function [DistError, Pos_y, Pos_z, A, B, C, D, E, F, Ar]  = amsRay(LAB,...
    LCD, Ip, r, LEF, dAy, dAz, ABry, ABrz, Brz, BrV1, Bn, dCx, dCz, dDx, dDz)
% Simulates Radial and Distance Errors with power loss over AMS interferometer
% Dr Jody Muelaner 2016, University of Bath, jody@muelaner.com
% V9 just renamed to amsRay for final output (was MistRay08)
%
% OUTPUTS (in mm):
% * DistError = Error in measurement path BCDE
% * Pos_y = y position of ray on detector
% * Pos_z = z position of ray on detector
% * Coordinates of points A - F
% INPUTS (in mm and deg):
% * Nominal interferometer geometry
%   * LAB = Distance from laser to splitter
%   * LCD = Distance between surfaces of spheres
%   * Ip = Position of splitter between spheres (0=on S1, 1=on S2)
%   * r = Radius of spheres
%   * LEF Distance from splitter to detector
% * Alignment error parameters for laser source
%   * dAy = Radial error of laser source in y direction
%   * dAz = Radial error of laser source in z direction
%   * ABry = Rotation of laser source about y axis in deg (+ve in CAD)
%   * ABrz = Rotation of laser source about z axis in deg (-ve in CAD)
% * Alignment error parameters for beam splitter
%   * Brz = Rotation of beam splitter about z axis in deg (+ve in CAD)
%   * BrV1 = Elevation angle of beam splitter in deg (rotation about V1) (-ve in CAD)
% * Alignment error parameters for sphere positions
%   * dCx = Radial error of first sphere in x direction
%   * dCz = Radial error of first sphere in z direction
%   * dDx = Radial error of second sphere in x direction
%   * dDz = Radial error of second sphere in z direction
% Requires external functions:
% * ReflectVector
% * Geom3D Toolbox
% * rodrigues_rot
%
% Geometry definitions in Geom3D:
% * Lines: [x0 y0 z0 dx dy dz] (point and direction vector)
% * Planes: [x0 y0 z0 dx1 dy1 dz1 dx2 dy2 dz2] (point and vectors in plane)
% * Spheres: [xc yc zc  R]
%
%Coordinate system has:
% * +ve x-axis pointing from beam splitter to laser source
% * +ve y-axis pointing from beam splitter to first sphere
% * +ve z-axis pointing up normal to plane of interferometer
%
% Validaton:
% Simulated points A-F validated against a SolidWorks beam model to within 10 nm

%% Convert input angles
ABry = degtorad(ABry); %Angle now in radians
ABrz = degtorad(ABrz); %Angle now in radians
Brz = degtorad(Brz); %Angle now in radians
BrV1 = degtorad(BrV1); %Angle now in radians

%% Define Geometry of reflecting surfaces

% Plane of splitter: [x0 y0 z0 dx1 dy1 dz1 dx2 dy2 dz2]
% * Point on plane at origin
% * V1 in plane of splitter and on x-y at 45 deg + Brz
% * V2 in plane nominally normal to V1 rotated Bxy about V1
splitterV1 = [-sin((pi/4)-Brz) cos((pi/4)-Brz) 0]; %Actual V1
splitterV1 = normalizeVector3d(splitterV1); %normalized V1
splitterV2 = [0 0 1]; %nominal V2
splitterV2 = rodrigues_rot(splitterV2,splitterV1,BrV1); %Actual V2
splitterV2 = normalizeVector3d(splitterV2); %normalized V1
splitter = [0 0 0 splitterV1 splitterV2]; %Plane of splitter, passing origin
B_d = planeNormal(splitter); %Normalized surface normal for splitter
Bn = Bn*B_d; %Resolve components of splitter offset
splitter = [Bn splitterV1 splitterV2]; %Actual plane of splitter

%Define planes of source and detector
source = [LAB 0 0 0 1 0 0 0 1];
detector = [-LEF 0 0 0 1 0 0 0 1]; 

%Spheres: [xc yc zc  R]
sphere1 = [dCx (LCD*Ip)+r dCz r]; % sphere 1 centre coordinates
sphere2 = [dDx (LCD*(Ip-1))-r dDz r]; % sphere 2 centre coordinates

%% Calculate actual path AB
A = [LAB, dAy, dAz];  %xyz position of laser

% Convert angles to rotation matracies
ABry = [cos(ABry) 0 sin(ABry);0 1 0; -sin(ABry) 0 cos(ABry)];
ABrz = [cos(ABrz) -sin(ABrz) 0;sin(ABrz) cos(ABrz) 0;0 0 1];

% Find line AB
AB_d = [-1 0 0]; %nominal direction of AB
AB_d = ABry*AB_d'; %rotate about y
AB_d = ABrz*AB_d; %rotate about z
AB_d = normalizeVector3d(AB_d'); %normalized direction vector
AB = [A AB_d]; %Line AB in parametric form

% Find point B: where AB intersects with splitter
B = intersectLinePlane(AB, splitter); %intersect line and plane

%% Calculate actual path BC

BC_d = ReflectVector(AB_d, B_d); %Normalized direction of BC
BC = [B BC_d]; %Line BC in parametric form

%Before finding point C ensure BC will intersect with sphere1 (<r)
%And that there is no excessive divergence (<r/3)
plane1 = [sphere1(1:3),1,0,0,0,0,1];
intersect1 = intersectLinePlane(BC, plane1);
if pdist([sphere1(1:3);intersect1],'euclidean')>r/3
    %Function returns nan and then exits
    DistError =  nan; Pos_y = nan; Pos_z = nan; 
    C = [nan, nan, nan]; D = [nan, nan, nan]; E = [nan, nan, nan]; F = [nan, nan, nan];
    return
end

C = intersectLineSphere(BC, sphere1); % 2 possible positions for C

sizeC = size(C);
if sizeC(1) == 2 %If two points of interestion select one with smallest y-coordinate
    if C(1,2) > C(2,2)
        C = C(2,:);
    else
        C = C(1,:);
    end
end


%% Calculate actual path CD

%Find normal to sphere at C
sphere1_c = sphere1(1:3); %centre of sphere 1
C_d = normalizeVector3d(C - sphere1_c); % Normalized surface normal at C
CD_d = ReflectVector(BC_d, C_d); %Normalized direction of BC
CD = [C CD_d]; %Line CD in parametric form

%Before finding point D ensure CD will intersect with sphere2 (<r)
%And that there is no excessive divergence (<r/3)
plane2 = [sphere2(1:3),1,0,0,0,0,1];
intersect2 = intersectLinePlane(CD, plane2);
if pdist([sphere2(1:3);intersect2],'euclidean')>r/3
    %Function returns nan and stops
    DistError =  nan; Pos_y = nan; Pos_z = nan; 
    D = [nan, nan, nan]; E = [nan, nan, nan]; F = [nan, nan, nan];
    return
end

D = intersectLineSphere(CD, sphere2); % 2 possible positions for D

sizeD = size(D);
if sizeD(1) == 2 %If two points of interestion select one with largest y-coordinate
    if D(1,2) < D(2,2)
        D = D(2,:);
    else
        D = D(1,:);
    end
end

%% Calculate actual path DE

%Find normal to sphere at D
sphere2_c = sphere2(1:3); %centre of sphere 2
D_d = normalizeVector3d(D - sphere2_c); % Normalized surface normal at D
DE_d = ReflectVector(CD_d, D_d); %Normalized direction of BC
DE = [D DE_d]; %Line CD in parametric form
E = intersectLinePlane(DE, splitter); %intersect line and plane

%If the y-coordinate of E is less than that of D then fails on this condition!
%Check DE travels forwards towards the real splitter and detector
if E(2)<D(2)
    %Function returns nan and exits
    DistError =  nan; Pos_y = nan; Pos_z = nan; F = [nan, nan, nan];
    return
end

%Check that DE intersects the splitter before the detector
E_det = intersectLinePlane(DE, detector); %intersect line and plane
if pdist([D;E],'euclidean')>pdist([D;E_det],'euclidean')
    %Function returns nan and exits
    DistError =  nan; Pos_y = nan; Pos_z = nan; F = [nan, nan, nan];
    return
end

%% Calculate actual path EF
EF_d = ReflectVector(DE_d, B_d); %Normalized direction of EF
EF = [E EF_d]; %Line BC in parametric form
F = intersectLinePlane(EF, detector); %intersect line and plane

%% Calculate reference path ArF
% AB_d is normalized direction of AB
ArF = [F AB_d]; %Line ArF in parametric form
Ar = intersectLinePlane(ArF, source); %Find starting point for reference arm

%% Output final results
DistError = norm(A-B) + norm(B-C) + norm(C-D) + norm(D-E) + norm(E-F) - norm(Ar-F) - (2*LCD);
% This shouldn't be RQD! DistError = 0 - DistError; % Give it the right sign!
Pos_y = F(2);
Pos_z = F(3);
end
