function [meanE,RangeE,TotalP] = amsSim(r, r_det, Ip, dIx, dIz, dIrx, dIrz)
% Simulates AMS interferometer (multiple rays) for one
% configuration and error set
% V6 just renamed amsSim (was amsSim05)
% 
%Inputs
% * r & r_det: radius of spheres and detector (mm)
% * Ip: Beam splitter position (% of distance from S1 to S2)
% * dIx & dIz: Translations of interferometer assembly (mm)
% * dIrx & dIrz: Rotations of interferometer assembly (deg)

%% Inputs defined within function:
LAB=10; LCD=1000; LEF=10; % IFM dimensions (Length AB, CD & EF)
dAy = 0.00; dAz = 0.00; ABry = 0.00; ABrz = -0.00; % Laser aligned to detector
Brz = 0; BrV1 = 0; Bn = 0; % Splitter aligned to laser/detector
P0 = 1; w0 = 1; I0 = 2*P0/(pi*(w0^2)); % Power (W), beam width (mm) & peak intensity (W/mm^2)

% Calculate Sphere translations wrt interferometer assembly
dIrx = degtorad(dIrx); dIrz = degtorad(dIrz); %Convert to radians
dCx = LCD*Ip*tan(dIrz) - dIx; dDx = -dIx - LCD*(1-Ip)*tan(dIrz);
dCz = LCD*Ip*tan(dIrx) - dIz; dDz = -dIz - LCD*(1-Ip)*tan(dIrx);

% Parameters for numerical integration over beam
ndydz = 40; % Set number of increments in each direction
r_beam_max = w0 * 6; %Integrate over upto +/- 6 standard deviations
LimSc = 0.02; %Amount to expand each limit in each direction before integration

%Set initial beam limits and search parameters
BeamYmin = -r_beam_max; BeamYmax = r_beam_max;
BeamZmin = -r_beam_max; BeamZmax = r_beam_max;
Step = r_beam_max/50; % Initial step size for pattern searches

%% Find centre of sphere 1
Beam=[0 0]; % Start search at centre of beam
LB = [BeamYmin, BeamZmin]; UB = [BeamYmax, BeamZmax]; % Set limits for search
options = psoptimset('InitialMeshSize',Step,'TolFun',1e-12,'TolMesh',1e-12,'TolX',1e-24,'Display','off',...
    'MeshContraction',0.5,'MeshExpansion',2,'CompletePoll','on','MaxIter',1000);
[Beam] = patternsearch(@(Beam)objFunC(dCx, dCz, LAB, LCD, Ip, r, LEF, Beam,...
    ABry, ABrz, Brz, BrV1, Bn, dCx, dCz, dDx, dDz),Beam,[],[],[],[],LB,UB,options);

% Verification: Remove latter...
%disp('Found centre of sphere 1')
%[~, ~, ~,~,~,C]  = MistRay08(LAB, LCD, Ip, r, LEF, Beam(1), Beam(2),...
%    ABry, ABrz, Brz, BrV1, Bn, dCx, dCz, dDx, dDz); % Verify found centre
%distX = C(1)-dCx; distZ = C(3)-dCz; dist = sqrt(distX^2+distZ^2); % Dist.

%% Find centre of sphere 2
[Beam] = patternsearch(@(Beam)objFunD(dDx, dDz, LAB, LCD, Ip, r, LEF, Beam,...
    ABry, ABrz, Brz, BrV1, Bn, dCx, dCz, dDx, dDz),Beam,[],[],[],[],LB,UB,options);

% Verification: Remove latter...
%disp('Found centre of sphere 2')
%[~, ~, ~,~,~,~,D]  = MistRay08(LAB, LCD, Ip, r, LEF, Beam(1), Beam(2),...
%    ABry, ABrz, Brz, BrV1, Bn, dCx, dCz, dDx, dDz); % Verify found centre
%distX = D(1)-dDx; distZ = D(3)-dDz; dist = sqrt(distX^2+distZ^2); % Dist.

%% Find corners of detector
[BeamMin] = patternsearch(@(Beam)objFunF(-r_det, -r_det, LAB, LCD, Ip, r, LEF, Beam,...
    ABry, ABrz, Brz, BrV1, Bn, dCx, dCz, dDx, dDz),Beam,[],[],[],[],LB,UB,options);
% Verification: Remove latter...
%disp('Found lower corner of detector')
%[~, ~, ~,~,~,~,~,~,F]  = MistRay08(LAB, LCD, Ip, r, LEF, BeamMin(1), BeamMin(2),...
%    ABry, ABrz, Brz, BrV1, Bn, dCx, dCz, dDx, dDz) % Verify found centre
%distY = F(2)+r_det; distZ = F(3)+r_det; dist = sqrt(distY^2+distZ^2) % Dist.


[BeamMax] = patternsearch(@(Beam)objFunF(r_det, r_det, LAB, LCD, Ip, r, LEF, Beam,...
    ABry, ABrz, Brz, BrV1, Bn, dCx, dCz, dDx, dDz),Beam,[],[],[],[],LB,UB,options);
% Verification: Remove latter...
%disp('Found upper corner of detector')
%[~, ~, ~,~,~,~,~,~,F]  = MistRay08(LAB, LCD, Ip, r, LEF, BeamMax(1), BeamMax(2),...
%    ABry, ABrz, Brz, BrV1, Bn, dCx, dCz, dDx, dDz) % Verify found centre
%distY = F(2)-r_det; distZ = F(3)-r_det; dist = sqrt(distY^2+distZ^2) % Dist.

%% Update beam limits & expand to ensure integration covers detector
BeamYmin = BeamMin(1); BeamZmin = BeamMin(2);
BeamYmax = BeamMax(1); BeamZmax = BeamMax(2);
Yrange = BeamYmax-BeamYmin;
BeamYmin = BeamYmin - (LimSc * Yrange); BeamYmax = BeamYmax + (LimSc * Yrange);
Zrange = BeamZmax-BeamZmin;
BeamZmin = BeamZmin - (LimSc * Zrange); BeamZmax = BeamZmax + (LimSc * Zrange);

%% Numerical integration (raster over bounds determined above)
dy = (BeamYmax-BeamYmin)/(ndydz-1); %Set step size in y direction
dz = (BeamZmax-BeamZmin)/(ndydz-1); %Set step size in z direction
results = zeros(ndydz^2,5); % Empty array to store results

for i=1:ndydz %Outter loop scanns in z from BeamZmin to BeamZmax
    rdAz = BeamZmin + dz*(i - 1);
    
    for k=1:ndydz %Inner loop scanns in y from BeamMinY to BeamMaxY
        rdAy = BeamYmin + dy*(k - 1);
        row = k + ndydz*(i - 1); % row to store results
        [results(row,3), Pos_y, Pos_z]  = amsRay(LAB, LCD, Ip, r, LEF, rdAy, rdAz, ABry, ABrz, Brz, BrV1, Bn, dCx, dCz, dDx, dDz);
        results(row,1) = sqrt(Pos_y^2+Pos_z^2); % Find radial position of ray on detector
        
        results(row,4) = i; results(row,5) = k;
        
        %For power remove the initial alignment error and create new variables:
        rdAy_beam = rdAy - dAy; % Actual position of ray in beam frame
        rdAz_beam = rdAz - dAz; % Actual position of ray in beam frame
        pos_r_beam = sqrt(rdAy_beam^2+rdAz_beam^2); % Radial position of ray in beam
        Pd = I0*exp(-2*pos_r_beam^2/w0^2); % Power density for this ray
        results(row,2) = Pd*dy*dz; % Power for this finite region of beam
    end
end

results = results(results(:, 1) < r_det, :); % Filter results on detector

% Calculate summary results
if length(results)>10
    meanE = 1000*mean(results(:,3)); % Result in microns
    RangeE = 1000*(max(results(:,3))-min(results(:,3))); % Result in microns
    TotalP = sum(results(:,2))*(10^9); % Fractional power (10^-9)
else
    meanE=NaN; RangeE=NaN; TotalP=NaN;
end
outStr=strcat('COMPLETED integration: meanE=',num2str(meanE),'microns. RangeE=',num2str(RangeE),'microns. TotalP=',num2str(TotalP),'10^-9 Fractional power');
disp(outStr)
end