%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% A_IntCp: Analytical integration of heat capacity
%          from Shomate equations
%
% DCpdT = A_IntCp('species',T1,T2)
%
% Where T1 and T2 are temperatures in degrees K
% and DCpdT is returned in kJ/mol
% T1 is typically a reference temperature, while
% T2 may be given as a vector. In this case, the
% output DCpdT will be a row vector.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function Delta_H_vect = A_IntCp(species,T1,T2_vect)

%% Import data
[coefficients, T_ranges] = getdata(species);

%% Begin loop over vector
Delta_H_vect = zeros(1,length(T2_vect));
for element = 1:length(T2_vect)
T2 = T2_vect(element);
%% Check range is ok

if (T1 == T2)
    Delta_H = 0;

elseif min([T1 T2]) < min(T_ranges)
    error( ... 
        'Lower temperature %4.2fK is below lower bound for data (%dK)', ...
        min([T1 T2]), min(T_ranges));
elseif max([T1 T2]) > max(T_ranges)
    error( ...
        'Upper temperature %4.2fK is above upper bound for data (%dK)', ...
        max([T1 T2]), max(T_ranges));

end

%% Allocate temperature range and initialise integral

T_upper = max([T1 T2]);
T_lower = min([T1 T2]);

Delta_H = 0;

%% Iterate through temperature ranges, applying analytical integration as 
%  appropriate
%
%  (For each step, T_ranges(i) is the lower limit for the current
%  integration region and T_ranges(i+1) is the upper limit for the region,
%  as defined in the data set)
i=1;
while T_ranges(i) < T_upper

    if T_ranges(i+1) <= T_lower
        % Do nothing
    else
        % Fix lower bound for this integration step
        if T_ranges(i) <= T_lower
            T_step_lower_bound = T_lower;
        else
            T_step_lower_bound = T_ranges(i);
        end
        % Fix upper bound for this integration step
        if T_ranges(i+1) >= T_upper
            T_step_upper_bound = T_upper;
        else
            T_step_upper_bound = T_ranges(i+1);
        end
        
        Delta_H = Delta_H + ...
            IntCp(coefficients(i,:),T_step_lower_bound,T_step_upper_bound);
    end
    
   % disp(Delta_H);
    
    i = i+1;
end

%% Check sign, correct if necessary

if T1 > T2
    Delta_H = -Delta_H;
end

Delta_H_vect(element) = Delta_H;
end

function IntCp = IntCp(coefficient_row, T_step_lower_bound, T_step_upper_bound);

%%%%% Shomate Eqn correlation form:
%%%%% Cp = A + B(t) + C(t^2) + D(t^3) + E(t^-2)
%%%%%
%%%%% Analytical integral:
%%%%% int(Cp)dt = At + (B/2)(t^2) + (C/3)(t^3) + (D/4)(t^4) - E(t^-1) + k
%%%%% where k is constant of integration and A-E = coefficient_row(1:5)
%%%%%
%%%%% t = T/1000
%%%%% Note that this factor results in answers being given in kJ/mol
%%%%% rather than J/mol

t1 = T_step_lower_bound/1000;
t2 = T_step_upper_bound/1000;

t1_row = [t1 t1^2 t1^3 t1^4 t1^-1];
t2_row = [t2 t2^2 t2^3 t2^4 t2^-1];
factors = [1 0.5 1/3 0.25 -1];

IntCp = sum(coefficient_row(1:5).*factors.*(t2_row-t1_row));

function [coefficients, T_ranges] = getdata(species)

switch species

%%%%%%%%%%%% Data for Oxygen %%%%%%%%%%%
% Chase, M.W., Jr., NIST-JANAF Themochemical Tables, Fourth Edition,
% J. Phys. Chem. Ref. Data, Monograph 9, 1998, 1-1951. [all data]

    case {'O2' 'o2' 'Oxygen' 'oxygen'}

     T_ranges=[100; 700; 2000; 6000];
 
 
	coefficients(1,:) = [31.32234
			-20.23531
			57.86644
			-36.50624
			-0.007374
			-8.903471
			246.7945
			0]';
  
	coefficients(2,:) = [30.03235
			8.772972
			-3.988133
			0.788313
			-0.741599
			-11.32468
			236.1663
			0]';
    
	coefficients(3,:) = [20.91111
			10.72071
			-2.020498
			0.146449
			9.245722
			5.337651
			237.6185
			0]';
     

%%%%%%%%%%%% Data for Nitrogen %%%%%%%%%%%
% Chase, M.W., Jr., NIST-JANAF Themochemical Tables, Fourth Edition,
% J. Phys. Chem. Ref. Data, Monograph 9, 1998, 1-1951. [all data]

    case {'N2' 'n2' 'Nitrogen' 'nitrogen'}

        T_ranges=[100; 500; 2000; 6000]; 
      
      
	coefficients(1,:) = [28.98641
			1.853978
			-9.647459
			16.63537
			0.000117
			-8.671914
			226.4168
			0];
   
    coefficients(2,:) = [19.50583
			19.88705
			-8.598535
			1.369784
			0.527601
			-4.935202
			212.39
			0];
    
	coefficients(3,:) = [35.51872
			1.128728
			-0.196103
			0.014662
			-4.55376
			-18.97091
			224.981
			0];

%%%%%%%%%%% End of data %%%%%%%%%%%%%%%

  end

