%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                              %
%  GaN_Cp_lit.m:  Comparative heat capacities of GaN from literature           %
%                  and ab-initio phonon calculations                           %
%                                                                              %
%  Note that thermal expansion is disregarded such that Cp and Cv are used     %
%  interchangably                                                              %
%                                                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function GaN_Cp_lit()

%% Fundamental constants and setup
k_B = 1.3806488E-23; % J/K
N_A = 6.02214129E23; % /mol

source = {'phonopy','Danilchenko','Jacob','Leitner'};
linestyle = {'k-','k--','k:','k-.'};
name = {'PBEsol with harmonic approximation','Adiabatic calorimetry measurements and fit by Danilchenko et al. (2006)', ...
    'Solid-state EMF measurements and fit by Jacob et al. (2007)', ...
    'Calvet calorimetry by Leitner et al. (2003), \n fit incorporating data from Koschenko et al. (1979) and Itagaki and Yamaguchi (1990)'};

%% Ab initio results from phonopy-FHI-aims

FU.phonopy         = 2; % Formula units / unit cell

phonopy = importdata('data/GaN/phonopy-FHI-aims-free_energy.dat');

data.('phonopy').T = phonopy.data(:,1);
data.('phonopy').Cp = phonopy.data(:,4)*k_B*N_A/FU.phonopy; % J/mol

%% Fitted experimental data by Danilchenko et al (2006)
data.('Danilchenko').T = [1:1:2000];

data.('Danilchenko').Cp = zeros(1,length(data.('Danilchenko').T));
for i = 1:length(data.('Danilchenko').Cp)
  data.('Danilchenko').Cp(i) = Cp_Danilchenko(data.('Danilchenko').T(i));
end


%% Fitted experimental data by Jacob et al (2007)
data.('Jacob').T = [350:1:1075];
data.('Jacob').Cp = 74.424 - 0.00106*data.('Jacob').T ...
    + 46720./(data.('Jacob').T.^2) - 685.9./(data.('Jacob').T.^0.5);

%% Fitted experimental data by Leitner et al
%%%%%%% Thermochimica Acta 401 (2003) 169.
%%%%%%%  including data from:
%%%%%%% K. Itagaki, K. Yamaguchi, Thermochimica Acta 163 (1990) 1.
%%%%%%% V.I. Koschenko et al., Izv. Akad. Nauk SSSR, Neorg. Mater. 16 (1999)
%%%%%%% 1686.

data.('Leitner').T = [200:1:1300];
data.('Leitner').Cp = 49.552 + 5.440E-3*data.('Leitner').T ...
    - 2.190E6*data.('Leitner').T.^-2 + 2.460E8*data.('Leitner').T.^-3;

%% Plotting

close all;
figure;
hold on
for i=1:length(source)
plot(data.(source{i}).T,data.(source{i}).Cp,linestyle{i});
end
hold off

xlabel('Temperature / K','interpreter','latex');
ylabel('$C_p$ / J mol$^{-1}$ K$^{-1}$','interpreter','latex');
%legend(name,'Location','SouthEast','interpreter','latex');

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   Function implements combined Debye/Einstein model     %
%   according to Danilchenko et al (2006)                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function Cp = Cp_Danilchenko(T)

  av = 6.02214129E23;    % /  mol^-1
  k_B = 1.380650424E-23; % / J K^-1
  Theta_D = 365; % /K
  Theta_E = 880; % /K

  if T < 2
    Cp = 0;
% Safety measure - these values are negligible but the exponents
% overflow and cause errors. A more rigorous approach would
% avoid this through the use of analytical solutions; these are
% readily available for the Debye model, but not for this hybrid.

  elseif T > 720
    Cp = 3*av*k_B + ...                        % Dulong and Petit law
	3*av*k_B*(Theta_E/T)^2 * (exp(Theta_E/T))/((exp(Theta_E/T)-1)^2); 
  else
    Cp = 3*av*k_B*3*(T/Theta_D)^3*quad(@F,0,Theta_D/T) +  ...
     3*av*k_B*(Theta_E/T)^2 * (exp(Theta_E/T))/((exp(Theta_E/T)-1)^2);
  end

end


function FT = F(x)
  FT = (x.^4).*exp(x)./((exp(x) - 1).^2);
end