%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                              %
%   H_supercell: calculates the chemical potentials of defect-containing       %
%                supercells from ground-state DFT and phonon calculations,     %
%                assuming thermal are same as pure material.                   %
%                                                                              %
%     usage: mu = H_solid('species',T,P)                                       %
%                                                                              %
%     where H is the enthalpy in J/mol, T is the temperature in K              %
%     and P is the absolute pressure in Pa. 'species' is a string specifying   %
%     the material. For vectorised calculations T may be a row vector and P    %
%     may be a column vector, in which case mu is a matrix of values which     %
%     correspond to each combination.                                          %
%                                                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%    Requires: FHI-aims geometry file for supercell and phonopy thermal        %
%             properties data for parent species:                              %
%                             data/(species)/geometry.in                       %
%                             data/(species)/phonopy-FHI-aims-free-energy.dat  %
%                                                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function H = H_supercell(species,T,P)

eV2J = 1.602176565E-19; % J/eV
N_A  = 6.02214129E23;   % /mol

E_DFT_eV.('GaN')   = -0.109256874892143E6;  % eV (run 0234)
E_DFT_eV.('S_72')  = -0.196717956674226E7;  % eV (run 0164)
E_DFT_eV.('S_128') = -0.349677592856924E7;  % eV (run 0210)
E_DFT_eV.('S_300') = -0.819482171562133E7;  % eV (run 0217)

FU.('Ga')          = 8;  % Formula units / unit cell
FU.('GaN')         = 2;  % Formula units / unit cell
FU.('Ga2O3')       = 4;  % Formula units / unit cell
FU.('S_72')        = 36; % GaN Formula units / supercell without defect
FU.('S_128')       = 64; % GaN Formula units / supercell without defect
FU.('S_300')       = 150;% GaN Formula units / supercell without defect

thermal=importdata(strcat('data/GaN/phonopy-FHI-aims-free_energy.dat'));

F = interp1(thermal.data(:,1),thermal.data(:,2),T,'spline') * FU.(species)/ FU.('GaN');  % eV
negTS =  interp1(thermal.data(:,1),thermal.data(:,5),T,'spline') * FU.(species)/ FU.('GaN');  % eV
U = F - negTS; % eV

U = kron(U,ones(length(P),1)); % Copy rows to complete matrix

geometry = importdata(strcat('data/',species,'/geometry.in'));
V = dot(geometry.data(1,:),cross(geometry.data(2,:),geometry.data(3,:))); % AA^3

PV = P*V*1E-30; % J
PV = kron(PV,ones(1,length(T))); % Copy columns to complete matrix

H = ((E_DFT_eV.(species) + U)*eV2J + PV)*N_A; % J/mol

%disp(E_DFT_eV.(species)*eV2J*N_A/FU.(species));
%disp(F*eV2J*N_A);
%disp(PV*N_A/FU.(species));
