%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                              %
% Sig.m: Calculate the entropy of an ideal gas using Shomate equations with T, %
%        followed by a pressure correction.                                    %
%                                                                              %
% Usage: S = Sig('species',T,P)                                                %
%        where is is the entropy in J/molK, T is the temperature in K, P is    %
%        the partial pressure in Pa, 'species' is a string identifying the gas.%
%        T may be a (1xn) row vector and P may be a (mx1) column vector, in    %
%        which case S is returned as an m x n matrix.                          %
%                                                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function S = Sig(species,T,P)

P0 = 1E5; % Reference state is 1 bar for all data used here
          % (A later version of this program may need to support
          %  multiple reference states.)

S_T = zeros(1,length(T));
[coefficients, T_ranges] = getdata(species);

          % Step through valid temperature ranges, computing all valid S_Ts

for i=1:length(T_ranges)-1

          % This is then divided by 1000 to normalise to the correct units
          %  for the Shomate correlation

  t = T/1000;

  S_T(i,:) = coefficients(i,1)*log(t) + coefficients(i,2)*t + ...
      coefficients(i,3)*(t.^2)/2 + coefficients(i,4)*(t.^3)/3 - ...
      coefficients(i,5)./(2*t.^2) + coefficients(i,7);
          % Multiply out-of-range values by zero
  S_T(i,:) = S_T(i,:).*(T>=T_ranges(i) & T<T_ranges(i+1));
end
          % Collapse S_T to single row by summing columns
S_T = sum(S_T,1);
          % Duplicate rows with Kronecker tensor to deal with different pressures
S_T = kron(S_T,ones(length(P),1));

% Change in entropy with pressure at constant pressure
% Expansion with kronecker tensor follows log as log is costly operation
% S = S_0 - nR log(p/p_0)
S_TP = S_T - kron(8.314*log(P/P0),ones(1,length(T)));

% Return solution
S = S_TP;


function [coefficients, T_ranges] = getdata(species)

switch species

%%%%%%%%%%%% Data for Oxygen %%%%%%%%%%%
% Chase, M.W., Jr., NIST-JANAF Themochemical Tables, Fourth Edition,
% J. Phys. Chem. Ref. Data, Monograph 9, 1998, 1-1951. [all data]

    case {'O2' 'o2' 'Oxygen' 'oxygen'}

     T_ranges=[100; 700; 2000; 6000];
 
 
	coefficients(1,:) = [31.32234
			-20.23531
			57.86644
			-36.50624
			-0.007374
			-8.903471
			246.7945
			0]';
  
	coefficients(2,:) = [30.03235
			8.772972
			-3.988133
			0.788313
			-0.741599
			-11.32468
			236.1663
			0]';
    
	coefficients(3,:) = [20.91111
			10.72071
			-2.020498
			0.146449
			9.245722
			5.337651
			237.6185
			0]';
     

%%%%%%%%%%%% Data for Nitrogen %%%%%%%%%%%
% Chase, M.W., Jr., NIST-JANAF Themochemical Tables, Fourth Edition,
% J. Phys. Chem. Ref. Data, Monograph 9, 1998, 1-1951. [all data]

    case {'N2' 'n2' 'Nitrogen' 'nitrogen'}

        T_ranges=[100; 500; 2000; 6000]; 
      
      
	coefficients(1,:) = [28.98641
			1.853978
			-9.647459
			16.63537
			0.000117
			-8.671914
			226.4168
			0];
   
    coefficients(2,:) = [19.50583
			19.88705
			-8.598535
			1.369784
			0.527601
			-4.935202
			212.39
			0];
    
	coefficients(3,:) = [35.51872
			1.128728
			-0.196103
			0.014662
			-4.55376
			-18.97091
			224.981
			0];

%%%%%%%%%%% End of data %%%%%%%%%%%%%%%

  end

