%
% Standard formation enthalpies of compounds
%
% enthalpies.m
%

%% DFT energies

Ga_DFT = -0.425137737497745E6;    % eV (run 0222)
GaN_DFT = -0.109256874892143E6;   % eV (run 0234)
Ga2O3_DFT = -0.449656504853252E6; % eV (run 0230)
N2_DFT = -0.297008353771862E4;    % eV (run 0236)
O2_DFT = -0.408004839112704E4;    % eV (run 0237)

%% Solid vibrational energy integration

Solids.name = {'Ga','GaN','Ga2O3'};
Solids.FU = [8,2,4]; % Formula units in cell

for i = 1:length(Solids.name)
    % Import data from phonopy calculations
    phonopy(i).raw = importdata(strcat('data/',Solids.name{i},'/phonopy-FHI-aims-free_energy.dat'));
    % Get column of internal energy change from zero (including zero-point energy)
    % by subtracting (-TS) contribution from Helmholtz energy F
    phonopy(i).U = phonopy(i).raw.data(:,2) - phonopy(i).raw.data(:,5); % eV/cell
    phonopy(i).U298 = interp1(phonopy(i).raw.data(:,1),phonopy(i).U,298.15,'cubic');
    phonopy(i).zpe = phonopy(i).U(1); % eV/cell
end

%% Solid PV work

PV = zeros(length(Solids.name));
for i = 1:length(Solids.name)
    
    geometry(i).raw = importdata(strcat('data/',Solids.name{i},'/geometry.in'));
    
    geometry(i).a = geometry(i).raw.data(1,:);
    geometry(i).b = geometry(i).raw.data(2,:);
    geometry(i).c = geometry(i).raw.data(3,:);
    
        % Unit cell volume in cubic angstroms
    geometry(i).volumeAA = dot(cross(geometry(i).a,geometry(i).b),geometry(i).c);
    % Convert to m^3
    geometry(i).volume = geometry(i).volumeAA * 1E-30;
    
    PV(i) = 1E5 * geometry(i).volume;
end

%% Gas standard enthalpy from zero

O2_H298 = 8.680E3; % J/mol
N2_H298 = 8.670E3; % J/mol

%% Gas zero point energies

gas_zpe; % Literature ZPE for gases from external file


%% Ab initio standard enthalpies

av = 6.02214129E23;       % /mol
eV2J = 1.602176565E-19;   % eV/J

H_Ga = (Ga_DFT + phonopy(1).U298)*eV2J*av/Solids.FU(1);
H_GaN = (GaN_DFT + phonopy(2).U298)*eV2J*av/Solids.FU(2);
H_Ga2O3 = (Ga2O3_DFT + phonopy(3).U298)*eV2J*av/Solids.FU(3);

H_N2 = N2_DFT*eV2J*av + N2_H298 + zpe_N2_Jmol;
H_O2 = O2_DFT*eV2J*av + O2_H298 + zpe_O2_Jmol;


%% Formation enthalpies

DHF_GaN = H_GaN - (H_Ga + 0.5*H_N2);
DHF_Ga2O3 = H_Ga2O3 - (2*H_Ga + 1.5*H_O2);

fprintf('Enthalpy of formation of GaN:   %4.2f kJ/mol\n',DHF_GaN/1000);
fprintf('Enthalpy of formation of Ga2O3: %4.2f kJ/mol\n',DHF_Ga2O3/1000);

%% Generate bar charts

%%%% DFT ground-state energy %%%%
formation.GaN.dft = (GaN_DFT/Solids.FU(2) - (Ga_DFT/Solids.FU(1) + 0.5*N2_DFT)) ...
                    *av*eV2J;
formation.Ga2O3.dft = (Ga2O3_DFT/Solids.FU(3) - (2*Ga_DFT/Solids.FU(1) + 1.5*O2_DFT)) ...
                    *av*eV2J;
%%%% Zero-point energy %%%%
formation.GaN.zpe = (phonopy(2).zpe/Solids.FU(2))*av*eV2J ...
                    - (phonopy(1).zpe/Solids.FU(1))*av*eV2J ...
                    - 0.5*zpe_N2_Jmol;
formation.Ga2O3.zpe = (phonopy(3).zpe/Solids.FU(3))*av*eV2J ...
                - (2*phonopy(1).zpe/Solids.FU(1))*av*eV2J ...
                - 1.5*zpe_O2_Jmol;
%%%% Integration to 298.15K %%%%
formation.GaN.cp  = ((phonopy(2).U298-phonopy(2).zpe)/Solids.FU(2))*av*eV2J ...
                    - ((phonopy(1).U298-phonopy(1).zpe)/Solids.FU(1))*av*eV2J ...
                    - 0.5*N2_H298;
formation.Ga2O3.cp  = ((phonopy(3).U298-phonopy(3).zpe)/Solids.FU(3))*av*eV2J ...
                    - (2*(phonopy(1).U298-phonopy(1).zpe)/Solids.FU(1))*av*eV2J ...
                    - 1.5*O2_H298;
%%%%  Experimental values %%%%
formation.GaN.exp = -110E3;       % J/mol
formation.Ga2O3.exp = -1089.1E3;  % J/mol

y = [formation.GaN.dft, formation.GaN.zpe, formation.GaN.cp, DHF_GaN, formation.GaN.exp]/1000; % kJ/mol
fig1 = bar(y,0.5,'facecolor',[0.5,0.5,0.5]);
ylabel('Energy / kJ mol$^{-1}$','interpreter','latex');
set(gca,'XTickLabel',{'$\Delta E^\text{DFT}$','$\Delta E^\text{ZP}$', ...
               '$\int^{298.15}_0 \Delta C_p \text{d}T$' ...
               ,'$\Delta H^\theta_{f,GaN}$','Experimental'});
set(gcf,'Units','centimeters');
set(gcf,'Position',[0 0 8.6 6]);

figure;

y = [formation.Ga2O3.dft, formation.Ga2O3.zpe, formation.Ga2O3.cp, DHF_Ga2O3, formation.Ga2O3.exp]/1000; % kJ/mol
fig1 = bar(y,0.5,'facecolor',[0.5,0.5,0.5]);
ylabel('Energy / kJ mol$^{-1}$','interpreter','latex');
set(gca,'XTickLabel',{'$\Delta E^\text{DFT}$','$\Delta E^\text{ZP}$', ...
               '$\int^{298.15}_0 \Delta C_p \text{d}T$' ...
               ,'$\Delta H^\theta_{f,\ce{Ga2O3}}$','Experimental'});
set(gcf,'Units','centimeters');
set(gcf,'Position',[0 0 8.6 6]);
