%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                              %
%  mu_solid: calculates the chemical potentials of solid materials from        %
%            ground-state DFT and phonon calculations                          %
%                                                                              %
%     usage: mu = mu_solid('species',T,P)                                      %
%                                                                              %
%     where mu is the chemical potential in J/mol, T is the temperature in K   %
%     and P is the absolute pressure in Pa. 'species' is a string specifying   %
%     the material. For vectorised calculations T may be a row vector and P    %
%     may be a column vector, in which case mu is a matrix of values which     %
%     correspond to each combination.                                          %
%                                                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%    Requires: FHI-aims geometry file and phonopy thermal properties data for  %
%              each species: data/(species)/geometry.in                        %
%                            data/(species)/phonopy-FHI-aims-free-energy.dat   %
%                                                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function mu = mu_solid(species,T,P)

eV2J = 1.602176565E-19; % J/eV
N_A  = 6.02214129E23;   % /mol

E_DFT_eV.('Ga')    = -0.425137737497745E6;  % eV (run 0222)
E_DFT_eV.('GaN')   = -0.109256874892143E6;  % eV (run 0234)
E_DFT_eV.('Ga2O3') = -0.449656504853252E6;  % eV (run 0230)
E_DFT_eV.('S_72')  = -0.196717956674226E7;  % eV (run 0164)
E_DFT_eV.('S_128') = -0.349677592856924E7;  % eV (run 0210)

FU.('Ga')          = 8; % Formula units / unit cell
FU.('GaN')         = 2; % Formula units / unit cell
FU.('Ga2O3')       = 4; % Formula units / unit cell
FU.('S_72')        = 1; % Supercell
FU.('S_128')       = 1; % Supercell

thermal=importdata(strcat('data/',species,'/phonopy-FHI-aims-free_energy.dat'));

F = interp1(thermal.data(:,1),thermal.data(:,2),T,'spline');  % eV
F = kron(F,ones(length(P),1)); % Copy rows to complete matrix

geometry = importdata(strcat('data/',species,'/geometry.in'));
V = dot(geometry.data(1,:),cross(geometry.data(2,:),geometry.data(3,:))); % AA^3

PV = P*V*1E-30; % J
PV = kron(PV,ones(1,length(T))); % Copy columns to complete matrix

  mu = ((E_DFT_eV.(species) + F)*eV2J + PV)*N_A/FU.(species); % J/mol

%disp(E_DFT_eV.(species)*eV2J*N_A/FU.(species));
%disp(F*eV2J*N_A/FU.(species));
%disp(PV*N_A/FU.(species));
