/*  sandpile32d-128.c
 *  Abelian sandpile model in 32d computes height probability at the origin only
 *  Created by Antal A. Jarai and Minwei Sun.
 */

/* To compile:  mpicc -o sandpile32d-128 sandpile32d-128.c mt19937ar-aj.c jump_mt19937-aj.c
 * To run:  mpirun ./sandpile32d-128 <number of samples>
 * mt19937ar.c  source code of Mersenne Twister from author's website:
 * http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/MT2002/emt19937ar.html
 */

#include "mt19937ar-aj.h"   /* Header file for Random-Number Generator */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <mpi.h>

#define MASTER 0
#define SAMPLE_REQUEST 3
#define SAMPLE_REPLY 4

#define d 32
#define L 128
#define m 2 /* (unsigned int)pow(L, 6/d) */
#define mpower 1
#define ROWSIZE 256
#define HASHSIZE 0x1000000 /* */
#define HASHSIZEmONE 0xffffff /* HASHSIZE - 1 */

MPI_Comm world;
int sample_request;

unsigned long int prob[64] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
unsigned long int total_prob[64] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
float probability[64];
unsigned long int seed;   /* Seed for random number generator */
unsigned long int nb[64];
FILE *probability_file;
unsigned long *pf;
State s0;

static struct node *stack[HASHSIZE];
unsigned long int stack_pointer;
struct node
{
    struct node *next; /* next entry in the chain */
    struct node *treepos;
    unsigned int coordinates[d];
    unsigned char treeneighbour;
    unsigned char sand;
    unsigned char intree;
    unsigned long int distance;
    
};
static struct node *hashtab[HASHSIZE]; /* pointer table */
static struct node vertexdata[HASHSIZE]; /* actual node, not pointer */
struct node *freevertex = vertexdata; /* pointer to the first entry of vertexdata */
unsigned long int usage = 0;


/* hash: form hash value for coordinates point */
unsigned long int hash(unsigned int *coordinates)
{
    unsigned long int hashval;
    unsigned int i;
    
    hashval = 0;
    for (i = 0; i < d; i++)
    {
        hashval <<= mpower; /* mpower is 4 & binary operation *16 */
        hashval += coordinates[i];
    }
    return (hashval & HASHSIZEmONE); /* moduleo HASHSIZE */
}

/* install returns pointer to the vertex, it comes back either address where we can put the new vertex or pointer to the vertex which has been visited before */

struct node *install(unsigned int *coordinates)
{
    struct node *np, *nplast;
    unsigned long int hashval;
    char i, found;
    
    hashval = hash(coordinates);
    np = hashtab[hashval];
    
    if (np == NULL) /* hashval has not been used before */
    {
        hashtab[hashval] = freevertex;/* pointer to the first free vertex */
    }
    else
    {
        
        for (nplast = np ; np != NULL ; np = (np -> next) )
        {
            found = 1; /* 1 is ture, 0 is false */
            for ( i = 0 ; i < d; i++) /* check whether the actual coordinates with same hashval are equal */
                if (coordinates[i] != (np ->coordinates) [i])
                    found = 0;
            if (found)
                return np;
            else
                nplast = np;
            
        }
        (nplast -> next) = freevertex;
    }
    usage++;
    if (usage == HASHSIZE)
    {
        printf("Hashtable full!\n");
        return NULL;
    }
    for (i=0; i<d; i++)
        (freevertex->coordinates[i]) = coordinates[i];
    return freevertex++;
    
}

/* Runs a random walk from startpos until the tree is hit using install. Assume that the sink is in the tree.
 The origin is with coordinates all 128 and the boundary point has coordinates with one of the entries is 0 or 256 */

/* Initializes toppling counts and probability */
void initialize()
{
    long int i;
    for (i=0; i < 64; i++)
        prob[i] = 0;
    stack_pointer = 0;
}

#define checksink(i)                                \
if (!(newpos[i]&0b11111111)) /* newpos[i] = 0 or 256 */  \
{						    \
reached = 1; /* random walk reached the sink */ \
if (! (pos->treepos = install(newpos)))        \
return -1;			     \
pos->distance = 1; /* pos is on the boundary, 1 step to the sink */	\
pos->intree = 1;							\
\
nextpos = pos->treepos;						\
nextpos->distance = 0;						\
nextpos->intree = 1;						\
}

int run(unsigned int *startpos)
{
    struct node *pos, *nextpos;
    unsigned int newpos[d];
    register unsigned char stepa, stepb, stepc, stepd, stepe, stepf;
    unsigned long int steps;
    unsigned int i;
    unsigned long int dist;
    char reached;
    
    if (!(pos = install(startpos)))
        return -1;
    if (pos->intree)
        return 0;
    
    reached = 0;/* either reach the sink or the tree */
    for (i = 0; i < d; i++)
        newpos[i] = pos->coordinates[i];
    
    while ( !reached )
    {
        stepa = (genrand_int32() >> 26);
        stepf = (stepa & 0x1);
        stepa >>= 1;
        stepe = (stepa & 0x1);
        stepa >>= 1;
        stepd = (stepa & 0x1);
        stepa >>= 1;
        stepc = (stepa & 0x1);
        stepa >>= 1;
        stepb = (stepa & 0x1);
        stepa >>= 1;
        
        if (stepf)
        {
            if (stepe)
            {
                if (stepd)
                {
                    if (stepc)
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[0] = pos->coordinates[0] + 1;
                                pos->treeneighbour = 0;
                                checksink(0)
                            }
                            else
                            {
                                newpos[0] = pos->coordinates[0] - 1;
                                pos->treeneighbour = 32;
                                checksink(0)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[1] = pos->coordinates[1] + 1;
                                pos->treeneighbour = 1;
                                checksink(1)
                            }
                            else
                            {
                                newpos[1] = pos->coordinates[1] - 1;
                                pos->treeneighbour = 33;
                                checksink(1)
                            }
                        }
                    }
                    else /* stepc = 0 */
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[2] = pos->coordinates[2] + 1;
                                pos->treeneighbour = 2;
                                checksink(2)
                            }
                            else
                            {
                                newpos[2] = pos->coordinates[2] - 1;
                                pos->treeneighbour = 34;
                                checksink(2)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[3] = pos->coordinates[3] + 1;
                                pos->treeneighbour = 3;
                                checksink(3)
                            }
                            else
                            {
                                newpos[3] = pos->coordinates[3] - 1;
                                pos->treeneighbour = 35;
                                checksink(3)
                            }
                        }
                    }
                }
                else /* stepd = 0 */
                {
                    if (stepc)
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[4] = pos->coordinates[4] + 1;
                                pos->treeneighbour = 4;
                                checksink(4)
                            }
                            else
                            {
                                newpos[4] = pos->coordinates[4] - 1;
                                pos->treeneighbour = 36;
                                checksink(4)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[5] = pos->coordinates[5] + 1;
                                pos->treeneighbour = 5;
                                checksink(5)
                            }
                            else
                            {
                                newpos[5] = pos->coordinates[5] - 1;
                                pos->treeneighbour = 37;
                                checksink(5)
                            }
                        }
                    }
                    else
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[6] = pos->coordinates[6] + 1;
                                pos->treeneighbour = 6;
                                checksink(6)
                            }
                            else
                            {
                                newpos[6] = pos->coordinates[6] - 1;
                                pos->treeneighbour = 38;
                                checksink(6)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[7] = pos->coordinates[7] + 1;
                                pos->treeneighbour = 7;
                                checksink(7)
                            }
                            else
                            {
                                newpos[7] = pos->coordinates[7] - 1;
                                pos->treeneighbour = 39;
                                checksink(7)
                            }
                        }
                    }
                }
            }
            else
            {
                if (stepd)
                {
                    if (stepc)
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[8] = pos->coordinates[8] + 1;
                                pos->treeneighbour = 8;
                                checksink(8)
                            }
                            else
                            {
                                newpos[8] = pos->coordinates[8] - 1;
                                pos->treeneighbour = 40;
                                checksink(8)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[9] = pos->coordinates[9] + 1;
                                pos->treeneighbour = 9;
                                checksink(9)
                            }
                            else
                            {
                                newpos[9] = pos->coordinates[9] - 1;
                                pos->treeneighbour = 41;
                                checksink(9)
                            }
                        }
                    }
                    else
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[10] = pos->coordinates[10] + 1;
                                pos->treeneighbour = 10;
                                checksink(10)
                            }
                            else
                            {
                                newpos[10] = pos->coordinates[10] - 1;
                                pos->treeneighbour = 42;
                                checksink(10)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[11] = pos->coordinates[11] + 1;
                                pos->treeneighbour = 11;
                                checksink(11)
                            }
                            else
                            {
                                newpos[11] = pos->coordinates[11] - 1;
                                pos->treeneighbour = 43;
                                checksink(11)
                            }
                        }
                    }
                }
                else
                {
                    if (stepc)
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[12] = pos->coordinates[12] + 1;
                                pos->treeneighbour = 12;
                                checksink(12)
                            }
                            else
                            {
                                newpos[12] = pos->coordinates[12] - 1;
                                pos->treeneighbour = 44;
                                checksink(12)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[13] = pos->coordinates[13] + 1;
                                pos->treeneighbour = 13;
                                checksink(13)
                            }
                            else
                            {
                                newpos[13] = pos->coordinates[13] - 1;
                                pos->treeneighbour = 45;
                                checksink(13)
                            }
                        }
                    }
                    else
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[14] = pos->coordinates[14] + 1;
                                pos->treeneighbour = 14;
                                checksink(14)
                            }
                            else
                            {
                                newpos[14] = pos->coordinates[14] - 1;
                                pos->treeneighbour = 46;
                                checksink(14)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[15] = pos->coordinates[15] + 1;
                                pos->treeneighbour = 15;
                                checksink(15)
                            }
                            else
                            {
                                newpos[15] = pos->coordinates[15] - 1;
                                pos->treeneighbour = 47;
                                checksink(15)
                            }
                        }
                    }
                }
            }
        }
        else
        {
            if (stepe)
            {
                if (stepd)
                {
                    if (stepc)
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[16] = pos->coordinates[16] + 1;
                                pos->treeneighbour = 16;
                                checksink(16)
                            }
                            else
                            {
                                newpos[16] = pos->coordinates[16] - 1;
                                pos->treeneighbour = 48;
                                checksink(16)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[17] = pos->coordinates[17] + 1;
                                pos->treeneighbour = 17;
                                checksink(17)
                            }
                            else
                            {
                                newpos[17] = pos->coordinates[17] - 1;
                                pos->treeneighbour = 49;
                                checksink(17)
                            }
                        }
                    }
                    else
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[18] = pos->coordinates[18] + 1;
                                pos->treeneighbour = 18;
                                checksink(18)
                            }
                            else
                            {
                                newpos[18] = pos->coordinates[18] - 1;
                                pos->treeneighbour = 50;
                                checksink(18)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[19] = pos->coordinates[19] + 1;
                                pos->treeneighbour = 19;
                                checksink(19)
                            }
                            else
                            {
                                newpos[19] = pos->coordinates[19] - 1;
                                pos->treeneighbour = 51;
                                checksink(19)
                            }
                        }
                    }
                }
                else
                {
                    if (stepc)
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[20] = pos->coordinates[20] + 1;
                                pos->treeneighbour = 20;
                                checksink(20)
                            }
                            else
                            {
                                newpos[20] = pos->coordinates[20] - 1;
                                pos->treeneighbour = 52;
                                checksink(20)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[21] = pos->coordinates[21] + 1;
                                pos->treeneighbour = 21;
                                checksink(21)
                            }
                            else
                            {
                                newpos[21] = pos->coordinates[21] - 1;
                                pos->treeneighbour = 53;
                                checksink(21)
                            }
                        }
                    }
                    else
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[22] = pos->coordinates[22] + 1;
                                pos->treeneighbour = 22;
                                checksink(22)
                            }
                            else
                            {
                                newpos[22] = pos->coordinates[22] - 1;
                                pos->treeneighbour = 54;
                                checksink(22)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[23] = pos->coordinates[23] + 1;
                                pos->treeneighbour = 23;
                                checksink(23)
                            }
                            else
                            {
                                newpos[23] = pos->coordinates[23] - 1;
                                pos->treeneighbour = 55;
                                checksink(23)
                            }
                        }
                    }
                }
            }
            else
            {
                if (stepd)
                {
                    if (stepc)
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[24] = pos->coordinates[24] + 1;
                                pos->treeneighbour = 24;
                                checksink(24)
                            }
                            else
                            {
                                newpos[24] = pos->coordinates[24] - 1;
                                pos->treeneighbour = 56;
                                checksink(24)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[25] = pos->coordinates[25] + 1;
                                pos->treeneighbour = 25;
                                checksink(25)
                            }
                            else
                            {
                                newpos[25] = pos->coordinates[25] - 1;
                                pos->treeneighbour = 57;
                                checksink(25)
                            }
                        }
                    }
                    else
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[26] = pos->coordinates[26] + 1;
                                pos->treeneighbour = 26;
                                checksink(26)
                            }
                            else
                            {
                                newpos[26] = pos->coordinates[26] - 1;
                                pos->treeneighbour = 58;
                                checksink(26)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[27] = pos->coordinates[27] + 1;
                                pos->treeneighbour = 27;
                                checksink(27)
                            }
                            else
                            {
                                newpos[27] = pos->coordinates[27] - 1;
                                pos->treeneighbour = 59;
                                checksink(27)
                            }
                        }
                    }
                }
                else
                {
                    if (stepc)
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[28] = pos->coordinates[28] + 1;
                                pos->treeneighbour = 28;
                                checksink(28)
                            }
                            else
                            {
                                newpos[28] = pos->coordinates[28] - 1;
                                pos->treeneighbour = 60;
                                checksink(28)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[29] = pos->coordinates[29] + 1;
                                pos->treeneighbour = 29;
                                checksink(29)
                            }
                            else
                            {
                                newpos[29] = pos->coordinates[29] - 1;
                                pos->treeneighbour = 61;
                                checksink(29)
                            }
                        }
                    }
                    else
                    {
                        if (stepb)
                        {
                            if (stepa)
                            {
                                newpos[30] = pos->coordinates[30] + 1;
                                pos->treeneighbour = 30;
                                checksink(30)
                            }
                            else
                            {
                                newpos[30] = pos->coordinates[30] - 1;
                                pos->treeneighbour = 62;
                                checksink(30)
                            }
                        }
                        else
                        {
                            if (stepa)
                            {
                                newpos[31] = pos->coordinates[31] + 1;
                                pos->treeneighbour = 31;
                                checksink(31)
                            }
                            else
                            {
                                newpos[31] = pos->coordinates[31] - 1;
                                pos->treeneighbour = 63;
                                checksink(31)
                            }
                        }
                    }
                }
            }
        }
        
        if (!(pos->treepos = install(newpos)))
            return -1;
        pos = pos->treepos;/* overwrite any old step taken */
        if (pos->intree)
            reached = 1; /* random walk reached the tree */
    }
    steps = 0;
    reached = 0;
    pos = install(startpos);
    while (!reached)
    {
        pos = pos->treepos;
        steps++;
        if (pos->intree)
            reached = 1;
    }
    
    dist = pos->distance + steps;/* Copy distance of the vertex where tree was hit and add the number of steps */
    
    /* Compute distances for each point along the path */
    pos = install(startpos);
    while (steps)
    {
        pos->intree = 1; /* mark the vertex as in the tree */
        pos->sand = -1; /* initialize sand height to be -1 */
        pos->distance = dist;
        pos = pos->treepos;
        dist--;
        steps--;
    }
    return 0;
}

/* Form the sandheight for the coordinates point */
int sandheight(unsigned int *coordinates)
{
    struct node *pos;
    int i;
    unsigned int neighbour[d];
    unsigned long int disttoroot;
    unsigned long int dist[64];
    unsigned char height;
    unsigned char found;
    
    if (run(coordinates))
        return -1;
    pos = install(coordinates);
    disttoroot = pos->distance;
   
    height = 0; /* the sand height of sink */
    if (disttoroot) /* skip the root */
    {
        /* Using run to compute random walks of all its neighbours first */
        for (i = 0; i < d; i++)
            neighbour[i] = coordinates[i];
        for (i = 0; i < d; i++)
        {
            if (i)
                neighbour[i-1]--;
           
            neighbour[i]++;
            if (run(neighbour))
                return -1;
            
            dist[i] = install(neighbour)->distance;
        }
        neighbour[d-1]--;
        for (i = 0; i < d; i++)
        {
            if (i)
                neighbour[i-1]++;
    
            neighbour[i]--;
            if (run(neighbour))
                return -1;
            
            dist[i+d] = install(neighbour)->distance;
        }
        height = 63;
        found = 0;
        for (i = 63; i >= 0; i--)
        {
            if (dist[i]+1 < disttoroot)
                height -= 1;
            if (dist[i]+1 == disttoroot) /* burnt in the previous step */
            {
                if (pos->treeneighbour == i) /* the chosen one in the ordering */
                    found = 1;
                if (found == 0)
                    height -= 1;
            }
        }
    }
    pos->sand = height;
    return 0;
}

int sandprob()
{
    struct node *posorigin;
    unsigned int i;
    unsigned int origin[d];
    
    for (i = 0; i < d; i++)
        origin[i] = 128;
    posorigin = install(origin);
    if (sandheight(posorigin->coordinates))
        return -1;
    prob[posorigin->sand]++;
    
    return 0;
}

/* Carries out n independent sandheight */
unsigned long int prob_indept(unsigned long int n)
{
    register unsigned long int i, j, k;
    unsigned long int x;
    unsigned int point[d];
    
    k = n;
    for (i = 0; i < n; i++)
    {
        freevertex = vertexdata;
        for (j = 0; j < HASHSIZE; j++)
        {
            hashtab[j] = NULL;
            vertexdata[j].sand = -1;
        }
        for (j = 0; j < usage; j++)
        {
            vertexdata[j].next = NULL;
            vertexdata[j].intree = 0;
        }
        usage = 0;
        
        if (sandprob())
            k--;
    }
    return k;
}

int main(int argc, char *argv[])
{
    int color;
    int numprocs, myid, j, jump;
    MPI_Comm workers;
    MPI_Status status;
    unsigned long int n, num_batch, batch_remain_sent, batch_remain_received;
    unsigned long int i, k, total_k, x;
    char *tail;
    
    MPI_Init(&argc, &argv);
    world = MPI_COMM_WORLD;
    MPI_Comm_size(world, &numprocs);
    MPI_Comm_rank(world, &myid);
    
    if (myid == MASTER)
    {
        printf("Number of arguments: %d \n", argc);
        n = strtoul(argv[argc-1], &tail, 10);
        num_batch = n;
        batch_remain_sent = num_batch;
        batch_remain_received = num_batch;
        printf("Number of samples requested: %lu\n", n);
        jump = atoi(argv[argc-2]);
        printf("Jump requested: %d\n", jump);
    }
    
    MPI_Bcast(&num_batch, 1, MPI_UNSIGNED_LONG, MASTER, world);
    MPI_Bcast(&jump, 1, MPI_INT, MASTER, world);
    color = (myid > num_batch);
    MPI_Comm_split(world, color, 0, &workers);
    
    init_genrand(1); /* Set seed = 1 */
    pf = (unsigned long *)calloc(P_SIZE, sizeof(unsigned long));
    
    if (myid == MASTER)
    {
        FILE *fin;
        char c;
        
        /* read the file clist.txt, and set the coefficient */
        if ((fin = fopen("clist_mt19937.txt", "r")) == NULL){
            printf("File read error.\n");
            exit(1);
        }
        for (j = MEXP - 1; j>-1; j--){
            c = fgetc(fin);
            if (c == '1')
                set_coef(pf, j, 1);
        }
    }
    
    MPI_Bcast(pf, P_SIZE, MPI_UNSIGNED_LONG, MASTER, world);
    if (color == 0 && myid > 0)
    {
        jump_ahead(jump + myid - 1);  /* Jump ahead by (myid-1)*2^100 */
    }
    
    if (color == 0)
    {
        initialize();
        k = 0;
    }
    /* Master node initializes its arrays for probability */
    if (myid == MASTER)
    {
        for (i=0; i < 64; i++)
            probability[i] = 0;
    }
    /* Main loop for Master Node */
    if (myid == MASTER)
    {
        sample_request = 1;
        for (j = 1; (j < numprocs && j <= num_batch); j++)
        {
            MPI_Send(&sample_request, 1, MPI_INT, j, SAMPLE_REQUEST, world);
            batch_remain_sent--;
        }
        while (batch_remain_received > 0)
        {
            MPI_Recv(&sample_request, 1, MPI_INT, MPI_ANY_SOURCE, SAMPLE_REPLY, world, &status);
            batch_remain_received--;
            if (batch_remain_sent > 0)
            {
                MPI_Send(&sample_request, 1, MPI_INT, status.MPI_SOURCE, SAMPLE_REQUEST, world);
                batch_remain_sent--;
            }
        }
        /* Send worker nodes a request to terminate computation */
        sample_request = 0;
        for (j = 1; (j < numprocs && j <= num_batch); j++)
            MPI_Send(&sample_request, 1, MPI_INT, j, SAMPLE_REQUEST, world);
    }
    /* Main loop for worker nodes */
    if (myid > 0 && color == 0)
    {
        MPI_Recv(&sample_request, 1, MPI_INT, MASTER, SAMPLE_REQUEST, world, &status);
        while (sample_request != 0)
        {
            k += prob_indept(1);
            MPI_Send(&sample_request, 1, MPI_INT, MASTER, SAMPLE_REPLY, world);
            MPI_Recv(&sample_request, 1, MPI_INT, MASTER, SAMPLE_REQUEST, world, &status);
        }
    }
    MPI_Reduce(&prob, &total_prob, 64, MPI_UNSIGNED_LONG, MPI_SUM, MASTER, workers);
    MPI_Reduce(&k, &total_k, 1, MPI_UNSIGNED_LONG, MPI_SUM, MASTER, workers);
    MPI_Comm_free(&workers);
    
    if (myid == MASTER)
    {
        printf("Number of samples generated = %lu\n",total_k);
        for (i = 0; i < 64; i++)
            probability[i] = ((float) total_prob[i])/total_k;
        probability_file = fopen("probability32d-128.txt", "w");
        for (i = 0; i < 64; i++)
        {
            fprintf(probability_file, "%f\n", probability[i]);
        }
        fclose(probability_file);
    }
    MPI_Finalize();
}

