/* sandpile8192sink.c : Parallel version using MPI
 * Simulated independent stationary configurations
 * in the 2D Abelian sandpile model with Dirichlet bounday conditions
 * and checked which sites topple when adding a particle at the origin 0.
 * Specialized to L = 8192, giving a box of size 2L x 2L
 * sink is all the boundaries
 * Created by Antal A. Jarai and Minwei Sun
 */

/* To compile:  mpicc -o sandpile8192sink sandpile8192sink.c mt19937ar-aj.c jump_mt19937-aj.c
 * To run:  mpirun ./sandpile8192sink <number of samples>
 * mt19937ar.c  source code of Mersenne Twister from author's website:
 * http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/MT2002/emt19937ar.html 
 */

#define L (1<<13)  /* Linear radius of box for simulation */
#define ROWSIZE 2*L /* 2*L */
#define ARRAYSIZE 2*L*2*L /* (2*L * 2*L) */
#define BUFFERSIZE ARRAYSIZE
unsigned long int treeneighbour[ARRAYSIZE];
unsigned long int intree[ARRAYSIZE];
unsigned long int distance[ARRAYSIZE];

#include "mt19937ar-aj.h"   /* Header file for random-number generator */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mpi.h"
#define MASTER 0
#define SAMPLE_REQUEST 3
#define SAMPLE_REPLY 4


MPI_Comm world;
int sample_request;
int myid;
#define west(pos) ((pos & 0b1111111111111100000000000000) ? (pos - ROWSIZE) : (pos | 0b1111111111111100000000000000))
#define east(pos) (((pos & 0b1111111111111100000000000000) ^ 0b1111111111111100000000000000) ? (pos + ROWSIZE) : (pos & 0b0000000000000011111111111111))
#define north(pos)  ((pos & 0b0000000000000011111111111111) ? (pos - 1) : (pos | 0b0000000000000011111111111111) )
#define south(pos) (((pos & 0b0000000000000011111111111111) ^ 0b0000000000000011111111111111) ? (pos + 1) : (pos & 0b1111111111111100000000000000))
#define sand intree
#define topple treeneighbour
#define stack distance

long int stack_pointer;  /* Stack pointer for topplings */
unsigned long int box[513][513];
unsigned long int total_box[513][513];
unsigned long int vertices[L];
unsigned long int total_vertices[L];
float portion[L];
unsigned long int prob[4] = { 0, 0, 0, 0};
unsigned long int total_prob[4] = { 0, 0, 0 , 0};
float probability[4];
unsigned long long int sdistinct;
unsigned long long int total_sdistinct;
unsigned long long int sall;
unsigned long long int total_sall;
unsigned long int numwaves[2000];
unsigned long int total_numwaves[2000];
float wave[2000];

unsigned long int seed;   /* Seed for random number generator */
unsigned long int nb[4];
FILE *box_file;
FILE *portion_file;
FILE *probability_file;
FILE *wave_file;
unsigned long *pf;
State s0;

/* Initialize the tables of neighbours, toppling site counts, probability and the number of waves */
void initialize()
{
    long int i, j;

    for (i=0; i < L; i++)
        vertices[i] = 0;
    for (i=0; i < 4; i++)
        prob[i] = 0;
    for (i=0; i <= 512; i++)
        for (j=0; j <= 512; j++)
            {
                box[i][j] = 0;
            }
    for (i=0; i < 2000; i++)
        numwaves[i] = 0;
    
    nb[0] = west(0);
    nb[1] = east(0);
    nb[2] = south(0);
    nb[3] = north(0);
    stack_pointer = 0;
    sdistinct = 0;
    sall = 0;
   
}

/* Runs a random walk from startpos until tree is hit. Assumes that at least the root is marked as in the tree */
void run(unsigned long int startpos)
{
    register unsigned long int pos;
    unsigned long int newpos;
    register unsigned char stepLR, stepdir;
    unsigned long int steps;
    unsigned long int dist;

    pos = startpos;

    while (intree[pos] == 0)
    {
        stepdir = (genrand_int32() >> 30);
        stepLR = (stepdir & 0x1);
		stepdir >>= 1;
        if (stepLR)
        {
            if (stepdir)
            {
                newpos = east(pos);
                treeneighbour[pos] = 1;
            }
                else
            {
                newpos = west(pos);
                treeneighbour[pos] = 0;
            }
        }
        else
        {
            if (stepdir)
            {
                newpos = north(pos);
                treeneighbour[pos] = 3;
            }
            else
            {
                newpos = south(pos);
                treeneighbour[pos] = 2;
            }
        }
        pos = newpos;
    }
   
    /* Compute distance to point where it hit */
    pos = startpos;
    steps = 0;
    while (intree[pos] == 0)
    {
        switch(treeneighbour[pos]) {
            case 0:
                newpos = west(pos);
                break;
            case 1:
                newpos = east(pos);
                break;
            case 2:
                newpos = south(pos);
                break;
            case 3:
                newpos = north(pos);
                break;
        }
        pos = newpos;
    
        steps++;           /* Count number of steps */
    }
    dist = distance[pos] + steps; /* Copy distance of the point where tree was hit and add the number of steps */
    /* Compute distances for each point along the path */
    pos = startpos;
    while (intree[pos] == 0)
    {
        intree[pos] = 1;  /* Mark the vertex as in the tree */
        distance[pos] = dist;  /* Record its distance */
        dist--;  /* Distance of next point on the path */
    
        switch(treeneighbour[pos]) {
            case 0:
                newpos = west(pos);
                break;
            case 1:
                newpos = east(pos);
                break;
            case 2:
                newpos = south(pos);
                break;
            case 3:
                newpos = north(pos);
                break;
        }
        pos = newpos;
    }
}

/* Computes the whole tree with distances */
void tree()
{
    register unsigned long int i;
    unsigned long int pos, sinkpos;

    for (i = 0; i < ARRAYSIZE; i++)
        intree[i] = 0;      /* Mark all vertices as not in the tree */
    for (i = 0; i < ROWSIZE; i++)
    {
        sinkpos = ROWSIZE*i + L;
        intree[sinkpos] = 1;  /* Mark the sink as being in the tree */
        distance[sinkpos] = 0;  /* Sink has distance 0 from the root */
        
        sinkpos = ROWSIZE*L + i;
        intree[sinkpos] = 1;
        distance[sinkpos] = 0;
    }
    run(0);
    
    for (pos = 1; pos < ARRAYSIZE; pos++)
        run(pos);
}

/* Compute the whole sandpile from the tree */ 
void sandconf()
{
    unsigned long int dist[4];
    register unsigned long int pos, sinkpos;
    unsigned long int disttoroot;
    unsigned char height;
    unsigned char found;
    int i;

    for (pos = 0; pos < ARRAYSIZE; pos++)
    {
        disttoroot = distance[pos];
        if (disttoroot) /* Skip the root */
        {
            disttoroot--;
            dist[0] = distance[west(pos)];
            dist[1] = distance[east(pos)];
            dist[2] = distance[south(pos)];
            dist[3] = distance[north(pos)];
            height = 3;
            found = 0;
            for (i = 3; i >= 0; i--)
            {
                
                if (dist[i] < disttoroot)
                    height--;
                if (dist[i] == disttoroot)
                {
                    if (treeneighbour[pos] == i)
                        found = 1;
                    if (found == 0)
                        height--;
                }
            }
            sand[pos] = height;
        }
    }
    for (i = 0; i < ROWSIZE; i++)
        {
            sinkpos = ROWSIZE*i + L;
            sand[sinkpos] = -1;
            sinkpos = ROWSIZE*L + i;
            sand[sinkpos] = -1;
        }
}

/* Pushes pointer to site onto the stack */
void push(unsigned long int site){
  if (stack_pointer < BUFFERSIZE)
    stack[stack_pointer++] = site;
  else 
    printf("Error: stack overflow\n");
}

/* Pop a pointer to site from the stack */
unsigned long int pop(void){
  if (stack_pointer > 0)
    return stack[--stack_pointer];
  else {
    printf("Error: cannot pop from empty stack\n");
    exit(-1);
  }
}

/* Compute avalanche using waves */
void avalanche1()
{
    register unsigned long int pos, sinkpos, temp;
    register unsigned long int i;
    unsigned long int n;
    unsigned long int posnb[4];
    unsigned long int nwaves;

    for (i = 0; i < ARRAYSIZE; i++)
        topple[i] = 0;
    nwaves = 0;
    sand[0]++;
    if (sand[0] == 4)
    {
        topple[0] = 1;
        sand[0]--;
        sdistinct++;
        while (sand[0] == 3)
        {
            sand[0] -= 4;
            nwaves++;
            sall++;
            for (n = 0; n < 4; n++)
            {
                if (++sand[nb[n]] == 4)
                    push(nb[n]);
            }
            while (stack_pointer != 0)
            {
                pos = pop();
                sall++;
                if (!topple[pos])
                {
                    sdistinct++;
                    topple[pos] = 1;
                }
                sand[pos] -= 4;
                if (sand[temp = west(pos)] == 3)
                {
                    push(temp);
                }
                if ((temp & (0x3fff << 14)) ^ (L << 14)) /* not sinkpos and initialized sand[sinkpos] = -1 */
                    sand[temp]++;
                if (sand[temp = east(pos)] == 3)
                {
                    push(temp);
                }
                if ((temp & (0x3fff << 14)) ^ (L << 14))
                    sand[temp]++;
                if (sand[temp = north(pos)] == 3)
                {
                    push(temp);
                }
                if ((temp & 0x3fff) ^ L)
                    sand[temp]++;
                if (sand[temp = south(pos)] == 3)
                {
                    push(temp);
                }
                if ((temp & 0x3fff) ^ L)
                    sand[temp]++;
            }
        }
        sand[0]++;
    }
    if (nwaves < 2000)
        numwaves[nwaves]++;
}


/* Carries out n independent avalanches */ 
void vertex_indept(unsigned long int n)
{
    register unsigned long int i;
    long int x, y;

    for (i = 0; i < n; i++)
    {
        tree();
        sandconf();
        prob[sand[0]]++;
        avalanche1();
      
        for (x = 0; x < L; x++)
            if (topple[x])
                vertices[x]++;
        for (x = -256; x <= 256; x++)
            for (y = -256; y <= 256; y++)
                if (topple[(x & 0x3fff) + (y & 0x3fff)*ROWSIZE])
                        box[x+256][y+256]++;
    }
}
/* Main code - Read required number of samples from command line,
 * run the simulation and write results to file */
void main(int argc, char *argv[])
{
    int color;
    int numprocs, j;
    MPI_Comm workers;
    MPI_Status status;
    unsigned long int n, num_batch, batch_remain_sent, batch_remain_received;
    unsigned long int i, jump;
    int x, y;
    char *tail;
    
    MPI_Init(&argc, &argv);
    world = MPI_COMM_WORLD;
    MPI_Comm_size(world, &numprocs);
    MPI_Comm_rank(world, &myid);
    /* Master node reads number of samples from command line */
    if (myid == MASTER)
    {
        printf("Number of arguments: %d \n", argc);
        n = strtoul(argv[argc-1], &tail, 10);
        num_batch = n;
        batch_remain_sent = num_batch;
        batch_remain_received = num_batch;
        /* Code assumes n is a multiple of 100. */
        printf("Number of samples requested: %lu\n", n);
        jump = atoi(argv[argc-2]);
        printf("Jump requested: %d\n", jump);
    }
    /* Master node broadcasts number of batches required */
    MPI_Bcast(&num_batch, 1, MPI_UNSIGNED_LONG, MASTER, world);
    MPI_Bcast(&jump, 1, MPI_UNSIGNED_LONG, MASTER, world);
    color = (myid > num_batch);
    MPI_Comm_split(world, color, 0, &workers);
    /* Random number server initializes seed */
    init_genrand(1234567); /* Set seed = 1 */
    pf = (unsigned long *)calloc(P_SIZE, sizeof(unsigned long));
    /* Initialize jump ahead */
    if (myid == MASTER)
    {
        FILE *fin;
        char c;
        
        /* read the file clist.txt, and set the coefficient */
        if ((fin = fopen("clist_mt19937.txt", "r")) == NULL){
            printf("File read error.\n");
            exit(1);
        }
        for (j = MEXP - 1; j>-1; j--){
            c = fgetc(fin);
            if (c == '1')
                set_coef(pf, j, 1);
        }
    }
    MPI_Bcast(pf, P_SIZE, MPI_UNSIGNED_LONG, MASTER, world);
    if (color == 0 && myid > 0)
    {
        jump_ahead(jump + myid - 1);  /* Jump ahead by (myid-1)*2^100 */
    }
    /* Worker nodes and Master node initialize their arrays */
    if (color == 0)
        initialize();
    /* Master node initializes its arrays for  portion  and  probability */
    if (myid == MASTER)
    {
        for (i=0; i < L; i++)
            portion[i] = 0;
        for (i=0; i < 4; i++)
            probability[i] = 0;
    }
    /* Main loop for Master Node */
    if (myid == MASTER)
    {
        sample_request = 1;
        for (j = 1; (j < numprocs && j <= num_batch); j++)
        {
            MPI_Send(&sample_request, 1, MPI_INT, j, SAMPLE_REQUEST, world);
            batch_remain_sent--;
        }
        while (batch_remain_received > 0)
        {
            MPI_Recv(&sample_request, 1, MPI_INT, MPI_ANY_SOURCE, SAMPLE_REPLY, world, &status);
            batch_remain_received--;
            if (batch_remain_sent > 0)
            {
                MPI_Send(&sample_request, 1, MPI_INT, status.MPI_SOURCE, SAMPLE_REQUEST, world);
                batch_remain_sent--;
            }
        }
        /* Send worker nodes a request to terminate computation */
        sample_request = 0;
        for (j = 1; (j < numprocs && j <= num_batch); j++)
            MPI_Send(&sample_request, 1, MPI_INT, j, SAMPLE_REQUEST, world);
        /* Send server node a request to terminate computation */
    }
    /* Main loop for worker nodes */
    if (myid > 0 && color == 0)
    {
        MPI_Recv(&sample_request, 1, MPI_INT, MASTER, SAMPLE_REQUEST, world, &status);
        while (sample_request != 0)
        {
            vertex_indept(1);
            MPI_Send(&sample_request, 1, MPI_INT, MASTER, SAMPLE_REPLY, world);
            MPI_Recv(&sample_request, 1, MPI_INT, MASTER, SAMPLE_REQUEST, world, &status);
        }
    }
    MPI_Reduce(&box, &total_box, 513*513, MPI_UNSIGNED_LONG, MPI_SUM, MASTER, workers);
    MPI_Reduce(&vertices, &total_vertices, L, MPI_UNSIGNED_LONG, MPI_SUM, MASTER, workers);
    MPI_Reduce(&prob, &total_prob, 4, MPI_UNSIGNED_LONG, MPI_SUM, MASTER, workers);
    MPI_Reduce(&sdistinct, &total_sdistinct, 1, MPI_UNSIGNED_LONG_LONG, MPI_SUM, MASTER, workers);
    MPI_Reduce(&sall, &total_sall, 1, MPI_UNSIGNED_LONG_LONG, MPI_SUM, MASTER, workers);
    MPI_Reduce(&numwaves, &total_numwaves, 2000, MPI_UNSIGNED_LONG, MPI_SUM, MASTER, workers);
    MPI_Comm_free(&workers);
    /* Master node computes  portion  and  probability  and writes them to file */
    if (myid == MASTER)
    {
        printf("Average number of topplings <s> = %f\n", ((double) total_sall) / n);
        printf("Average number of distinct topplings <sdistinct> = %f\n", ((double) total_sdistinct) / n);
        
        for (i = 0; i < 4; i++)
            probability[i] = ((float) total_prob[i])/n;
        for (x = 0; x < L; x++)
            portion[x] = ((float) total_vertices[x])/n;
        for (i = 0; i < 2000; i++)
            wave[i] = ((float) total_numwaves[i])/n;
        
        
        portion_file = fopen("portion8192sink.txt", "w");
        if (portion_file == NULL)
            printf("Could not open portion file");
        probability_file = fopen("probability8192sink.txt", "w");
        if (probability_file == NULL)
            printf("Could not open probability file");
        box_file = fopen("box8192sink.txt", "w");
        if (box_file == NULL)
            printf("Could not open box file");
        wave_file = fopen("wave8192sink.txt", "w");
        if (wave_file == NULL)
            printf("Could not open wave file");
        
        for (i = 0; i < L; i++)
            fprintf(portion_file, "%f\n", portion[i]);
        fclose(portion_file);
        for (i = 0; i < 4; i++)
            fprintf(probability_file, "%f\n", probability[i]);
        fclose(probability_file);
        for (x = -256; x <= 256; x++)
            for (y = -256; y <= 256; y++)
                fprintf(box_file, "%d %d %lu\n", x, y, total_box[x+256][y+256]);
        fclose(box_file);
        for (i = 0; i < 2000; i++)
            fprintf(wave_file, "%f\n", wave[i]);
        fclose(wave_file);
    }
    MPI_Finalize();
}
