/* sandpile5d-32m32.c
 *  Abelian sandpile model in 5d with Dirichlet boundary conditions using hashing
 * Created by Antal A. Jarai and Minwei Sun.
 */

/* To compile:  mpicc -o sandpile5d-32m32 sandpile5d-32m32.c mt19937ar-aj.c jump_mt19937-aj.c
 * To run:  mpirun ./sandpile5d-32m32 <number of samples>
 * mt19937ar.c  source code of Mersenne Twister from author's website:
 * http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/MT2002/emt19937ar.html
 */

#include "mt19937ar-aj.h"   /* Header file for Random-Number Generator */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <mpi.h>

#define MASTER 0
#define SAMPLE_REQUEST 3
#define SAMPLE_REPLY 4

#define d 5
#define L 32
#define m 32 /* (unsigned int)pow(L, 6/d) */
#define mpower 5
#define ROWSIZE 64
#define HASHSIZE 0x2000000 /* m^5=2^20 */
#define HASHSIZEmONE 0x1ffffff /* HASHSIZE - 1 */

MPI_Comm world;
int sample_request;

unsigned long int vertices[L];
unsigned long int total_vertices[L];
float portion[L];
unsigned long int prob[10] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
unsigned long int total_prob[10] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
float probability[10];
unsigned long int seed;   /* Seed for random number generator */
unsigned long int nb[10];
FILE *portion_file;
FILE *probability_file;
unsigned long *pf;
State s0;

static struct node *stack[HASHSIZE];
unsigned long int stack_pointer;
struct node
{
    struct node *next; /* next entry in the chain */
    struct node *treepos;
    unsigned int coordinates[d];
    unsigned char treeneighbour;
    unsigned char topple;
    unsigned char sand;
    unsigned char intree;
    unsigned long int distance;
    
};
static struct node *hashtab[HASHSIZE]; /* pointer table */
static struct node vertexdata[HASHSIZE]; /* actual node, not pointer */
struct node *freevertex = vertexdata; /* pointer to the first entry of vertexdata */
unsigned long int usage = 0;


/* hash: form hash value for coordinates point */
unsigned long int hash(unsigned int *coordinates)
{
    unsigned long int hashval;
    unsigned int i;
    
    hashval = 0;
    for (i = 0; i < d; i++)
    {
        hashval <<= mpower; /* mpower is 5 & binary operation *32 */
        hashval += coordinates[i];
    }

    return (hashval & HASHSIZEmONE); /* moduleo HASHSIZE */
}

/* install returns pointer to the vertex, it comes back either address where we can put the new vertex or pointer to the vertex which has been visited before */

struct node *install(unsigned int *coordinates)
{
    struct node *np, *nplast;
    unsigned long int hashval;
    char i, found;
    
    hashval = hash(coordinates);
    np = hashtab[hashval];
    if (np == NULL) /* hashval has not been used before */
    {
        hashtab[hashval] = freevertex;/* pointer to the first free vertex */
    }
    else
    {
        
        for (nplast = np ; np != NULL ; np = (np -> next) )
        {
            found = 1; /* 1 is ture, 0 is false */
            for ( i = 0 ; i < d; i++) /* check whether the actual coordinates with same hashval are equal */
                if (coordinates[i] != (np ->coordinates) [i])
                    found = 0;
            if (found)
                return np;
            else
                nplast = np;
            
        }
        (nplast -> next) = freevertex;
    }
    usage++;
    if (usage == HASHSIZE)
    {
        return NULL;
    }
    for (i=0; i<d; i++)
        (freevertex->coordinates[i]) = coordinates[i];
    return freevertex++;
    
}

/* Runs a random walk from startpos until the tree is hit using install. Assume that the sink is in the tree.
 The origin is with coordinates all 32 and the boundary point has coordinates with one of the entries is 0 or 64 */

/* Initializes toppling counts and probability */
void initialize()
{
    long int i;
    for (i=0; i <= L; i++)
        vertices[i] = 0;
    for (i=0; i < 10; i++)
        prob[i] = 0;
    stack_pointer = 0;
}

#define checksink(i)                                \
if (!(newpos[i]&0b111111)) /* newpos[i] = 0 or 256 */  \
  {                                                   \
  reached = 1; /* random walk reached the sink */   \
  if (! (pos->treepos = install(newpos)))	    \
    return -1;							     \
  pos->distance = 1; /* pos is on the boundary, 1 step to the sink */	\
  pos->intree = 1;							\
									\
  nextpos = pos->treepos;						\
  nextpos->distance = 0;						\
  nextpos->intree = 1;							\
 }

int run(unsigned int *startpos)
{
  struct node *pos, *initpos, *nextpos;
    unsigned int newpos[d];
    register unsigned char stepa, stepb, stepc, stepd; /* stepa indicates up or down, stepb,stepc,stepd indicates which entry */
    unsigned long int steps;
    unsigned int i;
    unsigned long int dist;
    char reached;
    
    if (!(initpos = install(startpos)))
        return -1;
    if ( (pos = initpos) ->intree)
        return 0;
    
    for (i = 0; i < d; i++)
        if (!(pos->coordinates[i] & 0b111111)) /* it is the sink */
        {
            pos->distance = 0;
            pos->intree = 1;
            return 0;
        }
    
    reached = 0;/* either reach the sink or the tree */
    for (i = 0; i < d; i++)
        newpos[i] = pos->coordinates[i];
    
    while ( !reached )
    {
        stepa = (genrand_int32() >> 28);
        stepd = (stepa & 0x1);
        stepa >>= 1;
        stepc = (stepa & 0x1);
        stepa >>= 1;
        stepb = (stepa & 0x1);
        stepa >>= 1;
        /* b is the first bit, c is the second one, d is the third one and a is the last one. The first three bits indicate which entry of the coordinates and the last bit indicates up or down. */
        
        while ( (stepb == 1) && !((stepc == 0) && (stepd == 0))) /* discard six random numbers */
        {
            stepa= (genrand_int32() >> 28);
            stepd = (stepa & 0x1);
            stepa >>= 1;
            stepc = (stepa & 0x1);
            stepa >>= 1;
            stepb = (stepa & 0x1);
            stepa >>= 1;
        }
        
        if (stepb)
        {
            if (stepa)
            {
                newpos[0] = pos->coordinates[0] + 1;
                pos->treeneighbour = 0;
                checksink(0)
            }
            else
            {
                newpos[0] = pos->coordinates[0] - 1;
                pos->treeneighbour = 5;
                checksink(0)
            }
        }
        else
        {
            if (stepc)
            {
                if (stepd)
                {
                    if (stepa)
                    {
                        newpos[1] = pos->coordinates[1] + 1;
                        pos->treeneighbour = 1;
                        checksink(1)
                    }
                    else
                    {
                        newpos[1] = pos->coordinates[1] - 1;
                        pos->treeneighbour = 6;
                        checksink(1)
                    }
                }
                else
                {
                    if (stepa)
                    {
                        newpos[2] = pos->coordinates[2] + 1;
                        pos->treeneighbour = 2;
                        checksink(2)
                    }
                    else
                    {
                        newpos[2] = pos->coordinates[2] - 1;
                        pos->treeneighbour = 7;
                        checksink(2)
                    }
                }
            }
            else
            {
                if (stepd)
                {
                    if (stepa)
                    {
                        newpos[3] = pos->coordinates[3] + 1;
                        pos->treeneighbour = 3;
                        checksink(3)
                    }
                    else
                    {
                        newpos[3] = pos->coordinates[3] - 1;
                        pos->treeneighbour = 8;
                        checksink(3)
                    }
                }
                else
                {
                    if (stepa)
                    {
                        newpos[4] = pos->coordinates[4] + 1;
                        pos->treeneighbour = 4;
                        checksink(4)
                    }
                    else
                    {
                        newpos[4] = pos->coordinates[4] - 1;
                        pos->treeneighbour = 9;
                        checksink(4)
                    }
                }
            }
        }
        if (!(pos->treepos = install(newpos)))
            return -1;
        pos = pos->treepos;/* overwrite any old step taken */
        if (pos->intree)
            reached = 1; /* random walk reached the tree */
    }
    steps = 0;
    reached = 0;
    pos = install(startpos);
    while (!reached)
    {
        pos = pos->treepos;
        steps++;
        if (pos->intree)
            reached = 1;
    }
    
    dist = pos->distance + steps;/* Copy distance of the vertex where tree was hit and add the number of steps */
    /* Compute distances for each point along the path */
    pos = initpos;
    while (steps)
    {
        pos->intree = 1; /* mark the vertex as in the tree */
        pos->sand = -1; /* initialize sand height to be -1 */
        pos->distance = dist;
        pos = pos->treepos;
        dist--;
        steps--;
    }
    return 0;
}


/* Form the sandheight for the coordinates point */
int sandheight(unsigned int *coordinates)
{
    struct node *pos;
    int i;
    unsigned int neighbour[d];
    unsigned long int disttoroot;
    unsigned long int dist[10];
    unsigned char height;
    unsigned char found;
    
    if (run(coordinates))
        return -1;
    pos = install(coordinates);
    disttoroot = pos->distance;
    height = 0; /* the sand height of sink */
    
    if (disttoroot) /* skip the root */
    {
        /* Using run to compute random walks of all its neighbours first */
        for (i = 0; i < d; i++)
            neighbour[i] = coordinates[i];
        for (i = 0; i < d; i++)
        {
            if (i)
                neighbour[i-1]--;
            neighbour[i]++;
            if (run(neighbour))
                return -1;
    
            dist[i] = install(neighbour)->distance;
        }
        neighbour[d-1]--;
        for (i = 0; i < d; i++)
        {
            if (i)
                neighbour[i-1]++;
            neighbour[i]--;
            if (run(neighbour))
                return -1;
    
            dist[i+d] = install(neighbour)->distance;
        }
        height = 9;
        found = 0;
        for (i = 9; i >= 0; i--)
        {
            if (dist[i]+1 < disttoroot)
                height -= 1;
            if (dist[i]+1 == disttoroot) /* burnt in the previous step */
            {
                if (pos->treeneighbour == i) /* the chosen one in the ordering */
                    found = 1;
                if (found == 0)
                    height -= 1;
            }
        }
    }
    pos->sand = height;
    return 0;
}

/* Pushes pointer to site onto the stack */
void push(struct node *site)
{
    if (stack_pointer < HASHSIZE)
        stack[stack_pointer++] = site;
    else
        printf("Error: stack overflow\n");
}

/* Pop a pointer to site from the stack */
struct node *pop(void)
{
    if (stack_pointer > 0)
        return stack[--stack_pointer];
    else {
        printf("Error: cannot pop from empty stack\n");
        exit(-1);
    }
}

/* Compute avalanche using waves and as much memory as we actually need. */
int avalanche()
{
    struct node *pos, *posorigin;
    unsigned int i;
    struct node *neighb;
    unsigned int neighbour[d];
    unsigned int origin[d];
    
    for (i = 0; i < d; i++)
        origin[i] = 32;
    posorigin = install(origin);
    if (sandheight(posorigin->coordinates))
        return -1;
    prob[posorigin->sand]++;
    if (posorigin->sand == 9)
    {
        posorigin->topple = 1;
	posorigin->sand++;
        while (posorigin->sand == 10)
        {
            posorigin->sand -= 10;
            for (i = 0; i < d; i++)
                neighbour[i] = posorigin->coordinates[i];
            for (i = 0; i < d; i++)
            {
	      neighbour[i]++;
	      neighb = install(neighbour);
	      if (!(++neighb->sand) ) /* initialize sand height to be -1, sand height has not been computed before */
		{
		  if (sandheight(neighbour))
		    return -1;
		  neighb->sand++;
		}
	      if (neighb->sand == 10)
		push(neighb);
	      neighbour[i] -= 2;
	      neighb = install(neighbour);
	      if (!(++neighb->sand) )
		{
		  if (sandheight(neighbour))
		    return -1;
		  neighb->sand++;
		}
	      if (neighb->sand == 10)
		push(neighb);
	      neighbour[i]++;
            }
            
            while (stack_pointer)
            {
                pos = pop();
                
                if (pos != posorigin)   /* Check it is not the origin */
                {
                    pos->topple = 1;
                    pos->sand -= 10;
                    for (i = 0; i < d; i++)
                        neighbour[i] = pos->coordinates[i];
                    for (i = 0; i < d; i++)
                    {
                        if((neighbour[i]++) &0b111111) /* not the sink */
                        {
                            neighb = install(neighbour);
                            if (! (++neighb->sand))
                            {
                                if (sandheight(neighbour))
                                    return -1;
                                neighb->sand++;
                            }
                            if (neighb->sand == 10)
                                push(neighb);
                        }
                        if ((neighbour[i] -= 2)&0b111111)
                        {
                            neighb = install(neighbour);
                            if (! (++neighb->sand) )
                            {
                                if (sandheight(neighbour))
                                    return -1;
                                neighb->sand++;
                            }
                            if (neighb->sand == 10)
                                push(neighb);
                        }
                        neighbour[i]++;
                    }
                }
                
            }
        }
        
    }
    return 0;
}


/* Carries out n independent avalanches */
unsigned long int vertex_indept(unsigned long int n)
{
    register unsigned long int i, j, k;
    unsigned long int x;
    unsigned int point[d];
    struct node *pos;
    
    k = n;
    for (i = 0; i < n; i++)
    {
        freevertex = vertexdata;
        for (j = 0; j < HASHSIZE; j++)
        {
            hashtab[j] = NULL;
            vertexdata[j].sand = -1;
        }
        for (j = 0; j < usage; j++)
        {
            vertexdata[j].next = NULL;
            vertexdata[j].intree = 0;
            vertexdata[j].topple = 0;
        }
        usage = 0;
        
        if (avalanche())
            k--;
        else
	  {
	    for (j = 0; j < d; j++)
	      point[j] = 32;
	    for (x = 0; x < L; x++)
	      {
		pos = install(point);
		if (pos->topple)
		  vertices[x] += 1;
		point[0]++;
	      }
	  }
    }
    return k;
}

int main(int argc, char *argv[])
{
    int color;
    int numprocs, myid, j, jump;
    MPI_Comm workers;
    MPI_Status status;
    unsigned long int n, num_batch, batch_remain_sent, batch_remain_received;
    unsigned long int i, k, total_k, x;
    char *tail;
    
    MPI_Init(&argc, &argv);
    world = MPI_COMM_WORLD;
    MPI_Comm_size(world, &numprocs);
    MPI_Comm_rank(world, &myid);
    
    if (myid == MASTER)
    {
        printf("Number of arguments: %d \n", argc);
        n = strtoul(argv[argc-1], &tail, 10);
        num_batch = n;
        batch_remain_sent = num_batch;
        batch_remain_received = num_batch;
        printf("Number of samples requested: %lu\n", n);
        jump = atoi(argv[argc-2]);
        printf("Jump requested: %d\n", jump);
    }
    
    MPI_Bcast(&num_batch, 1, MPI_UNSIGNED_LONG, MASTER, world);
    MPI_Bcast(&jump, 1, MPI_INT, MASTER, world);
    color = (myid > num_batch);
    MPI_Comm_split(world, color, 0, &workers);
    
    init_genrand(1); /* Set seed = 1 */
    pf = (unsigned long *)calloc(P_SIZE, sizeof(unsigned long));
    
    if (myid == MASTER)
    {
        FILE *fin;
        char c;
        
        /* read the file clist.txt, and set the coefficient */
        if ((fin = fopen("clist_mt19937.txt", "r")) == NULL){
            printf("File read error.\n");
            exit(1);
        }
        for (j = MEXP - 1; j>-1; j--){
            c = fgetc(fin);
            if (c == '1')
                set_coef(pf, j, 1);
        }
    }
    
    MPI_Bcast(pf, P_SIZE, MPI_UNSIGNED_LONG, MASTER, world);
    if (color == 0 && myid > 0)
    {
        jump_ahead(jump + myid - 1);  /* Jump ahead by (myid-1)*2^100 */
    }
    
    if (color == 0)
    {
        initialize();
	k = 0;
    }
    /* Master node initializes its arrays for  portion  and  probability */
    if (myid == MASTER)
     { 
      for (i=0; i < L; i++)
          portion[i] = 0;
      for (i=0; i < 10; i++)
          probability[i] = 0;
    }
    /* Main loop for Master Node */
    if (myid == MASTER)
    {
        sample_request = 1;
        for (j = 1; (j < numprocs && j <= num_batch); j++)
        {
            MPI_Send(&sample_request, 1, MPI_INT, j, SAMPLE_REQUEST, world);
            batch_remain_sent--;
        }
        while (batch_remain_received > 0)
        {
            MPI_Recv(&sample_request, 1, MPI_INT, MPI_ANY_SOURCE, SAMPLE_REPLY, world, &status);
            batch_remain_received--;
            if (batch_remain_sent > 0)
            {
                MPI_Send(&sample_request, 1, MPI_INT, status.MPI_SOURCE, SAMPLE_REQUEST, world);
                batch_remain_sent--;
            }
        }
        /* Send worker nodes a request to terminate computation */
        sample_request = 0;
        for (j = 1; (j < numprocs && j <= num_batch); j++)
            MPI_Send(&sample_request, 1, MPI_INT, j, SAMPLE_REQUEST, world);
    }
    /* Main loop for worker nodes */
    if (myid > 0 && color == 0)
    {
        MPI_Recv(&sample_request, 1, MPI_INT, MASTER, SAMPLE_REQUEST, world, &status);
        while (sample_request != 0)
        {
            k += vertex_indept(1);
            MPI_Send(&sample_request, 1, MPI_INT, MASTER, SAMPLE_REPLY, world);
            MPI_Recv(&sample_request, 1, MPI_INT, MASTER, SAMPLE_REQUEST, world, &status);
        }
    }
    MPI_Reduce(&vertices, &total_vertices, L, MPI_UNSIGNED_LONG, MPI_SUM, MASTER, workers);
    MPI_Reduce(&prob, &total_prob, 10, MPI_UNSIGNED_LONG, MPI_SUM, MASTER, workers);
    MPI_Reduce(&k, &total_k, 1, MPI_UNSIGNED_LONG, MPI_SUM, MASTER, workers);
    MPI_Comm_free(&workers);
    
    if (myid == MASTER)
    {
        printf("Number of samples generated = %lu\n",total_k);
        for (i = 0; i < 10; i++)
            probability[i] = ((float) total_prob[i])/total_k;
        for (x = 0; x < L; x++)
            portion[x] = ((float) total_vertices[x])/total_k;
        portion_file = fopen("portion5d-32m32.txt", "w");
        probability_file = fopen("probability5d-32m32.txt", "w");
        for (i = 0; i < L; i++)
        {
            fprintf(portion_file, "%f\n", portion[i]);
        }
        fclose(portion_file);
        for (i = 0; i < 10; i++)
        {
            fprintf(probability_file, "%f\n", probability[i]);
        }
        fclose(probability_file);
    }
    MPI_Finalize();
}

