
# Exploratory Data Analysis: US Crime Projected Grid ----------------------

# This R script produces plots for the census tract and gridded data produced from the R script DATA/PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/CountDataGen_final.R which are the data projected on the UTM coordinates.
# Note that the only socio-economic variables used within my thesis were the total population and average income and so the final R code for data generation was tailored accordingly, with the code for the inclusion of other socio-economic variables commented out and we do the same below. For any additional variables we comment out they're inclusion and plot outputs and this can be easily uncommented if required to generate the relevant plots.

# Author: Nadeen Khaleel


# Set WD and Load Libraries -----------------------------------------------

library("rstudioapi")
library(ggplot2)
library(gridExtra)
library(ggmap)
library(dplyr)
library(zipcode)
library(readr)
library(revgeo)
library(sf)

rbPal <- colorRampPalette(c('blue','red','yellow'))


# Los Angeles -------------------------------------------------------------

# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))
setwd("./LA")

load("../../PROCESSED_DATA/SHAPEFILES/LACityCT.rda")
ct_LA.proj <- lwgeom::st_transform_proj(ct_LA,"epsg:32611")

xy.coord <- sapply(1:length(ct_LA.proj$geometry),function(i){return(as.numeric(st_centroid(ct_LA.proj$geometry[[i]])))}) # returns a 2xlength(b$geometry) matrix, where each column is for each geometry and row one is longitude and row two is latitude
ct_LA.proj$x <- xy.coord[1,]
ct_LA.proj$y <- xy.coord[2,]


# Load in the crime data that has processed
la_gta <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/LA/la_gta.rds")
la_hom <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/LA/la_hom.rds")
la_hom_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/LA/la_hom_sf.rds")
la_gta_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/LA/la_gta_sf.rds")
la_hom_sf.proj <- lwgeom::st_transform_proj(la_hom_sf,"epsg:32611")
la_gta_sf.proj <- lwgeom::st_transform_proj(la_gta_sf,"epsg:32611")

la_gta15 <- la_gta[which(la_gta$Y==2015),]
la_hom15 <- la_hom[which(la_hom$Y==2015),]


# Los Angeles: Simple Summaries -------------------------------------------

summary(la_gta15)
dim(la_gta15)
# [1] 15972    32
summary(la_hom15)
dim(la_hom15)
# [1] 283  32


la_gta15_sf.proj <- la_gta_sf.proj[which(la_gta_sf.proj$Y==2015),]
la_hom15_sf.proj <- la_hom_sf.proj[which(la_hom_sf.proj$Y==2015),]

la_hom15_sp.proj <- as(la_hom15_sf.proj,"Spatial")
la_gta15_sp.proj <- as(la_gta15_sf.proj,"Spatial")

# Incidents per Year
hom.year <- ggplot(data=la_hom,aes(Y)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1)) + ggtitle("Homicides per Year in Los Angeles")
gta.year <- ggplot(data=la_gta,aes(Y)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1)) + ggtitle("Motor Vehicle Thefts per Year in Los Angeles")

pdf("LA_CrimevYear_proj_final.pdf",h=8,w=10,pointsize=14)
grid.arrange(hom.year,gta.year,ncol=2)
dev.off()


# Consider Y=2015 and consider Incidents per Month
hom.month15 <- ggplot(data=la_hom[la_hom$Y==2015,],aes(MY)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1)) + ggtitle("Homicides per Month in Los Angeles during 2015")
gta.month15 <- ggplot(data=la_gta[la_gta$Y==2015,],aes(MY)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1)) + ggtitle("Motor Vehicle Thefts per Month in Los Angeles during 2015")

pdf("LA_CrimevMonth2015_proj_final.pdf",h=8,w=10,pointsize=14)
grid.arrange(hom.month15,gta.month15,ncol=2)
dev.off()



# Plot as time series
ts_lahom <- la_hom %>% count(MY)
ts_laGTA <- la_gta %>% count(MY)
ts_lahom <- ts_lahom[order(ts_lahom$MY),]
ts_laGTA <- ts_lahom[order(ts_laGTA$MY),]
ts_lahom <- ts(ts_lahom$n,start = c(2010,01),end = c(2018,11),frequency=12)
ts_laGTA <- ts(ts_laGTA$n,start = c(2010,01),end = c(2018,11),frequency=12)


pdf("LA_CrimeTimeSeries_proj_final.pdf",h=8,w=10,pointsize=14)
par(mfrow=c(1,2))
plot(ts_lahom,main="Time Series of Homicides in LA")
plot(ts_laGTA,main="Time Series of Motor Vehicle Thefts in LA")
dev.off()


# There were several cases that classified for homicide in LA, what proportion of each class is there?
table(la_hom$`Crime Code Description`)
#       CRIMINAL HOMICIDE MANSLAUGHTER, NEGLIGENT 
#                   2469                       5 
table(la_hom[la_hom$Y==2015,]$`Crime Code Description`)
#       CRIMINAL HOMICIDE MANSLAUGHTER, NEGLIGENT 
#                   282                       1 
crime_lahom.plot <- ggplot(la_hom,aes(`Crime Code Description`)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1))
crime_lahom.plot



# Los Angeles: Spatial Plots ----------------------------------------------

# Los Angeles:  Map plots ####

# Bounding box for stamen map function
latla_low <- min(min(la_gta$lat),min(la_hom$lat)) - 0.1
lonla_low <- min(min(la_gta$lon),min(la_hom$lon)) - 0.1
latla_up <- max(max(la_gta$lat),max(la_hom$lat)) + 0.1
lonla_up <- max(max(la_gta$lon),max(la_hom$lon)) + 0.1

# Produce base map plot of LA to add anything else too.
gm_LA <- get_stamenmap(bbox=c(left=(lonla_low),right=(lonla_up),top=(latla_up),bottom=(latla_low)),zoom=10,maptype = "toner",source="stamen")
plot(gm_LA)

LA_map <- ggmap(gm_LA)

LA_map_GTA <- LA_map + geom_point(data = la_gta,aes(x=lon,y=lat), colour="red", size=0.1) +
  ggtitle("Plot of Locations of Stolen Vehicle Incidents in LA")
LA_map_GTA

pdf("LA_GTAMap15_final.pdf",h=8,w=10,pointsize=14)
LA_map_GTA2 <- LA_map + geom_point(data = la_gta15,aes(x=lon,y=lat), colour="red", size=1) +
  ggtitle("Plot of Locations of Stolen Vehicle Incidents in LA")
LA_map_GTA2
dev.off()

LA_map_Hom <- LA_map + geom_point(data = la_hom,aes(x=lon,y=lat), colour="red", size=0.1) + ggtitle("Plot of Locations of Homicide Incidents in LA")
LA_map_Hom
pdf("LA_HomMap15_final.pdf",h=8,w=10,pointsize=14)
LA_map_Hom2 <- LA_map + geom_point(data = la_hom15,aes(x=lon,y=lat), colour="red", size=1) + ggtitle("Plot of Locations of Homicide Incidents in LA")
LA_map_Hom2
dev.off()

pdf("LA_Crime_Maps_All_final.pdf",h=8,w=10,pointsize=14)
grid.arrange(LA_map_Hom,LA_map_GTA,ncol=2)
dev.off()

LA_map_GTA <- LA_map + geom_point(data = la_gta15,aes(x=lon,y=lat), colour="red", size=0.1) +
  ggtitle("Plot of Locations of Stolen Vehicle Incidents in LA in 2015")# + theme(text = element_text(size=20))
LA_map_GTA

LA_map_Hom <- LA_map + geom_point(data = la_hom15,aes(x=lon,y=lat), colour="red", size=0.1) + ggtitle("Plot of Locations of Homicide Incidents in LA in 2015")
LA_map_Hom

pdf("LA_Crime_Maps_2015_final.pdf",h=8,w=10,pointsize=14)
grid.arrange(LA_map_Hom,LA_map_GTA,ncol=2)
dev.off()

# Census Tract Plots
pdf("LA_CT_final.pdf",h=8,w=10,pointsize = 14)
plot(st_geometry(ct_LA), col = sf.colors(20, categorical = TRUE), border = 'grey', 
     axes = FALSE, bty="n", main="Los Angeles City Census Tracts")
dev.off()
pdf("LA_CT_proj_final.pdf",h=8,w=10,pointsize = 14)
plot(st_geometry(ct_LA.proj), col = sf.colors(20, categorical = TRUE), border = 'grey', 
     axes = FALSE, bty="n", main="Los Angeles City Census Tracts")
dev.off()

pdf("LACTHom2015_proj_final.pdf",h=8,w=10,pointsize = 14)
plot(st_geometry(ct_LA.proj), col = sf.colors(20, categorical = TRUE), border = 'grey', 
     axes = FALSE, bty="n", main="2015 Homicide Incidents in LA",cex.main=2)
plot(st_geometry(la_hom15_sf.proj),add=TRUE,pch=19,col="red")
dev.off()

pdf("LACTGTA2015_proj_final.pdf",h=8,w=10,pointsize = 14)
plot(st_geometry(ct_LA.proj), col = sf.colors(20, categorical = TRUE), border = 'grey', 
     axes = FALSE, bty="n", main="2015 Motor Vehicle Theft Incidents in LA",cex.main=2)
plot(st_geometry(la_gta15_sf.proj),add=TRUE,pch=19,col="red")
dev.off()


# Los Angeles: Plotting Gridded Data along with Census Tract Data ---------

# Create a wrapper function to plot the required socio-economic variables and crime gridded data where the input allows for us to alter the grid resolution we are interested in without replicating the code multiple times for each, just requires a call to this function for each resolution of interest. We can either use st_geometry() in the plot function with alterations to the binning and the "standard" plot just uses the plot function to extract the relevant variable.
# As well as the average income and population variables, we originally included a few other socio-economic variables however these were not included in any of the modelling of the crime data and updated data generation code has these commented out and so I will comment out the lines including these variables but retain the code so that it is complete as was originally run. If they are of interest, they can easily be uncommented to produce the relevant plots.

plot_la_grid <- function(grid_cellsx,grid_cellsy,standard=TRUE){
  
  # Load the required gridded data and socio-economic variables.
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/LA/LA2015CT",grid_cellsx,grid_cellsy,"CountData_projFinal.rda"))
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/LA/LAGridCells",grid_cellsx,grid_cellsy,"_projFinal.rda"))
  
  LA_ctpop_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTPop_15_proj.rds")
  LA_ctinc_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTInc_15_0imp_proj.rds")
  # LA_ctsex_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTSex_15_proj.rds")
  # LA_ctage_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTAge_15_proj.rds")
  # LA_ctocc_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTOcc_15_proj.rds")
  # LA_ctfood_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTFood_15_proj.rds")
  
  
  # Plot the gridded crime counts against the socio-economic variables interpolated on to the grids.
  # Average Income
  pdf(paste0("LAHomCountvIncImp",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=10,w=20,pointsize = 14)
  plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$inc,main="Counts of Homicide Normalised by Population Counts against\n Average Income",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Average Income")
  dev.off()
  
  pdf(paste0("LAGTACountvIncImp",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=10,w=20,pointsize = 14)
  plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$inc,main="Counts of Motor Vehicle Theft Normalised by Population Counts against\n Average Income",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Average Income")
  dev.off()
  
  # # Sex
  # pdf(paste0("LAHomCountvSex",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$msex,main="Counts of Homicide Normalised by Population Counts against Proportion of Males",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Males")
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$fsex,main="Counts of Homicide Normalised by Population Counts against Proportion of Females",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Females")
  # dev.off()
  # 
  # pdf(paste0("LAGTACountvSex",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$msex,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Males",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Males")
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$fsex,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Females",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Females")
  # dev.off()
  # 
  # # Occupied
  # pdf(paste0("LAHomCountvOcc",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$own,main="Counts of Homicide Normalised by Population Counts against Proportion of Properties Owned",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Owned Properties")
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$rent,main="Counts of Homicide Normalised by Population Counts against Proportion of Properties Rented",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Rented Properties")
  # dev.off()
  # 
  # pdf(paste0("LAGTACountvOcc",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$own,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Properties Owned",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Owned Properties")
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$rent,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Properties Rented",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Rented Properties")
  # dev.off()
  # 
  # # Food
  # pdf(paste0("LAHomCountvFood",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=10,w=20,pointsize = 14)
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$food,main="Counts of Homicide Normalised by Population Counts against\n Percentage of Households Receiving Food Stamps",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Percentage of Households Receiving Food Stamps")
  # dev.off()
  # 
  # pdf(paste0("LAGTACountvFood",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=10,w=20,pointsize = 14)
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$food,main="Counts of Motor Vehicle Theft Normalised by Population Counts against\n Percentage of Households Receiving Food Stamps",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Percentage of Households Receiving Food Stamps")
  # dev.off()
  
  # Now to plot heat maps for the crime, homicide and motor vehicle thefts for the census tracts as well as the gridded data.
  if (standard==FALSE){ # want to create our own bins for the heat maps
    la_cells$homcount <- hom_countdf$hom; la_cells$gtacount <- gta_countdf$gta
    la_cells.temp <- la_cells
    
    # First we will plot the grid crime data
    bin.upperlim <- max(la_cells.temp$homcount)
    
    # Bins for different colours 
    la_cells.temp$bin <- cut(la_cells.temp$homcount, c(-0.5,0.5,1.5,2.5,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(la_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    
    # Re-name because I wanted to separate out 0, 1 and 2 homicides...
    my.levels[1] <- " 0"
    my.levels[2] <- " 1"
    my.levels[3] <- " 2"
    my.levels[4] <- " 3+"
    
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(la_cells.temp$bin))))
    la_cells.temp$Col <- a[as.numeric(la_cells.temp$bin)]
    
    pdf(paste0("LAHom2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(la_cells.temp), col = la_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("Map of Homicides in LA\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(la_cells.temp$Col),title=expression("Homicide"),cex=1,xpd=NA)
    dev.off()
    
    bin.upperlim <- max(la_cells.temp$gtacount)
    
    # Bins for different colours 
    la_cells.temp$bin <- cut(la_cells.temp$gtacount, c(-0.5,0.5,1.5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(la_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    
    # Re-name levels
    length(my.levels)
    my.levels[1] <- " 0"
    my.levels[2] <- " 1"
    my.levels[3] <- " 2 to 10"
    my.levels[length(my.levels)] <- " 75+"
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(la_cells.temp$bin))))
    la_cells.temp$Col <- a[as.numeric(la_cells.temp$bin)]
    
    pdf(paste0("LAGTA2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(la_cells.temp), col = la_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("Map of Motor Vehicle Thefts in LA\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(la_cells.temp$Col),title=expression("MVT"),cex=1,xpd=NA)
    dev.off()
    
    # Heat Maps for Socio-economic Variables
    # First plot the census tract then grid socio-economic variables for each variable
    
    # Population ##
    LA_ctpop_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTPop_15_proj.rds")
    
    # Census Tract Heat Map: Population
    if (round(max(LA_ctpop_15$pop)/5000)!=max(LA_ctpop_15$pop)/5000){
      bin.upperlim <-5000*(round(max(LA_ctpop_15$pop)/5000)+1)
    } else {
      bin.upperlim <- max(LA_ctpop_15$pop)
    }
    
    # Bins for different colours 
    LA_ctpop_15$bin <- cut(LA_ctpop_15$pop, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(LA_ctpop_15$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(LA_ctpop_15$bin))))
    LA_ctpop_15$Col <- a[as.numeric(LA_ctpop_15$bin)]
    
    pdf("LAPop2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(ct_LA.proj), col = LA_ctpop_15$Col, border = 'black', axes = FALSE, bty="n", main="2015 Population in LA Census Tracts",cex.main=2)
    legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(LA_ctpop_15$Col),title=expression("Population (" *10^{3}*")"),cex=1,xpd=NA)
    dev.off()
    
    # Grid Heat Map: Population
    la_cells.temp <- la_cells
    la_cells.temp$pop <- hom_countdf$pop
    if (round(max(la_cells.temp$pop)/5000)!=max(la_cells.temp$pop)/5000){
      bin.upperlim <- 5000*(round(max(la_cells.temp$pop)/5000)+1)
    } else {
      bin.upperlim <- max(la_cells.temp$pop)
    }
    
    # Bins for different colours 
    la_cells.temp$bin <- cut(la_cells.temp$pop, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(la_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(la_cells.temp$bin))))
    la_cells.temp$Col <- a[as.numeric(la_cells.temp$bin)]
    
    pdf(paste0("LAPop2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(la_cells), col = la_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Population in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(la_cells.temp$Col),title=expression("Population (" *10^{3}*")"),cex=1,xpd=NA)
    dev.off()
    
    
    # Average Income ##
    # Census Tract Heat Map: Average Income
    if (round(max(LA_ctinc_15$inc,na.rm = TRUE)/1e5)!=max(LA_ctinc_15$inc,na.rm = TRUE)/1e5){
      bin.upperlim <-1e5*(round(max(LA_ctinc_15$inc,na.rm = TRUE)/1e5)+1)
    } else {
      bin.upperlim <- max(LA_ctinc_15$inc,na.rm = TRUE)
    }
    
    LA_ctinc_15$bin <- cut(LA_ctinc_15$inc,seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(LA_ctinc_15$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("0000","",lev5)
    my.levels <- lev6
    
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(LA_ctinc_15$bin))))
    LA_ctinc_15$Col <- a[as.numeric(LA_ctinc_15$bin)]
    
    pdf("LAIncImp2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(ct_LA.proj), col = LA_ctinc_15$Col, border = 'black', axes = FALSE, bty="n", main="2015 Mean Income in LA Census Tracts",cex.main=2)
    legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(LA_ctinc_15$Col),title=expression("Income (" *10^{4}*")"),cex=1,xpd=NA)
    dev.off()
    
    # Grid Heat Map: Average Income
    la_cells.temp <- la_cells
    la_cells.temp$inc <- hom_countdf$inc
    if (round(max(la_cells.temp$inc)/1e5)!=max(la_cells.temp$inc)/1e5){
      bin.upperlim <- 1e5*(round(max(la_cells.temp$inc)/1e5)+1)
    } else {
      bin.upperlim <- max(la_cells.temp$inc)
    }
    
    # Bins for different colours 
    la_cells.temp$bin <- cut(la_cells.temp$inc, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(la_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("0000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(la_cells.temp$bin))))
    la_cells.temp$Col <- a[as.numeric(la_cells.temp$bin)]
    
    pdf(paste0("LAIncImp2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(la_cells), col = la_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Average Income in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(la_cells.temp$Col),title=expression("Income (" *10^{4}*")"),cex=1,xpd=NA)
    dev.off()
    
    
    # # Sex ##
    # # Census Tract Heat Map: Sex
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # LA_ctsex_15$binm <- cut(LA_ctsex_15$mprop, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # LA_ctsex_15$binf <- cut(LA_ctsex_15$fprop, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(LA_ctsex_15$binm)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # am <- sort(rbPal(length(levels(LA_ctsex_15$binm))))
    # LA_ctsex_15$Colm <- am[as.numeric(LA_ctsex_15$binm)]
    # af <- sort(rbPal(length(levels(LA_ctsex_15$binf))))
    # LA_ctsex_15$Colf <- af[as.numeric(LA_ctsex_15$binf)]
    # 
    # pdf("LASex2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(ct_LA.proj), col = LA_ctsex_15$Colm, border = 'black', axes = FALSE, bty="n", main="2015 Proportion of Males\n in LA Census Tracts",cex.main=1.4)
    # plot(st_geometry(ct_LA.proj), col = LA_ctsex_15$Colf, border = 'black', axes = FALSE, bty="n", main="2015 Proportion of Females\n in LA Census Tracts",cex.main=1.4)
    # legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = am, legend = my.levels, col = unique(LA_ctsex_15$Colm),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # # Grid Heat Map: Sex
    # la_cells.temp <- la_cells
    # la_cells.temp$m <- hom_countdf$msex
    # la_cells.temp$f <- hom_countdf$fsex
    # 
    # bin.upperlim <- 1
    # 
    # 
    # # Bins for different colours 
    # la_cells.temp$binm <- cut(la_cells.temp$m, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # la_cells.temp$binf <- cut(la_cells.temp$f, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(la_cells.temp$binm)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # am <- sort(rbPal(length(levels(la_cells.temp$binm))))
    # la_cells.temp$Colm <- am[as.numeric(la_cells.temp$binm)]
    # af <- sort(rbPal(length(levels(la_cells.temp$binf))))
    # la_cells.temp$Colf <- af[as.numeric(la_cells.temp$binf)]
    # 
    # pdf(paste0("LASex2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(la_cells), col = la_cells.temp$Colm, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion of Males\n in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.4)
    # plot(st_geometry(la_cells), col = la_cells.temp$Colf, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion of Females\n in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.4)
    # legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = am, legend = my.levels, col = unique(la_cells.temp$Colm),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # 
    # # Home Occupier ##
    # # Census Tract Heat Map: Owned and Rented Properties
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # LA_ctocc_15$binown <- cut(LA_ctocc_15$ownprop, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # LA_ctocc_15$binrent <- cut(LA_ctocc_15$rentprop, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(LA_ctocc_15$binown)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # aown <- sort(rbPal(length(levels(LA_ctocc_15$binown))))
    # LA_ctocc_15$Colown <- aown[as.numeric(LA_ctocc_15$binown)]
    # arent <- sort(rbPal(length(levels(LA_ctocc_15$binrent))))
    # LA_ctocc_15$Colrent <- arent[as.numeric(LA_ctocc_15$binrent)]
    # 
    # pdf("LAOcc2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(ct_LA.proj), col = LA_ctocc_15$Colown, border = 'black', axes = FALSE, bty="n", main="2015 Proportion Properties\n Owned in LA Census Tracts",cex.main=1.3)
    # plot(st_geometry(ct_LA.proj), col = LA_ctocc_15$Colrent, border = 'black', axes = FALSE, bty="n", main="2015 Proportion Properties\n Rented in LA Census Tracts",cex.main=1.3)
    # legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = aown, legend = my.levels, col = unique(LA_ctocc_15$Colown),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # # Grid Heat Map: Owned and Rented Properties
    # la_cells.temp <- la_cells
    # la_cells.temp$own <- hom_countdf$own
    # la_cells.temp$rent <- hom_countdf$rent
    # 
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # la_cells.temp$binown <- cut(la_cells.temp$own, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # la_cells.temp$binrent <- cut(la_cells.temp$rent, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(la_cells.temp$binown)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # aown <- sort(rbPal(length(levels(la_cells.temp$binown))))
    # la_cells.temp$Colown <- aown[as.numeric(la_cells.temp$binown)]
    # arent <- sort(rbPal(length(levels(la_cells.temp$binrent))))
    # la_cells.temp$Colrent <- arent[as.numeric(la_cells.temp$binrent)]
    # 
    # pdf(paste0("LAOcc2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(la_cells), col = la_cells.temp$Colown, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion Properties\n Owned in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # plot(st_geometry(la_cells), col = la_cells.temp$Colrent, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion Properties\n Rented in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = aown, legend = my.levels, col = unique(la_cells.temp$Colown),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # 
    # # Food Stamps ##
    # # Census Tract Heat Map: Food Stamps
    # bin.upperlim <- 100
    # 
    # # Bins for different colours 
    # LA_ctfood_15$bin <- cut(LA_ctfood_15$proprec, c(-0.5,0,5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(LA_ctfood_15$bin)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # my.levels[1] <- " 0"
    # 
    # # Create a function to generate a continuous color palette
    # a <- sort(rbPal(length(levels(LA_ctfood_15$bin))))
    # LA_ctfood_15$Col <- a[as.numeric(LA_ctfood_15$bin)]
    # 
    # pdf("LAFood2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    # par(oma=c(0,0,0,8))
    # plot(st_geometry(ct_LA.proj), col = LA_ctfood_15$Col, border = 'black', axes = FALSE, bty="n", main="2015 Percentage of Houses Receiving\n Food Stamps in LA Census Tracts",cex.main=1.3)
    # legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(LA_ctfood_15$Col),title=expression("Percentage"),cex=1,xpd=NA)
    # dev.off()
    # 
    # # Grid Heat Map: Food Stamps
    # la_cells.temp <- la_cells
    # la_cells.temp$food <- hom_countdf$food
    # 
    # bin.upperlim <- 100 # 100%
    # # Bins for different colours 
    # la_cells.temp$bin <- cut(la_cells.temp$food, c(-0.5,0,5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(la_cells.temp$bin)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # my.levels[1] <- " 0"
    # 
    # # Create a function to generate a continuous color palette
    # a <- sort(rbPal(length(levels(la_cells.temp$bin))))
    # la_cells.temp$Col <- a[as.numeric(la_cells.temp$bin)]
    # 
    # pdf(paste0("LAFood2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,0,8))
    # plot(st_geometry(la_cells), col = la_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Percentage of Houses Receiving\n Food Stamps in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(la_cells.temp$Col),title=expression("Percentage"),cex=1,xpd=NA)
    # dev.off()
    # 
  } else { # standard plots, using the sf format and just a simple plot() function call and no alteration of the bins for the colours
    
    la_cellstemp <- la_cells
    la_cellstemp$homcount <- hom_countdf$hom; la_cellstemp$gtacount <- gta_countdf$gta
    
    # First plot the grid crime data
    pdf(paste0("LAHom2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    plot(la_cellstemp["homcount"], main=paste0("Map of Homicides in LA\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    dev.off()
    
    pdf(paste0("LAGTA2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    plot(la_cellstemp["gtacount"], main=paste0("Map of Motor Vehicle Thefts in LA\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    dev.off()
    
    # Heat Maps for Socio-economic Variables
    # First plot the census tract then grid socio-economic variables for each variable
    ct_LA.projtemp <- ct_LA.proj
    ct_LA.projtemp$pop <- LA_ctpop_15$pop[match(ct_LA.proj$GEOID,LA_ctpop_15$geoid2)]
    ct_LA.projtemp$inc <- LA_ctinc_15$inc[match(ct_LA.proj$GEOID,LA_ctinc_15$geoid2)]
    # ct_LA.projtemp$msex <- LA_ctsex_15$mprop[match(ct_LA.proj$GEOID,LA_ctsex_15$geoid2)]
    # ct_LA.projtemp$fsex <- LA_ctsex_15$fprop[match(ct_LA.proj$GEOID,LA_ctsex_15$geoid2)]
    # ct_LA.projtemp$own <- LA_ctocc_15$ownprop[match(ct_LA.proj$GEOID,LA_ctocc_15$geoid2)]
    # ct_LA.projtemp$rent <- LA_ctocc_15$rentprop[match(ct_LA.proj$GEOID,LA_ctocc_15$geoid2)]
    # ct_LA.projtemp$food <- LA_ctfood_15$proprec[match(ct_LA.proj$GEOID,LA_ctfood_15$geoid2)]
    
    la_cellstemp$pop <- hom_countdf$pop
    la_cellstemp$inc <- hom_countdf$inc
    # la_cellstemp$msex <- hom_countdf$msex
    # la_cellstemp$fsex <- hom_countdf$fsex
    # la_cellstemp$own <- hom_countdf$own
    # la_cellstemp$rent <- hom_countdf$rent
    # la_cellstemp$food <- hom_countdf$food
    
    
    # Population ##
    # Census Tract Heat Map: Population
    pdf("LAPop2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(ct_LA.projtemp["pop"], main=NULL,cex.main=2)
    mtext("2015 Population in LA Census Tracts",side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    # Grid Heat Map: Population
    pdf(paste0("LAPop2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(la_cellstemp["pop"], main=NULL,cex.main=2)
    mtext(paste0("2015 Population in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    
    # Average Income ##
    # Census Tract Heat Map: Average Income
    pdf("LAIncImp2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(ct_LA.projtemp["inc"], main=NULL,cex.main=2)
    mtext("2015 Mean Income in LA Census Tracts",side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    # Grid Heat Map: Average Income
    pdf(paste0("LAIncImp2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(la_cellstemp["inc"], main=NULL,cex.main=2)
    mtext(paste0("2015 Average Income in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    # # Sex ##
    # # Census Tract Heat Map: Sex
    # pdf("LASex2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(ct_LA.projtemp["msex"], main=NULL,cex.main=1.4)
    # mtext("2015 Proportion of Males\n in LA Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # plot(ct_LA.projtemp["fsex"], main=NULL,cex.main=1.4)
    # mtext("2015 Proportion of Females\n in LA Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # # Grid Heat Map: Sex
    # pdf(paste0("LASex2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(la_cellstemp["msex"], main=NULL,cex.main=1.4)
    # mtext(paste0("2015 Proportion of Males\n in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # plot(la_cellstemp["fsex"], main=NULL,cex.main=1.4)
    # mtext(paste0("2015 Proportion of Females\n in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # 
    # # Home Occupier ##
    # # Census Tract Heat Map: Owned and Rented Properties
    # pdf("LAOcc2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(ct_LA.projtemp["own"], main=NULL,cex.main=1.3)
    # mtext("2015 Proportion Properties\n Owned in LA Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # plot(ct_LA.projtemp["rent"], main=NULL,cex.main=1.3)
    # mtext("2015 Proportion Properties\n Rented in LA Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # # Grid Heat Map: Owned and Rented Properties
    # pdf(paste0("LAOcc2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(la_cellstemp["own"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Proportion Properties\n Owned in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # plot(la_cellstemp["rent"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Proportion Properties\n Rented in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # # Food Stamps ##
    # # Census Tract Heat Map: Food Stamps
    # pdf("LAFood2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(ct_LA.projtemp["food"], main=NULL,cex.main=1.3)
    # mtext("2015 Percentage of Houses Receiving\n Food Stamps in LA Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # # Grid Heat Map: Food Stamps
    # pdf(paste0("LAFood2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(la_cellstemp["food"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Percentage of Houses Receiving\n Food Stamps in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
  }
}


grid_cellsxvec <- c(10,24,48,95)
grid_cellsyvec <- c(15,36,72,144)

for (i in 1:4){
  plot_la_grid(grid_cellsx=grid_cellsxvec[i],grid_cellsy=grid_cellsyvec[i],standard=TRUE)
}

# New York ----------------------------------------------------------------

# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))
setwd("./NYC")

load("../../PROCESSED_DATA/SHAPEFILES/NYCityCT.rda")
ct_NY.proj <- lwgeom::st_transform_proj(ct_NY,"epsg:32618")

xy.coord <- sapply(1:length(ct_NY.proj$geometry),function(i){return(as.numeric(st_centroid(ct_NY.proj$geometry[[i]])))}) # returns a 2xlength(b$geometry) matrix, where each column is for each geometry and row one is longitude and row two is latitude

ct_NY.proj$x <- xy.coord[1,] # [1] 
ct_NY.proj$y <- xy.coord[2,] # [1] 


# Load in the crime data that has processed
nyc_gta <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/NYC/nyc_gta.rds")
nyc_hom <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/NYC/nyc_hom.rds")
nyc_hom_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/NYC/nyc_hom_sf.rds")
nyc_gta_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/NYC/nyc_gta_sf.rds")
nyc_hom_sf.proj <- lwgeom::st_transform_proj(nyc_hom_sf,"epsg:32618")
nyc_gta_sf.proj <- lwgeom::st_transform_proj(nyc_gta_sf,"epsg:32618")

nyc_gta15 <- nyc_gta[which(nyc_gta$Y==2015),]
nyc_hom15 <- nyc_hom[which(nyc_hom$Y==2015),]


# New York: Simple Summaries ----------------------------------------------

summary(nyc_gta15)
dim(nyc_gta15)
# [1] 7312   39

summary(nyc_hom15)
dim(nyc_hom15)
# [1] 357  39

nyc_gta15_sf.proj <- nyc_gta_sf.proj[which(nyc_gta_sf.proj$Y==2015),]
nyc_hom15_sf.proj <- nyc_hom_sf.proj[which(nyc_hom_sf.proj$Y==2015),]

nyc_hom15_sp.proj <- as(nyc_hom15_sf.proj,"Spatial")
nyc_gta15_sp.proj <- as(nyc_gta15_sf.proj,"Spatial")

# Incidents per Year
hom.year <- ggplot(data=nyc_hom,aes(Y)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1)) + ggtitle("Homicides per Year in New York")
gta.year <- ggplot(data=nyc_gta,aes(Y)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1)) + ggtitle("Motor Vehicle Thefts per Year in New York")

pdf("NY_CrimevYear_proj_final.pdf",h=8,w=10,pointsize=14)
grid.arrange(hom.year,gta.year,ncol=2)
dev.off()


# Consider Y=2015 and consider Incidents per Month
hom.month15 <- ggplot(data=nyc_hom[nyc_hom$Y==2015,],aes(MY)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1)) + ggtitle("Homicides per Month in New York during 2015")
gta.month15 <- ggplot(data=nyc_hom[nyc_hom$Y==2015,],aes(MY)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1)) + ggtitle("Motor Vehicle Thefts per Month in New York during 2015")

pdf("NY_CrimevMonth2015_proj_final.pdf",h=8,w=10,pointsize=14)
grid.arrange(hom.month15,gta.month15,ncol=2)
dev.off()


# There were several cases that classified for homicide in NY, what proportion of each class is there?
table(nyc_hom$OFNS_DESC)
# HOMICIDE-NEGLIGENT-VEHICLE  HOMICIDE-NEGLIGENT,UNCLASSIFIE MURDER & NON-NEGL. MANSLAUGHTER 
# 99                             137                            4748 
table(nyc_hom[which(nyc_hom$Y==2015),]$OFNS_DESC)
# HOMICIDE-NEGLIGENT-VEHICLE  HOMICIDE-NEGLIGENT,UNCLASSIFIE MURDER & NON-NEGL. MANSLAUGHTER 
# 3                               9                             345 
crime_nychom.plot <- ggplot(nyc_hom,aes(OFNS_DESC)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1))
crime_nychom.plot



# New York: Spatial Plots -------------------------------------------------

# New York: Map Plots -----------------------------------------------------

# Bounding box for stamen map function
latnyc_low <- min(min(nyc_gta$Latitude),min(nyc_hom$Latitude)) - 0.1
lonnyc_low <- min(min(nyc_gta$Longitude),min(nyc_hom$Longitude)) - 0.1
latnyc_up <- max(max(nyc_gta$Latitude),max(nyc_hom$Latitude)) + 0.1
lonnyc_up <- max(max(nyc_gta$Longitude),max(nyc_hom$Longitude)) + 0.1

# Produce base map plot of NY to add anything else too.
gm_NY <- get_stamenmap(bbox=c(left=(lonnyc_low),right=(lonnyc_up),top=(latnyc_up),bottom=(latnyc_low)),zoom=10,maptype = "toner",source="stamen")
plot(gm_NY)

NY_map <- ggmap(gm_NY)


NY_map_GTA <- NY_map + geom_point(data = nyc_gta,aes(x=Longitude,y=Latitude), colour="red", size=0.1) +
  ggtitle("Plot of Locations of Stolen Vehicle Incidents in NY")
NY_map_GTA

pdf("NY_GTAMap15_final.pdf",h=8,w=10,pointsize=14)
NY_map_GTA2 <- NY_map + geom_point(data = nyc_gta15,aes(x=Longitude,y=Latitude), colour="red", size=1) +
  ggtitle("Plot of Locations of Stolen Vehicle Incidents in NY")
NY_map_GTA2
dev.off()

NY_map_Hom <- NY_map + geom_point(data = nyc_hom,aes(x=Longitude,y=Latitude), colour="red", size=0.1) + ggtitle("Plot of Locations of Homicide Incidents in NY")
NY_map_Hom
pdf("NY_HomMap15_final.pdf",h=8,w=10,pointsize=14)
NY_map_Hom2 <- NY_map + geom_point(data = nyc_hom15,aes(x=Longitude,y=Latitude), colour="red", size=1) + ggtitle("Plot of Locations of Homicide Incidents in NY")
NY_map_Hom2
dev.off()

pdf("NY_Crime_Maps_All_final.pdf",h=8,w=10,pointsize=14)
grid.arrange(NY_map_Hom,NY_map_GTA,ncol=2)
dev.off()


NY_map_GTA <- NY_map + geom_point(data = nyc_gta15,aes(x=Longitude,y=Latitude), colour="red", size=0.1) +
  ggtitle("Plot of Locations of Stolen Vehicle Incidents in NY in 2015")
NY_map_GTA

NY_map_Hom <- NY_map + geom_point(data = nyc_hom15,aes(x=Longitude,y=Latitude), colour="red", size=0.1) + ggtitle("Plot of Locations of Homicide Incidents in NY in 2015")
NY_map_Hom

pdf("NY_Crime_Maps_2015_final.pdf",h=8,w=10,pointsize=14)
grid.arrange(NY_map_Hom,NY_map_GTA,ncol=2)
dev.off()

# Census Tract Plots
pdf("NY_CT_final.pdf",h=8,w=10,pointsize = 14)
plot(st_geometry(ct_NY), col = sf.colors(20, categorical = TRUE), border = 'grey', 
     axes = FALSE, bty="n", main="New York City Census Tracts")
dev.off()
pdf("NY_CT_proj_final.pdf",h=8,w=10,pointsize = 14)
plot(st_geometry(ct_NY.proj), col = sf.colors(20, categorical = TRUE), border = 'grey', 
     axes = FALSE, bty="n", main="New York City Census Tracts")
dev.off()

pdf("NYCTHom2015_proj_final.pdf",h=8,w=10,pointsize = 14)
plot(st_geometry(ct_NY.proj), col = sf.colors(20, categorical = TRUE), border = 'grey', 
     axes = FALSE, bty="n", main="2015 Homicide Incidents in NY",cex.main=2)
plot(st_geometry(nyc_hom15_sf.proj),add=TRUE,pch=19,col="red")
dev.off()

pdf("NYCTGTA2015_proj_final.pdf",h=8,w=10,pointsize = 14)
plot(st_geometry(ct_NY.proj), col = sf.colors(20, categorical = TRUE), border = 'grey', 
     axes = FALSE, bty="n", main="2015 Motor Vehicle Theft Incidents in NY",cex.main=2)
plot(st_geometry(nyc_gta15_sf.proj),add=TRUE,pch=19,col="red")
dev.off()



# New York: Plotting Gridded Data along with Census Tract Data ------------

# Create a wrapper function to plot the required socio-economic variables and crime gridded data where the input allows for us to alter the grid resolution we are interested in without replicating the code multiple times for each, just requires a call to this function for each resolution of interest. We can either use st_geometry() in the plot function with alterations to the binning and the "standard" plot just uses the plot function to extract the relevant variable.
# As well as the average income and population variables, we originally included a few other socio-economic variables however these were not included in any of the modelling of the crime data and updated data generation code has these commented out and so I will comment out the lines including these variables but retain the code so that it is complete as was originally run. If they are of interest, they can easily be uncommented to produce the relevant plots.

plot_nyc_grid <- function(grid_cellsx,grid_cellsy,standard=FALSE){
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/NYC/NY2015CT",grid_cellsx,grid_cellsy,"CountData_projFinal.rda"))
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/NYC/NYGridCells",grid_cellsx,grid_cellsy,"_projFinal.rda"))
  
  NY_ctpop_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTPop_15_proj.rds")
  NY_ctinc_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTInc_15_0imp_proj.rds")
  # NY_ctsex_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTSex_15_proj.rds")
  # NY_ctage_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTAge_15_proj.rds")
  # NY_ctocc_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTOcc_15_proj.rds")
  # NY_ctfood_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTFood_15_proj.rds")
  
  # Plot the gridded crime counts against the socio-economic variables interpolated on to the grids.
  # Average Income
  pdf(paste0("NYHomCountvIncImp",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=10,w=20,pointsize = 14)
  plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$inc,main="Counts of Homicide Normalised by Population Counts against\n Average Income",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Average Income")
  dev.off()
  
  pdf(paste0("NYGTACountvIncImp",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=10,w=20,pointsize = 14)
  plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$inc,main="Counts of Motor Vehicle Theft Normalised by Population Counts against\n Average Income",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Average Income")
  dev.off()
  
  # # Sex
  # pdf(paste0("NYHomCountvSex",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$msex,main="Counts of Homicide Normalised by Population Counts against Proportion of Males",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Males")
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$fsex,main="Counts of Homicide Normalised by Population Counts against Proportion of Females",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Females")
  # dev.off()
  # 
  # pdf(paste0("NYGTACountvSex",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$msex,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Males",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Males")
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$fsex,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Females",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Females")
  # dev.off()
  # 
  # # Occupied
  # pdf(paste0("NYHomCountvOcc",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$own,main="Counts of Homicide Normalised by Population Counts against Proportion of Properties Owned",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Owned Properties")
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$rent,main="Counts of Homicide Normalised by Population Counts against Proportion of Properties Rented",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Rented Properties")
  # dev.off()
  # 
  # pdf(paste0("NYGTACountvOcc",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$own,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Properties Owned",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Owned Properties")
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$rent,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Properties Rented",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Rented Properties")
  # dev.off()
  # 
  # # Food
  # pdf(paste0("NYHomCountvFood",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=10,w=20,pointsize = 14)
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$food,main="Counts of Homicide Normalised by Population Counts against\n Percentage of Households Receiving Food Stamps",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Percentage of Households Receiving Food Stamps")
  # dev.off()
  # 
  # pdf(paste0("NYGTACountvFood",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=10,w=20,pointsize = 14)
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$food,main="Counts of Motor Vehicle Theft Normalised by Population Counts against\n Percentage of Households Receiving Food Stamps",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Percentage of Households Receiving Food Stamps")
  # dev.off()
  
  # Now to plot heat maps for the crime, homicide and motor vehicle thefts for the census tracts as well as the gridded data.
  if (standard==FALSE){ # want to create our own bins for the heat maps
    nyc_cells$homcount <- hom_countdf$hom; nyc_cells$gtacount <- gta_countdf$gta
    nyc_cells.temp <- nyc_cells
    
    # First we will plot the grid crime data
    bin.upperlim <- max(nyc_cells.temp$homcount)
    
    # Bins for different colours 
    nyc_cells.temp$bin <- cut(nyc_cells.temp$homcount, c(-0.5,0.5,1.5,2.5,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(nyc_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    
    # Re-name because I wanted to separate out 0, 1 and 2 homicides...
    my.levels[1] <- " 0"
    my.levels[2] <- " 1"
    my.levels[3] <- " 2"
    my.levels[4] <- " 3+"
    
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(nyc_cells.temp$bin))))
    nyc_cells.temp$Col <- a[as.numeric(nyc_cells.temp$bin)]
    
    pdf(paste0("NYHom2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(nyc_cells.temp), col = nyc_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("Map of Homicides in New York\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(nyc_cells.temp$Col),title=expression("Homicide"),cex=1,xpd=NA)
    dev.off()
    
    
    bin.upperlim <- max(nyc_cells.temp$gtacount)
    
    # Bins for different colours 
    nyc_cells.temp$bin <- cut(nyc_cells.temp$gtacount, c(-0.5,0.5,1.5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(nyc_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    
    # Re-name levels
    length(my.levels)
    my.levels[1] <- " 0"
    my.levels[2] <- " 1"
    my.levels[3] <- " 2 to 10"
    my.levels[length(my.levels)] <- " 75+"
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(nyc_cells.temp$bin))))
    nyc_cells.temp$Col <- a[as.numeric(nyc_cells.temp$bin)]
    
    pdf(paste0("NYGTA2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(nyc_cells.temp), col = nyc_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("Map of Motor Vehicle Thefts in New York\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(nyc_cells.temp$Col),title=expression("MVT"),cex=1,xpd=NA)
    dev.off()
    
    # Heat Maps for Socio-economic Variables
    # First plot the census tract then grid socio-economic variables for each variable
    
    # Population ##
    NY_ctpop_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTPop_15_proj.rds")
    
    # Census Tract Heat Map: Population
    if (round(max(NY_ctpop_15$pop)/5000)!=max(NY_ctpop_15$pop)/5000){
      bin.upperlim <-5000*(round(max(NY_ctpop_15$pop)/5000)+1)
    } else {
      bin.upperlim <- max(NY_ctpop_15$pop)
    }
    
    # Bins for different colours 
    NY_ctpop_15$bin <- cut(NY_ctpop_15$pop, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(NY_ctpop_15$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(NY_ctpop_15$bin))))
    NY_ctpop_15$Col <- a[as.numeric(NY_ctpop_15$bin)]
    
    pdf("NYPop2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(ct_NY.proj), col = NY_ctpop_15$Col, border = 'black', axes = FALSE, bty="n", main="2015 Population in New York\n Census Tracts",cex.main=2)
    legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(NY_ctpop_15$Col),title=expression("Population (" *10^{3}*")"),cex=1,xpd=NA)
    dev.off()
    
    # Grid Heat Map: Population
    nyc_cells.temp <- nyc_cells
    nyc_cells.temp$pop <- hom_countdf$pop
    if (round(max(nyc_cells.temp$pop)/5000)!=max(nyc_cells.temp$pop)/5000){
      bin.upperlim <- 5000*(round(max(nyc_cells.temp$pop)/5000)+1)
    } else {
      bin.upperlim <- max(nyc_cells.temp$pop)
    }
    
    # Bins for different colours 
    nyc_cells.temp$bin <- cut(nyc_cells.temp$pop, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(nyc_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(nyc_cells.temp$bin))))
    nyc_cells.temp$Col <- a[as.numeric(nyc_cells.temp$bin)]
    
    pdf(paste0("NYPop2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(nyc_cells), col = nyc_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Population in New York on\n ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(nyc_cells.temp$Col),title=expression("Population (" *10^{3}*")"),cex=1,xpd=NA)
    dev.off()
    
    
    # Average Income ##
    # Census Tract Heat Map: Average Income
    if (round(max(NY_ctinc_15$inc,na.rm = TRUE)/1e5)!=max(NY_ctinc_15$inc,na.rm = TRUE)/1e5){
      bin.upperlim <-1e5*(round(max(NY_ctinc_15$inc,na.rm = TRUE)/1e5)+1)
    } else {
      bin.upperlim <- max(NY_ctinc_15$inc,na.rm = TRUE)
    }
    
    NY_ctinc_15$bin <- cut(NY_ctinc_15$inc, seq(0,bin.upperlim,length.out=6), include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(NY_ctinc_15$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("0000","",lev5)
    my.levels <- lev6
    
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(NY_ctinc_15$bin))))
    NY_ctinc_15$Col <- a[as.numeric(NY_ctinc_15$bin)]
    
    pdf("NYIncImp2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(ct_NY.proj), col = NY_ctinc_15$Col, border = 'black', axes = FALSE, bty="n", main="2015 Mean Income in New York\n Census Tracts",cex.main=2)
    legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(NY_ctinc_15$Col),title=expression("Income (" *10^{4}*")"),cex=1,xpd=NA)
    dev.off()
    
    # Grid Heat Map: Average Income
    nyc_cells.temp <- nyc_cells
    nyc_cells.temp$inc <- hom_countdf$inc
    if (round(max(nyc_cells.temp$inc)/1e6)!=max(nyc_cells.temp$inc)/1e6){
      bin.upperlim <- 1e6*(round(max(nyc_cells.temp$inc)/1e6)+1)
    } else {
      bin.upperlim <- max(nyc_cells.temp$inc)
    }
    
    # Bins for different colours
    nyc_cells.temp$bin <- cut(nyc_cells.temp$inc, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(nyc_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("0000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(nyc_cells.temp$bin))))
    nyc_cells.temp$Col <- a[as.numeric(nyc_cells.temp$bin)]
    
    pdf(paste0("NYIncImp2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(nyc_cells), col = nyc_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Average Income in New York\n on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(nyc_cells.temp$Col),title=expression("Income (" *10^{4}*")"),cex=1,xpd=NA)
    dev.off()
    
    
    # # Sex ##
    # # Census Tract Heat Map: Sex
    # bin.upperlim <- 1
    # 
    # 
    # # Bins for different colours 
    # NY_ctsex_15$binm <- cut(NY_ctsex_15$mprop, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # NY_ctsex_15$binf <- cut(NY_ctsex_15$fprop, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(NY_ctsex_15$binm)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # am <- sort(rbPal(length(levels(NY_ctsex_15$binm))))
    # NY_ctsex_15$Colm <- am[as.numeric(NY_ctsex_15$binm)]
    # af <- sort(rbPal(length(levels(NY_ctsex_15$binf))))
    # NY_ctsex_15$Colf <- af[as.numeric(NY_ctsex_15$binf)]
    # 
    # pdf("NYSex2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(ct_NY.proj), col = NY_ctsex_15$Colm, border = 'black', axes = FALSE, bty="n", main="2015 Proportion of Males\n in New York Census Tracts",cex.main=1.4)
    # plot(st_geometry(ct_NY.proj), col = NY_ctsex_15$Colf, border = 'black', axes = FALSE, bty="n", main="2015 Proportion of Females\n in New York Census Tracts",cex.main=1.4)
    # legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = am, legend = my.levels, col = unique(NY_ctsex_15$Colm),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # # Grid Heat Map: Sex
    # nyc_cells.temp <- nyc_cells
    # nyc_cells.temp$m <- hom_countdf$msex
    # nyc_cells.temp$f <- hom_countdf$fsex
    # 
    # bin.upperlim <- 1
    # 
    # 
    # # Bins for different colours 
    # nyc_cells.temp$binm <- cut(nyc_cells.temp$m, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # nyc_cells.temp$binf <- cut(nyc_cells.temp$f, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(nyc_cells.temp$binm)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # am <- sort(rbPal(length(levels(nyc_cells.temp$binm))))
    # nyc_cells.temp$Colm <- am[as.numeric(nyc_cells.temp$binm)]
    # af <- sort(rbPal(length(levels(nyc_cells.temp$binf))))
    # nyc_cells.temp$Colf <- af[as.numeric(nyc_cells.temp$binf)]
    # 
    # pdf(paste0("NYSex2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(nyc_cells), col = nyc_cells.temp$Colm, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion of Males\n in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.4)
    # plot(st_geometry(nyc_cells), col = nyc_cells.temp$Colf, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion of Females\n in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.4)
    # legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = am, legend = my.levels, col = unique(nyc_cells.temp$Colm),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # 
    # # Home Occupier ##
    # # Census Tract Heat Map: Owned and Rented Properties
    # bin.upperlim <- 1
    # 
    # 
    # # Bins for different colours 
    # NY_ctocc_15$binown <- cut(NY_ctocc_15$ownprop, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # NY_ctocc_15$binrent <- cut(NY_ctocc_15$rentprop, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(NY_ctocc_15$binown)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # aown <- sort(rbPal(length(levels(NY_ctocc_15$binown))))
    # NY_ctocc_15$Colown <- aown[as.numeric(NY_ctocc_15$binown)]
    # arent <- sort(rbPal(length(levels(NY_ctocc_15$binrent))))
    # NY_ctocc_15$Colrent <- arent[as.numeric(NY_ctocc_15$binrent)]
    # 
    # pdf("NYOcc2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(ct_NY.proj), col = NY_ctocc_15$Colown, border = 'black', axes = FALSE, bty="n", main="2015 Proportion Properties\n Owned in New York Census Tracts",cex.main=1.3)
    # plot(st_geometry(ct_NY.proj), col = NY_ctocc_15$Colrent, border = 'black', axes = FALSE, bty="n", main="2015 Proportion Properties\n Rented in New York Census Tracts",cex.main=1.3)
    # legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = aown, legend = my.levels, col = unique(NY_ctocc_15$Colown),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # # Grid Heat Map: Owned and Rented Properties
    # nyc_cells.temp <- nyc_cells
    # nyc_cells.temp$own <- hom_countdf$own
    # nyc_cells.temp$rent <- hom_countdf$rent
    # 
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # nyc_cells.temp$binown <- cut(nyc_cells.temp$own, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # nyc_cells.temp$binrent <- cut(nyc_cells.temp$rent, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(nyc_cells.temp$binown)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # aown <- sort(rbPal(length(levels(nyc_cells.temp$binown))))
    # nyc_cells.temp$Colown <- aown[as.numeric(nyc_cells.temp$binown)]
    # arent <- sort(rbPal(length(levels(nyc_cells.temp$binrent))))
    # nyc_cells.temp$Colrent <- arent[as.numeric(nyc_cells.temp$binrent)]
    # 
    # pdf(paste0("NYOcc2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(nyc_cells), col = nyc_cells.temp$Colown, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion Properties\n Owned in New York\n on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # plot(st_geometry(nyc_cells), col = nyc_cells.temp$Colrent, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion Properties\n Rented in New York\n on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = aown, legend = my.levels, col = unique(nyc_cells.temp$Colown),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # 
    # # Food Stamps ##
    # # Census Tract Heat Map: Food Stamps
    # 
    # bin.upperlim <- 100
    # 
    # # Bins for different colours 
    # NY_ctfood_15$bin <- cut(NY_ctfood_15$proprec, c(-0.5,0,5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(NY_ctfood_15$bin)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # my.levels[1] <- " 0"
    # 
    # # Create a function to generate a continuous color palette
    # a <- sort(rbPal(length(levels(NY_ctfood_15$bin))))
    # NY_ctfood_15$Col <- a[as.numeric(NY_ctfood_15$bin)]
    # 
    # pdf("NYFood2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    # par(oma=c(0,0,0,8))
    # plot(st_geometry(ct_NY.proj), col = NY_ctfood_15$Col, border = 'black', axes = FALSE, bty="n", main="2015 Percentage of Houses Receiving\n Food Stamps in New York\n Census Tracts",cex.main=1.3)
    # legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(NY_ctfood_15$Col),title=expression("Percentage"),cex=1,xpd=NA)
    # dev.off()
    # 
    # # Grid Heat Map: Food Stamps
    # nyc_cells.temp <- nyc_cells
    # nyc_cells.temp$food <- hom_countdf$food
    # 
    # bin.upperlim <- 100 # 100%
    # # Bins for different colours 
    # nyc_cells.temp$bin <- cut(nyc_cells.temp$food, c(-0.5,0,5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(nyc_cells.temp$bin)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # my.levels[1] <- " 0"
    # 
    # # Create a function to generate a continuous color palette
    # a <- sort(rbPal(length(levels(nyc_cells.temp$bin))))
    # nyc_cells.temp$Col <- a[as.numeric(nyc_cells.temp$bin)]
    # 
    # pdf(paste0("NYFood2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,0,8))
    # plot(st_geometry(nyc_cells), col = nyc_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Percentage of Houses Receiving\n Food Stamps in New York\n on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(nyc_cells.temp$Col),title=expression("Percentage"),cex=1,xpd=NA)
    # dev.off()
  } else { # standard plots, using the sf format and just a simple plot() function call and no alteration of the bins for the colours
    nyc_cellstemp <- nyc_cells
    nyc_cellstemp$homcount <- hom_countdf$hom; nyc_cellstemp$gtacount <- gta_countdf$gta
    
    # First plot the grid crime data
    pdf(paste0("NYHom2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    plot(nyc_cellstemp["homcount"], main=paste0("Map of Homicides in New York\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    dev.off()
    
    pdf(paste0("NYGTA2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    plot(nyc_cellstemp["gtacount"], main=paste0("Map of Motor Vehicle Thefts in New York\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    dev.off()
    
    # Heat Maps for Socio-economic Variables
    # First plot the census tract then grid socio-economic variables for each variable
    ct_NY.projtemp <- ct_NY.proj
    ct_NY.projtemp$pop <- NY_ctpop_15$pop[match(ct_NY.proj$GEOID,NY_ctpop_15$geoid2)]
    ct_NY.projtemp$inc <- NY_ctinc_15$inc[match(ct_NY.proj$GEOID,NY_ctinc_15$geoid2)]
    # ct_NY.projtemp$msex <- NY_ctsex_15$mprop[match(ct_NY.proj$GEOID,NY_ctsex_15$geoid2)]
    # ct_NY.projtemp$fsex <- NY_ctsex_15$fprop[match(ct_NY.proj$GEOID,NY_ctsex_15$geoid2)]
    # ct_NY.projtemp$own <- NY_ctocc_15$ownprop[match(ct_NY.proj$GEOID,NY_ctocc_15$geoid2)]
    # ct_NY.projtemp$rent <- NY_ctocc_15$rentprop[match(ct_NY.proj$GEOID,NY_ctocc_15$geoid2)]
    # ct_NY.projtemp$food <- NY_ctfood_15$proprec[match(ct_NY.proj$GEOID,NY_ctfood_15$geoid2)]
    
    nyc_cellstemp$pop <- hom_countdf$pop
    nyc_cellstemp$inc <- hom_countdf$inc
    # nyc_cellstemp$msex <- hom_countdf$msex
    # nyc_cellstemp$fsex <- hom_countdf$fsex
    # nyc_cellstemp$own <- hom_countdf$own
    # nyc_cellstemp$rent <- hom_countdf$rent
    # nyc_cellstemp$food <- hom_countdf$food
    
    
    # Population ##
    # Census Tract Heat Map: Population
    pdf("NYPop2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(ct_NY.projtemp["pop"], main=NULL,cex.main=2)
    mtext("2015 Population in New York Census Tracts",side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    # Grid Heat Map: Population
    pdf(paste0("NYPop2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(nyc_cellstemp["pop"], main=NULL,cex.main=2)
    mtext(paste0("2015 Population in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    
    # Average Income ##
    # Census Tract Heat Map: Average Income
    pdf("NYIncImp2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(ct_NY.projtemp["inc"], main=NULL,cex.main=2)
    mtext("2015 Mean Income in New York Census Tracts",side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    # Grid Heat Map: Average Income
    pdf(paste0("NYIncImp2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(nyc_cellstemp["inc"], main=NULL,cex.main=2)
    mtext(paste0("2015 Average Income in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    
    # # Sex ##
    # # Census Tract Heat Map: Sex
    # pdf("NYSex2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(ct_NY.projtemp["msex"], main=NULL,cex.main=1.4)
    # mtext("2015 Proportion of Males\n in New York Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # plot(ct_NY.projtemp["fsex"], main=NULL,cex.main=1.4)
    # mtext("2015 Proportion of Females\n in New York Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # # Grid Heat Map: Sex
    # pdf(paste0("NYSex2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(nyc_cellstemp["msex"], main=NULL,cex.main=1.4)
    # mtext(paste0("2015 Proportion of Males\n in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # plot(nyc_cellstemp["fsex"], main=NULL,cex.main=1.4)
    # mtext(paste0("2015 Proportion of Females\n in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # 
    # # Home Occupier ##
    # # Census Tract Heat Map: Owned and Rented Properties
    # pdf("NYOcc2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(ct_NY.projtemp["own"], main=NULL,cex.main=1.3)
    # mtext("2015 Proportion Properties\n Owned in New York Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # plot(ct_NY.projtemp["rent"], main=NULL,cex.main=1.3)
    # mtext("2015 Proportion Properties\n Rented in New York Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # # Grid Heat Map: Owned and Rented Properties
    # pdf(paste0("NYOcc2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(nyc_cellstemp["own"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Proportion Properties\n Owned in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # plot(nyc_cellstemp["rent"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Proportion Properties\n Rented in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # 
    # # Food Stamps ##
    # # Census Tract Heat Map: Food Stamps
    # pdf("NYFood2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(ct_NY.projtemp["food"], main=NULL,cex.main=1.3)
    # mtext("2015 Percentage of Houses Receiving\n Food Stamps in New York Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # # Grid Heat Map: Food Stamps
    # pdf(paste0("NYFood2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(nyc_cellstemp["food"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Percentage of Houses Receiving\n Food Stamps in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
  }
}

grid_cellsxvec <- c(10,24,47,94)
grid_cellsyvec <- c(10,24,48,96)

for (i in 1:4){
  plot_nyc_grid(grid_cellsx=grid_cellsxvec[i],grid_cellsy=grid_cellsyvec[i],standard=TRUE)
}

# Portland ----------------------------------------------------------------

# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))
setwd("./Portland")

load("../../PROCESSED_DATA/SHAPEFILES/PCityCT.rda")
ct_P.proj <- lwgeom::st_transform_proj(ct_P,"epsg:32610")

xy.coord <- sapply(1:length(ct_P.proj$geometry),function(i){return(as.numeric(st_centroid(ct_P.proj$geometry[[i]])))}) # returns a 2xlength(b$geometry) matrix, where each column is for each geometry and row one is longitude and row two is latitude

ct_P.proj$x <- xy.coord[1,] # [1] 
ct_P.proj$y <- xy.coord[2,] # [1] 

# Load in the crime data that has processed
p_gta <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/Portland/p_gta.rds")
p_hom <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/Portland/p_hom.rds")
p_hom_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/Portland/p_hom_sf.rds")
p_gta_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/Portland/p_gta_sf.rds")
p_hom_sf.proj <- lwgeom::st_transform_proj(p_hom_sf,"epsg:32610")
p_gta_sf.proj <- lwgeom::st_transform_proj(p_gta_sf,"epsg:32610")

p_gta15 <- p_gta[which(p_gta$Y==2015),]
p_hom15 <- p_hom[which(p_hom$Y==2015),]

# Portland: Simple Summaries ----------------------------------------------

summary(p_gta15)
dim(p_gta15)
# [1] 3002   23
summary(p_hom15)
dim(p_hom15)
# [1]  1 23

p_gta15_sf.proj <- p_gta_sf.proj[which(p_gta_sf.proj$Y==2015),]
p_hom15_sf.proj <- p_hom_sf.proj[which(p_hom_sf.proj$Y==2015),]

p_hom15_sp.proj <- as(p_hom15_sf.proj,"Spatial")
p_gta15_sp.proj <- as(p_gta15_sf.proj,"Spatial")

# Incidents per Year
hom.year <- ggplot(data=p_hom,aes(Y)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1)) + ggtitle("Homicides per Year in New York")
gta.year <- ggplot(data=p_gta,aes(Y)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1)) + ggtitle("Motor Vehicle Thefts per Year in New York")

pdf("P_CrimevYear_proj_final.pdf",h=8,w=10,pointsize=14)
grid.arrange(hom.year,gta.year,ncol=2)
dev.off()


# Consider Y=2015 and consider Incidents per Month
hom.month15 <- ggplot(data=p_hom[p_hom$Y==2015,],aes(MY)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1)) + ggtitle("Homicides per Month in New York during 2015")
gta.month15 <- ggplot(data=p_hom[p_hom$Y==2015,],aes(MY)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1)) + ggtitle("Motor Vehicle Thefts per Month in New York during 2015")

pdf("P_CrimevMonth2015_proj_final.pdf",h=8,w=10,pointsize=14)
grid.arrange(hom.month15,gta.month15,ncol=2)
dev.off()

# There were several cases that classified for homicide in Portland, what proportion of each class is there?
table(p_hom$`Offense Type`)
# HOMICIDE-NEGLIGENT-VEHICLE  HOMICIDE-NEGLIGENT,UNCLASSIFIE MURDER & NON-NEGL. MANSLAUGHTER 
# 99                             137                            4748 
table(p_hom[which(p_hom$Y==2015),]$`Offense Type`)
# HOMICIDE-NEGLIGENT-VEHICLE  HOMICIDE-NEGLIGENT,UNCLASSIFIE MURDER & NON-NEGL. MANSLAUGHTER 
# 3                               9                             345 
crime_phom.plot <- ggplot(p_hom,aes(`Offense Type`)) + geom_bar() + theme(axis.text.x=element_text(angle=90,hjust=1))
crime_phom.plot

# Portland: Spatial Plots -------------------------------------------------

# Portland: Map Plots -----------------------------------------------------

# Bounding box for stamen map function
latp_low <- min(min(p_gta$lat),min(p_hom$lat)) - 0.1
lonp_low <- min(min(p_gta$lon),min(p_hom$lon)) - 0.1
latp_up <- max(max(p_gta$lat),max(p_hom$lat)) + 0.1
lonp_up <- max(max(p_gta$lon),max(p_hom$lon)) + 0.1

# Produce base map plot of NY to add anything else too.
gm_P <- get_stamenmap(bbox=c(left=(lonp_low),right=(lonp_up),top=(latp_up),bottom=(latp_low)),zoom=10,maptype = "toner",source="stamen")
plot(gm_P)

P_map <- ggmap(gm_P)


P_map_GTA <- P_map + geom_point(data = p_gta,aes(x=lon,y=lat), colour="red", size=0.1) +
  ggtitle("Plot of Locations of Stolen Vehicle Incidents in Portland")
P_map_GTA

pdf("P_GTAMap15_final.pdf",h=8,w=10,pointsize=14)
P_map_GTA2 <- P_map + geom_point(data = p_gta15,aes(x=lon,y=lat), colour="red", size=1) +
  ggtitle("Plot of Locations of Stolen Vehicle Incidents in Portland")
P_map_GTA2
dev.off()

P_map_Hom <- P_map + geom_point(data = p_hom,aes(x=lon,y=lat), colour="red", size=0.1) + ggtitle("Plot of Locations of Homicide Incidents in Portland")
P_map_Hom
pdf("P_HomMap15_final.pdf",h=8,w=10,pointsize=14)
P_map_Hom2 <- P_map + geom_point(data = p_hom15,aes(x=lon,y=lat), colour="red", size=1) + ggtitle("Plot of Locations of Homicide Incidents in Portland")
P_map_Hom2
dev.off()

pdf("P_Crime_Maps_All_final.pdf",h=8,w=10,pointsize=14)
grid.arrange(P_map_Hom,P_map_GTA,ncol=2)
dev.off()

P_map_GTA <- P_map + geom_point(data = p_gta15,aes(x=lon,y=lat), colour="red", size=0.1) +
  ggtitle("Plot of Locations of Stolen Vehicle Incidents in Portland in 2015")
P_map_GTA

P_map_Hom <- P_map + geom_point(data = p_hom15,aes(x=lon,y=lat), colour="red", size=0.1) + ggtitle("Plot of Locations of Homicide Incidents in Portland in 2015")
P_map_Hom

pdf("P_Crime_Maps_2015_final.pdf",h=8,w=10,pointsize=14)
grid.arrange(P_map_Hom,P_map_GTA,ncol=2)
dev.off()

# Census Tract Plots
pdf("P_CT_final.pdf",h=8,w=10,pointsize = 14)
plot(st_geometry(ct_P), col = sf.colors(20, categorical = TRUE), border = 'grey', 
     axes = FALSE, bty="n", main="Portland City Census Tracts")
dev.off()
pdf("P_CT_proj_final.pdf",h=8,w=10,pointsize = 14)
plot(st_geometry(ct_P.proj), col = sf.colors(20, categorical = TRUE), border = 'grey', 
     axes = FALSE, bty="n", main="Portland City Census Tracts")
dev.off()

pdf("PTHom2015_proj_final.pdf",h=8,w=10,pointsize = 14)
plot(st_geometry(ct_P.proj), col = sf.colors(20, categorical = TRUE), border = 'grey', 
     axes = FALSE, bty="n", main="2015 Homicide Incidents in Portland",cex.main=2)
plot(st_geometry(p_hom15_sf.proj),add=TRUE,pch=19,col="red")
dev.off()

pdf("PTGTA2015_proj_final.pdf",h=8,w=10,pointsize = 14)
plot(st_geometry(ct_P.proj), col = sf.colors(20, categorical = TRUE), border = 'grey', 
     axes = FALSE, bty="n", main="2015 Motor Vehicle Theft Incidents in Portland",cex.main=2)
plot(st_geometry(p_gta15_sf.proj),add=TRUE,pch=19,col="red")
dev.off()



# Portland: Plotting Gridded Data along with Census Tract Data ------------

# Create a wrapper function to plot the required socio-economic variables and crime gridded data where the input allows for us to alter the grid resolution we are interested in without replicating the code multiple times for each, just requires a call to this function for each resolution of interest. We can either use st_geometry() in the plot function with alterations to the binning and the "standard" plot just uses the plot function to extract the relevant variable.
# As well as the average income and population variables, we originally included a few other socio-economic variables however these were not included in any of the modelling of the crime data and updated data generation code has these commented out and so I will comment out the lines including these variables but retain the code so that it is complete as was originally run. If they are of interest, they can easily be uncommented to produce the relevant plots.

plot_p_grid <- function(grid_cellsx,grid_cellsy,standard=FALSE){
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/Portland/P2015CT",grid_cellsx,grid_cellsy,"CountData_projFinal.rda"))
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/Portland/PGridCells",grid_cellsx,grid_cellsy,"_projFinal.rda"))
  
  P_ctpop_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTPop_15_proj.rds")
  P_ctinc_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTInc_15_0imp_proj.rds")
  # P_ctsex_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTSex_15_proj.rds")
  # P_ctage_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTAge_15_proj.rds")
  # P_ctocc_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTOcc_15_proj.rds")
  # P_ctfood_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTFood_15_proj.rds")
  
  # Plot the gridded crime counts against the socio-economic variables interpolated on to the grids.  
  # Average Income
  pdf(paste0("PHomCountvIncImp",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=10,w=20,pointsize = 14)
  plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$inc,main="Counts of Homicide Normalised by Population Counts against\n Average Income",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Average Income")
  dev.off()
  
  pdf(paste0("PGTACountvIncImp",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=10,w=20,pointsize = 14)
  plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$inc,main="Counts of Motor Vehicle Theft Normalised by Population Counts against\n Average Income",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Average Income")
  dev.off()
  
  # # Sex
  # pdf(paste0("PHomCountvSex",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$msex,main="Counts of Homicide Normalised by Population Counts against Proportion of Males",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Males")
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$fsex,main="Counts of Homicide Normalised by Population Counts against Proportion of Females",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Females")
  # dev.off()
  # 
  # pdf(paste0("PGTACountvSex",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$msex,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Males",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Males")
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$fsex,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Females",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Females")
  # dev.off()
  # 
  # # Occupied
  # pdf(paste0("PHomCountvOcc",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$own,main="Counts of Homicide Normalised by Population Counts against Proportion of Properties Owned",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Owned Properties")
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$rent,main="Counts of Homicide Normalised by Population Counts against Proportion of Properties Rented",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Rented Properties")
  # dev.off()
  # 
  # pdf(paste0("PGTACountvOcc",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$own,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Properties Owned",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Owned Properties")
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$rent,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Properties Rented",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Rented Properties")
  # dev.off()
  # 
  # # Food
  # pdf(paste0("PHomCountvFood",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=10,w=20,pointsize = 14)
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$food,main="Counts of Homicide Normalised by Population Counts against\n Percentage of Households Receiving Food Stamps",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Percentage of Households Receiving Food Stamps")
  # dev.off()
  # 
  # pdf(paste0("PGTACountvFood",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=10,w=20,pointsize = 14)
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$food,main="Counts of Motor Vehicle Theft Normalised by Population Counts against\n Percentage of Households Receiving Food Stamps",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Percentage of Households Receiving Food Stamps")
  # dev.off()
  
  if (standard==FALSE){ # want to create our own bins for the heat maps
    p_cells$homcount <- hom_countdf$hom; p_cells$gtacount <- gta_countdf$gta
    p_cells.temp <- p_cells
    
    # First we will plot the grid crime data
    bin.upperlim <- max(p_cells.temp$homcount)
    
    # Bins for different colours 
    p_cells.temp$bin <- cut(p_cells.temp$homcount, c(-0.5,0.5,1.5,2.5,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(p_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    
    # Re-name because I wanted to separate out 0, 1 and 2 homicides...
    my.levels[1] <- " 0"
    my.levels[2] <- " 1"
    my.levels[3] <- " 2"
    my.levels[4] <- " 3+"
    
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(p_cells.temp$bin))))
    p_cells.temp$Col <- a[as.numeric(p_cells.temp$bin)]
    
    pdf(paste0("PHom2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(p_cells.temp), col = p_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("Map of Homicides in Portland\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(p_cells.temp$Col),title=expression("Homicide"),cex=1,xpd=NA)
    dev.off()
    
    bin.upperlim <- max(p_cells.temp$gtacount)
    
    # Bins for different colours 
    p_cells.temp$bin <- cut(p_cells.temp$gtacount, c(-0.5,0.5,1.5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(p_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    
    # Re-name levels
    length(my.levels)
    my.levels[1] <- " 0"
    my.levels[2] <- " 1"
    my.levels[3] <- " 2 to 10"
    my.levels[length(my.levels)] <- " 75+"
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(p_cells.temp$bin))))
    p_cells.temp$Col <- a[as.numeric(p_cells.temp$bin)]
    
    pdf(paste0("PGTA2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(p_cells.temp), col = p_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("Map of Motor Vehicle Thefts in Portland\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(p_cells.temp$Col),title=expression("MVT"),cex=1,xpd=NA)
    dev.off()
    
    
    # Heat Maps for Socio-economic Variables
    # First plot the census tract then grid socio-economic variables for each variable
    
    # Population ##
    P_ctpop_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTPop_15_proj.rds")
    
    # Census Tract Heat Map: Population
    if (round(max(P_ctpop_15$pop)/5000)!=max(P_ctpop_15$pop)/5000){
      bin.upperlim <-5000*(round(max(P_ctpop_15$pop)/5000)+1)
    } else {
      bin.upperlim <- max(P_ctpop_15$pop)
    }
    
    # Bins for different colours 
    P_ctpop_15$bin <- cut(P_ctpop_15$pop, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(P_ctpop_15$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(P_ctpop_15$bin))))
    P_ctpop_15$Col <- a[as.numeric(P_ctpop_15$bin)]
    
    pdf("PPop2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(ct_P.proj), col = P_ctpop_15$Col, border = 'black', axes = FALSE, bty="n", main="2015 Population in Portland Census Tracts",cex.main=2)
    legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(P_ctpop_15$Col),title=expression("Population (" *10^{3}*")"),cex=1,xpd=NA)
    dev.off()
    
    # Grid Heat Map: Population
    p_cells.temp <- p_cells
    p_cells.temp$pop <- hom_countdf$pop
    if (round(max(p_cells.temp$pop)/5000)!=max(p_cells.temp$pop)/5000){
      bin.upperlim <- 5000*(round(max(p_cells.temp$pop)/5000)+1)
    } else {
      bin.upperlim <- max(p_cells.temp$pop)
    }
    
    # Bins for different colours 
    p_cells.temp$bin <- cut(p_cells.temp$pop, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(p_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(p_cells.temp$bin))))
    p_cells.temp$Col <- a[as.numeric(p_cells.temp$bin)]
    
    pdf(paste0("PPop2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(p_cells), col = p_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Population in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(p_cells.temp$Col),title=expression("Population (" *10^{3}*")"),cex=1,xpd=NA)
    dev.off()
    
    
    # Average Income ##
    # Census Tract Heat Map: Average Income
    if (round(max(P_ctinc_15$inc,na.rm = TRUE)/1e4)!=max(P_ctinc_15$inc,na.rm = TRUE)/1e4){
      bin.upperlim <-1e4*(round(max(P_ctinc_15$inc,na.rm = TRUE)/1e4)+1)
    } else {
      bin.upperlim <- max(P_ctinc_15$inc,na.rm = TRUE)
    }
    
    
    P_ctinc_15$bin <- cut(P_ctinc_15$inc,seq(0,bin.upperlim,length.out=6), include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(P_ctinc_15$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("0000","",lev5)
    my.levels <- lev6
    
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(P_ctinc_15$bin))))
    P_ctinc_15$Col <- a[as.numeric(P_ctinc_15$bin)]
    
    pdf("PIncImp2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(ct_P.proj), col = P_ctinc_15$Col, border = 'black', axes = FALSE, bty="n", main="2015 Mean Income in Portland Census Tracts",cex.main=2)
    legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(P_ctinc_15$Col),title=expression("Income (" *10^{4}*")"),cex=1,xpd=NA)
    dev.off()
    
    
    # Grid Heat Map: Average Income
    p_cells.temp <- p_cells
    p_cells.temp$inc <- hom_countdf$inc
    if (round(max(p_cells.temp$inc)/1e4)!=max(p_cells.temp$inc)/1e4){
      bin.upperlim <- 1e4*(round(max(p_cells.temp$inc)/1e4)+1)
    } else {
      bin.upperlim <- max(p_cells.temp$inc)
    }
    
    # Bins for different colours 
    p_cells.temp$bin <- cut(p_cells.temp$inc,seq(0,bin.upperlim,length.out=6), include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(p_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("0000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(p_cells.temp$bin))))
    p_cells.temp$Col <- a[as.numeric(p_cells.temp$bin)]
    
    pdf(paste0("PIncImp2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(p_cells), col = p_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Average Income in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(p_cells.temp$Col),title=expression("Income (" *10^{4}*")"),cex=1,xpd=NA)
    dev.off()
    
    
    # # Sex ##
    # # Census Tract Heat Map: Sex
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # P_ctsex_15$binm <- cut(P_ctsex_15$mprop, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # P_ctsex_15$binf <- cut(P_ctsex_15$fprop, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(P_ctsex_15$binm)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # am <- sort(rbPal(length(levels(P_ctsex_15$binm))))
    # P_ctsex_15$Colm <- am[as.numeric(P_ctsex_15$binm)]
    # af <- sort(rbPal(length(levels(P_ctsex_15$binf))))
    # P_ctsex_15$Colf <- af[as.numeric(P_ctsex_15$binf)]
    # 
    # pdf("PSex2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(ct_P.proj), col = P_ctsex_15$Colm, border = 'black', axes = FALSE, bty="n", main="2015 Proportion of Males\n in Portland Census Tracts",cex.main=1.4)
    # plot(st_geometry(ct_P.proj), col = P_ctsex_15$Colf, border = 'black', axes = FALSE, bty="n", main="2015 Proportion of Females\n in Portland Census Tracts",cex.main=1.4)
    # legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = am, legend = my.levels, col = unique(P_ctsex_15$Colm),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # # Grid Heat Map: Sex
    # p_cells.temp <- p_cells
    # p_cells.temp$m <- hom_countdf$msex
    # p_cells.temp$f <- hom_countdf$fsex
    # 
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # p_cells.temp$binm <- cut(p_cells.temp$m, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # p_cells.temp$binf <- cut(p_cells.temp$f, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(p_cells.temp$binm)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # am <- sort(rbPal(length(levels(p_cells.temp$binm))))
    # p_cells.temp$Colm <- am[as.numeric(p_cells.temp$binm)]
    # af <- sort(rbPal(length(levels(p_cells.temp$binf))))
    # p_cells.temp$Colf <- af[as.numeric(p_cells.temp$binf)]
    # 
    # pdf(paste0("PSex2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(p_cells), col = p_cells.temp$Colm, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion of Males\n in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.4)
    # plot(st_geometry(p_cells), col = p_cells.temp$Colf, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion of Females\n in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.4)
    # legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = am, legend = my.levels, col = unique(p_cells.temp$Colm),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # 
    # # Home Occupier ##
    # # Census Tract Heat Map: Owned and Rented Properties
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # P_ctocc_15$binown <- cut(P_ctocc_15$ownprop, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # P_ctocc_15$binrent <- cut(P_ctocc_15$rentprop, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(P_ctocc_15$binown)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # aown <- sort(rbPal(length(levels(P_ctocc_15$binown))))
    # P_ctocc_15$Colown <- aown[as.numeric(P_ctocc_15$binown)]
    # arent <- sort(rbPal(length(levels(P_ctocc_15$binrent))))
    # P_ctocc_15$Colrent <- arent[as.numeric(P_ctocc_15$binrent)]
    # 
    # pdf("POcc2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(ct_P.proj), col = P_ctocc_15$Colown, border = 'black', axes = FALSE, bty="n", main="2015 Proportion Properties\n Owned in Portland Census Tracts",cex.main=1.3)
    # plot(st_geometry(ct_P.proj), col = P_ctocc_15$Colrent, border = 'black', axes = FALSE, bty="n", main="2015 Proportion Properties\n Rented in Portland Census Tracts",cex.main=1.3)
    # legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = aown, legend = my.levels, col = unique(P_ctocc_15$Colown),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # # Grid Heat Map: Owned and Rented Properties
    # p_cells.temp <- p_cells
    # p_cells.temp$own <- hom_countdf$own
    # p_cells.temp$rent <- hom_countdf$rent
    # 
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # p_cells.temp$binown <- cut(p_cells.temp$own, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # p_cells.temp$binrent <- cut(p_cells.temp$rent, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(p_cells.temp$binown)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # aown <- sort(rbPal(length(levels(p_cells.temp$binown))))
    # p_cells.temp$Colown <- aown[as.numeric(p_cells.temp$binown)]
    # arent <- sort(rbPal(length(levels(p_cells.temp$binrent))))
    # p_cells.temp$Colrent <- arent[as.numeric(p_cells.temp$binrent)]
    # 
    # pdf(paste0("POcc2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(p_cells), col = p_cells.temp$Colown, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion Properties\n Owned in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # plot(st_geometry(p_cells), col = p_cells.temp$Colrent, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion Properties\n Rented in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = aown, legend = my.levels, col = unique(p_cells.temp$Colown),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # 
    # # Food Stamps ##
    # # Census Tract Heat Map: Food Stamps
    # bin.upperlim <- 100
    # 
    # # Bins for different colours 
    # P_ctfood_15$bin <- cut(P_ctfood_15$proprec, c(-0.5,0,5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(P_ctfood_15$bin)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # my.levels[1] <- " 0"
    # 
    # # Create a function to generate a continuous color palette
    # a <- sort(rbPal(length(levels(P_ctfood_15$bin))))
    # P_ctfood_15$Col <- a[as.numeric(P_ctfood_15$bin)]
    # 
    # pdf("PFood2015CT_proj_final.pdf",h=8,w=10,pointsize=14)
    # par(oma=c(0,0,0,8))
    # plot(st_geometry(ct_P.proj), col = P_ctfood_15$Col, border = 'black', axes = FALSE, bty="n", main="2015 Percentage of Houses Receiving\n Food Stamps in Portland Census Tracts",cex.main=1.3)
    # legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(P_ctfood_15$Col),title=expression("Percentage"),cex=1,xpd=NA)
    # dev.off()
    # 
    # # Grid Heat Map: Food Stamps
    # p_cells.temp <- p_cells
    # p_cells.temp$food <- hom_countdf$food
    # 
    # bin.upperlim <- 100 # 100%
    # # Bins for different colours 
    # p_cells.temp$bin <- cut(p_cells.temp$food, c(-0.5,0,5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(p_cells.temp$bin)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # my.levels[1] <- " 0"
    # 
    # # Create a function to generate a continuous color palette
    # a <- sort(rbPal(length(levels(p_cells.temp$bin))))
    # p_cells.temp$Col <- a[as.numeric(p_cells.temp$bin)]
    # 
    # pdf(paste0("PFood2015Grid",grid_cellsx,grid_cellsy,"_proj_final.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,0,8))
    # plot(st_geometry(p_cells), col = p_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Percentage of Houses Receiving\n Food Stamps in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(p_cells.temp$Col),title=expression("Percentage"),cex=1,xpd=NA)
    # dev.off()
  } else {
    p_cellstemp <- p_cells
    p_cellstemp$homcount <- hom_countdf$hom; p_cellstemp$gtacount <- gta_countdf$gta
    
    # First plot the grid crime data
    pdf(paste0("PHom2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    plot(p_cellstemp["homcount"], main=paste0("Map of Homicides in Portland\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    dev.off()
    
    pdf(paste0("PGTA2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    plot(p_cellstemp["gtacount"], main=paste0("Map of Motor Vehicle Thefts in Portland\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    dev.off()
    
    # Heat Maps for Socio-economic Variables
    # First plot the census tract then grid socio-economic variables for each variable
    ct_P.projtemp <- ct_P.proj
    ct_P.projtemp$pop <- P_ctpop_15$pop[match(ct_P.proj$GEOID,P_ctpop_15$geoid2)]
    ct_P.projtemp$inc <- P_ctinc_15$inc[match(ct_P.proj$GEOID,P_ctinc_15$geoid2)]
    # ct_P.projtemp$msex <- P_ctsex_15$mprop[match(ct_P.proj$GEOID,P_ctsex_15$geoid2)]
    # ct_P.projtemp$fsex <- P_ctsex_15$fprop[match(ct_P.proj$GEOID,P_ctsex_15$geoid2)]
    # ct_P.projtemp$own <- P_ctocc_15$ownprop[match(ct_P.proj$GEOID,P_ctocc_15$geoid2)]
    # ct_P.projtemp$rent <- P_ctocc_15$rentprop[match(ct_P.proj$GEOID,P_ctocc_15$geoid2)]
    # ct_P.projtemp$food <- P_ctfood_15$proprec[match(ct_P.proj$GEOID,P_ctfood_15$geoid2)]
    
    p_cellstemp$pop <- hom_countdf$pop
    p_cellstemp$inc <- hom_countdf$inc
    # p_cellstemp$msex <- hom_countdf$msex
    # p_cellstemp$fsex <- hom_countdf$fsex
    # p_cellstemp$own <- hom_countdf$own
    # p_cellstemp$rent <- hom_countdf$rent
    # p_cellstemp$food <- hom_countdf$food
    
    
    # Population ##
    # Census Tract Heat Map: Population
    pdf("PPop2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(ct_P.projtemp["pop"], main=NULL,cex.main=2)
    mtext("2015 Population in Portland\n Census Tracts",side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    # Grid Heat Map: Population
    pdf(paste0("PPop2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(p_cellstemp["pop"], main=NULL,cex.main=2)
    mtext(paste0("2015 Population in Portland\n on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    
    # Average Income ##
    # Census Tract Heat Map: Average Income
    pdf("PIncImp2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(ct_P.projtemp["inc"], main=NULL,cex.main=2)
    mtext("2015 Mean Income in Portland\n Census Tracts",side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    # Grid Heat Map: Average Income
    pdf(paste0("PIncImp2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(p_cellstemp["inc"], main=NULL,cex.main=2)
    mtext(paste0("2015 Average Income in Portland\n on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    
    # # Sex ##
    # # Census Tract Heat Map: Sex
    # pdf("PSex2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(ct_P.projtemp["msex"], main=NULL,cex.main=1.4)
    # mtext("2015 Proportion of Males\n in Portland Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # plot(ct_P.projtemp["fsex"], main=NULL,cex.main=1.4)
    # mtext("2015 Proportion of Females\n in Portland Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # # Grid Heat Map: Sex
    # pdf(paste0("PSex2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(p_cellstemp["msex"], main=NULL,cex.main=1.4)
    # mtext(paste0("2015 Proportion of Males\n in Portland on\n ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # plot(p_cellstemp["fsex"], main=NULL,cex.main=1.4)
    # mtext(paste0("2015 Proportion of Females\n in Portland on\n ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # 
    # # Home Occupier ##
    # # Census Tract Heat Map: Owned and Rented Properties
    # pdf("POcc2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(ct_P.projtemp["own"], main=NULL,cex.main=1.3)
    # mtext("2015 Proportion Properties\n Owned in Portland\n Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # plot(ct_P.projtemp["rent"], main=NULL,cex.main=1.3)
    # mtext("2015 Proportion Properties\n Rented in Portland\n Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # # Grid Heat Map: Owned and Rented Properties
    # pdf(paste0("POcc2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(p_cellstemp["own"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Proportion Properties\n Owned in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # plot(p_cellstemp["rent"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Proportion Properties\n Rented in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # 
    # # Food Stamps ##
    # # Census Tract Heat Map: Food Stamps
    # pdf("PFood2015CT_projSFPlot_final.pdf",h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(ct_P.projtemp["food"], main=NULL,cex.main=1.3)
    # mtext("2015 Percentage of Houses Receiving\n Food Stamps in Portland\n Census Tracts",side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # # Grid Heat Map: Food Stamps
    # pdf(paste0("PFood2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_final.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(p_cellstemp["food"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Percentage of Houses Receiving\n Food Stamps in Portland\n on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
  }
}

grid_cellsxvec <- c(8,19,38,76)
grid_cellsyvec <- c(6,13,26,52)

for (i in 1:4){
  plot_p_grid(grid_cellsx=grid_cellsxvec[i],grid_cellsy=grid_cellsyvec[i],standard=TRUE)
}


# Session Information -----------------------------------------------------

sessionInfo()
