
# Introduction ------------------------------------------------------------

## Exploratory Data Analysis for the Homicide and Motor Vehicle Theft for Los Angeles, New York, Portland and Fayetteville and with the covariate data on the census tract level.
# Note that the only socio-economic variables used within my thesis were the total population and average income and so the final R code for data generation was tailored accordingly, with the code for the inclusion of other socio-economic variables commented out and we do the same below. For any additional variables we comment out they're inclusion and plot outputs and this can be easily uncommented if required to generate the relevant plots.

# Author: Nadeen Khaleel


# Set WD and Load Libraries -----------------------------------------------

library("rstudioapi")
library(ggplot2)
library(gridExtra)
library(ggmap)
library(dplyr)
library(zipcode)
library(readr)
library(revgeo)
library(sf)
library(maptools)

rbPal <- colorRampPalette(c('blue','red','yellow'))

# Los Angeles -------------------------------------------------------------

# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))
setwd("./LA")

load("../../PROCESSED_DATA/SHAPEFILES/LACityCT.rda")
ct_LA.proj <- lwgeom::st_transform_proj(ct_LA,"epsg:32611")

xy.coord <- sapply(1:length(ct_LA.proj$geometry),function(i){return(as.numeric(st_centroid(ct_LA.proj$geometry[[i]])))}) # returns a 2xlength(b$geometry) matrix, where each column is for each geometry and row one is longitude and row two is latitude
ct_LA.proj$x <- xy.coord[1,]
ct_LA.proj$y <- xy.coord[2,]

# Load in the crime data that has processed
la_gta <- readRDS("X:/Maths/ResearchProjects/TRSmith/TRSmith2/DATA_LAPTOP/DATA/PROCESSED DATA/CRIME/LA/la_gta.rds")
la_hom <- readRDS("X:/Maths/ResearchProjects/TRSmith/TRSmith2/DATA_LAPTOP/DATA/PROCESSED DATA/CRIME/LA/la_hom.rds")
la_hom_sf <- readRDS("X:/Maths/ResearchProjects/TRSmith/TRSmith2/DATA_LAPTOP/DATA/PROCESSED DATA/CRIME/LA/la_hom_sf.rds")
la_gta_sf <- readRDS("X:/Maths/ResearchProjects/TRSmith/TRSmith2/DATA_LAPTOP/DATA/PROCESSED DATA/CRIME/LA/la_gta_sf.rds")
la_hom_sf.proj <- lwgeom::st_transform_proj(la_hom_sf,"epsg:32611")
la_gta_sf.proj <- lwgeom::st_transform_proj(la_gta_sf,"epsg:32611")

la_gta15 <- la_gta[which(la_gta$Y==2015),]
la_hom15 <- la_hom[which(la_hom$Y==2015),]

la_gta15_sf.proj <- la_gta_sf.proj[which(la_gta_sf.proj$Y==2015),]
la_hom15_sf.proj <- la_hom_sf.proj[which(la_hom_sf.proj$Y==2015),]

la_hom15_sp.proj <- as(la_hom15_sf.proj,"Spatial")
la_gta15_sp.proj <- as(la_gta15_sf.proj,"Spatial")

# Los Angeles: Plotting Shifted and Scaled Gridded Data -------------------

# Create a wrapper function to plot the required socio-economic variables and crime gridded data where the input allows for us to alter the grid resolution we are interested in without replicating the code multiple times for each, just requires a call to this function for each resolution of interest. We can either use st_geometry() in the plot function with alterations to the binning and the "standard" plot just uses the plot function to extract the relevant variable.
# As well as the average income and population variables, we originally included a few other socio-economic variables however these were not included in any of the modelling of the crime data and updated data generation code has these commented out and so I will comment out the lines including these variables but retain the code so that it is complete as was originally run. If they are of interest, they can easily be uncommented to produce the relevant plots.

plot_la_grid <- function(grid_cellsx,grid_cellsy,standard=TRUE){
  
  # Load the required gridded data and socio-economic variables.
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/LA/LA2015CT",grid_cellsx,grid_cellsy,"CountData_projFinalScale.rda"))
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/LA/LAGridCells",grid_cellsx,grid_cellsy,"_projFinal.rda"))
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/LA/LAWindowProjScale.rda"))
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/LA/LAOrdDF",grid_cellsx,grid_cellsy,"_projFinalScale.rda"))
  
  bbox <- lacity_geom.proj@bbox
  lacity_spshift <- elide(lacity_geom.proj,shift=-c(bbox[1,1],bbox[2,1]))
  bbox_shift <- lacity_spshift@bbox
  lacity_spscale <- elide(lacity_spshift,scale=max(bbox_shift)/1e4)
  
  la_cellssp <- as(la_cells,"Spatial")
  la_cells_shift <- elide(la_cellssp,shift=-c(bbox[1,1],bbox[2,1]))
  la_cells_scale <- elide(la_cells_shift ,scale=max(bbox_shift)/1e4)
  
  la_cells.scale <- st_as_sf(la_cells_scale)
  la_cells.scale <- la_cells.scale[ord.df.la$ind,]
  
  LA_ctpop_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTPop_15_proj.rds")
  LA_ctinc_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTinc_15_0imp_proj.rds")
  # LA_ctsex_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTSex_15_proj.rds")
  # LA_ctage_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTAge_15_proj.rds")
  # LA_ctocc_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTOcc_15_proj.rds")
  # LA_ctfood_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTFood_15_proj.rds")
  
  
  # Plot the gridded crime counts against the socio-economic variables interpolated on to the grids.
  # Average Income
  pdf(paste0("LAHomCountvIncImp",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=10,w=20,pointsize = 14)
  plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$inc,main="Counts of Homicide Normalised by Population Counts against\n Average Income",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Average Income")
  dev.off()
  
  pdf(paste0("LAGTACountvIncImp",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=10,w=20,pointsize = 14)
  plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$inc,main="Counts of Motor Vehicle Theft Normalised by Population Counts against\n Average Income",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Average Income")
  dev.off()
  
  # # Sex
  # pdf(paste0("LAHomCountvSex",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$msex,main="Counts of Homicide Normalised by Population Counts against Proportion of Males",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Males")
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$fsex,main="Counts of Homicide Normalised by Population Counts against Proportion of Females",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Females")
  # dev.off()
  # 
  # pdf(paste0("LAGTACountvSex",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$msex,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Males",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Males")
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$fsex,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Females",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Females")
  # dev.off()
  # 
  # # Occupied
  # pdf(paste0("LAHomCountvOcc",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$own,main="Counts of Homicide Normalised by Population Counts against Proportion of Properties Owned",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Owned Properties")
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$rent,main="Counts of Homicide Normalised by Population Counts against Proportion of Properties Rented",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Rented Properties")
  # dev.off()
  # 
  # pdf(paste0("LAGTACountvOcc",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$own,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Properties Owned",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Owned Properties")
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$rent,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Properties Rented",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Rented Properties")
  # dev.off()
  # 
  # # Food
  # pdf(paste0("LAHomCountvFood",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=10,w=20,pointsize = 14)
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$food,main="Counts of Homicide Normalised by Population Counts against\n Percentage of Households Receiving Food Stamps",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Percentage of Households Receiving Food Stamps")
  # dev.off()
  # 
  # pdf(paste0("LAGTACountvFood",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=10,w=20,pointsize = 14)
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$food,main="Counts of Motor Vehicle Theft Normalised by Population Counts against\n Percentage of Households Receiving Food Stamps",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Percentage of Households Receiving Food Stamps")
  # dev.off()
  
  if (standard==FALSE){ # want to create our own bins for the heat maps
    la_cells.scale$homcount <- hom_countdf$hom; la_cells.scale$gtacount <- gta_countdf$gta
    la_cells.temp <- la_cells.scale
    
    bin.upperlim <- max(la_cells.temp$homcount)
    
    # Bins for different colours 
    la_cells.temp$bin <- cut(la_cells.temp$homcount, c(-0.5,0.5,1.5,2.5,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(la_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    
    # Re-name because I wanted to separate out 0, 1 and 2 homicides...
    my.levels[1] <- " 0"
    my.levels[2] <- " 1"
    my.levels[3] <- " 2"
    my.levels[4] <- " 3+"
    
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(la_cells.temp$bin))))
    la_cells.temp$Col <- a[as.numeric(la_cells.temp$bin)]
    
    pdf(paste0("LAHom2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(la_cells.temp), col = la_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("Map of Homicides in LA\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"))#,cex.main=2)
    legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(la_cells.temp$Col),title=expression("Homicide"),cex=1,xpd=NA)
    dev.off()
    
    
    bin.upperlim <- max(la_cells.temp$gtacount)
    
    # Bins for different colours 
    la_cells.temp$bin <- cut(la_cells.temp$gtacount, c(-0.5,0.5,1.5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(la_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    
    # Re-name levels
    length(my.levels)
    my.levels[1] <- " 0"
    my.levels[2] <- " 1"
    my.levels[3] <- " 2 to 10"
    my.levels[length(my.levels)] <- " 75+"
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(la_cells.temp$bin))))
    la_cells.temp$Col <- a[as.numeric(la_cells.temp$bin)]
    
    pdf(paste0("LAGTA2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(la_cells.temp), col = la_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("Map of Motor Vehicle Thefts in LA\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(la_cells.temp$Col),title=expression("MVT"),cex=1,xpd=NA)
    dev.off()
    
    
    # Heat Maps for Socio-economic Variables
    # Plot the gridded socio-economic variables for each variable
    
    # Population ##
    LA_ctpop_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/LA_CTPop_15_proj.rds")
    
    # Grid Heat Map: Population
    la_cells.temp <- la_cells.scale
    la_cells.temp$pop <- hom_countdf$pop
    if (round(max(la_cells.temp$pop)/5000)!=max(la_cells.temp$pop)/5000){
      bin.upperlim <- 5000*(round(max(la_cells.temp$pop)/5000)+1)
    } else {
      bin.upperlim <- max(la_cells.temp$pop)
    }
    
    # Bins for different colours 
    la_cells.temp$bin <- cut(la_cells.temp$pop, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(la_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(la_cells.temp$bin))))
    la_cells.temp$Col <- a[as.numeric(la_cells.temp$bin)]
    
    pdf(paste0("LAPop2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(la_cells.temp), col = la_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Population in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(la_cells.temp$Col),title=expression("Population (" *10^{3}*")"),cex=1,xpd=NA)
    dev.off()
    
    
    # Average Income ##
    # Grid Heat Map: Average Income
    la_cells.temp <- la_cells.scale
    la_cells.temp$inc <- hom_countdf$inc
    if (round(max(la_cells.temp$inc)/1e5)!=max(la_cells.temp$inc)/1e5){
      bin.upperlim <- 1e5*(round(max(la_cells.temp$inc)/1e5)+1)
    } else {
      bin.upperlim <- max(la_cells.temp$inc)
    }
    
    # Bins for different colours
    la_cells.temp$bin <- cut(la_cells.temp$inc, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(la_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("0000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(la_cells.temp$bin))))
    la_cells.temp$Col <- a[as.numeric(la_cells.temp$bin)]
    
    pdf(paste0("LAIncImp2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(la_cells.temp), col = la_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Average Income in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(la_cells.temp$Col),title=expression("Income (" *10^{4}*")"),cex=1,xpd=NA)
    dev.off()
    
    # # Sex
    # # Grid Heat Map: Sex
    # la_cells.temp <- la_cells.scale
    # la_cells.temp$m <- hom_countdf$msex
    # la_cells.temp$f <- hom_countdf$fsex
    # 
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # la_cells.temp$binm <- cut(la_cells.temp$m, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # la_cells.temp$binf <- cut(la_cells.temp$f, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(la_cells.temp$binm)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # am <- sort(rbPal(length(levels(la_cells.temp$binm))))
    # la_cells.temp$Colm <- am[as.numeric(la_cells.temp$binm)]
    # af <- sort(rbPal(length(levels(la_cells.temp$binf))))
    # la_cells.temp$Colf <- af[as.numeric(la_cells.temp$binf)]
    # 
    # pdf(paste0("LASex2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(la_cells.temp), col = la_cells.temp$Colm, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion of Males\n in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.4)
    # plot(st_geometry(la_cells.temp), col = la_cells.temp$Colf, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion of Females\n in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.4)
    # legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = am, legend = my.levels, col = unique(la_cells.temp$Colm),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # 
    # # Home Occupier
    # # Grid Heat Map: Owned and Rented
    # la_cells.temp <- la_cells.scale
    # la_cells.temp$own <- hom_countdf$own
    # la_cells.temp$rent <- hom_countdf$rent
    # 
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # la_cells.temp$binown <- cut(la_cells.temp$own, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # la_cells.temp$binrent <- cut(la_cells.temp$rent, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(la_cells.temp$binown)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # aown <- sort(rbPal(length(levels(la_cells.temp$binown))))
    # la_cells.temp$Colown <- aown[as.numeric(la_cells.temp$binown)]
    # arent <- sort(rbPal(length(levels(la_cells.temp$binrent))))
    # la_cells.temp$Colrent <- arent[as.numeric(la_cells.temp$binrent)]
    # 
    # pdf(paste0("LAOcc2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(la_cells.temp), col = la_cells.temp$Colown, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion Properties\n Owned in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # plot(st_geometry(la_cells.temp), col = la_cells.temp$Colrent, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion Properties\n Rented in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = aown, legend = my.levels, col = unique(la_cells.temp$Colown),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # 
    # # Food Stamps
    # # Grid Heat Map: Food Stamps
    # la_cells.temp <- la_cells.scale
    # la_cells.temp$food <- hom_countdf$food
    # 
    # bin.upperlim <- 100 # 100%
    # # Bins for different colours 
    # la_cells.temp$bin <- cut(la_cells.temp$food, c(-0.5,0,5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(la_cells.temp$bin)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # my.levels[1] <- " 0"
    # 
    # # Create a function to generate a continuous color palette
    # a <- sort(rbPal(length(levels(la_cells.temp$bin))))
    # la_cells.temp$Col <- a[as.numeric(la_cells.temp$bin)]
    # 
    # pdf(paste0("LAFood2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,0,8))
    # plot(st_geometry(la_cells.temp), col = la_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Percentage of Houses Receiving\n Food Stamps in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # legend(x=4e5,y=3.775e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(la_cells.temp$Col),title=expression("Percentage"),cex=1,xpd=NA)
    # dev.off()
  } else { # standard plots, using the sf format and just a simple plot() function call and no alteration of the bins for the colours
    la_cellstemp <- la_cells.scale
    la_cellstemp$homcount <- hom_countdf$hom; la_cellstemp$gtacount <- gta_countdf$gta
    
    # First plot the grid crime data
    pdf(paste0("LAHom2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(la_cellstemp["homcount"], main=NULL)
    mtext(paste0("Map of Homicides in LA\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    pdf(paste0("LAGTA2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(la_cellstemp["gtacount"], main=NULL)
    mtext(paste0("Map of Motor Vehicle Thefts in LA\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    # Heat Maps for Socio-economic Variables
    # First plot the census tract then grid socio-economic variables for each variable
    ct_LA.projtemp <- ct_LA.proj
    ct_LA.projtemp$pop <- LA_ctpop_15$pop[match(ct_LA.proj$GEOID,LA_ctpop_15$geoid2)]
    ct_LA.projtemp$inc <- LA_ctinc_15$inc[match(ct_LA.proj$GEOID,LA_ctinc_15$geoid2)]
    # ct_LA.projtemp$msex <- LA_ctsex_15$mprop[match(ct_LA.proj$GEOID,LA_ctsex_15$geoid2)]
    # ct_LA.projtemp$fsex <- LA_ctsex_15$fprop[match(ct_LA.proj$GEOID,LA_ctsex_15$geoid2)]
    # ct_LA.projtemp$own <- LA_ctocc_15$ownprop[match(ct_LA.proj$GEOID,LA_ctocc_15$geoid2)]
    # ct_LA.projtemp$rent <- LA_ctocc_15$rentprop[match(ct_LA.proj$GEOID,LA_ctocc_15$geoid2)]
    # ct_LA.projtemp$food <- LA_ctfood_15$proprec[match(ct_LA.proj$GEOID,LA_ctfood_15$geoid2)]
    
    la_cellstemp$pop <- hom_countdf$pop
    la_cellstemp$inc <- hom_countdf$inc
    # la_cellstemp$msex <- hom_countdf$msex
    # la_cellstemp$fsex <- hom_countdf$fsex
    # la_cellstemp$own <- hom_countdf$own
    # la_cellstemp$rent <- hom_countdf$rent
    # la_cellstemp$food <- hom_countdf$food
    
    
    # Population ##
    # Grid Heat Map: Population
    pdf(paste0("LAPop2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(la_cellstemp["pop"], main=NULL) #,cex.main=2)
    mtext(paste0("2015 Population in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    
    # Average Income ##
    # Grid Heat Map: Average Income
    pdf(paste0("LAIncImp2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(la_cellstemp["inc"], main=NULL) #,cex.main=2)
    mtext(paste0("2015 Average Income in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    
    # # Sex ##
    # # Grid Heat Map: Sex
    # pdf(paste0("LASex2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(la_cellstemp["msex"], main=NULL,cex.main=1.4)
    # mtext(paste0("2015 Proportion of Males\n in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # plot(la_cellstemp["fsex"], main=NULL,cex.main=1.4)
    # mtext(paste0("2015 Proportion of Females\n in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # 
    # # Home Occupier ##
    # # Grid Heat Map: Owned and Rented Properties
    # pdf(paste0("LAOcc2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(la_cellstemp["own"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Proportion Properties\n Owned in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # plot(la_cellstemp["rent"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Proportion Properties\n Rented in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # 
    # # Food Stamps ##
    # # Grid Heat Map: Food Stamps
    # pdf(paste0("LAFood2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(la_cellstemp["food"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Percentage of Houses Receiving\n Food Stamps in LA on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
  }
}


grid_cellsxvec <- c(10,24,48,95)
grid_cellsyvec <- c(15,36,72,144)


for (i in 1:4){
  plot_la_grid(grid_cellsx=grid_cellsxvec[i],grid_cellsy=grid_cellsyvec[i],standard=TRUE)
}

# New York ----------------------------------------------------------------

# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))
setwd("./NYC")

load("../../PROCESSED_DATA/SHAPEFILES/NYCityCT.rda")
ct_NY.proj <- lwgeom::st_transform_proj(ct_NY,"epsg:32618")

xy.coord <- sapply(1:length(ct_NY.proj$geometry),function(i){return(as.numeric(st_centroid(ct_NY.proj$geometry[[i]])))}) # returns a 2xlength(b$geometry) matrix, where each column is for each geometry and row one is longitude and row two is latitude

ct_NY.proj$x <- xy.coord[1,]
ct_NY.proj$y <- xy.coord[2,] 


# Load in the crime data that has processed
nyc_gta <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/NYC/nyc_gta.rds")
nyc_hom <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/NYC/nyc_hom.rds")
nyc_hom_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/NYC/nyc_hom_sf.rds")
nyc_gta_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/NYC/nyc_gta_sf.rds")
nyc_hom_sf.proj <- lwgeom::st_transform_proj(nyc_hom_sf,"epsg:32618")
nyc_gta_sf.proj <- lwgeom::st_transform_proj(nyc_gta_sf,"epsg:32618")

nyc_gta15 <- nyc_gta[which(nyc_gta$Y==2015),]
nyc_hom15 <- nyc_hom[which(nyc_hom$Y==2015),]

nyc_gta15_sf.proj <- nyc_gta_sf.proj[which(nyc_gta_sf.proj$Y==2015),]
nyc_hom15_sf.proj <- nyc_hom_sf.proj[which(nyc_hom_sf.proj$Y==2015),]

nyc_hom15_sp.proj <- as(nyc_hom15_sf.proj,"Spatial")
nyc_gta15_sp.proj <- as(nyc_gta15_sf.proj,"Spatial")


# New York: Plotting Shifted and Scaled Gridded Data ----------------------

# Create a wrapper function to plot the required socio-economic variables and crime gridded data where the input allows for us to alter the grid resolution we are interested in without replicating the code multiple times for each, just requires a call to this function for each resolution of interest. We can either use st_geometry() in the plot function with alterations to the binning and the "standard" plot just uses the plot function to extract the relevant variable.
# As well as the average income and population variables, we originally included a few other socio-economic variables however these were not included in any of the modelling of the crime data and updated data generation code has these commented out and so I will comment out the lines including these variables but retain the code so that it is complete as was originally run. If they are of interest, they can easily be uncommented to produce the relevant plots.

plot_nyc_grid <- function(grid_cellsx,grid_cellsy,standard=FALSE){
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/NYC/NY2015CT",grid_cellsx,grid_cellsy,"CountData_projFinalScale.rda"))
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/NYC/NYGridCells",grid_cellsx,grid_cellsy,"_projFinal.rda"))
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/NYC/NYWindowProjScale.rda"))
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/NYC/NYOrdDF",grid_cellsx,grid_cellsy,"_projFinalScale.rda"))
  
  
  bbox <- nycity_geom.proj@bbox
  nycity_spshift <- elide(nycity_geom.proj,shift=-c(bbox[1,1],bbox[2,1]))
  bbox_shift <- nycity_spshift@bbox
  nycity_spscale <- elide(nycity_spshift,scale=max(bbox_shift)/1e4)
  
  nyc_cellssp <- as(nyc_cells,"Spatial")
  nyc_cells_shift <- elide(nyc_cellssp,shift=-c(bbox[1,1],bbox[2,1]))
  nyc_cells_scale <- elide(nyc_cells_shift ,scale=max(bbox_shift)/1e4)
  
  nyc_cells.scale <- st_as_sf(nyc_cells_scale)
  nyc_cells.scale <- nyc_cells.scale[ord.df.ny$ind,]
  
  NY_ctpop_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTPop_15_proj.rds")
  NY_ctinc_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTinc_15_0imp_proj.rds")
  # NY_ctsex_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTSex_15_proj.rds")
  # NY_ctage_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTAge_15_proj.rds")
  # NY_ctocc_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTOcc_15_proj.rds")
  # NY_ctfood_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTFood_15_proj.rds")
  
  # Plot the gridded crime counts against the socio-economic variables interpolated on to the grids.
  # Average Income
  pdf(paste0("NYHomCountvIncImp",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=10,w=20,pointsize = 14)
  plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$inc,main="Counts of Homicide Normalised by Population Counts against\n Average Income",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Average Income")
  dev.off()
  
  pdf(paste0("NYGTACountvIncImp",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=10,w=20,pointsize = 14)
  plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$inc,main="Counts of Motor Vehicle Theft Normalised by Population Counts against\n Average Income",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Average Income")
  dev.off()
  
  # # Sex
  # pdf(paste0("NYHomCountvSex",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$msex,main="Counts of Homicide Normalised by Population Counts against Proportion of Males",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Males")
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$fsex,main="Counts of Homicide Normalised by Population Counts against Proportion of Females",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Females")
  # dev.off()
  # 
  # pdf(paste0("NYGTACountvSex",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$msex,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Males",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Males")
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$fsex,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Females",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Females")
  # dev.off()
  # 
  # # Occupied
  # pdf(paste0("NYHomCountvOcc",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$own,main="Counts of Homicide Normalised by Population Counts against Proportion of Properties Owned",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Owned Properties")
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$rent,main="Counts of Homicide Normalised by Population Counts against Proportion of Properties Rented",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Rented Properties")
  # dev.off()
  # 
  # pdf(paste0("NYGTACountvOcc",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$own,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Properties Owned",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Owned Properties")
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$rent,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Properties Rented",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Rented Properties")
  # dev.off()
  # 
  # # Food
  # pdf(paste0("NYHomCountvFood",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=10,w=20,pointsize = 14)
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$food,main="Counts of Homicide Normalised by Population Counts against\n Percentage of Households Receiving Food Stamps",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Percentage of Households Receiving Food Stamps")
  # dev.off()
  # 
  # pdf(paste0("NYGTACountvFood",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=10,w=20,pointsize = 14)
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$food,main="Counts of Motor Vehicle Theft Normalised by Population Counts against\n Percentage of Households Receiving Food Stamps",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Percentage of Households Receiving Food Stamps")
  # dev.off()
  
  # Now to plot heat maps for the crime, homicide and motor vehicle thefts for the gridded data.
  if (standard==FALSE){ # want to create our own bins for the heat maps
    nyc_cells.scale$homcount <- hom_countdf$hom; nyc_cells.scale$gtacount <- gta_countdf$gta
    nyc_cells.temp <- nyc_cells.scale
    
    # First we will plot the grid crime data
    bin.upperlim <- max(nyc_cells.temp$homcount)
    
    # Bins for different colours 
    nyc_cells.temp$bin <- cut(nyc_cells.temp$homcount, c(-0.5,0.5,1.5,2.5,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(nyc_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    
    # Re-name because I wanted to separate out 0, 1 and 2 homicides...
    my.levels[1] <- " 0"
    my.levels[2] <- " 1"
    my.levels[3] <- " 2"
    my.levels[4] <- " 3+"
    
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(nyc_cells.temp$bin))))
    nyc_cells.temp$Col <- a[as.numeric(nyc_cells.temp$bin)]
    
    pdf(paste0("NYHom2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(nyc_cells.temp), col = nyc_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("Map of Homicides in New York\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(nyc_cells.temp$Col),title=expression("Homicide"),cex=1,xpd=NA)
    dev.off()
    
    bin.upperlim <- max(nyc_cells.temp$gtacount)
    
    # Bins for different colours 
    nyc_cells.temp$bin <- cut(nyc_cells.temp$gtacount, c(-0.5,0.5,1.5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(nyc_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    
    # Re-name levels
    length(my.levels)
    my.levels[1] <- " 0"
    my.levels[2] <- " 1"
    my.levels[3] <- " 2 to 10"
    my.levels[length(my.levels)] <- " 75+"
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(nyc_cells.temp$bin))))
    nyc_cells.temp$Col <- a[as.numeric(nyc_cells.temp$bin)]
    
    pdf(paste0("NYGTA2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(nyc_cells.temp), col = nyc_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("Map of Motor Vehicle Thefts in New York\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(nyc_cells.temp$Col),title=expression("MVT"),cex=1,xpd=NA)
    dev.off()
    
    # Heat Maps for Socio-economic Variables
    # First plot the census tract then grid socio-economic variables for each variable
    
    # Population ##
    NY_ctpop_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/NY_CTPop_15_proj.rds")
    
    # Grid Heat Map: Population
    nyc_cells.temp <- nyc_cells.scale
    nyc_cells.temp$pop <- hom_countdf$pop
    if (round(max(nyc_cells.temp$pop)/5000)!=max(nyc_cells.temp$pop)/5000){
      bin.upperlim <- 5000*(round(max(nyc_cells.temp$pop)/5000)+1)
    } else {
      bin.upperlim <- max(nyc_cells.temp$pop)
    }
    
    # Bins for different colours 
    nyc_cells.temp$bin <- cut(nyc_cells.temp$pop, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(nyc_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(nyc_cells.temp$bin))))
    nyc_cells.temp$Col <- a[as.numeric(nyc_cells.temp$bin)]
    
    pdf(paste0("NYPop2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(nyc_cells.temp), col = nyc_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Population in New York on\n ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(nyc_cells.temp$Col),title=expression("Population (" *10^{3}*")"),cex=1,xpd=NA)
    dev.off()
    
    
    # Average Income ##
    # Grid Heat Map: Average Income
    nyc_cells.temp <- nyc_cells.scale
    nyc_cells.temp$inc <- hom_countdf$inc
    if (round(max(nyc_cells.temp$inc)/1e6)!=max(nyc_cells.temp$inc)/1e6){
      bin.upperlim <- 1e6*(round(max(nyc_cells.temp$inc)/1e6)+1)
    } else {
      bin.upperlim <- max(nyc_cells.temp$inc)
    }
    
    # Bins for different colours 
    nyc_cells.temp$bin <- cut(nyc_cells.temp$inc, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(nyc_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("0000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(nyc_cells.temp$bin))))
    nyc_cells.temp$Col <- a[as.numeric(nyc_cells.temp$bin)]
    
    pdf(paste0("NYIncImp2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(nyc_cells.temp), col = nyc_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Average Income in New York\n on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(nyc_cells.temp$Col),title=expression("Income (" *10^{4}*")"),cex=1,xpd=NA)
    dev.off()
    
    
    # # Sex ##
    # # Grid Heat Map: Sex
    # nyc_cells.temp <- nyc_cells.scale
    # nyc_cells.temp$m <- hom_countdf$msex
    # nyc_cells.temp$f <- hom_countdf$fsex
    # 
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # nyc_cells.temp$binm <- cut(nyc_cells.temp$m, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # nyc_cells.temp$binf <- cut(nyc_cells.temp$f, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(nyc_cells.temp$binm)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # am <- sort(rbPal(length(levels(nyc_cells.temp$binm))))
    # nyc_cells.temp$Colm <- am[as.numeric(nyc_cells.temp$binm)]
    # af <- sort(rbPal(length(levels(nyc_cells.temp$binf))))
    # nyc_cells.temp$Colf <- af[as.numeric(nyc_cells.temp$binf)]
    # 
    # pdf(paste0("NYSex2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(nyc_cells.temp), col = nyc_cells.temp$Colm, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion of Males\n in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.4)
    # plot(st_geometry(nyc_cells.temp), col = nyc_cells.temp$Colf, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion of Females\n in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.4)
    # legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = am, legend = my.levels, col = unique(nyc_cells.temp$Colm),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # 
    # # Home Occupier ##
    # # Grid Heat Map: Owned and Rented Properties
    # nyc_cells.temp <- nyc_cells.scale
    # nyc_cells.temp$own <- hom_countdf$own
    # nyc_cells.temp$rent <- hom_countdf$rent
    # 
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # nyc_cells.temp$binown <- cut(nyc_cells.temp$own, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # nyc_cells.temp$binrent <- cut(nyc_cells.temp$rent, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(nyc_cells.temp$binown)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # aown <- sort(rbPal(length(levels(nyc_cells.temp$binown))))
    # nyc_cells.temp$Colown <- aown[as.numeric(nyc_cells.temp$binown)]
    # arent <- sort(rbPal(length(levels(nyc_cells.temp$binrent))))
    # nyc_cells.temp$Colrent <- arent[as.numeric(nyc_cells.temp$binrent)]
    # 
    # pdf(paste0("NYOcc2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(nyc_cells.temp), col = nyc_cells.temp$Colown, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion Properties\n Owned in New York\n on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # plot(st_geometry(nyc_cells.temp), col = nyc_cells.temp$Colrent, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion Properties\n Rented in New York\n on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = aown, legend = my.levels, col = unique(nyc_cells.temp$Colown),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # 
    # # Food Stamps ##
    # # Grid Heat Map: Food Stamps
    # nyc_cells.temp <- nyc_cells.scale
    # nyc_cells.temp$food <- hom_countdf$food
    # 
    # bin.upperlim <- 100 # 100%
    # # Bins for different colours 
    # nyc_cells.temp$bin <- cut(nyc_cells.temp$food, c(-0.5,0,5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(nyc_cells.temp$bin)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # my.levels[1] <- " 0"
    # 
    # # Create a function to generate a continuous color palette
    # a <- sort(rbPal(length(levels(nyc_cells.temp$bin))))
    # nyc_cells.temp$Col <- a[as.numeric(nyc_cells.temp$bin)]
    # 
    # pdf(paste0("NYFood2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,0,8))
    # plot(st_geometry(nyc_cells.temp), col = nyc_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Percentage of Houses Receiving\n Food Stamps in New York\n on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # legend(x=6.15e5,y=4.515e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(nyc_cells.temp$Col),title=expression("Percentage"),cex=1,xpd=NA)
    # dev.off()
  } else { # standard plots, using the sf format and just a simple plot() function call and no alteration of the bins for the colours
    nyc_cellstemp <- nyc_cells.scale
    nyc_cellstemp$homcount <- hom_countdf$hom; nyc_cellstemp$gtacount <- gta_countdf$gta
    
    # First plot the grid crime data
    pdf(paste0("NYHom2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(nyc_cellstemp["homcount"], main=NULL)
    mtext(paste0("Map of Homicides in New York\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    pdf(paste0("NYGTA2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(nyc_cellstemp["gtacount"], main=NULL)
    mtext(paste0("Map of Motor Vehicle Thefts in New York\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    # Heat Maps for Socio-economic Variables
    # First plot the census tract then grid socio-economic variables for each variable
    ct_NY.projtemp <- ct_NY.proj
    ct_NY.projtemp$pop <- NY_ctpop_15$pop[match(ct_NY.proj$GEOID,NY_ctpop_15$geoid2)]
    ct_NY.projtemp$inc <- NY_ctinc_15$inc[match(ct_NY.proj$GEOID,NY_ctinc_15$geoid2)]
    # ct_NY.projtemp$msex <- NY_ctsex_15$mprop[match(ct_NY.proj$GEOID,NY_ctsex_15$geoid2)]
    # ct_NY.projtemp$fsex <- NY_ctsex_15$fprop[match(ct_NY.proj$GEOID,NY_ctsex_15$geoid2)]
    # ct_NY.projtemp$own <- NY_ctocc_15$ownprop[match(ct_NY.proj$GEOID,NY_ctocc_15$geoid2)]
    # ct_NY.projtemp$rent <- NY_ctocc_15$rentprop[match(ct_NY.proj$GEOID,NY_ctocc_15$geoid2)]
    # ct_NY.projtemp$food <- NY_ctfood_15$proprec[match(ct_NY.proj$GEOID,NY_ctfood_15$geoid2)]
    
    nyc_cellstemp$pop <- hom_countdf$pop
    nyc_cellstemp$inc <- hom_countdf$inc
    # nyc_cellstemp$msex <- hom_countdf$msex
    # nyc_cellstemp$fsex <- hom_countdf$fsex
    # nyc_cellstemp$own <- hom_countdf$own
    # nyc_cellstemp$rent <- hom_countdf$rent
    # nyc_cellstemp$food <- hom_countdf$food
    
    
    # Population ##
    # Grid Heat Map: Population
    pdf(paste0("NYPop2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(nyc_cellstemp["pop"], main=NULL) #,cex.main=2)
    mtext(paste0("2015 Population in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    
    # Average Income ##
    # Grid Heat Map: Average Income
    pdf(paste0("NYIncImp2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(nyc_cellstemp["inc"], main=NULL) #,cex.main=2)
    mtext(paste0("2015 Average Income in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    
    # # Sex ##
    # # Grid Heat Map: Sex
    # pdf(paste0("NYSex2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(nyc_cellstemp["msex"], main=NULL,cex.main=1.4)
    # mtext(paste0("2015 Proportion of Males\n in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # plot(nyc_cellstemp["fsex"], main=NULL,cex.main=1.4)
    # mtext(paste0("2015 Proportion of Females\n in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # 
    # # Home Occupier ##
    # # Grid Heat Map: Owned and Rented Properties
    # pdf(paste0("NYOcc2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(nyc_cellstemp["own"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Proportion Properties\n Owned in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # plot(nyc_cellstemp["rent"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Proportion Properties\n Rented in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # 
    # # Food Stamps ##
    # # Grid Heat Map: Food Stamps
    # pdf(paste0("NYFood2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(nyc_cellstemp["food"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Percentage of Houses Receiving\n Food Stamps in New York on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
  }
}


grid_cellsxvec <- c(10,24,47,94)
grid_cellsyvec <- c(10,24,48,96)

for (i in 1:4){
  plot_nyc_grid(grid_cellsx=grid_cellsxvec[i],grid_cellsy=grid_cellsyvec[i],standard=TRUE)
}

# Portland ----------------------------------------------------------------

# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))
setwd("./Portland")

load("../../PROCESSED_DATA/SHAPEFILES/PCityCT.rda")
ct_P.proj <- lwgeom::st_transform_proj(ct_P,"epsg:32610")

xy.coord <- sapply(1:length(ct_P.proj$geometry),function(i){return(as.numeric(st_centroid(ct_P.proj$geometry[[i]])))}) # returns a 2xlength(b$geometry) matrix, where each column is for each geometry and row one is longitude and row two is latitude

ct_P.proj$x <- xy.coord[1,] # [1] 
ct_P.proj$y <- xy.coord[2,] # [1] 

# Load in the crime data that has processed
p_gta <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/Portland/p_gta.rds")
p_hom <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/Portland/p_hom.rds")
p_hom_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/Portland/p_hom_sf.rds")
p_gta_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/Portland/p_gta_sf.rds")
p_hom_sf.proj <- lwgeom::st_transform_proj(p_hom_sf,"epsg:32610")
p_gta_sf.proj <- lwgeom::st_transform_proj(p_gta_sf,"epsg:32610")

p_gta15 <- p_gta[which(p_gta$Y==2015),]
p_hom15 <- p_hom[which(p_hom$Y==2015),]

p_gta15_sf.proj <- p_gta_sf.proj[which(p_gta_sf.proj$Y==2015),]
p_hom15_sf.proj <- p_hom_sf.proj[which(p_hom_sf.proj$Y==2015),]

p_hom15_sp.proj <- as(p_hom15_sf.proj,"Spatial")
p_gta15_sp.proj <- as(p_gta15_sf.proj,"Spatial")


# Portland: Plotting Shifted and Scaled Gridded Data ----------------------

# Create a wrapper function to plot the required socio-economic variables and crime gridded data where the input allows for us to alter the grid resolution we are interested in without replicating the code multiple times for each, just requires a call to this function for each resolution of interest. We can either use st_geometry() in the plot function with alterations to the binning and the "standard" plot just uses the plot function to extract the relevant variable.
# As well as the average income and population variables, we originally included a few other socio-economic variables however these were not included in any of the modelling of the crime data and updated data generation code has these commented out and so I will comment out the lines including these variables but retain the code so that it is complete as was originally run. If they are of interest, they can easily be uncommented to produce the relevant plots.

plot_p_grid <- function(grid_cellsx,grid_cellsy,standard=FALSE){
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/Portland/P2015CT",grid_cellsx,grid_cellsy,"CountData_projFinalScale.rda"))
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/Portland/PGridCells",grid_cellsx,grid_cellsy,"_projFinal.rda"))
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/Portland/PWindowProjScale.rda"))
  load(paste0("../../PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/Portland/POrdDF",grid_cellsx,grid_cellsy,"_projFinalScale.rda"))
  
  bbox <- pcity_geom.proj@bbox
  pcity_spshift <- elide(pcity_geom.proj,shift=-c(bbox[1,1],bbox[2,1]))
  bbox_shift <- pcity_spshift@bbox
  pcity_spscale <- elide(pcity_spshift,scale=max(bbox_shift)/1e4)
  
  p_cellssp <- as(p_cells,"Spatial")
  p_cells_shift <- elide(p_cellssp,shift=-c(bbox[1,1],bbox[2,1]))
  p_cells_scale <- elide(p_cells_shift ,scale=max(bbox_shift)/1e4)
  
  p_cells.scale <- st_as_sf(p_cells_scale)
  p_cells.scale <- p_cells.scale[ord.df.p$ind,]
  
  P_ctpop_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTPop_15_proj.rds")
  P_ctinc_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTinc_15_0imp_proj.rds")
  # P_ctsex_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTSex_15_proj.rds")
  # P_ctage_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTAge_15_proj.rds")
  # P_ctocc_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTOcc_15_proj.rds")
  # P_ctfood_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTFood_15_proj.rds")
  
  # Plot the gridded crime counts against the socio-economic variables interpolated on to the grids.  
  # Average Income
  pdf(paste0("PHomCountvIncImp",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=10,w=20,pointsize = 14)
  plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$inc,main="Counts of Homicide Normalised by Population Counts against\n Average Income",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Average Income")
  dev.off()
  
  pdf(paste0("PGTACountvIncImp",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=10,w=20,pointsize = 14)
  plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$inc,main="Counts of Motor Vehicle Theft Normalised by Population Counts against\n Average Income",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Average Income")
  dev.off()
  
  # # Sex
  # pdf(paste0("PHomCountvSex",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$msex,main="Counts of Homicide Normalised by Population Counts against Proportion of Males",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Males")
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$fsex,main="Counts of Homicide Normalised by Population Counts against Proportion of Females",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Females")
  # dev.off()
  # 
  # pdf(paste0("PGTACountvSex",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$msex,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Males",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Males")
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$fsex,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Females",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Females")
  # dev.off()
  # 
  # # Occupied
  # pdf(paste0("PHomCountvOcc",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$own,main="Counts of Homicide Normalised by Population Counts against Proportion of Properties Owned",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Owned Properties")
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$rent,main="Counts of Homicide Normalised by Population Counts against Proportion of Properties Rented",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Proportion of Rented Properties")
  # dev.off()
  # 
  # pdf(paste0("PGTACountvOcc",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=20,w=15,pointsize = 14)
  # par(mfrow=c(2,1))
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$own,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Properties Owned",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Owned Properties")
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$rent,main="Counts of Motor Vehicle Theft Normalised by Population Counts against Proportion of Properties Rented",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Proportion of Rented Properties")
  # dev.off()
  # 
  # # Food
  # pdf(paste0("PHomCountvFood",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=10,w=20,pointsize = 14)
  # plot(hom_countdf$hom/hom_countdf$pop,hom_countdf$food,main="Counts of Homicide Normalised by Population Counts against\n Percentage of Households Receiving Food Stamps",xlab = "Counts of Homicide Divided by Total Population Counts",ylab="Percentage of Households Receiving Food Stamps")
  # dev.off()
  # 
  # pdf(paste0("PGTACountvFood",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=10,w=20,pointsize = 14)
  # plot(gta_countdf$gta/gta_countdf$pop,gta_countdf$food,main="Counts of Motor Vehicle Theft Normalised by Population Counts against\n Percentage of Households Receiving Food Stamps",xlab = "Counts of Motor Vehicle Theft Divided by Total Population Counts",ylab="Percentage of Households Receiving Food Stamps")
  # dev.off()
  
  if (standard==FALSE){ # want to create our own bins for the heat maps
    p_cells.scale$homcount <- hom_countdf$hom; p_cells.scale$gtacount <- gta_countdf$gta
    p_cells.temp <- p_cells.scale.scale
    
    # First we will plot the grid crime data
    bin.upperlim <- max(p_cells.temp$homcount)
    
    # Bins for different colours 
    p_cells.temp$bin <- cut(p_cells.temp$homcount, c(-0.5,0.5,1.5,2.5,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(p_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    
    # Re-name because I wanted to separate out 0, 1 and 2 homicides...
    my.levels[1] <- " 0"
    my.levels[2] <- " 1"
    my.levels[3] <- " 2"
    my.levels[4] <- " 3+"
    
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(p_cells.temp$bin))))
    p_cells.temp$Col <- a[as.numeric(p_cells.temp$bin)]
    
    pdf(paste0("PHom2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(p_cells.temp), col = p_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("Map of Homicides in Portland\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(p_cells.temp$Col),title=expression("Homicide"),cex=1,xpd=NA)
    dev.off()
    
    bin.upperlim <- max(p_cells.temp$gtacount)
    
    # Bins for different colours 
    p_cells.temp$bin <- cut(p_cells.temp$gtacount, c(-0.5,0.5,1.5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(p_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    
    # Re-name levels
    length(my.levels)
    my.levels[1] <- " 0"
    my.levels[2] <- " 1"
    my.levels[3] <- " 2 to 10"
    my.levels[length(my.levels)] <- " 75+"
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(p_cells.temp$bin))))
    p_cells.temp$Col <- a[as.numeric(p_cells.temp$bin)]
    
    pdf(paste0("PGTA2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(p_cells.temp), col = p_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("Map of Motor Vehicle Thefts in Portland\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(p_cells.temp$Col),title=expression("MVT"),cex=1,xpd=NA)
    dev.off()
    
    
    # Heat Maps for Socio-economic Variables
    # First plot the census tract then grid socio-economic variables for each variable
    
    # Population ##
    P_ctpop_15 <- readRDS("../../PROCESSED_DATA/COVARIATES/P_CTPop_15_proj.rds")
    
    # Grid Heat Map: Population
    p_cells.temp <- p_cells.scale.scale
    p_cells.temp$pop <- hom_countdf$pop
    if (round(max(p_cells.temp$pop)/5000)!=max(p_cells.temp$pop)/5000){
      bin.upperlim <- 5000*(round(max(p_cells.temp$pop)/5000)+1)
    } else {
      bin.upperlim <- max(p_cells.temp$pop)
    }
    
    # Bins for different colours 
    p_cells.temp$bin <- cut(p_cells.temp$pop, seq(0,bin.upperlim,length.out=6),include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(p_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(p_cells.temp$bin))))
    p_cells.temp$Col <- a[as.numeric(p_cells.temp$bin)]
    
    pdf(paste0("PPop2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(p_cells.temp), col = p_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Population in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(p_cells.temp$Col),title=expression("Population (" *10^{3}*")"),cex=1,xpd=NA)
    dev.off()
    
    
    # Average Income ##
    # Grid Heat Map: Average Income
    p_cells.temp <- p_cells.scale
    p_cells.temp$inc <- hom_countdf$inc
    if (round(max(p_cells.temp$inc)/1e4)!=max(p_cells.temp$inc)/1e4){
      bin.upperlim <- 1e4*(round(max(p_cells.temp$inc)/1e4)+1)
    } else {
      bin.upperlim <- max(p_cells.temp$inc)
    }
    
    # Bins for different colours 
    p_cells.temp$bin <- cut(p_cells.temp$inc,seq(0,bin.upperlim,length.out=6), include.lowest = TRUE, dig.lab = 10)
    
    # labels for the legend
    lev = levels(p_cells.temp$bin)
    lev2 <- gsub("\\,", " to ", lev)
    lev3 <- gsub("\\]$", "", lev2)
    lev4 <- gsub("\\(|\\)", " ", lev3)
    lev5 <- gsub("^\\[", " ", lev4)
    my.levels <- lev5
    lev6 <- gsub("0000","",lev5)
    my.levels <- lev6
    
    # Create a function to generate a continuous color palette
    a <- sort(rbPal(length(levels(p_cells.temp$bin))))
    p_cells.temp$Col <- a[as.numeric(p_cells.temp$bin)]
    
    pdf(paste0("PIncImp2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,0,8))
    plot(st_geometry(p_cells.temp), col = p_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Average Income in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=2)
    legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(p_cells.temp$Col),title=expression("Income (" *10^{4}*")"),cex=1,xpd=NA)
    dev.off()
    
    
    # # Sex ##
    # # Grid Heat Map: Sex
    # p_cells.temp <- p_cells.scale
    # p_cells.temp$m <- hom_countdf$msex
    # p_cells.temp$f <- hom_countdf$fsex
    # 
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # p_cells.temp$binm <- cut(p_cells.temp$m, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # p_cells.temp$binf <- cut(p_cells.temp$f, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(p_cells.temp$binm)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # am <- sort(rbPal(length(levels(p_cells.temp$binm))))
    # p_cells.temp$Colm <- am[as.numeric(p_cells.temp$binm)]
    # af <- sort(rbPal(length(levels(p_cells.temp$binf))))
    # p_cells.temp$Colf <- af[as.numeric(p_cells.temp$binf)]
    # 
    # pdf(paste0("PSex2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(p_cells.temp), col = p_cells.temp$Colm, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion of Males\n in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.4)
    # plot(st_geometry(p_cells.temp), col = p_cells.temp$Colf, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion of Females\n in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.4)
    # legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = am, legend = my.levels, col = unique(p_cells.temp$Colm),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # 
    # # Home Occupier ##
    # # Grid Heat Map: Owned and Rented Properties
    # p_cells.temp <- p_cells.scale
    # p_cells.temp$own <- hom_countdf$own
    # p_cells.temp$rent <- hom_countdf$rent
    # 
    # bin.upperlim <- 1
    # 
    # # Bins for different colours 
    # p_cells.temp$binown <- cut(p_cells.temp$own, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # p_cells.temp$binrent <- cut(p_cells.temp$rent, c(0,0.25,0.5,0.75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(p_cells.temp$binown)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # # Create a function to generate a continuous color palette
    # aown <- sort(rbPal(length(levels(p_cells.temp$binown))))
    # p_cells.temp$Colown <- aown[as.numeric(p_cells.temp$binown)]
    # arent <- sort(rbPal(length(levels(p_cells.temp$binrent))))
    # p_cells.temp$Colrent <- arent[as.numeric(p_cells.temp$binrent)]
    # 
    # pdf(paste0("POcc2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(mfrow=c(1,2),oma=c(0,0,0,8))
    # plot(st_geometry(p_cells.temp), col = p_cells.temp$Colown, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion Properties\n Owned in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # plot(st_geometry(p_cells.temp), col = p_cells.temp$Colrent, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Proportion Properties\n Rented in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = aown, legend = my.levels, col = unique(p_cells.temp$Colown),title=expression("Proportion"),cex=1,xpd=NA)
    # dev.off()
    # 
    # 
    # # Food Stamps ##
    # # Grid Heat Map: Food Stamps
    # p_cells.temp <- p_cells.scale
    # p_cells.temp$food <- hom_countdf$food
    # 
    # bin.upperlim <- 100 # 100%
    # # Bins for different colours 
    # p_cells.temp$bin <- cut(p_cells.temp$food, c(-0.5,0,5,10,25,50,75,bin.upperlim),include.lowest = TRUE, dig.lab = 10)
    # 
    # # labels for the legend
    # lev = levels(p_cells.temp$bin)
    # lev2 <- gsub("\\,", " to ", lev)
    # lev3 <- gsub("\\]$", "", lev2)
    # lev4 <- gsub("\\(|\\)", " ", lev3)
    # lev5 <- gsub("^\\[", " ", lev4)
    # my.levels <- lev5
    # 
    # my.levels[1] <- " 0"
    # 
    # # Create a function to generate a continuous color palette
    # a <- sort(rbPal(length(levels(p_cells.temp$bin))))
    # p_cells.temp$Col <- a[as.numeric(p_cells.temp$bin)]
    # 
    # pdf(paste0("PFood2015Grid",grid_cellsx,grid_cellsy,"_proj_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,0,8))
    # plot(st_geometry(p_cells.temp), col = p_cells.temp$Col, border = 'black', axes = FALSE, bty="n", main=paste0("2015 Percentage of Houses Receiving\n Food Stamps in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),cex.main=1.3)
    # legend(x=5.52e5,y=5.049e6,inset = c(-0.5,0.25),fill = a, legend = my.levels, col = unique(p_cells.temp$Col),title=expression("Percentage"),cex=1,xpd=NA)
    # dev.off()
  } else {
    p_cellstemp <- p_cells.scale
    p_cellstemp$homcount <- hom_countdf$hom; p_cellstemp$gtacount <- gta_countdf$gta
    
    # First plot the grid crime data
    pdf(paste0("PHom2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(p_cellstemp["homcount"], main=NULL)
    mtext(paste0("Map of Homicides in Portland\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    pdf(paste0("PGTA2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(p_cellstemp["gtacount"], main=NULL)
    mtext(paste0("Map of Motor Vehicle Thefts in Portland\n in 2015 in a ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    # Heat Maps for Socio-economic Variables
    # Plot the gridded socio-economic variables for each variable
    ct_P.projtemp <- ct_P.proj
    ct_P.projtemp$pop <- P_ctpop_15$pop[match(ct_P.proj$GEOID,P_ctpop_15$geoid2)]
    ct_P.projtemp$inc <- P_ctinc_15$inc[match(ct_P.proj$GEOID,P_ctinc_15$geoid2)]
    # ct_P.projtemp$msex <- P_ctsex_15$mprop[match(ct_P.proj$GEOID,P_ctsex_15$geoid2)]
    # ct_P.projtemp$fsex <- P_ctsex_15$fprop[match(ct_P.proj$GEOID,P_ctsex_15$geoid2)]
    # ct_P.projtemp$own <- P_ctocc_15$ownprop[match(ct_P.proj$GEOID,P_ctocc_15$geoid2)]
    # ct_P.projtemp$rent <- P_ctocc_15$rentprop[match(ct_P.proj$GEOID,P_ctocc_15$geoid2)]
    # ct_P.projtemp$food <- P_ctfood_15$proprec[match(ct_P.proj$GEOID,P_ctfood_15$geoid2)]
    
    p_cellstemp$pop <- hom_countdf$pop
    p_cellstemp$inc <- hom_countdf$inc
    # p_cellstemp$msex <- hom_countdf$msex
    # p_cellstemp$fsex <- hom_countdf$fsex
    # p_cellstemp$own <- hom_countdf$own
    # p_cellstemp$rent <- hom_countdf$rent
    # p_cellstemp$food <- hom_countdf$food
    
    
    # Population ##
    # Grid Heat Map: Population
    pdf(paste0("PPop2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(p_cellstemp["pop"], main=NULL) #,cex.main=2)
    mtext(paste0("2015 Population in Portland\n on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    
    # Average Income ##
    # Grid Heat Map: Average Income
    pdf(paste0("PIncImp2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    par(oma=c(0,0,3,0))
    plot(p_cellstemp["inc"], main=NULL) #,cex.main=2)
    mtext(paste0("2015 Average Income in Portland\n on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    dev.off()
    
    
    # # Sex ##
    # # Grid Heat Map: Sex
    # pdf(paste0("PSex2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(p_cellstemp["msex"], main=NULL,cex.main=1.4)
    # mtext(paste0("2015 Proportion of Males\n in Portland on\n ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # plot(p_cellstemp["fsex"], main=NULL,cex.main=1.4)
    # mtext(paste0("2015 Proportion of Females\n in Portland on\n ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # 
    # # Home Occupier ##
    # # Grid Heat Map: Owned and Rented Properties
    # pdf(paste0("POcc2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(p_cellstemp["own"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Proportion Properties\n Owned in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # plot(p_cellstemp["rent"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Proportion Properties\n Rented in Portland on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
    # 
    # 
    # # Food Stamps ##
    # # Grid Heat Map: Food Stamps
    # pdf(paste0("PFood2015Grid",grid_cellsx,grid_cellsy,"_projSFPlot_finalscale.pdf"),h=8,w=10,pointsize=14)
    # par(oma=c(0,0,3,0))
    # plot(p_cellstemp["food"], main=NULL,cex.main=1.3)
    # mtext(paste0("2015 Percentage of Houses Receiving\n Food Stamps in Portland\n on ",grid_cellsx,"x",grid_cellsy," Grid"),side=3,line=0,outer = T,cex=1.5)
    # dev.off()
  }
}


grid_cellsxvec <- c(8,19,38,76)
grid_cellsyvec <- c(6,13,26,52)


for (i in 1:4){
  plot_p_grid(grid_cellsx=grid_cellsxvec[i],grid_cellsy=grid_cellsyvec[i],standard=TRUE)
}


# Session Information -----------------------------------------------------

sessionInfo()