
# Minimum Contrast: Los Angeles 2014 --------------------------------------

# This R script takes the 2014 Los Angeles crime data as well as the ACS total population and average income variables for 2014 to implement the minimum contrast method. This will provide estimates for the range and variance of the point patterns, with the goal of using these estimates, in particular the range estimates, to guide our upper limit of the grid and mesh resolution of the Grid-Mesh Optimisation method implemented over the LA polygon.
# Initially, we created the census tract data as for the 2015 data as this would have allowed us to use covariate effect estimates from the ppm() function to generate an estimate of the intensity function over Los Angeles, however it was no longer used when the code was updated. The, now redundant, code for the data generation is still included for interest.
# Additionally, we only combined the PCF or Ripley's K function with the identity function or log() transformation within the minimum contrast implementation that guided our decisions, we later ran the combination of Ripley's K function with the fourth root transformation for interest. It reasonably agreed with some of the other results and while it was not originally included within this script, the necessary R code has been included in the relevant sections.

# Author: Nadeen Khaleel

# Setwd and Load Libraries ------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))

library(dplyr)
library(readr)
library(revgeo)
library(sf)
library(lwgeom)
library(stringr)
library(purrr)
library(sp)
library(maptools)
library(spatstat)
library(lgcp)
library(raster)

# Los Angeles 2014 --------------------------------------------------------
# First we need to set-up the 2014 socio-economic variables (only population and average income) for the LA census tracts, accessed through the ACS. The methods should follow those from the R scripts CovDataGen_final.R and CovDataGen_Inc_final.R in the DATA/RAW_DATA/COVARIATES DIRECTORY, where the latter is used for the income and dealing with missing data that correspond to census tracts with zero estimated households.

# LA Census Tracts
load("../../PROCESSED_ DATA/SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")

# Project to UTM
ct_LA.proj <- lwgeom::st_transform_proj(ct_LA,"epsg:32611")
latlon <- sapply(1:length(ct_LA.proj$geometry),function(i){return(as.numeric(st_centroid(ct_LA.proj$geometry[[i]])))}) # returns a 2xlength(b$geometry) matrix, where each column is for each geometry and row one is longitude and row two is latitude

ct_LA.proj$x <- latlon[1,]
ct_LA.proj$y <- latlon[2,]


# Los Angeles 2014: Population --------------------------------------------
# We will now extract the necessary population data and check for missing values before saving the final data set. The ACS data contains estimate for the population as well errors for these estimates so we will store these as well.

la_pop_2014 <- read_csv("../../RAW_DATA/COVARIATES/LA/ACSDT5Y2014.B01003_2020-07-24T133057/ACSDT5Y2014.B01003_data_with_overlays_2020-07-24T133054.csv")

# Sort out the census tract identification for merging, as in the 2015 LA data we had geoid2.
geo2 <- str_extract_all(la_pop_2014$GEO_ID[2:length(la_pop_2014$GEO_ID)],"\\d+")
la_pop_2014$GEO.id2 <- la_pop_2014$GEO_ID
la_pop_2014$GEO.id2[1] <- "id2"
la_pop_2014$GEO.id2[2:length(la_pop_2014$GEO_ID)] <- map_chr(geo2,2)

# Extract the data for the census tracts of interest
LA_ctpop_14_0 <- la_pop_2014[la_pop_2014$GEO.id2%in%ct_LA.proj$GEOID,]

# Assign the relevant (UTM) coordinates
LA_ctpop_14_0$y <- ct_LA.proj[ct_LA.proj$GEOID%in%LA_ctpop_14_0$GEO.id2,]$y
LA_ctpop_14_0$x <- ct_LA.proj[ct_LA.proj$GEOID%in%LA_ctpop_14_0$GEO.id2,]$x

# These columns were selected by checking the necessary column names in the metadata
LA_ctpop_14 <- LA_ctpop_14_0[,c(1,5,2,3,4,6,7)]
colnames(LA_ctpop_14) <- c("geoid","geoid2","name","pop","err","y","x")
LA_ctpop_14$pop <- as.numeric(LA_ctpop_14$pop)
# No missing data (usually, the as.numeric would have picked up on these and produce a warning)

# Save
saveRDS(LA_ctpop_14,"LA_CTPop_14_proj.rds")
LA_ctpop_14 <- readRDS("LA_CTPop_14_proj.rds")


# Los Angeles 2014: Average Income (Imputed-Models) -----------------------
# We will now extract the average income data. It is important to note that the results from the following code assigns missing average income data for a census tract a value of zero if there is an estimated zero total households therefore total income of zero. This follows the method for the 2015 crime data for use within the models.

la_inc_2014 <- read_csv("../../RAW_DATA/COVARIATES/LA/ACSST5Y2014.S1902_2020-07-24T133414/ACSST5Y2014.S1902_data_with_overlays_2020-07-24T133400.csv")

# Sort out the census tract identification for merging, as in the 2015 LA data we had geoid2.
geo2 <- str_extract_all(la_inc_2014$GEO_ID[2:length(la_inc_2014$GEO_ID)],"\\d+")
la_inc_2014$GEO.id2 <- la_inc_2014$GEO_ID
la_inc_2014$GEO.id2[1] <- "id2"
la_inc_2014$GEO.id2[2:length(la_inc_2014$GEO_ID)] <- map_chr(geo2,2)

# Extract the data for the census tracts of interest
LA_ctinc_14_0 <- la_inc_2014[la_inc_2014$GEO.id2%in%ct_LA.proj$GEOID,]

# Assign the relevant (UTM) coordinates
LA_ctinc_14_0$y <- ct_LA.proj[ct_LA.proj$GEOID%in%LA_ctinc_14_0$GEO.id2,]$y
LA_ctinc_14_0$x <- ct_LA.proj[ct_LA.proj$GEOID%in%LA_ctinc_14_0$GEO.id2,]$x

# Take the mean income and margin of error, which according to the metadata is given by S1902_C01_001E/S1902_C01_001M and is in slot 57 and 58, while lat and lon are in slot 112 and 113
LA_ctinc_14 <- LA_ctinc_14_0[,c(1,111,2,3,57,58,112,113)]
colnames(LA_ctinc_14) <- c("geoid","geoid2","name","total.inc","inc","err","y","x")
LA_ctinc_14$inc <- as.numeric(LA_ctinc_14$inc)
LA_ctinc_14$total.inc <- as.numeric(LA_ctinc_14$total.inc) 

# We have missing data
sum(is.na(LA_ctinc_14$inc))
# [1] 9

# Assign those with zero estimated total households a zero value.
LA_ctinc_14$inc[LA_ctinc_14$total.inc==0] <- 0
sum(is.na(LA_ctinc_14$inc))
# [1] 3
# There is still some missing data, so still need to impute.

# Which census tracts have the missing data?
ind.na <- which(is.na(LA_ctinc_14$inc))
plot(ct_LA.proj$geometry,axes=T,main="Census Tracts with Missing Income Data")
ind.na.ct <- match(LA_ctinc_14$geoid2[ind.na],ct_LA.proj$GEOID)
plot(ct_LA.proj[ind.na.ct,]$geometry,add=T,col="red")

# Load the LA County census tract to plot and get a visual representation of the census tracts with the missing data.
load("../../PROCESSED_DATA/SHAPEFILES/CENSUS_TRACTS/LACountyCT.rda")
ct_LA_County.proj <- lwgeom::st_transform_proj(ct_LA_County,"epsg:32611")
ct_LA_Countysp.proj <- as(ct_LA_County.proj,"Spatial")

plot(ct_LA_County.proj$geometry,border="red")
plot(ct_LA.proj[ct_LA.proj$GEOID%in%LA_ctinc_14$geoid2[ind.na],]$geometry,col="red",add=T)
plot(ct_LA.proj$geometry,add=T,border="blue")

# In order to deal with the missing data in the income, I will use the average of each census tracts neighbours to impute the missing data. Therefore we must load in the neighbourhood matrix for Los Angeles county.
load("../../PROCESSED_DATA/SHAPEFILES/CENSUS_TRACTS/LACountyNB.rda")

# Extract only the rows from the neighbourhood matrix for the census tracts that have missing income data.
ct_ind <- match(LA_ctinc_14$geoid2[ind.na],ct_LA_Countysp$GEOID) # which geometries are the missings from
nb_ind <- sapply(1:length(ct_ind),function(i){which(unname(nb[ct_ind[i],]))}) # who are their neighbours
nb_ct <- sapply(1:length(nb_ind),function(i){match(ct_LA_Countysp$GEOID[nb_ind[[i]]],ct_LA_County.proj$GEOID)}) # which geometries on the county scale have we selected

# Visually inspect that we have the correct census tracts
plot(ct_LA_County.proj[unlist(nb_ct),]$geometry,col="magenta",add=T)

# Find the indices for the county with the indices of the income data frame
inc_ind <- sapply(1:length(nb_ct),function(i){match(ct_LA_County.proj$GEOID[nb_ct[[i]]],la_inc_2014$GEO.id2)})
sum(is.na(as.numeric(unlist(la_inc_2014[unique(unlist(inc_ind)),57]))))

# Extract the indices for census tracts with NAs for the average income in the list elements of neighbours for each of the missing data census tracts
cty_na_ind <- sapply(1:length(inc_ind),function(i){inc_ind[[i]][is.na(as.numeric(unlist(la_inc_2014[inc_ind[[i]],6])))]})
# Extract the indices within the county data
cty_na_ct <- sapply(1:length(cty_na_ind),function(i){match(la_inc_2014$GEO.id2[cty_na_ind[[i]]],ct_LA_County.proj$GEOID)})

# These are just to visually inspect missing data between the county and city.
plot(ct_LA_County.proj[unlist(cty_na_ct),]$geometry,col="yellow",border="blue",add=T) # which are NA in county data
plot(ct_LA.proj[ct_LA.proj$GEOID%in%LA_ctinc_14$geoid2[ind.na],]$geometry,col="red",border="yellow",add=T)

# Now to impute the average of the neighbours (recall nb_ind is a list for each ind.na)
# nb_inc extracts the average income for each element of the list of neighbours for each of the 10 census tracts within missing data.
nb_inc <- sapply(1:length(nb_ind),function(i){as.numeric(unlist(la_inc_2014[match(ct_LA_Countysp$GEOID[nb_ind[[i]]],la_inc_2014$GEO.id2),6]))})


# Impute the mean, making sure to ignore any missing data.
LA_ctinc_14_imp <- LA_ctinc_14
mean.inc <- lapply(nb_inc,mean,na.rm=T)
LA_ctinc_14_imp$inc[ind.na] <- unlist(mean.inc)

# Check that we no longer have missing data for the average income.
sum(is.na(LA_ctinc_14))
# [1] 9
sum(is.na(LA_ctinc_14_imp))
# [1] 0

# Save
saveRDS(LA_ctinc_14_imp,"LA_ctinc_14_0imp_proj.rds")
LA_ctinc_14_imp <- readRDS("LA_ctinc_14_0imp_proj.rds")



# Los Angeles 2014: Census Tract Count Data -------------------------------
# In this section we create census tract count data for the 2014 crimes in Los Angeles, similar to the set-up in CountDataGen_CT_final.R in DATA/PROCESSED_DATA/CRIME/COUNT_DATA_CENSUS_TRACTS.
# Note: while this count data was originally used to generate the pixel images for the mean field in the following section, it is no longer necessary but the code to generate the census tract data is not removed for interest.

# LA Data
load("../../PROCESSED_DATA/SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")

# LA Point Patterns
# Load and project to UTM
la_hom <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/la_hom.rds")
la_gta <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/la_gta.rds")
la_hom_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/la_hom_sf.rds")
la_gta_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/la_gta_sf.rds")
la_hom_sf.proj <- lwgeom::st_transform_proj(la_hom_sf,"epsg:32611")
la_gta_sf.proj <- lwgeom::st_transform_proj(la_gta_sf,"epsg:32611")

# LA Socio-economic Variables
LA_ctpop_14 <- readRDS("./LA_CTPop_14_proj.rds")
LA_ctinc_14 <- readRDS("./LA_CTInc_14_0imp_proj.rds")

# Project to UTM: both the census tracts and their union - which will form a boundary for the city.
ct_LA.proj <- lwgeom::st_transform_proj(ct_LA,"epsg:32611")
lacity_boundary <- st_union(ct_LA)
lacity_boundary.proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611")
lacity_geom <- as(lacity_boundary,"Spatial")
lacity_geom.proj <- as(lacity_boundary.proj,"Spatial")
p4s <- lacity_geom.proj@proj4string


# Extract 2014 data
la_hom14_sf.proj <- la_hom_sf.proj[which(la_hom_sf.proj$Y==2014),]
la_gta14_sf.proj <- la_gta_sf.proj[which(la_gta_sf.proj$Y==2014),]

# Calculate area of the census tracts and get coordinates for the center coordinates of the census tracts
ct.area <- st_area(ct_LA.proj)
ct.centres <- st_centroid(ct_LA.proj)
ct.centres <- st_coordinates(ct.centres)

# Homicide Counts
hom_ctLA_cov <- lengths(st_covers(ct_LA.proj,la_hom14_sf.proj))

# Motor Vehicle Theft Counts
gta_ctLA_cov <- lengths(st_covers(ct_LA.proj,la_gta14_sf.proj))

# Generate data frames for the two crime counts
ct_homcount.df <- data.frame(ct=ct_LA.proj$TRACTCE,x=ct.centres[,1],y=ct.centres[,2],ctarea=ct.area,hom=hom_ctLA_cov)
ct_gtacount.df <- data.frame(ct=ct_LA.proj$TRACTCE,x=ct.centres[,1],y=ct.centres[,2],ctarea=ct.area,gta=gta_ctLA_cov)

geo2ctLA <- function(x){ct_LA$TRACTCE[which(ct_LA$GEOID==x)]}

# Get the required socio-economic variables values for each census tract in the count data frames above.
count.pop <- LA_ctpop_14$pop[match(ct_homcount.df$ct,as.character(sapply(1:length(LA_ctpop_14$geoid2),function(i){geo2ctLA(LA_ctpop_14$geoid2[i])})))]
count.inc <- LA_ctinc_14$inc[match(ct_homcount.df$ct,as.character(sapply(1:length(LA_ctinc_14$geoid2),function(i){geo2ctLA(LA_ctinc_14$geoid2[i])})))]

# Place the variables into the count data frames.
ct_homcount.df$pop <- count.pop
ct_homcount.df$inc <- count.inc

ct_gtacount.df$pop <- count.pop
ct_gtacount.df$inc <- count.inc

# Turn data frames into spatial objects, in particular sf objects
spdf_ct_homcount <- SpatialPointsDataFrame(coords=ct_homcount.df[,c("x","y")],data=ct_homcount.df,proj4string = p4s)
sf_homcount <- st_as_sf(spdf_ct_homcount)
spdf_ct_gtacount <- SpatialPointsDataFrame(coords=ct_gtacount.df[,c("x","y")],data=ct_gtacount.df,proj4string = p4s)
sf_gtacount <- st_as_sf(spdf_ct_gtacount)

# Save
save(hom_countdf=ct_homcount.df,gta_countdf=ct_gtacount.df,file=paste0("LA2014CTCountData_proj_Final.rda"))
save(sf_homcount,sf_gtacount,file=paste0("LA2014CTSFCountData_proj_Final.rda"))



# Los Angeles 2014: Minimum Contrast --------------------------------------
# This section sets up the point patterns as well as the pixel images for the covariates and uses these within ppm() to estimate coefficients for the intercept and covariate effects in order to generate the mean field, lambda, over the study region. This can then be included in the minimum.contrast function.
# Note that while we originally have the count data over the census tracts above generated and loaded below, updated code made them redundant and while the lines of code have not been removed they are unnecessary.
# We also re-load all of the data required below, in the case of having the data created above, then this section is only needed to implement the minimum.contrast functions.

# # Load the necessary count data - this used to provide us with the variable data at the census tract level, however we proceeded with another method.
# load("LA2014CTCountData_proj_Final.rda")
# hom_countdf <- ct_homcount.df; gta_countdf <- ct_gtacount.df
# load("LA2014CTSFCountData_proj_Final.rda")

# LA Data
load("../../PROCESSED_DATA/SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")

# Load point data and project to UTM
la_hom_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/la_hom_sf.rds")
la_gta_sf <- readRDS("../../PROCESSED_DATA/CRIME/POINT_PATTERN/la_gta_sf.rds")
la_hom_sf.proj <- lwgeom::st_transform_proj(la_hom_sf,"epsg:32611")
la_gta_sf.proj <- lwgeom::st_transform_proj(la_gta_sf,"epsg:32611")

# Project to UTM the union of the census tracts - Los Angeles City boundary
lacity_boundary <- st_union(ct_LA)
lacity_proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611")
lacity_sp <- as(lacity_proj,"Spatial")
bbox <- lacity_sp@bbox
lacity_spshift <- elide(lacity_sp,shift=-c(bbox[1,1],bbox[2,1]))
bbox_shift <- lacity_spshift@bbox
lacity_spscale <- elide(lacity_spshift,scale=max(bbox_shift)/1e4)
W <- as.owin.SpatialPolygons((lacity_spscale))

# Extract 2014 data from sf object
la_gta14 <- la_gta_sf[which(la_gta_sf$Y==2014),]
la_hom14 <- la_hom_sf[which(la_hom_sf$Y==2014),]

# Project the 2014 LA crime to UTM and scale and shift as with the LA city boundary
bbox <- lacity_sp@bbox
bbox_shift <- lacity_spshift@bbox
la_hom.proj <- lwgeom::st_transform_proj(la_hom14,"epsg:32611")
la_hom.projsp <- as(la_hom.proj,"Spatial")
la_hom_spshift <- elide(la_hom.projsp,bb=lacity_sp@bbox,shift=-c(bbox[1,1],bbox[2,1]))
la_hom_spscale <- elide(la_hom_spshift,bb=lacity_spshift@bbox,scale=max(bbox_shift)/1e4)
bbox <- lacity_sp@bbox
bbox_shift <- lacity_spshift@bbox
la_gta.proj <- lwgeom::st_transform_proj(la_gta14,"epsg:32611")
la_gta.projsp <- as(la_gta.proj,"Spatial")
la_gta_spshift <- elide(la_gta.projsp,bb=lacity_sp@bbox,shift=-c(bbox[1,1],bbox[2,1]))
la_gta_spscale <- elide(la_gta_spshift,bb=lacity_spshift@bbox,scale=max(bbox_shift)/1e4)

# Turn into ppp object - the shifted and scaled point data, not the project to UTM point data
la_hom.proj.loc <- la_hom_spscale@coords
lahom.ppp <- as.ppp(la_hom.proj.loc,W) # 1 point outside boundary
la_gta.proj.loc <- la_gta_spscale@coords
lagta.ppp <- as.ppp(la_gta.proj.loc,W) # 41 points outside boundary

# Store the scale and shift information for further use
bbox <- lacity_sp@bbox
bbox_shift <- lacity_spshift@bbox


# The below we created from the count data, over the census tracts but no longer used as the rasters and pixel images for the ppm() modelling were generated another way.
# # Generate standardised variables for the sf count data and shift and scale as necessary
# hom.df <- as(sf_homcount,"Spatial")
# hom.df$zpop <- (hom.df$pop - mean(hom.df$pop))/sd(hom.df$pop)
# hom.df$zinc <- (hom.df$inc - mean(hom.df$inc))/sd(hom.df$inc)
# hom.dfshift <- elide(hom.df,bb=bbox,shift=-c(bbox[1,1],bbox[2,1]))
# hom.dfscale <- elide(hom.dfshift,bb=bbox_shift,scale=max(bbox_shift)/1e4)
# 
# gta.df <- as(sf_gtacount,"Spatial")
# gta.df$zpop <- (gta.df$pop - mean(gta.df$pop))/sd(gta.df$pop)
# gta.df$zinc <- (gta.df$inc - mean(gta.df$inc))/sd(gta.df$inc)
# gta.dfshift <- elide(gta.df,bb=bbox,shift=-c(bbox[1,1],bbox[2,1]))
# gta.dfscale <- elide(gta.dfshift,bb=bbox_shift,scale=max(bbox_shift)/1e4)


# Resolution for the rasters (roughly 200m-by-200m)
nr <- 359
nc <- 236

# Assign covariates to census tracts
count.cells <- ct_LA.proj
count.cells$pop <- LA_ctpop_14$pop[match(count.cells$TRACTCE,as.character(sapply(1:length(LA_ctpop_14$geoid2),function(i){geo2ctLA(LA_ctpop_14$geoid2[i])})))]
count.cells$inc <- LA_ctinc_14$inc[match(count.cells$TRACTCE,as.character(sapply(1:length(LA_ctinc_14$geoid2),function(i){geo2ctLA(LA_ctinc_14$geoid2[i])})))]
W.proj <- as.owin.SpatialPolygons((lacity_sp))

# We now want to generate the covariates over a fine grid, but not interpolated, just assigning the values to the census tracts the cells lie within.
g <- quadrats(W.proj,nx=nc,ny=nr)
la_cells <- as(g,"SpatialPolygons")
la_cells_centre <- t(sapply(la_cells@polygons, function(x){x@Polygons[[1]]@labpt}))
cells.centre.df <- data.frame(x=la_cells_centre[,1],y=la_cells_centre[,2])
coordinates(cells.centre.df) <- ~ x + y
cells.centre.df <- st_as_sf(cells.centre.df)
st_crs(cells.centre.df) <- st_crs(count.cells)

# Assign the necessary values
cttocell <- st_intersects(cells.centre.df,count.cells)
cells.centre.df$pop[which(lengths(cttocell)!=0)] <- count.cells$pop[unlist(cttocell)]
# NA to 0
cells.centre.df$pop[is.na(cells.centre.df$pop)] <- 0
cells.centre.df$inc[which(lengths(cttocell)!=0)] <- count.cells$inc[unlist(cttocell)]
# NA to 0
cells.centre.df$inc[is.na(cells.centre.df$inc)] <- 0

# Calculate standardised variables and then shift and scale as with the window, so that unit change in x or y direction is related to a real distance shift of 10km and the bottom=left corner of the bounding box of the LA polygon, lies on the origin, (0,0)
cells.centre.df <- as(cells.centre.df,"Spatial")
cells.centre.df$zpop <- (cells.centre.df$pop - mean(cells.centre.df$pop))/sd(cells.centre.df$pop)
cells.centre.df$zinc <- (cells.centre.df$inc - mean(cells.centre.df$inc))/sd(cells.centre.df$inc)
cells.centre.dfshift <- elide(cells.centre.df,bb=bbox,shift=-c(bbox[1,1],bbox[2,1]))
cells.centre.dfscale <- elide(cells.centre.dfshift,bb=bbox_shift,scale=max(bbox_shift)/1e4)

# Create raster which can then be used to produce pixel images
w.r <- raster(lacity_spscale,nrow=nr,ncol=nc)
pop.ras <- rasterize(cells.centre.dfscale,w.r,field=cells.centre.dfscale$pop)
inc.ras <- rasterize(cells.centre.dfscale,w.r,field=cells.centre.dfscale$inc)
zpop.ras <- rasterize(cells.centre.dfscale,w.r,field=cells.centre.dfscale$zpop)
zinc.ras <- rasterize(cells.centre.dfscale,w.r,field=cells.centre.dfscale$zinc)
int.ras <- rasterize(cells.centre.dfscale,w.r,field=rep(1,length(cells.centre.dfscale)))

# Need a base to cover the entire window, including the region outside the window, otherwise there are warnings about the covariates not covering the LA polygon when fitting the ppm() model
b.r <- raster(extent(pop.ras),nrow=pop.ras@nrows,ncol=pop.ras@ncols)
bbox.base <- owin(xrange=c(extent(pop.ras)[1],extent(pop.ras)[2]),yrange=c(extent(pop.ras)[3],extent(pop.ras)[4]))
gc.df <- gridcenters(bbox.base,1e3,1e3)
gc.df <- data.frame(x=gc.df$x,y=gc.df$y)
coordinates(gc.df) <- ~ x + y
b.ras <- rasterize(gc.df,b.r,rep(0,length(gc.df)))

# Generate the pixel images
popb.im <- as.im(merge(pop.ras,b.ras))
incb.im <- as.im(merge(inc.ras,b.ras))
zpopb.im <- as.im(merge(zpop.ras,b.ras))
zincb.im <- as.im(merge(zinc.ras,b.ras))
intb.im <- as.im(merge(int.ras,b.ras))

# Generate Mean field pixel images for homicide and motor vehicle point patterns using the estimates from the ppm() function in order to call within the minimum.contrast function below.
# We will only consider the models for the standardised variables.
mod.hom <- ppm(lahom.ppp ~ c1 + c2, covariates=list(c1=zpopb.im,c2=zincb.im))
homlambda.im <- exp(intb.im*mod.hom$coef[[1]] + zpopb.im*mod.hom$coef[[2]] + zincb.im*mod.hom$coef[[3]])
mod.gta <- ppm(lagta.ppp ~ c1 + c2, covariates=list(c1=zpopb.im,c2=zincb.im))
# Warning message:
# In countingweights(id, areas) :
# some tiles with positive area do not contain any quadrature points: relative error = 1.1%
gtalambda.im <- exp(intb.im*mod.gta$coef[[1]] + zpopb.im*mod.gta$coef[[2]] + zincb.im*mod.gta$coef[[3]])


# Los Angeles 2014: Minimum Contrast: Homicide ----------------------------
# Implement the minimum.contrast function to the 2014 homicide point pattern using different combinations of for different functions (PCF or Ripley's K) and for different transformations. Note that the fourth root transformation in combination with Ripley's K was not originally run within this script but later implemented and therefore the code has been included.

# PCF with no transformation
minc.hom <- minimum.contrast(lahom.ppp,"matern",intens = homlambda.im,nu=1)
minc.hom

# PCF with log() transformation
minc.hom <- minimum.contrast(lahom.ppp,"matern",intens = homlambda.im,nu=1,transform=log)
minc.hom

# Ripley's K with no transformation
minc.hom <- minimum.contrast(lahom.ppp,"matern",method = "K",intens = homlambda.im,nu=1)
minc.hom

# Ripley's K with log() transformation
minc.hom <- minimum.contrast(lahom.ppp,"matern",method = "K",intens = homlambda.im,nu=1,transform=log)
minc.hom


# Ripley's K with fourth root transformation (the below was not initially run with the above values and so not included in thesis results but run later to compare).
minc.hom <- minimum.contrast(lahom.ppp,"matern",method = "K",intens = homlambda.im,nu=1,power=0.25)
minc.hom


# Los Angeles 2014: Minimum Contrast: Motor Vehicle Theft -----------------
# Implement the minimum.contrast function to the 2014 motor vehicle theft point pattern using different combinations of functions (PCF or Ripley's K) and for different transformations. Note that the fourth root transformation in combination with Ripley's K was not originally run within this script but later implemented and therefore the code necessary has been included.

# PCF with no transformation
minc.gta <- minimum.contrast(lagta.ppp,"matern",intens = gtalambda.im,nu=1)
minc.gta

# PCF with log() transformation
minc.gta <- minimum.contrast(lagta.ppp,"matern",intens = gtalambda.im,nu=1,transform=log)
minc.gta

# Ripley's K with no transformation
minc.gta <- minimum.contrast(lagta.ppp,"matern",method = "K",intens = gtalambda.im,nu=1)
minc.gta

# Ripley's K with log() transformation
minc.gta <- minimum.contrast(lagta.ppp,"matern",method = "K",intens = gtalambda.im,nu=1,transform=log)
minc.gta


# Ripley's K with fourth root transformation (the below was not initially run with the above values and so not included in thesis results but run later to compare).
minc.gta <- minimum.contrast(lagta.ppp,"matern",method = "K",intens = gtalambda.im,nu=1,power=0.25)
minc.gta



# sessionInfo() -----------------------------------------------------------

sessionInfo()
