
# Creating Census Tract Level Count Data ----------------------------------

# This R script contains the code for creating the count data for homicides and motor vehicle thefts at the census tract level for LA, NYC and Portland. The final count data frames contain the counts, census tract areas, the projected centres of the census tracts as well as values for socio-economic variables.
# NOTE: in the below code, the loading of and assignment of socio-economic variables other than total population and average income have been commented out. When run these were included for my data, however as they are not used for any modelling, they are not necessary and the only variables of interest would be population and income. Therefore, while the lines of code are kept in order to maintain all code used to generate the final count data used, they are commented out and can be un-commented as needed.

# Author: Nadeen Khaleel


# Setwd and Load Libraries ------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))

library(sp)
library(spatstat)
library(sf)
library(dplyr)
library(readr)
library(maptools)


# Los Angeles -------------------------------------------------------------
# This section of code creates the count data for the 2015 incidents homicides and motor vehicle thefts in LA over the census tracts where the final data frames will also contain a selection of  socio-economica variables for the corresponding census tracts.

# Los Angeles: Set-Up -----------------------------------------------------
# Change the working directory as we want to save the data in separate folders for each city. We then load the census tract data produced in DATA/RAW_DATA/SHAPEFILES/CENSUS_TRACTS and project these to UTM coordinates. We also load the spatial point patterns for the two crimes and the socio-economic variables where we project the points of homicide and motor vehicle thefts.

# LA Data
setwd("./LA")

# LA Census Tracts
load("../../../SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")

# LA Point Patterns
# Load and project to UTM
la_hom <- readRDS("../../POINT_PATTERN/la_hom.rds")
la_gta <- readRDS("../../POINT_PATTERN/la_gta.rds")
la_hom_sf <- readRDS("../../POINT_PATTERN/la_hom_sf.rds")
la_gta_sf <- readRDS("../../POINT_PATTERN/la_gta_sf.rds")
la_hom_sf.proj <- lwgeom::st_transform_proj(la_hom_sf,"epsg:32611")
la_gta_sf.proj <- lwgeom::st_transform_proj(la_gta_sf,"epsg:32611")

# LA Socio-economic Variables
LA_ctpop_15 <- readRDS("../../../COVARIATES/LA_CTPop_15_proj.rds")
LA_ctinc_15 <- readRDS("../../../COVARIATES/LA_CTInc_15_0imp_proj.rds")
# LA_ctsex_15 <- readRDS("../../../COVARIATES/LA_CTSex_15_proj.rds")
# LA_ctage_15 <- readRDS("../../../COVARIATES/LA_CTAge_15_proj.rds")
# LA_ctocc_15 <- readRDS("../../../COVARIATES/LA_CTOcc_15_proj.rds")
# LA_ctfood_15 <- readRDS("../../../COVARIATES/LA_CTFood_15_proj.rds")


# Project to UTM: both the census tracts and their union - which will form a boundary for the city. While the union of the census tracts is not needed, the creation of it in order to generate the proj4string which is used later is important and they will not be removed.
ct_LA.proj <- lwgeom::st_transform_proj(ct_LA,"epsg:32611")
lacity_boundary <- st_union(ct_LA)
lacity_boundary.proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611")
lacity_geom <- as(lacity_boundary,"Spatial")
lacity_geom.proj <- as(lacity_boundary.proj,"Spatial")
p4s <- lacity_geom.proj@proj4string


# Los Angeles: 2015 Crime Data Points to Census Tract Counts --------------
# We want to extract the crime data (homicide and motor vehicle theft) for the year 2015 and then turn count data over all of the census tracts with the projected points and sensus tracts formed above. Note that having noticed a mis-match in total counts and total number of points for the motor vehicle thefts we do find that there are duplications in the motor vehicle theft data, where it appears as though some points were double counted - perhaps they lay on the boundary of two census tracts, with no particular way to discern where they belong we randomly assign them to one of the census tracts they lie on the border of.

# Extract 2015 data
la_hom15_sf.proj <- la_hom_sf.proj[which(la_hom_sf.proj$Y==2015),]
la_gta15_sf.proj <- la_gta_sf.proj[which(la_gta_sf.proj$Y==2015),]

# Calculate area of the census tracts and get coordinates for the center coordinates of the census tracts
ct.area <- st_area(ct_LA.proj)
ct.centres <- st_centroid(ct_LA.proj)
ct.centres <- st_coordinates(ct.centres)

# Calculate the counts of crime from 2015 in each census tract by using st_covers. Having noticed a mis-match in the number of points and total counts for motor vehicle thefts for LA I check these for duplicates and try to resolve the issue by removing th relevant duplicates.
# The order of the numbers in the list should follow the order of the census tract names in ct_LA.proj

# Homicide Counts
hom_int_ct <- st_covers(ct_LA.proj,la_hom15_sf.proj)
# Check we don't have duplicates
sum(duplicated(unlist(hom_int_ct)))
# [1] 0

# Motor Vehicle Theft Counts
gta_int_ct <- st_covers(ct_LA.proj,la_gta15_sf.proj)
# Check for duplicates - there are some (two), need to find them and remove one of the duplicates for each.
sum(duplicated(unlist(gta_int_ct)))
# [1] 2
which(duplicated(unlist(gta_int_ct)))
# [1] 308 321
dup.points <- unlist(gta_int_ct)[which(duplicated(unlist(gta_int_ct)))]
dup.points
# [1]  555 1074
ind.dup <- which(unlist(gta_int_ct)%in%dup.points)
cs_gta <- cumsum(lengths(gta_int_ct))
which.ct <- sapply(1:length(ind.dup),function(i){max(which(cs_gta<ind.dup[i]))})
which.ct # which of the elements of the list contain the duplicated values
# [1] 17 17 18 18
which.ct <- which.ct + 1
gta_int_ct[unique(which.ct)]
# [[1]]
# [1]  541  555  600  729  850  903 1032 1074
# 
# [[2]]
# [1]   40  441  456  471  555  641  657  731  753  776  825  926  948 1007 1037 1046 1061 1074

# This makes sense as the census tracts that `overlap' both are probably next to each other, but how to split the points across each?
st_geometry(la_gta15_sf.proj)[dup.points]
# Geometry set for 2 features 
# geometry type:  POINT
# dimension:      XY
# bbox:           xmin: 383354.9 ymin: 3771348 xmax: 383354.9 ymax: 3771348
# projected CRS:  WGS 84 / UTM zone 11N
# POINT (383354.9 3771348)
# POINT (383354.9 3771348)

# These are the same location, so will randomly allocate the points and remove them from the other census tract.
set.seed(25)
loc <- sample(unique(which.ct),2,replace = TRUE)
loc
# [1] 18 18
inc.ind <- loc
rem.ind <- rep(NA,2)
rem.ind[1] <- setdiff(unique(which.ct),inc.ind[1])
rem.ind[2] <- setdiff(unique(which.ct),inc.ind[2])
rem.ind
# [1] 19 19

gta_int_ct[[rem.ind[1]]]
# [1]   40  441  456  471  555  641  657  731  753  776  825  926  948 1007 1037 1046 1061 1074
gta_int_ct[[rem.ind[1]]] <- setdiff(gta_int_ct[[rem.ind[1]]],dup.points[1])
gta_int_ct[[rem.ind[2]]]
# [1]   40  441  456  471  641  657  731  753  776  825  926  948 1007 1037 1046 1061 1074
gta_int_ct[[rem.ind[2]]] <- setdiff(gta_int_ct[[rem.ind[2]]],dup.points[2])

# Check that the duplicates have been removed.
sum(duplicated(unlist(gta_int_ct)))
# [1] 0

# Get counts of homicides and motor vehicle thefts for each.
hom_ctLA_cov <- lengths(hom_int_ct)
gta_ctLA_cov <- lengths(gta_int_ct)


# Los Angeles: Census Tract Count Data Frames -----------------------------
# Here we will take the counts created above and form a data frame for each crime, for the census tracts in LA we will then extract the necessary values of the socio-economic variables and place them in these data frames. We then save these data frames as they are and as sf objects.

ct_homcount.df <- data.frame(ct=ct_LA.proj$TRACTCE,x=ct.centres[,1],y=ct.centres[,2],ctarea=ct.area,hom=hom_ctLA_cov)
ct_gtacount.df <- data.frame(ct=ct_LA.proj$TRACTCE,x=ct.centres[,1],y=ct.centres[,2],ctarea=ct.area,gta=gta_ctLA_cov)

geo2ctLA <- function(x){ct_LA$TRACTCE[which(ct_LA$GEOID==x)]}

# Get the required socio-economic variables values for each census tract in the count data frames above.
count.pop <- LA_ctpop_15$pop[match(ct_homcount.df$ct,as.character(sapply(1:length(LA_ctpop_15$geoid2),function(i){geo2ctLA(LA_ctpop_15$geoid2[i])})))]
count.inc <- LA_ctinc_15$inc[match(ct_homcount.df$ct,as.character(sapply(1:length(LA_ctinc_15$geoid2),function(i){geo2ctLA(LA_ctinc_15$geoid2[i])})))]
# count.msex <- LA_ctsex_15$mprop[match(ct_homcount.df$ct,as.character(sapply(1:length(LA_ctsex_15$geoid2),function(i){geo2ctLA(LA_ctsex_15$geoid2[i])})))]
# count.fsex <- LA_ctsex_15$fprop[match(ct_homcount.df$ct,as.character(sapply(1:length(LA_ctsex_15$geoid2),function(i){geo2ctLA(LA_ctsex_15$geoid2[i])})))]
# count.food <- LA_ctfood_15$proprec[match(ct_homcount.df$ct,as.character(sapply(1:length(LA_ctfood_15$geoid2),function(i){geo2ctLA(LA_ctfood_15$geoid2[i])})))]
# count.own <- LA_ctocc_15$ownprop[match(ct_homcount.df$ct,as.character(sapply(1:length(LA_ctocc_15$geoid2),function(i){geo2ctLA(LA_ctocc_15$geoid2[i])})))]
# count.rent <- LA_ctocc_15$rentprop[match(ct_homcount.df$ct,as.character(sapply(1:length(LA_ctocc_15$geoid2),function(i){geo2ctLA(LA_ctocc_15$geoid2[i])})))]

# Place the variables into the count data frames.
ct_homcount.df$pop <- count.pop
ct_homcount.df$inc <- count.inc
# ct_homcount.df$msex <- count.msex
# ct_homcount.df$fsex <- count.fsex
# ct_homcount.df$food <- count.food
# ct_homcount.df$own <- count.own
# ct_homcount.df$rent <- count.rent

ct_gtacount.df$pop <- count.pop
ct_gtacount.df$inc <- count.inc
# ct_gtacount.df$msex <- count.msex
# ct_gtacount.df$fsex <- count.fsex
# ct_gtacount.df$food <- count.food
# ct_gtacount.df$own <- count.own
# ct_gtacount.df$rent <- count.rent

# Turn data frames into spatial objects, in particular sf objects
spdf_ct_homcount <- SpatialPointsDataFrame(coords=ct_homcount.df[,c("x","y")],data=ct_homcount.df,proj4string = p4s)
sf_homcount <- st_as_sf(spdf_ct_homcount)
spdf_ct_gtacount <- SpatialPointsDataFrame(coords=ct_gtacount.df[,c("x","y")],data=ct_gtacount.df,proj4string = p4s)
sf_gtacount <- st_as_sf(spdf_ct_gtacount)

# Save
save(hom_countdf=ct_homcount.df,gta_countdf=ct_gtacount.df,file=paste0("LA2015CTCountData_projFinal.rda"))
save(sf_homcount,sf_gtacount,file=paste0("LA2015CTSFCountData_projFinal.rda"))


# Los Angeles: Re-set WD --------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))


# New York ----------------------------------------------------------------
# This section of code creates the count data for the 2015 incidents homicides and motor vehicle thefts in NYC over the census tracts where the final data frames will also contain a selection of  socio-economica variables for the corresponding census tracts.

# New York: Set-Up --------------------------------------------------------
# Change the working directory as we want to save the data in separate folders for each city. We then load the census tract data produced in DATA/RAW_DATA/SHAPEFILES/CENSUS_TRACTS and project these to UTM coordinates. We also load the spatial point patterns for the two crimes and the socio-economic variables where we project the points of homicide and motor vehicle thefts.

# NYC Data
setwd("./NYC")

# NYC Census Tracts
load("../../../SHAPEFILES/CENSUS_TRACTS/NYCityCT.rda")

# NYC Point Pattern
# Load and project to UTM
nyc_hom <- readRDS("../../POINT_PATTERN/nyc_hom.rds")
nyc_gta <- readRDS("../../POINT_PATTERN/nyc_gta.rds")
nyc_hom_sf <- readRDS("../../POINT_PATTERN/nyc_hom_sf.rds")
nyc_gta_sf <- readRDS("../../POINT_PATTERN/nyc_gta_sf.rds")
nyc_hom_sf.proj <- lwgeom::st_transform_proj(nyc_hom_sf,"epsg:32618")
nyc_gta_sf.proj <- lwgeom::st_transform_proj(nyc_gta_sf,"epsg:32618")

# NYC Socio-economic Variables
NY_ctpop_15 <- readRDS("../../../COVARIATES/NY_CTPop_15_proj.rds")
NY_ctinc_15 <- readRDS("../../../COVARIATES/NY_CTInc_15_0imp_proj.rds")
# NY_ctsex_15 <- readRDS("../../../COVARIATES/NY_CTSex_15_proj.rds")
# NY_ctage_15 <- readRDS("../../../COVARIATES/NY_CTAge_15_proj.rds")
# NY_ctocc_15 <- readRDS("../../../COVARIATES/NY_CTOcc_15_proj.rds")
# NY_ctfood_15 <- readRDS("../../../COVARIATES/NY_CTFood_15_proj.rds")

# Project to UTM: both the census tracts and their union - which will form a boundary for the city. While the union of the census tracts is not needed, the creation of it in order to generate the proj4string which is used later is important and they will not be removed.
ct_NY.proj <- lwgeom::st_transform_proj(ct_NY,"epsg:32618")
nycity_boundary <- st_union(ct_NY)
nycity_boundary.proj <- lwgeom::st_transform_proj(nycity_boundary,"epsg:32618")
nycity_geom <- as(nycity_boundary,"Spatial")
nycity_geom.proj <- as(nycity_boundary.proj,"Spatial")
nycity_win <- as.owin.SpatialPolygons((nycity_geom))
p4s <- nycity_geom.proj@proj4string


# New York: 2015 Crime Data Points to Census Tract Counts -----------------
# We want to extract the crime data (homicide and motor vehicle theft) for the year 2015 and then turn count data over all of the census tracts with the projected points and sensus tracts formed above.

# Extract 2015 data
nyc_hom15_sf.proj <- nyc_hom_sf.proj[which(nyc_hom_sf.proj$Y==2015),]
nyc_gta15_sf.proj <- nyc_gta_sf.proj[which(nyc_gta_sf.proj$Y==2015),]

# Calculate area of the census tracts and get coordinates for the center coordinates of the census tracts
ct.area <- st_area(ct_NY.proj)
ct.centres <- st_centroid(ct_NY.proj)
ct.centres <- st_coordinates(ct.centres)

# Calculate the counts of crime from 2015 in each census tract by using st_covers.
# The order of the numbers in the list should follow the order of the census tract names in ct_NY.proj

# Homicide Counts
hom_ctNY_cov <- lengths(st_covers(ct_NY.proj,nyc_hom15_sf.proj))
# Motor Vehicle Theft Counts
gta_ctNY_cov <- lengths(st_covers(ct_NY.proj,nyc_gta15_sf.proj))


# New York: Census Tract Count Data Frames --------------------------------
# Here we will take the counts created above and form a data frame for each crime, for the census tracts in NYC we will then extract the necessary values of the socio-economic variables and place them in these data frames. We then save these data frames as they are and as sf objects.

# Need to use GEOID for New York as the census tracts are not unique within the city due to multiple counties.
ct_homcount.df <- data.frame(ct=ct_NY.proj$GEOID,x=ct.centres[,1],y=ct.centres[,2],ctarea=ct.area,hom=hom_ctNY_cov)
ct_gtacount.df <- data.frame(ct=ct_NY.proj$GEOID,x=ct.centres[,1],y=ct.centres[,2],ctarea=ct.area,gta=gta_ctNY_cov)

# Get the required socio-economic variables values for each census tract in the count data frames above.
count.pop <- NY_ctpop_15$pop[match(ct_homcount.df$ct,as.character(NY_ctpop_15$geoid2))]
count.inc <- NY_ctinc_15$inc[match(ct_homcount.df$ct,as.character(NY_ctinc_15$geoid2))]
# count.msex <- NY_ctsex_15$mprop[match(ct_homcount.df$ct,as.character(NY_ctsex_15$geoid2))]
# count.fsex <- NY_ctsex_15$fprop[match(ct_homcount.df$ct,as.character(NY_ctsex_15$geoid2))]
# count.food <- NY_ctfood_15$proprec[match(ct_homcount.df$ct,as.character(NY_ctfood_15$geoid2))]
# count.own <- NY_ctocc_15$ownprop[match(ct_homcount.df$ct,as.character(NY_ctocc_15$geoid2))]
# count.rent <- NY_ctocc_15$rentprop[match(ct_homcount.df$ct,as.character(NY_ctocc_15$geoid2))]

# Place the variables into the count data frames.
ct_homcount.df$pop <- count.pop
ct_homcount.df$inc <- count.inc
# ct_homcount.df$msex <- count.msex
# ct_homcount.df$fsex <- count.fsex
# ct_homcount.df$food <- count.food
# ct_homcount.df$own <- count.own
# ct_homcount.df$rent <- count.rent

ct_gtacount.df$pop <- count.pop
ct_gtacount.df$inc <- count.inc
# ct_gtacount.df$msex <- count.msex
# ct_gtacount.df$fsex <- count.fsex
# ct_gtacount.df$food <- count.food
# ct_gtacount.df$own <- count.own
# ct_gtacount.df$rent <- count.rent

# Turn data frames into spatial objects, in particular sf objects
spdf_ct_homcount <- SpatialPointsDataFrame(coords=ct_homcount.df[,c("x","y")],data=ct_homcount.df,proj4string = p4s)
sf_homcount <- st_as_sf(spdf_ct_homcount)
spdf_ct_gtacount <- SpatialPointsDataFrame(coords=ct_gtacount.df[,c("x","y")],data=ct_gtacount.df,proj4string = p4s)
sf_gtacount <- st_as_sf(spdf_ct_gtacount)

# Save
save(ct_homcount.df,ct_gtacount.df,file=paste0("NY2015CTCountData_projFinal.rda"))
save(sf_homcount,sf_gtacount,file=paste0("NY2015CTSFCountData_projFinal.rda"))


# New York: Re-set WD -----------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))



# Portland ----------------------------------------------------------------
# This section of code creates the count data for the 2015 incidents homicides and motor vehicle thefts in Portland over the census tracts where the final data frames will also contain a selection of  socio-economica variables for the corresponding census tracts.

# Portland: Set-Up --------------------------------------------------------
# Change the working directory as we want to save the data in separate folders for each city. We then load the census tract data produced in DATA/RAW_DATA/SHAPEFILES/CENSUS_TRACTS and project these to UTM coordinates. We also load the spatial point patterns for the two crimes and the socio-economic variables where we project the points of homicide and motor vehicle thefts.

# Portland Data
setwd("./Portland")

# Portland Census Tracts
load("../../../SHAPEFILES/CENSUS_TRACTS/PCityCT.rda")

# Portland Point Pattern
# Load and project to UTM
p_hom <- readRDS("../../POINT_PATTERN/p_hom.rds")
p_gta <- readRDS("../../POINT_PATTERN/p_gta.rds")
p_hom_sf <- readRDS("../../POINT_PATTERN/p_hom_sf.rds")
p_gta_sf <- readRDS("../../POINT_PATTERN/p_gta_sf.rds")
p_hom_sf.proj <- lwgeom::st_transform_proj(p_hom_sf,"epsg:32610")
p_gta_sf.proj <- lwgeom::st_transform_proj(p_gta_sf,"epsg:32610")

# NYC Socio-economic Variables
P_ctpop_15 <- readRDS("../../../COVARIATES/P_CTPop_15_proj.rds")
P_ctinc_15 <- readRDS("../../../COVARIATES/P_CTInc_15_0imp_proj.rds")
# P_ctsex_15 <- readRDS("../../../COVARIATES/P_CTSex_15_proj.rds")
# P_ctage_15 <- readRDS("../../../COVARIATES/P_CTAge_15_proj.rds")
# P_ctocc_15 <- readRDS("../../../COVARIATES/P_CTOcc_15_proj.rds")
# P_ctfood_15 <- readRDS("../../../COVARIATES/P_CTFood_15_proj.rds")

# Project to UTM: both the census tracts and their union - which will form a boundary for the city. While the union of the census tracts is not needed, the creation of it in order to generate the proj4string which is used later is important and they will not be removed.
ct_P.proj <- lwgeom::st_transform_proj(ct_P,"epsg:32610")
pcity_boundary <- st_union(ct_P)
pcity_boundary.proj <- lwgeom::st_transform_proj(pcity_boundary,"epsg:32610")
pcity_geom <- as(pcity_boundary,"Spatial")
pcity_geom.proj <- as(pcity_boundary.proj,"Spatial")
p4s <- pcity_geom.proj@proj4string


# Portland: 2015 Crime Data Points to Census Tract Counts -----------------
# We want to extract the crime data (homicide and motor vehicle theft) for the year 2015 and then turn count data over all of the census tracts with the projected points and sensus tracts formed above.

# Extract 2015 data
p_hom15_sf.proj <- p_hom_sf.proj[which(p_hom_sf.proj$Y==2015),]
p_gta15_sf.proj <- p_gta_sf.proj[which(p_gta_sf.proj$Y==2015),]

# Calculate area of the census tracts and get coordinates for the center coordinates of the census tracts
ct.area <- st_area(ct_P.proj)
ct.centres <- st_centroid(ct_P.proj)
ct.centres <- st_coordinates(ct.centres)

# Calculate the counts of crime from 2015 in each census tract by using st_covers.
# The order of the numbers in the list should follow the order of the census tract names in ct_P.proj

# Homicide Counts
hom_ctP_cov <- lengths(st_covers(ct_P.proj,p_hom15_sf.proj))

# Motor Vehicle Theft Counts
gta_ctP_cov <- lengths(st_covers(ct_P.proj,p_gta15_sf.proj))


# Portland: Census Tract Count Data Frames --------------------------------
# Here we will take the counts created above and form a data frame for each crime, for the census tracts in Portland we will then extract the necessary values of the socio-economic variables and place them in these data frames. We then save these data frames as they are and as sf objects.

ct_homcount.df <- data.frame(ct=ct_P.proj$TRACTCE,x=ct.centres[,1],y=ct.centres[,2],ctarea=ct.area,hom=hom_ctP_cov)
ct_gtacount.df <- data.frame(ct=ct_P.proj$TRACTCE,x=ct.centres[,1],y=ct.centres[,2],ctarea=ct.area,gta=gta_ctP_cov)

geo2ctP <- function(x){ct_P$TRACTCE[which(ct_P$GEOID==x)]}

# Get the required socio-economic variables values for each census tract in the count data frames above.
count.pop <- P_ctpop_15$pop[match(ct_homcount.df$ct,as.character(sapply(1:length(P_ctpop_15$geoid2),function(i){geo2ctP(P_ctpop_15$geoid2[i])})))]
count.inc <- P_ctinc_15$inc[match(ct_homcount.df$ct,as.character(sapply(1:length(P_ctinc_15$geoid2),function(i){geo2ctP(P_ctinc_15$geoid2[i])})))]
# count.msex <- P_ctsex_15$mprop[match(ct_homcount.df$ct,as.character(sapply(1:length(P_ctsex_15$geoid2),function(i){geo2ctP(P_ctsex_15$geoid2[i])})))]
# count.fsex <- P_ctsex_15$fprop[match(ct_homcount.df$ct,as.character(sapply(1:length(P_ctsex_15$geoid2),function(i){geo2ctP(P_ctsex_15$geoid2[i])})))]
# count.food <- P_ctfood_15$proprec[match(ct_homcount.df$ct,as.character(sapply(1:length(P_ctfood_15$geoid2),function(i){geo2ctP(P_ctfood_15$geoid2[i])})))]
# count.own <- P_ctocc_15$ownprop[match(ct_homcount.df$ct,as.character(sapply(1:length(P_ctocc_15$geoid2),function(i){geo2ctP(P_ctocc_15$geoid2[i])})))]
# count.rent <- P_ctocc_15$rentprop[match(ct_homcount.df$ct,as.character(sapply(1:length(P_ctocc_15$geoid2),function(i){geo2ctP(P_ctocc_15$geoid2[i])})))]

# Place the variables into the count data frames.
ct_homcount.df$pop <- count.pop
ct_homcount.df$inc <- count.inc
# ct_homcount.df$msex <- count.msex
# ct_homcount.df$fsex <- count.fsex
# ct_homcount.df$food <- count.food
# ct_homcount.df$own <- count.own
# ct_homcount.df$rent <- count.rent

ct_gtacount.df$pop <- count.pop
ct_gtacount.df$inc <- count.inc
# ct_gtacount.df$msex <- count.msex
# ct_gtacount.df$fsex <- count.fsex
# ct_gtacount.df$food <- count.food
# ct_gtacount.df$own <- count.own
# ct_gtacount.df$rent <- count.rent

# Turn data frames into spatial objects, in particular sf objects
spdf_ct_homcount <- SpatialPointsDataFrame(coords=ct_homcount.df[,c("x","y")],data=ct_homcount.df,proj4string = p4s)
sf_homcount <- st_as_sf(spdf_ct_homcount)
spdf_ct_gtacount <- SpatialPointsDataFrame(coords=ct_gtacount.df[,c("x","y")],data=ct_gtacount.df,proj4string = p4s)
sf_gtacount <- st_as_sf(spdf_ct_gtacount)

# Save
save(hom_countdf=ct_homcount.df,gta_countdf=ct_gtacount.df,file=paste0("P2015CTCountData_projFinal.rda"))
save(sf_homcount,sf_gtacount,file=paste0("P2015CTSFCountData_projFinal.rda"))


# Portland: Re-set WD -----------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))



# sessionInfo() -----------------------------------------------------------

sessionInfo()