
# Creating Projected Gridded Counts for Modelling Crime Data --------------

# This R scripts takes the crime point patterns as well as census-tract level socio-economic variables in order to generate a count data frame for each crime within each city that contains the counts of crime within a grid cell as well as the interpolated socio-econonic variables and other variables like the areas of the cells (or rather their area when intersected with the census tracts).
# The outputs from this code are projected UTM coordinates and in the R script CounDataGen_Scale_final.R, we take these projected count data sets and shift and then scale them so that a unit shift in the x or y direction translates into a 10km shift unlike the 1m shift in the data created below, additionally, the shifting results in the bounding box of the city polygons with the bottom-left corner at (0,0) so all of the coordinates are shifted and scaled towards the origin.

# NOTE: in the below code, the loading of and assignment of socio-economic variables other than total population and average income have been commented out. When run these were included for my data, however as they are not used for any modelling, they are not necessary and the only variables of interest would be population and income. Therefore, while the lines of code are kept in order to maintain all code used to generate the final count data used, they are commented out and can be un-commented as needed.

# Author: Nadeen Khaleel

# Setwd and Load Libraries ------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))

library(sp)
library(spatstat)
library(sf)
library(dplyr)
library(readr)
library(maptools)

dim.ind <- 1 # 1 2 3 4 5 # which grid cell dimensions do we want? 200mx200m (1) 500mx500m (2) 1kmx1km (3) or 2kmx2km (4) or 5kmx5km (5)

# Los Angeles -------------------------------------------------------------
# We want to generate gridded count data for the homicides and motor vehicle thefts in Los Angeles with the purpose of modelling the gridded data with INLA algorithm as well as the INLA within MCMC algorithm.

# Los Angeles: Set-Up -----------------------------------------------------
# Change the working directory as we want to save the data in separate folders for each city. We then load the census tract data produced in DATA/RAW_DATA/SHAPEFILES/CENSUS_TRACTS and project these to UTM coordinates. We also load the spatial point patterns for the two crimes and the socio-economic variables where we project the points of homicide and motor vehicle thefts.

# LA Data
setwd("./LA")

# LA Census Tracts
load("../../../SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")

# LA Point Patterns
# Load and project to UTM 
la_hom <- readRDS("../../POINT_PATTERN/la_hom.rds")
la_gta <- readRDS("../../POINT_PATTERN/la_gta.rds")
la_hom_sf <- readRDS("../../POINT_PATTERN/la_hom_sf.rds")
la_gta_sf <- readRDS("../../POINT_PATTERN/la_gta_sf.rds")
la_hom_sf.proj <- lwgeom::st_transform_proj(la_hom_sf,"epsg:32611")
la_gta_sf.proj <- lwgeom::st_transform_proj(la_gta_sf,"epsg:32611")

# LA Socio-economic Variables
LA_ctpop_15 <- readRDS("../../../COVARIATES/LA_CTPop_15_proj.rds")
LA_ctinc_15 <- readRDS("../../../COVARIATES/LA_CTInc_15_0imp_proj.rds") # use the imputed average income where missing data where the estimated total households is equal  to 0 are set to 0 and then the remaining missing data is imputed.
# LA_ctsex_15 <- readRDS("../../../COVARIATES/LA_CTSex_15_proj.rds")
# LA_ctage_15 <- readRDS("../../../COVARIATES/LA_CTAge_15_proj.rds")
# LA_ctocc_15 <- readRDS("../../../COVARIATES/LA_CTOcc_15_proj.rds")
# LA_ctfood_15 <- readRDS("../../../COVARIATES/LA_CTFood_15_proj.rds")


# Project to UTM: both the census tracts and their union - which will form a boundary for the city which is used to create the point patterns.
ct_LA.proj <- lwgeom::st_transform_proj(ct_LA,"epsg:32611")
lacity_boundary <- st_union(ct_LA)
lacity_boundary.proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611")
lacity_geom <- as(lacity_boundary,"Spatial")
lacity_geom.proj <- as(lacity_boundary.proj,"Spatial")
lacity_win <- as.owin.SpatialPolygons((lacity_geom))
lacity_win.proj <- as.owin.SpatialPolygons((lacity_geom.proj))


# Los Angeles: Set-Up Grids -----------------------------------------------
# We now want to set-up the grids for different cell-widths, namely: 5km, 2km, 1km, 500m and 200m. The grid resolution that will be used for modelling depends on the results of the Grid-Mesh Optimisation method and so we build all of the required (projected) gridded crime data for Los Angeles.

diff(lacity_win.proj$yrange)
# [1] 71787.39
diff(lacity_win.proj$xrange)
# [1] 47170.91

# So for approximately 1kmx1km, 2kmx2km  grids etc need following dims
x.range <- diff(lacity_win.proj$xrange)
y.range <- diff(lacity_win.proj$yrange)

grid_cellsxvec <- ceiling(x.range/(1e3*c(0.2,0.5,1,2,5)))
grid_cellsyvec <- ceiling(y.range/(1e3*c(0.2,0.5,1,2,5)))

grid_cellsx <- grid_cellsxvec[dim.ind]
grid_cellsy <- grid_cellsyvec[dim.ind]


# Los-Angeles: Create Crime Point Pattern Data ----------------------------
# Here we turn the location data for homicides and motor vehicle thefts into point patterns, object of type ppp, in order to use quadratcount() to later generate the count data over the grids.

# Extract 2015 data
la_hom15_sf.proj <- la_hom_sf.proj[which(la_hom_sf.proj$Y==2015),]
la_gta15_sf.proj <- la_gta_sf.proj[which(la_gta_sf.proj$Y==2015),]

la_hom15_sp.proj <- as(la_hom15_sf.proj,"Spatial")
la_gta15_sp.proj <- as(la_gta15_sf.proj,"Spatial")

la_hom15_loc.proj <- la_hom15_sp.proj@coords
colnames(la_hom15_loc.proj) <- c("x","y")
la_gta15_loc.proj <- la_gta15_sp.proj@coords
colnames(la_gta15_loc.proj) <- c("x","y")

# Turn in to ppp object
# Note that for some of these the points are lying slightly outside the window of interest, possibly due to the holes and we do not want to shift their location too much as we are gridded to reasonably fine resolutions, so we will leave the data unaltered. This will be consistent across all grids and is consis tent with the census tract count data.
la_hom15.proj.ppp <- as.ppp(la_hom15_loc.proj,lacity_win.proj) # 1 point lying outside window?
la_gta15.proj.ppp <- as.ppp(la_gta15_loc.proj,lacity_win.proj) # 35 points lying outside window?


# Los Angeles: Generate Grids ---------------------------------------------
# We now use the grid dimensions calculated above to create the grid cells using quadrats() and store these as spatial objects.

la_cells <- quadrats(lacity_win.proj,grid_cellsx,grid_cellsy)
quad.la <- la_cells
la_cells <- as(la_cells,"SpatialPolygons")
la_cells_centre <- t(sapply(la_cells@polygons, function(x){x@Polygons[[1]]@labpt}))
la_cells <- st_as_sf(la_cells)
la_cells$ID <- 1:length(la_cells$geometry)
p4s <- lacity_geom.proj@proj4string
la_cells <- st_set_crs(la_cells,p4s@projargs)

# Save
# Want to save the grid cells, la_cells, which are objects of class sf, while also saving the quadrats themselves, quad.la.
save(la_cells,file=paste0("LAGridCells",grid_cellsx,grid_cellsy,"_projFinal.rda"))
save(quad.la,file=paste0("LAQuad",grid_cellsx,grid_cellsy,"_projFinal.rda"))

# Calculate the cell areas for data frame and interpolation.
cell_areas <- st_area(la_cells)


# Los Angeles: Generate Gridded Count Data --------------------------------
# We now use the grid dimensions with quadratcount() to generate the count data sets with centroids provided from the above calculations. We include these locations as well as the counts and cell areas and an identifier for the cell for the interpolation of the socio-economic variables.

la_hom15.proj.qcount <- quadratcount(la_hom15.proj.ppp,nx=grid_cellsx,ny=grid_cellsy)
la_gta15.proj.qcount <- quadratcount(la_gta15.proj.ppp,nx=grid_cellsx,ny=grid_cellsy)


la_hom15.proj.count <- data.frame(x=la_cells_centre[,1],y=la_cells_centre[,2],count=unname(la_hom15.proj.qcount))
la_hom15.proj.count$count.Var1 <- as.numeric(la_hom15.proj.count$count.Var1)
colnames(la_hom15.proj.count) <- c("x","y","cell","count")
la_hom15.proj.count$cellarea <- cell_areas

la_gta15.proj.count <- data.frame(x=la_cells_centre[,1],y=la_cells_centre[,2],count=unname(la_gta15.proj.qcount))
la_gta15.proj.count$count.Var1 <- as.numeric(la_gta15.proj.count$count.Var1)
colnames(la_gta15.proj.count) <- c("x","y","cell","count")
la_gta15.proj.count$cellarea <- cell_areas


# Los Angeles: Socio-economic Variable Generation -------------------------
# We will now use the intersections between the grid cells and census tracts to calculate weighted sums for the interpolation of the socio-economic variables.

# Calculate the intersection of the grid cells and census tracts and calculate the areas of these intersections
s_intLA <- st_intersection(ct_LA.proj,la_cells)
s_areaLA <- as.numeric(st_area(s_intLA))

# Create a data frame for these intersections, with identifiers for the census tracts and cells
int_areaLA <- data.frame(ct=s_intLA$TRACTCE,cell=s_intLA$ID,area=as.numeric(s_areaLA))
int_areaLA <- int_areaLA[order(int_areaLA$ct),]

# Create a data frame for the census tracts only and calculate the areas of the census tracts
ct_areaLA <- data.frame(ct=ct_LA$TRACTCE,totpoly_area=as.numeric(st_area(ct_LA.proj)))
ct_areaLA <- ct_areaLA[order(ct_areaLA$ct),]

# Add the census tract areas to the intersection data frame and calculate the proportion of the intersected areas with respect to the census tract total areas.
int_areaLA$totpoly_area <- ct_areaLA$totpoly_area[match(int_areaLA$ct,ct_areaLA$ct)]
int_areaLA$proppoly_area <- int_areaLA$area/int_areaLA$totpoly_area

# Create a data frame for the grid cells only and calculate the areas of the grid cells
cell_areaLA <- data.frame(cell=la_cells$ID,totcell_area=as.numeric(cell_areas))
cell_areaLA <- cell_areaLA[order(cell_areaLA$cell),]

# Add the grid cell areas to the intersection data frame and calculate the proportion of the intersected areas with respect to the grid cell total areas.
int_areaLA$totcell_area <- cell_areaLA$totcell_area[match(int_areaLA$cell,cell_areaLA$cell)]
int_areaLA$propcell_area <- int_areaLA$area/int_areaLA$totcell_area


geo2ctLA <- function(x){ct_LA$TRACTCE[which(ct_LA$GEOID==x)]}

# Assign the census tract level socio-economic variable values for each census tract in the intersection data frame
int_areaLA$pop <- LA_ctpop_15$pop[match(int_areaLA$ct,as.character(sapply(1:length(LA_ctpop_15$geoid2),function(i){geo2ctLA(LA_ctpop_15$geoid2[i])})))]
int_areaLA$inc <- LA_ctinc_15$inc[match(int_areaLA$ct,as.character(sapply(1:length(LA_ctinc_15$geoid2),function(i){geo2ctLA(LA_ctinc_15$geoid2[i])})))]
# int_areaLA$msex <- LA_ctsex_15$mprop[match(int_areaLA$ct,as.character(sapply(1:length(LA_ctsex_15$geoid2),function(i){geo2ctLA(LA_ctsex_15$geoid2[i])})))]
# int_areaLA$fsex <- LA_ctsex_15$fprop[match(int_areaLA$ct,as.character(sapply(1:length(LA_ctsex_15$geoid2),function(i){geo2ctLA(LA_ctsex_15$geoid2[i])})))]
# int_areaLA$food <- LA_ctfood_15$proprec[match(int_areaLA$ct,as.character(sapply(1:length(LA_ctfood_15$geoid2),function(i){geo2ctLA(LA_ctfood_15$geoid2[i])})))]
# int_areaLA$own <- LA_ctocc_15$ownprop[match(int_areaLA$ct,as.character(sapply(1:length(LA_ctocc_15$geoid2),function(i){geo2ctLA(LA_ctocc_15$geoid2[i])})))]
# int_areaLA$rent <- LA_ctocc_15$rentprop[match(int_areaLA$ct,as.character(sapply(1:length(LA_ctocc_15$geoid2),function(i){geo2ctLA(LA_ctocc_15$geoid2[i])})))]

# For each proportion of areas, proportion of the cell or proportion of the census tract polygons, we calculate the product of the proportion with the value of the socio-economic variables for the relevant census tract and grid cell intersection.

int_areaLA$propcell.pop <- int_areaLA$propcell_area*int_areaLA$pop
int_areaLA$proppoly.pop <- int_areaLA$proppoly_area*int_areaLA$pop

int_areaLA$propcell.inc <- int_areaLA$propcell_area*int_areaLA$inc
int_areaLA$proppoly.inc <- int_areaLA$proppoly_area*int_areaLA$inc

# int_areaLA$propcell.msex <- int_areaLA$propcell_area*int_areaLA$msex
# int_areaLA$proppoly.msex <- int_areaLA$proppoly_area*int_areaLA$msex
# 
# int_areaLA$propcell.fsex <- int_areaLA$propcell_area*int_areaLA$fsex
# int_areaLA$proppoly.fsex <- int_areaLA$proppoly_area*int_areaLA$fsex
# 
# int_areaLA$propcell.food <- int_areaLA$propcell_area*int_areaLA$food
# int_areaLA$proppoly.food <- int_areaLA$proppoly_area*int_areaLA$food
# 
# int_areaLA$propcell.own <- int_areaLA$propcell_area*int_areaLA$own
# int_areaLA$proppoly.own <- int_areaLA$proppoly_area*int_areaLA$own
# 
# int_areaLA$propcell.rent <- int_areaLA$propcell_area*int_areaLA$rent
# int_areaLA$proppoly.rent <- int_areaLA$proppoly_area*int_areaLA$rent


int_areaLA$cellx <-  la_cells_centre[int_areaLA$cell,1]
int_areaLA$celly <-  la_cells_centre[int_areaLA$cell,2]

# Save
saveRDS(int_areaLA,file=paste0("LA_Cov_",grid_cellsx,grid_cellsy,"_projFinal.rds"))


# Combine counts with socio-economic variables
# First, calculate the areal weighted sums for the variables
# a. population and average income
# unlike the output from CountDataGen_GMO_final.R, we use column 17 here which uses the proportion of the area of the grid cell that is intersected with each census tract for the average income. However, we do not alter the weights we've used for the population which remain as the proportion of the area of the census tract that is intersected with each grid.
sumLA2525.a <- aggregate(int_areaLA[, c(16,17)], list(cell=int_areaLA$cell), sum, na.rm = TRUE)

# This new line of code is to create sumLA2525 where, previously, it was created by combining sumLA2525.a and sumLA2525.b, however as the only socio-economic variables considered are total population and average income we have commented out the code to include the other variables, although they remain in this R script as it was the original code to create the data sets. If further variables are used, the code for sumLA2525.b and sumLA2525 where combines sumLA2525.a nd sumLA2525.b can be uncommented, and the following line of code must be commented.
sumLA2525 <- sumLA2525.a

# b. all remaining socio-economic variables
# these are all weighted sums using the proportion of the area of the grid cell that is intersected with each census tract
# sumLA2525.b <- aggregate(int_areaLA[, c(19,21,23,25,27)],list(cell=int_areaLA$cell), sum, na.rm = TRUE)
# sumLA2525 <- merge(sumLA2525.a,sumLA2525.b,by="cell",all=TRUE)


# Los Angeles: Generate Final Count Data Sets -----------------------------
# Merge the counts for each crime with the interpolated variables

hom_countdf <- data.frame("cell"=la_hom15.proj.count$cell,"hom"=la_hom15.proj.count$count,"area"=la_hom15.proj.count$cellarea,"x"=la_hom15.proj.count$x,"y"=la_hom15.proj.count$y)
hom_countdf$pop <- sumLA2525$proppoly.pop
hom_countdf$inc <- sumLA2525$propcell.inc
# hom_countdf$msex <- sumLA2525$propcell.msex
# hom_countdf$fsex <- sumLA2525$propcell.fsex
# hom_countdf$food <- sumLA2525$propcell.food
# hom_countdf$own <- sumLA2525$propcell.own
# hom_countdf$rent <- sumLA2525$propcell.rent

# Turn homicide data frame into spatial objects, in particular sf
spdf_homcount <- SpatialPointsDataFrame(coords=hom_countdf[,c("x","y")],data=hom_countdf,proj4string = p4s)
sf_homcount <- st_as_sf(spdf_homcount)


gta_countdf <- data.frame("cell"=la_gta15.proj.count$cell,"gta"=la_gta15.proj.count$count,"area"=la_gta15.proj.count$cellarea,"x"=la_gta15.proj.count$x,"y"=la_gta15.proj.count$y)
gta_countdf$pop <- sumLA2525$proppoly.pop
gta_countdf$inc <- sumLA2525$propcell.inc
# gta_countdf$msex <- sumLA2525$propcell.msex
# gta_countdf$fsex <- sumLA2525$propcell.fsex
# gta_countdf$food <- sumLA2525$propcell.food
# gta_countdf$own <- sumLA2525$propcell.own
# gta_countdf$rent <- sumLA2525$propcell.rent

# Turn motor vehicle theft data frame into spatial objects, in particular sf 
spdf_gtacount <- SpatialPointsDataFrame(coords=gta_countdf[,c("x","y")],data=gta_countdf,proj4string = p4s)
sf_gtacount <- st_as_sf(spdf_gtacount)

# Save
save(hom_countdf,gta_countdf,file=paste0("LA2015CT",grid_cellsx,grid_cellsy,"CountData_projFinal.rda"))
save(sf_homcount,sf_gtacount,file=paste0("LA2015CT",grid_cellsx,grid_cellsy,"SFCountData_projFinal.rda"))


# Los Angeles: Re-set WD --------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))


# New York ----------------------------------------------------------------
# We want to generate gridded count data for the homicides and motor vehicle thefts in New York with the purpose of modelling the gridded data with INLA algorithm as well as the INLA within MCMC algorithm.

# New York: Set-Up --------------------------------------------------------
# Change the working directory as we want to save the data in separate folders for each city. We then load the census tract data produced in DATA/RAW_DATA/SHAPEFILES/CENSUS_TRACTS and project these to UTM coordinates. We also load the spatial point patterns for the two crimes and the socio-economic variables where we project the points of homicide and motor vehicle thefts.

# NYC Data
setwd("./NYC")

# NYC Census Tracts
load("../../../SHAPEFILES/CENSUS_TRACTS/NYCityCT.rda")

# NYC Point Patterns
# Load and project to UTM 
nyc_hom <- readRDS("../../POINT_PATTERN//nyc_hom.rds")
nyc_gta <- readRDS("../../POINT_PATTERN//nyc_gta.rds")
nyc_hom_sf <- readRDS("../../POINT_PATTERN//nyc_hom_sf.rds")
nyc_gta_sf <- readRDS("../../POINT_PATTERN//nyc_gta_sf.rds")
nyc_hom_sf.proj <- lwgeom::st_transform_proj(nyc_hom_sf,"epsg:32618")
nyc_gta_sf.proj <- lwgeom::st_transform_proj(nyc_gta_sf,"epsg:32618")

# NY Socio-economic Variables
NY_ctpop_15 <- readRDS("../../../COVARIATES/NY_CTPop_15_proj.rds")
NY_ctinc_15 <- readRDS("../../../COVARIATES/NY_CTInc_15_0imp_proj.rds") # use the imputed average income where missing data where the estimated total households is equal  to 0 are set to 0 and then the remaining missing data is imputed.
# NY_ctsex_15 <- readRDS("../../../COVARIATES/NY_CTSex_15_proj.rds")
# NY_ctage_15 <- readRDS("../../../COVARIATES/NY_CTAge_15_proj.rds")
# NY_ctocc_15 <- readRDS("../../../COVARIATES/NY_CTOcc_15_proj.rds")
# NY_ctfood_15 <- readRDS("../../../COVARIATES/NY_CTFood_15_proj.rds")


# Project to UTM: both the census tracts and their union - which will form a boundary for the city for creating the point patterns.
ct_NY.proj <- lwgeom::st_transform_proj(ct_NY,"epsg:32618")
nycity_boundary <- st_union(ct_NY)
nycity_boundary.proj <- lwgeom::st_transform_proj(nycity_boundary,"epsg:32618")
nycity_geom <- as(nycity_boundary,"Spatial")
nycity_geom.proj <- as(nycity_boundary.proj,"Spatial")
nycity_win <- as.owin.SpatialPolygons((nycity_geom))
nycity_win.proj <- as.owin.SpatialPolygons((nycity_geom.proj))


# New York: Set-Up Grids --------------------------------------------------
# We now want to set-up the grids for different cell-widths, namely: 5km, 2km, 1km, 500m and 200m.

diff(nycity_win.proj$yrange)
# [1] 47628.82
diff(nycity_win.proj$xrange)
# [1] 46984.43

# So for approximately 1kmx1km, 2kmx2km  grids etc need following dims
x.range <- diff(nycity_win.proj$xrange)
y.range <- diff(nycity_win.proj$yrange)

grid_cellsxvec <- ceiling(x.range/(1e3*c(0.2,0.5,1,2,5)))
grid_cellsyvec <- ceiling(y.range/(1e3*c(0.2,0.5,1,2,5)))

grid_cellsx <- grid_cellsxvec[dim.ind]
grid_cellsy <- grid_cellsyvec[dim.ind]


# New York: Create Crime Point Pattern Data -------------------------------
# Here we turn the location data for homicides and motor vehicle thefts into point patterns, object of type ppp, in order to use quadratcount() to later generate the count data over the grids.

# Extract 2015 data
nyc_hom15_sf.proj <- nyc_hom_sf.proj[which(nyc_hom_sf.proj$Y==2015),]
nyc_gta15_sf.proj <- nyc_gta_sf.proj[which(nyc_gta_sf.proj$Y==2015),]

nyc_hom15_sp.proj <- as(nyc_hom15_sf.proj,"Spatial")
nyc_gta15_sp.proj <- as(nyc_gta15_sf.proj,"Spatial")

nyc_hom15_loc.proj <- nyc_hom15_sp.proj@coords
colnames(nyc_hom15_loc.proj) <- c("x","y")
nyc_gta15_loc.proj <- nyc_gta15_sp.proj@coords
colnames(nyc_gta15_loc.proj) <- c("x","y")

# Turn in to ppp object
nyc_hom15.proj.ppp <- as.ppp(nyc_hom15_loc.proj,nycity_win.proj)
nyc_gta15.proj.ppp <- as.ppp(nyc_gta15_loc.proj,nycity_win.proj)


# New York: Generate Grids ------------------------------------------------
# We now use the grid dimensions calculated above to create the grid cells using quadrats() and store these as spatial objects.

nyc_cells <- quadrats(nycity_win.proj,grid_cellsx,grid_cellsy)
quad.ny <- nyc_cells
nyc_cells <- as(nyc_cells,"SpatialPolygons")
nyc_cells_centre <- t(sapply(nyc_cells@polygons, function(x){x@Polygons[[1]]@labpt}))
nyc_cells <- st_as_sf(nyc_cells)
nyc_cells$ID <- 1:length(nyc_cells$geometry)
p4s <- nycity_geom.proj@proj4string
nyc_cells <- st_set_crs(nyc_cells,p4s@projargs)

# Save
# Want to save the grid cells, nyc_cells, which are objects of class sf, while also saving the quadrats themselves, quad.ny.
save(nyc_cells,file=paste0("NYGridCells",grid_cellsx,grid_cellsy,"_projFinal.rda"))
save(quad.ny,file=paste0("NYQuad",grid_cellsx,grid_cellsy,"_projFinal.rda"))

# Calculate cell areas for data frame and interpolation
cell_areas <- st_area(nyc_cells)


# New York: Generate Gridded Count Data -----------------------------------
# We now use the grid dimensions with quadratcount() to generate the count data sets with centroids provided from the above calculations. We include these locations as well as the counts and cell areas and an identifier for the cell for the interpolation of the socio-economic variables.

nyc_hom15.proj.qcount <- quadratcount(nyc_hom15.proj.ppp,nx=grid_cellsx,ny=grid_cellsy)
nyc_gta15.proj.qcount <- quadratcount(nyc_gta15.proj.ppp,nx=grid_cellsx,ny=grid_cellsy)


nyc_hom15.proj.count <- data.frame(x=nyc_cells_centre[,1],y=nyc_cells_centre[,2],count=unname(nyc_hom15.proj.qcount))
nyc_hom15.proj.count$count.Var1 <- as.numeric(nyc_hom15.proj.count$count.Var1)
colnames(nyc_hom15.proj.count) <- c("x","y","cell","count")
nyc_hom15.proj.count$cellarea <- cell_areas

nyc_gta15.proj.count <- data.frame(x=nyc_cells_centre[,1],y=nyc_cells_centre[,2],count=unname(nyc_gta15.proj.qcount))
nyc_gta15.proj.count$count.Var1 <- as.numeric(nyc_gta15.proj.count$count.Var1)
colnames(nyc_gta15.proj.count) <- c("x","y","cell","count")
nyc_gta15.proj.count$cellarea <- cell_areas


# New York: Socio-economic Variable Generation ----------------------------
# We will now use the intersections between the grid cells and census tracts to calculate weighted sums for the interpolation of the socio-economic variables.

# Calculate the intersection of the grid cells and census tracts and calculate the areas of these intersections
s_intNY <- st_intersection(ct_NY.proj,nyc_cells)
s_areaNY <- as.numeric(st_area(s_intNY))

# Create a data frame for these intersections, with identifiers for the census tracts and cells - however rather than using the CT code using the GEOID as census tracts code are duplicated across some of the counties within the city.
int_areaNY <- data.frame(ct=s_intNY$GEOID,cell=s_intNY$ID,area=as.numeric(s_areaNY))
int_areaNY <- int_areaNY[order(int_areaNY$ct),]

# Create a data frame for the census tracts only and calculate the areas of the census tracts
ct_areaNY <- data.frame(ct=ct_NY$GEOID,totpoly_area=as.numeric(st_area(ct_NY.proj)))
ct_areaNY <- ct_areaNY[order(ct_areaNY$ct),]

# Add the census tract areas to the intersection data frame and calculate the proportion of the intersected areas with respect to the census tract total areas.
int_areaNY$totpoly_area <- ct_areaNY$totpoly_area[match(int_areaNY$ct,ct_areaNY$ct)]
int_areaNY$proppoly_area <- int_areaNY$area/int_areaNY$totpoly_area

# Create a data frame for the grid cells only and calculate the areas of the grid cells
cell_areaNY <- data.frame(cell=nyc_cells$ID,totcell_area=as.numeric(cell_areas))
cell_areaNY <- cell_areaNY[order(cell_areaNY$cell),]

# Add the grid cell areas to the intersection data frame and calculate the proportion of the intersected areas with respect to the grid cell total areas.
int_areaNY$totcell_area <- cell_areaNY$totcell_area[match(int_areaNY$cell,cell_areaNY$cell)]
int_areaNY$propcell_area <- int_areaNY$area/int_areaNY$totcell_area


# Note that the number of unique census tract codes are not the same as the number of unique geoids, this is because there are multiple counties within New York City with the same census tract codes. Instead use GEOID in ct_NY.proj

int_areaNY$pop <- NY_ctpop_15$pop[match(int_areaNY$ct,as.character(NY_ctpop_15$geoid2))]
int_areaNY$inc <- NY_ctinc_15$inc[match(int_areaNY$ct,as.character(NY_ctinc_15$geoid2))]
# int_areaNY$msex <- NY_ctsex_15$mprop[match(int_areaNY$ct,as.character(NY_ctsex_15$geoid2))]
# int_areaNY$fsex <- NY_ctsex_15$fprop[match(int_areaNY$ct,as.character(NY_ctsex_15$geoid2))]
# int_areaNY$food <- NY_ctfood_15$proprec[match(int_areaNY$ct,as.character(NY_ctfood_15$geoid2))]
# int_areaNY$own <- NY_ctocc_15$ownprop[match(int_areaNY$ct,as.character(NY_ctocc_15$geoid2))]
# int_areaNY$rent <- NY_ctocc_15$rentprop[match(int_areaNY$ct,as.character(NY_ctocc_15$geoid2))]

# For each proportion of areas, proportion of the cell or proportion of the census tract polygons, we calculate the product of the proportion with the value of the socio-economic variables for the relevant census tract and grid cell intersection.

int_areaNY$propcell.pop <- int_areaNY$propcell_area*int_areaNY$pop
int_areaNY$proppoly.pop <- int_areaNY$proppoly_area*int_areaNY$pop

int_areaNY$propcell.inc <- int_areaNY$propcell_area*int_areaNY$inc
int_areaNY$proppoly.inc <- int_areaNY$proppoly_area*int_areaNY$inc

# int_areaNY$propcell.msex <- int_areaNY$propcell_area*int_areaNY$msex
# int_areaNY$proppoly.msex <- int_areaNY$proppoly_area*int_areaNY$msex
# 
# int_areaNY$propcell.fsex <- int_areaNY$propcell_area*int_areaNY$fsex
# int_areaNY$proppoly.fsex <- int_areaNY$proppoly_area*int_areaNY$fsex
# 
# int_areaNY$propcell.food <- int_areaNY$propcell_area*int_areaNY$food
# int_areaNY$proppoly.food <- int_areaNY$proppoly_area*int_areaNY$food
# 
# int_areaNY$propcell.own <- int_areaNY$propcell_area*int_areaNY$own
# int_areaNY$proppoly.own <- int_areaNY$proppoly_area*int_areaNY$own
# 
# int_areaNY$propcell.rent <- int_areaNY$propcell_area*int_areaNY$rent
# int_areaNY$proppoly.rent <- int_areaNY$proppoly_area*int_areaNY$rent


int_areaNY$cellx <-  nyc_cells_centre[int_areaNY$cell,1]
int_areaNY$celly <-  nyc_cells_centre[int_areaNY$cell,2]

# Save
saveRDS(int_areaNY,file=paste0("NY_Cov_",grid_cellsx,grid_cellsy,"_projFinal.rds"))

# Combine counts with socio-economic variables
# First, calculate the areal weighted sums for the variables
# a. population and average income
# For the average income we use column 17 here which uses the proportion of the area of the grid cell that is intersected with each census tract. However, we do not alter the weights we've used for the population which remain as the proportion of the area of the census tract that is intersected with each grid.
sumNY2525.a <- aggregate(int_areaNY[, c(16,17)], list(cell=int_areaNY$cell), sum, na.rm = TRUE)

# This new line of code is to create sumNY2525 where, previously, it was created by combining sumNY2525.a and sumNY2525.b, however as the only socio-economic variables considered are total population and average income we have commented out the code to include the other variables, although they remain in this R script as it was the original code to create the data sets. If further variables are used, the code for sumNY2525.b and sumNY2525 where combines sumNY2525.a nd sumNY2525.b can be uncommented, and the following line of code must be commented.
sumNY2525 <- sumNY2525.a

# b. all remaining socio-economic variables
# these are all weighted sums using the proportion of the area of the grid cell that is intersected with each census tract
# sumNY2525.b <- aggregate(int_areaNY[, c(19,21,23,25,27)],list(cell=int_areaNY$cell), sum, na.rm = TRUE)
# sumNY2525 <- merge(sumNY2525.a,sumNY2525.b,by="cell",all=TRUE)


# New York: Generate Final Count Data Sets --------------------------------
# Merge the counts for each crime with the interpolated variables

hom_countdf <- data.frame("cell"=nyc_hom15.proj.count$cell,"hom"=nyc_hom15.proj.count$count,"area"=nyc_hom15.proj.count$cellarea,"x"=nyc_hom15.proj.count$x,"y"=nyc_hom15.proj.count$y)
hom_countdf$pop <- sumNY2525$proppoly.pop
hom_countdf$inc <- sumNY2525$propcell.inc
# hom_countdf$msex <- sumNY2525$propcell.msex
# hom_countdf$fsex <- sumNY2525$propcell.fsex
# hom_countdf$food <- sumNY2525$propcell.food
# hom_countdf$own <- sumNY2525$propcell.own
# hom_countdf$rent <- sumNY2525$propcell.rent

# Turn homicide data frame into spatial objects, in particular sf 
spdf_homcount <- SpatialPointsDataFrame(coords=hom_countdf[,c("x","y")],data=hom_countdf,proj4string = p4s)
sf_homcount <- st_as_sf(spdf_homcount)


gta_countdf <- data.frame("cell"=nyc_gta15.proj.count$cell,"gta"=nyc_gta15.proj.count$count,"area"=nyc_gta15.proj.count$cellarea,"x"=nyc_gta15.proj.count$x,"y"=nyc_gta15.proj.count$y)
gta_countdf$pop <- sumNY2525$proppoly.pop
gta_countdf$inc <- sumNY2525$propcell.inc
# gta_countdf$msex <- sumNY2525$propcell.msex
# gta_countdf$fsex <- sumNY2525$propcell.fsex
# gta_countdf$food <- sumNY2525$propcell.food
# gta_countdf$own <- sumNY2525$propcell.own
# gta_countdf$rent <- sumNY2525$propcell.rent

# Turn motor vehicle theft data frame into spatial objects, in particular sf 
spdf_gtacount <- SpatialPointsDataFrame(coords=gta_countdf[,c("x","y")],data=gta_countdf,proj4string = p4s)
sf_gtacount <- st_as_sf(spdf_gtacount)

# Save
save(hom_countdf,gta_countdf,file=paste0("NY2015CT",grid_cellsx,grid_cellsy,"CountData_projFinal.rda"))
save(sf_homcount,sf_gtacount,file=paste0("NY2015CT",grid_cellsx,grid_cellsy,"SFCountData_projFinal.rda"))


# New York: Re-set WD -----------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))



# Portland ----------------------------------------------------------------
# We want to generate gridded count data for the homicides and motor vehicle thefts in Portland with the purpose of modelling the gridded data with INLA algorithm as well as the INLA within MCMC algorithm. However, the Portland data is never really considered using these algorithms within my thesis, but it is used to briefly test the speed of the Multivariate INLA within MCMC algorithm.

# Portland: Set-Up --------------------------------------------------------
# Change the working directory as we want to save the data in separate folders for each city. We then load the census tract data produced in DATA/RAW_DATA/SHAPEFILES/CENSUS_TRACTS and project these to UTM coordinates. We also load the spatial point patterns for the two crimes and the socio-economic variables where we project the points of homicide and motor vehicle thefts.

# Portland Data
setwd("./Portland")

# Portland Census Tracts
load("../../../SHAPEFILES/CENSUS_TRACTS/PCityCT.rda")

# Portland Point Patterns
# Load and project ot UTM
p_hom <- readRDS("../../POINT_PATTERN/p_hom.rds")
p_gta <- readRDS("../../POINT_PATTERN/p_gta.rds")
p_hom_sf <- readRDS("../../POINT_PATTERN/p_hom_sf.rds")
p_gta_sf <- readRDS("../../POINT_PATTERN/p_gta_sf.rds")
p_hom_sf.proj <- lwgeom::st_transform_proj(p_hom_sf,"epsg:32610")
p_gta_sf.proj <- lwgeom::st_transform_proj(p_gta_sf,"epsg:32610")

# Portland Socio-economic Variables
P_ctpop_15 <- readRDS("../../../COVARIATES/P_CTPop_15_proj.rds")
P_ctinc_15 <- readRDS("../../../COVARIATES/P_CTInc_15_0imp_proj.rds") # use the imputed average income where missing data where the estimated total households is equal  to 0 are set to 0 and then the remaining missing data is imputed.
# P_ctsex_15 <- readRDS("../../../COVARIATES/P_CTSex_15_proj.rds")
# P_ctage_15 <- readRDS("../../../COVARIATES/P_CTAge_15_proj.rds")
# P_ctocc_15 <- readRDS("../../../COVARIATES/P_CTOcc_15_proj.rds")
# P_ctfood_15 <- readRDS("../../../COVARIATES/P_CTFood_15_proj.rds")

# Project to UTM: both the census tracts and their union - which will form a boundary for the city for creating the point patterns.
ct_P.proj <- lwgeom::st_transform_proj(ct_P,"epsg:32610")
pcity_boundary <- st_union(ct_P)
pcity_boundary.proj <- lwgeom::st_transform_proj(pcity_boundary,"epsg:32610")
pcity_geom <- as(pcity_boundary,"Spatial")
pcity_geom.proj <- as(pcity_boundary.proj,"Spatial")
pcity_win <- as.owin.SpatialPolygons((pcity_geom))
pcity_win.proj <- as.owin.SpatialPolygons((pcity_geom.proj))


# Portland: Set-Up Grids --------------------------------------------------
# We now want to set-up the grids for different cell-widths, namely: 5km, 2km, 1km, 500m and 200m. 

diff(pcity_win.proj$yrange)
# [1] 25639.97
diff(pcity_win.proj$xrange)
# [1] 37954.78

# So for approximately grids need following dims
x.range <- diff(pcity_win.proj$xrange)
y.range <- diff(pcity_win.proj$yrange)

grid_cellsxvec <- ceiling(x.range/(1e3*c(0.2,0.5,1,2,5)))
grid_cellsyvec <- ceiling(y.range/(1e3*c(0.2,0.5,1,2,5)))

grid_cellsx <- grid_cellsxvec[dim.ind]
grid_cellsy <- grid_cellsyvec[dim.ind]


# Portland: Create Crime Point Pattern Data -------------------------------
# Here we turn the location data for homicides and motor vehicle thefts into point patterns, object of type ppp, in order to use quadratcount() to later generate the count data over the grids.

# Extract 2015 data
p_hom15_sf.proj <- p_hom_sf.proj[which(p_hom_sf.proj$Y==2015),]
p_gta15_sf.proj <- p_gta_sf.proj[which(p_gta_sf.proj$Y==2015),]

p_hom15_sp.proj <- as(p_hom15_sf.proj,"Spatial")
p_gta15_sp.proj <- as(p_gta15_sf.proj,"Spatial")

p_hom15_loc.proj <- p_hom15_sp.proj@coords
colnames(p_hom15_loc.proj) <- c("x","y")
p_gta15_loc.proj <- p_gta15_sp.proj@coords
colnames(p_gta15_loc.proj) <- c("x","y")

# Turn in to ppp object
p_hom15.proj.ppp <- as.ppp(p_hom15_loc.proj,pcity_win.proj)
p_gta15.proj.ppp <- as.ppp(p_gta15_loc.proj,pcity_win.proj) # 1 point lies outside window


# Portland: Generate Grids ------------------------------------------------
# We now use the grid dimensions calculated above to create the grid cells using quadrats() and store these as spatial objects.

p_cells <- quadrats(pcity_win.proj,grid_cellsx,grid_cellsy)
quad.p <- p_cells
p_cells <- as(p_cells,"SpatialPolygons")
p_cells_centre <- t(sapply(p_cells@polygons, function(x){x@Polygons[[1]]@labpt}))
p_cells <- st_as_sf(p_cells)
p_cells$ID <- 1:length(p_cells$geometry)
p4s <- pcity_geom.proj@proj4string
p_cells <- st_set_crs(p_cells,p4s@projargs)

# Save
# Want to save the grid cells, la_cells, which are objects of class sf, while also saving the quadrats themselves, quad.la.
save(p_cells,file=paste0("PGridCells",grid_cellsx,grid_cellsy,"_projFinal.rda"))
save(quad.p,file=paste0("PQuad",grid_cellsx,grid_cellsy,"_projFinal.rda"))

# Calculate the cell areas for data frame and interpolation.
cell_areas <- st_area(p_cells)


# Portland: Generate Gridded Count Data -----------------------------------
# We now use the grid dimensions with quadratcount() to generate the count data sets with centroids provided from the above calculations. We include these locations as well as the counts and cell areas and an identifier for the cell for the interpolation of the socio-economic variables.

p_hom15.proj.qcount <- quadratcount(p_hom15.proj.ppp,nx=grid_cellsx,ny=grid_cellsy)
p_gta15.proj.qcount <- quadratcount(p_gta15.proj.ppp,nx=grid_cellsx,ny=grid_cellsy)


p_hom15.proj.count <- data.frame(x=p_cells_centre[,1],y=p_cells_centre[,2],count=unname(p_hom15.proj.qcount))
p_hom15.proj.count$count.Var1 <- as.numeric(p_hom15.proj.count$count.Var1)
colnames(p_hom15.proj.count) <- c("x","y","cell","count")
p_hom15.proj.count$cellarea <- cell_areas

p_gta15.proj.count <- data.frame(x=p_cells_centre[,1],y=p_cells_centre[,2],count=unname(p_gta15.proj.qcount))
p_gta15.proj.count$count.Var1 <- as.numeric(p_gta15.proj.count$count.Var1)
colnames(p_gta15.proj.count) <- c("x","y","cell","count")
p_gta15.proj.count$cellarea <- cell_areas


# Portland: Socio-economic Variable Generation ----------------------------
# We will now use the intersections between the grid cells and census tracts to calculate weighted sums for the interpolation of the socio-economic variables.


# Calculate the intersection of the grid cells and census tracts and calculate the areas of these intersections
s_intP <- st_intersection(ct_P.proj,p_cells)
s_areaP <- as.numeric(st_area(s_intP))

# Create a data frame for these intersections, with identifiers for the census tracts and cells
int_areaP <- data.frame(ct=s_intP$TRACTCE,cell=s_intP$ID,area=as.numeric(s_areaP))
int_areaP <- int_areaP[order(int_areaP$ct),]

# Create a data frame for the census tracts only and calculate the areas of the census tracts
ct_areaP <- data.frame(ct=ct_P$TRACTCE,totpoly_area=as.numeric(st_area(ct_P.proj)))
ct_areaP <- ct_areaP[order(ct_areaP$ct),]

# Add the census tract areas to the intersection data frame and calculate the proportion of the intersected areas with respect to the census tract total areas.
int_areaP$totpoly_area <- ct_areaP$totpoly_area[match(int_areaP$ct,ct_areaP$ct)]
int_areaP$proppoly_area <- int_areaP$area/int_areaP$totpoly_area

# Create a data frame for the grid cells only and calculate the areas of the grid cells
cell_areaP <- data.frame(cell=p_cells$ID,totcell_area=as.numeric(cell_areas))
cell_areaP <- cell_areaP[order(cell_areaP$cell),]

# Add the grid cell areas to the intersection data frame and calculate the proportion of the intersected areas with respect to the grid cell total areas.
int_areaP$totcell_area <- cell_areaP$totcell_area[match(int_areaP$cell,cell_areaP$cell)]
int_areaP$propcell_area <- int_areaP$area/int_areaP$totcell_area


geo2ctP <- function(x){ct_P$TRACTCE[which(ct_P$GEOID==x)]}

# Assign the census tract level socio-economic variable values for each census tract in the intersection data frame
int_areaP$pop <- P_ctpop_15$pop[match(int_areaP$ct,as.character(sapply(1:length(P_ctpop_15$geoid2),function(i){geo2ctP(P_ctpop_15$geoid2[i])})))]
int_areaP$inc <- P_ctinc_15$inc[match(int_areaP$ct,as.character(sapply(1:length(P_ctinc_15$geoid2),function(i){geo2ctP(P_ctinc_15$geoid2[i])})))]
# int_areaP$msex <- P_ctsex_15$mprop[match(int_areaP$ct,as.character(sapply(1:length(P_ctsex_15$geoid2),function(i){geo2ctP(P_ctsex_15$geoid2[i])})))]
# int_areaP$fsex <- P_ctsex_15$fprop[match(int_areaP$ct,as.character(sapply(1:length(P_ctsex_15$geoid2),function(i){geo2ctP(P_ctsex_15$geoid2[i])})))]
# int_areaP$food <- P_ctfood_15$proprec[match(int_areaP$ct,as.character(sapply(1:length(P_ctfood_15$geoid2),function(i){geo2ctP(P_ctfood_15$geoid2[i])})))]
# int_areaP$own <- P_ctocc_15$ownprop[match(int_areaP$ct,as.character(sapply(1:length(P_ctocc_15$geoid2),function(i){geo2ctP(P_ctocc_15$geoid2[i])})))]
# int_areaP$rent <- P_ctocc_15$rentprop[match(int_areaP$ct,as.character(sapply(1:length(P_ctocc_15$geoid2),function(i){geo2ctP(P_ctocc_15$geoid2[i])})))]

# For each proportion of areas, proportion of the cell or proportion of the census tract polygons, we calculate the product of the proportion with the value of the socio-economic variables for the relevant census tract and grid cell intersection.

int_areaP$propcell.pop <- int_areaP$propcell_area*int_areaP$pop
int_areaP$proppoly.pop <- int_areaP$proppoly_area*int_areaP$pop

int_areaP$propcell.inc <- int_areaP$propcell_area*int_areaP$inc
int_areaP$proppoly.inc <- int_areaP$proppoly_area*int_areaP$inc

# int_areaP$propcell.msex <- int_areaP$propcell_area*int_areaP$msex
# int_areaP$proppoly.msex <- int_areaP$proppoly_area*int_areaP$msex
# 
# int_areaP$propcell.fsex <- int_areaP$propcell_area*int_areaP$fsex
# int_areaP$proppoly.fsex <- int_areaP$proppoly_area*int_areaP$fsex
# 
# int_areaP$propcell.food <- int_areaP$propcell_area*int_areaP$food
# int_areaP$proppoly.food <- int_areaP$proppoly_area*int_areaP$food
# 
# int_areaP$propcell.own <- int_areaP$propcell_area*int_areaP$own
# int_areaP$proppoly.own <- int_areaP$proppoly_area*int_areaP$own
# 
# int_areaP$propcell.rent <- int_areaP$propcell_area*int_areaP$rent
# int_areaP$proppoly.rent <- int_areaP$proppoly_area*int_areaP$rent


int_areaP$cellx <-  p_cells_centre[int_areaP$cell,1]
int_areaP$celly <-  p_cells_centre[int_areaP$cell,2]

# Save
saveRDS(int_areaP,file=paste0("P_Cov_",grid_cellsx,grid_cellsy,"_projFinal.rds"))


# Combine counts with socio-economic variables
# First, calculate the areal weighted sums for the variables
# a. population and average income
# For the average income we use column 17 here which uses the proportion of the area of the grid cell that is intersected with each census tract. However, we do not alter the weights we've used for the population which remain as the proportion of the area of the census tract that is intersected with each grid.
sumP2525.a <- aggregate(int_areaP[, c(16,17)], list(cell=int_areaP$cell), sum, na.rm = TRUE)

# This new line of code is to create sumP2525 where, previously, it was created by combining sumP2525.a and sumLA2525.b, however as the only socio-economic variables considered are total population and average income we have commented out the code to include the other variables, although they remain in this R script as it was the original code to create the data sets. If further variables are used, the code for sumP2525.b and sumLA2525 where combines sumP2525.a nd sumP2525.b can be uncommented, and the following line of code must be commented.
sumP2525 <- sumP2525.a

# b. all remaining socio-economic variables
# these are all weighted sums using the proportion of the area of the grid cell that is intersected with each census tract
# sumP2525.b <- aggregate(int_areaP[, c(19,21,23,25,27)],list(cell=int_areaP$cell), sum, na.rm = TRUE)
# sumP2525 <- merge(sumP2525.a,sumP2525.b,by="cell",all=TRUE)


# Portland: Generate Final Count Data Sets --------------------------------
# Merge the counts for each crime with the interpolated variables

hom_countdf <- data.frame("cell"=p_hom15.proj.count$cell,"hom"=p_hom15.proj.count$count,"area"=p_hom15.proj.count$cellarea,"x"=p_hom15.proj.count$x,"y"=p_hom15.proj.count$y)
hom_countdf$pop <- sumP2525$proppoly.pop
hom_countdf$inc <- sumP2525$propcell.inc
# hom_countdf$msex <- sumP2525$propcell.msex
# hom_countdf$fsex <- sumP2525$propcell.fsex
# hom_countdf$food <- sumP2525$propcell.food
# hom_countdf$own <- sumP2525$propcell.own
# hom_countdf$rent <- sumP2525$propcell.rent

# Turn homicide data frame into spatial objects, in particular sf
spdf_homcount <- SpatialPointsDataFrame(coords=hom_countdf[,c("x","y")],data=hom_countdf,proj4string = p4s)
sf_homcount <- st_as_sf(spdf_homcount)


gta_countdf <- data.frame("cell"=p_gta15.proj.count$cell,"gta"=p_gta15.proj.count$count,"area"=p_gta15.proj.count$cellarea,"x"=p_gta15.proj.count$x,"y"=p_gta15.proj.count$y)
gta_countdf$pop <- sumP2525$proppoly.pop
gta_countdf$inc <- sumP2525$propcell.inc
# gta_countdf$msex <- sumP2525$propcell.msex
# gta_countdf$fsex <- sumP2525$propcell.fsex
# gta_countdf$food <- sumP2525$propcell.food
# gta_countdf$own <- sumP2525$propcell.own
# gta_countdf$rent <- sumP2525$propcell.rent

# Turn motor vehicle theft data frame into spatial objects, in particular sf 
spdf_gtacount <- SpatialPointsDataFrame(coords=gta_countdf[,c("x","y")],data=gta_countdf,proj4string = p4s)
sf_gtacount <- st_as_sf(spdf_gtacount)

# Save
save(hom_countdf,gta_countdf,file=paste0("P2015CT",grid_cellsx,grid_cellsy,"CountData_projFinal.rda"))
save(sf_homcount,sf_gtacount,file=paste0("P2015CT",grid_cellsx,grid_cellsy,"SFCountData_projFinal.rda"))


# Portland: Re-set WD -----------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))



# sessionInfo() -----------------------------------------------------------

sessionInfo()

