
# US City Census Tracts and Neighbourhood Matrices ------------------------

# This R script takes the state level census tract shapefiles and uses these in combination with the US City boundaries shapefiles in ../BOUNDARIES/ to extract the relevant census tract shapefiles within the city of interest. We also produce neighbourhood matrices, both for the city itself as well as for the full county(/ies) that the cities lie within, to aid in the creation of our covariates.

# Author: Nadeen Khaleel

# Setwd and Load Libraries ------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))

library(dplyr)
library(readr)
library(revgeo)
library(sf)
library(lwgeom)
library(maptools)
library(sp)
library(stringr)
library(rgeos)

# Los Angeles Census Tracts -----------------------------------------------
# In this section we will use the California census tract shapefile in combination with the Los Angeles City boundary to extract the Los Angeles City census tracts from the those identified to be for Los Angeles County, where there is a particualr overlap.

# Los Angeles: Load California Census Tract Shapefile ---------------------
# NOTE: Originally accesssed: 24/09/2019 from ftp://ftp2.census.gov/geo/tiger/TIGER2015/TRACT/, from https://www.census.gov/cgi-bin/geo/shapefiles/index.php select "Access our FTP site for additional downloading options" with FIPS codes for choosing the correct state found in https://www.census.gov/geographies/reference-files/2015/demo/popest/2015-fips.html, where you can download a .xls file foro the 2015 geocodes for the states.

ct.shape <- st_read("./tl_2015_06_tract/tl_2015_06_tract.shp") 

# Coordinate datum
# ct.shape <- lwgeom::st_transform_proj(ct.shape,"+init=epsg:4326") # originally run code, however with newer package, need to use the line below, without '+init='
ct.shape <- lwgeom::st_transform_proj(ct.shape,"epsg:4326")

# Need to project the shapefile to UTM in order to use st_intersection().
# ct.shape.proj <- lwgeom::st_transform_proj(ct.shape,"+init=epsg:32611") # newer package version doesn't require '+init='
ct.shape.proj <- lwgeom::st_transform_proj(ct.shape,"epsg:32611")


# Los Angeles: Extract LA County Census Tracts ----------------------------
# Extract LA County, FIPS "037"

ct.shape.projSubCT <- ct.shape.proj[ct.shape.proj$COUNTYFP=="037",]
# ct_LA_County <- lwgeom::st_transform_proj(ct.shape.projSubCT,"+init=epsg:4326") # Newer package version requires the below code
ct_LA_County <- lwgeom::st_transform_proj(ct.shape.projSubCT,"epsg:4326")

# Save the county, unprojected coordinate system (back from the UTM)
save(ct_LA_County,file="LACountyCT.rda") 


# Los Angeles: Extract City Census Tracts ---------------------------------
# Now we want to extract the census tracts IN LA City not just LA County, so we use the LA City boundary shapefile to try to extract the necessary census tracts.

la_boundary <- st_read("../BOUNDARIES/City Boundaries for Los Angeles County/geo_export_ded155ce-fcf8-43a1-b781-9267f1e0fcb6.shp")
lacity_boundary <- la_boundary[la_boundary$abbr=="LAX",]

# Issue with the LA City Boundary and the Census Tract Data: some census tracts are partially intersected so need to decide at what point does a small intersection no longer classify as the census tract belonging in the city.
ct_LA <- ct_LA_County
# ct_LA.proj <- lwgeom::st_transform_proj(ct_LA,"+init=epsg:32611") # older packager version
# lacity_boundary.proj <- lwgeom::st_transform_proj(lacity_boundary,"+init=epsg:32611") # older packager version
ct_LA.proj <- lwgeom::st_transform_proj(ct_LA,"epsg:32611")
lacity_boundary.proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611")
a.proj <- st_intersection(ct_LA.proj,lacity_boundary.proj)
b.proj <- ct_LA.proj[ct_LA.proj$GEOID%in%unique(a.proj$GEOID),]

# Find census tracts that have at least a certain percentage overlap
a2.proj <- st_collection_extract(a.proj, type = c("POLYGON"),warn = FALSE)
# So here, due to the intersection I believe some of the areas were split and considered their own polygon, so try to find them and merge them.
sum(duplicated(a2.proj$GEOID))
# [1] 38
length(unique(a2.proj$GEOID))
# [1] 1169
a2.proj %<>% group_by(GEOID,TRACTCE,NAME) %>% summarise()

# There's a lot of intersection slightly even though I don't believe some of these census tracts lie within LA, so instead consider the areas produced by the intersection, if the area is very small (relative to the census tract) then do not include it as it may be a result of the way the intersection was calculated.
a.censustract.proj <- a2.proj$TRACTCE
a.areas.proj <- st_area(a2.proj)

# Consider all of the census tracts highlighted by the intersection
full.ct.la <- ct_LA.proj[ct_LA.proj$TRACTCE%in%a.censustract.proj,]

# Calculate the areas for the census tracts, in order to compare the area of the intersection with the LA City boundary with that census tract.
la.areas.proj <- st_area(full.ct.la)
ordering <- match(a.censustract.proj,full.ct.la$TRACTCE) # order the output from the full census tracts using this
area.comp.proj <- data.frame(ct=a.censustract.proj,int=a.areas.proj,full=la.areas.proj[ordering])
area.comp.proj$prop <- as.numeric(area.comp.proj$int)/as.numeric(area.comp.proj$full)

# Extract the census tracts with certain percentage intersections
fiveperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.05)]
tenperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.1)]
quarter.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.25)]
half.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.5)]
threequarter.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.75)]
eightyperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.8)]
ninetyperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.9)]
ninetyfiveperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.95)]

# Plot these areas, filled in according to the percentage of census tract area intersected with city.
pdf("LAIntersectionAreaProp2ProjNew.pdf",h=15,w=15,pointsize = 14)
par(mfrow=c(2,2))
plot(ct_LA.proj$geometry,border="blue",main="less than 0.9")
plot(ct_LA.proj[ct_LA$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_LA.proj$geometry,border="blue",main="less than 0.9 or 0.5")
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_LA.proj$geometry,border="blue",main="less than 0.9 or 0.5 or 0.25")
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%quarter.proj,]$geometry,col="orange",add=TRUE)
plot(ct_LA.proj$geometry,border="blue",main="less than 0.9 or 0.5 or 0.25 or 0.1")
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%quarter.proj,]$geometry,col="orange",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%tenperc.proj,]$geometry,col="red",add=TRUE)
plot(ct_LA.proj$geometry,border="blue",main="less than 0.9 or 0.5 or 0.25 or 0.1 or 0.05")
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%quarter.proj,]$geometry,col="orange",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%tenperc.proj,]$geometry,col="red",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%fiveperc.proj,]$geometry,col="black",add=TRUE)
dev.off()


pdf("LAIntersectionAreaPropHigher2ProjNew.pdf",h=15,w=15,pointsize = 14)
par(mfrow=c(2,2))
plot(ct_LA.proj$geometry,border="blue",main="less than 0.95")
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%ninetyfiveperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_LA.proj$geometry,border="blue",main="less than 0.95 or 0.9")
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%ninetyfiveperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_LA.proj$geometry,border="blue",main="less than 0.95 or 0.9 or 0.8")
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%ninetyfiveperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%eightyperc.proj,]$geometry,col="orange",add=TRUE)
plot(ct_LA.proj$geometry,border="blue",main="less than 0.95 or 0.9 or 0.8 or 0.75")
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%ninetyfiveperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%eightyperc.proj,]$geometry,col="orange",add=TRUE)
plot(ct_LA.proj[ct_LA.proj$TRACTCE%in%threequarter.proj,]$geometry,col="red",add=TRUE)
dev.off()

# So, I believe that I should only keep the census tracts where the proportion of overlap between the census tracts and the LA City Boundary from the shapefile is at least 25% of that census tract, and the above saved plots indicate this is a reasonable cut-off value.
sel.census.proj <- area.comp.proj[which(area.comp.proj$prop>=0.25),]$ct
lacity_CT.proj <- ct_LA.proj[ct_LA.proj$TRACTCE%in%sel.census.proj,]
# Remove any which are just water, with no land.
which(lacity_CT.proj$ALAND==0)
# [1] 748
lacity_CT.proj <- lacity_CT.proj[-which(lacity_CT.proj$ALAND==0),]

# Now compare the extracted census tracts with the originally extract census tracts.
pdf("LACitywithCensusTracts2ProjNew.pdf",h=15,w=15,pointsize = 14)
par(mfrow=c(1,3))
plot(ct_LA.proj$geometry,border="blue",main="Intersection between CT and Boundary")
plot(a2.proj$geometry,col="darkgreen",add=TRUE)
plot(ct_LA.proj$geometry,border="blue",main="Extracting Census Tracts with > 25% Coverage from Intersection")
plot(lacity_CT.proj$geometry,col="red",add=TRUE)
plot(ct_LA.proj$geometry,border="blue",main="Overlap to Compare Differences")
plot(lacity_CT.proj$geometry,col="red",add=TRUE)
plot(a2.proj$geometry,col="darkgreen",add=TRUE)
dev.off()

# Return back from UTM projection
ct_LA.proj <- lacity_CT.proj
# ct_LA <- lwgeom::st_transform_proj(ct_LA.proj,"+init=epsg:4326") # older package version from original code
ct_LA <- lwgeom::st_transform_proj(ct_LA.proj,"epsg:4326")

# Save LA City census tracts
save(ct_LA,file="LACityCT.rda")


# Los Angeles: Neighbourhood Matrix ---------------------------------------
# For some of the socio-economic variables we may want to perform some simple interpolation if there is missing data present, therefore I need to build a neighbourhood matrix to be able to identify neighbours of the census tracts with missing data. The neighbours will include those in LA County and not just LA City: so produce two output files, one for just LA City and one for all of LA County.

# City neighbourhood matrix
ct_LAsp <- as(ct_LA,"Spatial")
nb <- gTouches(ct_LAsp,byid = TRUE)
lab.geo <- ct_LAsp$GEOID
lab.ct <- ct_LAsp$TRACTCE
save(nb,lab.ct,lab.ct,ct_LAsp,file="LACityNB.rda")

# County neighbourhood matrix
ct_LA_Countysp <- as(ct_LA_County,"Spatial")
nb <- gTouches(ct_LA_Countysp,byid = TRUE)
lab.geo <- ct_LA_Countysp$GEOID
lab.ct <- ct_LA_Countysp$TRACTCE
save(nb,lab.geo,lab.ct,ct_LA_Countysp,file="LACountyNB.rda")


# New York ----------------------------------------------------------------
# In this section we will use the New York State census tract shapefile in combination with the New York City boundary to extract the New York City census tracts from those identified for the counties (boroughs) that form the city.

# New York: Load New York State Census Tract Shapefile --------------------
# NOTE: Originally accessed: 24/09/2019 from ftp://ftp2.census.gov/geo/tiger/TIGER2015/TRACT/, from https://www.census.gov/cgi-bin/geo/shapefiles/index.php select "Access our FTP site for additional downloading options" with FIPS codes for choosing the correct state found in https://www.census.gov/geographies/reference-files/2015/demo/popest/2015-fips.html, where you can download a .xls file foro the 2015 geocodes for the states.

ct.shape <- st_read("./tl_2015_36_tract/tl_2015_36_tract.shp")

# Coordinate datum
# ct.shape <- lwgeom::st_transform_proj(ct.shape,"+init=epsg:4326") # originally run code, however with newer package, need to use the line below, without '+init='
ct.shape <- lwgeom::st_transform_proj(ct.shape,"epsg:4326")

# Need to project the shapefile to UTM in order to use st_intersection().
# ct.shape.proj <- lwgeom::st_transform_proj(ct.shape,"+init=epsg:32618")
ct.shape.proj <- lwgeom::st_transform_proj(ct.shape,"epsg:32618") # newer package version doesn't require '+init='


# New York: Extract County Census Tracts ----------------------------------
# Select the required counties from the ct.shape file which are:
# Bronx County - 005; Kings County - 047; NY County - 061; Queens County - 081 and Richmond County - 085
# All found from https://www.census.gov/quickfacts/fact/table/richmondcountystatenislandboroughnewyork,queenscountyqueensboroughnewyork,newyorkcountymanhattanboroughnewyork,kingscountybrooklynboroughnewyork,bronxcountybronxboroughnewyork/PST045219
# These codes are the last 3 digits of the FIPS code, while the first two refer to the state, NY-36
# Then consider the intersection with these.

ct.shape.projSubCT <- ct.shape.proj[ct.shape.proj$COUNTYFP%in%c("005","047","061","081","085"),]
# ct_NY_County <- lwgeom::st_transform_proj(ct.shape.projSubCT,"+init=epsg:4326")
ct_NY_County <- lwgeom::st_transform_proj(ct.shape.projSubCT,"epsg:4326") # newer package version doesn't require '+init='

# Save the county, unprojected coordinate system (back from the UTM)
save(ct_NY_County,file="NYCCountyCT.rda")


# New York: Extract City Census Tracts ------------------------------------

nyc_boundary <- st_read("../BOUNDARIES/Borough Boundaries NYC/geo_export_3dcef28d-8b06-44cb-b8a3-5254b5580b2c.shp")

# Issue with the NY City Boundary and the Census Tract Data: some census tracts are partially intersected so need to decide at what point does a small intersection no longer classify as the census tract belonging in the city.
ct_NY <- ct_NY_County
# ct_NY.proj <- lwgeom::st_transform_proj(ct_NY,"+init=epsg:32618") # older package version
# nyc_boundary.proj <- lwgeom::st_transform_proj(nyc_boundary,"+init=epsg:32618") # older package version
ct_NY.proj <- lwgeom::st_transform_proj(ct_NY,"epsg:32618")
nyc_boundary.proj <- lwgeom::st_transform_proj(nyc_boundary,"epsg:32618")
a.proj <- st_intersection(ct_NY.proj,nyc_boundary.proj)
b.proj <- ct_NY.proj[ct_NY.proj$GEOID%in%unique(a.proj$GEOID),]

# Find census tracts that have at least a certain percentage overlap
a2.proj <- st_collection_extract(a.proj, type = c("POLYGON"),warn = FALSE)
# So here, due to the intersection I believe some of the areas were split and considered their own polygon, so try to find them and merge them.
sum(duplicated(a2.proj$GEOID))
# [1] 90
length(unique(a2.proj$GEOID))
# [1] 2165
a2.proj %<>% group_by(GEOID,TRACTCE,NAME) %>% summarise()

# There's a lot of intersection slightly even though I don't believe some of these census tracts lie within New York City, so instead consider the areas produced by the intersection, if the area is very small (relative to the census tract) then do not include it as it may be a result of the way the intersection was calculated.
a.censustract.proj <- a2.proj$TRACTCE
a.areas.proj <- st_area(a2.proj)

# Consider all of the census tracts highlighted by the intersection
full.ct.ny <- ct_NY.proj[ct_NY.proj$TRACTCE%in%a.censustract.proj,]

# There is are two extra elements in the full.ct.ny: further investigation uncovers that they are fully water census tracts
which(full.ct.ny$GEOID%in%setdiff(full.ct.ny$GEOID,a2.proj$GEOID))
which(full.ct.ny$ALAND==0)
# [1]  696 1620 1941
setdiff(full.ct.ny$GEOID,a2.proj$GEOID)
# [1] "36047990100" "36081990100"
which(full.ct.ny$GEOID%in%setdiff(full.ct.ny$GEOID,a2.proj$GEOID))
# [1]  696 1620

# Lets remove these two census tracts
full.ct.ny <- full.ct.ny[-c(696,1620),]

# Calculate the areas for the census tracts, in order to compare the area of the intersection with the NY City boundary with that census tract.
nyc.areas.proj <- st_area(full.ct.ny)
ordering <- match(a.censustract.proj,full.ct.ny$TRACTCE) # order the output from the full census tracts using this
area.comp.proj <- data.frame(ct=a.censustract.proj,int=a.areas.proj,full=nyc.areas.proj[ordering])
area.comp.proj$prop <- as.numeric(area.comp.proj$int)/as.numeric(area.comp.proj$full)


# Extract the census tracts with certain percentage intersections
oneperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.01)]
twoperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.02)]
threeperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.03)]
fourperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.04)]
fiveperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.05)]
tenperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.1)]
quarter.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.25)]
half.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.5)]
threequarter.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.75)]
eightyperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.8)]
ninetyperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.9)]
ninetyfiveperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.95)]


# Plot these areas, filled in according to the percentage of census tract area intersected with city.
pdf("NYCIntersectionAreaProp2ProjNew.pdf",h=15,w=15,pointsize = 14)
par(mfrow=c(2,2))
plot(ct_NY.proj$geometry,border="blue",main="less than 0.9")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.9 or 0.5")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.9 or 0.5 or 0.25")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%quarter.proj,]$geometry,col="orange",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.9 or 0.5 or 0.25 or 0.1")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%quarter.proj,]$geometry,col="orange",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%tenperc.proj,]$geometry,col="red",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.9 or 0.5 or 0.25 or 0.1 or 0.05")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%quarter.proj,]$geometry,col="orange",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%tenperc.proj,]$geometry,col="red",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%fiveperc.proj,]$geometry,col="black",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.9 or 0.5 or 0.25 or 0.1 or 0.01")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%quarter.proj,]$geometry,col="orange",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%tenperc.proj,]$geometry,col="red",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%fiveperc.proj,]$geometry,col="black",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%oneperc.proj,]$geometry,col="magenta",add=TRUE)
dev.off()

pdf("NYCIntersectionAreaPropLower2ProjNew.pdf",h=15,w=15,pointsize = 14)
par(mfrow=c(2,2))
plot(ct_NY.proj$geometry,border="blue",main="less than 0.1")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%tenperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.1 or 0.05")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%tenperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%fiveperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.1 or 0.05 or 0.04")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%tenperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%fiveperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%fourperc.proj,]$geometry,col="orange",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.1 or 0.05 or 0.04 or 0.03")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%tenperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%fiveperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%fourperc.proj,]$geometry,col="orange",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%threeperc.proj,]$geometry,col="red",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.1 or 0.05 or 0.04 or 0.03 or 0.02")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%tenperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%fiveperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%fourperc.proj,]$geometry,col="orange",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%threeperc.proj,]$geometry,col="red",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%twoperc.proj,]$geometry,col="black",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.1 or 0.05 or 0.04 or 0.03 or 0.02 or 0.01")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%tenperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%fiveperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%fourperc.proj,]$geometry,col="orange",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%threeperc.proj,]$geometry,col="red",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%twoperc.proj,]$geometry,col="black",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%oneperc.proj,]$geometry,col="magenta",add=TRUE)
dev.off()


pdf("NYCIntersectionAreaPropHigher2ProjNew.pdf",h=15,w=15,pointsize = 14)
par(mfrow=c(2,2))
plot(ct_NY.proj$geometry,border="blue",main="less than 0.95")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyfiveperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.95 or 0.9")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyfiveperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.95 or 0.9 or 0.8")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyfiveperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%eightyperc.proj,]$geometry,col="orange",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="less than 0.95 or 0.9 or 0.8 or 0.75")
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyfiveperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%eightyperc.proj,]$geometry,col="orange",add=TRUE)
plot(ct_NY.proj[ct_NY.proj$TRACTCE%in%threequarter.proj,]$geometry,col="red",add=TRUE)
dev.off()

# So, I believe that I should only keep the census tracts where the proportion of overlap between the census tracts and the NY City Boundary from the shapefile is at least 1% of that census tract, and the below plots indicate this is a reasonable cut-off value.
sel.census.proj <- area.comp.proj[which(area.comp.proj$prop>=0.01),]$ct
nycity_CT.proj <- ct_NY.proj[ct_NY.proj$TRACTCE%in%sel.census.proj,]

# Now compare the extracted census tracts with the originally extract census tracts.
pdf("NYCitywithCensusTracts2ProjNew.pdf",h=15,w=15,pointsize = 14)
par(mfrow=c(1,3))
plot(ct_NY.proj$geometry,border="blue",main="Intersection between CT and Boundary")
plot(a2.proj$geometry,col="darkgreen",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="Extracting Census Tracts with > 25% Coverage from Intersection") # NOTE: Incorrect percentage in title!!
plot(nycity_CT.proj$geometry,col="red",add=TRUE)
plot(ct_NY.proj$geometry,border="blue",main="Overlap to Compare Differences")
plot(nycity_CT.proj$geometry,col="red",add=TRUE)
plot(a2.proj$geometry,col="darkgreen",add=TRUE)
dev.off()

# Return back from UTM projection
ct_NY.proj <- nycity_CT.proj
# ct_NY <- lwgeom::st_transform_proj(ct_NY.proj,"+init=epsg:4326") # older package version from original code
ct_NY <- lwgeom::st_transform_proj(ct_NY.proj,"+init=epsg:4326")

# Save NY City census tracts
save(ct_NY,file="NYCityCT.rda")


# New York: Neighbourhood Matrix ------------------------------------------
# For some of the socio-economic variables we may want to perform some simple interpolation if there is missing data present, therefore I need to build a neighbourhood matrix to be able to identify neighbours of the census tracts with missing data. The neighbours will include those in the counties containing New York and not just NY City: so produce two output files, one for just NY City and one for all of the counties that contain NY.

# City neighbourhood matrix
ct_NYsp <- as(ct_NY,"Spatial")
nb <- gTouches(ct_NYsp,byid = TRUE)
lab.geo <- ct_NYsp$GEOID
lab.ct <- ct_NYsp$TRACTCE
save(nb,lab.ct,lab.ct,ct_NYsp,file="NYCityNB.rda")

# County neighbourhood matrix
ct_NY_Countysp <- as(ct_NY_County,"Spatial")
nb <- gTouches(ct_NY_Countysp,byid = TRUE)
lab.geo <- ct_NY_Countysp$GEOID
lab.ct <- ct_NY_Countysp$TRACTCE
save(nb,lab.ct,lab.ct,ct_NY_Countysp,file="NYCountyNB.rda")



# Portland ----------------------------------------------------------------
# In this section we will use the Oregon census tract shapefile in combination with the Portland City boundary to extract the New York City census tracts from those identified for the counties that form the city.


# Portland: Load Oregon Census Tract Shapefile ----------------------------
# NOTE: Originally accessed: 24/09/2019 from ftp://ftp2.census.gov/geo/tiger/TIGER2015/TRACT/, from https://www.census.gov/cgi-bin/geo/shapefiles/index.php select "Access our FTP site for additional downloading options" with FIPS codes for choosing the correct state found in https://www.census.gov/geographies/reference-files/2015/demo/popest/2015-fips.html, where you can download a .xls file foro the 2015 geocodes for the states.

ct.shape <- st_read("./tl_2015_41_tract/tl_2015_41_tract.shp")

# Coordinate datum
# ct.shape <- lwgeom::st_transform_proj(ct.shape,"+init=epsg:4326") # originally run code, however with newer package, need to use the line below, without '+init='
ct.shape <- lwgeom::st_transform_proj(ct.shape,"epsg:4326")

# Need to project the shapefile to UTM in order to use st_intersection().
# ct.shape.proj <- lwgeom::st_transform_proj(ct.shape,"+init=epsg:32610")
ct.shape.proj <- lwgeom::st_transform_proj(ct.shape,"epsg:32610") # newer package version doesn't require '+init='


# Portland: Extract County Census Tracts ----------------------------------
# Select the required counties from the ct.shape files which are:
# Multnomah County - 051; Clackamas County - 005 and Washington County - 067
# Then consider the intersection with these.

ct.shape.projSubCT <- ct.shape.proj[ct.shape.proj$COUNTYFP%in%c("005","051","067"),]
# ct_P_County <- lwgeom::st_transform_proj(ct.shape.projSubCT,"+init=epsg:4326")
ct_P_County <- lwgeom::st_transform_proj(ct.shape.projSubCT,"epsg:4326") # newer package version doesn't require '+init='

# Save the county, unprojected coordinate system (back from the UTM)
save(ct_P_County,file="PCountyCT.rda")


# Portland: Extract City Census Tracts ------------------------------------

or_boundary <- st_read("../BOUNDARIES/City_Boundaries_Portland/City_Boundaries.shp")
p_boundary <- or_boundary[or_boundary$CITYNAME=="Portland",]

# Issue with the Portland City Boundary and the Census Tract Data: some census tracts are partially intersected so need to decide at what point does a small intersection no longer classify as the census tract belonging in the city.
ct_P <- ct_P_County
# ct_P.proj <- lwgeom::st_transform_proj(ct_P,"+init=epsg:32610") # older package version
# p_boundary.proj <- lwgeom::st_transform_proj(p_boundary,"+init=epsg:32610") # older package version
ct_P.proj <- lwgeom::st_transform_proj(ct_P,"epsg:32610")
p_boundary.proj <- lwgeom::st_transform_proj(p_boundary,"epsg:32610")
a.proj <- st_intersection(ct_P.proj,p_boundary.proj)
b.proj <- ct_P.proj[ct_P.proj$GEOID%in%unique(a.proj$GEOID),]


# Find census tracts that have at least a certain percentage overlap
a2.proj <- st_collection_extract(a.proj, type = c("POLYGON"),warn = FALSE)
plot(a2.proj$geometry)
plot(p_boundary.proj$geometry,add=T,border="red")

# So here, due to the intersection I believe some of the areas were split and considered their own polygon, so try to find them and merge them.
sum(duplicated(a2.proj$GEOID))
# [1] 0
length(unique(a2.proj$GEOID))
# [1] 166
a2.proj %<>% group_by(GEOID,TRACTCE,NAME) %>% summarise()
# 
# There's a lot of intersection slightly even though I don't believe some of these census tracts lie within Portlamd, so instead consider the areas produced by the intersection, if the area is very small then do not include it as it may be a result of the way the intersection was calculated.
a.censustract.proj <- a2.proj$TRACTCE
a.areas.proj <- st_area(a2.proj)

# Consider all of the census tracts highlighted by the intersection
full.ct.p <- ct_P.proj[ct_P.proj$TRACTCE%in%a.censustract.proj,]

# We have a duplication:
ctprojcheck <- ct_P.proj[ct_P.proj$TRACTCE%in%a.censustract.proj,]$TRACTCE
which(duplicated(ctprojcheck)) # 132
which(ctprojcheck==ctprojcheck[132]) #90
projcheck <- ct_P.proj[ct_P.proj$TRACTCE%in%a.censustract.proj,]
projcheck[c(90,132),]
# Simple feature collection with 2 features and 12 fields
# geometry type:  MULTIPOLYGON
# dimension:      XY
# bbox:           xmin: 521029.2 ymin: 4970525 xmax: 606101.3 ymax: 5046902
# CRS:            +init=epsg:32610
# STATEFP COUNTYFP TRACTCE       GEOID NAME          NAMELSAD MTFCC FUNCSTAT      ALAND  AWATER    INTPTLAT     INTPTLON
# 313      41      051  980000 41051980000 9800 Census Tract 9800 G5020        S    4078535 1594068 +45.5616432 -122.7062017
# 461      41      005  980000 41005980000 9800 Census Tract 9800 G5020        S 1632985294 6684470 +45.0641988 -121.9884098
# geometry
# 313 MULTIPOLYGON (((521040.3 50...
#                     461 MULTIPOLYGON (((557576.7 50...
# We can see that there are the two census tract codes, one in Multnomah County (051) and the other in Clackamas County (005), plotting shows that the census tract in Clackamas is not in the city and we can also see that through
# st_intersection(projcheck[c(90,132),],p_boundary.proj)
# Simple feature collection with 1 feature and 17 fields
# geometry type:  POLYGON
# dimension:      XY
# bbox:           xmin: 521029.2 ymin: 5042758 xmax: 525151.4 ymax: 5046902
# CRS:            +init=epsg:32610
# STATEFP COUNTYFP TRACTCE       GEOID NAME          NAMELSAD MTFCC FUNCSTAT   ALAND  AWATER    INTPTLAT     INTPTLON OBJECTID CITYNAME
# 313      41      051  980000 41051980000 9800 Census Tract 9800 G5020        S 4078535 1594068 +45.5616432 -122.7062017       33 Portland
# Shape_Leng Shape_Area       AREA                       geometry
# 313   276025.7  765886653 4044453717 POLYGON ((521040.3 5045757,...
# Therefore, the 90th element and the census tract 9800 with county code 051 is the correct one.

# Remove the duplicate
full.ct.p <- full.ct.p[-132,]

# Calculate the areas for the census tracts, in order to compare the area of the intersection with the NY City boundary with that census tract.
p.areas.proj <- st_area(full.ct.p)
ordering <- match(a.censustract.proj,full.ct.p$TRACTCE) # order the output from the full census tracts using this
area.comp.proj <- data.frame(ct=a.censustract.proj,int=a.areas.proj,full=p.areas.proj[ordering])
area.comp.proj$prop <- as.numeric(area.comp.proj$int)/as.numeric(area.comp.proj$full)


# Extract the census tracts with certain percentage intersections
oneperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.01)]
fiveperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.05)]
tenperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.1)]
quarter.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.25)]
half.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.5)]
threequarter.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.75)]
eightyperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.8)]
ninetyperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.9)]
ninetyfiveperc.proj <- area.comp.proj$ct[which(area.comp.proj$prop<0.95)]

# Plot these areas, filled in according to the percentage of census tract area intersected with city.
pdf("PIntersectionAreaProp2ProjNew.pdf",h=15,w=15,pointsize = 14)
par(mfrow=c(2,2))
plot(ct_P.proj$geometry,border="blue",main="less than 0.9")
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_P.proj$geometry,border="blue",main="less than 0.9 or 0.5")
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_P.proj$geometry,border="blue",main="less than 0.9 or 0.5 or 0.25")
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%quarter.proj,]$geometry,col="orange",add=TRUE)
plot(ct_P.proj$geometry,border="blue",main="less than 0.9 or 0.5 or 0.25 or 0.1")
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%quarter.proj,]$geometry,col="orange",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%tenperc.proj,]$geometry,col="red",add=TRUE)
plot(ct_P.proj$geometry,border="blue",main="less than 0.9 or 0.5 or 0.25 or 0.1 or 0.05")
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%quarter.proj,]$geometry,col="orange",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%tenperc.proj,]$geometry,col="red",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%fiveperc.proj,]$geometry,col="black",add=TRUE)
plot(ct_P.proj$geometry,border="blue",main="less than 0.9 or 0.5 or 0.25 or 0.1 or 0.01")
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%half.proj,]$geometry,col="green",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%quarter.proj,]$geometry,col="orange",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%tenperc.proj,]$geometry,col="red",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%fiveperc.proj,]$geometry,col="black",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%oneperc.proj,]$geometry,col="magenta",add=TRUE)
dev.off()


pdf("PIntersectionAreaPropHigher2ProjNew.pdf",h=15,w=15,pointsize = 14)
par(mfrow=c(2,2))
plot(ct_P.proj$geometry,border="blue",main="less than 0.95")
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyfiveperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_P.proj$geometry,border="blue",main="less than 0.95 or 0.9")
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyfiveperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_P.proj$geometry,border="blue",main="less than 0.95 or 0.9 or 0.8")
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyfiveperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%eightyperc.proj,]$geometry,col="orange",add=TRUE)
plot(ct_P.proj$geometry,border="blue",main="less than 0.95 or 0.9 or 0.8 or 0.75")
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyfiveperc.proj,]$geometry,col="darkgreen",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%ninetyperc.proj,]$geometry,col="green",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%eightyperc.proj,]$geometry,col="orange",add=TRUE)
plot(ct_P.proj[ct_P.proj$TRACTCE%in%threequarter.proj,]$geometry,col="red",add=TRUE)
dev.off()


# So, I believe that I should only keep the census tracts where the proportion of overlap between the census tracts and the Portland City Boundary from the shapefile is at least 1% of that census tract, and the below plots indicate this is a reasonable cut-off value.
sel.census.proj <- area.comp.proj[which(area.comp.proj$prop>=0.01),]$ct
pcity_CT.proj <- ct_P.proj[ct_P.proj$TRACTCE%in%sel.census.proj,]

# Duplicate needs to be removed, this has to be done again as the selected census tract will extract both values in ct_P.proj.
which(duplicated(pcity_CT.proj$TRACTCE))
# [1] 126
pcity_CT.proj$TRACTCE[126]
# [1] 980000
# 581 Levels: 000100 000200 000201 000202 000203 ... 990101
which(pcity_CT.proj$TRACTCE=="980000")
# [1]  86 126
plot(pcity_CT.proj$geometry)
plot(pcity_CT.proj[126,]$geometry,col="red",add=T)
# Yes remove row 126 as it is definitely not in Portland
pcity_CT.proj <- pcity_CT.proj[-126,]

# Now compare the extracted census tracts with the originally extract census tracts.
pdf("PCitywithCensusTracts2ProjNew.pdf",h=15,w=15,pointsize = 14)
par(mfrow=c(1,3))
plot(ct_P.proj$geometry,border="blue",main="Intersection between CT and Boundary")
plot(a2.proj$geometry,col="darkgreen",add=TRUE)
plot(ct_P.proj$geometry,border="blue",main="Extracting Census Tracts with > 5% Coverage from Intersection")
plot(pcity_CT.proj$geometry,col="red",add=TRUE)
plot(ct_P.proj$geometry,border="blue",main="Overlap to Compare Differences")
plot(pcity_CT.proj$geometry,col="red",add=TRUE)
plot(a2.proj$geometry,col="darkgreen",add=TRUE)
dev.off()


# Return back from UTM projection
ct_P.proj <- pcity_CT.proj
# ct_P <- lwgeom::st_transform_proj(ct_P.proj,"+init=epsg:4326") # older package version
ct_P <- lwgeom::st_transform_proj(ct_P.proj,"epsg:4326")

# Save Portland City census tracts
save(ct_P,file="PCityCT.rda")


# Portland: Neighbourhood Matrix ------------------------------------------
# For some of the socio-economic variables we may want to perform some simple interpolation if there is missing data present, therefore I need to build a neighbourhood matrix to be able to identify neighbours of the census tracts with missing data. The neighbours will include those in the counties containing Portland and not just those in Portland City: so produce two output files, one for just Portland City and one for all of the counties that contain Portland.

# City neighbourhood matrix
ct_Psp <- as(ct_P,"Spatial")
nb <- gTouches(ct_Psp,byid = TRUE)
lab.geo <- ct_Psp$GEOID
lab.ct <- ct_Psp$TRACTCE
save(nb,lab.ct,lab.ct,ct_Psp,file="PCityNB.rda")

# County neighbourhood matrix
ct_P_Countysp <- as(ct_P_County,"Spatial")
nb <- gTouches(ct_P_Countysp,byid = TRUE)
lab.geo <- ct_P_Countysp$GEOID
lab.ct <- ct_P_Countysp$TRACTCE
save(nb,lab.ct,lab.ct,ct_P_Countysp,file="PCountyNB.rda")



# sessionInfo() -----------------------------------------------------------

sessionInfo()
