
# INLA within MCMC: Example 5.1 -------------------------------------------

# This code simulates the data from the paper: "Markov Chain Monte Carlo with the Integrated Nested Laplace Approximation" by Gomez-Rubio and Rue (6-04-17 - at time of originally implementing this code)

# This follows Section 5 - Simulation Study, concentrating on 5.1 - Bivariate linear regression.

# This R script only contains the code to implement the INLA within MCMC algorithm for the example, after simulate the data as specified in the paper. When implementing this, the aim of this R script is to provide the step-by-step explanation of the Bayesian Model Averaging (run through INLABMA:::fitmargBMA2), through examples for conditional posterior marginals for three of the models (conditioned on three of the MH sample values) as well plots of the splines and combination steps for the BMA. The plots for this are placed in Appendix B of my thesis.


# Please note that this was some original INLA within MCMC code, and so I was not nearly as efficient as newer IwM code where I save the conditional models as I ran the MH step, therefore we have to wait for these to re-run: an important reason to thin the above. Additionally, these models are all saved as inla objects which take up a large amount of memory, in newer code I extract the necessary marginals to store as the matrices they are output as to save space, allowing for more iterations with less space consumed by the INLA output.

# Author: Nadeen Khaleel

# Setwd and Load Libraries ------------------------------------------------

# Either setwd() to the source file location, or run the following:
library("rstudioapi")
setwd(dirname(getActiveDocumentContext()$path))

library(INLA)
library(INLABMA)
library(mvtnorm)
library(stringr)
library(dplyr)
library(gridExtra)

# Simulation --------------------------------------------------------------

# Simulating the dataset with the following specifications: 

# y_i = alpha + beta_1u_1i + beta_2u_2i + epsilon_i,  i=1,...,n 
# where y is the response variable, u_1, u_2 are the two covariates, epsilon_i
# is a Gaussian error term with zero mean and precision tau.

# As in the paper specified above, the following values are used:

set.seed(1)

alpha_0 <- 3
beta_1_0 <- 2
beta_2_0 <- -2
tau_0 <- 1

n <- 100

# The two covariates, u_1 and u_2, will be simulated from a U(0,1) distribution.

u_1 <- as.matrix(runif(n,0,1))
u_2 <- as.matrix(runif(n,0,1))

# Simulate the zero-mean Gaussian error.

epsilon <- as.matrix(rnorm(n,mean=0,sd=sqrt(1/tau_0)))

# Now to use these simulated values to calculate the response, y.

y <- alpha_0*as.matrix(rep(1,n)) + beta_1_0*u_1 + beta_2_0*u_2 + epsilon

dat <- as.data.frame(cbind(y,u_1,u_2))
colnames(dat) <- c("y","u_1","u_2")

# INLA WITHIN MCMC --------------------------------------------------------

# INLA WITHIN MCMC: MH ----------------------------------------------------
# Run a Metropolis-Hastings MCMC chain.

# MCMC-MH function, runs Metropolis Hastings chain with Gaussian proposal.
# Input:
# data      : data vector (y)
# u_1, u_2  : rest of data
# fx        : fixed hyperpriors vector (sd_beta)
# init      : initial values for the chain (beta_1,beta_2)
# its       : number of iterations of the algorithm (length of chain)
# burnin    : burn-in for the chain
# 

# Output:
# theta     : matrix with posterior samples
# lik.val   : likelihood at the samples in theta
# acc_rat   : acceptance ratio

## Set-up (vectors and acceptance counter)

init <- c(0,0)
fx <- 1e3
sd <- (0.75)
its <- 1e4
n.burnin <- 1e3

n.a <- 0
theta <- matrix(rep(0,(length(init)*(its+1))),nrow=length(init))
lik.val <- matrix(rep(0,(its+1)),nrow=1)
log.post.val <- matrix(rep(0,(its+1)),nrow=1)

## Initial calculations
theta[,1] <- init
formula <- y ~ 1 + offset(init[1]*u_1 + init[2]*u_2)
cond_i <- inla(formula,data=dat, family="gaussian",control.compute=list(mlik=TRUE),control.fixed = list(prec.intercept=0.001),control.family = list(hyper = list(prec = list(prior = "loggamma",param = c(1, 0.00005)))))
lik.val[1] <- cond_i$mlik[2]
log.post.val[1] <- lik.val[1] + dmvnorm(c(init[1],init[2]),mean=c(0,0),sigma=fx[1]*diag(2),log=TRUE)

for (i in 1:its){
  theta.p <- theta[,i] + sd*rmvnorm(1,mean=rep(0,length(theta[,1])),sigma=diag(length(theta[,1])))
  formula <- y ~ 1 + offset(theta.p[1]*u_1 + theta.p[2]*u_2)
  cond_i.p <- inla(formula,data=dat, family="gaussian",control.compute=list(mlik=TRUE),control.fixed = list(prec.intercept=0.001),control.family = list(hyper = list(prec = list(prior = "loggamma",param = c(1, 0.00005)))))
  lik.val.p <- cond_i.p$mlik[2]
  log.post.p <- lik.val.p + dmvnorm(c(theta.p[1],theta.p[2]),mean=c(0,0),sigma=fx[1]*diag(2),log=TRUE)
  alpha <- exp(log.post.p - log.post.val[i]) # Using Random Walk MH
  if (alpha > runif(1)){
    n.a <- n.a + 1
    theta[,i+1] <- theta.p
    log.post.val[i+1] <- log.post.p
    lik.val[i+1] <- lik.val.p
  } else {
    n.a <- n.a
    theta[,i+1] <- theta[,i]
    log.post.val[i+1] <- log.post.val[i]
    lik.val[i+1] <- lik.val[i]
  }
  # The following commands give percentage completion
  
  if (floor(i*100/its)!=floor((i-1)*100/its)) {
    cat(paste(".",i*100/its,"."))
  }
}

# Acceptance Rate
n.a/its


# # Plotting
# 
# plot(1:(its+1),theta[1,],type="l",xlab="iteration",ylab=expression(beta_1))
# plot(1:(its+1),theta[2,],type="l",xlab="iteration",ylab=expression(beta_2))
# 
# 
# hist(theta[1,(1e3:1e4)],main=~beta[1])
# hist(theta[2,(1e3:1e4)],main=~beta[2])
# 
# plot(density(theta[1,(1e3:1e4)],bw=0.1),main=~beta[1])
# plot(density(theta[2,(1e3:1e4)],bw=0.1),main=~beta[2])
# 
# mean(theta[1,(1e3:1e4)])
# sd(theta[1,(1e3:1e4)])
# quantile(theta[1,(1e3:1e4)],c(0.025,0.25,0.5,0.75,0.975))
# 
# mean(theta[2,(1e3:1e4)])
# sd(theta[2,(1e3:1e4)])
# quantile(theta[2,(1e3:1e4)],c(0.025,0.25,0.5,0.75,0.975))




# INLA WITHIN MCMC: BMA ---------------------------------------------------
# Using INLABMA, in particular fitmargBMA2 to calculate the marginals for alpha and tau. As mentioned above, as this was some original code for the INLA within MCMC code and so it is not as efficient, in particular, where the conditional models fit are not stored at each step and so we have to take the samples (from the full, post-burning or after thinning) and re-implement the INLA algorithm conditional on these parameter values.

# Thinning of the samples
theta_thin <- theta[,(n.burnin + 9*c(0:1000) + 1)]

# Fit INLA models conditional on the beta values sampled in the MH step
mod.names <- c()
save.names <- c()
mod.all <- list()
for (i in 1:dim(theta_thin)[2]){
beta_1 <- theta_thin[1,i]
beta_2 <- theta_thin[2,i]
temp.mod <- inla(formula = y ~ 1 + offset(beta_1*u_1 + beta_2*u_2),data=dat,family="gaussian",control.fixed = list(prec.intercept=0.001),control.family = list(hyper = list(prec = list(prior = "loggamma",param = c(1, 0.00005)))))
mod.names[i] <- paste("modalt",i,sep=".")
assign(mod.names[i],temp.mod)
mod.all[[mod.names[i]]] <- get(mod.names[i])
save.name <- paste("inlamodalt",i,"rda",sep=".")
save(list=mod.names[i],file=save.name)
}
all.objs <- ls()
mod.objs <- all.objs[all.objs%in%c(mod.names,"mod.all")]
save(list = mod.objs,file="modelsalt.rda")

# Implement the BMA using fitmargBMA2 from the INLABMA package
load("./modelsalt.rda")
listmarg<-c("marginals.fixed", "marginals.hyperpar")
ws <- rep(1/length(mod.all),length(mod.all))
margeff<-mclapply(listmarg, function(X){INLABMA:::fitmargBMA2(mod.all, ws, X)})
names(margeff)<-listmarg

plot(margeff$marginals.fixed$`(Intercept)`,type="l",main=expression(paste("INLA BMA:",~alpha,sep="")),xlab=~alpha,ylab="Density")
plot(margeff$marginals.hyperpar$`Precision for the Gaussian observations`,type="l",main=~tau)


alpha.mu <- inla.emarginal(function(x){x},margeff$marginals.fixed$`(Intercept)`)
alpha.mu2 <- inla.emarginal(function(x){x^2},margeff$marginals.fixed$`(Intercept)`)
alpha.sd <- sqrt(alpha.mu2-alpha.mu^2)

tau.mu <- inla.emarginal(function(x){x},margeff$marginals.hyperpar$`Precision for the Gaussian observations`)
tau.mu2 <- inla.emarginal(function(x){x^2},margeff$marginals.hyperpar$`Precision for the Gaussian observations`)
tau.sd <- sqrt(tau.mu2-tau.mu^2)



# BMA Plots: Step-by-Step BMA ---------------------------------------------
# The goal of the following code, in particular the plots, is to illustrate the steps taken by the INLABMA:::fitmargBMA2 function to perform the Bayesian Model Averaging for the INLA within MCMC algorithm.

# Matrices for the posterior marginals
head(modalt.1$marginals.fixed$`(Intercept)`)
head(modalt.50$marginals.fixed$`(Intercept)`)
head(modalt.100$marginals.fixed$`(Intercept)`)

# spline functions for the posterior marginals
marg.spline.1 <- INLABMA::mysplinefun(modalt.1$marginals.fixed$`(Intercept)`)
marg.spline.50 <- INLABMA::mysplinefun(modalt.50$marginals.fixed$`(Intercept)`)
marg.spline.100 <- INLABMA::mysplinefun(modalt.100$marginals.fixed$`(Intercept)`)

# x range for the marginals, xx points are the set of values that the BMA approximate marginal results are calculated at
xx <- margeff$marginals.fixed$`(Intercept)`[,1]
xrange1 <- c(min(xx,modalt.1$marginals.fixed$`(Intercept)`[,1]),max(xx,modalt.1$marginals.fixed$`(Intercept)`[,1]))
xrange50 <- c(min(modalt.50$marginals.fixed$`(Intercept)`[,1],xx),max(xx,modalt.50$marginals.fixed$`(Intercept)`[,1]))
xrange100 <- c(min(modalt.100$marginals.fixed$`(Intercept)`[,1],xx),max(xx,modalt.100$marginals.fixed$`(Intercept)`[,1]))

# Plot the posterior marginals
pdf("ExBMAalt_PostMarg.pdf",h=8,w=15,pointsize=18)
par(mfrow=c(1,3))
plot(modalt.1$marginals.fixed$`(Intercept)`,xlim=xrange1,pch=19,xlab=~alpha,ylab="Density")
title(main=expression(paste("Model 1: Posterior Marginal of ",~alpha,sep="")))
plot(modalt.50$marginals.fixed$`(Intercept)`,xlim=xrange50,pch=19,xlab=~alpha,ylab="Density")
title(main=expression(paste("Model 50: Posterior Marginal of ",~alpha,sep="")))
plot(modalt.100$marginals.fixed$`(Intercept)`,xlim=xrange100,pch=19,xlab=~alpha,ylab="Density")
title(main=expression(paste("Model 100: Posterior Marginal of ",~alpha,sep="")))
dev.off()

# Plot the posterior marginals with spline functions overlaid
pdf("ExBMAalt_PostMargSpline.pdf",h=8,w=15,pointsize=18)
par(mfrow=c(1,3),mar=c(4,4,3,1),oma=c(0,0,0,10))
plot(modalt.1$marginals.fixed$`(Intercept)`,xlim=xrange1,pch=19,xlab=~alpha,ylab="Density")
title(main=expression(atop(paste("Model 1: Posterior Marginal of ",~alpha,sep="")," with Spline Function Overlayed")))
curve(marg.spline.1(x),col="red",add=TRUE,lwd=2)

plot(modalt.50$marginals.fixed$`(Intercept)`,xlim=xrange50,pch=19,xlab=~alpha,ylab="Density")
title(main=expression(atop(paste("Model 50: Posterior Marginal of ",~alpha,sep="")," with Spline Function Overlayed")))
curve(marg.spline.50(x),col="red",add=TRUE,lwd=2)

plot(modalt.100$marginals.fixed$`(Intercept)`,xlim=xrange100,pch=19,xlab=~alpha,ylab="Density")
title(main=expression(atop(paste("Model 100: Posterior Marginal of ",~alpha,sep="")," with Spline Function Overlayed")))
curve(marg.spline.100(x),col="red",add=TRUE,lwd=2)

legend(x=4.2,y=2.5,c("Posterior Marginal",expression(paste("Spline Function, s"[~alpha]^{i},sep=""))),pch=c(19,NA),lty=c(NA,1),lwd = c(NA,2),col=c("black","red"),xpd=NA,bty="n")
dev.off()


# Plot the posterior marginals with spline functions overlaid with xx points
pdf("ExBMAalt_PostMargSplineBMAPoints.pdf",h=8,w=15,pointsize=18)
par(mfrow=c(1,3),mar=c(4,4,3,1),oma=c(0,0,0,10))
plot(modalt.1$marginals.fixed$`(Intercept)`,xlim=xrange1,pch=19,xlab=~alpha,ylab="Density")
title(main=expression(atop(paste("Model 1: Posterior Marginal of ",~alpha, " with ",sep=""),paste(" Spline Function and ", ~alpha[p]," Values for BMA",sep=""))))
curve(marg.spline.1(x),col="red",add=TRUE,lwd=2)
points(xx,rep(-0.1,length(xx)),col="blue",pch=19,cex=0.45)
points(xx,marg.spline.1(xx),col="blue",pch=4,lwd=1.25)

plot(modalt.50$marginals.fixed$`(Intercept)`,xlim=xrange50,pch=19,xlab=~alpha,ylab="Density")
title(main=expression(atop(paste("Model 50: Posterior Marginal of ",~alpha, " with ",sep=""),paste(" Spline Function and ", ~alpha[p]," Values for BMA",sep=""))))
curve(marg.spline.50(x),col="red",add=TRUE,lwd=2)
points(xx,rep(-0.1,length(xx)),col="blue",pch=19,cex=0.45)
points(xx,marg.spline.50(xx),col="blue",pch=4,lwd=1.25)

plot(modalt.100$marginals.fixed$`(Intercept)`,xlim=xrange100,pch=19,xlab=~alpha,ylab="Density")
title(main=expression(atop(paste("Model 100: Posterior Marginal of ",~alpha, " with ",sep=""),paste(" Spline Function and ", ~alpha[p]," Values for BMA",sep=""))))
curve(marg.spline.100(x),col="red",add=TRUE,lwd=2)
points(xx,rep(-0.1,length(xx)),col="blue",pch=19,cex=0.45)
points(xx,marg.spline.100(xx),col="blue",pch=4,lwd=1.25)

legend(x=4.15,y=2.5,c("Posterior Marginal",expression(paste("Spline Function, s"[~alpha]^{i},sep="")),expression(paste(~alpha[p]," values for BMA")),expression(paste("s"[~alpha]^{i},"(",~alpha[p],") for BMA"," ",sep=""))),pch=c(19,NA,19,4),lty=c(NA,1,NA,NA),lwd = c(NA,2,NA,1.25),pt.cex = c(1,1,0.45,1),col=c("black","red","blue","blue"),xpd=NA,bty="n")

dev.off()


# Plot the with xx points and spline but nothing else
pdf("ExBMAalt_SplineBMAPoints.pdf",h=8,w=15,pointsize=18)
par(mfrow=c(1,3),mar=c(4,4,3,1),oma=c(0,0,0,10))
plot(modalt.1$marginals.fixed$`(Intercept)`[,1],marg.spline.1(modalt.1$marginals.fixed$`(Intercept)`[,1]),col="red",add=TRUE,lwd=2,type="l",xlim=xrange1,xlab=~alpha,ylab="Density")
title(main=expression(atop(paste("Model 1: Posterior Marginal of ",~alpha, " with ",sep=""),paste(" Spline Function and ", ~alpha[p]," Values for BMA",sep=""))))
points(xx,rep(-0.1,length(xx)),col="blue",pch=19,cex=0.45)
points(xx,marg.spline.1(xx),col="blue",pch=4,lwd=2.25)

plot(modalt.50$marginals.fixed$`(Intercept)`[,1],marg.spline.50(modalt.50$marginals.fixed$`(Intercept)`[,1]),col="red",add=TRUE,lwd=2,type="l",xlim=xrange50,xlab=~alpha,ylab="Density")
title(main=expression(atop(paste("Model 50: Posterior Marginal of ",~alpha, " with ",sep=""),paste(" Spline Function and ", ~alpha[p]," Values for BMA",sep=""))))
points(xx,rep(-0.1,length(xx)),col="blue",pch=19,cex=0.45)
points(xx,marg.spline.50(xx),col="blue",pch=4,lwd=2.25)

plot(modalt.100$marginals.fixed$`(Intercept)`[,1],marg.spline.100(modalt.100$marginals.fixed$`(Intercept)`[,1]),col="red",add=TRUE,lwd=2,type="l",xlim=xrange100,xlab=~alpha,ylab="Density")
title(main=expression(atop(paste("Model 100: Posterior Marginal of ",~alpha, " with ",sep=""),paste(" Spline Function and ", ~alpha[p]," Values for BMA",sep=""))))
points(xx,rep(-0.1,length(xx)),col="blue",pch=19,cex=0.45)
points(xx,marg.spline.100(xx),col="blue",pch=4,lwd=2.25)

legend(x=4.15,y=2.5,c(expression(paste("Spline Function, s"[~alpha]^{i},sep="")),expression(paste(~alpha[p]," values for BMA")),expression(paste("s"[~alpha]^{i},"(",~alpha[p],") for BMA"," ",sep=""))),pch=c(NA,19,4),lty=c(1,NA,NA),lwd = c(2,NA,1.25),pt.cex = c(1,0.45,1),col=c("red","blue","blue"),xpd=NA,bty="n")

dev.off()


## Plot the all marginals on support for final approximate marginals, post-BMA
pdf("ExBMAalt_BMAPointsAll.pdf",h=8,w=15,pointsize=14)
par(mfrow=c(1,1),mar=c(4,4,3,1),oma=c(0,0,0,10))
plot(xx,marg.spline.1(xx),xlim=xrange1,pch=4,lwd=2,xlab=~alpha,ylab="Density")
title(main=expression(paste("s"[~alpha]^{i},"(",~alpha[p],") for BMA",", i=1,50,100"," ",sep="")))
points(xx,marg.spline.50(xx),col="red",pch=4,lwd=2)
points(xx,marg.spline.100(xx),col="blue",pch=4,lwd=2)

legend(x=4.1,y=2.5,c("Model 1","Model 50","Model 100"),pch=c(4,4,4),lty=c(NA,NA,NA),lwd = c(2,2,2),col=c("black","red","blue"),xpd=NA,bty="n")

dev.off()

## Plot the all marginals on support for final approximate marginals, post-BMA, with the final BMA overlaid
pdf("ExBMAalt_BMAPointsAllBMA.pdf",h=8,w=15,pointsize=18)
par(mfrow=c(1,1),mar=c(4,4,3,1),oma=c(0,0,0,10))
plot(xx,marg.spline.1(xx),xlim=xrange1,col="dark green",pch=4,lwd=2,xlab=~alpha,ylab="Density")
title(main=expression(paste("s"[~alpha]^{i},"(",~alpha[p],") for BMA, i=1,50,100, with BMA Values",sep="")))
points(xx,marg.spline.50(xx),col="red",pch=4,lwd=2)
points(xx,marg.spline.100(xx),col="blue",pch=4,lwd=2)
points(margeff$marginals.fixed$`(Intercept)`,col="black",pch=19,lwd=2)

legend(x=4.1,y=2.5,c("Model 1","Model 50","Model 100","BMA"),pch=c(4,4,4,19),lty=c(NA,NA,NA),lwd = c(2,2,2,2),col=c("dark green","red","blue","black"),xpd=NA,bty="n")

dev.off()

# Final intercept BMA approximate posterior marginal
pdf("ExBMAalt_FinalBMA.pdf",h=8,w=15,pointsize=14)
par(mfrow=c(1,1),mar=c(4,4,3,1))
plot(margeff$marginals.fixed$`(Intercept)`,xlab=~alpha,pch=19,lwd=2)
title(expression(paste("BMA Posterior Marginal for ",~alpha,sep="")))
dev.off()



# sessionInfo() -----------------------------------------------------------

sessionInfo()
