
# Gaussian Field for Parameter Values that Produced Errors ----------------

# We want to consider the Gaussian field produced from the parameter values that resulted in errors for some grid and mesh combinations in the SBC simulation study. We also consider the dame for those that produced more than 10 FFT warnings when inla was running.

# Author: Nadeen Khaleel

# Setwd and Load Libraries ------------------------------------------------

# Either setwd() to the source file location, or run the following:
library("rstudioapi")
setwd(dirname(getActiveDocumentContext()$path))

library(spatstat)
library(raster)


# SBC: Parameters for Errors ----------------------------------------------
# What parameter values result in errors?

load("../GAUSSIAN_OUTPUT/GridMeshSGSBCSS.rda")
err.vec <- rep(NA,50)
ii <- sum(!is.na(err.vec))
for (l in 1:4){
  if (sum(!is.na(run.out.final[[l]]$mess.ls$err))!=0){
    err.vec[(ii+1):(ii+sum(!is.na(run.out.final[[l]]$mess.ls$err)))] <- which(!is.na(run.out.final[[l]]$mess.ls$err))
    ii <- sum(!is.na(err.vec))
  }
}
err.vec
err.vec.u <- unique(err.vec[!is.na(err.vec)])
err.vec.u
true.theta.err <- true.theta.final[err.vec.u,]

true.theta.err
# beta0       beta1     beta2       sigma        rho      tau
# 1156  76.68879  11.1816367  35.77290 0.143827016 35.2681690 15.81514
# 2291 -21.55799  -0.6290255 -16.99816 0.005167183  1.6328183 37.09222
# 1424  54.88868 -63.4418042 -81.78120 0.008975999  0.7135963 13.10510

nrow(true.theta.err)

# SBC: Gaussian Field for Errors ------------------------------------------
# What does the Gaussian field look like for the runs which resulted in errors.

load("../GAUSSIAN_CODE/GridMeshSGSSCov.rda")

max.mu <- rep(NA,nrow(true.theta.err))

W <- owin(c(0,5),c(0,5))
gridx <- 50; gridy <- 50

minx <- W$xrange[1]; maxx <- W$xrange[2]; miny <- W$yrange[1]; maxy <- W$yrange[2]; midx <- 0.5*(maxx-minx)/gridx; midy <- 0.5*(maxy-miny)/gridy
x <- seq(from=minx+midx,to=maxx-midx,by=2*midx)
y <- seq(from=miny+midy,to=maxy-midy,by=2*midy)
Grid <- expand.grid(x, y)

xy <- cbind(Grid$Var1,Grid$Var2)
cov1 <- extract(cov1.ras,xy); cov2 <- extract(cov2.ras,xy)

for (k in 1:nrow(true.theta.err)){
  theta <- true.theta.err[k,]
  beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
  mu <- rep(beta.0,dim(Grid)[1]) + beta.1*cov1 + beta.2*cov2
  
  print("Values ",k)
  print(theta)
  print(max(mu))
  max.mu[k] <- max(mu)
  print(mean(mu))
  print(median(mu))
  print(sum(mu>10)/length(mu))
}
# 
# [1] "Values "
# beta0    beta1   beta2    sigma      rho      tau
# 1156 76.68879 11.18164 35.7729 0.143827 35.26817 15.81514
# [1] 184.3677
# [1] 102.3319
# [1] 102.0322
# [1] 1
# [1] "Values "
# beta0      beta1     beta2       sigma      rho      tau
# 2291 -21.55799 -0.6290255 -16.99816 0.005167183 1.632818 37.09222
# [1] 0.2709207
# [1] -31.34041
# [1] -31.19886
# [1] 0
# [1] "Values "
# beta0    beta1    beta2       sigma       rho     tau
# 1424 54.88868 -63.4418 -81.7812 0.008975999 0.7135963 13.1051
# [1] 182.8461
# [1] -23.16162
# [1] -22.91204
# [1] 0.3036



# SBC: Parameters for FFT -------------------------------------------------
# What parameter values result in more than 10 FFT warnings.

errfft.vec <- rep(NA,200)
ii <- sum(!is.na(errfft.vec))
for (l in 1:4){
  if (sum((run.out.final[[l]]$mess.ls$FFT)>10,na.rm=T)){
    errfft.vec[(ii+1):(ii+sum((run.out.final[[l]]$mess.ls$FFT)>10,na.rm=T))] <- which((run.out.final[[l]]$mess.ls$FFT)>10)
    ii <- sum(!is.na(errfft.vec))
  }
}
errfft.vec
errfft.vec.u <- unique(errfft.vec[!is.na(errfft.vec)])
errfft.vec.u
true.theta.err <- true.theta.final[errfft.vec.u,]

true.theta.err
# beta0       beta1        beta2        sigma          rho       tau
# 262  -25.2205642 -10.6758999  -2.83919163 1.0357489760    1.2574731 35.688689
# 414  -70.8316145   7.9654479 -42.16744423 0.2475388325  121.7309755  5.523062
# 538  -27.8337080 -36.6949436  20.23092253 0.0004531475    2.0806697  6.053843
# 569  -17.1719322  28.9196265 -54.14510203 0.5134261713  895.6003883 16.192712
# 587  -85.0208966  -1.6076044  53.55862318 0.7267165769  448.9031638 16.927243
# 807   37.6435076   0.2532543 -43.39469840 2.3343479624    2.9229401 36.892985
# 947    2.7904943 -48.6199771  24.45419298 1.1711801840    2.0837243  9.529548
# 976  -10.1041812 -47.6708298  17.65784594 1.5081059117    1.9417542 11.072543
# 1018  67.5425431   1.4012798  -8.49899259 2.2247447131    1.4939535 18.404274
# 1029  -6.5642731  15.9477496  21.76952257 0.6672986271    0.3951701  7.241151
# 1138  17.7318616 -16.0281145   9.93264999 1.1131154252    1.3045247 24.481217
# 1165  18.4132657  37.9246893 -54.85268341 1.5859328349    0.7210107 14.695253
# 1245 -17.9375284 -69.5537134   1.85374846 0.0062497746  143.7907751  9.316056
# 1292  -1.9062388 -30.3077222  16.07486143 2.2266788255    3.6061748 10.538284
# 1594  17.7055843 -12.0784138  37.67915082 0.0231245006   11.9683070 22.982516
# 1662 -19.5486614  63.6213799  39.82529049 1.7170140568    0.8396312 18.724519
# 1767  55.1007322 -13.1877404  -7.18578558 0.0127483075    5.7385581  7.557147
# 1775  32.9326127  18.7585553 -18.34445517 0.0239119726  319.3186263 33.695734
# 1826  33.0094230  56.8332278 -14.01939598 0.0022977344    2.3013610 13.694283
# 1896  -7.8910264 -56.0013442  25.06066605 0.0123756604   20.3748804 30.202421
# 1942   7.3631411  10.4046818 -14.04796576 3.0364351356    4.2414104 31.021718
# 2152 -10.5652521  47.7731137 -34.33093016 0.0314440811    7.2825460 35.649694
# 2213 -36.4479336 -70.7403372  10.66141650 0.0025906497   13.8174862  8.317204
# 2308 -26.5680447 -10.5575304  29.41771234 1.1949964644    1.8834535  4.037396
# 2507  -5.8666779  62.8345007  86.20097744 0.0241067250    4.4533821 17.542720
# 12    22.7887970  15.8084011 -18.18590706 0.0377743765    7.8701069 13.177834
# 131   25.1553062   9.9760777   7.11024159 2.9343483543    1.4686701 33.737507
# 199   20.9516730  70.6809543 -69.91490091 3.7764778433    1.2739840  5.813121
# 248   20.9551915  17.0349149  -7.79809779 3.6504692888    1.4620576  4.894299
# 433  -33.4191475   6.4477045 -39.54385101 2.4334637465    1.3911695 17.360926
# 512   16.4145579   9.7102935 -17.52669683 2.0530609691    1.8758229  1.258695
# 704    2.8978762 -31.4341059  43.76610885 1.5297261473    0.8626578 27.915033
# 787   35.5245740  -9.5852691 -27.02384689 0.0067397256    1.0301237 12.002744
# 924   18.5685490  24.7372694  -6.99616629 3.2657766516    0.8301924  9.031065
# 937  -27.4099907 -44.8970278 -40.71354758 2.8983675090    1.0696391 40.631337
# 988  -31.3184420  12.0120957  59.93093621 0.0215715063    9.1983813 12.622825
# 997  -77.7327213  24.8063634 -37.29710106 0.3390765837 3654.7830370  8.020451
# 1049  46.3841906  -5.5418238 -69.76613912 0.0008633389   14.1374079 28.855638
# 1054  35.0937819   5.0128508  -0.80563288 3.0653735076    2.1523490 12.169554
# 1091  -1.5890741 -61.3377885  13.87114300 0.0469044354    2.8603790  6.161851
# 1143  55.2558281  -6.6006199 -18.31642572 0.0160174718    1.0219323 10.619322
# 1453   5.2926700  39.2906950 -33.69043544 0.0197590796    0.7739851 16.543831
# 1497 -51.2205763 -42.3586174  51.46305598 2.5275456520    0.9153595 42.044649
# 1544 -13.1125738  10.4470260  -1.40839175 4.0184174055    0.7063964 13.036194
# 1681  -1.2784730   8.6699256 -14.60398251 1.7766603491    0.7250872 55.528857
# 1702 -37.8388549   1.7053469  13.27679319 4.5459797266    2.8922863 22.994110
# 1722  27.0421621  17.2444576  49.29888193 2.0245741357    1.0196529 32.939232
# 1792  -3.1221650 -54.7414735   9.83850395 0.1624358892  516.2329325 19.037747
# 1808  25.7180356  36.7130881  -3.08254097 3.0833346568    1.9008006  9.681762
# 1874  12.7125901  21.0777505  22.85046084 0.0850951498  142.4616381  5.963942
# 1954  13.1533771  37.7035067 -50.93017813 1.0823351490    0.5695155 10.690256
# 2043  -7.0058032 -12.1452055 -19.37274770 0.0736624685    0.3527016  3.698530
# 2103   2.0834346  57.4590350   7.77788303 0.0030558143  103.8464061  2.588006
# 2117 -20.9256832  39.7163180  20.83178981 0.0135361052   17.0846887 25.300998
# 2203  36.9245837 -35.9993837  61.24482339 0.0392599677    3.2462261  5.410701
# 2210 -20.2200886 -22.2200067 -54.39026016 4.9660102085    1.8176063 10.524492
# 2261  70.0154020   2.5220432  28.92595903 3.5654678256    1.3124860 30.126299
# 2289   8.8625208 -12.6991591 -43.03194216 3.2590246388    2.1232821 25.843873
# 2324   0.4323911 -55.2213541  -0.78169944 3.6003074747    1.8011756 16.126657
# 2348  52.2408767 -42.9856405 -27.27181295 0.0134372920    2.3736903 21.824331
# 2384 -22.6839638 -37.4699039  15.23956780 1.8453430032    1.3456451 11.023937
# 2435 -35.3404272  30.4538098   4.45198376 3.3830250979    2.1846675 34.452073
# 17    -0.4543259 -17.8257538   5.93873455 0.0174052114   16.9583381  5.758898
# 68    -4.9765819 -37.9112032 -28.11912805 0.0097745052    5.4770819 29.903928
# 175   25.7468148  12.3690278  42.57079146 8.5914046384    1.2510503  4.689888
# 177  -46.7351889 -13.4484407  41.36604298 5.4127219575    0.7573849  9.113723
# 258  -21.2506399   0.7387042  -0.52088560 0.0608194363   38.7050315 12.854070
# 393  -48.2212470   1.3812242  27.92560629 0.0434786389    1.6554240  7.289727
# 401   -3.4835573 -41.9509013  20.81173542 2.9688537993    1.4535821  6.635318
# 416  -33.6011195  13.6443713 -71.58998233 0.0164503173    1.1674306 17.940285
# 598   51.0253264  19.2679764   1.92608266 1.2812809728    0.5238544 29.436057
# 635   -8.9960342 -22.0351319 -11.72394259 0.0517795557  130.3387546  6.890232
# 691  -30.4524079 -21.4726560 -44.95936742 2.2123584579    0.9136189 14.604990
# 727    4.7822308  24.2781041   4.59541459 1.5986729801    0.5840254 21.193553
# 742  -24.8578125  22.5405151 -80.41855174 1.9923927633    0.6602199 19.482321
# 819  -29.3017784 -44.8496560 -19.45027454 0.0154967467    0.4241285  6.264043
# 873  -14.7963444  15.6911350   3.46899289 1.4887033022    0.7462092 21.854715
# 1057  41.1001040 -10.0500817   4.61417540 0.0235888525  221.4634314 27.405039
# 1060  18.5840586 -38.2873027 -25.07085585 0.0387586180    8.2126598  7.036516
# 1240 -49.6316611  14.1262362 -20.61821310 0.0805207814   48.5155482 61.523514
# 1255   6.0598251  17.9178767  21.37611622 0.0072776124    0.9158425 14.432394
# 1534  29.0073969  35.2517674  11.81521742 2.3703436405    0.8875787 34.496461
# 1733  49.7236928 -17.0338599   5.95837808 0.0320699453   18.0991836 10.181583
# 1765 -19.8067959 -68.7552891 -50.19699630 3.6880014388    0.9364694 34.763038
# 1799  -5.5943721 -55.1277927 -25.00035727 2.2195889787    1.1907296 16.767241
# 1833  -8.0003052   6.2338388  -3.09471507 0.0537654666    1.2538007  7.758198
# 1983 -34.0753157   8.2945497 -15.48567860 0.0242077919   12.6377913  6.245994
# 2190  16.6889802  -0.9605723 -50.37654924 1.9141125295    0.7896841 18.421004
# 2235 -28.5163873 -38.5351398  23.62711588 0.8898283539  313.7090949 26.980163
# 2291 -21.5579903  -0.6290255 -16.99815636 0.0051671834    1.6328183 37.092218
# 2301  21.3069094  15.5288887 -14.62132583 2.3697501900    0.9746516 11.885727
# 2310 -14.3668126 -10.3716586 -15.17013474 0.0225626936    1.0837241 20.646780
# 2315  -9.3100350 -30.7209315 -23.39980387 2.6844474904    1.5462305 10.320185
# 2339  -7.9725927 -37.8067972 -17.86093162 1.5566704936    0.6119787 18.654544
# 2362 -33.8195608  38.4879177 -21.78816349 2.2314626634    1.8087917  5.923805
# 2369  16.7612730   4.0718850  51.95688059 0.1303387944  325.5266285 19.990592
# 2406  23.5227550 -13.0609186  80.21535733 0.0121074561    1.3566011  9.986399
# 2463  10.2996504 -39.4690871  50.08924421 0.0197484310    2.4353969  6.623022
# 2552  37.9782026   3.7562482  44.59143505 0.0324745660   55.2525667 13.046281
# 55     7.9816209  18.9180146  39.82519790 0.0280796978    3.5377945 24.606530
# 92    -3.5228100 -26.3623533  26.28138847 0.4457491837  508.6488164 24.392600
# 97   -35.5611575 -75.2943925 -15.03060687 0.0174445973    1.3084312  9.815538
# 114    2.7303890 -10.1417400  43.54341395 0.0517141025   62.4457787 33.449760
# 369   71.6920163  24.5522031  41.73244083 0.7293049308  104.4745535 75.148096
# 417   42.5246506  20.3874913 -51.44565490 0.0306638882    0.5880227 11.541761
# 427  -27.6368769  17.2055759  24.55039859 0.1581087956   29.4930163 33.818900
# 431   -1.7310457 -61.2707900 -27.00126462 0.0075624993    1.2921023  2.098873
# 467    3.6214225 -18.1509123 -36.06353303 0.1237685011   26.4418370 25.170518
# 504    5.0158298 -11.1202574  -6.06692935 0.0290056909   17.6162227 24.115324
# 530   -1.8738995  16.5573896 -25.20939415 0.0533744301    4.1374348 51.349251
# 536  -35.4746506   4.9290579   1.21223739 0.5405779761  108.0283424 32.335659
# 576  -57.1945026 -35.0215031 -46.94388571 0.2742630871  362.5869797 11.628722
# 612  -45.7200890 -19.0118472 -16.53426397 0.0372818575    4.4719543  5.798423
# 636   53.4898594  -4.2525134 -35.69278647 0.0511313246    0.3574896 10.247094
# 638   23.0542859 -18.3126143  -9.48945049 0.9850146570  134.2325575 32.411305
# 743    2.4565712  24.0026234 -25.05527092 0.0071697252    0.8266276 16.920230
# 753  -61.3913482   9.0362110   0.04006848 0.1337366848    8.9388014 44.007631
# 820   57.3164049 -21.8626435 -38.23214735 0.4561251720  167.5464252 34.633922
# 954  -15.6676513 -53.6721668 -81.98182444 0.0709087325    9.0748510 20.050087
# 1093  26.2402291 -50.8935588  -9.49527726 0.0151327395    6.4971012 24.078570
# 1108  -9.0318331 -21.3147931  57.39712687 0.0135784013    1.9163673 17.150689
# 1213   7.1062315 -73.3199083  -8.82425535 0.2993455601   33.7614204 48.672338
# 1258  29.7492420  18.1653519 -43.51530955 0.1950880979   31.7647664 29.311468
# 1330  15.9375747  21.9846128 -13.52255592 0.0406769024    4.1385828 25.502586
# 1339  38.2229152  11.4993524  10.28825924 0.0397703173    4.9169852 46.550936
# 1341 -71.1980625  55.5470967   8.99954922 0.0219866249    2.0944774 29.539115
# 1470  -2.7665443 -26.2628665 -12.30613596 0.0330676113    6.3539859 12.320686
# 1481  -0.1947145  47.4703896   3.75651503 0.0004324033    2.7731058 17.061803
# 1577  67.0783271  -7.3578174  -7.21294148 0.0584801764    7.6092724  7.092982
# 1663  34.4189346 -17.7390678  20.26873817 0.5766062555   52.2977466 17.141696
# 1757  40.5130298 -25.5361772  12.56889236 0.5027638938  172.1453923 19.000244
# 1824   1.9262268  -1.4605382 -35.66695684 0.0215755900    1.3269422 10.735469
# 1973  30.0495540 -25.0322359 -46.08300138 0.0406844762    3.8596853 17.888315
# 2032   8.4920883  10.1963718 -11.97262895 0.1912894036   93.1535238 38.491157
# 2054 -15.3021637 -42.6557265  24.74003587 0.0197977327    0.6318304 29.706625
# 2087 -14.9657955 -40.3345892   1.28610053 0.0365017179    1.2908537 37.562833
# 2164  18.2220490  -2.3892257 -10.15072935 0.0647130682   22.0531858 18.566509
# 2381  -5.1179452   4.9354494  21.19280229 0.0692124460  286.8998441 20.476723
# 2401  59.3488536  -0.8901199  -9.53011940 0.0943471514   11.2152695 36.012155
# 2494 -59.9725057 -62.6030628 -19.29313148 0.5966114106   52.1804584 21.852871
# 2535 -29.7715735  32.1599903 -48.17936759 0.0616239177   24.0369206  5.453190

nrow(true.theta.err)

# SBC: Gaussian Field for FFT ---------------------------------------------
# What does the Gaussian field look like for the runs which resulted in more than 10 FFT warnings.

load("../GAUSSIAN_CODE/GridMeshSGSSCov.rda")

max.mu <- rep(NA,nrow(true.theta.err))

W <- owin(c(0,5),c(0,5))
gridx <- 50; gridy <- 50

minx <- W$xrange[1]; maxx <- W$xrange[2]; miny <- W$yrange[1]; maxy <- W$yrange[2]; midx <- 0.5*(maxx-minx)/gridx; midy <- 0.5*(maxy-miny)/gridy
x <- seq(from=minx+midx,to=maxx-midx,by=2*midx)
y <- seq(from=miny+midy,to=maxy-midy,by=2*midy)
Grid <- expand.grid(x, y)

xy <- cbind(Grid$Var1,Grid$Var2)
cov1 <- extract(cov1.ras,xy); cov2 <- extract(cov2.ras,xy)

for (k in 1:nrow(true.theta.err)){
  theta <- true.theta.err[k,]
  beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
  mu <- rep(beta.0,dim(Grid)[1]) + beta.1*cov1 + beta.2*cov2
  
  print("Values ")
  print(theta)
  print(max(mu))
  max.mu[k] <- max(mu)
  # print(exp(max(mu$v)))
  print(mean(mu))
  print(median(mu))
  print(sum(mu>10)/length(mu))
  # print(mean(exp(mu$v)))
}

# [1] "Values "
# beta0    beta1     beta2    sigma      rho      tau
# 262 -25.22056 -10.6759 -2.839192 1.035749 1.257473 35.68869
# [1] -11.69957
# [1] -32.27598
# [1] -32.16539
# [1] 0
# [1] "Values "
# beta0    beta1     beta2     sigma     rho      tau
# 414 -70.83161 7.965448 -42.16744 0.2475388 121.731 5.523062
# [1] -10.63595
# [1] -90.2134
# [1] -89.5602
# [1] 0
# [1] "Values "
# beta0     beta1    beta2        sigma     rho      tau
# 538 -27.83371 -36.69494 20.23092 0.0004531475 2.08067 6.053843
# [1] 43.16764
# [1] -35.39454
# [1] -35.47557
# [1] 0.0264
# [1] "Values "
# beta0    beta1    beta2     sigma      rho      tau
# 569 -17.17193 28.91963 -54.1451 0.5134262 895.6004 16.19271
# [1] 71.98285
# [1] -32.47275
# [1] -31.72269
# [1] 0.0896
# [1] "Values "
# beta0     beta1    beta2     sigma      rho      tau
# 587 -85.0209 -1.607604 53.55862 0.7267166 448.9032 16.92724
# [1] 44.72872
# [1] -56.03894
# [1] -56.4693
# [1] 0.0104
# [1] "Values "
# beta0     beta1    beta2    sigma     rho      tau
# 807 37.64351 0.2532543 -43.3947 2.334348 2.92294 36.89298
# [1] 94.55034
# [1] 13.62337
# [1] 13.94533
# [1] 0.5656
# [1] "Values "
# beta0     beta1    beta2   sigma      rho      tau
# 947 2.790494 -48.61998 24.45419 1.17118 2.083724 9.529548
# [1] 93.81292
# [1] -8.536003
# [1] -8.628389
# [1] 0.2636
# [1] "Values "
# beta0     beta1    beta2    sigma      rho      tau
# 976 -10.10418 -47.67083 17.65785 1.508106 1.941754 11.07254
# [1] 70.93733
# [1] -24.72619
# [1] -24.79392
# [1] 0.11
# [1] "Values "
# beta0   beta1     beta2    sigma      rho      tau
# 1018 67.54254 1.40128 -8.498993 2.224745 1.493953 18.40427
# [1] 79.54562
# [1] 63.53136
# [1] 63.68438
# [1] 1
# [1] "Values "
# beta0    beta1    beta2     sigma       rho      tau
# 1029 -6.564273 15.94775 21.76952 0.6672986 0.3951701 7.241151
# [1] 74.64287
# [1] 13.73003
# [1] 13.69986
# [1] 0.5872
# [1] "Values "
# beta0     beta1   beta2    sigma      rho      tau
# 1138 17.73186 -16.02811 9.93265 1.113115 1.304525 24.48122
# [1] 50.16763
# [1] 15.03924
# [1] 14.89471
# [1] 0.6908
# [1] "Values "
# beta0    beta1     beta2    sigma       rho      tau
# 1165 18.41327 37.92469 -54.85268 1.585933 0.7210107 14.69525
# [1] 125.6338
# [1] 7.337098
# [1] 7.934973
# [1] 0.4768
# [1] "Values "
# beta0     beta1    beta2       sigma      rho      tau
# 1245 -17.93753 -69.55371 1.853748 0.006249775 143.7908 9.316056
# [1] 76.25238
# [1] -52.57792
# [1] -52.04658
# [1] 0.058
# [1] "Values "
# beta0     beta1    beta2    sigma      rho      tau
# 1292 -1.906239 -30.30772 16.07486 2.226679 3.606175 10.53828
# [1] 55.91249
# [1] -8.504189
# [1] -8.54168
# [1] 0.1624
# [1] "Values "
# beta0     beta1    beta2     sigma      rho      tau
# 1594 17.70558 -12.07841 37.67915 0.0231245 11.96831 22.98252
# [1] 105.3596
# [1] 32.48013
# [1] 32.06966
# [1] 0.8588
# [1] "Values "
# beta0    beta1    beta2    sigma       rho      tau
# 1662 -19.54866 63.62138 39.82529 1.717014 0.8396312 18.72452
# [1] 196.4282
# [1] 35.24443
# [1] 35.4024
# [1] 0.6996
# [1] "Values "
# beta0     beta1     beta2      sigma      rho      tau
# 1767 55.10073 -13.18774 -7.185786 0.01274831 5.738558 7.557147
# [1] 72.18288
# [1] 44.3381
# [1] 44.38358
# [1] 1
# [1] "Values "
# beta0    beta1     beta2      sigma      rho      tau
# 1775 32.93261 18.75856 -18.34446 0.02391197 319.3186 33.69573
# [1] 83.38986
# [1] 32.34426
# [1] 32.50219
# [1] 0.9432
# [1] "Values "
# beta0    beta1    beta2       sigma      rho      tau
# 1826 33.00942 56.83323 -14.0194 0.002297734 2.301361 13.69428
# [1] 173.6934
# [1] 54.35543
# [1] 54.39823
# [1] 0.9116
# [1] "Values "
# beta0     beta1    beta2      sigma      rho      tau
# 1896 -7.891026 -56.00134 25.06067 0.01237566 20.37488 30.20242
# [1] 92.9175
# [1] -22.66569
# [1] -22.94451
# [1] 0.1712
# [1] "Values "
# beta0    beta1     beta2    sigma     rho      tau
# 1942 7.363141 10.40468 -14.04797 3.036435 4.24141 31.02172
# [1] 36.4857
# [1] 4.88142
# [1] 5.007094
# [1] 0.294
# [1] "Values "
# beta0    beta1     beta2      sigma      rho      tau
# 2152 -10.56525 47.77311 -34.33093 0.03144408 7.282546 35.64969
# [1] 114.3034
# [1] -5.169688
# [1] -4.755992
# [1] 0.3192
# [1] "Values "
# beta0     beta1    beta2      sigma      rho      tau
# 2213 -36.44793 -70.74034 10.66142 0.00259065 13.81749 8.317204
# [1] 66.26418
# [1] -66.79527
# [1] -66.55458
# [1] 0.0304
# [1] "Values "
# beta0     beta1    beta2    sigma      rho      tau
# 2308 -26.56804 -10.55753 29.41771 1.194996 1.883453 4.037396
# [1] 42.25614
# [1] -15.61114
# [1] -15.88134
# [1] 0.064
# [1] "Values "
# beta0   beta1    beta2      sigma      rho      tau
# 2507 -5.866678 62.8345 86.20098 0.02410673 4.453382 17.54272
# [1] 315.1516
# [1] 74.33185
# [1] 74.29907
# [1] 0.8376
# [1] "Values "
# beta0   beta1     beta2      sigma      rho      tau
# 12 22.7888 15.8084 -18.18591 0.03777438 7.870107 13.17783
# [1] 66.1102
# [1] 20.77564
# [1] 20.74785
# [1] 0.8028
# [1] "Values "
# beta0    beta1    beta2    sigma     rho      tau
# 131 25.15531 9.976078 7.110242 2.934348 1.46867 33.73751
# [1] 60.05861
# [1] 34.22873
# [1] 34.25288
# [1] 1
# [1] "Values "
# beta0    beta1    beta2    sigma      rho      tau
# 199 20.95167 70.68095 -69.9149 3.776478 1.273984 5.813121
# [1] 211.304
# [1] 18.2928
# [1] 18.85471
# [1] 0.57
# [1] "Values "
# beta0    beta1     beta2    sigma      rho      tau
# 248 20.95519 17.03491 -7.798098 3.650469 1.462058 4.894299
# [1] 64.17766
# [1] 25.3521
# [1] 25.42047
# [1] 0.934
# [1] "Values "
# beta0    beta1     beta2    sigma     rho      tau
# 433 -33.41915 6.447705 -39.54385 2.433464 1.39117 17.36093
# [1] 22.38265
# [1] -52.11926
# [1] -51.40694
# [1] 8e-04
# [1] "Values "
# beta0    beta1    beta2    sigma      rho      tau
# 512 16.41456 9.710294 -17.5267 2.053061 1.875823 1.258695
# [1] 45.49533
# [1] 11.64072
# [1] 11.85524
# [1] 0.5688
# [1] "Values "
# beta0     beta1    beta2    sigma       rho      tau
# 704 2.897876 -31.43411 43.76611 1.529726 0.8626578 27.91503
# [1] 110.717
# [1] 11.13297
# [1] 10.66187
# [1] 0.5108
# [1] "Values "
# beta0     beta1     beta2       sigma      rho      tau
# 787 35.52457 -9.585269 -27.02385 0.006739726 1.030124 12.00274
# [1] 69.08024
# [1] 15.56926
# [1] 15.73939
# [1] 0.628
# [1] "Values "
# beta0    beta1     beta2    sigma       rho      tau
# 924 18.56855 24.73727 -6.996166 3.265777 0.8301924 9.031065
# [1] 80.06494
# [1] 27.36206
# [1] 27.39626
# [1] 0.88
# [1] "Values "
# beta0     beta1     beta2    sigma      rho      tau
# 937 -27.40999 -44.89703 -40.71355 2.898368 1.069639 40.63134
# [1] 46.50241
# [1] -73.09425
# [1] -73.10692
# [1] 0.01
# [1] "Values "
# beta0   beta1    beta2      sigma      rho      tau
# 988 -31.31844 12.0121 59.93094 0.02157151 9.198381 12.62283
# [1] 137.5526
# [1] 8.194974
# [1] 8.022328
# [1] 0.4752
# [1] "Values "
# beta0    beta1    beta2     sigma      rho      tau
# 997 -77.73272 24.80636 -37.2971 0.3390766 3654.783 8.020451
# [1] -7.18434
# [1] -85.76686
# [1] -85.33143
# [1] 0
# [1] "Values "
# beta0     beta1     beta2        sigma      rho      tau
# 1049 46.38419 -5.541824 -69.76614 0.0008633389 14.13741 28.85564
# [1] 134.0991
# [1] 4.715698
# [1] 4.847285
# [1] 0.4548
# [1] "Values "
# beta0    beta1      beta2    sigma      rho      tau
# 1054 35.09378 5.012851 -0.8056329 3.065374 2.152349 12.16955
# [1] 47.38459
# [1] 37.21637
# [1] 37.19816
# [1] 1
# [1] "Values "
# beta0     beta1    beta2      sigma      rho      tau
# 1091 -1.589074 -61.33779 13.87114 0.04690444 2.860379 6.161851
# [1] 91.19764
# [1] -25.32784
# [1] -25.16617
# [1] 0.1624
# [1] "Values "
# beta0    beta1     beta2      sigma      rho      tau
# 1143 55.25583 -6.60062 -18.31643 0.01601747 1.021932 10.61932
# [1] 78.07908
# [1] 41.67711
# [1] 41.79078
# [1] 0.9976
# [1] "Values "
# beta0   beta1     beta2      sigma       rho      tau
# 1453 5.29267 39.2907 -33.69044 0.01975908 0.7739851 16.54383
# [1] 109.5898
# [1] 6.694301
# [1] 6.972193
# [1] 0.458
# [1] "Values "
# beta0     beta1    beta2    sigma       rho      tau
# 1497 -51.22058 -42.35862 51.46306 2.527546 0.9153595 42.04465
# [1] 77.42199
# [1] -44.30483
# [1] -44.52811
# [1] 0.062
# [1] "Values "
# beta0    beta1     beta2    sigma       rho      tau
# 1544 -13.11257 10.44703 -1.408392 4.018417 0.7063964 13.03619
# [1] 12.82637
# [1] -8.538402
# [1] -8.591927
# [1] 0.0012
# [1] "Values "
# beta0    beta1     beta2   sigma       rho      tau
# 1681 -1.278473 8.669926 -14.60398 1.77666 0.7250872 55.52886
# [1] 23.83845
# [1] -4.959334
# [1] -4.756459
# [1] 0.044
# [1] "Values "
# beta0    beta1    beta2   sigma      rho      tau
# 1702 -37.83885 1.705347 13.27679 4.54598 2.892286 22.99411
# [1] -2.067017
# [1] -29.57543
# [1] -29.64374
# [1] 0
# [1] "Values "
# beta0    beta1    beta2    sigma      rho      tau
# 1722 27.04216 17.24446 49.29888 2.024574 1.019653 32.93923
# [1] 178.5819
# [1] 63.32215
# [1] 62.97583
# [1] 0.96
# [1] "Values "
# beta0     beta1    beta2     sigma      rho      tau
# 1792 -3.122165 -54.74147 9.838504 0.1624359 516.2329 19.03775
# [1] 77.61178
# [1] -25.72212
# [1] -25.2712
# [1] 0.1348
# [1] "Values "
# beta0    beta1     beta2    sigma      rho      tau
# 1808 25.71804 36.71309 -3.082541 3.083335 1.900801 9.681762
# [1] 119.1105
# [1] 42.8316
# [1] 42.56915
# [1] 0.9424
# [1] "Values "
# beta0    beta1    beta2      sigma      rho      tau
# 1874 12.71259 21.07775 22.85046 0.08509515 142.4616 5.963942
# [1] 105.3913
# [1] 36.23948
# [1] 36.22821
# [1] 0.9144
# [1] "Values "
# beta0    beta1     beta2    sigma       rho      tau
# 1954 13.15338 37.70351 -50.93018 1.082335 0.5695155 10.69026
# [1] 118.6921
# [1] 4.146724
# [1] 4.60382
# [1] 0.4392
# [1] "Values "
# beta0     beta1     beta2      sigma       rho     tau
# 2043 -7.005803 -12.14521 -19.37275 0.07366247 0.3527016 3.69853
# [1] 21.09414
# [1] -24.01604
# [1] -23.96849
# [1] 0.0056
# [1] "Values "
# beta0    beta1    beta2       sigma      rho      tau
# 2103 2.083435 57.45904 7.777883 0.003055814 103.8464 2.588006
# [1] 163.3546
# [1] 35.88102
# [1] 35.34203
# [1] 0.7736
# [1] "Values "
# beta0    beta1    beta2      sigma      rho    tau
# 2117 -20.92568 39.71632 20.83179 0.01353611 17.08469 25.301
# [1] 109.0706
# [1] 11.03695
# [1] 10.93194
# [1] 0.5116
# [1] "Values "
# beta0     beta1    beta2      sigma      rho      tau
# 2203 36.92458 -35.99938 61.24482 0.03925997 3.246226 5.410701
# [1] 185.0469
# [1] 52.54554
# [1] 51.69752
# [1] 0.8784
# [1] "Values "
# beta0     beta1     beta2   sigma      rho      tau
# 2210 -20.22009 -22.22001 -54.39026 4.96601 1.817606 10.52449
# [1] 49.5622
# [1] -61.88534
# [1] -61.84505
# [1] 0.0152
# [1] "Values "
# beta0    beta1    beta2    sigma      rho     tau
# 2261 70.0154 2.522043 28.92596 3.565468 1.312486 30.1263
# [1] 145.9045
# [1] 87.40676
# [1] 87.29471
# [1] 1
# [1] "Values "
# beta0     beta1     beta2    sigma      rho      tau
# 2289 8.862521 -12.69916 -43.03194 3.259025 2.123282 25.84387
# [1] 60.32899
# [1] -21.59864
# [1] -21.28937
# [1] 0.114
# [1] "Values "
# beta0     beta1      beta2    sigma      rho      tau
# 2324 0.4323911 -55.22135 -0.7816994 3.600307 1.801176 16.12666
# [1] 73.4377
# [1] -28.32404
# [1] -28.03862
# [1] 0.1192
# [1] "Values "
# beta0     beta1     beta2      sigma     rho      tau
# 2348 52.24088 -42.98564 -27.27181 0.01343729 2.37369 21.82433
# [1] 111.6533
# [1] 15.0175
# [1] 14.90173
# [1] 0.5704
# [1] "Values "
# beta0    beta1    beta2    sigma      rho      tau
# 2384 -22.68396 -37.4699 15.23957 1.845343 1.345645 11.02394
# [1] 42.78179
# [1] -33.42004
# [1] -33.35365
# [1] 0.03
# [1] "Values "
# beta0    beta1    beta2    sigma      rho      tau
# 2435 -35.34043 30.45381 4.451984 3.383025 2.184668 34.45207
# [1] 50.52982
# [1] -17.24395
# [1] -17.4795
# [1] 0.0712
# [1] "Values "
# beta0     beta1    beta2      sigma      rho      tau
# 17 -0.4543259 -17.82575 5.938735 0.01740521 16.95834 5.758898
# [1] 28.9876
# [1] -6.291602
# [1] -6.331748
# [1] 0.0576
# [1] "Values "
# beta0    beta1     beta2       sigma      rho      tau
# 68 -4.976582 -37.9112 -28.11913 0.009774505 5.477082 29.90393
# [1] 51.36561
# [1] -40.06897
# [1] -40.15717
# [1] 0.0416
# [1] "Values "
# beta0    beta1    beta2    sigma     rho      tau
# 175 25.74681 12.36903 42.57079 8.591405 1.25105 4.689888
# [1] 152.2802
# [1] 55.78202
# [1] 55.42095
# [1] 0.9632
# [1] "Values "
# beta0     beta1    beta2    sigma       rho      tau
# 177 -46.73519 -13.44844 41.36604 5.412722 0.7573849 9.113723
# [1] 49.56066
# [1] -30.61146
# [1] -31.03498
# [1] 0.0448
# [1] "Values "
# beta0     beta1      beta2      sigma      rho      tau
# 258 -21.25064 0.7387042 -0.5208856 0.06081944 38.70503 12.85407
# [1] -19.32275
# [1] -21.16166
# [1] -21.1552
# [1] 0
# [1] "Values "
# beta0    beta1    beta2      sigma      rho      tau
# 393 -48.22125 1.381224 27.92561 0.04347864 1.655424 7.289727
# [1] 23.23659
# [1] -31.9717
# [1] -32.16456
# [1] 0.002
# [1] "Values "
# beta0    beta1    beta2    sigma      rho      tau
# 401 -3.483557 -41.9509 20.81174 2.968854 1.453582 6.635318
# [1] 74.67946
# [1] -13.41678
# [1] -13.43094
# [1] 0.182
# [1] "Values "
# beta0    beta1     beta2      sigma      rho      tau
# 416 -33.60112 13.64437 -71.58998 0.01645032 1.167431 17.94028
# [1] 68.67314
# [1] -66.44459
# [1] -65.33842
# [1] 0.0252
# [1] "Values "
# beta0    beta1    beta2    sigma       rho      tau
# 598 51.02533 19.26798 1.926083 1.281281 0.5238544 29.43606
# [1] 104.2875
# [1] 61.9792
# [1] 61.89306
# [1] 1
# [1] "Values "
# beta0     beta1     beta2      sigma      rho      tau
# 635 -8.996034 -22.03513 -11.72394 0.05177956 130.3388 6.890232
# [1] 19.27211
# [1] -26.82181
# [1] -26.76496
# [1] 0.0056
# [1] "Values "
# beta0     beta1     beta2    sigma       rho      tau
# 691 -30.45241 -21.47266 -44.95937 2.212358 0.9136189 14.60499
# [1] 29.61183
# [1] -66.48589
# [1] -66.39316
# [1] 0.004
# [1] "Values "
# beta0   beta1    beta2    sigma       rho      tau
# 727 4.782231 24.2781 4.595415 1.598673 0.5840254 21.19355
# [1] 74.49286
# [1] 19.79119
# [1] 19.55557
# [1] 0.738
# [1] "Values "
# beta0    beta1     beta2    sigma       rho      tau
# 742 -24.85781 22.54052 -80.41855 1.992393 0.6602199 19.48232
# [1] 94.60629
# [1] -58.05199
# [1] -56.82236
# [1] 0.0612
# [1] "Values "
# beta0     beta1     beta2      sigma       rho      tau
# 819 -29.30178 -44.84966 -19.45027 0.01549675 0.4241285 6.264043
# [1] 27.75271
# [1] -63.12831
# [1] -62.77804
# [1] 0.0052
# [1] "Values "
# beta0    beta1    beta2    sigma       rho      tau
# 873 -14.79634 15.69113 3.468993 1.488703 0.7462092 21.85471
# [1] 30.85619
# [1] -4.818262
# [1] -5.03618
# [1] 0.0672
# [1] "Values "
# beta0     beta1    beta2      sigma      rho      tau
# 1057 41.1001 -10.05008 4.614175 0.02358885 221.4634 27.40504
# [1] 59.34293
# [1] 38.5136
# [1] 38.4712
# [1] 1
# [1] "Values "
# beta0    beta1     beta2      sigma     rho      tau
# 1060 18.58406 -38.2873 -25.07086 0.03875862 8.21266 7.036516
# [1] 72.25887
# [1] -15.0048
# [1] -15.10775
# [1] 0.19
# [1] "Values "
# beta0    beta1     beta2      sigma      rho      tau
# 1240 -49.63166 14.12624 -20.61821 0.08052078 48.51555 61.52351
# [1] -9.63928
# [1] -53.86118
# [1] -53.63052
# [1] 0
# [1] "Values "
# beta0    beta1    beta2       sigma       rho      tau
# 1255 6.059825 17.91788 21.37612 0.007277612 0.9158425 14.43239
# [1] 89.67218
# [1] 27.14561
# [1] 27.02178
# [1] 0.8424
# [1] "Values "
# beta0    beta1    beta2    sigma       rho      tau
# 1534 29.0074 35.25177 11.81522 2.370344 0.8875787 34.49646
# [1] 136.3907
# [1] 53.66239
# [1] 53.39655
# [1] 0.9772
# [1] "Values "
# beta0     beta1    beta2      sigma      rho      tau
# 1733 49.72369 -17.03386 5.958378 0.03206995 18.09918 10.18158
# [1] 78.22572
# [1] 44.30349
# [1] 44.27158
# [1] 0.9996
# [1] "Values "
# beta0     beta1   beta2    sigma       rho      tau
# 1765 -19.8068 -68.75529 -50.197 3.688001 0.9364694 34.76304
# [1] 81.59944
# [1] -83.00498
# [1] -83.23171
# [1] 0.036
# [1] "Values "
# beta0     beta1     beta2    sigma     rho      tau
# 1799 -5.594372 -55.12779 -25.00036 2.219589 1.19073 16.76724
# [1] 64.5719
# [1] -47.78098
# [1] -47.31096
# [1] 0.06
# [1] "Values "
# beta0    beta1     beta2      sigma      rho      tau
# 1833 -8.000305 6.233839 -3.094715 0.05376547 1.253801 7.758198
# [1] 7.887422
# [1] -6.525425
# [1] -6.522756
# [1] 0
# [1] "Values "
# beta0   beta1     beta2      sigma      rho      tau
# 1983 -34.07532 8.29455 -15.48568 0.02420779 12.63779 6.245994
# [1] -8.561878
# [1] -38.43938
# [1] -38.2303
# [1] 0
# [1] "Values "
# beta0      beta1     beta2    sigma       rho    tau
# 2190 16.68898 -0.9605723 -50.37655 1.914113 0.7896841 18.421
# [1] 81.95558
# [1] -11.83923
# [1] -11.56727
# [1] 0.2196
# [1] "Values "
# beta0     beta1    beta2     sigma      rho      tau
# 2235 -28.51639 -38.53514 23.62712 0.8898284 313.7091 26.98016
# [1] 49.13769
# [1] -35.13096
# [1] -35.41745
# [1] 0.0332
# [1] "Values "
# beta0      beta1     beta2       sigma      rho      tau
# 2291 -21.55799 -0.6290255 -16.99816 0.005167183 1.632818 37.09222
# [1] 0.2709207
# [1] -31.34041
# [1] -31.19886
# [1] 0
# [1] "Values "
# beta0    beta1     beta2   sigma       rho      tau
# 2301 21.30691 15.52889 -14.62133 2.36975 0.9746516 11.88573
# [1] 62.9113
# [1] 21.13416
# [1] 21.21294
# [1] 0.838
# [1] "Values "
# beta0     beta1     beta2      sigma      rho      tau
# 2310 -14.36681 -10.37166 -15.17013 0.02256269 1.083724 20.64678
# [1] 8.297753
# [1] -28.12861
# [1] -28.09454
# [1] 0
# [1] "Values "
# beta0     beta1    beta2    sigma     rho      tau
# 2315 -9.310035 -30.72093 -23.3998 2.684447 1.54623 10.32019
# [1] 36.94141
# [1] -38.08835
# [1] -38.10241
# [1] 0.0192
# [1] "Values "
# beta0    beta1     beta2   sigma       rho      tau
# 2339 -7.972593 -37.8068 -17.86093 1.55667 0.6119787 18.65454
# [1] 40.17166
# [1] -37.30255
# [1] -37.07514
# [1] 0.03
# [1] "Values "
# beta0    beta1     beta2    sigma      rho      tau
# 2362 -33.81956 38.48792 -21.78816 2.231463 1.808792 5.923805
# [1] 65.05807
# [1] -26.2058
# [1] -26.11293
# [1] 0.0636
# [1] "Values "
# beta0    beta1    beta2     sigma      rho      tau
# 2369 16.76127 4.071885 51.95688 0.1303388 325.5266 19.99059
# [1] 152.2877
# [1] 47.76467
# [1] 47.66185
# [1] 0.9056
# [1] "Values "
# beta0     beta1    beta2      sigma      rho      tau
# 2406 23.52275 -13.06092 80.21536 0.01210746 1.356601 9.986399
# [1] 205.7644
# [1] 61.46565
# [1] 60.01558
# [1] 0.8816
# [1] "Values "
# beta0     beta1    beta2      sigma      rho      tau
# 2463 10.29965 -39.46909 50.08924 0.01974843 2.435397 6.623022
# [1] 134.9013
# [1] 17.93279
# [1] 17.5936
# [1] 0.5868
# [1] "Values "
# beta0    beta1    beta2      sigma      rho      tau
# 2552 37.9782 3.756248 44.59144 0.03247457 55.25257 13.04628
# [1] 154.7409
# [1] 64.7207
# [1] 64.5692
# [1] 0.9856
# [1] "Values "
# beta0    beta1   beta2     sigma      rho      tau
# 55 7.981621 18.91801 39.8252 0.0280797 3.537795 24.60653
# [1] 138.9529
# [1] 39.84763
# [1] 39.81452
# [1] 0.8688
# [1] "Values "
# beta0     beta1    beta2     sigma      rho     tau
# 92 -3.52281 -26.36235 26.28139 0.4457492 508.6488 24.3926
# [1] 63.80505
# [1] -2.417161
# [1] -2.651931
# [1] 0.258
# [1] "Values "
# beta0     beta1     beta2     sigma      rho      tau
# 97 -35.56116 -75.29439 -15.03061 0.0174446 1.308431 9.815538
# [1] 59.6313
# [1] -82.54226
# [1] -81.58648
# [1] 0.0192
# [1] "Values "
# beta0     beta1    beta2     sigma      rho      tau
# 114 2.730389 -10.14174 43.54341 0.0517141 62.44578 33.44976
# [1] 102.7098
# [1] 21.76177
# [1] 20.88562
# [1] 0.6832
# [1] "Values "
# beta0   beta1    beta2     sigma      rho     tau
# 369 71.69202 24.5522 41.73244 0.7293049 104.4746 75.1481
# [1] 217.0439
# [1] 107.5091
# [1] 107.2855
# [1] 1
# [1] "Values "
# beta0    beta1     beta2      sigma       rho      tau
# 417 42.52465 20.38749 -51.44565 0.03066389 0.5880227 11.54176
# [1] 122.7355
# [1] 24.35025
# [1] 24.79453
# [1] 0.6892
# [1] "Values "
# beta0    beta1   beta2     sigma      rho     tau
# 427 -27.63688 17.20558 24.5504 0.1581088 29.49302 33.8189
# [1] 62.60729
# [1] -5.149857
# [1] -5.210563
# [1] 0.1992
# [1] "Values "
# beta0     beta1     beta2       sigma      rho      tau
# 431 -1.731046 -61.27079 -27.00126 0.007562499 1.292102 2.098873
# [1] 76.22764
# [1] -48.18176
# [1] -47.6901
# [1] 0.0748
# [1] "Values "
# beta0     beta1     beta2     sigma      rho      tau
# 467 3.621423 -18.15091 -36.06353 0.1237685 26.44184 25.17052
# [1] 52.51201
# [1] -25.75772
# [1] -25.64993
# [1] 0.0724
# [1] "Values "
# beta0     beta1     beta2      sigma      rho      tau
# 504 5.01583 -11.12026 -6.066929 0.02900569 17.61622 24.11532
# [1] 19.4274
# [1] -4.063789
# [1] -4.028761
# [1] 0.0344
# [1] "Values "
# beta0    beta1     beta2      sigma      rho      tau
# 530 -1.8739 16.55739 -25.20939 0.05337443 4.137435 51.34925
# [1] 45.30704
# [1] -7.411638
# [1] -7.118525
# [1] 0.126
# [1] "Values "
# beta0    beta1    beta2    sigma      rho      tau
# 536 -35.47465 4.929058 1.212237 0.540578 108.0283 32.33566
# [1] -20.98697
# [1] -32.27205
# [1] -32.33401
# [1] 0
# [1] "Values "
# beta0    beta1     beta2     sigma     rho      tau
# 576 -57.1945 -35.0215 -46.94389 0.2742631 362.587 11.62872
# [1] 15.1853
# [1] -101.2812
# [1] -101.0797
# [1] 4e-04
# [1] "Values "
# beta0     beta1     beta2      sigma      rho      tau
# 612 -45.72009 -19.01185 -16.53426 0.03728186 4.471954 5.798423
# [1] -15.10621
# [1] -64.67235
# [1] -64.65525
# [1] 0
# [1] "Values "
# beta0     beta1     beta2      sigma       rho      tau
# 636 53.48986 -4.252513 -35.69279 0.05113132 0.3574896 10.24709
# [1] 97.46601
# [1] 31.44512
# [1] 31.60653
# [1] 0.8472
# [1] "Values "
# beta0     beta1    beta2     sigma      rho      tau
# 638 23.05429 -18.31261 -9.48945 0.9850147 134.2326 32.41131
# [1] 46.4355
# [1] 8.381219
# [1] 8.430783
# [1] 0.454
# [1] "Values "
# beta0    beta1     beta2       sigma       rho      tau
# 743 2.456571 24.00262 -25.05527 0.007169725 0.8266276 16.92023
# [1] 67.48353
# [1] 0.8230453
# [1] 1.025742
# [1] 0.3132
# [1] "Values "
# beta0    beta1      beta2     sigma      rho      tau
# 753 -61.39135 9.036211 0.04006848 0.1337367 8.938801 44.00763
# [1] -37.44726
# [1] -56.73464
# [1] -56.76856
# [1] 0
# [1] "Values "
# beta0     beta1     beta2     sigma      rho      tau
# 820 57.3164 -21.86264 -38.23215 0.4561252 167.5464 34.63392
# [1] 111.1565
# [1] 24.82675
# [1] 25.11831
# [1] 0.7068
# [1] "Values "
# beta0     beta1     beta2      sigma      rho      tau
# 954 -15.66765 -53.67217 -81.98182 0.07090873 9.074851 20.05009
# [1] 104.9915
# [1] -88.81905
# [1] -88.76778
# [1] 0.0436
# [1] "Values "
# beta0     beta1     beta2      sigma      rho      tau
# 1093 26.24023 -50.89356 -9.495277 0.01513274 6.497101 24.07857
# [1] 90.56114
# [1] -5.145882
# [1] -4.645182
# [1] 0.3192
# [1] "Values "
# beta0     beta1    beta2     sigma      rho      tau
# 1108 -9.031833 -21.31479 57.39713 0.0135784 1.916367 17.15069
# [1] 125.4986
# [1] 11.97909
# [1] 11.41811
# [1] 0.5168
# [1] "Values "
# beta0     beta1     beta2     sigma      rho      tau
# 1213 7.106231 -73.31991 -8.824255 0.2993456 33.76142 48.67234
# [1] 99.60728
# [1] -35.40825
# [1] -34.96616
# [1] 0.1528
# [1] "Values "
# beta0    beta1     beta2     sigma      rho      tau
# 1258 29.74924 18.16535 -43.51531 0.1950881 31.76477 29.31147
# [1] 98.17971
# [1] 14.84857
# [1] 15.37768
# [1] 0.584
# [1] "Values "
# beta0    beta1     beta2     sigma      rho      tau
# 1330 15.93757 21.98461 -13.52256 0.0406769 4.138583 25.50259
# [1] 72.73312
# [1] 19.68726
# [1] 19.86035
# [1] 0.7624
# [1] "Values "
# beta0    beta1    beta2      sigma      rho      tau
# 1339 38.22292 11.49935 10.28826 0.03977032 4.916985 46.55094
# [1] 83.39646
# [1] 49.84622
# [1] 49.84109
# [1] 1
# [1] "Values "
# beta0   beta1    beta2      sigma      rho      tau
# 1341 -71.19806 55.5471 8.999549 0.02198662 2.094477 29.53911
# [1] 86.48119
# [1] -37.70117
# [1] -38.0842
# [1] 0.0796
# [1] "Values "
# beta0     beta1     beta2      sigma      rho      tau
# 1470 -2.766544 -26.26287 -12.30614 0.03306761 6.353986 12.32069
# [1] 30.67394
# [1] -23.0846
# [1] -22.90588
# [1] 0.0284
# [1] "Values "
# beta0    beta1    beta2        sigma      rho     tau
# 1481 -0.1947145 47.47039 3.756515 0.0004324033 2.773106 17.0618
# [1] 129.842
# [1] 26.24202
# [1] 25.85002
# [1] 0.7096
# [1] "Values "
# beta0     beta1     beta2      sigma      rho      tau
# 1577 67.07833 -7.357817 -7.212941 0.05848018 7.609272 7.092982
# [1] 79.71557
# [1] 59.29057
# [1] 59.28647
# [1] 1
# [1] "Values "
# beta0     beta1    beta2     sigma      rho     tau
# 1663 34.41893 -17.73907 20.26874 0.5766063 52.29775 17.1417
# [1] 85.44922
# [1] 36.60105
# [1] 36.62925
# [1] 0.9708
# [1] "Values "
# beta0     beta1    beta2     sigma      rho      tau
# 1757 40.51303 -25.53618 12.56889 0.5027639 172.1454 19.00024
# [1] 87.96286
# [1] 34.41112
# [1] 34.40291
# [1] 0.9412
# [1] "Values "
# beta0     beta1     beta2      sigma      rho      tau
# 1824 1.926227 -1.460538 -35.66696 0.02157559 1.326942 10.73547
# [1] 47.64022
# [1] -18.67222
# [1] -18.38495
# [1] 0.07
# [1] "Values "
# beta0     beta1   beta2      sigma      rho      tau
# 1973 30.04955 -25.03224 -46.083 0.04068448 3.859685 17.88831
# [1] 93.93335
# [1] -8.434853
# [1] -8.14145
# [1] 0.2828
# [1] "Values "
# beta0    beta1     beta2     sigma      rho      tau
# 2032 8.492088 10.19637 -11.97263 0.1912894 93.15352 38.49116
# [1] 36.50546
# [1] 7.058497
# [1] 7.079828
# [1] 0.37
# [1] "Values "
# beta0     beta1    beta2      sigma       rho      tau
# 2054 -15.30216 -42.65573 24.74004 0.01979773 0.6318304 29.70663
# [1] 68.82029
# [1] -23.4107
# [1] -23.51616
# [1] 0.1108
# [1] "Values "
# beta0     beta1    beta2      sigma      rho      tau
# 2087 -14.9658 -40.33459 1.286101 0.03650172 1.290854 37.56283
# [1] 39.82181
# [1] -34.93647
# [1] -34.64751
# [1] 0.028
# [1] "Values "
# beta0     beta1     beta2      sigma      rho      tau
# 2164 18.22205 -2.389226 -10.15073 0.06471307 22.05319 18.56651
# [1] 29.97984
# [1] 11.3476
# [1] 11.40211
# [1] 0.588
# [1] "Values "
# beta0    beta1   beta2      sigma      rho      tau
# 2381 -5.117945 4.935449 21.1928 0.06921245 286.8998 20.47672
# [1] 55.77767
# [1] 9.20752
# [1] 9.113678
# [1] 0.474
# [1] "Values "
# beta0      beta1     beta2      sigma      rho      tau
# 2401 59.34885 -0.8901199 -9.530119 0.09434715 11.21527 36.01215
# [1] 71.24633
# [1] 53.58863
# [1] 53.61835
# [1] 1
# [1] "Values "
# beta0     beta1     beta2     sigma      rho      tau
# 2494 -59.97251 -62.60306 -19.29313 0.5966114 52.18046 21.85287
# [1] 19.40283
# [1] -102.8168
# [1] -102.3167
# [1] 8e-04
# [1] "Values "
# beta0    beta1     beta2      sigma      rho     tau
# 2535 -29.77157 32.15999 -48.17937 0.06162392 24.03692 5.45319
# [1] 61.63919
# [1] -40.09045
# [1] -39.58616
# [1] 0.0488


# sessionInfo() -----------------------------------------------------------

sessionInfo()
