
# Mean Field for Parameter Values that Produced Errors --------------------

# We want to consider the mean field produced from the parameter values that resulted in errors for some grid and mesh combinations in the SBC simulation study. We also consider the dame for those that produced more than 10 FFT warnings when inla was running.
# We produce some plots of the mean field for the error parameter values.

# Author: Nadeen Khaleel


# Setwd and Load Libraries ------------------------------------------------


# Either setwd() to the source file location, or run the following:
library("rstudioapi")
setwd(dirname(getActiveDocumentContext()$path))

library(spatstat)


# SBC Simulation Study ----------------------------------------------------

# SBC: Parameters for Errors ----------------------------------------------
# What parameter values result in errors?


load("../IRREGPOLLGCP_OUTPUT/GridMeshIrregPolLGCPSBCSS.rda")

err.vec <- rep(NA,50)
ii <- sum(!is.na(err.vec))
for (j in 1:4){
  for (l in 1:4){
    if (sum(!is.na(run.out.final[[j]][[l]]$mess.ls$err))!=0){
      err.vec[(ii+1):(ii+sum(!is.na(run.out.final[[j]][[l]]$mess.ls$err)))] <- which(!is.na(run.out.final[[j]][[l]]$mess.ls$err))
      ii <- sum(!is.na(err.vec))
    }
  }
}
err.vec
err.vec.u <- unique(err.vec[!is.na(err.vec)])
err.vec.u
true.theta.err <- true.theta.final[err.vec.u,]

true.theta.err
# beta0     beta1         beta2      sigma        rho
# 974 4.901376 0.6865738 -0.6760356772 0.16881047  8.9836321
# 141 3.458145 1.1426511  0.2180417934 1.41053908  2.4073466
# 361 2.770775 0.8604464 -0.0008532766 2.64060007  0.7227725
# 251 3.563836 1.6418819 -0.1485273737 0.50045351  6.4121983
# 540 2.622071 2.0068904  0.3154202662 1.09058928 13.2516835
# 546 3.272435 1.6049552  0.1248945258 0.02355465  1.5625934
# 921 3.838233 1.0204554  0.9710081271 3.22773325  2.3116741
# 856 2.790247 1.3829485 -0.5149555332 0.71766116  0.8212983
# 479 3.324580 1.7586057 -0.3961438982 0.12103486  1.1870172
# 738 4.634244 0.4135613 -0.3944151605 0.06802721  0.4596939
# 115 3.132307 0.8972784  0.3238263825 1.66367921  0.8381427
# 196 2.815459 1.9147213 -0.3111797902 1.41680704  6.6995612
# 434 3.795388 1.2906254  0.9202399988 0.01359262  2.0121125
# 459 2.710127 1.5512663 -0.2214070031 1.69102799  1.3596002
# 472 4.163235 1.4942236  0.3698132435 1.87673846  2.4006565
# 816 4.168051 2.0151076 -0.5420550548 0.79816279  2.5783449
# 851 3.614131 2.1249227 -0.7000832531 0.87851995  2.4848514
# 732 3.670358 1.2474259 -0.0814762128 0.61725170 22.2272025
# 245 1.918399 0.8569167  0.3503282390 2.29309516  1.1288912
# 313 4.455193 0.6850343 -1.2495869882 0.06740977  0.6484623
# 914 3.188174 0.9555199 -0.8612637384 0.28588288 18.6797224
# 948 3.762640 1.5465412  0.5342759250 1.80562001  3.1251001

nrow(true.theta.err)


# SBC: Mean Field for Errors ----------------------------------------------
# What does the mean field look like for the runs which resulted in errors.

load("../IRREGPOLLGCP_CODE/GridMeshIrregPolLGCPSSCov.rda")

max.mu <- rep(NA,nrow(true.theta.err))

for (k in 1:nrow(true.theta.err)){
  theta <- true.theta.err[k,]
  beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
  mu <- beta.0*intb.im + beta.1*popb.im + beta.2*incb.im
  
  print("Values ",k)
  print(theta)
  print(max(mu$v))
  max.mu[k] <- max(mu$v)
  print(exp(max(mu$v)))
  print(mean(mu$v))
  print(median(mu$v))
  print(sum(mu$v>10)/length(mu$v))
  print(mean(exp(mu$v)))
}
# 
# [1] "Values "
# beta0     beta1      beta2     sigma      rho
# 974 4.901376 0.6865738 -0.6760357 0.1688105 8.983632
# [1] 9.296193
# [1] 10896.46
# [1] 1.913078
# [1] 0
# [1] 0
# [1] 61.34051
# [1] "Values "
# beta0    beta1     beta2    sigma      rho
# 141 3.458145 1.142651 0.2180418 1.410539 2.407347
# [1] 17.14739
# [1] 27990876
# [1] 1.349764
# [1] 0
# [1] 0.00128653
# [1] 1176.492
# [1] "Values "
# beta0     beta1         beta2  sigma       rho
# 361 2.770775 0.8604464 -0.0008532766 2.6406 0.7227725
# [1] 11.90954
# [1] 148678.5
# [1] 1.081474
# [1] 0
# [1] 0.0001534394
# [1] 23.86048
# [1] "Values "
# beta0    beta1      beta2     sigma      rho
# 251 3.563836 1.641882 -0.1485274 0.5004535 6.412198
# [1] 19.96119
# [1] 466697437
# [1] 1.391016
# [1] 0
# [1] 0.002396015
# [1] 16124.8
# [1] "Values "
# beta0   beta1     beta2    sigma      rho
# 540 2.622071 2.00689 0.3154203 1.090589 13.25168
# [1] 26.18652
# [1] 235862845565
# [1] 1.023432
# [1] 0
# [1] 0.00443794
# [1] 4964947
# [1] "Values "
# beta0    beta1     beta2      sigma      rho
# 546 3.272435 1.604955 0.1248945 0.02355465 1.562593
# [1] 21.21495
# [1] 1635065683
# [1] 1.277278
# [1] 0
# [1] 0.002667485
# [1] 44247.45
# [1] "Values "
# beta0    beta1     beta2    sigma      rho
# 921 3.838233 1.020455 0.9710081 3.227733 2.311674
# [1] 22.7383
# [1] 7500949433
# [1] 1.498118
# [1] 0
# [1] 0.004426137
# [1] 181503.6
# [1] "Values "
# beta0    beta1      beta2     sigma       rho
# 856 2.790247 1.382949 -0.5149555 0.7176612 0.8212983
# [1] 13.95231
# [1] 1146602
# [1] 1.089074
# [1] 0
# [1] 0.0006019546
# [1] 138.397
# [1] "Values "
# beta0    beta1      beta2     sigma      rho
# 479 3.32458 1.758606 -0.3961439 0.1210349 1.187017
# [1] 19.20773
# [1] 219690596
# [1] 1.297631
# [1] 0
# [1] 0.001900288
# [1] 10461.88
# [1] "Values "
# beta0     beta1      beta2      sigma       rho
# 738 4.634244 0.4135613 -0.3944152 0.06802721 0.4596939
# [1] 7.316395
# [1] 1504.769
# [1] 1.808812
# [1] 0
# [1] 0
# [1] 42.8192
# [1] "Values "
# beta0     beta1     beta2    sigma       rho
# 115 3.132307 0.8972784 0.3238264 1.663679 0.8381427
# [1] 15.00146
# [1] 3273791
# [1] 1.222584
# [1] 0
# [1] 0.0007317879
# [1] 193.1355
# [1] "Values "
# beta0    beta1      beta2    sigma      rho
# 196 2.815459 1.914721 -0.3111798 1.416807 6.699561
# [1] 20.95991
# [1] 1266985739
# [1] 1.098914
# [1] 0
# [1] 0.002337
# [1] 42365.49
# [1] "Values "
# beta0    beta1   beta2      sigma      rho
# 434 3.795388 1.290625 0.92024 0.01359262 2.012113
# [1] 24.94046
# [1] 67842794718
# [1] 1.481395
# [1] 0
# [1] 0.005795288
# [1] 1407262
# [1] "Values "
# beta0    beta1     beta2    sigma    rho
# 459 2.710127 1.551266 -0.221407 1.691028 1.3596
# [1] 17.62796
# [1] 45261040
# [1] 1.057802
# [1] 0
# [1] 0.001274727
# [1] 1990.299
# [1] "Values "
# beta0    beta1     beta2    sigma      rho
# 472 4.163235 1.494224 0.3698132 1.876738 2.400657
# [1] 22.66452
# [1] 6967495276
# [1] 1.624971
# [1] 0
# [1] 0.004496955
# [1] 195050.1
# [1] "Values "
# beta0    beta1      beta2     sigma      rho
# 816 4.168051 2.015108 -0.5420551 0.7981628 2.578345
# [1] 21.7433
# [1] 2773277842
# [1] 1.62685
# [1] 0
# [1] 0.003765167
# [1] 131952.7
# [1] "Values "
# beta0    beta1      beta2   sigma      rho
# 851 3.614131 2.124923 -0.7000833 0.87852 2.484851
# [1] 21.23662
# [1] 1670880257
# [1] 1.410648
# [1] 0
# [1] 0.002938955
# [1] 102560.5
# [1] "Values "
# beta0    beta1       beta2     sigma     rho
# 732 3.670358 1.247426 -0.08147621 0.6172517 22.2272
# [1] 16.3506
# [1] 12617574
# [1] 1.432594
# [1] 0
# [1] 0.001133091
# [1] 693.4421
# [1] "Values "
# beta0     beta1     beta2    sigma      rho
# 245 1.918399 0.8569167 0.3503282 2.293095 1.128891
# [1] 13.6217
# [1] 823810.7
# [1] 0.7487788
# [1] 0
# [1] 0.0002950758
# [1] 49.01974
# [1] "Values "
# beta0     beta1     beta2      sigma       rho
# 313 4.455193 0.6850343 -1.249587 0.06740977 0.6484623
# [1] 7.271565
# [1] 1438.801
# [1] 1.738926
# [1] 0
# [1] 0
# [1] 43.73852
# [1] "Values "
# beta0     beta1      beta2     sigma      rho
# 914 3.188174 0.9555199 -0.8612637 0.2858829 18.67972
# [1] 9.521634
# [1] 13651.9
# [1] 1.24439
# [1] 0
# [1] 0
# [1] 14.05414
# [1] "Values "
# beta0    beta1     beta2   sigma    rho
# 948 3.76264 1.546541 0.5342759 1.80562 3.1251
# [1] 24.0063
# [1] 26656611932
# [1] 1.468613
# [1] 0
# [1] 0.005075303
# [1] 751167

min(max.mu)
# [1] 7.271565
max(max.mu)
# [1] 26.18652
mean(max.mu)
# [1] 17.44893
hist(max.mu,breaks=25)

sum.mu <- rep(NA,nrow(true.theta.err))
sum.expmu <- rep(NA,nrow(true.theta.err))
for (k in 1:nrow(true.theta.err)){
  theta <- true.theta.err[k,]
  beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
  mu <- beta.0*intb.im + beta.1*popb.im + beta.2*incb.im
  
  print(theta)
  print(sum(mu$v))
  sum.mu[k] <- sum(mu$v)
  sum.expmu <- sum(exp(mu$v))
  print(sum(exp(mu$v)))
} # includes outside W LA window
# beta0     beta1      beta2     sigma      rho
# 974 4.901376 0.6865738 -0.6760357 0.1688105 8.983632
# [1] 162083.6
# [1] 5197014
# beta0    beta1     beta2    sigma      rho
# 141 3.458145 1.142651 0.2180418 1.410539 2.407347
# [1] 114357.4
# [1] 99677121
# beta0     beta1         beta2  sigma       rho
# 361 2.770775 0.8604464 -0.0008532766 2.6406 0.7227725
# [1] 91626.76
# [1] 2021556
# beta0    beta1      beta2     sigma      rho
# 251 3.563836 1.641882 -0.1485274 0.5004535 6.412198
# [1] 117852.5
# [1] 1366157837
# beta0   beta1     beta2    sigma      rho
# 540 2.622071 2.00689 0.3154203 1.090589 13.25168
# [1] 86709.28
# [1] 420650175447
# beta0    beta1     beta2      sigma      rho
# 546 3.272435 1.604955 0.1248945 0.02355465 1.562593
# [1] 108216.1
# [1] 3748821259
# beta0    beta1     beta2    sigma      rho
# 921 3.838233 1.020455 0.9710081 3.227733 2.311674
# [1] 126926.5
# [1] 15377710464
# beta0    beta1      beta2     sigma       rho
# 856 2.790247 1.382949 -0.5149555 0.7176612 0.8212983
# [1] 92270.68
# [1] 11725547
# beta0    beta1      beta2     sigma      rho
# 479 3.32458 1.758606 -0.3961439 0.1210349 1.187017
# [1] 109940.5
# [1] 886372744
# beta0     beta1      beta2      sigma       rho
# 738 4.634244 0.4135613 -0.3944152 0.06802721 0.4596939
# [1] 153249.8
# [1] 3627814
# beta0     beta1     beta2    sigma       rho
# 115 3.132307 0.8972784 0.3238264 1.663679 0.8381427
# [1] 103582.2
# [1] 16363211
# beta0    beta1      beta2    sigma      rho
# 196 2.815459 1.914721 -0.3111798 1.416807 6.699561
# [1] 93104.42
# [1] 3589373419
# beta0    beta1   beta2      sigma      rho
# 434 3.795388 1.290625 0.92024 0.01359262 2.012113
# [1] 125509.7
# [1] 119228848551
# beta0    beta1     beta2    sigma    rho
# 459 2.710127 1.551266 -0.221407 1.691028 1.3596
# [1] 89621.2
# [1] 168626100
# beta0    beta1     beta2    sigma      rho
# 472 4.163235 1.494224 0.3698132 1.876738 2.400657
# [1] 137674
# [1] 16525426101
# beta0    beta1      beta2     sigma      rho
# 816 4.168051 2.015108 -0.5420551 0.7981628 2.578345
# [1] 137833.3
# [1] 11179560728
# beta0    beta1      beta2   sigma      rho
# 851 3.614131 2.124923 -0.7000833 0.87852 2.484851
# [1] 119515.7
# [1] 8689335035
# beta0    beta1       beta2     sigma     rho
# 732 3.670358 1.247426 -0.08147621 0.6172517 22.2272
# [1] 121375.1
# [1] 58751184
# beta0     beta1     beta2    sigma      rho
# 245 1.918399 0.8569167 0.3503282 2.293095 1.128891
# [1] 63439.54
# [1] 4153148
# beta0     beta1     beta2      sigma       rho
# 313 4.455193 0.6850343 -1.249587 0.06740977 0.6484623
# [1] 147328.8
# [1] 3705703
# beta0     beta1      beta2     sigma      rho
# 914 3.188174 0.9555199 -0.8612637 0.2858829 18.67972
# [1] 105429.7
# [1] 1190723
# beta0    beta1     beta2   sigma    rho
# 948 3.76264 1.546541 0.5342759 1.80562 3.1251
# [1] 124426.8
# [1] 63641871423

min(sum.mu)
# [1] 63439.54
min(sum.expmu)
# [1] 63641871423
max(sum.mu)
# [1] 162083.6
max(sum.expmu)
# [1] 63641871423

mean(sum.mu)
# [1] 115094.3
mean(sum.expmu)
# [1] 63641871423


pdf("IrregPolMeanFieldandExpMeanFieldforErr.pdf",height = 5, width=12.5)
for (k in 1:nrow(true.theta.err)){
  theta <- true.theta.err[k,]
  beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
  mu <- beta.0*intb.im + beta.1*popb.im + beta.2*incb.im
  
  par(mfrow=c(1,2))
  plot(mu,main=paste0("mu for ",k))
  plot(exp(mu),paste0("exp(mu) for ",k))
}
dev.off()

# SBC: Parameters for TIME ERROR Warnings ---------------------------------
# What parameter values result in time errors, but were re-run successfully?

load("../IRREGPOLLGCP_OUTPUT/GridMeshIrregPolLGCPSBCSS.rda")

err.vec <- rep(NA,50)
ii <- sum(!is.na(err.vec))
for (j in 1:4){
  for (l in 1:4){
    if (sum(!is.na(run.out.final[[j]][[l]]$mess.ls$warning))!=0){
      err.vec[(ii+1):(ii+sum(!is.na(run.out.final[[j]][[l]]$mess.ls$warning)))] <- which(!is.na(run.out.final[[j]][[l]]$mess.ls$warning))
      ii <- sum(!is.na(err.vec))
    }
  }
}
err.vec
# [1] 181 196 254 349 400 655 948 983 569 112 121 211 395 655  41 116 163 712 738 780 155 208 341 395 434 478 569 655 816 143 341 395 412 434 569 655
# [37] 732 816 851  NA  NA  NA  NA  NA  NA  NA  NA  NA  NA  NA
err.vec.u <- unique(err.vec[!is.na(err.vec)])
err.vec.u
# [1] 181 196 254 349 400 655 948 983 569 112 121 211 395  41 116 163 712 738 780 155 208 341 434 478 816 143 412 732 851
true.theta.err <- true.theta.final[err.vec.u,]

true.theta.err
# beta0      beta1       beta2      sigma        rho
# 181 1.057061  1.7593337  0.10300560 0.05327072  6.4613684
# 196 2.815459  1.9147213 -0.31117979 1.41680704  6.6995612
# 254 2.977315  1.6438785  0.07851470 1.61666957  9.1478419
# 349 3.311370  1.6370538  0.02118976 0.07314742  0.9364609
# 400 3.528875  1.7471088 -0.34341657 0.11854588  1.5843971
# 655 1.979246  1.6750574  0.89225712 0.17289646  1.7308649
# 948 3.762640  1.5465412  0.53427592 1.80562001  3.1251001
# 983 3.202194  1.6726778 -0.21342359 0.78313307 11.1846101
# 569 1.636562  1.8974675  0.43878962 1.36929494  1.0587260
# 112 3.856830  2.3516583 -0.19206268 0.48931492  5.4973838
# 121 5.231728  1.2136411  0.43286741 0.03430835 24.9241405
# 211 4.000165  1.6701135  0.37788538 0.03113893  4.7165546
# 395 3.640675  1.5305456 -0.06851556 2.48742528  4.6413853
# 41  4.551687  1.5820339 -0.40277300 0.79714099  4.8151733
# 116 2.548651 -0.1852790 -0.35308802 4.33914141  0.3925233
# 163 2.435150  0.8247821 -0.38308599 3.27009450  1.1708859
# 712 5.089383  1.4457264 -0.21503061 0.14607542  1.8149148
# 738 4.634244  0.4135613 -0.39441516 0.06802721  0.4596939
# 780 2.621741  0.9835234 -0.35103265 1.85236813  0.5095075
# 155 4.033903  1.5302857  0.25825558 0.80088808  1.2651081
# 208 3.085795  1.8433148 -0.22116594 0.23583019  8.4888739
# 341 3.807896  1.5596690 -0.39829260 0.83128539  4.3270474
# 434 3.795388  1.2906254  0.92024000 0.01359262  2.0121125
# 478 3.118912  1.4770446  0.58850220 2.15332872 53.0049222
# 816 4.168051  2.0151076 -0.54205505 0.79816279  2.5783449
# 143 1.914680  1.6540858 -0.34772182 1.99110418  0.6363664
# 412 2.493601  2.0582779  0.15098600 0.24907350  3.0057429
# 732 3.670358  1.2474259 -0.08147621 0.61725170 22.2272025
# 851 3.614131  2.1249227 -0.70008325 0.87851995  2.4848514

nrow(true.theta.err)
# [1] 29


# SBC: Mean Field for TIME ERROR Warnings ---------------------------------
# What does the mean field look like for the runs which resulted in errors.

load("../IRREGPOLLGCP_CODE/GridMeshIrregPolLGCPSSCov.rda")

max.mu <- rep(NA,nrow(true.theta.err))

for (k in 1:nrow(true.theta.err)){
  theta <- true.theta.err[k,]
  beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
  mu <- beta.0*intb.im + beta.1*popb.im + beta.2*incb.im
  
  print(paste0("Values ",k))
  print(theta)
  print(max(mu$v))
  max.mu[k] <- max(mu$v)
  print(exp(max(mu$v)))
  print(mean(mu$v))
  print(median(mu$v))
  print(sum(mu$v>10)/length(mu$v))
  print(mean(exp(mu$v)))
}
# [1] "Values 1"
# beta0    beta1     beta2      sigma      rho
# 181 1.057061 1.759334 0.1030056 0.05327072 6.461368
# [1] 20.48519
# [1] 788144553
# [1] 0.4125862
# [1] 0
# [1] 0.001463576
# [1] 19269.07
# [1] "Values 2"
# beta0    beta1      beta2    sigma      rho
# 196 2.815459 1.914721 -0.3111798 1.416807 6.699561
# [1] 20.95991
# [1] 1266985739
# [1] 1.098914
# [1] 0
# [1] 0.002337
# [1] 42365.49
# [1] "Values 3"
# beta0    beta1     beta2   sigma      rho
# 254 2.977315 1.643878 0.0785147 1.61667 9.147842
# [1] 21.00483
# [1] 1325205521
# [1] 1.162089
# [1] 0
# [1] 0.002277985
# [1] 35754.44
# [1] "Values 4"
# beta0    beta1      beta2      sigma       rho
# 349 3.31137 1.637054 0.02118976 0.07314742 0.9364609
# [1] 20.86012
# [1] 1146661135
# [1] 1.292476
# [1] 0
# [1] 0.002549455
# [1] 32519.45
# [1] "Values 5"
# beta0    beta1      beta2     sigma      rho
# 400 3.528875 1.747109 -0.3434166 0.1185459 1.584397
# [1] 19.66349
# [1] 346533846
# [1] 1.377371
# [1] 0
# [1] 0.002266182
# [1] 14955.8
# [1] "Values 6"
# beta0    beta1     beta2     sigma      rho
# 655 1.979246 1.675057 0.8922571 0.1728965 1.730865
# [1] 26.71542
# [1] 400274082870
# [1] 0.7725282
# [1] 0
# [1] 0.004331712
# [1] 8036233
# [1] "Values 7"
# beta0    beta1     beta2   sigma    rho
# 948 3.76264 1.546541 0.5342759 1.80562 3.1251
# [1] 24.0063
# [1] 26656611932
# [1] 1.468613
# [1] 0
# [1] 0.005075303
# [1] 751167
# [1] "Values 8"
# beta0    beta1      beta2     sigma      rho
# 983 3.202194 1.672678 -0.2134236 0.7831331 11.18461
# [1] 19.46696
# [1] 284703344
# [1] 1.249862
# [1] 0
# [1] 0.001841273
# [1] 10538.57
# [1] "Values 9"
# beta0    beta1     beta2    sigma      rho
# 569 1.636562 1.897467 0.4387896 1.369295 1.058726
# [1] 24.91232
# [1] 65960578787
# [1] 0.6387738
# [1] 0
# [1] 0.003116
# [1] 1510456
# [1] "Values 10"
# beta0    beta1      beta2     sigma      rho
# 112 3.85683 2.351658 -0.1920627 0.4893149 5.497384
# [1] 27.48917
# [1] 867747612837
# [1] 1.505376
# [1] 0
# [1] 0.007990652
# [1] 18011631
# [1] "Values 11"
# beta0    beta1     beta2      sigma      rho
# 121 5.231728 1.213641 0.4328674 0.03430835 24.92414
# [1] 21.2397
# [1] 1676042993
# [1] 2.042019
# [1] 0
# [1] 0.005417591
# [1] 57631.44
# [1] "Values 12"
# beta0    beta1     beta2      sigma      rho
# 211 4.000165 1.670114 0.3778854 0.03113893 4.716555
# [1] 24.42801
# [1] 40639694756
# [1] 1.561322
# [1] 0
# [1] 0.005441197
# [1] 1020033
# [1] "Values 13"
# beta0    beta1       beta2    sigma      rho
# 395 3.640675 1.530546 -0.06851556 2.487425 4.641385
# [1] 19.42176
# [1] 272120858
# [1] 1.421008
# [1] 0
# [1] 0.002006515
# [1] 9515.357
# [1] "Values 14"
# beta0    beta1     beta2    sigma      rho
# 41 4.551687 1.582034 -0.402773 0.797141 4.815173
# [1] 18.51126
# [1] 109480500
# [1] 1.776589
# [1] 0
# [1] 0.00249044
# [1] 6923.063
# [1] "Values 15"
# beta0     beta1     beta2    sigma       rho
# 116 2.548651 -0.185279 -0.353088 4.339141 0.3925233
# [1] 3.124648
# [1] 22.75189
# [1] 0.9947752
# [1] 0
# [1] 0
# [1] 6.116848
# [1] "Values 16"
# beta0     beta1     beta2    sigma      rho
# 163 2.43515 0.8247821 -0.383086 3.270095 1.170886
# [1] 8.884919
# [1] 7222.232
# [1] 0.9504742
# [1] 0
# [1] 0
# [1] 7.508327
# [1] "Values 17"
# beta0    beta1      beta2     sigma      rho
# 712 5.089383 1.445726 -0.2150306 0.1460754 1.814915
# [1] 18.93072
# [1] 166536222
# [1] 1.98646
# [1] 0
# [1] 0.003009773
# [1] 8486.689
# [1] "Values 18"
# beta0     beta1      beta2      sigma       rho
# 738 4.634244 0.4135613 -0.3944152 0.06802721 0.4596939
# [1] 7.316395
# [1] 1504.769
# [1] 1.808812
# [1] 0
# [1] 0
# [1] 42.8192
# [1] "Values 19"
# beta0     beta1      beta2    sigma       rho
# 780 2.621741 0.9835234 -0.3510326 1.852368 0.5095075
# [1] 10.6014
# [1] 40191.21
# [1] 1.023303
# [1] 0
# [1] 5.901516e-05
# [1] 15.25555
# [1] "Values 20"
# beta0    beta1     beta2     sigma      rho
# 155 4.033903 1.530286 0.2582556 0.8008881 1.265108
# [1] 22.1279
# [1] 4074030949
# [1] 1.574491
# [1] 0
# [1] 0.003835985
# [1] 111280.1
# [1] "Values 21"
# beta0    beta1      beta2     sigma      rho
# 208 3.085795 1.843315 -0.2211659 0.2358302 8.488874
# [1] 21.10922
# [1] 1471018218
# [1] 1.20443
# [1] 0
# [1] 0.002596667
# [1] 45833.07
# [1] "Values 22"
# beta0    beta1      beta2     sigma      rho
# 341 3.807896 1.559669 -0.3982926 0.8312854 4.327047
# [1] 17.56152
# [1] 42351903
# [1] 1.486277
# [1] 0
# [1] 0.001581606
# [1] 2751.21
# [1] "Values 23"
# beta0    beta1   beta2      sigma      rho
# 434 3.795388 1.290625 0.92024 0.01359262 2.012113
# [1] 24.94046
# [1] 67842794718
# [1] 1.481395
# [1] 0
# [1] 0.005795288
# [1] 1407262
# [1] "Values 24"
# beta0    beta1     beta2    sigma      rho
# 478 3.118912 1.477045 0.5885022 2.153329 53.00492
# [1] 23.15402
# [1] 11367420508
# [1] 1.217356
# [1] 0
# [1] 0.003835985
# [1] 299638.4
# [1] "Values 25"
# beta0    beta1      beta2     sigma      rho
# 816 4.168051 2.015108 -0.5420551 0.7981628 2.578345
# [1] 21.7433
# [1] 2773277842
# [1] 1.62685
# [1] 0
# [1] 0.003765167
# [1] 131952.7
# [1] "Values 26"
# beta0    beta1      beta2    sigma       rho
# 143 1.91468 1.654086 -0.3477218 1.991104 0.6363664
# [1] 17.03014
# [1] 24894069
# [1] 0.7473274
# [1] 0
# [1] 0.0009678485
# [1] 1232.349
# [1] "Values 27"
# beta0    beta1    beta2     sigma      rho
# 412 2.493601 2.058278 0.150986 0.2490735 3.005743
# [1] 25.43892
# [1] 1.11682e+11
# [1] 0.9732883
# [1] 0
# [1] 0.003847788
# [1] 2286242
# [1] "Values 28"
# beta0    beta1       beta2     sigma     rho
# 732 3.670358 1.247426 -0.08147621 0.6172517 22.2272
# [1] 16.3506
# [1] 12617574
# [1] 1.432594
# [1] 0
# [1] 0.001133091
# [1] 693.4421
# [1] "Values 29"
# beta0    beta1      beta2   sigma      rho
# 851 3.614131 2.124923 -0.7000833 0.87852 2.484851
# [1] 21.23662
# [1] 1670880257
# [1] 1.410648
# [1] 0
# [1] 0.002938955
# [1] 102560.5

min(max.mu)
# [1] 3.124648
max(max.mu)
# [1] 27.48917
mean(max.mu)
# [1] 19.61087
hist(max.mu,breaks=25)

sum.mu <- rep(NA,nrow(true.theta.err))
sum.expmu <- rep(NA,nrow(true.theta.err))
for (k in 1:nrow(true.theta.err)){
  theta <- true.theta.err[k,]
  beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
  mu <- beta.0*intb.im + beta.1*popb.im + beta.2*incb.im
  
  print(theta)
  print(sum(mu$v))
  sum.mu[k] <- sum(mu$v)
  sum.expmu <- sum(exp(mu$v))
  print(sum(exp(mu$v)))
} # includes outside W LA window
# beta0    beta1     beta2      sigma      rho
# 181 1.057061 1.759334 0.1030056 0.05327072 6.461368
# [1] 34955.96
# [1] 1632552817
# beta0    beta1      beta2    sigma      rho
# 196 2.815459 1.914721 -0.3111798 1.416807 6.699561
# [1] 93104.42
# [1] 3589373419
# beta0    beta1     beta2   sigma      rho
# 254 2.977315 1.643878 0.0785147 1.61667 9.147842
# [1] 98456.84
# [1] 3029258849
# beta0    beta1      beta2      sigma       rho
# 349 3.31137 1.637054 0.02118976 0.07314742 0.9364609
# [1] 109503.7
# [1] 2755177790
# beta0    beta1      beta2     sigma      rho
# 400 3.528875 1.747109 -0.3434166 0.1185459 1.584397
# [1] 116696.4
# [1] 1267115592
# beta0    beta1     beta2     sigma      rho
# 655 1.979246 1.675057 0.8922571 0.1728965 1.730865
# [1] 65451.68
# [1] 680861767070
# beta0    beta1     beta2   sigma    rho
# 948 3.76264 1.546541 0.5342759 1.80562 3.1251
# [1] 124426.8
# [1] 63641871423
# beta0    beta1      beta2     sigma      rho
# 983 3.202194 1.672678 -0.2134236 0.7831331 11.18461
# [1] 105893.3
# [1] 892869895
# beta0    beta1     beta2    sigma      rho
# 569 1.636562 1.897467 0.4387896 1.369295 1.058726
# [1] 54119.48
# [1] 127971854321
# beta0    beta1      beta2     sigma      rho
# 112 3.85683 2.351658 -0.1920627 0.4893149 5.497384
# [1] 127541.5
# [1] 1.526017e+12
# beta0    beta1     beta2      sigma      rho
# 121 5.231728 1.213641 0.4328674 0.03430835 24.92414
# [1] 173008
# [1] 4882766148
# beta0    beta1     beta2      sigma      rho
# 211 4.000165 1.670114 0.3778854 0.03113893 4.716555
# [1] 132281.5
# [1] 86421241418
# beta0    beta1       beta2    sigma      rho
# 395 3.640675 1.530546 -0.06851556 2.487425 4.641385
# [1] 120393.5
# [1] 806179097
# beta0    beta1     beta2    sigma      rho
# 41 4.551687 1.582034 -0.402773 0.797141 4.815173
# [1] 150519.7
# [1] 586549590
# beta0     beta1     beta2    sigma       rho
# 116 2.548651 -0.185279 -0.353088 4.339141 0.3925233
# [1] 84281.33
# [1] 518243.9
# beta0     beta1     beta2    sigma      rho
# 163 2.43515 0.8247821 -0.383086 3.270095 1.170886
# [1] 80527.98
# [1] 636135.5
# beta0    beta1      beta2     sigma      rho
# 712 5.089383 1.445726 -0.2150306 0.1460754 1.814915
# [1] 168300.8
# [1] 719026228
# beta0     beta1      beta2      sigma       rho
# 738 4.634244 0.4135613 -0.3944152 0.06802721 0.4596939
# [1] 153249.8
# [1] 3627814
# beta0     beta1      beta2    sigma       rho
# 780 2.621741 0.9835234 -0.3510326 1.852368 0.5095075
# [1] 86698.35
# [1] 1292511
# beta0    beta1     beta2     sigma      rho
# 155 4.033903 1.530286 0.2582556 0.8008881 1.265108
# [1] 133397.1
# [1] 9428094538
# beta0    beta1      beta2     sigma      rho
# 208 3.085795 1.843315 -0.2211659 0.2358302 8.488874
# [1] 102044.1
# [1] 3883161142
# beta0    beta1      beta2     sigma      rho
# 341 3.807896 1.559669 -0.3982926 0.8312854 4.327047
# [1] 125923.3
# [1] 233093530
# beta0    beta1   beta2      sigma      rho
# 434 3.795388 1.290625 0.92024 0.01359262 2.012113
# [1] 125509.7
# [1] 119228848551
# beta0    beta1     beta2    sigma      rho
# 478 3.118912 1.477045 0.5885022 2.153329 53.00492
# [1] 103139.3
# [1] 25386562989
# beta0    beta1      beta2     sigma      rho
# 816 4.168051 2.015108 -0.5420551 0.7981628 2.578345
# [1] 137833.3
# [1] 11179560728
# beta0    beta1      beta2    sigma       rho
# 143 1.91468 1.654086 -0.3477218 1.991104 0.6363664
# [1] 63316.56
# [1] 104409529
# beta0    beta1    beta2     sigma      rho
# 412 2.493601 2.058278 0.150986 0.2490735 3.005743
# [1] 82460.88
# [1] 193699534519
# beta0    beta1       beta2     sigma     rho
# 732 3.670358 1.247426 -0.08147621 0.6172517 22.2272
# [1] 121375.1
# [1] 58751184
# beta0    beta1      beta2   sigma      rho
# 851 3.614131 2.124923 -0.7000833 0.87852 2.484851
# [1] 119515.7
# [1] 8689335035

min(sum.mu)
# [1] 34955.96
min(sum.expmu)
# [1] 8689335035
max(sum.mu)
# [1] 173008
max(sum.expmu)
# [1] 8689335035

mean(sum.mu)
# [1] 110135.4
mean(sum.expmu)
# [1] 8689335035


pdf("IrregPolMeanFieldandExpMeanFieldforTimeErr.pdf",height = 5, width=12.5)
for (k in 1:nrow(true.theta.err)){
  theta <- true.theta.err[k,]
  beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
  mu <- beta.0*intb.im + beta.1*popb.im + beta.2*incb.im
  
  par(mfrow=c(1,2))
  plot(mu,main=paste0("mu for ",k))
  plot(exp(mu),paste0("exp(mu) for ",k))
}
dev.off()



# SBC: Parameters for FFT -------------------------------------------------
# What parameter values result in more than 10 FFT warnings.

library(spatstat)

load("../IRREGPOLLGCP_OUTPUT/GridMeshIrregPolLGCPSBCSS.rda")
err.vec <- rep(NA,200)
ii <- sum(!is.na(err.vec))
for (j in 1:4){
  for (l in 1:4){
    if (sum((run.out.final[[j]][[l]]$mess.ls$FFT)>10,na.rm=T)!=0){
      err.vec[(ii+1):(ii+sum((run.out.final[[j]][[l]]$mess.ls$FFT)>10,na.rm=T))] <- which((run.out.final[[j]][[l]]$mess.ls$FFT)>10)
      ii <- sum(!is.na(err.vec))
    }
  }
}
err.vec
err.vec.u <- unique(err.vec[!is.na(err.vec)])
err.vec.u
true.theta.err <- true.theta.final[err.vec.u,]

true.theta.err
# beta0        beta1       beta2       sigma          rho
# 961 4.9153458  0.340883824 -0.52560684 0.091544715  277.2849040
# 182 1.5128099  0.180575250 -1.21881228 2.626382220   28.8353349
# 6   1.7114818  0.576155297 -0.76081442 0.219780296   15.0713585
# 39  0.9192488  0.156004945 -0.84524019 0.059391414    6.4561086
# 162 1.8440002  0.658604955 -0.45053016 0.293233947    1.3272201
# 207 4.0962493  0.281478355 -0.55488481 0.061575195    5.0153202
# 299 1.1925594  0.473991716 -0.11945791 1.815465927   60.1268700
# 313 4.4551928  0.685034287 -1.24958699 0.067409770    0.6484623
# 348 4.4273578  0.647190462 -0.71839802 0.064882934    4.2795833
# 378 2.0992014  0.690736557 -0.79134370 0.717446696   17.5144596
# 390 4.4857515  0.398467693 -1.16988468 0.104767779    0.8471358
# 425 3.3473837  0.512554241 -0.56403672 0.014915842    2.8969104
# 429 0.3355598  0.133020562 -0.48600231 0.181661371   13.2994405
# 530 1.3165718 -0.340822387 -0.55387938 0.029805186    1.2242924
# 539 3.5300045  0.503979149 -0.55030482 0.161889808   27.1075596
# 543 3.0856605  0.208515044 -0.39275805 0.132070008    6.0438813
# 580 0.6524381  0.284893028 -0.01585666 0.651687184    1.0047777
# 589 2.7669131  0.603440644 -0.26502798 0.098816802    1.1954871
# 660 4.7324747  0.653273423 -0.89750619 0.064228156    2.9863257
# 686 2.3338203  0.701685132 -0.93462923 0.048220849    1.2243507
# 817 2.1552092  0.287493538 -0.50287345 0.063263553    1.5758840
# 820 2.1737127  0.780146499 -0.38330588 0.007419395    1.1368313
# 824 4.2916847  0.150752602 -1.21789917 0.028733677    0.8666620
# 841 1.5594621  0.413316837 -1.24306795 0.122867163   12.3572260
# 858 3.9999486  0.642742411 -0.51253442 0.047191558    1.2145535
# 867 0.9374531 -0.476308855 -0.94136096 0.127639527    2.0488729
# 937 1.0745668  0.617798779 -0.91843439 0.645673655   61.4518682
# 976 5.3158236  0.733182744 -0.69061142 0.006424272    2.0164573
# 998 1.4245812  0.877346141 -1.10223093 0.473337772   66.9201520
# 511 2.2652646  0.862438613 -0.85445405 0.307970101    4.2959205
# 665 3.9630941  0.361317186 -0.55266849 0.006868955    1.2252492
# 269 3.7338509  0.275875449 -0.88281796 0.047334764    2.9381834
# 300 1.1781728  0.007068367 -0.47963224 0.196396549    3.8094033
# 364 2.9647484  0.564764438 -0.11795536 0.056192815  114.4852559
# 372 0.8570927  0.190850289 -0.59843674 0.179974413    0.9488316
# 518 2.7118621  0.244825594 -0.13234441 0.661470658   27.9008254
# 600 3.3444973  0.385934584 -0.60943241 0.309433802    4.2337813
# 664 0.9529113  0.470758113 -0.92304947 0.521685818    1.5041437
# 765 4.0583152  0.426196596 -0.07594704 0.027980486    1.0807741
# 818 3.6046347  0.409452972  0.18496188 0.221612398    4.5662472
# 24  2.7256496  0.550047288 -0.92796997 0.085998213   84.0991073
# 287 1.7999390  0.479008007 -0.72711215 0.179700351   31.0995077
# 531 2.6284665  0.518271645  0.17923559 1.286221250 5515.5949321
# 815 3.6316208  0.033469341 -0.56451022 0.175952247    3.1861357
# 853 2.2348160  0.870868062 -0.93726177 1.000492915  103.4953553
# 958 1.5791170 -0.219420561 -0.60489504 0.137705882    0.9128991
# 964 2.3548680  0.674928077 -1.62209459 0.125853795    0.6428203
# 968 4.6817026  0.516666544 -0.04896226 0.015833152    2.6459977

nrow(true.theta.err)


# SBC: Mean Field for FFT -------------------------------------------------
# What does the mean field look like for the runs which resulted in more than 10 FFT warnings.

load("../IRREGPOLLGCP_OUTPUT/GridMeshIrregPolLGCPSSCov.rda")

max.mu <- rep(NA,nrow(true.theta.err))

for (k in 1:nrow(true.theta.err)){
  theta <- true.theta.err[k,]
  beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
  mu <- beta.0*intb.im + beta.1*popb.im + beta.2*incb.im
  
  print("Values ")
  print(theta)
  print(max(mu$v))
  max.mu[k] <- max(mu$v)
  print(exp(max(mu$v)))
  print(mean(mu$v))
  print(median(mu$v))
  print(sum(mu$v>10)/length(mu$v))
  print(mean(exp(mu$v)))
}
# 
# [1] "Values "
# beta0     beta1      beta2      sigma      rho
# 961 4.915346 0.3408838 -0.5256068 0.09154471 277.2849
# [1] 6.579238
# [1] 719.9906
# [1] 1.91853
# [1] 0
# [1] 0
# [1] 56.84628
# [1] "Values "
# beta0     beta1     beta2    sigma      rho
# 182 1.51281 0.1805752 -1.218812 2.626382 28.83533
# [1] 2.795961
# [1] 16.37837
# [1] 0.5904715
# [1] 0
# [1] 0
# [1] 3.152928
# [1] "Values "
# beta0     beta1      beta2     sigma      rho
# 6 1.711482 0.5761553 -0.7608144 0.2197803 15.07136
# [1] 4.871691
# [1] 130.5415
# [1] 0.668016
# [1] 0
# [1] 0
# [1] 3.035092
# [1] "Values "
# beta0     beta1      beta2      sigma      rho
# 39 0.9192488 0.1560049 -0.8452402 0.05939141 6.456109
# [1] 1.866852
# [1] 6.467903
# [1] 0.3587961
# [1] 0
# [1] 0
# [1] 1.788037
# [1] "Values "
# beta0    beta1      beta2     sigma     rho
# 162 1.844 0.658605 -0.4505302 0.2932339 1.32722
# [1] 6.599771
# [1] 734.9272
# [1] 0.7197399
# [1] 0
# [1] 0
# [1] 3.508709
# [1] "Values "
# beta0     beta1      beta2      sigma     rho
# 207 4.096249 0.2814784 -0.5548848 0.06157519 5.01532
# [1] 5.140471
# [1] 170.7961
# [1] 1.598825
# [1] 0
# [1] 0
# [1] 25.59499
# [1] "Values "
# beta0     beta1      beta2    sigma      rho
# 299 1.192559 0.4739917 -0.1194579 1.815466 60.12687
# [1] 5.383594
# [1] 217.8037
# [1] 0.4654731
# [1] 0
# [1] 0
# [1] 2.066616
# [1] "Values "
# beta0     beta1     beta2      sigma       rho
# 313 4.455193 0.6850343 -1.249587 0.06740977 0.6484623
# [1] 7.271565
# [1] 1438.801
# [1] 1.738926
# [1] 0
# [1] 0
# [1] 43.73852
# [1] "Values "
# beta0     beta1     beta2      sigma      rho
# 348 4.427358 0.6471905 -0.718398 0.06488293 4.279583
# [1] 8.348753
# [1] 4224.91
# [1] 1.728062
# [1] 0
# [1] 0
# [1] 37.78249
# [1] "Values "
# beta0     beta1      beta2     sigma      rho
# 378 2.099201 0.6907366 -0.7913437 0.7174467 17.51446
# [1] 6.217324
# [1] 501.3597
# [1] 0.8193486
# [1] 0
# [1] 0
# [1] 4.30625
# [1] "Values "
# beta0     beta1     beta2     sigma       rho
# 390 4.485751 0.3984677 -1.169885 0.1047678 0.8471358
# [1] 6.139805
# [1] 463.9633
# [1] 1.750854
# [1] 0
# [1] 0
# [1] 45.64932
# [1] "Values "
# beta0     beta1      beta2      sigma     rho
# 425 3.347384 0.5125542 -0.5640367 0.01491584 2.89691
# [1] 6.466401
# [1] 643.165
# [1] 1.306532
# [1] 0
# [1] 0
# [1] 12.62972
# [1] "Values "
# beta0     beta1      beta2     sigma      rho
# 429 0.3355598 0.1330206 -0.4860023 0.1816614 13.29944
# [1] 0.9616964
# [1] 2.616131
# [1] 0.1309738
# [1] 0
# [1] 0
# [1] 1.190905
# [1] "Values "
# beta0      beta1      beta2      sigma      rho
# 530 1.316572 -0.3408224 -0.5538794 0.02980519 1.224292
# [1] 2.266516
# [1] 9.645737
# [1] 0.513877
# [1] 0
# [1] 0
# [1] 2.460453
# [1] "Values "
# beta0     beta1      beta2     sigma      rho
# 539 3.530005 0.5039791 -0.5503048 0.1618898 27.10756
# [1] 6.608557
# [1] 741.4126
# [1] 1.377812
# [1] 0
# [1] 0
# [1] 14.99546
# [1] "Values "
# beta0    beta1      beta2   sigma      rho
# 543 3.085661 0.208515 -0.3927581 0.13207 6.043881
# [1] 3.9091
# [1] 49.85407
# [1] 1.204378
# [1] 0
# [1] 0
# [1] 9.439817
# [1] "Values "
# beta0    beta1       beta2     sigma      rho
# 580 0.6524381 0.284893 -0.01585666 0.6516872 1.004778
# [1] 3.567909
# [1] 35.4424
# [1] 0.254656
# [1] 0
# [1] 0
# [1] 1.396488
# [1] "Values "
# beta0     beta1     beta2     sigma      rho
# 589 2.766913 0.6034406 -0.265028 0.0988168 1.195487
# [1] 7.527401
# [1] 1858.269
# [1] 1.079966
# [1] 0
# [1] 0
# [1] 8.025494
# [1] "Values "
# beta0     beta1      beta2      sigma      rho
# 660 4.732475 0.6532734 -0.8975062 0.06422816 2.986326
# [1] 8.220598
# [1] 3716.725
# [1] 1.847153
# [1] 0
# [1] 0
# [1] 52.21187
# [1] "Values "
# beta0     beta1      beta2      sigma      rho
# 686 2.33382 0.7016851 -0.9346292 0.04822085 1.224351
# [1] 6.160596
# [1] 473.7104
# [1] 0.9109238
# [1] 0
# [1] 0
# [1] 5.375285
# [1] "Values "
# beta0     beta1      beta2      sigma      rho
# 817 2.155209 0.2874935 -0.5028734 0.06326355 1.575884
# [1] 3.395959
# [1] 29.84327
# [1] 0.8412092
# [1] 0
# [1] 0
# [1] 4.154859
# [1] "Values "
# beta0     beta1      beta2       sigma      rho
# 820 2.173713 0.7801465 -0.3833059 0.007419395 1.136831
# [1] 8.217283
# [1] 3704.424
# [1] 0.8484314
# [1] 0
# [1] 0
# [1] 5.4676
# [1] "Values "
# beta0     beta1     beta2      sigma      rho
# 824 4.291685 0.1507526 -1.217899 0.02873368 0.866662
# [1] 5.548238
# [1] 256.7847
# [1] 1.675106
# [1] 0
# [1] 0
# [1] 42.04649
# [1] "Values "
# beta0     beta1     beta2     sigma      rho
# 841 1.559462 0.4133168 -1.243068 0.1228672 12.35723
# [1] 3.29808
# [1] 27.06062
# [1] 0.6086806
# [1] 0
# [1] 0
# [1] 3.098204
# [1] "Values "
# beta0     beta1      beta2      sigma      rho
# 858 3.999949 0.6427424 -0.5125344 0.04719156 1.214553
# [1] 8.442452
# [1] 4639.918
# [1] 1.561238
# [1] 0
# [1] 0
# [1] 25.03223
# [1] "Values "
# beta0      beta1     beta2     sigma      rho
# 867 0.9374531 -0.4763089 -0.941361 0.1276395 2.048873
# [1] 2.456781
# [1] 11.6672
# [1] 0.3659015
# [1] 0
# [1] 0
# [1] 2.294834
# [1] "Values "
# beta0     beta1      beta2     sigma      rho
# 937 1.074567 0.6177988 -0.9184344 0.6456737 61.45187
# [1] 4.183251
# [1] 65.57872
# [1] 0.4194189
# [1] 0
# [1] 0
# [1] 1.939272
# [1] "Values "
# beta0     beta1      beta2       sigma      rho
# 976 5.315824 0.7331827 -0.6906114 0.006424272 2.016457
# [1] 10.09441
# [1] 24207.39
# [1] 2.074843
# [1] 0
# [1] 1.180303e-05
# [1] 94.96793
# [1] "Values "
# beta0     beta1     beta2     sigma      rho
# 998 1.424581 0.8773461 -1.102231 0.4733378 66.92015
# [1] 6.390407
# [1] 596.0991
# [1] 0.5560346
# [1] 0
# [1] 0
# [1] 2.695226
# [1] "Values "
# beta0     beta1      beta2     sigma     rho
# 511 2.265265 0.8624386 -0.8544541 0.3079701 4.29592
# [1] 7.77148
# [1] 2371.978
# [1] 0.8841655
# [1] 0
# [1] 0
# [1] 5.432707
# [1] "Values "
# beta0     beta1      beta2       sigma      rho
# 665 3.963094 0.3613172 -0.5526685 0.006868955 1.225249
# [1] 5.738847
# [1] 310.7059
# [1] 1.546853
# [1] 0
# [1] 0
# [1] 22.42473
# [1] "Values "
# beta0     beta1     beta2      sigma      rho
# 269 3.733851 0.2758754 -0.882818 0.04733476 2.938183
# [1] 4.935472
# [1] 139.1388
# [1] 1.457376
# [1] 0
# [1] 0
# [1] 19.95328
# [1] "Values "
# beta0       beta1      beta2     sigma      rho
# 300 1.178173 0.007068367 -0.4796322 0.1963965 3.809403
# [1] 1.721382
# [1] 5.592249
# [1] 0.4598579
# [1] 0
# [1] 0
# [1] 1.992432
# [1] "Values "
# beta0     beta1      beta2      sigma      rho
# 364 2.964748 0.5647644 -0.1179554 0.05619281 114.4853
# [1] 8.131163
# [1] 3398.749
# [1] 1.157184
# [1] 0
# [1] 0
# [1] 10.00866
# [1] "Values "
# beta0     beta1      beta2     sigma       rho
# 372 0.8570927 0.1908503 -0.5984367 0.1799744 0.9488316
# [1] 1.678846
# [1] 5.359367
# [1] 0.3345356
# [1] 0
# [1] 0
# [1] 1.611686
# [1] "Values "
# beta0     beta1      beta2     sigma      rho
# 518 2.711862 0.2448256 -0.1323444 0.6614707 27.90083
# [1] 4.575573
# [1] 97.08361
# [1] 1.058479
# [1] 0
# [1] 0
# [1] 6.595964
# [1] "Values "
# beta0     beta1      beta2     sigma      rho
# 600 3.344497 0.3859346 -0.6094324 0.3094338 4.233781
# [1] 5.189113
# [1] 179.3094
# [1] 1.305406
# [1] 0
# [1] 0
# [1] 12.49623
# [1] "Values "
# beta0     beta1      beta2     sigma      rho
# 664 0.9529113 0.4707581 -0.9230495 0.5216858 1.504144
# [1] 2.712866
# [1] 15.07241
# [1] 0.371935
# [1] 0
# [1] 0
# [1] 1.78799
# [1] "Values "
# beta0     beta1       beta2      sigma      rho
# 765 4.058315 0.4261966 -0.07594704 0.02798049 1.080774
# [1] 8.049726
# [1] 3132.936
# [1] 1.584019
# [1] 0
# [1] 0
# [1] 25.73987
# [1] "Values "
# beta0    beta1     beta2     sigma      rho
# 818 3.604635 0.409453 0.1849619 0.2216124 4.566247
# [1] 9.354156
# [1] 11546.71
# [1] 1.406941
# [1] 0
# [1] 0
# [1] 19.49291
# [1] "Values "
# beta0     beta1    beta2      sigma      rho
# 24 2.72565 0.5500473 -0.92797 0.08599821 84.09911
# [1] 5.192674
# [1] 179.9492
# [1] 1.06386
# [1] 0
# [1] 0
# [1] 7.532915
# [1] "Values "
# beta0    beta1      beta2     sigma      rho
# 287 1.799939 0.479008 -0.7271121 0.1797004 31.09951
# [1] 4.169311
# [1] 64.67086
# [1] 0.7025422
# [1] 0
# [1] 0
# [1] 3.216227
# [1] "Values "
# beta0     beta1     beta2    sigma      rho
# 531 2.628466 0.5182716 0.1792356 1.286221 5515.595
# [1] 9.414165
# [1] 12260.83
# [1] 1.025928
# [1] 0
# [1] 0
# [1] 9.220473
# [1] "Values "
# beta0      beta1      beta2     sigma      rho
# 815 3.631621 0.03346934 -0.5645102 0.1759522 3.186136
# [1] 4.247706
# [1] 69.9448
# [1] 1.417474
# [1] 0
# [1] 0
# [1] 17.05227
# [1] "Values "
# beta0     beta1      beta2    sigma      rho
# 853 2.234816 0.8708681 -0.9372618 1.000493 103.4954
# [1] 7.591757
# [1] 1981.792
# [1] 0.8722809
# [1] 0
# [1] 0
# [1] 5.275385
# [1] "Values "
# beta0      beta1     beta2     sigma       rho
# 958 1.579117 -0.2194206 -0.604895 0.1377059 0.9128991
# [1] 2.475295
# [1] 11.88521
# [1] 0.6163522
# [1] 0
# [1] 0
# [1] 2.950266
# [1] "Values "
# beta0     beta1     beta2     sigma       rho
# 964 2.354868 0.6749281 -1.622095 0.1258538 0.6428203
# [1] 4.877999
# [1] 131.3675
# [1] 0.919139
# [1] 0
# [1] 0
# [1] 6.822055
# [1] "Values "
# beta0     beta1       beta2      sigma      rho
# 968 4.681703 0.5166665 -0.04896226 0.01583315 2.645998
# [1] 9.825855
# [1] 18506.09
# [1] 1.827336
# [1] 0
# [1] 0
# [1] 52.36279


min(max.mu)
# [1] 0.9616964
max(max.mu)
# [1] 10.09441

mean(max.mu)
# [1] 5.553834


sum.mu <- rep(NA,nrow(true.theta.err))
sum.expmu <- rep(NA,nrow(true.theta.err))
for (k in 1:nrow(true.theta.err)){
  theta <- true.theta.err[k,]
  beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
  mu <- beta.0*intb.im + beta.1*popb.im + beta.2*incb.im
  
  print(theta)
  print(sum(mu$v))
  sum.mu[k] <- sum(mu$v)
  sum.expmu <- sum(exp(mu$v))
  print(sum(exp(mu$v)))
}



# Traditional Simulation Study --------------------------------------------

# Traditional: Parameters for Errors --------------------------------------
# What parameter values result in errors?

load("../IRREGPOLLGCP_OUTPUT/GridMeshIrregPolLGCPTradSS.rda")

err.vec <- rep(NA,50)
ii <- sum(!is.na(err.vec))
for (j in 1:4){
  for (l in 1:4){
    if (sum(!is.na(run.out.final[[j]][[l]]$mess.ls$err))!=0){
      err.vec[(ii+1):(ii+sum(!is.na(run.out.final[[j]][[l]]$mess.ls$err)))] <- which(!is.na(run.out.final[[j]][[l]]$mess.ls$err))
      ii <- sum(!is.na(err.vec))
    }
  }
}
err.vec
err.vec.u <- unique(err.vec[!is.na(err.vec)])
err.vec.u
true.theta.err <- true.theta.final[err.vec.u,]

true.theta.err
# [1] beta0 beta1 beta2 sigma rho  
# <0 rows> (or 0-length row.names)

nrow(true.theta.err)


# Traditional: Mean Field for Errors --------------------------------------
# What does the mean field look like for the runs which resulted in errors.

load("../IRREGPOLLGCP_OUTPUT/GridMeshIrregPolLGCPTradSS.rda")
err.vec <- rep(NA,200)
ii <- sum(!is.na(err.vec))
for (j in 1:4){
  for (l in 1:4){
    if (sum((run.out.final[[j]][[l]]$mess.ls$FFT)>10,na.rm=T)!=0){
      err.vec[(ii+1):(ii+sum((run.out.final[[j]][[l]]$mess.ls$FFT)>10,na.rm=T))] <- which((run.out.final[[j]][[l]]$mess.ls$FFT)>10)
      ii <- sum(!is.na(err.vec))
    }
  }
}
err.vec
err.vec.u <- unique(err.vec[!is.na(err.vec)])
err.vec.u
true.theta.err <- true.theta.final[err.vec.u,]

true.theta.err
# [1] beta0 beta1 beta2 sigma rho  
# <0 rows> (or 0-length row.names)


nrow(true.theta.err)


# sessionInfo() -----------------------------------------------------------

sessionInfo()