
# SBC Mean Field Irreg Pol Wrapper Function: Relabelled -------------------

# This is a wrapper function that will produce summaries for the frequencies of the ranks for SBC as well as assessing the uniformity of the frequencies.
# This function is for considering the fixed effects and hyperparameters of a spatial model, for latent effects, there will be another function to perform similarly.
# This R script contains the functions that produce some of the plots with the altered axis labels, output with suffix *_Relabel.pdf. Any plots that we don't want to reproduce with different labels are commented out so that they aren't unnecessarily saved.

# Author: Nadeen Khaleel

# Libraries ---------------------------------------------------------------

library(purrr)
library(ggplot2)
library(rlist)
library(magrittr)
library(grid)
library(gridExtra)
library(stringr)
library(scales)

# Sub-functions -----------------------------------------------------------

plot.ranks <- function(N,L,ranks,main,cexmain,cexlab,cexaxis){
  d1 <- qbinom(0.005,N,1/(L+1))
  d2 <- qbinom(0.995,N,1/(L+1))
  m <- N/(L+1)
  xcoord <- c(-0.5,L+0.5,L+0.5,-0.5)
  ycoord <- c(d1,d1,d2,d2)
  
  hist(ranks,breaks=seq(-0.5,L+0.5),main=main,xlab = "Rank Statistic",cex.main=cexmain, cex.lab=cexlab, cex.axis=cexaxis)
  polygon(xcoord,ycoord,col=rgb(0.5,0,0.5,0.25),border = NA)
  segments(-0.5,m,L+0.5,m,col="blue")
}

freq.func <- function(x){
  df <- as.data.frame(table(x))
  names(df) <- c("ranks","freq")
  return(df)
}

summary.stats <- function(N,L,freq,m,lower.bound,upper.bound){
  out <- (sum(freq<lower.bound)+sum(freq>upper.bound))/(L+1) # L + 1 ranks, {0,...,L}
  dist <- sum(sqrt((freq-m)^2))
  return(list("outside.bounds"=out,"dist.sum"=dist))
}

models_sbc <- function(freq.ranks,test="Chisq"){
  
  if (test=="Chisq"){
    ranks.df <- data.frame(Freq=freq.ranks$freq,ranks=(as.numeric(as.character(freq.ranks$ranks))))
    fit.const <- glm(Freq~1,data = ranks.df, family = "poisson")
    fit.ranks <- glm(Freq~ranks,data = ranks.df, family = "poisson")
    fit.ranks2 <- glm(Freq~ranks+I(ranks^2),data = ranks.df, family = "poisson")
    
    a1 <- anova(fit.const,fit.ranks,test = "Chisq")
    a2 <- anova(fit.const,fit.ranks2,test = "Chisq")
    
    if (!is.na(a1$`Pr(>Chi)`[2])){
      if (a1$`Pr(>Chi)`[2]>0.01){ # prefer const over rank
        if (a2$`Pr(>Chi)`[2]>0.01){ # prefer const over rank^2
          u.divergence <- "FALSE"
          mod <- fit.const
        } else { # prefer rank^2 over const - therefore as rank < const < rank^2
          u.divergence <- "TRUE"
          mod <- fit.ranks2
        }
      } else { # prefer rank over const
        if (a2$`Pr(>Chi)`[2]>0.01){ # prefer const over rank^2 - therefore rank^2 < const < rank
          u.divergence <- "TRUE"
          mod <- fit.ranks
        } else {
          a3 <- anova(fit.ranks,fit.ranks2,test = "Chisq")
          if (!is.na(a3$`Pr(>Chi)`[2])){
            if (a3$`Pr(>Chi)`[2]<0.01){
              u.divergence <- "TRUE"
              mod <- fit.ranks2
            } else {
              u.divergence <- "TRUE"
              mod <- fit.ranks
            }
          } else { # if null, then ranks^2 term is NA and I believe the two non-constant models are the same
            u.divergence <- "TRUE"
            mod <- fit.ranks
          }
        }
      } 
    } else if (dim(ranks.df)[1]==1){ # NAs for non-constant models
      u.divergence="TRUE"
      mod <- fit.ranks
    }
    
    
  } else {
    
    # OR - use step() function and check dim?
    full.fit <- glm(Freq~ranks+I(ranks^2),data = ranks.df, family = "poisson")
    mod <- step(full.fit)
    if (length(final.fit$coefficients)==1){
      u.divergence=="FALSE"
    } else {
      u.divergence=="TRUE"
    }
  }
  return(list("div"=u.divergence,"fit.mod"=mod))
}

models_sbc.ranks <- function(models,div){
  order.names <- rep(NA,sum(lengths(div)))
  order.param <- rep(NA,sum(lengths(div)))
  r1.vals <- rep(NA,sum(lengths(div)))
  r2.vals <- rep(NA,sum(lengths(div)))
  d <- 1
  for (i in 1:length(div)){
    if (length(div[[i]])!=0){
      order.names[min(which(is.na(order.names))):(min(which(is.na(order.names)))+length(div[[i]])-1)] <- i
      order.param[min(which(is.na(order.param))):(min(which(is.na(order.param)))+length(div[[i]])-1)] <- div[[i]]
      for (j in 1:length(div[[i]])){
        r1.vals[d] <- abs(unname(models[[i]][[div[[i]][j]]]$fit.mod$coefficients[2]))
        r2.vals[d] <- abs(unname(models[[i]][[div[[i]][j]]]$fit.mod$coefficients[3]))
        d <- d + 1
      }
    }
  }
  df <- data.frame(name.ind=order.names,param.ind=order.param,rank1=r1.vals,rank2=r2.vals)
  df <- df[order(df$rank2,df$rank1,decreasing=TRUE),]
  return(list("order.df"=df))
}

rank.vals <- function(prior.vals,dap.samps){rowSums(dap.samps<prior.vals)}

# One go-round
run.same <- function(N,L,mu,sigma){
  prior.vals <- rnorm(N)
  dap.samps <- matrix(rnorm(N*L),nrow=N,byrow=TRUE)
  ranks <- rank.vals(prior.vals,dap.samps)
}


# Main Function -----------------------------------------------------------


sbc_checks_latent <- function(N,L,rank.values,grids,meshes,plots=FALSE,plot.save=FALSE,plots.min=FALSE,plot.cap=5,worst.plot="grid",plot.name=c("irregpolsumdisttest","irregpoloutsideboundstest","irregpolsbcdivergences")){
  # N - total number of simulations
  # L - total number of posterior samples
  # rank.values - the data output from the simulation study, extracting only the mean field results
  # grids - character string for names of the grids for each list item of rank.values
  # meshes - character string for names of the meshes for each list item of rank.values
  # plots - want to produce plots for the different summaries and the SBC output?
  # plot.save - save plots or print only?
  # plots.min - minimise the number of SBC plots produced - returning the "worst" as ranked by the coefficients
  # plot.cap - if plots.min=TRUE, then how many of the "worst" plots for each grid/mesh do we want to plot
  # worst.plot - plot the worst plots for the grids or the worst plots for the mesh
  # plot.name - if you want the plots to be saved, provide a vector of length 3, each element in is a character string that you want the plots to be saved as: one for the proportion outside the boundary, another for average distance from mean and the final is the plots of the SBC histograms.
  
  # Want grids and meshes for each rank.values item (regardless of repetition), if this is not the case, create the lists, where hold grid fixed and go down meshes first..
  
  if (length(meshes)!=length(rank.values)){
    meshes <- rep(meshes,length(unique(grids)))
  }
  
  r.n <- length(rank.values)
  l.m <- length(unique(meshes))
  l.g <- length(unique(grids))
  
  levels.m <- unique(meshes)
  levels.g <- unique(grids)
  
  names.gm <- names(rank.values)
  
  grids.names <- grids
  ord.grids.names <- unique(grids.names)
  for (i in 1:l.g){
    grids.names <- str_replace(grids.names,ord.grids.names[i],paste0("Grid ",i))
  }
  meshes.names <- meshes
  ord.meshes.names <- unique(meshes.names)[order(unique(meshes.names),decreasing = T)]
  for (i in 1:l.m){
    meshes.names <- str_replace(meshes.names,ord.meshes.names[i],paste0("Mesh ",i)) # assume grid and mesh labels are in order
  }
  
  lower.bound <- qbinom(0.005,N,1/(L+1))
  upper.bound <- qbinom(0.995,N,1/(L+1))
  m <- N/(L+1)
  
  # As a comparison for the stats, sample from the SAME distribution
  mult.same.ranks <- replicate(1e3,run.same(N,L))
  freq.multsame.ranks <- apply(mult.same.ranks,2,freq.func)
  below.multsame.ranks <- unlist(lapply(1:1e3,FUN=function(i){sum(freq.multsame.ranks[[i]]$freq<lower.bound)}))
  above.multsame.ranks <- unlist(lapply(1:1e3,FUN=function(i){sum(freq.multsame.ranks[[i]]$freq>upper.bound)}))
  dist.multsame.ranks <- unlist(lapply(1:1e3,FUN=function(i){sum(sqrt((freq.multsame.ranks[[i]]$freq-m)^2))}))
  outside.multsame.ranks <- (below.multsame.ranks+above.multsame.ranks)/(L+1) # L + 1 ranks, {0,...,L}
  
  ave.outside.ranks <- mean(outside.multsame.ranks)
  ave.dist.ranks <- mean(dist.multsame.ranks)
  
  sd.outside.ranks <- sd(outside.multsame.ranks)
  sd.dist.ranks <- sd(dist.multsame.ranks)
  
  f.n <- sapply(1:length(rank.values),function(i){dim(rank.values[[i]])[2]}) # how many ``parameters'' [f(theta) terms are we considering]
  if (is.null(f.n)){
    f.n <- length(rank.values[[1]])
  }
  names.f <- lapply(1:length(rank.values),function(i){names(rank.values[[i]])})
  
  
  sub.freq.tabs <- function(x,param.names){
    r <- rank.values[[x]]
    out <- lapply(1:f.n[x],function(i){freq.func(r[[i]])})
    names(out) <- paste0("Ranks_",param.names)
    out
  }
  freq.tabs <- lapply(1:r.n,function(x,param.names){sub.freq.tabs(x,param.names[[x]])},names.f)
  names(freq.tabs) <- names.gm
  
  
  sub.summary <- function(x,param.names){
    f <- freq.tabs[[x]]
    out <- sapply(1:f.n[x],function(i){summary.stats(N,L,f[[i]]$freq,m,lower.bound,upper.bound)})
    colnames(out) <- param.names
    out
  }
  summaries <- lapply(1:r.n,function(x,param.names){sub.summary(x,param.names[[x]])},names.f)
  names(summaries) <- names.gm
  
  OutsideBounds <- unname(unlist(summaries %>% map(~.x[1, ]))) # first row is outside.bounds
  MeanDist <- unname(unlist(summaries %>% map(~.x[2, ]))) # second row is dist.sum
  sum.df <- data.frame(Grid=rep(grids.names,f.n),Mesh=rep(meshes.names,f.n),Parameter=unlist(names.f),OutsideBounds=OutsideBounds,MeanDist=MeanDist)
  
  sum.df$Grid <- factor(as.character(sum.df$Grid),levels=unique(sum.df$Grid)[order(unique(sum.df$Grid))])
  sum.df$Mesh <- factor(as.character(sum.df$Mesh),levels=unique(sum.df$Mesh)[order(unique(sum.df$Mesh),decreasing=FALSE)])
  
  # Setting up labels for the plots
  var.lab.orig <- as.character(sum.df$Parameter)
  num <- as.numeric(str_extract(var.lab.orig, "[0-9]+"))
  char <- str_extract(var.lab.orig, "[aA-zZ]+")
  
  plot.x.lab <- char
  plot.x.lab
  for (i in which(!is.na(num))){
    plot.x.lab[i] <- paste0(plot.x.lab[i],"[",num[i],"]")
  }
  plot.x.lab
  
  if (sum(plot.x.lab=="Int")>0){
    plot.x.lab[plot.x.lab=="Int"] <- "beta[0]"
  }
  sum.df$Label <- plot.x.lab
  
  
  sub.models_sbc <- function(x,param.names){
    f <- freq.tabs[[x]]
    out <- lapply(1:f.n[x],function(i){models_sbc(f[[i]])})
    names(out) <- param.names
    out
  }
  
  # unif.check <- lapply(1:r.n,function(x,param.names){sub.models_sbc(x,param.names[[x]])},names.f)
  # names(unif.check) <- names.gm
  
  opar <- par()
  
  if (plots==TRUE && plots.min==FALSE){
    
    # Data Frames for plotting
    # unif.div <- lapply(1:r.n,function(i){list.which(unif.check[[i]],div=="TRUE")})
    # names(unif.div) <- names.gm
    # if (sum(lengths(unif.div))==0){
    #   print("No detected divergences from uniformity within SBC.")
    # } else {
    #   unif.div.cov <- list()
    #   unif.div.cov <- lapply(1:r.n,function(i){unif.div.cov[[i]] <- matrix(rep(NA,3*length(unif.div[[i]])),ncol=3)})
    #   unif.div.cov <- lapply(1:r.n,function(x){
    #     uc <- unif.check[[x]]; ud <- unif.div[[x]]
    #     
    #     if (length(ud)!=0){ # some grid-mesh combos have no problems arising...
    #       for (i in 1:length(ud)){
    #         unif.div.cov[[x]][i,1:length(uc[[ud[i]]]$fit.mod$coefficients)] <- uc[[ud[i]]]$fit.mod$coefficients
    #       }
    #       unif.div.cov[[x]] <- round(unif.div.cov[[x]],8)
    #     }
    #   })
    #   
    #   sub.fitted_values <- function(x,param.names){
    #     ud <- unif.div[[x]]; uc <- unif.check[[x]]
    #     out <- lapply(ud,function(i){uc[[i]]$fit.mod$fitted.values})
    #     names(out) <- param.names[ud]
    #     out
    #   }
    #   y <- lapply(1:r.n,function(x,param.names){sub.fitted_values(x,param.names[[x]])},names.f)
    #   names(y) <- names.gm
    # }
    
    outside.ylim <- rep(ave.outside.ranks) + c(-1,1)*2*sd.outside.ranks
    outside.ylim[1] <- max(0,outside.ylim[1])
    dist.ylim <- rep(ave.dist.ranks) + c(-1,1)*2*sd.dist.ranks
    
    if (plot.save==TRUE){
      # First plot the distance summaries
      p1 <- ggplot(sum.df,aes(Mesh,MeanDist)) + facet_wrap(~sum.df$Grid) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Mesh))-0.5,xmax=max(as.numeric(sum.df$Mesh))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=2,aes(color=Mesh)) + facet_wrap(~Grid) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
      p2 <- ggplot(sum.df,aes(Grid,MeanDist)) + facet_wrap(~sum.df$Mesh) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Grid))-0.5,xmax=max(as.numeric(sum.df$Grid))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=2,aes(color=Grid)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
      p1
      p2
      ggsave(paste0(plot.name[1],"byGrid_Relabel.pdf"),plot=p1,width = 15,height = 10,device = "pdf")
      ggsave(paste0(plot.name[1],"byMesh_Relabel.pdf"),plot=p2,width = 15,height = 10,device = "pdf")
      
      # Second plot the outside of bounds summaries
      p1 <- ggplot(sum.df,aes(Mesh,OutsideBounds)) + facet_wrap(~sum.df$Grid) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Mesh))-0.5,xmax=max(as.numeric(sum.df$Mesh))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=2,aes(color=Mesh)) + facet_wrap(~Grid) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
      p2 <- ggplot(sum.df,aes(Grid,OutsideBounds)) + facet_wrap(~sum.df$Mesh) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Grid))-0.5,xmax=max(as.numeric(sum.df$Grid))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=2,aes(color=Grid)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
      p1
      p2
      ggsave(paste0(plot.name[2],"byGrid_Relabel.pdf"),plot=p1,width = 15,height = 10,device="pdf")
      ggsave(paste0(plot.name[2],"byMesh_Relabel.pdf"),plot=p2,width = 15,height = 10,device="pdf")
      
      # Plot the sbc runs that had problems
      # if (sum(lengths(unif.div)==0)){
      #   print("No detected divergences from uniformity within SBC so no plot produced.")
      # } else {
      #   pdf(paste0(plot.name[3],"_Relabel.pdf"),h=10,w=15,pointsize=14)
      #   if (sum(lengths(unif.div))==1){
      #     par(oma=opar$oma+c(5,0,0,0),mar=opar$mar+c(0,1,0,0))
      #   } else if (sum(lengths(unif.div))>1){
      #     par(oma=opar$oma+c(5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
      #   }
      #   
      #   Grid_Mesh <- paste0(grids.names," and ",meshes.names)
      #   
      #   for (i in 1:length(unif.div)){
      #     if (length(unif.div[[i]])!=0){
      #       for (j in 1:length(unif.div[[i]])){
      #         plot.ranks(N,L,rank.values[[i]][[unif.div[[i]][j]]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(names.f[[i]][unif.div[[i]][j]]), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
      #         points(as.numeric(as.character(freq.tabs[[i]][[unif.div[[i]][j]]]$ranks)),y[[i]][[j]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
      #         int.mtext <- paste0("Int=",format(unif.div.cov[[i]][jj,1],scientific=TRUE,digits=3))
      #         cov.mtext <- paste0("rank=",format(unif.div.cov[[i]][jj,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.div.cov[[i]][jj,3],scientific=TRUE,digits=3))
      #         int.mtext <- str_replace_all(int.mtext,"e","x10^")
      #         cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
      #         int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
      #         cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
      #         mtext(text=int.mtext,side=1,line=5,cex=1.5) 
      #         mtext(text=cov.mtext,side=1,line=6,cex=1.5) 
      #       }
      #     }
      #   }
      #   dev.off()
      # }
    } else {
      # First plot the distance summaries
      p1 <- ggplot(sum.df,aes(Mesh,MeanDist)) + facet_wrap(~sum.df$Grid) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Mesh))-0.5,xmax=max(as.numeric(sum.df$Mesh))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + facet_wrap(~Grid) + geom_boxplot(size=2,aes(color=Mesh)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
      p2 <- ggplot(sum.df,aes(Grid,MeanDist)) + facet_wrap(~sum.df$Mesh) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Grid))-0.5,xmax=max(as.numeric(sum.df$Grid))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=2,aes(color=Grid)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
      print(p1)
      print(p2)
      
      # Second plot the outside of bounds summaries
      p4 <- ggplot(sum.df,aes(Mesh,OutsideBounds)) + facet_wrap(~sum.df$Grid) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Mesh))-0.5,xmax=max(as.numeric(sum.df$Mesh))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=2,aes(color=Mesh)) + facet_wrap(~Grid) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
      p5 <- ggplot(sum.df,aes(Grid,OutsideBounds)) + facet_wrap(~sum.df$Mesh) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Grid))-0.5,xmax=max(as.numeric(sum.df$Grid))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=2,aes(color=Grid)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
      print(p4)
      print(p5)
      
      
      # Plot the sbc runs that had problems
      
      # if (sum(lengths(unif.div)==0)){
      #   print("No detected divergences from uniformity within SBC so no plots produced.")
      # } else {
      #   if (sum(lengths(unif.div))==1){
      #     par(oma=opar$oma+c(5,0,0,0),mar=opar$mar+c(0,1,0,0))
      #   } else if (sum(lengths(unif.div))>1){
      #     par(oma=opar$oma+c(5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
      #   }
      #   
      #   Grid_Mesh <- paste0(grids.names," and ",meshes.names)
      #   
      #   for (i in 1:length(unif.div)){
      #     if (length(unif.div[[i]])!=0){
      #       for (j in 1:length(unif.div[[i]])){
      #         plot.ranks(N,L,rank.values[[i]][[unif.div[[i]][j]]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(names.f[[i]][unif.div[[i]][j]]), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
      #         points(as.numeric(as.character(freq.tabs[[i]][[unif.div[[i]][j]]]$ranks)),y[[i]][[j]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
      #         int.mtext <- paste0("Int=",format(unif.div.cov[[i]][jj,1],scientific=TRUE,digits=3))
      #         cov.mtext <- paste0("rank=",format(unif.div.cov[[i]][jj,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.div.cov[[i]][jj,3],scientific=TRUE,digits=3))
      #         int.mtext <- str_replace_all(int.mtext,"e","x10^")
      #         cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
      #         int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
      #         cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
      #         mtext(text=int.mtext,side=1,line=5,cex=1.5) 
      #         mtext(text=cov.mtext,side=1,line=6,cex=1.5) 
      #       }
      #     }
      #   }
      # } 
    }
  } else if (plots==TRUE && plots.min==TRUE){
    # unif.div <- lapply(1:r.n,function(i){list.which(unif.check[[i]],div=="TRUE")})
    # names(unif.div) <- names.gm
    # if (sum(lengths(unif.div))==0){
    #   print("No detected divergences from uniformity within SBC.")
    #   sel.plot <- list("order.df"=data.frame())
    # } else {
    #   unif.div.cov <- list()
    #   unif.div.cov <- lapply(1:r.n,function(i){unif.div.cov[[i]] <- matrix(rep(NA,3*length(unif.div[[i]])),ncol=3)})
    #   unif.div.cov <- lapply(1:r.n,function(x){
    #     uc <- unif.check[[x]]; ud <- unif.div[[x]]
    #     if (length(ud)!=0){ # some grid-mesh combos have no problems arising...
    #       for (i in 1:length(ud)){
    #         unif.div.cov[[x]][i,1:length(uc[[ud[i]]]$fit.mod$coefficients)] <- uc[[ud[i]]]$fit.mod$coefficients
    #       }
    #       unif.div.cov[[x]] <- round(unif.div.cov[[x]],8)
    #     }
    #   })
    #   
    #   sub.fitted_values <- function(x,param.names){
    #     ud <- unif.div[[x]]; uc <- unif.check[[x]]
    #     out <- lapply(ud,function(i){uc[[i]]$fit.mod$fitted.values})
    #     names(out) <- param.names[ud]
    #     out
    #   }
    #   y <- lapply(1:r.n,function(x,param.names){sub.fitted_values(x,param.names[[x]])},names.f)
    #   names(y) <- names.gm
    #   
    #   sel.plot <- models_sbc.ranks(unif.check,unif.div)
    # }
    
    
    outside.ylim <- rep(ave.outside.ranks) + c(-1,1)*2*sd.outside.ranks
    outside.ylim[1] <- max(0,outside.ylim[1])
    dist.ylim <- rep(ave.dist.ranks) + c(-1,1)*2*sd.dist.ranks
    
    # if (nrow(sel.plot$order.df)<=plot.cap){
    #   if (plot.save==TRUE){
    #     # First plost the distance summaries
    #     p1 <- ggplot(sum.df,aes(Mesh,MeanDist)) + facet_wrap(~sum.df$Grid) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Mesh))-0.5,xmax=max(as.numeric(sum.df$Mesh))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Mesh)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
    #     p2 <- ggplot(sum.df,aes(Grid,MeanDist)) + facet_wrap(~sum.df$Mesh) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Grid))-0.5,xmax=max(as.numeric(sum.df$Grid))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Grid)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
    #     p1
    #     p2
    #     ggsave(paste0(plot.name[1],"byGrid_Relabel.pdf"),plot=p1,width = 15,height = 10,device = "pdf")
    #     ggsave(paste0(plot.name[1],"byMesh_Relabel.pdf"),plot=p2,width = 15,height = 10,device = "pdf")
    #     
    #     # Second plot the outside of bounds summaries
    #     p1 <- ggplot(sum.df,aes(x=Mesh,y=OutsideBounds)) + facet_wrap(~sum.df$Grid) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Mesh))-0.5,xmax=max(as.numeric(sum.df$Mesh))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Mesh)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
    #     p2 <- ggplot(sum.df,aes(x=Grid,y=OutsideBounds)) + facet_wrap(~sum.df$Mesh) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Grid))-0.5,xmax=max(as.numeric(sum.df$Grid))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Grid)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
    #     p1
    #     p2
    #     ggsave(paste0(plot.name[2],"byGrid_Relabel.pdf"),plot=p1,width = 15,height = 10,device = "pdf")
    #     ggsave(paste0(plot.name[2],"byMesh_Relabel.pdf"),plot=p2,width = 15,height = 10,device = "pdf")
    #     
    #     # Plot the sbc runs that had problems
    #     # if (sum(lengths(unif.div))==0){
    #     #   print("No detected divergences from uniformity within SBC so no plot produced.")
    #     # } else {
    #     #   pdf(paste0(plot.name[3],"_Relabel.pdf"),h=10,w=15,pointsize=14)
    #     #   if (sum(lengths(unif.div))==1){
    #     #     par(oma=opar$oma+c(5,0,0,0),mar=opar$mar+c(0,1,0,0))
    #     #   } else if (sum(lengths(unif.div))>1){
    #     #     par(oma=opar$oma+c(5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
    #     #   }
    #     #   
    #     #   Grid_Mesh <- paste0(grids.names," and ",meshes.names)
    #     #   
    #     #   for (i in 1:length(unif.div)){
    #     #     if (length(unif.div[[i]])!=0){
    #     #       for (j in 1:length(unif.div[[i]])){
    #     #         plot.ranks(N,L,rank.values[[i]][[unif.div[[i]][j]]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(names.f[[i]][unif.div[[i]][j]]), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
    #     #         points(as.numeric(as.character(freq.tabs[[i]][[unif.div[[i]][j]]]$ranks)),y[[i]][[j]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
    #     #         int.mtext <- paste0("Int=",format(unif.div.cov[[i]][jj,1],scientific=TRUE,digits=3))
    #     #         cov.mtext <- paste0("rank=",format(unif.div.cov[[i]][jj,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.div.cov[[i]][jj,3],scientific=TRUE,digits=3))
    #     #         int.mtext <- str_replace_all(int.mtext,"e","x10^")
    #     #         cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
    #     #         int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
    #     #         cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
    #     #         mtext(text=int.mtext,side=1,line=5,cex=1.5) 
    #     #         mtext(text=cov.mtext,side=1,line=6,cex=1.5) 
    #     #       }
    #     #     }
    #     #   }
    #     #   dev.off()
    #     # } 
    #     
    #   } else {
    #     # First plot the distance summaries
    #     p1 <- ggplot(sum.df,aes(Mesh,MeanDist)) + facet_wrap(~sum.df$Grid) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Mesh))-0.5,xmax=max(as.numeric(sum.df$Mesh))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Mesh)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
    #     p2 <- ggplot(sum.df,aes(Grid,MeanDist)) + facet_wrap(~sum.df$Mesh) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Grid))-0.5,xmax=max(as.numeric(sum.df$Grid))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Grid)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
    #     print(p1)
    #     print(p2)
    #     
    #     # Second plot the outside of bounds summaries
    #     p4 <- ggplot(sum.df,aes(Mesh,OutsideBounds)) + facet_wrap(~sum.df$Grid) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Mesh))-0.5,xmax=max(as.numeric(sum.df$Mesh))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Mesh)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
    #     p5 <- ggplot(sum.df,aes(Grid,OutsideBounds)) + facet_wrap(~sum.df$Mesh) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Grid))-0.5,xmax=max(as.numeric(sum.df$Grid))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Grid)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
    #     print(p4)
    #     print(p5)
    #     
    #     
    #     # Plot the sbc runs that had problems
    #     
    #     # if (sum(lengths(unif.div))==0){
    #     #   print("No detected divergences from uniformity within SBC.")
    #     # } else {
    #     #   if (sum(lengths(unif.div))==1){
    #     #     par(oma=opar$oma+c(5,0,0,0),mar=opar$mar+c(0,1,0,0))  
    #     #   } else if (sum(lengths(unif.div))>1){
    #     #     par(oma=opar$oma+c(5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
    #     #   }
    #     #   
    #     #   Grid_Mesh <- paste0(grids.names," and ",meshes.names)
    #     #   
    #     #   for (i in 1:length(unif.div)){
    #     #     if (length(unif.div[[i]])!=0){
    #     #       for (j in 1:length(unif.div[[i]])){
    #     #         plot.ranks(N,L,rank.values[[i]][[unif.div[[i]][j]]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(names.f[[i]][unif.div[[i]][j]]), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
    #     #         points(as.numeric(as.character(freq.tabs[[i]][[unif.div[[i]][j]]]$ranks)),y[[i]][[j]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
    #     #         int.mtext <- paste0("Int=",format(unif.div.cov[[i]][jj,1],scientific=TRUE,digits=3))
    #     #         cov.mtext <- paste0("rank=",format(unif.div.cov[[i]][jj,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.div.cov[[i]][jj,3],scientific=TRUE,digits=3))
    #     #         int.mtext <- str_replace_all(int.mtext,"e","x10^")
    #     #         cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
    #     #         int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
    #     #         cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
    #     #         mtext(text=int.mtext,side=1,line=5,cex=1.5) 
    #     #         mtext(text=cov.mtext,side=1,line=6,cex=1.5) 
    #     #       }
    #     #     }
    #     #   }
    #     # }
    #   }
    # } else {
      
      if (plot.save==TRUE){
        # First plot the distance summaries
        p1 <- ggplot(sum.df,aes(Mesh,MeanDist)) + facet_wrap(~sum.df$Grid) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Mesh))-0.5,xmax=max(as.numeric(sum.df$Mesh))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Mesh)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
        p2 <- ggplot(sum.df,aes(Grid,MeanDist)) + facet_wrap(~sum.df$Mesh) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Grid))-0.5,xmax=max(as.numeric(sum.df$Grid))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Grid)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
        print(p1)
        print(p2)
        ggsave(paste0(plot.name[1],"byGrid_Relabel.pdf"),plot=p1,width = 15,height = 10,device = "pdf")
        ggsave(paste0(plot.name[1],"byMesh_Relabel.pdf"),plot=p2,width = 15,height = 10,device = "pdf")
        
        # Second plot the outside of bounds summaries
        p1 <- ggplot(sum.df,aes(Mesh,OutsideBounds)) + facet_wrap(~sum.df$Grid) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Mesh))-0.5,xmax=max(as.numeric(sum.df$Mesh))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Mesh)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
        p2 <- ggplot(sum.df,aes(Grid,OutsideBounds)) + facet_wrap(~sum.df$Mesh) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Grid))-0.5,xmax=max(as.numeric(sum.df$Grid))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Grid)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
        print(p1)
        print(p2)
        ggsave(paste0(plot.name[2],"byGrid_Relabel.pdf"),plot=p1,width = 15,height = 10,device = "pdf")
        ggsave(paste0(plot.name[2],"byMesh_Relabel.pdf"),plot=p2,width = 15,height = 10,device = "pdf")
        
        
        # Plot the sbc runs that had problems
        
        # if (worst.plot=="grid"){
        #   for (ii in 1:l.g){
        #     ss <- strsplit(plot.name[3], "[.]")
        #     pdf(paste0(ss[[1]][1],levels.g[ii],"_Relabel.pdf"),h=10,w=15,pointsize=14)
        #     
        #     par(oma=opar$oma+c(5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
        #     
        #     ind <- grep(names(unif.check),pattern = paste0(unique(grids)[ii]))#,"$"))
        #     sel.plot.df.sub <- sel.plot$order.df
        #     sel.plot.df.sub <- sel.plot.df.sub[sel.plot.df.sub$name.ind%in%ind,]
        #     which.plot.i <- sel.plot.df.sub$name.ind[1:plot.cap]
        #     which.plot.j <- sel.plot.df.sub$param.ind[1:plot.cap]
        #     
        #     Grid_Mesh <- paste0(grids.names," and ",meshes.names)
        #     
        #     for (k in 1:length(which.plot.i)){
        #       i <- which.plot.i[k]; j <- which.plot.j[k]
        #       jj <- which(unif.div[[i]]==j)
        #       plot.ranks(N,L,rank.values[[i]][[j]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(names.f[[i]][j]), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
        #       points(as.numeric(as.character(freq.tabs[[i]][[j]]$ranks)),y[[i]][[jj]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
        #       int.mtext <- paste0("Int=",format(unif.div.cov[[i]][jj,1],scientific=TRUE,digits=3))
        #       cov.mtext <- paste0("rank=",format(unif.div.cov[[i]][jj,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.div.cov[[i]][jj,3],scientific=TRUE,digits=3))
        #       int.mtext <- str_replace_all(int.mtext,"e","x10^")
        #       cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
        #       int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
        #       cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
        #       mtext(text=int.mtext,side=1,line=5,cex=1.5)
        #       mtext(text=cov.mtext,side=1,line=6,cex=1.5)
        #     }
        #     
        #     dev.off()
        #   }
        # } else {
        #   for (ii in 1:l.m){
        #     ss <- strsplit(plot.name[3], "[.]")
        #     pdf(paste0(ss[[1]][1],levels.m[ii],"_Relabel.pdf"),h=10,w=15,pointsize=14)
        #     
        #     par(oma=opar$oma+c(5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
        #     
        #     ind <- grep(names(unif.check),pattern = paste0(unique(meshes)[ii],"$"))
        #     sel.plot.df.sub <- sel.plot$order.df
        #     sel.plot.df.sub <- sel.plot.df.sub[sel.plot.df.sub$name.ind%in%ind,]
        #     which.plot.i <- sel.plot.df.sub$name.ind[1:plot.cap]
        #     which.plot.j <- sel.plot.df.sub$param.ind[1:plot.cap]
        #     
        #     Grid_Mesh <- paste0(grids.names," and ",meshes.names)
        #     
        #     for (k in 1:length(which.plot.i)){
        #       i <- which.plot.i[k]; j <- which.plot.j[k]
        #       jj <- which(unif.div[[i]]==j)
        #       plot.ranks(N,L,rank.values[[i]][[j]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(names.f[[i]][j]), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
        #       points(as.numeric(as.character(freq.tabs[[i]][[j]]$ranks)),y[[i]][[jj]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
        #       int.mtext <- paste0("Int=",format(unif.div.cov[[i]][jj,1],scientific=TRUE,digits=3))
        #       cov.mtext <- paste0("rank=",format(unif.div.cov[[i]][jj,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.div.cov[[i]][jj,3],scientific=TRUE,digits=3))
        #       int.mtext <- str_replace_all(int.mtext,"e","x10^")
        #       cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
        #       int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
        #       cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
        #       mtext(text=int.mtext,side=1,line=5,cex=1.5) 
        #       mtext(text=cov.mtext,side=1,line=6,cex=1.5)
        #     }
        #     dev.off()
        #   }
        # }
      } else {
        # First plot the distance summaries
        p1 <- ggplot(sum.df,aes(Mesh,MeanDist)) + facet_wrap(~sum.df$Grid) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Mesh))-0.5,xmax=max(as.numeric(sum.df$Mesh))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Mesh)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
        p2 <- ggplot(sum.df,aes(Grid,MeanDist)) + facet_wrap(~sum.df$Mesh) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Grid))-0.5,xmax=max(as.numeric(sum.df$Grid))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Grid)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
        print(p1)
        print(p2)
        
        # Second plot the outside of bounds summaries
        p4 <- ggplot(sum.df,aes(Mesh,OutsideBounds)) + facet_wrap(~sum.df$Grid) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Mesh))-0.5,xmax=max(as.numeric(sum.df$Mesh))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Mesh)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
        p5 <- ggplot(sum.df,aes(Grid,OutsideBounds)) + facet_wrap(~sum.df$Mesh) + geom_rect(data=sum.df,xmin=min(as.numeric(sum.df$Grid))-0.5,xmax=max(as.numeric(sum.df$Grid))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_boxplot(size=1.5,aes(color=Grid)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size = 20),legend.position = "none")
        print(p4)
        print(p5)
        
        
        # Plot the sbc runs that had problems
        
        # if (worst.plot=="grid"){
        #   for (ii in 1:l.g){
        #     sel.plot.df.sub <- sel.plot$order.df
        #     which.plot.i <- sel.plot.df.sub$name.ind[1:plot.cap]
        #     which.plot.j <- sel.plot.df.sub$param.ind[1:plot.cap]
        #     
        #     par(oma=opar$oma+c(5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
        #     
        #     Grid_Mesh <- paste0(grids.names," and ",meshes.names)
        #     
        #     for (k in 1:length(which.plot.i)){
        #       i <- which.plot.i[k]; j <- which.plot.j[k]
        #       jj <- which(unif.div[[i]]==j)
        #       plot.ranks(N,L,rank.values[[i]][[j]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(names.f[[i]][j]), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
        #       points(as.numeric(as.character(freq.tabs[[i]][[j]]$ranks)),y[[i]][[jj]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
        #       int.mtext <- paste0("Int=",format(unif.div.cov[[i]][jj,1],scientific=TRUE,digits=3))
        #       cov.mtext <- paste0("rank=",format(unif.div.cov[[i]][jj,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.div.cov[[i]][jj,3],scientific=TRUE,digits=3))
        #       int.mtext <- str_replace_all(int.mtext,"e","x10^")
        #       cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
        #       int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
        #       cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
        #       mtext(text=int.mtext,side=1,line=5,cex=1.5) 
        #       mtext(text=cov.mtext,side=1,line=6,cex=1.5) 
        #     }
        #   }
        # } else {
        #   for (ii in 1:l.m){
        #     ind <- grep(names(unif.check),pattern = paste0(unique(meshes)[ii],"$"))
        #     sel.plot.df.sub <- sel.plot$order.df
        #     sel.plot.df.sub <- sel.plot.df.sub[sel.plot.df.sub$name.ind%in%ind,]
        #     which.plot.i <- sel.plot.df.sub$name.ind[1:plot.cap]
        #     which.plot.j <- sel.plot.df.sub$param.ind[1:plot.cap]
        #     
        #     par(oma=opar$oma+c(5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
        #     
        #     Grid_Mesh <- paste0(grids.names," and ",meshes.names)
        #     
        #     for (k in 1:length(which.plot.i)){
        #       i <- which.plot.i[k]; j <- which.plot.j[k]
        #       jj <- which(unif.div[[i]]==j)
        #       plot.ranks(N,L,rank.values[[i]][[j]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(names.f[[i]][j]), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
        #       points(as.numeric(as.character(freq.tabs[[i]][[j]]$ranks)),y[[i]][[jj]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
        #       int.mtext <- paste0("Int=",format(unif.div.cov[[i]][jj,1],scientific=TRUE,digits=3))
        #       cov.mtext <- paste0("rank=",format(unif.div.cov[[i]][jj,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.div.cov[[i]][jj,3],scientific=TRUE,digits=3))
        #       int.mtext <- str_replace_all(int.mtext,"e","x10^")
        #       cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
        #       int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
        #       cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
        #       mtext(text=int.mtext,side=1,line=5,cex=1.5) 
        #       mtext(text=cov.mtext,side=1,line=6,cex=1.5) 
        #     }
        #   }
        # }
      }
    # }
  }
  
  return(list("summary"=sum.df)) #,"unif.models"=unif.check,"unif.divergence.ind"=unif.div))
}
