
# SBC Fixed Effects Irreg Pol Wrapper Function: Relabelled ----------------

# This is a wrapper function that will produce summaries for the frequencies of the ranks for SBC as well as assessing the uniformity of the frequencies.
# This function is for considering the fixed effects and hyperparameters of a spatial model, for latent effects, there will be another function to perform similarly.
# This R script contains the functions that produce some of the plots with the altered axis labels, output with suffix *_Relabel.pdf. Any plots that we don't want to reproduce with different labels are commented out so that they aren't unnecessarily saved.

# Author: Nadeen Khaleel

# Libraries ---------------------------------------------------------------

library(purrr)
library(ggplot2)
library(rlist)
library(magrittr)
library(grid)
library(gridExtra)
library(stringr)
library(tidyverse)
library(scales)
library(rlang)

opar <- par()

# Sub-functions -----------------------------------------------------------

plot.ranks <- function(N,L,ranks,main,cexmain,cexlab,cexaxis){
  d1 <- qbinom(0.005,N,1/(L+1))
  d2 <- qbinom(0.995,N,1/(L+1))
  m <- N/(L+1)
  xcoord <- c(-0.5,L+0.5,L+0.5,-0.5)
  ycoord <- c(d1,d1,d2,d2)
  
  hist(ranks,breaks=seq(-0.5,L+0.5),main=main,xlab = "Rank Statistic",cex.main=cexmain,cex.lab=cexlab,cex.axis=cexaxis)
  polygon(xcoord,ycoord,col=rgb(0.5,0,0.5,0.25),border = NA)
  segments(-0.5,m,L+0.5,m,col="blue")
}

freq.func <- function(x){
  df <- as.data.frame(table(x))
  names(df) <- c("ranks","freq")
  return(df)
}

summary.stats <- function(N,L,freq,m,lower.bound,upper.bound){
  out <- (sum(freq<lower.bound)+sum(freq>upper.bound))/(L+1) # L+1 ranks {0,...,L}
  dist <- sum(sqrt((freq-m)^2))
  return(list("outside.bounds"=out,"dist.sum"=dist))
}

models_sbc <- function(freq.ranks,test="Chisq"){
  
  if (test=="Chisq"){
    ranks.df <- data.frame(Freq=freq.ranks$freq,ranks=(as.numeric(as.character(freq.ranks$ranks))))
    fit.const <- glm(Freq~1,data = ranks.df, family = "poisson")
    fit.ranks <- glm(Freq~ranks,data = ranks.df, family = "poisson")
    fit.ranks2 <- glm(Freq~ranks+I(ranks^2),data = ranks.df, family = "poisson")
    
    a1 <- anova(fit.const,fit.ranks,test = "Chisq")
    a2 <- anova(fit.const,fit.ranks2,test = "Chisq")
    
    
    if (a1$`Pr(>Chi)`[2]>0.01){ # prefer const over rank
      if (a2$`Pr(>Chi)`[2]>0.01){ # prefer const over rank^2
        u.divergence <- "FALSE"
        mod <- fit.const
      } else { # prefer rank^2 over const - therefore as rank < const < rank^2
        u.divergence <- "TRUE"
        mod <- fit.ranks2
      }
    } else { # prefer rank over const
      if (a2$`Pr(>Chi)`[2]>0.01){ # prefer const over rank^2 - therefore rank^2 < const < rank
        u.divergence <- "TRUE"
        mod <- fit.ranks
      } else {
        a3 <- anova(fit.ranks,fit.ranks2,test = "Chisq")
        if (a3$`Pr(>Chi)`[2]<0.01){
          u.divergence <- "TRUE"
          mod <- fit.ranks2
        } else {
          u.divergence <- "TRUE"
          mod <- fit.ranks
        }
      }
    }
  } else {
    
    # OR - use step() function and check dim?
    full.fit <- glm(Freq~ranks+I(ranks^2),data = ranks.df, family = "poisson")
    mod <- step(full.fit)
    if (length(final.fit$coefficients)==1){
      u.divergence=="FALSE"
    } else {
      u.divergence=="TRUE"
    }
  }
  return(list("div"=u.divergence,"fit.mod"=mod))
}

rank.vals <- function(prior.vals,dap.samps){rowSums(dap.samps<prior.vals)}

# One go-round
run.same <- function(N,L,mu,sigma){
  prior.vals <- rnorm(N)
  dap.samps <- matrix(rnorm(N*L),nrow=N,byrow=TRUE)
  ranks <- rank.vals(prior.vals,dap.samps)
}


# Main Function -----------------------------------------------------------


sbc_checks_param <- function(N,L,rank.values,run,param.names,grids,meshes,plots=FALSE,plot.save=FALSE,plot.name=c("IrregPolsumdisttest_param.pdf","IrregPoloutsideboundstest_param.pdf","IrregPolsbcdivergences_param","IrregPolsbcnondivergences_param")){
  # N - total number of simulations
  # L - total number of posterior samples
  # rank.values - the data output from the simulation study, extracting only the parameter results
  # grids - character string for names of the grids for each list item of rank.values
  # meshes - character string for names of the meshes for each list item of rank.values
  # plots - want to produce plots for the different summaries and the SBC output?
  # plot.save - save plots or print only?
  # plot.name - if you want the plots to be saved, provide a vector of length 3, each element in is a character string that you want the plots to be saved as: one for the proportion outside the boundary, another for average distance from mean and the final is the plots of the SBC histograms.
  
  # Want grids and meshes for each rank.values item (regardless of repetition), if this is not the case, create the lists, where hold grid fixed and go down meshes first..
  if (length(grids)!=length(rank.values)){
    if (length(rank.values)%%length(grids)!=0){
      stop("grids must be vector of character string of grid label for each rank.values")
    }
    grids <- rep(grids,each=length(unique(meshes)))
  }
  if (length(meshes)!=length(rank.values)){
    meshes <- rep(meshes,length(unique(grids)))
  }
  
  r.n <- length(rank.values)
  l.g <- length(unique(grids))
  l.m <- length(unique(meshes))
  
  names.gm <- names(rank.values)
  
  grids.names <- grids
  ord.grids.names <- unique(grids.names)
  for (i in 1:l.g){
    grids.names <- str_replace(grids.names,ord.grids.names[i],paste0("Grid ",i))
  }
  meshes.names <- meshes
  ord.meshes.names <- unique(meshes.names)[order(unique(meshes.names),decreasing = T)]
  for (i in 1:l.m){
    meshes.names <- str_replace(meshes.names,ord.meshes.names[i],paste0("Mesh ",i)) # assume grid and mesh labels are in order
  }
  
  lower.bound <- qbinom(0.005,N,1/(L+1))
  upper.bound <- qbinom(0.995,N,1/(L+1))
  m <- N/(L+1)
  
  # As a comparison for the stats, sample from the SAME distribution
  mult.same.ranks <- replicate(1e3,run.same(N,L))
  freq.multsame.ranks <- apply(mult.same.ranks,2,freq.func)
  below.multsame.ranks <- unlist(lapply(1:1e3,FUN=function(i){sum(freq.multsame.ranks[[i]]$freq<lower.bound)}))
  above.multsame.ranks <- unlist(lapply(1:1e3,FUN=function(i){sum(freq.multsame.ranks[[i]]$freq>upper.bound)}))
  dist.multsame.ranks <- unlist(lapply(1:1e3,FUN=function(i){sum(sqrt((freq.multsame.ranks[[i]]$freq-m)^2))}))
  outside.multsame.ranks <- (below.multsame.ranks+above.multsame.ranks)/(L+1) # L+1 ranks {0,...,L}
  
  ave.outside.ranks <- mean(outside.multsame.ranks)
  ave.dist.ranks <- mean(dist.multsame.ranks)
  
  sd.outside.ranks <- sd(outside.multsame.ranks)
  sd.dist.ranks <- sd(dist.multsame.ranks)
  
  f.n <- dim(rank.values[[1]])[2] # how many ``parameters'' [f(theta) terms are we considering]
  if (is.null(f.n)){
    f.n <- length(rank.values[[1]])
  }
  names.f <- param.names # parameter names
  
  sub.freq.tabs <- function(x,param.names){
    r <- rank.values[[x]]
    out <- lapply(1:f.n,function(i){freq.func(r[[i]])})
    names(out) <- paste0("Ranks_",param.names)
    out
  }
  
  freq.tabs <- lapply(1:r.n,function(x,param.names){sub.freq.tabs(x,param.names)},names.f)
  names(freq.tabs) <- names.gm
  
  
  sub.summary <- function(x,param.names){
    f <- freq.tabs[[x]]
    out <- sapply(1:f.n,function(i){summary.stats(N,L,f[[i]]$freq,m,lower.bound,upper.bound)})
    colnames(out) <- param.names
    out
  }
  summaries <- lapply(1:r.n,function(x,param.names){sub.summary(x,param.names)},names.f)
  names(summaries) <- names.gm
  
  OutsideBounds <- unname(unlist(summaries %>% map(~.x[1, ]))) # first row is outside.bounds
  SumDist <- unname(unlist(summaries %>% map(~.x[2, ]))) # second row is dist.sum
  
  sum.df <- data.frame(Grid=rep(grids.names,each=f.n),Mesh=rep(meshes.names,each=f.n),Parameter=rep(names.f,l.g*l.m),OutsideBounds=OutsideBounds,SumDist=SumDist)
  sum.df$Grid <- factor(as.character(sum.df$Grid),levels=unique(sum.df$Grid)[order(unique(sum.df$Grid),decreasing=FALSE)])
  sum.df$Mesh <- factor(as.character(sum.df$Mesh),levels=unique(sum.df$Mesh)[order(unique(sum.df$Mesh),decreasing=FALSE)])
  
  # Setting up labels for the plots
  var.lab.orig <- as.character(sum.df$Parameter)
  num <- as.numeric(str_extract(var.lab.orig, "[0-9]+"))
  char <- str_extract(var.lab.orig, "[aA-zZ]+")
  
  plot.x.lab <- char
  plot.x.lab
  for (i in which(!is.na(num))){
    plot.x.lab[i] <- paste0(plot.x.lab[i],"[",num[i],"]")
  }
  plot.x.lab
  
  if (sum(plot.x.lab=="Int")>0){
    plot.x.lab[plot.x.lab=="Int"] <- "beta[0]"
  }
  sum.df$Label <- plot.x.lab

  
  sub.models_sbc <- function(x,param.names){
    f <- freq.tabs[[x]]
    out <- lapply(1:f.n,function(i){models_sbc(f[[i]])})
    names(out) <- param.names
    out
  }
  
  # unif.check <- lapply(1:r.n,function(x,param.names){sub.models_sbc(x,param.names)},names.f)
  # names(unif.check) <- names.gm
  
  
  # Original par settings for plots
  opar <- par()
  
  if (plots==TRUE){
    
    # Data Frames for plotting
    # unif.div <- lapply(1:r.n,function(i){list.which(unif.check[[i]],div=="TRUE")})
    # unif.ndiv <- lapply(1:r.n,function(i){list.which(unif.check[[i]],div=="FALSE")}) # WANT TO PLOT NON-DIVERGENCES FOR PARAMETERS TOO
    # names(unif.div) <- names.gm
    # names(unif.ndiv) <- names.gm
    # 
    # if (sum(lengths(unif.div))==0){
    #   print("No detected divergences from uniformity within SBC.")
    #   # Will want to plot the histograms for all as well.
    #   unif.ndiv.cov <- list()
    #   unif.ndiv.cov <- lapply(1:r.n,function(i){unif.ndiv.cov[[i]] <- matrix(rep(NA,3*length(unif.ndiv[[i]])),ncol=3)})
    #   unif.ndiv.cov <- lapply(1:r.n,function(x){
    #     uc <- unif.check[[x]]; ud <- unif.ndiv[[x]]
    #     
    #     if (length(ud)!=0){
    #       for (i in 1:length(ud)){
    #         unif.ndiv.cov[[x]][i,1:length(uc[[ud[i]]]$fit.mod$coefficients)] <- uc[[ud[i]]]$fit.mod$coefficients
    #       }
    #       unif.ndiv.cov[[x]] <- round(unif.ndiv.cov[[x]],8)
    #     }
    #   })
    #   
    #   nsub.fitted_values <- function(x,param.names){
    #     ud <- unif.ndiv[[x]]; uc <- unif.check[[x]]
    #     out <- lapply(ud,function(i){uc[[i]]$fit.mod$fitted.values})
    #     names(out) <- param.names[ud]
    #     out
    #   }
    #   y.n <- lapply(1:r.n,function(x,param.names){nsub.fitted_values(x,param.names)},names.f)
    #   names(y.n) <- names.gm
    # } else {
    #   unif.div.cov <- list()
    #   unif.div.cov <- lapply(1:r.n,function(i){unif.div.cov[[i]] <- matrix(rep(NA,3*length(unif.div[[i]])),ncol=3)})
    #   unif.div.cov <- lapply(1:r.n,function(x){
    #     uc <- unif.check[[x]]; ud <- unif.div[[x]]
    #     
    #     if (length(ud)!=0){
    #       for (i in 1:length(ud)){
    #         unif.div.cov[[x]][i,1:length(uc[[ud[i]]]$fit.mod$coefficients)] <- uc[[ud[i]]]$fit.mod$coefficients
    #       }
    #       unif.div.cov[[x]] <- round(unif.div.cov[[x]],8)
    #     }
    #   })
    #   
    #   sub.fitted_values <- function(x,param.names){
    #     ud <- unif.div[[x]]; uc <- unif.check[[x]]
    #     out <- lapply(ud,function(i){uc[[i]]$fit.mod$fitted.values})
    #     names(out) <- param.names[ud]
    #     out
    #   }
    #   y <- lapply(1:r.n,function(x,param.names){sub.fitted_values(x,param.names)},names.f)
    #   names(y) <- names.gm
    #   
    #   # Will want to plot the histograms for all non-divergent parameters as well.
    #   unif.ndiv.cov <- list()
    #   unif.ndiv.cov <- lapply(1:r.n,function(i){unif.ndiv.cov[[i]] <- matrix(rep(NA,3*length(unif.ndiv[[i]])),ncol=3)})
    #   unif.ndiv.cov <- lapply(1:r.n,function(x){
    #     uc <- unif.check[[x]]; ud <- unif.ndiv[[x]]
    #     
    #     if (length(ud)!=0){
    #       for (i in 1:length(ud)){
    #         unif.ndiv.cov[[x]][i,1:length(uc[[ud[i]]]$fit.mod$coefficients)] <- uc[[ud[i]]]$fit.mod$coefficients
    #       }
    #       unif.ndiv.cov[[x]] <- round(unif.ndiv.cov[[x]],8)
    #     }
    #   })
    #   
    #   nsub.fitted_values <- function(x,param.names){
    #     ud <- unif.ndiv[[x]]; uc <- unif.check[[x]]
    #     out <- lapply(ud,function(i){uc[[i]]$fit.mod$fitted.values})
    #     names(out) <- param.names[ud]
    #     out
    #   }
    #   y.n <- lapply(1:r.n,function(x,param.names){nsub.fitted_values(x,param.names)},names.f)
    #   names(y.n) <- names.gm
    # }
    
    outside.ylim <- rep(ave.outside.ranks) + c(-1,1)*2*sd.outside.ranks
    outside.ylim[1] <- max(0,outside.ylim[1])
    dist.ylim <- rep(ave.dist.ranks) + c(-1,1)*2*sd.dist.ranks
    
    crit.df <- data.frame(Grid=rep(unique(grids.names),each=8*N),Mesh=rep(rep(unique(meshes.names),each=(2*N)),4),Criterion=rep(rep(c("WAIC","DIC"),each=N),16))
    crit.df$Value <- rep(NA,nrow(crit.df))
    start.ind <- 1 # need to account for grid mesh
    for (i in 1:length(run)){
      for (j in 1:length(run[[i]])){
        crit.df$Value[start.ind:(start.ind+length(run[[i]][[j]]$run.df$waic)-1)] <- run[[i]][[j]]$run.df$waic
        start.ind <- start.ind + length(run[[i]][[j]]$run.df$waic)
        crit.df$Value[start.ind:(start.ind+length(run[[i]][[j]]$run.df$dic)-1)] <- run[[i]][[j]]$run.df$dic
        start.ind <- start.ind + length(run[[i]][[j]]$run.df$dic)
      }
    }
    crit.df$Grid <- as.factor(crit.df$Grid)
    crit.df$Mesh <- factor(as.character(crit.df$Mesh),levels=unique(crit.df$Mesh)[order(unique(crit.df$Mesh),decreasing=FALSE)])
    crit.df$Criterion <- as.factor(crit.df$Criterion)
    crit.df$Value <- as.numeric(crit.df$Value)
    
    critsum.df <- crit.df
    critsum.df %<>%
      group_by(Grid,Mesh,Criterion) %>%
      summarise(mean = mean(Value,na.rm=TRUE),sd=sd(Value,na.rm=TRUE), n = n())
    
    
    err.df <- data.frame(Grid=rep(unique(grids.names),each=12*N),Mesh=rep(rep(unique(meshes.names),each=(3*N)),4),Criterion=rep(rep(c("Error","Warning","FFT"),each=N),16),Run=rep(1:N,3*4*4))
    err.df$Value <- rep(NA,nrow(err.df))
    start.ind <- 1
    for (i in 1:length(run)){
      for (j in 1:length(run[[i]])){
        err <- run[[i]][[j]]$mess.ls$error
        err[!is.na(err)] <- 1
        err[is.na(err)] <- 0
        warn <- run[[i]][[j]]$mess.ls$warning
        warn[!is.na(warn)] <- 1
        warn[is.na(warn)] <- 0
        fft <- run[[i]][[j]]$mess.ls$FFT
        err.df$Value[start.ind:(start.ind+length(err)-1)] <- err
        start.ind <- start.ind + length(err)
        err.df$Value[start.ind:(start.ind+length(warn)-1)] <- warn
        start.ind <- start.ind + length(warn)
        err.df$Value[start.ind:(start.ind+length(fft)-1)] <- fft
        start.ind <- start.ind + length(fft)
      }
    }
    err.df$Grid <- as.factor(err.df$Grid)
    err.df$Mesh <- factor(as.character(err.df$Mesh),levels=unique(err.df$Mesh)[order(unique(err.df$Mesh),decreasing=FALSE)])
    err.df$Criterion <- as.factor(err.df$Criterion)
    err.df$Value <- as.numeric(err.df$Value)
    
    errsum.df <- err.df
    errsum.df %<>%
      group_by(Grid,Mesh,Criterion) %>%
      summarise(mean = mean(Value,na.rm=TRUE),sd=sd(Value,na.rm=TRUE), n = n())
    
    fft.ind <- sapply(1:length(run),function(i){which(run[[i]]$mess.ls$FFT > 5)})
    theta.val.fft <- lapply(1:length(run),function(i){true.theta.final[fft.ind[[i]],]})
    err.ind <- sapply(1:length(run),function(i){which(!is.na(run[[i]]$mess.ls$error))})
    theta.val.err <- lapply(1:length(run),function(i){true.theta.final[err.ind[[i]],]})
    
    
    if (plot.save==TRUE){
      # First plost the distance summaries
      p1 <- ggplot(sum.df,aes(x=Label,y=SumDist)) + scale_x_discrete(labels = label_parse()) + facet_wrap(~Grid) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(as.factor(sum.df$Parameter)))-0.5,xmax=max(as.numeric(as.factor(sum.df$Parameter)))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_point(size=5,aes(color=as.factor(Mesh))) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size=30),legend.title=element_text(size=25),legend.text=element_text(size=20)) + labs(color = "Mesh")
      p2 <- ggplot(sum.df,aes(x=Label,y=SumDist)) + scale_x_discrete(labels = label_parse()) + facet_wrap(~Mesh) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(as.factor(sum.df$Parameter)))-0.5,xmax=max(as.numeric(as.factor(sum.df$Parameter)))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_point(size=5,aes(color=as.factor(Grid))) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size=30),legend.title=element_text(size=25),legend.text=element_text(size=20)) + labs(color = "Grid")
      p1
      p2
      print(p1)
      print(p2)
      ggsave(paste0(plot.name[1],"_byGrid_Relabel.pdf"),plot=p1,width = 15,height = 10,device = "pdf")
      ggsave(paste0(plot.name[1],"_byMesh_Relabel.pdf"),plot=p2,width = 15,height = 10,device = "pdf")
      
      # Second plot the outside of bounds summaries
      p1 <- ggplot(sum.df,aes(x=Label,y=OutsideBounds)) + scale_x_discrete(labels = label_parse()) + facet_wrap(~Grid) + geom_rect(data=sum.df,xmin=min(as.numeric(as.factor(sum.df$Parameter)))-0.5,xmax=max(as.numeric(as.factor(sum.df$Parameter)))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_point(size=5,aes(color=Mesh)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size=30),legend.title=element_text(size=25),legend.text=element_text(size=20)) + labs(color = "Mesh")
      p2 <- ggplot(sum.df,aes(x=Label,y=OutsideBounds)) + scale_x_discrete(labels = label_parse()) + facet_wrap(~Mesh) + geom_rect(data=sum.df,xmin=min(as.numeric(as.factor(sum.df$Parameter)))-0.5,xmax=max(as.numeric(as.factor(sum.df$Parameter)))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_point(size=5,aes(color=Grid)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size=30),legend.title=element_text(size=25),legend.text=element_text(size=20)) + labs(color = "Grid")
      p1
      p2
      print(p1)
      print(p2)
      ggsave(paste0(plot.name[2],"_byGrid_Relabel.pdf"),plot=p1,width = 15,height = 10,device="pdf")
      ggsave(paste0(plot.name[2],"_byMesh_Relabel.pdf"),plot=p2,width = 15,height = 10,device="pdf")
      
      # Plot the sbc runs that had problems
      # if (sum(lengths(unif.div)==0)){
      #   print("No detected divergences from uniformity within SBC so no plot produced.")
      #   pdf(paste0(plot.name[4],"_Relabel.pdf"),h=10,w=15,pointsize=14)
      #   par(oma=opar$oma+c(2.5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
      #   
      #   Grid_Mesh <- paste0(grids.names," and ", meshes.names)
      #   
      #   names.plot.param <- char
      #   names.plot.param
      #   for (i in which(!is.na(num))){
      #     names.plot.param[i] <- paste0(names.plot.param[i],"[.parse_expr(",num[i],")]")
      #   }
      #   
      #   if (sum(names.plot.param=="int")>0){
      #     names.plot.param[names.plot.param=="int"] <- "beta[0]"
      #   }
      #   
      #   for (i in 1:length(unif.ndiv)){
      #     if (length(unif.ndiv[[i]])!=0){
      #       for (j in 1:length(unif.ndiv[[i]])){
      #         param.plot.name <- names.plot.param[unif.ndiv[[i]][j]]
      #         plot.ranks(N,L,rank.values[[i]][[unif.ndiv[[i]][j]]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(parse_expr(param.plot.name)), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
      #         points(as.numeric(as.character(freq.tabs[[i]][[unif.ndiv[[i]][j]]]$ranks)),y.n[[i]][[j]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
      #         int.mtext <- paste0("Int=",format(unif.ndiv.cov[[i]][j,1],scientific=TRUE,digits=3))
      #         cov.mtext <- paste0("rank=",format(unif.ndiv.cov[[i]][j,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.ndiv.cov[[i]][j,3],scientific=TRUE,digits=3))
      #         int.mtext <- str_replace_all(int.mtext,"e","x10^")
      #         cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
      #         int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
      #         cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
      #         mtext(text=int.mtext,side=1,line=5,cex=1.5)
      #         mtext(text=cov.mtext,side=1,line=6,cex=1.5)  
      #       } 
      #     }
      #   }
      #   dev.off()
      # } else {
      #   pdf(paste0(plot.name[3],"_Relabel.pdf"),h=10,w=15,pointsize=14)
      #   if (sum(lengths(unif.div))==1){
      #     par(oma=opar$oma+c(2.5,0,0,0),mar=opar$mar+c(0,1,0,0))
      #   } else if (sum(lengths(unif.div))>1){
      #     par(oma=opar$oma+c(2.5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
      #   }
      #   
      #   Grid_Mesh <- paste0(grids.names," and ", meshes.names)
      #   
      #   names.plot.param <- names.f
      #   num <- as.numeric(str_extract(names.plot.param, "[0-9]+"))
      #   char <- str_to_lower(str_extract(names.plot.param, "[aA-zZ]+"))
      #   
      #   names.plot.param <- char
      #   names.plot.param
      #   for (i in which(!is.na(num))){
      #     names.plot.param[i] <- paste0(names.plot.param[i],"[",num[i],"]")
      #   }
      #   
      #   if (sum(names.plot.param=="int")>0){
      #     names.plot.param[names.plot.param=="int"] <- "beta[0]"
      #   }
      #   
      #   for (i in 1:length(unif.div)){
      #     if (length(unif.div[[i]])!=0){
      #       for (j in 1:length(unif.div[[i]])){
      #         param.plot.name <- names.plot.param[unif.div[[i]][j]]
      #         plot.ranks(N,L,rank.values[[i]][[unif.div[[i]][j]]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(parse_expr(param.plot.name)), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
      #         points(as.numeric(as.character(freq.tabs[[i]][[unif.div[[i]][j]]]$ranks)),y[[i]][[j]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
      #         int.mtext <- paste0("Int=",format(unif.div.cov[[i]][j,1],scientific=TRUE,digits=3))
      #         cov.mtext <- paste0("rank=",format(unif.div.cov[[i]][j,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.div.cov[[i]][j,3],scientific=TRUE,digits=3))
      #         int.mtext <- str_replace_all(int.mtext,"e","x10^")
      #         cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
      #         int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
      #         cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
      #         mtext(text=int.mtext,side=1,line=5,cex=1.5)
      #         mtext(text=cov.mtext,side=1,line=6,cex=1.5)  
      #       } 
      #     }
      #   }
      #   dev.off()
      #   
      #   pdf(paste0(plot.name[4],"_Relabel.pdf"),h=10,w=15,pointsize=14)
      #   # Non-divergences too
      #   if (sum(lengths(unif.ndiv))==1){
      #     par(oma=opar$oma+c(2.5,0,0,0),mar=opar$mar+c(0,1,0,0))
      #   } else if (sum(lengths(unif.ndiv))>1){
      #     par(oma=opar$oma+c(2.5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
      #   }
      #   
      #   Grid_Mesh <- paste0(grids.names," and ", meshes.names)
      #   
      #   names.plot.param <- names.f
      #   num <- as.numeric(str_extract(names.plot.param, "[0-9]+"))
      #   char <- str_to_lower(str_extract(names.plot.param, "[aA-zZ]+"))
      #   
      #   names.plot.param <- char
      #   names.plot.param
      #   for (i in which(!is.na(num))){
      #     names.plot.param[i] <- paste0(names.plot.param[i],"[",num[i],"]")
      #   }
      #   
      #   if (sum(names.plot.param=="int")>0){
      #     names.plot.param[names.plot.param=="int"] <- "beta[0]"
      #   }
      #   
      #   for (i in 1:length(unif.ndiv)){
      #     if (length(unif.ndiv[[i]])!=0){
      #       for (j in 1:length(unif.ndiv[[i]])){
      #         param.plot.name <- names.plot.param[unif.ndiv[[i]][j]]
      #         plot.ranks(N,L,rank.values[[i]][[unif.ndiv[[i]][j]]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(parse_expr(param.plot.name)), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
      #         points(as.numeric(as.character(freq.tabs[[i]][[unif.ndiv[[i]][j]]]$ranks)),y.n[[i]][[j]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
      #         int.mtext <- paste0("Int=",format(unif.ndiv.cov[[i]][j,1],scientific=TRUE,digits=3))
      #         cov.mtext <- paste0("rank=",format(unif.ndiv.cov[[i]][j,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.ndiv.cov[[i]][j,3],scientific=TRUE,digits=3))
      #         int.mtext <- str_replace_all(int.mtext,"e","x10^")
      #         cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
      #         int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
      #         cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
      #         mtext(text=int.mtext,side=1,line=5,cex=1.5)
      #         mtext(text=cov.mtext,side=1,line=6,cex=1.5)  
      #       } 
      #     }
      #   }
      #   dev.off()
      # }
      
      print(critsum.df,n=nrow(critsum.df))
      
      p3a <- ggplot(critsum.df) + geom_point(aes(x=Mesh,y=mean,col=Grid,size=2)) + xlab("Mesh") + facet_wrap(~Criterion) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20)) + guides(size=FALSE, color = guide_legend(override.aes = list(size = 3)))
      p3b <- ggplot(critsum.df) + geom_point(aes(x=Mesh,y=mean,col=Grid,size=2)) + xlab("Mesh") + facet_wrap(~Criterion) + ylim(c(4.5e2,7.2e5)) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20)) + guides(size=FALSE, color = guide_legend(override.aes = list(size = 3)))
      p3c <- ggplot(critsum.df) + geom_point(aes(x=Mesh,y=mean,col=Grid,size=2)) + xlab("Mesh") + facet_wrap(~Criterion) + ylim(c(4.5e2,3.6e4)) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20)) + guides(size=FALSE, color = guide_legend(override.aes = list(size = 3)))
      
      print(p3a)
      print(p3b)
      print(p3c)
      
      # ggsave("IrregPolLGCPSBCDICWAICMean100_Relabel.pdf",plot=p3a,width = 15,height = 10,device="pdf")
      # ggsave("IrregPolLGCPSBCDICWAICMean90_Relabel.pdf",plot=p3b,width = 15,height = 10,device="pdf")
      # ggsave("IrregPolLGCPSBCDICWAICMean75_Relabel.pdf",plot=p3c,width = 15,height = 10,device="pdf")
      
      p4err <- ggplot(err.df[err.df$Criterion=="Error",]) + geom_point(aes(x=Run,y=Value,col=Grid,size=2)) + facet_wrap(~Mesh+Grid,ncol = 4,scales="free_y") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
      p4war <- ggplot(err.df[err.df$Criterion=="Warning",]) + geom_point(aes(x=Run,y=Value,col=Grid,size=2)) + facet_wrap(~Mesh+Grid,ncol = 4,scales="free_y") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
      p4fft <- ggplot(err.df[err.df$Criterion=="FFT",]) + geom_point(aes(x=Run,y=Value,col=Grid,size=2)) + facet_wrap(~Mesh+Grid,ncol = 4,scales="free_y") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
      p5 <- ggplot(errsum.df) + geom_point(aes(x=Mesh,y=mean,col=Grid,size=2)) + xlab("Mesh") + facet_wrap(~Criterion,scales="free_y") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),axis.text.x=element_text(angle = 45, vjust = 1, hjust = 1),strip.text.x = element_text(size=20)) + guides(size=FALSE, color = guide_legend(override.aes = list(size = 3)))
      
      print(p4err)
      print(p4war)
      print(p4fft)
      print(p5)
      
      # ggsave("IrregPolLGCPSBCErr_Relabel.pdf",plot=p4err,width = 15,height = 10,device="pdf")
      # ggsave("IrregPolLGCPSBCWar_Relabel.pdf",plot=p4war,width = 15,height = 10,device="pdf")
      # ggsave("IrregPolLGCPSBCFFT_Relabel.pdf",plot=p4fft,width = 15,height = 10,device="pdf")
      # ggsave("IrregPolLGCPSBCErrMean_Relabel.pdf",plot=p5,width = 15,height = 10,device="pdf")
      
      col.ind <- c("red","blue","orange","magenta") # mesh 1,2,3,4
      # pdf("IrregPolLGCPSBCFFTThetaTrueValues_Relabel.pdf",h=10,w=15,pointsize=14)
      m <- matrix(c(1,2,3,4,5,6,7,7,7),nrow = 3,ncol = 3,byrow = TRUE)
      layout(mat = m,heights = c(0.4,0.4,0.2))
      for (i in 1:length(names.f)){
        par(mar = c(3,3,2,2))
        plot(true.theta.final[,i],pch=19,main=(names.f[i]),ylab=names.f[i])
        for (j in 1:length(theta.val.fft)){
          points(fft.ind[[j]],theta.val.fft[[j]][,i],pch=19,col=col.ind[j])
        }
      }
      plot(1, type = "n", axes=FALSE, xlab="", ylab="")
      legend("bottom",legend = names(run), text.width = max(sapply(names(run), strwidth)),col=col.ind, lwd=5, cex=1, horiz = TRUE)
      # dev.off()
      
      # pdf("IrregPolLGCPSBCErrorThetaTrueValues_Relabel.pdf",h=10,w=15,pointsize=14)
      m <- matrix(c(1,2,3,4,5,6,7,7,7),nrow = 3,ncol = 3,byrow = TRUE)
      layout(mat = m,heights = c(0.4,0.4,0.2))
      for (i in 1:length(names.f)){
        par(mar = c(3,3,2,2))
        plot(true.theta.final[,i],pch=19,main=(names.f[i]),ylab=names.f[i])
        for (j in 1:length(theta.val.err)){
          points(err.ind[[j]],theta.val.err[[j]][,i],pch=19,col=col.ind[j])
        }
      }
      plot(1, type = "n", axes=FALSE, xlab="", ylab="")
      legend("bottom",legend = names(run), text.width = max(sapply(names(run), strwidth)),col=col.ind, lwd=5, cex=1, horiz = TRUE)
      # dev.off()
      
      cols.fft <- matrix(rep(character(nrow(true.theta.final)),length(run)),ncol=length(run))
      cols.fft[] <- "black"
      for (i in 1:length(run)){
        cols.fft[fft.ind[[i]],i] <- col.ind[i]
      }
      mesh.names.title <- gsub("([a-z])([0-9])", "\\1 \\2", names(run))
      
      # pdf("IrregPolLGCPSBCFFTPairs_Relabel.pdf",h=10,w=15,pointsize=14)
      for (i in 1:length(run)){
        pairs(true.theta.final,col=cols.fft[,i],pch=19,upper.panel = NULL,main=paste0(mesh.names.title[i]," FFT"))
      }
      # dev.off()
      
      cols.err <- matrix(rep(character(nrow(true.theta.final)),length(run)),ncol=length(run))
      cols.err[] <- "black"
      for (i in 1:length(run)){
        cols.err[err.ind[[i]],i] <- col.ind[i]
      }
      
      # pdf("IrregPolLGCPSBCErrorPairs_Relabel.pdf",h=10,w=15,pointsize=14)
      for (i in 1:length(run)){
        pairs(true.theta.final,col=cols.err[,i],pch=19,upper.panel = NULL,main=paste0(mesh.names.title[i]," Error"))
      }
      # dev.off()
      
    } else {
      # First plot the distance summaries
      p1 <- ggplot(sum.df,aes(x=Label,y=SumDist)) + scale_x_discrete(labels = label_parse()) + facet_wrap(~Grid) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(as.factor(sum.df$Parameter)))-0.5,xmax=max(as.numeric(as.factor(sum.df$Parameter)))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_point(size=5,aes(color=Mesh)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size=30),legend.title=element_text(size=25),legend.text=element_text(size=20)) + labs(color = "Mesh")
      p2 <- ggplot(sum.df,aes(x=Label,y=SumDist)) + scale_x_discrete(labels = label_parse()) + facet_wrap(~Mesh) + scale_y_continuous(limits = c(dist.ylim[1], NA)) + geom_rect(data=sum.df,xmin=min(as.numeric(as.factor(sum.df$Parameter)))-0.5,xmax=max(as.numeric(as.factor(sum.df$Parameter)))+0.5,ymin=dist.ylim[1],ymax=dist.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.dist.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_point(size=5,aes(color=Grid)) + theme_bw() + ggtitle("Sum of Distances of Frequencies from Expected Average") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(D["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size=30),legend.title=element_text(size=25),legend.text=element_text(size=20)) + labs(color = "Grid")
      print(p1)
      print(p2)
      
      # Second plot the outside of bounds summaries
      p4 <- ggplot(sum.df,aes(x=Label,y=OutsideBounds)) + scale_x_discrete(labels = label_parse()) + facet_wrap(~Grid) + geom_rect(data=sum.df,xmin=min(as.numeric(as.factor(sum.df$Parameter)))-0.5,xmax=max(as.numeric(as.factor(sum.df$Parameter)))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_point(size=5,aes(color=Mesh)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size=30),legend.title=element_text(size=25),legend.text=element_text(size=20)) + labs(color = "Mesh")
      p5 <- ggplot(sum.df,aes(x=Label,y=OutsideBounds)) + scale_x_discrete(labels = label_parse()) + facet_wrap(~Mesh) + geom_rect(data=sum.df,xmin=min(as.numeric(as.factor(sum.df$Parameter)))-0.5,xmax=max(as.numeric(as.factor(sum.df$Parameter)))+0.5,ymin=outside.ylim[1],ymax=outside.ylim[2],fill="blue",alpha=0.01) + geom_hline(yintercept=rep(ave.outside.ranks,nrow(sum.df)),color="red",linetype="dashed",size=1.5) + geom_point(size=5,aes(color=Grid)) + theme_bw() + ggtitle("Proportion of Frequencies Outside Confidence Bounds") + xlab(expression(paste("f(",theta,")",sep=""))) + ylab(expression(O["i,j"])) + theme(plot.title = element_text(size=35),axis.title = element_text(size=35),axis.text = element_text(size=30),strip.text.x = element_text(size=30),legend.title=element_text(size=25),legend.text=element_text(size=20)) + labs(color = "Grid")
      print(p4)
      print(p5)
      
      
      # Plot the sbc runs that had problems
      # if (sum(lengths(unif.div)==0)){
        # print("No detected divergences from uniformity within SBC.")
        # 
        # par(oma=opar$oma+c(2.5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
        # 
        # Grid_Mesh <- paste0(grids.names," and ", meshes.names)
        # 
        # names.plot.param <- names.f
        # num <- as.numeric(str_extract(names.plot.param, "[0-9]+"))
        # char <- str_to_lower(str_extract(names.plot.param, "[aA-zZ]+"))
        # 
        # names.plot.param <- char
        # names.plot.param
        # for (i in which(!is.na(num))){
        #   names.plot.param[i] <- paste0(names.plot.param[i],"[",num[i],"]")
        # }
        # 
        # if (sum(names.plot.param=="int")>0){
        #   names.plot.param[names.plot.param=="int"] <- "beta[0]"
        # }
        # 
        # for (i in 1:length(unif.ndiv)){
        #   if (length(unif.ndiv[[i]])!=0){
        #     for (j in 1:length(unif.ndiv[[i]])){
        #       param.plot.name <- names.plot.param[unif.ndiv[[i]][j]]
        #       plot.ranks(N,L,rank.values[[i]][[unif.ndiv[[i]][j]]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(parse_expr(param.plot.name)), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
        #       points(as.numeric(as.character(freq.tabs[[i]][[unif.ndiv[[i]][j]]]$ranks)),y.n[[i]][[j]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
        #       int.mtext <- paste0("Int=",format(unif.ndiv.cov[[i]][j,1],scientific=TRUE,digits=3))
        #       cov.mtext <- paste0("rank=",format(unif.ndiv.cov[[i]][j,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.ndiv.cov[[i]][j,3],scientific=TRUE,digits=3))
        #       int.mtext <- str_replace_all(int.mtext,"e","x10^")
        #       cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
        #       int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
        #       cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
        #       mtext(text=int.mtext,side=1,line=5,cex=1.5)
        #       mtext(text=cov.mtext,side=1,line=6,cex=1.5)  
        #     } 
        #   }
        # }
      # } else {
        # if (sum(lengths(unif.div))==1){
        #   par(oma=opar$oma+c(2.5,0,0,0),mar=opar$mar+c(0,1,0,0))
        # } else if (sum(lengths(unif.div))>1){
        #   par(oma=opar$oma+c(2.5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
        # }
        # 
        # Grid_Mesh <- paste0(grids.names," and ", meshes.names)
        # 
        # names.plot.param <- names.f
        # num <- as.numeric(str_extract(names.plot.param, "[0-9]+"))
        # char <- str_to_lower(str_extract(names.plot.param, "[aA-zZ]+"))
        # 
        # names.plot.param <- char
        # names.plot.param
        # for (i in which(!is.na(num))){
        #   names.plot.param[i] <- paste0(names.plot.param[i],"[",num[i],"]")
        # }
        # 
        # if (sum(names.plot.param=="int")>0){
        #   names.plot.param[names.plot.param=="int"] <- "beta[0]"
        # }
        # 
        # for (i in 1:length(unif.div)){
        #   if (length(unif.div[[i]])!=0){
        #     for (j in 1:length(unif.div[[i]])){
        #       param.plot.name <- names.plot.param[unif.div[[i]][j]]
        #       plot.ranks(N,L,rank.values[[i]][[unif.div[[i]][j]]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(parse_expr(param.plot.name)), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
        #       points(as.numeric(as.character(freq.tabs[[i]][[unif.div[[i]][j]]]$ranks)),y[[i]][[j]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
        #       int.mtext <- paste0("Int=",format(unif.div.cov[[i]][j,1],scientific=TRUE,digits=3))
        #       cov.mtext <- paste0("rank=",format(unif.div.cov[[i]][j,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.div.cov[[i]][j,3],scientific=TRUE,digits=3))
        #       int.mtext <- str_replace_all(int.mtext,"e","x10^")
        #       cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
        #       int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
        #       cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
        #       mtext(text=int.mtext,side=1,line=5,cex=1.5) 
        #       mtext(text=cov.mtext,side=1,line=6,cex=1.5) 
        #     }
        #   }
        # }
        # 
      #   # Non-divergences too
      #   if (sum(lengths(unif.ndiv))==1){
      #     par(oma=opar$oma+c(2.5,0,0,0),mar=opar$mar+c(0,1,0,0))
      #   } else if (sum(lengths(unif.ndiv))>1){
      #     par(oma=opar$oma+c(2.5,0,0,0),mar=opar$mar+c(0,1,0,0),mfrow=c(1,2))
      #   }
      #   
      #   for (i in 1:length(unif.ndiv)){
      #     if (length(unif.ndiv[[i]])!=0){
      #       for (j in 1:length(unif.ndiv[[i]])){
      #         param.plot.name <- names.plot.param[unif.ndiv[[i]][j]]
      #         plot.ranks(N,L,rank.values[[i]][[unif.ndiv[[i]][j]]],main = bquote(atop("Ranks for f(" ~ theta ~ ") = " ~ .(parse_expr(param.plot.name)), ~ " and " ~ .(Grid_Mesh[i]))),cexmain=2,cexlab=2,cexaxis=1.5)
      #         points(as.numeric(as.character(freq.tabs[[i]][[unif.ndiv[[i]][j]]]$ranks)),y.n[[i]][[j]],type="l",lwd=2.5,pch=19,col="red",cex=0.5)
      #         int.mtext <- paste0("Int=",format(unif.ndiv.cov[[i]][j,1],scientific=TRUE,digits=3))
      #         cov.mtext <- paste0("rank=",format(unif.ndiv.cov[[i]][j,2],scientific=TRUE,digits=3),"; rank^2=",format(unif.ndiv.cov[[i]][j,3],scientific=TRUE,digits=3))
      #         int.mtext <- str_replace_all(int.mtext,"e","x10^")
      #         cov.mtext <- str_replace_all(cov.mtext,"e","x10^")
      #         int.mtext <- str_replace_all(int.mtext,"x10\\^\\+00","")
      #         cov.mtext <- str_replace_all(cov.mtext,"x10\\^\\+00","")
      #         mtext(text=int.mtext,side=1,line=5,cex=1.5)
      #         mtext(text=cov.mtext,side=1,line=6,cex=1.5)  
      #       } 
      #     }
      #   }
      # }
      
      p3a <- ggplot(critsum.df) + geom_point(aes(x=Mesh,y=mean,col=Grid,size=2)) + xlab("Mesh") + facet_wrap(~Criterion) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20)) + guides(size=FALSE, color = guide_legend(override.aes = list(size = 3)))
      p3b <- ggplot(critsum.df) + geom_point(aes(x=Mesh,y=mean,col=Grid,size=2)) + xlab("Mesh") + facet_wrap(~Criterion) + ylim(c(4.5e2,7.2e5)) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20)) + guides(size=FALSE, color = guide_legend(override.aes = list(size = 3)))
      p3c <- ggplot(critsum.df) + geom_point(aes(x=Mesh,y=mean,col=Grid,size=2)) + xlab("Mesh") + facet_wrap(~Criterion) + ylim(c(4.5e2,3.6e4)) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20)) + guides(size=FALSE, color = guide_legend(override.aes = list(size = 3)))
      
      print(p3a)
      print(p3b)
      print(p3c)
      
      p4err <- ggplot(err.df[err.df$Criterion=="Error",]) + geom_point(aes(x=Run,y=Value,col=Grid,size=2)) + facet_wrap(~Mesh+Grid,ncol = 4,scales="free_y") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
      p4war <- ggplot(err.df[err.df$Criterion=="Warning",]) + geom_point(aes(x=Run,y=Value,col=Grid,size=2)) + facet_wrap(~Mesh+Grid,ncol = 4,scales="free_y") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
      p4fft <- ggplot(err.df[err.df$Criterion=="FFT",]) + geom_point(aes(x=Run,y=Value,col=Grid,size=2)) + facet_wrap(~Mesh+Grid,ncol = 4,scales="free_y") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
      p5 <- ggplot(errsum.df) + geom_point(aes(x=,y=mean,col=Grid,size=2)) + xlab("Mesh") + facet_wrap(~Criterion) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),axis.text.x=element_text(angle = 45, vjust = 1, hjust = 1),strip.text.x = element_text(size=20))
      
      print(p4err)
      print(p4war)
      print(p4fft)
      print(p5)
      
      col.ind <- c("red","blue","orange","magenta") # mesh 1,2,3,4
      m <- matrix(c(1,2,3,4,5,6,7,7,7),nrow = 3,ncol = 3,byrow = TRUE)
      layout(mat = m,heights = c(0.4,0.4,0.2))
      for (i in 1:length(names.f)){
        par(mar = c(3,3,2,2))
        plot(true.theta.final[,i],pch=19,main=(names.f[i]),ylab=names.f[i])
        for (j in 1:length(theta.val.fft)){
          points(fft.ind[[j]],theta.val.fft[[j]][,i],pch=19,col=col.ind[j])
        }
      }
      plot(1, type = "n", axes=FALSE, xlab="", ylab="")
      legend("bottom",legend = names(run), text.width = max(sapply(names(run), strwidth)),col=col.ind, lwd=5, cex=1, horiz = TRUE)
      
      m <- matrix(c(1,2,3,4,5,6,7,7,7),nrow = 3,ncol = 3,byrow = TRUE)
      layout(mat = m,heights = c(0.4,0.4,0.2))
      for (i in 1:length(names.f)){
        par(mar = c(3,3,2,2))
        plot(true.theta.final[,i],pch=19,main=(names.f[i]),ylab=names.f[i])
        for (j in 1:length(theta.val.err)){
          points(err.ind[[j]],theta.val.err[[j]][,i],pch=19,col=col.ind[j])
        }
      }
      plot(1, type = "n", axes=FALSE, xlab="", ylab="")
      legend("bottom",legend = names(run), text.width = max(sapply(names(run), strwidth)),col=col.ind, lwd=5, cex=1, horiz = TRUE)
      
      
      cols.fft <- matrix(rep(character(nrow(true.theta.final)),length(run)),ncol=length(run))
      cols.fft[] <- "black"
      for (i in 1:length(run)){
        cols.fft[fft.ind[[i]],i] <- col.ind[i]
      }
      mesh.names.title <- gsub("([a-z])([0-9])", "\\1 \\2", names(run))
      
      
      for (i in 1:length(run)){
        p <- pairs(true.theta.final,col=cols.fft[,i],pch=19,upper.panel = NULL,main=paste0(mesh.names.title[i]," FFT"))
        print(p)
      }
      
      cols.err <- matrix(rep(character(nrow(true.theta.final)),length(run)),ncol=length(run))
      cols.err[] <- "black"
      for (i in 1:length(run)){
        cols.err[err.ind[[i]],i] <- col.ind[i]
      }
      for (i in 1:length(run)){
        p <- pairs(true.theta.final,col=cols.err[,i],pch=19,upper.panel = NULL,main=paste0(mesh.names.title[i]," Error"))
        print(p)
      }
    }
  }
  
  return(list("summary"=sum.df,"errors"=errsum.df,"crit"="critsum.df")) #,"unif.models"=unif.check,"unif.divergence.ind"=unif.div
}
