
# LA Polygon: Grid Mesh Optimisation: Traditional Simulation Study --------

# This script will implement the Traditional Simulation Study for LGCP example over the LA polygon window in order to check how the algorithm behaves with respect to things like parameter recovery for the different grid and mesh resolutions under consideration.
# It will output the timings and parameter posterior summaries (mean, quantiles) as well as the WAIC and DIC.
# The section for the simulation of the covariates and meshes are run before the code is moved to Balena, with the outputs saved in order to ensure there is no need to re-simulate for each iteration, especially with the expensive simulation of the grids and meshes and to ensure consistency in the mesh used.
# We have sub-functions that generates the data and the A matrix for the INLA runs to keep the implementation of the simulation study as compact as possible.

# Author: Nadeen Khaleel


# Arguments Read In -------------------------------------------------------

args=(commandArgs(TRUE))
print(args)

if (length(args)==0){
  print("No arguments supplied.")
  # Set default values.
  this.node = 1 # which "node" am I on - change per job
  total.nodes = 1 # how many nodes am I using?
  Nprocs.vec = 1 # vector where each element contains the number of processors for a particular node
  Nprocs.total = 1 # total number of processors across ALL nodes (/jobs)
  N = 1 # how many simulations?
  sim = 0 # start new or re-starting at last saved simulation?
}else{
  for (i in 1:length(args)){
    eval(parse(text=args[[i]]))
  }
}

# Example of R CMD BATCH command from job slurm scripts for implementing this simulation study:
# R CMD BATCH --vanilla '--args this.node=1 total.nodes=20 Nprocs.vec=rep(2,20) Nprocs.total=40 N=1000 sim=1'  GridMeshOptimIrregTrad_final.R gm_irregtradlgcp1.out
# We had 20 jobs each run across a separate node using similar commands to the above.


# Libraries ---------------------------------------------------------------

# Set up parallel computing
Nprocs <- Nprocs.vec[this.node]

library(doParallel)
library(foreach)
parallelCluster <- parallel::makeCluster(Nprocs)
print(parallelCluster)
registerDoParallel(parallelCluster)

# Indices for the processes to be split across the node, for the saving of output
# ind.procs <- 1:Nprocs.total
ind.procs.all <- 1:40 # using 20 nodes each with 2 jobs

# Can additionally alter the ordering of the processes so that certain processes can be on the same node, this is especially advantageous if we have a few processes on different nodes complete as the incomplete runs can be combined and reduce the total number of nodes required to complete.
# ind.procs.all <- c(4,5,1:3,6:40)

# Split the indices across the nodes, with the commented version more adaptable to any changes between nodes, however we consistently had 2 simulations per job so it was easier to use the uncommented version.
# ii <- (this.node-1)*Nprocs.vec[1]
# ind.procs <- ind.procs.all[(ii+1):(ii+Nprocs.vec[this.node])]
ii <- (this.node-1)*2 # 2 jobs per node
ind.procs <- ind.procs.all[(ii+1):(ii+2)]


ptm <- proc.time()
foreach(k = ind.procs) %dopar% {
  
  library(INLA)
  library(mvtnorm)
  library(sp)
  library(sf)
  library(spatstat)
  library(raster)
  library(maptools)
  library(stringr)
  library(rgeos)
  
  par.lic.filepath <- "./pardiso.lic" # file path for pardiso licence if in use
  inla.setOption(pardiso.license = par.lic.filepath)
  
  inla.setOption(num.threads=8) # 2 procs using 8 cores each per node
  
  # Functions ---------------------------------------------------------------
  
  cov.surface.gen <- function(W,theta,cov1.im,cov2.im,int.im){
    beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
    # Create pixel images
    mu <- beta.0*int.im + beta.1*cov1.im + beta.2*cov2.im
    
    return(mu)
  }
  
  data.gen <- function(W,theta,disc.full,mu,quads,coord,c1.list,c2.list){
    # W is the window
    # theta are the parameters required to generate the data.
    # disc.full is the vector of discretisations across the window for which the data will be produced at.
    # mu is the mean surface from fixed effects
    # quads is the list of pre-computed quadrats
    # coord is the list of pre-computed coordinates for the points
    # ci.list is the list of covariate values for each grid cell, extract over the polygons of the quadrats
    
    
    # Simulate from this process with the parameters as defned above.
    beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]]; sigma <- theta[[4]]; rho <- theta[[5]];
    
    # Produce the point pattern from LGCP
    lgcp.ppp <- rLGCP(model="matern",mu,var=(sigma)^2,scale=rho/2,nu=1,win = W,saveLambda = TRUE) # Note: default resolution of the simulated data is 128x128, if this is too coarse for the data aggregations (if one of the data aggregation requires a grid finer than 128x128), then also set dimyx=c(ny,nx).
    
    # If there are no points simulated, using the same parameters and mu, simulate point process again until we have non-zero data.
    sim.0 <- 1
    while(lgcp.ppp$n==0){
      sim.0 <- sim.0 + 1
      print(paste0("0 points simulated for data set",k," try for time",sim.0))
      lgcp.ppp <- rLGCP(model="matern",mu,var=(sigma)^2,scale=rho/2,nu=1,win = W,saveLambda = TRUE) # Note: default resolution of the simulated data is 128x128, if this is too coarse for the data aggregations (if one of the data aggregation requires a grid finer than 128x128), then also set dimyx=c(ny,nx).
    }
    
    grid.names <- c(paste0("grid",disc.full[,1],disc.full[,2]))
    mesh.names <- c(paste0("mesh",disc.full[,1],disc.full[,2]))
    data.list <- vector(mode="list",length=length(grid.names))
    names(data.list) <- grid.names
    
    for (i in 1:dim(disc.full)[1]){
      n.cell <- disc.full[i,]
      M <- n.cell[1]; N <- n.cell[2]
      cellsize <- c((W$xrange[2]-W$xrange[1])/n.cell[1],(W$yrange[2]-W$yrange[1])/n.cell[2])
      
      g <- quads[[i]] # pre-calculated for speed
      # Use coordinates with shifted centres so that all points lie within the window.
      ord <- t(coordinates(coord[[i]])) # t() so that ord matches the previous version of ord, and so the remaining code matches.
      a <- sapply(1:g$n,function(ii){(area.owin(g$tiles[[ii]]))})
      m <- matrix(as.numeric(unlist(str_extract_all(names(g$tiles),"\\d*\\d"))),ncol=2,byrow=TRUE)
      ord.df <- data.frame(ind=1:g$n,row=m[,1],col=m[,2]) # col matches the raster definition of col=x, and row=y which makes sense visually from the plots as x<--> and y ^inc(^)
      ord.df <- ord.df[order(ord.df$col),]
      
      x.ord <- ord[1,ord.df$ind]; y.ord <- ord[2,ord.df$ind]
      
      cell.area <- a[ord.df$ind]
      
      q <- quadratcount(lgcp.ppp,tess=g)
      count.df <- data.frame(x=x.ord,y=y.ord,count=as.vector(q)[ord.df$ind],area=cell.area)
      
      count.df$cov1 <- c1.list[[i]][ord.df$ind]
      count.df$cov2 <- c2.list[[i]][ord.df$ind]
      
      data.list[[i]] <- count.df
    }
    
    # Return count data list and true lambda from lgcp.ppp simulation
    return(list("data"=data.list,"lambda"=attributes(lgcp.ppp)$Lambda))
  }
  
  # Generate A matrix and stack data
  A_stack.gen <- function(data.gm,mesh,sigma.star,rho.star){
    the_spde <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[1],rho.star[2]),prior.sigma = c(sigma.star[1],sigma.star[2]))
    
    s.index <- inla.spde.make.index("field",n.spde=the_spde$n.spde)
    coords <- data.gm[,c("x","y")]
    coordinates(coords) <- ~ x + y
    A <- inla.spde.make.A(mesh, loc=coords)
    stk <- inla.stack(data=list(resp=data.gm$count),A=list(A,1),effects=list(c(s.index,list(int=1)),list(cov1=data.gm$cov1,cov2=data.gm$cov2,larea=log(data.gm$area))),tag='est')
    return(list("spde"=the_spde,"A"=A,"stack.est"=stk))
  }
  
  
  # Covariate and  Set-up --------------------------------------------------------
  
  # Set working directory to save the outputs before moving them over to Balena
  # library("rstudioapi")
  # # Either setwd() to the source file location, or run the following:
  # setwd(dirname(getActiveDocumentContext()$path))
  
  # # Will comment out after the required data is produced and saved, the data for our implementation was run from GridMeshOptimIrreg.R and then cp'd to GridMeshOptimIrregTrad.R
  # load("../../../DATA/PROCESSED_DATA/SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")
  # lacity_boundary <- st_union(ct_LA)
  # lacity_proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611")
  # lacity_sp <- as(lacity_proj,"Spatial")
  # 
  # # TRANSFORM THE WINDOW lacity_sp so that one unit is 10km
  # bbox <- lacity_sp@bbox
  # lacity_spshift <- elide(lacity_sp,shift=-c(bbox[1,1],bbox[2,1]))
  # bbox_shift <- lacity_spshift@bbox
  # lacity_spscale <- elide(lacity_spshift,scale=max(bbox_shift)/1e4)
  # W <- as.owin.SpatialPolygons((lacity_spscale))
  # 
  # load("../../../DATA/PROCESSED_DATA/COUNT_DATA_GMO/LA/LA2015CT236359CountData_proj.rda")
  # 
  # bbox <- lacity_sp@bbox
  # bbox_shift <- lacity_spshift@bbox
  # # z-stat for population and income
  # hom_countdf$zpop <- (hom_countdf$pop - mean(hom_countdf$pop))/sd(hom_countdf$pop)
  # hom_countdf$zinc <- (hom_countdf$inc - mean(hom_countdf$inc))/sd(hom_countdf$inc)
  # # Population
  # pop.df <- hom_countdf[,c("x","y","zpop")]
  # colnames(pop.df) <- c("x","y","zpop")
  # coordinates(pop.df) <- ~ x + y
  # pop.df.sf <- st_as_sf(pop.df)
  # pop.df.sf.proj <- st_set_crs(pop.df.sf,"epsg:32611")
  # pop.df.proj <- as(pop.df.sf.proj,"Spatial")
  # pop.dfshift <- elide(pop.df.proj,bb=lacity_sp@bbox,shift=-c(bbox[1,1],bbox[2,1]))
  # pop.dfscale <- elide(pop.dfshift,bb=lacity_spshift@bbox,scale=max(bbox_shift)/1e4)
  # # Average Income
  # inc.df <- hom_countdf[,c("x","y","zinc")]
  # colnames(inc.df) <- c("x","y","zinc")
  # coordinates(inc.df) <- ~ x + y
  # inc.df.sf <- st_as_sf(inc.df)
  # inc.df.sf.proj <- st_set_crs(inc.df.sf,"epsg:32611")
  # inc.df.proj <- as(inc.df.sf.proj,"Spatial")
  # inc.dfshift <- elide(inc.df.proj,bb=bbox,shift=-c(bbox[1,1],bbox[2,1]))
  # inc.dfscale <- elide(inc.dfshift,bb=bbox_shift,scale=max(bbox_shift)/1e4)
  # w.r <- raster(lacity_spscale,ncols=236,nrows=359)
  # pop.ras <- rasterize(pop.dfscale,w.r,field=pop.dfscale$zpop)
  # inc.ras <- rasterize(inc.dfscale,w.r,field=inc.dfscale$zinc)
  # int.ras <- rasterize(pop.dfscale,w.r,field=rep(1,length(pop.dfscale)))
  # # BASE RASTERLAYER - otherwise get errors that the covariates do not cover the window...
  # b.r <- raster(extent(pop.ras),pop.ras@nrows,pop.ras@ncols)
  # bbox.base <- owin(xrange=c(extent(pop.ras)[1],extent(pop.ras)[2]),yrange=c(extent(pop.ras)[3],extent(pop.ras)[4]))
  # gc.df <- gridcenters(bbox.base,500,500)
  # gc.df <- data.frame(x=gc.df$x,y=gc.df$y)
  # coordinates(gc.df) <- ~ x + y
  # b.ras <- rasterize(gc.df,b.r,rep(0,length(gc.df)))
  # # Creat Pixel Images
  # popb.im <- as.im(merge(pop.ras,b.ras))
  # incb.im <- as.im(merge(inc.ras,b.ras))
  # intb.im <- as.im(merge(int.ras,b.ras))
  
  cov.name <- paste0("GridMeshIrregPolLGCPSSCov.rda")
  # save(popb.im,incb.im,intb.im,pop.ras,inc.ras,int.ras,file=cov.name)
  
  # # Create the quadrat tessalations to save time
  # load("../../../DATA/PROCESSED_DATA/SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")
  # ct_LA.proj <- lwgeom::st_transform_proj(ct_LA,"epsg:32611")
  # lacity_boundary <- st_union(ct_LA)
  # lacity_boundary.proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611")
  # lacity_geom <- as(lacity_boundary,"Spatial")
  # lacity_geom.proj <- as(lacity_boundary.proj,"Spatial")
  # lacity_win <- as.owin.SpatialPolygons((lacity_geom))
  # lacity_win.proj <- as.owin.SpatialPolygons((lacity_geom.proj))
  # bbox <- lacity_geom.proj@bbox
  # lacity_spshift <- elide(lacity_geom.proj,shift=-c(bbox[1,1],bbox[2,1]))
  # bbox_shift <- lacity_spshift@bbox
  # lacity_spscale <- elide(lacity_spshift,scale=max(bbox_shift)/1e4)
  # W <- as.owin.SpatialPolygons((lacity_spscale))
  # 
  # # So for approximately 1kmx1km, 2kmx2km  grids etc need following dims
  # x.range <- diff(lacity_win.proj$xrange)
  # y.range <- diff(lacity_win.proj$yrange)
  # 
  # grid_cellsxvec <- ceiling(x.range/(1e3*c(5,2,1,0.5,0.2)))
  # grid_cellsyvec <- ceiling(y.range/(1e3*c(5,2,1,0.5,0.2)))
  # 
  # disc.full <- unname(cbind(grid_cellsxvec,grid_cellsyvec))
  # 
  # quad.gen <- function(window,cell.n,save.name){
  #   quad.names <- c(paste0("quad",cell.n[,1],cell.n[,2]))
  #   quad.list <- vector(mode="list",length=length(quad.names))
  #   names(quad.list) <- quad.names
  #   W <- window # added 04/2021 for re-run
  #   for (i in 1:dim(cell.n)[1]){
  #     print(i)
  #     n.cell <- cell.n[i,]
  #     M <- n.cell[1]; N <- n.cell[2]
  #     g <- quadrats(W,M,N)
  # 
  #     quad.list[[i]] <- g
  #     
  #   }
  #   save("quad"=quad.list,file=save.name)
  # #   save(list("quad"=quad.list),file=save.name) ## list() will not work (?? did not work for TradGridMeshQuickTest DataSetUp.R), was added in for possible re-run that was not done and therefore quadrats were not re-calculated. Corrected version of code is now included, for any further use.
  # }
  # 
  quad.file <- "QuadratsIrregPolLGCP.rda"
  # quad.gen(window=W,disc.full,quad.file)
  
  # # The below was run with the quadrats previously created.
  # # Now want to simulate the coordinates for the grid points, so that ALL of them lie within the window, which is not the case when initially calculating the centres of the grid, especially for the irregularly shaped intersections for the grid cells with the irregularly shaped polygon window.
  # polypoint <- function(quad){
  #   g <- quad
  #   g.sp <- as(g,"SpatialPolygons") # we need this to be sp
  #   ord <- sapply(1:g$n,function(ii){unlist(centroid.owin(g$tiles[[ii]]))})
  #   coords <- data.frame(x=ord[1,],y=ord[2,])
  #   coordinates(coords) <- ~ x + y
  #   
  #   coords.orig <- coords
  #   
  #   coord.poly.int <- st_intersects(st_as_sf(coords),st_as_sf(g.sp))
  #   ind.out <- which(lengths(coord.poly.int)==0) # which ones do not intersect with the polygon
  #   
  #   length(ind.out)
  #   ind.out.orig <- ind.out
  #   
  #   if (length(ind.out)>0){
  #     coord.df.poly <- data.frame(ind.out=ind.out,x.old=rep(NA,length(ind.out)),y.old=rep(NA,length(ind.out)),x.closestpoly=rep(NA,length(ind.out)),y.closestpoly=rep(NA,length(ind.out)))
  #     
  #     for (i in 1:length(ind.out)){
  #       l.poly <- length(g.sp@polygons[[ind.out[i]]]@Polygons)
  #       if (l.poly==1){
  #         poly.coord <- g.sp@polygons[[ind.out[i]]]@Polygons[[1]]@coords
  #         p = Polygon(poly.coord)
  #         ps = Polygons(list(p),1)
  #         sps = SpatialPolygons(list(ps))
  #         coord.df.poly[i,c("x.old","y.old")] <- unname(coordinates(coords[ind.out[i]]))
  #         coord.df.poly[i,c("x.closestpoly","y.closestpoly")] <- unname(coordinates(gCentroid(sps)))
  #       } else {
  #         d.vec <- rep(NA,l.poly)
  #         for (j in 1:l.poly){
  #           labpt <- data.frame(x=g.sp@polygons[[ind.out[i]]]@Polygons[[j]]@labpt[1],y=g.sp@polygons[[ind.out[i]]]@Polygons[[j]]@labpt[2])
  #           coordinates(labpt) <- ~ x + y
  #           d.vec[j] <- gDistance(coords[ind.out[i]],labpt)
  #         }
  #         m <- which.min(d.vec)
  #         poly.coord <- g.sp@polygons[[ind.out[i]]]@Polygons[[m]]@coords
  #         p = Polygon(poly.coord)
  #         ps = Polygons(list(p),1)
  #         sps = SpatialPolygons(list(ps))
  #         coord.df.poly[i,c("x.old","y.old")] <- unname(coordinates(coords[ind.out[i]]))
  #         coord.df.poly[i,c("x.closestpoly","y.closestpoly")] <- unname(coordinates(gCentroid(sps)))
  #       }
  #     }
  #     
  #     
  #     coords2 <- data.frame(coords)
  #     coords2[ind.out,] <- coord.df.poly[,c("x.closestpoly","y.closestpoly")]
  #     coordinates(coords2) <- ~ x + y
  #     coord.poly.int2 <- st_intersects(st_as_sf(coords2),st_as_sf(g.sp))
  #     ind.out2 <- which(lengths(coord.poly.int2)==0)
  #     
  #     length(ind.out2)
  #     
  #     # Rather than using the points on the boundary, maybe create a grid over the closest sub-polygon and find the closest point that also lies within the window
  #     
  #     if (length(ind.out2)>0){
  #       coord.df.grid <- data.frame(ind.out=ind.out2,x.old=rep(NA,length(ind.out2)),y.old=rep(NA,length(ind.out2)),x.closestgrid=rep(NA,length(ind.out2)),y.closestgrid=rep(NA,length(ind.out2)))
  #       
  #       for (i in 1:length(ind.out2)){
  #         l.poly <- length(g.sp@polygons[[ind.out2[i]]]@Polygons)
  #         if (l.poly==1){
  #           poly.coord <- g.sp@polygons[[ind.out2[i]]]@Polygons[[1]]@coords
  #           p = Polygon(poly.coord)
  #           ps = Polygons(list(p),1)
  #           sps = SpatialPolygons(list(ps))
  #           
  #           g.sub <- quadrats(sps,10,10)
  #           ord.sub <- sapply(1:g.sub$n,function(ii){unlist(centroid.owin(g.sub$tiles[[ii]]))})
  #           # Keep only the points IN the polygon
  #           coords.sub <- data.frame(x=ord.sub[1,],y=ord.sub[2,])
  #           coordinates(coords.sub) <- ~ x + y
  #           # Which lie outside the polygons?
  #           coord.poly.sub.int <- st_intersects(st_as_sf(coords.sub),st_as_sf(as(g.sub,"SpatialPolygons")))
  #           ord.sub.keep <- ord.sub[,lengths(coord.poly.sub.int)==1]
  #           
  #           coords.sub.keep <- data.frame(x=ord.sub.keep[1,],y=ord.sub.keep[2,])
  #           coordinates(coords.sub.keep) <- ~ x + y
  #           
  #           coords.nearest <- coordinates(gNearestPoints(coords[ind.out2[i]],coords.sub.keep))
  #           
  #           coord.df.grid[i,c("x.old","y.old")] <-  unname(coords.nearest[1,])
  #           coord.df.grid[i,c("x.closestgrid","y.closestgrid")] <-  unname(coords.nearest[2,])
  #         } else {
  #           d.vec <- rep(NA,l.poly)
  #           for (j in 1:l.poly){
  #             labpt <- data.frame(x=g.sp@polygons[[ind.out2[i]]]@Polygons[[j]]@labpt[1],y=g.sp@polygons[[ind.out2[i]]]@Polygons[[j]]@labpt[2])
  #             coordinates(labpt) <- ~ x + y
  #             d.vec[j] <- gDistance(coords[ind.out2[i]],labpt)
  #           }
  #           m <- which.min(d.vec)
  #           poly.coord <- g.sp@polygons[[ind.out2[i]]]@Polygons[[m]]@coords
  #           p = Polygon(poly.coord)
  #           ps = Polygons(list(p),1)
  #           sps = SpatialPolygons(list(ps))
  #           
  #           g.sub <- quadrats(sps,10,10)
  #           ord.sub <- sapply(1:g.sub$n,function(ii){unlist(centroid.owin(g.sub$tiles[[ii]]))})
  #           # Keep only the points IN the polygon
  #           coords.sub <- data.frame(x=ord.sub[1,],y=ord.sub[2,])
  #           coordinates(coords.sub) <- ~ x + y
  #           # Which lie outside the polygons?
  #           coord.poly.sub.int <- st_intersects(st_as_sf(coords.sub),st_as_sf(as(g.sub,"SpatialPolygons")))
  #           ord.sub.keep <- ord.sub[,lengths(coord.poly.sub.int)==1]
  #           
  #           coords.sub.keep <- data.frame(x=ord.sub.keep[1,],y=ord.sub.keep[2,])
  #           coordinates(coords.sub.keep) <- ~ x + y
  #           
  #           coords.nearest <- coordinates(gNearestPoints(coords[ind.out2[i]],coords.sub.keep))
  #           
  #           coord.df.grid[i,c("x.old","y.old")] <-  unname(coords.nearest[1,])
  #           coord.df.grid[i,c("x.closestgrid","y.closestgrid")] <-  unname(coords.nearest[2,])
  #         }
  #       }
  #       
  #       
  #       coords3 <- data.frame(coords2)
  #       coords3[ind.out2,] <- coord.df.grid[,c("x.closestgrid","y.closestgrid")]
  #       coordinates(coords3) <- ~ x + y
  #       coord.poly.int3 <- st_intersects(st_as_sf(coords3),st_as_sf(g.sp))
  #       ind.out3 <- which(lengths(coord.poly.int3)==0)
  #       
  #       length(ind.out3)
  #       
  #       coords.final <- coords3
  #       ind.out.final <- ind.out3
  #     } else {
  #       coords.final <- coords2
  #       ind.out.final <- ind.out2
  #     }
  #   } else {
  #     coords.final <- coords
  #     ind.out.final <- ind.out
  #   }
  #   
  #   return(list("ind.orig"=ind.out.orig,"ind.final"=ind.out.final,"coords.orig"=coords.orig,"coords.final"=coords.final))
  # }
  # 
  # load(quad.file)
  # library(stringr)
  # coord.gen <- function(quad,save.name){
  #   coord.names <- str_replace_all(names(quad.list),"quad","coord")
  #   coord.list <- vector(mode="list",length=length(coord.names))
  #   names(coord.list) <- coord.names
  #   
  #   for (i in 1:length(quad)){
  #     print(i)
  #     coord.alt <- polypoint(quad[[i]])
  #     print(coord.alt$ind.out.final)
  #     coord.list[[i]] <- coord.alt$coords.final
  #     
  #   }
  #   save("coord"=coord.list,file=save.name)
  # }
  # 
  coord.file <- "CoordsIrregPolLGCP.rda"
  # coord.gen(quad=quad.list,coord.file)
  
  # load(coord.file) # double check that the points are IN the window
  # for (i in 1:length(coord.list)){
  #   test.in <- st_intersects(st_as_sf(coord.list[[i]]),st_as_sf(lacity_spscale))
  #   print(sum(lengths(test.in)==0))
  # }
  
  # covgrid <- function(cov1.ras,cov2.ras,quad,covgrid.file){
  #   cov1grid.list <- vector(mode="list",length=length(quad))
  #   cov1.names <- str_replace_all(names(quad),"quad","cov1")
  #   names(cov1grid.list) <- cov1.names
  #   cov2grid.list <- vector(mode="list",length=length(quad))
  #   cov2.names <- str_replace_all(names(quad),"quad","cov2")
  #   names(cov2grid.list) <- cov2.names
  # 
  #   for (i in 1:length(quad)){
  #     print(i)
  #     g <- quad[[i]]
  #     g.sp <- as(g,"SpatialPolygons")
  #     cov1grid.list[[i]] <- raster::extract(cov1.ras,g.sp,weights=T,fun=mean)
  #     cov2grid.list[[i]] <- raster::extract(cov2.ras,g.sp,weights=T,fun=mean)
  #   }
  #   save(cov1grid.list,cov2grid.list,file=covgrid.file)
  # }
  # 
  covgrid.file <- "CovAggGridIrregPolLGCP.rda"
  # load(cov.name)
  # load(quad.file)
  # covgrid(pop.ras,inc.ras,quad.list,covgrid.file)
  
  # load("../../../DATA/PROCESSED_DATA/SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")
  # ct_LA.proj <- lwgeom::st_transform_proj(ct_LA,"epsg:32611")
  # lacity_boundary <- st_union(ct_LA)
  # lacity_boundary.proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611")
  # lacity_geom <- as(lacity_boundary,"Spatial")
  # lacity_geom.proj <- as(lacity_boundary.proj,"Spatial")
  # lacity_win <- as.owin.SpatialPolygons((lacity_geom))
  # lacity_win.proj <- as.owin.SpatialPolygons((lacity_geom.proj))
  # bbox <- lacity_geom.proj@bbox
  # lacity_spshift <- elide(lacity_geom.proj,shift=-c(bbox[1,1],bbox[2,1]))
  # bbox_shift <- lacity_spshift@bbox
  # lacity_spscale <- elide(lacity_spshift,scale=max(bbox_shift)/1e4)
  # W <- as.owin.SpatialPolygons((lacity_spscale))
  # 
  # # So for approximately 1kmx1km, 2kmx2km  grids etc need following dims
  # x.range <- diff(lacity_win.proj$xrange)
  # y.range <- diff(lacity_win.proj$yrange)
  # 
  # grid_cellsxvec <- ceiling(x.range/(1e3*c(5,2,1,0.5,0.2)))
  # grid_cellsyvec <- ceiling(y.range/(1e3*c(5,2,1,0.5,0.2)))
  # 
  # disc.full <- unname(cbind(grid_cellsxvec,grid_cellsyvec))
  # 
  # mesh.gen <- function(window,cell.n,quad,coord,save.name){ # was window rather than W previously, which is never used below
  #   W <- window
  #   mesh.names <- c(paste0("mesh",cell.n[,1],cell.n[,2]))
  #   mesh.list <- vector(mode="list",length=length(mesh.names))
  #   names(mesh.list) <- mesh.names
  #   for (i in 1:dim(cell.n)[1]){
  #     print(i)
  #     n.cell <- cell.n[i,]
  #     M <- n.cell[1]; N <- n.cell[2]
  # 
  #     cellsize <- c((W$xrange[2]-W$xrange[1])/n.cell[1],(W$yrange[2]-W$yrange[1])/n.cell[2])
  # 
  #     g <- quad[[i]]
  # 
  #     ord <- t(coordinates(coord[[i]])) # use t() in order to have it match the previous version of ord so that code will still work
  # 
  #     m <- matrix(as.numeric(unlist(str_extract_all(names(g$tiles),"\\d*\\d"))),ncol=2,byrow=TRUE)
  #     ord.df <- data.frame(ind=1:g$n,row=m[,1],col=m[,2])
  #     ord.df <- ord.df[order(ord.df$col),]
  # 
  #     x.ord <- ord[1,ord.df$ind]; y.ord <- ord[2,ord.df$ind]
  # 
  #     df <- data.frame(x=x.ord,y=y.ord)
  #     coords <- df[,c("x","y")]
  #     coordinates(coords) <- ~ x + y
  # 
  #     boundary <- as(W,"SpatialPolygons") # For the meshes
  # 
  #     mesh <- inla.mesh.2d(loc=coords, boundary=boundary, max.edge=c(max(cellsize), max(cellsize)+0.5), min.angle=c(30, 21),
  #                          max.n=c(48000, 16000), ## Safeguard against large meshes.
  #                          max.n.strict=c(128000, 128000), ## Don't build a huge mesh!
  #                          cutoff=0.0075, ## Filter away adjacent points.
  #                          offset=c(0.01, 1)) ## Offset for extra boundaries, if needed.
  # 
  #     mesh.list[[i]] <- mesh
  #   }
  #   save("mesh"=mesh.list,file=save.name)
  # }
  
  meshes.file <- "MeshesIrregPolLGCP.rda"
  # load(quad.file)
  # load(coord.file)
  # mesh.gen(window=W,disc.full,quad.list,coord.list,meshes.file)
  
  # Load LA polgyon window, W.la
  # load("../../../DATA/PROCESSED_DATA/SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")
  # lacity_boundary <- st_union(ct_LA)
  # lacity_proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611")
  # lacity_sp <- as(lacity_proj,"Spatial") #st_transform for crs...
  # 
  # # Transform the window
  # bbox <- lacity_sp@bbox
  # lacity_spshift <- elide(lacity_sp,shift=-c(bbox[1,1],bbox[2,1]))
  # bbox_shift <- lacity_spshift@bbox
  # lacity_spscale <- elide(lacity_spshift,scale=max(bbox_shift)/1e4)
  # W <- as.owin.SpatialPolygons((lacity_spscale))
  # lacity_win.proj <- as.owin.SpatialPolygons(lacity_sp)
  # 
  window.file <- "WindowsIrregPolLGCP.rda"
  # save(W,lacity_win.proj,file=window.file)
  
  # Simulations -------------------------------------------------------------
  
  
  # SBC Parameters for Parallelisation
  
  # Assign simulations per processors and then extract the required processors and simulations for each node/job
  M.it.total <- rep(N%/%Nprocs.total,length=Nprocs.total)
  if (N%%Nprocs.total!=0){M.it.total[1:(N%%Nprocs.total)] <- M.it.total[1:(N%%Nprocs.total)] + 1}
  
  
  fft.threshold <- 5 # how many "Fail to factorise Q" warnings accepted before a warning message is produced for user
  
  load(cov.name) # load the covariates
  
  # Prior for the Gaussian latent field covariance parameters
  alpha.rho <- 0.01; alpha.sigma <- 0.1; rho.0 <- 0.35; sigma.0 <- 2
  rho.star <- c(rho.0,alpha.rho) ; sigma.star <- c(sigma.0,alpha.sigma)
  
  # Saving the output
  save.file <- paste0("GridMeshIrregPolLGCPTradSS",k,".rda")
  print(save.file)
  
  
  load(window.file)
  x.range <- diff(lacity_win.proj$xrange)
  y.range <- diff(lacity_win.proj$yrange)
  
  grid_cellsxvec <- ceiling(x.range/(1e3*c(5,2,1,0.5)))
  grid_cellsyvec <- ceiling(y.range/(1e3*c(5,2,1,0.5)))
  
  disc.full <- unname(cbind(grid_cellsxvec,grid_cellsyvec))
  param <- c("Beta0","Beta1","Beta2","Sigma","Rho")
  N.gridx <- disc.full[,1]; N.gridy <- disc.full[,2]; mesh.edge <- apply(cbind(abs((W$xrange[2]-W$xrange[1]))/N.gridx,abs((W$yrange[2]-W$yrange[1]))/N.gridy), 1, max)
  N.g <- length(N.gridx); N.m <- length(mesh.edge); N.p <- length(param)
  
  grid.ind <- paste0("Grid",N.gridx,N.gridy); mesh.ind <- paste0("Mesh",signif(mesh.edge,2));
  
  load(meshes.file)
  mesh.list <- mesh.list[1:4] # the last element in the list is the 0.2kmx0.2km mesh, which we don't want to use.
  load(quad.file)
  quad.list <- quad.list[1:4] # the last element in the list is the 0.2kmx0.2km mesh, which we don't want to use.
  load(coord.file)
  coord.list <- coord.list[1:4] # the last element in the list is the 0.2kmx0.2km mesh, which we don't want to use.
  load(covgrid.file)
  cov1grid.list <- cov1grid.list[1:4] # the last element in the list is the 0.2kmx0.2km mesh, which we don't want to use.
  cov2grid.list <- cov2grid.list[1:4] # the last element in the list is the 0.2kmx0.2km mesh, which we don't want to use.
  
  # Set-up outputs if we being the simulations, or load up the previous simulations and find the correct indices to continue running.
  if (sim==0){
    p.length <- 1
    # Final Data list
    list.grid <- vector(mode="list",length=N.g)
    names(list.grid) <- grid.ind
    list.mesh <- vector(mode="list",length=N.m)
    names(list.mesh) <- mesh.ind
    list.param <- vector(mode="list",length=3)
    names(list.param) <- c("est.df","run.df","mess.ls")
    list.param$est.df <- data.frame(beta0=rep(NA,M.it.total[k]),beta0.sd=rep(NA,M.it.total[k]),beta0.cil=rep(NA,M.it.total[k]),beta0.ciu=rep(NA,M.it.total[k]),beta1=rep(NA,M.it.total[k]),beta1.sd=rep(NA,M.it.total[k]),beta1.cil=rep(NA,M.it.total[k]),beta1.ciu=rep(NA,M.it.total[k]),beta2=rep(NA,M.it.total[k]),beta2.sd=rep(NA,M.it.total[k]),beta2.cil=rep(NA,M.it.total[k]),beta2.ciu=rep(NA,M.it.total[k]),sigma=rep(NA,M.it.total[k]),sigma.sd=rep(NA,M.it.total[k]),sigma.cil=rep(NA,M.it.total[k]),sigma.ciu=rep(NA,M.it.total[k]),sigma.mode=rep(NA,M.it.total[k]),rho=rep(NA,M.it.total[k]),rho.sd=rep(NA,M.it.total[k]),rho.cil=rep(NA,M.it.total[k]),rho.ciu=rep(NA,M.it.total[k]),rho.mode=rep(NA,M.it.total[k]))
    # list.param$run.df <- list(time=rep(NA,M.it.total[k]),cpo=vector(mode="list",length=M.it.total[k]),waic=rep(NA,M.it.total[k]),dic=rep(NA,M.it.total[k])) # if we also want to output the CPO we can use this commented out line instead of the above line.
    list.param$run.df <- list(time=rep(NA,M.it.total[k]),waic=rep(NA,M.it.total[k]),dic=rep(NA,M.it.total[k]))
    list.param$mess.ls <- list(error=rep(NA,M.it.total[k]),warning=rep(NA,M.it.total[k]),FFT=rep(NA,M.it.total[k]),message=vector(mode="list",length=M.it.total[k]))
    
    list.meshparam <- lapply(list.mesh,function(x){x <- list.param})
    run.out <- lapply(list.grid,function(x){x <- list.meshparam})
    
    grid.start.ind <- 1
    mesh.start.ind <- 1
    
  } else{
    load(save.file)
    p.length <- sum(!is.na(run.out[[N.g]][[N.m]]$est.df$beta0)) + sum(!is.na(run.out[[N.g]][[N.m]]$mess.ls$error)) + 1
    
    g.s <- function(g){sapply(1:length(g),function(i){sum(!is.na(g[[i]]$est.df$beta0))})}
    e.s <- function(g){sapply(1:length(g),function(i){sum(!is.na(g[[i]]$mess.ls$error))})}
    s <- sapply(1:length(run.out),function(i){g.s(run.out[[i]]) + e.s(run.out[[i]])})
    ds <- diff(s)
    if (sum(ds)!=0){
      w <- which(ds!=0,arr.ind = T)
      grid.start.ind <- unname(w)[2]
      mesh.start.ind <- unname(w)[1] + 1
    } else if (sum(diff(t(s))!=0)){
      w <- which(diff(t(s))!=0,arr.ind=TRUE) # should be easy to extract the common row, then +1 to get the required GRID that needs to begin running...
      grid.start.ind <- unname(w)[1,1] + 1
      mesh.start.ind <- unname(w)[1,2]
    } else {
      grid.start.ind <- 1
      mesh.start.ind <- 1
    }
  } 
  
  # What values chosen?
  theta.tilde <- list(beta0.tilde=3,beta1.tilde=0.75,beta2.tilde=-0.5,sigma.tilde=1,rho.tilde=2)
  for (i in p.length:M.it.total[k]){
    
    if (i!=p.length){ # when we carry on, then we want to re-start at grid 1
      grid.start.ind <- 1
      mesh.start.ind <- 1
    }
    
    # If we are already part-way through running the Grid-Mesh combinations for a simulated data set, 1,...,N then there is no need to re-simulate the data, load up the pre-saved data and continue running at the next combination of grid and mesh resolutions.
    if (grid.start.ind==1 & mesh.start.ind==1){
      seed <- (k-1)*sum(M.it.total[0:(k-1)]) + i # (k-1) not really necessary but...
      set.seed(5*seed)
      mu.true <- cov.surface.gen(W=W,theta=theta.tilde,cov1.im=popb.im,cov2.im=incb.im,int.im=intb.im)
      data.sim <- data.gen(W=W,theta=theta.tilde,disc.full=disc.full,mu=mu.true,quads=quad.list,coord=coord.list,c1.list=cov1grid.list,c2.list=cov2grid.list) # simulate a data-set from pi(y|theta.tilde)
      save(data.sim,file=paste0("temp_data",k,"irreglgcptrad.rda"))
    } else {
      load(paste0("temp_data",k,"irreglgcptrad.rda"))
    }
    
    for (j in grid.start.ind:N.g){
      data <- data.sim$data[[j]]
      print(N.gridx[j])
      print(N.gridy[j])
      
      # If we move past the re-start location then we want the mesh to start at 1.
      if (j==grid.start.ind & i==p.length){
        mesh.start.ind <- mesh.start.ind
      } else {
        mesh.start.ind <- 1
      }
      
      for (l in mesh.start.ind:N.m){
        mesh <- mesh.list[[l]]
        print(mesh.edge[l])
        ind <- (j-1)*N.m + l
        str <- A_stack.gen(data.gm=data,mesh=mesh,sigma.star=sigma.star,rho.star=rho.star)
        start.time <- proc.time()
        fit.inla <- try(inla(resp ~ 0 + offset(larea) + int + cov1 + cov2 + f(field,model=str$spde), family="poisson", data=inla.stack.data(str$stack.est),control.predictor=list(A=inla.stack.A(str$stack.est),link=1,compute=TRUE),control.fixed=list(mean=list(int=3, cov1=0.75, cov2=-0.5),prec=list(int=1, cov1=4, cov2=4)),control.compute = list(config=TRUE,cpo=FALSE,waic=TRUE,dic=TRUE))) # include cpo=TRUE if we also want to consider the CPO output
        end.time <- proc.time()
        
        if (class(fit.inla)=="try-error"){
          # If there is an error, print the value of the offset that caused the error, otherwise, carry on.
          run.out[[j]][[l]]$mess.ls$error[i] <- "ERROR"
        } else if (length(grep('Fail to factorize',fit.inla$logfile)) > fft.threshold) {
          run.out[[j]][[l]]$mess.ls$FFT[i] <- length(grep('Fail to factorize',fit.inla$logfile))
          if (length(grep('WARNING',fit.inla$logfile)) > 0){
            run.out[[j]][[l]]$mess.ls$warning[i] <- "WARNING"
            run.out[[j]][[l]]$mess.ls$message[[i]] <- fit.inla$logfile[(grep('WARNING',fit.inla$logfile))]
          }
        } else if (length(grep('WARNING',fit.inla$logfile)) > 0) {
          run.out[[j]][[l]]$mess.ls$warning[i] <- "WARNING"
          run.out[[j]][[l]]$mess.ls$message[[i]] <- fit.inla$logfile[(grep('WARNING',fit.inla$logfile))]
        } else {
          run.out[[j]][[l]]$mess.ls$FFT[i] <- length(grep('Fail to factorize',fit.inla$logfile)) # in case there were some messages, but below the threshold, want to keep track of any messages.
        }  
        if (class(fit.inla)!="try-error"){
          time.taken <- unname(end.time[3] - start.time[3])
          
          # Put results of approximations into the output data set.
          run.out[[j]][[l]]$run.df$time[i] <- time.taken
          # run.out[[j]][[l]]$run.df$cpo[[i]] <- fit.inla$cpo # Need to tell inla to calculate this, uncomment line if we want to output the cpo
          run.out[[j]][[l]]$run.df$waic[i] <- fit.inla$waic$waic # Need to tell inla to calculate this too!
          run.out[[j]][[l]]$run.df$dic[i] <- fit.inla$dic$dic # Need to tell inla to calculate this too!
          # Posterior Mean
          run.out[[j]][[l]]$est.df$beta0[i] <- fit.inla$summary.fixed$mean[1]
          run.out[[j]][[l]]$est.df$beta1[i] <- fit.inla$summary.fixed$mean[2]
          run.out[[j]][[l]]$est.df$beta2[i] <- fit.inla$summary.fixed$mean[3]
          run.out[[j]][[l]]$est.df$sigma[i] <- fit.inla$summary.hyperpar$mean[2]
          run.out[[j]][[l]]$est.df$rho[i] <- fit.inla$summary.hyperpar$mean[1]
          # Posterior SD
          run.out[[j]][[l]]$est.df$beta0.sd[i] <- fit.inla$summary.fixed$sd[1]
          run.out[[j]][[l]]$est.df$beta1.sd[i] <- fit.inla$summary.fixed$sd[2]
          run.out[[j]][[l]]$est.df$beta2.sd[i] <- fit.inla$summary.fixed$sd[3]
          run.out[[j]][[l]]$est.df$sigma.sd[i] <- fit.inla$summary.hyperpar$sd[2]
          run.out[[j]][[l]]$est.df$rho.sd[i] <- fit.inla$summary.hyperpar$sd[1]
          # Mode for Sigma and Rho
          run.out[[j]][[l]]$est.df$sigma.mode[i] <- fit.inla$summary.hyperpar$mode[2]
          run.out[[j]][[l]]$est.df$rho.mode[i] <- fit.inla$summary.hyperpar$mode[1]
          # Posterior 2.5%
          run.out[[j]][[l]]$est.df$beta0.cil[i] <- fit.inla$summary.fixed$`0.025quant`[1]
          run.out[[j]][[l]]$est.df$beta1.cil[i] <- fit.inla$summary.fixed$`0.025quant`[2]
          run.out[[j]][[l]]$est.df$beta2.cil[i] <- fit.inla$summary.fixed$`0.025quant`[3]
          run.out[[j]][[l]]$est.df$sigma.cil[i] <- fit.inla$summary.hyperpar$`0.025quant`[2]
          run.out[[j]][[l]]$est.df$rho.cil[i] <- fit.inla$summary.hyperpar$`0.025quant`[1]
          # Posterior 97.5%
          run.out[[j]][[l]]$est.df$beta0.ciu[i] <- fit.inla$summary.fixed$`0.975quant`[1]
          run.out[[j]][[l]]$est.df$beta1.ciu[i] <- fit.inla$summary.fixed$`0.975quant`[2]
          run.out[[j]][[l]]$est.df$beta2.ciu[i] <- fit.inla$summary.fixed$`0.975quant`[3]
          run.out[[j]][[l]]$est.df$sigma.ciu[i] <- fit.inla$summary.hyperpar$`0.975quant`[2]
          run.out[[j]][[l]]$est.df$rho.ciu[i] <- fit.inla$summary.hyperpar$`0.975quant`[1]
        }
        save(run.out,file=save.file)
      }
    }
  }
  save(run.out,file=save.file)
  
}
# Stop the clock
print(proc.time() - ptm)

print(sessionInfo())

stopCluster(parallelCluster)
#################################################################################################

#Define arrays for storing result
rm(list=ls()) # Must finish with this.

