
# LA Polygon: Grid Mesh Optimisation: SBC Sim. Study: Space Errors --------

# This script will implement the re-runs for the Space Error Simulation-Grid_Mesh combinations from the original SBC Simulation Study implementation (GridMeshOptimIrreg_final.R) for LGCPs over the LA polygon window in order to check how the algorithm behaves with respect to the posterior distribution by considering the rank statistic outputs for both the parameters and the mean field.

# Unlike the original code, we will not limit the number of processors available for the run, instead allowing each simulation to use all 16 processors in the node, so there is no need for parallelisation.

# This R script takes in the completed data sets, bar the ERRORS present, which we know are due to space errors from the runs before the temp directories were fixed, GridMeshIrregPolLGCPSBCSSi_SPACEERRORFINAL.rda, and will then re-run the required Simulation-Grid-Mesh combinations.
# This R script outputs the files saved as GridMeshIrregPolLGCPSBCSSi.rda.

# Author: Nadeen Khaleel

# Arguments Read In -------------------------------------------------------

args=(commandArgs(TRUE))
print(args)

if (length(args)==0){
  print("No arguments supplied.")
  # Set default values.
  this.node = 1 # which "node" am I on - change per job
  Nprocs.total = 1 # total number of processors across ALL nodes (/jobs)
  N = 1 # how many simulations?
  L = 1
}else{
  for (i in 1:length(args)){
    eval(parse(text=args[[i]]))
  }
}

# Example of R CMD BATCH command from job slurm scripts for implementing this simulation study:
# R CMD BATCH --vanilla '--args this.node=1 Nprocs.total=40 N=1000 L=100'  GridMeshOptim_SpaceErrorRuns_final.R gm_spacererunsirregsbclgcp1.out
# We had 5 jobs, each one run across a separate node using similar commands to the above.


# Libraries ---------------------------------------------------------------

print(paste0("N = ",N))
print(paste0("L = ",L))

# Only a few cases, so can manually set this up, unlike the timing errors.
ind.procs.all <- c(7,9,11,27,40) 

sim.ind.vec <- c(5,8,4,5,8)
grid.ind.vec <- c(4,4,3,3,3)
mesh.ind.vec <- c(3,3,1,1,1)


# Local Directory is the Temp Directory in Compute Nodes
library(unixtools) # in r_packages
unixtools::set.tempdir("/local/")

ind.procs <- ind.procs.all[this.node]
print(paste0("Process ",ind.procs))

for (tt in ind.procs){
  tdirloc.k <- paste0("/local/intmpproc-",tt)
  print(paste0("Temp Directory ",tdirloc.k, " Created"))
  td.c <- try(dir.create(tdirloc.k))
  if (inherits(td.c, "try-error") || !td.c){
    stop(paste("Fail to create directory [", tdirloc.k, "]. Stop.", sep=""))
  }
}


ptm <- proc.time()
for (k in ind.procs){
  
  library(unixtools) # r_packages
  
  # Set-Up Temp folder for each process, which was already created above and just assign it as the temporary folder for process k (=1:40)
  tdirloc.k <- paste0("/local/intmpproc-",k)
  unixtools::set.tempdir(tdirloc.k)
  
  
  print(paste0("Set Temp Directory to ",tdirloc.k))
  
  
  library(INLA)
  library(mvtnorm)
  library(sp)
  library(sf)
  library(spatstat)
  library(raster)
  library(maptools)
  library(stringr)
  library(rgeos)
  
  
  par.lic.filepath <- "./pardiso.lic" # file path for pardiso licence if in use
  inla.setOption(pardiso.license = par.lic.filepath)
  inla.setOption(num.threads=16)
  
  # Functions ---------------------------------------------------------------
  
  prior.sim <- function(n=1,rho.star,sigma.star){
    # n is the number of samples of the parameters you want sampled from their priors.
    beta.0.sim <- rnorm(n,mean=3,sd=1); beta.1.sim <- rnorm(n,mean=0.75,sd=sqrt(0.25)); beta.2.sim <- rnorm(n,mean=-0.5,sd=sqrt(0.25))
    
    # For the covariance parameters of the latent Gaussian field, I have a joint prior on the marginal standard deviation and the range, however, in order to simulate from this I need the inverse CDF to generate this from a Uniform RV. However, as this is a joint distribution I cannot quite get the correct arrangement for the inverse. Therefore, I will look at the the prior on kappa (and turn this into a pdf for rho) and then look at the conditional distribution of tau given kappa (and turn this into a pdf for sigma).
    # (https://www.tandfonline.com/doi/pdf/10.1080/01621459.2017.1415907?needAccess=true) 
    # (https://arxiv.org/pdf/1503.00256.pdf)
    
    alpha.rho <- rho.star[2]; alpha.sigma <- sigma.star[2]; rho.0 <- rho.star[1]; sigma.0 <- sigma.star[1]
    u1 <- runif(n,0,1); u2 <- runif(n,0,1)
    rho.sim <- rho.0*log(alpha.rho)/log(u1)
    sigma.sim <- sigma.0*log(1-u2)/log(alpha.sigma)
    return(list(beta0.tilde=beta.0.sim,beta1.tilde=beta.1.sim,beta2.tilde=beta.2.sim,sigma.tilde=sigma.sim,rho.tilde=rho.sim))
  }
  
  cov.surface.gen <- function(W,theta,cov1.im,cov2.im,int.im){
    beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
    # Create pixel images
    mu <- beta.0*int.im + beta.1*cov1.im + beta.2*cov2.im
    
    return(mu)
  }
  
  data.gen <- function(W,theta,disc.full,mu,quads,coord,c1.list,c2.list){
    # W is the window
    # theta are the parameters required to generate the data.
    # disc.full is the vector of discretisations across the window for which the data will be produced at.
    # mu is the mean surface from fixed effects
    # quads is the list of pre-computed quadrats
    # coord is the list of pre-computed coordinates for the points
    # ci.list is the list of covariate values for each grid cell, extract over the polygons of the quadrats
    
    # Simulate from this process with the parameters as defined above.
    beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]]; sigma <- theta[[4]]; rho <- theta[[5]];
    
    # Produce the point pattern from LGCP
    lgcp.ppp <- try(rLGCP(model="matern",mu,var=(sigma)^2,scale=rho/2,nu=1,win = W,saveLambda = TRUE)) # Note: default resolution of the simulated data is 128x128, if this is too coarse for the data aggregations (if one of the data aggregation requires a grid finer than 128x128), then also set dimyx=c(ny,nx).
    
    if (class(lgcp.ppp)!="try-error"){ # error in creating LGCP? Example is too many points to simulate
      if (lgcp.ppp$n!=0){ # if there is non-zero points, go ahead and make the count data sets
        grid.names <- c(paste0("grid",disc.full[,1],disc.full[,2]))
        mesh.names <- c(paste0("mesh",disc.full[,1],disc.full[,2]))
        data.list <- vector(mode="list",length=length(grid.names))
        names(data.list) <- grid.names
        
        for (i in 1:dim(disc.full)[1]){
          
          n.cell <- disc.full[i,]
          M <- n.cell[1]; N <- n.cell[2]
          cellsize <- c((W$xrange[2]-W$xrange[1])/n.cell[1],(W$yrange[2]-W$yrange[1])/n.cell[2])
          g <- quads[[i]]
          ord <- t(coordinates(coord[[i]])) # t() so that ord matches the previous version of ord, and so the remaining code matches.
          a <- sapply(1:g$n,function(ii){(area.owin(g$tiles[[ii]]))})
          m <- matrix(as.numeric(unlist(str_extract_all(names(g$tiles),"\\d*\\d"))),ncol=2,byrow=TRUE)
          ord.df <- data.frame(ind=1:g$n,row=m[,1],col=m[,2]) # col matches the raster definition of col=x, and row=y which makes sense visually from the plots as x<--> and y ^inc(^)
          ord.df <- ord.df[order(ord.df$col),]
          
          x.ord <- ord[1,ord.df$ind]; y.ord <- ord[2,ord.df$ind]
          
          cell.area <- a[ord.df$ind]
          q <- quadratcount(lgcp.ppp,tess=g)
          count.df <- data.frame(x=x.ord,y=y.ord,count=as.vector(q)[ord.df$ind],area=cell.area)
          
          count.df$cov1 <- c1.list[[i]][ord.df$ind]
          count.df$cov2 <- c2.list[[i]][ord.df$ind]
          
          data.list[[i]] <- count.df
        }
        # Return if the data produced non-zero number of points and no error from rLGCP functions, so all data produced as required for remaining code.
        return(list("data"=data.list,"lambda"=attributes(lgcp.ppp)$Lambda,"err"=FALSE,"zero"=FALSE))
      } else {
        # Return if there were no point simulated from rLGCP function as we would need to make note of this in the output and re-simulate the parameters in order to re-simulate the data. This is why we have a different set-up for the Traditional SImulation Study which had a fixed set of parameters, so no need to stop data generation and re-simulate parameters, just continue to simulate until a point pattern with non-zero number of points is simulated.
        return(list("err"=TRUE,"zero"=TRUE,"lgcp"=lgcp.ppp)) # err=TRUE for while loop
      }
    } else {
      # Return if there was an error output from the rLGCP function, then we must re-simulate the parameters and the data accordingly, but must make a note in the output that a particular set of parameters resulted in error.
      return(list("err"=TRUE,"zero"=FALSE))
    }
    
  }
  
  # Generate A matrix and stack data
  A_stack.gen <- function(data.gm,mesh,sigma.star,rho.star){
    the_spde <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[1],rho.star[2]),prior.sigma = c(sigma.star[1],sigma.star[2]))
    
    s.index <- inla.spde.make.index("field",n.spde=the_spde$n.spde)
    coords <- data.gm[,c("x","y")]
    coordinates(coords) <- ~ x + y
    A <- inla.spde.make.A(mesh, loc=coords)
    stk <- inla.stack(data=list(resp=data.gm$count),A=list(A,1),effects=list(c(s.index,list(int=1)),list(cov1=data.gm$cov1,cov2=data.gm$cov2,larea=log(data.gm$area))),tag='est')
    
    return(list("spde"=the_spde,"A"=A,"stack.est"=stk))
  }
  
  
  # Covariate and  Set-up --------------------------------------------------------
  
  # Only require the file names here, code for their creation can be found in the original Traditional Simulation Study and SBC Simulation Study code.
  
  cov.name <- paste0("GridMeshIrregPolLGCPSSCov.rda")

  quad.file <- "QuadratsIrregPolLGCP.rda"
  
  coord.file <- "CoordsIrregPolLGCP.rda"
  
  covgrid.file <- "CovAggGridIrregPolLGCP.rda"
  
  meshes.file <- "MeshesIrregPolLGCP.rda"

  window.file <- "WindowsIrregPolLGCP.rda"

  # Simulations -------------------------------------------------------------
  
  
  # SBC Parameters for Parallelisation
  # Assign simulations per processors and then extract the required processors and simulations for each node/job
  M.it.total <- rep(N%/%Nprocs.total,length=Nprocs.total)
  if (N%%Nprocs.total!=0){M.it.total[1:(N%%Nprocs.total)] <- M.it.total[1:(N%%Nprocs.total)] + 1}
  
  # For the seed, so can more easily run another N=1000 if needed
  SEED.N <- 2*N
  SEED.M.it.total <- rep(SEED.N%/%Nprocs.total,length=Nprocs.total)
  if (SEED.N%%Nprocs.total!=0){SEED.M.it.total[1:(SEED.N%%Nprocs.total)] <- SEED.M.it.total[1:(SEED.N%%Nprocs.total)] + 1}
  
  fft.threshold <- 5 # how many "Fail to factorise Q" warnings accepted before a warning message is produced for user
  
  load(cov.name) # load the covariates
  
  # Prior for the Gaussian latent field covariance parameters
  alpha.rho <- 0.01; alpha.sigma <- 0.1; rho.0 <- 0.35; sigma.0 <- 2
  rho.star <- c(rho.0,alpha.rho) ; sigma.star <- c(sigma.0,alpha.sigma)
  
  pred <- FALSE # want to predict onto the same set of locations, additionally, this is only useful for the creation of the correct dimension data frames for the grid-mesh mean field outputs below, but this is unused otherwise.
  
  # Saving the output
  save.file <- paste0("GridMeshIrregPolLGCPSBCSS",k,".rda")
  print(save.file)
  
  
  load(window.file)
  x.range <- diff(lacity_win.proj$xrange)
  y.range <- diff(lacity_win.proj$yrange)
  
  grid_cellsxvec <- ceiling(x.range/(1e3*c(5,2,1,0.5)))
  grid_cellsyvec <- ceiling(y.range/(1e3*c(5,2,1,0.5)))
  
  disc.full <- unname(cbind(grid_cellsxvec,grid_cellsyvec))
  param <- c("Beta0","Beta1","Beta2","Sigma","Rho")
  N.gridx <- disc.full[,1]; N.gridy <- disc.full[,2]; mesh.edge <- apply(cbind(abs((W$xrange[2]-W$xrange[1]))/N.gridx,abs((W$yrange[2]-W$yrange[1]))/N.gridy), 1, max)
  N.g <- length(N.gridx); N.m <- length(mesh.edge); N.p <- length(param)
  
  grid.ind <- paste0("Grid",N.gridx,N.gridy); mesh.ind <- paste0("Mesh",signif(mesh.edge,2));
  
  load(meshes.file)
  mesh.list <- mesh.list[1:4] # the last element in the list is the 0.2kmx0.2km mesh, which we don't want to use.
  load(quad.file)
  quad.list <- quad.list[1:4] # the last element in the list is the 0.2kmx0.2km mesh, which we don't want to use.
  load(coord.file)
  coord.list <- coord.list[1:4] # the last element in the list is the 0.2kmx0.2km mesh, which we don't want to use.
  load(covgrid.file)
  cov1grid.list <- cov1grid.list[1:4]
  cov2grid.list <- cov2grid.list[1:4]
  
  # Locations for the prediction, currently on the finest resolution, but can be altered accordingly, maybe on the finest, but an in between?
  l.dat <- length(quad.list)
  final.loc.ind <- l.dat
  
  N.f <- sapply(1:length(quad.list),function(i){quad.list[[i]]$n})
  
  load.file <- paste0("GridMeshIrregPolLGCPSBCSS",k,"_SPACEERRORFINAL.rda")
  load(load.file)
  
  i <- sim.ind.vec[this.node]
  grid.ind <- grid.ind.vec[this.node]
  mesh.ind <- mesh.ind.vec[this.node]
  
  print(paste0("Process ",k, " with Simulation ",i, " and Grid ",grid.ind," Mesh ",mesh.ind))
  err.check <- run.out[[grid.ind]][[mesh.ind]]$mess.ls$error[i]
  print(err.check)
  
  if (err.check=="ERROR"){
    
    print("Simulate Data")
    seed <- (k-1)*sum(SEED.M.it.total[0:(k-1)]) + i
    set.seed(5*seed)
    
    theta.tilde <- prior.sim(n = 1,rho.star = rho.star,sigma.star = sigma.star) # simulate theta from pi(theta)
    mu.true <- cov.surface.gen(W = W,theta = theta.tilde,cov1.im = popb.im,cov2.im = incb.im,int.im = intb.im)
    data.sim <- data.gen(W = W,theta=theta.tilde,disc.full = disc.full,mu = mu.true,quads = quad.list,coord = coord.list,c1.list = cov1grid.list,c2.list = cov2grid.list) # simulate a dataset from pi(y|theta.tilde)
    
    while(data.sim$err==TRUE){
      
      err.count <- nrow(data.err.tracker)
      data.err.tracker[(err.count + 1),1] <- k
      data.err.tracker[(err.count + 1),2] <- i
      data.err.tracker[(err.count + 1),3] <- 5*seed
      if (data.sim$zero==FALSE){
        data.err.tracker[(err.count + 1),4] <- "err"
      } else {
        data.err.tracker[(err.count + 1),4] <- "zero"
      }
      data.err.tracker[(err.count + 1),5:(length(theta.tilde)+4)] <- unlist(theta.tilde)
      
      theta.tilde <- prior.sim(n = 1,rho.star = rho.star,sigma.star = sigma.star) # simulate theta from pi(theta)
      mu.true <- cov.surface.gen(W = W,theta = theta.tilde,cov1.im = popb.im,cov2.im = incb.im,int.im = intb.im)
      data.sim <- data.gen(W = W,theta=theta.tilde,disc.full = disc.full,mu = mu.true,quads = quad.list,coord = coord.list,c1.list = cov1grid.list,c2.list = cov2grid.list) # simulate a dataset from pi(y|theta.tilde)
    }
    save(data.sim,theta.tilde,mu.true,file=paste0("temp_data",k,"irregpolsbc_spaceerrorrerun.rda"))
    
    # Do not want to alter these vectors, will store a separate vector for seed and true theta to compare with the original, to make sure that the seed used to simulate the data is the same.
    # seed.vec[i] <- 5*seed
    # true.theta[i,] <- unname(unlist(theta.tilde))
    temp.theta <- unlist(theta.tilde)
    temp.seed <- 5*seed
    save(temp.seed,temp.theta,file=paste0("SpaceError",k,"ReRunInfo.rda"))
    
    print(temp.theta)
    print(temp.seed)
    
    lambda.im <- data.sim$lambda
    lambda.fun <- as.function.im(lambda.im,W)
    log.lambda.list <- vector(mode="list",length=length(data.sim$data))
    for (l.ind in 1:length(data.sim$data)){
      final.loc <- data.sim$data[[l.ind]][,c("x","y")]
      llist <- sapply(1:dim(final.loc)[1],function(ll,loc){lambda.fun(loc[ll,1],loc[ll,2])},final.loc)
      
      log.lambda.list[[l.ind]] <- log(llist)
    }
    
    print("Running INLA now.")
    for (j in grid.ind){
      data <- data.sim$data[[j]]
      print(paste0("Total Count: ",sum(data$count)))
      
      
      print(N.gridx[j])
      print(N.gridy[j])
      
      for (l in mesh.ind){
        mesh <- mesh.list[[l]]
        print(mesh.edge[l])
        ind <- (j-1)*N.m + l
        
        str <- A_stack.gen(data.gm = data,mesh = mesh,sigma.star = sigma.star,rho.star = rho.star)
        start.time <- proc.time()
        fit.inla <- try(inla(resp ~ 0 + offset(larea) + int + cov1 + cov2 + f(field,model=str$spde), family="poisson", data=inla.stack.data(str$stack.est),control.predictor=list(A=inla.stack.A(str$stack.est),link=1,compute=TRUE),control.fixed=list(mean=list(int=3, cov1=0.75, cov2=-0.5),prec=list(int=1, cov1=4, cov2=4)),control.compute = list(config=TRUE,cpo=FALSE,waic=TRUE,dic=TRUE))) # include cpo=TRUE if we also want to consider the CPO output, only if the output, pre-re-runs has already included space for storing the CPO
        end.time <- proc.time()
        
        if (class(fit.inla)=="try-error"){
          # If there is an error, print the value of the offset that caused the error, otherwise, carry on.
          run.out[[j]][[l]]$mess.ls$error[i] <- "ERROR"
        } else if (length(grep('Fail to factorize',fit.inla$logfile)) > fft.threshold) {
          run.out[[j]][[l]]$mess.ls$FFT[i] <- length(grep('Fail to factorize',fit.inla$logfile))
          if (length(grep('WARNING',fit.inla$logfile)) > 0){
            run.out[[j]][[l]]$mess.ls$warning[i] <- "WARNING"
            run.out[[j]][[l]]$mess.ls$message[[i]] <- fit.inla$logfile[(grep('WARNING',fit.inla$logfile))]
          }
        } else if (length(grep('WARNING',fit.inla$logfile)) > 0) {
          run.out[[j]][[l]]$mess.ls$warning[i] <- "WARNING"
          run.out[[j]][[l]]$mess.ls$message[[i]] <- fit.inla$logfile[(grep('WARNING',fit.inla$logfile))]
        } else {
          run.out[[j]][[l]]$mess.ls$FFT[i] <- length(grep('Fail to factorize',fit.inla$logfile)) # incase there were some messages, but below the threshold, want to keep track of any messages.
        }  
        if (class(fit.inla)!="try-error"){
          time.taken <- unname(end.time[3] - start.time[3])
          
          print(paste0("Placing Results in Position ",i," Grid ",j," and Mesh ",l))
          
          # Put results of approximations into the output data set.
          run.out[[j]][[l]]$run.df$time[i] <- time.taken
          # run.out[[j]][[l]]$run.df$cpo[[i]] <- fit.inla$cpo # Need to tell inla to calculate this, uncomment if you want to output CPO
          run.out[[j]][[l]]$run.df$waic[i] <- fit.inla$waic$waic # Need to tell inla to calculate this too!
          run.out[[j]][[l]]$run.df$dic[i] <- fit.inla$dic$dic # Need to tell inla to calculate this too!
          # Posterior Mean
          run.out[[j]][[l]]$est.df$beta0[i] <- fit.inla$summary.fixed$mean[1]
          run.out[[j]][[l]]$est.df$beta1[i] <- fit.inla$summary.fixed$mean[2]
          run.out[[j]][[l]]$est.df$beta2[i] <- fit.inla$summary.fixed$mean[3]
          run.out[[j]][[l]]$est.df$sigma[i] <- fit.inla$summary.hyperpar$mean[2]
          run.out[[j]][[l]]$est.df$rho[i] <- fit.inla$summary.hyperpar$mean[1]
          # Posterior SD
          run.out[[j]][[l]]$est.df$beta0.sd[i] <- fit.inla$summary.fixed$sd[1]
          run.out[[j]][[l]]$est.df$beta1.sd[i] <- fit.inla$summary.fixed$sd[2]
          run.out[[j]][[l]]$est.df$beta2.sd[i] <- fit.inla$summary.fixed$sd[3]
          run.out[[j]][[l]]$est.df$sigma.sd[i] <- fit.inla$summary.hyperpar$sd[2]
          run.out[[j]][[l]]$est.df$rho.sd[i] <- fit.inla$summary.hyperpar$sd[1]
          # Mode for Sigma and Rho
          run.out[[j]][[l]]$est.df$sigma.mode[i] <- fit.inla$summary.hyperpar$mode[2]
          run.out[[j]][[l]]$est.df$rho.mode[i] <- fit.inla$summary.hyperpar$mode[1]
          # Posterior 2.5%
          run.out[[j]][[l]]$est.df$beta0.cil[i] <- fit.inla$summary.fixed$`0.025quant`[1]
          run.out[[j]][[l]]$est.df$beta1.cil[i] <- fit.inla$summary.fixed$`0.025quant`[2]
          run.out[[j]][[l]]$est.df$beta2.cil[i] <- fit.inla$summary.fixed$`0.025quant`[3]
          run.out[[j]][[l]]$est.df$sigma.cil[i] <- fit.inla$summary.hyperpar$`0.025quant`[2]
          run.out[[j]][[l]]$est.df$rho.cil[i] <- fit.inla$summary.hyperpar$`0.025quant`[1]
          # Posterior 97.5%
          run.out[[j]][[l]]$est.df$beta0.ciu[i] <- fit.inla$summary.fixed$`0.975quant`[1]
          run.out[[j]][[l]]$est.df$beta1.ciu[i] <- fit.inla$summary.fixed$`0.975quant`[2]
          run.out[[j]][[l]]$est.df$beta2.ciu[i] <- fit.inla$summary.fixed$`0.975quant`[3]
          run.out[[j]][[l]]$est.df$sigma.ciu[i] <- fit.inla$summary.hyperpar$`0.975quant`[2]
          run.out[[j]][[l]]$est.df$rho.ciu[i] <- fit.inla$summary.hyperpar$`0.975quant`[1]
          
          # Shift error message to warning along with a message about the spacing issues.
          print(run.out[[j]][[l]]$mess.ls$error[i])
          run.out[[j]][[l]]$mess.ls$warning[i] <- paste0("SPACE ",run.out[[j]][[l]]$mess.ls$error[i])
          run.out[[j]][[l]]$mess.ls$message[[i]] <- paste0("This combination of grid and mesh for this simulation has been run with 16 processors available on a single node due to space on disk warning when /tmp/ was used rather than /local/ for the temporary files after which no such warning appeared. This was noted with an error: ",run.out[[j]][[l]]$mess.ls$error[i]," which has now been replaced.")
          run.out[[j]][[l]]$mess.ls$error[i] <- NA
          print(run.out[[j]][[l]]$mess.ls$error[i])
          print(run.out[[j]][[l]]$mess.ls$warning[i])
          
          # SBC
          print("Sampling now.")
          # Labelling for the latent field and parameter output samples to select from posterior sample function outputs
          contents <- fit.inla$misc$configs$contents
          # int indices
          id.int <- which(contents$tag=="int")
          ind.int <- contents$start[id.int] - 1 + (1:contents$length[id.int])
          # beta1 indices
          id.cov1 <- which(contents$tag=="cov1")
          ind.cov1 <- contents$start[id.cov1] - 1 + (1:contents$length[id.cov1])
          # beta2 indices
          id.cov2 <- which(contents$tag=="cov2")
          ind.cov2 <- contents$start[id.cov2] - 1 + (1:contents$length[id.cov2])
          
          theta.K <- inla.posterior.sample(L,fit.inla)
          thetahyperpar.K <- inla.hyperpar.sample(L,fit.inla)
          sum.int <- 0; sum.beta1 <- 0; sum.beta2 <- 0
          for (jj in 1:L){
            sum.int <- sum.int + as.numeric(theta.K[[jj]]$latent[ind.int,1] < theta.tilde$beta0.tilde)
            sum.beta1 <- sum.beta1 + as.numeric(theta.K[[jj]]$latent[ind.cov1,1] < theta.tilde$beta1.tilde)
            sum.beta2 <- sum.beta2 + as.numeric(theta.K[[jj]]$latent[ind.cov2,1] < theta.tilde$beta2.tilde)
          }
          
          gm[[ind]]$ranks.param[i,1] <- sum.int
          gm[[ind]]$ranks.param[i,2] <- sum.beta1
          gm[[ind]]$ranks.param[i,3] <- sum.beta2
          gm[[ind]]$ranks.param[i,4] <- sum(thetahyperpar.K[,2] < theta.tilde$sigma.tilde)
          gm[[ind]]$ranks.param[i,5] <- sum(thetahyperpar.K[,1] < theta.tilde$rho.tilde)
          
          
          mf.effect <- "APredictor";
          
          pred.ind <- inla.stack.index(str$stack.est, tag='est')$data
          
          log.lambda.true <- log.lambda.list[[j]]
          
          id.mfeffect <- which(contents$tag==mf.effect)
          ind.mfeffect <- (contents$start[id.mfeffect] - 1 + (1:contents$length[id.mfeffect]))[pred.ind]
          
          # NOTE: the below code has been changed after the simulation study was run, this is because a mistake was later found: the x$latent[ind.meffect] produces the linear predictor which is not what we have in log.lambda.true. The difference between the two is the inclusion of the offset of the log cell area in the lienar predictor (which is negative due to cell areas<1), however that can be easily rectified.
          # The commented out code for gm[[ind]]$ranks.mf[i,] is the code with the TYPO, and the un-commented code is the corrected version (which can be approached either via adding the log-areas to the rhs (loglambda) or subtracting it from the lhs (lin.predictor))
          # gm[[ind]]$ranks.mf[i,] <- rowSums(sapply(theta.K,function(x){x$latent[ind.mfeffect]<log.lambda.true}))
          gm[[ind]]$ranks.mf[i,] <- rowSums(sapply(theta.K,function(x){x$latent[ind.mfeffect]<log.lambda.true + log(data$area)}))
        }
        save(run.out,gm,true.theta,data.err.tracker,seed.vec,file=save.file)
      }
    }
    save(run.out,gm,true.theta,data.err.tracker,seed.vec,file=save.file)
  }
}
# Stop the clock
print(proc.time() - ptm)

# stopCluster(parallelCluster)
#################################################################################################

sessionInfo()
for (tt in ind.procs){
  tdirloc.k <- paste0("/local/intmpproc-",tt)
  print(paste0("Unlinking Temp Directory ", tdirloc.k))
  unlink(tdirloc.k,recursive=T)
}



#Define arrays for storing result
rm(list=ls()) # Must finish with this.

