
# LA Polygon: Grid Mesh Optimisation: SBC Sim. Study: Time Errors ---------

# This script will implement the re-runs for the Time Error Simulation-Grid_Mesh combinations from the original SBC Simulation Study implementation (GridMeshOptimIrreg_final.R) for LGCPs over the LA polygon window in order to check how the algorithm behaves with respect to the posterior distribution by considering the rank statistic outputs for both the parameters and the mean field.

# This R script loads up a pre-run data-set containing a data frame with a list of simulation, grid and mesh resolution indices and labels for the time errors (TIME ERROR/TIME ERROR 2/etc...) in order to re-run the required combinations. Unlike the original code, we will not limit the number of processors available for the run, instead allowing each simulation to use all 16 processors in the node, so there is no need for parallelisation.

# This R script takes in the completed data sets, bar the TIME ERRORS present, GridMeshIrregPolLGCPSBCSSi_TIMEERRORFINAL.rda, and will then re-run the required Simulation-Grid-Mesh combinations.
# This R script outputs the files saved as GridMeshIrregPolLGCPSBCSSi.rda. Either the time errors have been re-run, or they took too long or had an error output and therefore, no results for this S-G-M.

# Author: Nadeen Khaleel

# Arguments Read In -------------------------------------------------------

args=(commandArgs(TRUE))
print(args)

if (length(args)==0){
  print("No arguments supplied.")
  # Set default values.
  this.node = 1 # which "node" am I on - change per job
  total.nodes = 1 # how many nodes am I using?
  N = 1 # how many simulations?
  L = 1
}else{
  for (i in 1:length(args)){
    eval(parse(text=args[[i]]))
  }
}

# Example of R CMD BATCH command from job slurm scripts for implementing this simulation study:
# R CMD BATCH --vanilla '--args this.node=1 total.nodes=15 N=1000 L=100'  GridMeshOptim_TimeErrorRuns_final.R gm_rerunsirregsbclgcp1.out
# We had 15 jobs, each one run across a separate node using similar commands to the above.

# Libraries ---------------------------------------------------------------

# Local Directory is the Temp Directory in Compute Nodes
library(unixtools) # in r_packages
unixtools::set.tempdir("/local/")

ptm <- proc.time()

library(unixtools) # r_packages
library(INLA)
library(mvtnorm)
library(sp)
library(sf)
library(spatstat)
library(raster)
library(maptools)
library(stringr)
library(rgeos)


par.lic.filepath <- "./pardiso.lic" # file path for pardiso licence if in use
inla.setOption(pardiso.license = par.lic.filepath)
inla.setOption(num.threads=16)

# Functions ---------------------------------------------------------------

# Generate A matrix and stack data
A_stack.gen <- function(data.gm,mesh,sigma.star,rho.star){
  the_spde <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[1],rho.star[2]),prior.sigma = c(sigma.star[1],sigma.star[2]))
  
  s.index <- inla.spde.make.index("field",n.spde=the_spde$n.spde)
  coords <- data.gm[,c("x","y")]
  coordinates(coords) <- ~ x + y
  A <- inla.spde.make.A(mesh, loc=coords)
  stk <- inla.stack(data=list(resp=data.gm$count),A=list(A,1),effects=list(c(s.index,list(int=1)),list(cov1=data.gm$cov1,cov2=data.gm$cov2,larea=log(data.gm$area))),tag='est')
  
  return(list("spde"=the_spde,"A"=A,"stack.est"=stk))
}

# Covariate and  Set-up --------------------------------------------------------

# Only require the file names here, code for their creation can be found in the original Traditional Simulation Study and SBC Simulation Study code.

cov.name <- paste0("GridMeshIrregPolLGCPSSCov.rda")

quad.file <- "QuadratsIrregPolLGCP.rda"

coord.file <- "CoordsIrregPolLGCP.rda"

covgrid.file <- "CovAggGridIrregPolLGCP.rda"

meshes.file <- "MeshesIrregPolLGCP.rda"

window.file <- "WindowsIrregPolLGCP.rda"

# Simulations -------------------------------------------------------------

fft.threshold <- 5 # how many "Fail to factorise Q" warnings accepted before a warning message is produced for user

load(cov.name) # load the covariates

# Prior for the Gaussian latent field covariance parameters
alpha.rho <- 0.01; alpha.sigma <- 0.1; rho.0 <- 0.35; sigma.0 <- 2
rho.star <- c(rho.0,alpha.rho) ; sigma.star <- c(sigma.0,alpha.sigma)

pred <- FALSE # want to predict onto the same set of locations, additionally, this is only useful for the creation of the correct dimension data frames for the grid-mesh mean field outputs below, but this is unused otherwise.

load(window.file)
x.range <- diff(lacity_win.proj$xrange)
y.range <- diff(lacity_win.proj$yrange)

grid_cellsxvec <- ceiling(x.range/(1e3*c(5,2,1,0.5)))
grid_cellsyvec <- ceiling(y.range/(1e3*c(5,2,1,0.5)))

disc.full <- unname(cbind(grid_cellsxvec,grid_cellsyvec))
param <- c("Beta0","Beta1","Beta2","Sigma","Rho")
N.gridx <- disc.full[,1]; N.gridy <- disc.full[,2]; mesh.edge <- apply(cbind(abs((W$xrange[2]-W$xrange[1]))/N.gridx,abs((W$yrange[2]-W$yrange[1]))/N.gridy), 1, max)
N.g <- length(N.gridx); N.m <- length(mesh.edge); N.p <- length(param)

grid.ind <- paste0("Grid",N.gridx,N.gridy); mesh.ind <- paste0("Mesh",signif(mesh.edge,2));

load(meshes.file)
mesh.list <- mesh.list[1:4] # the last element in the list is the 0.2kmx0.2km mesh, which we don't want to use.
load(quad.file)
quad.list <- quad.list[1:4] # the last element in the list is the 0.2kmx0.2km mesh, which we don't want to use.
load(coord.file)
coord.list <- coord.list[1:4] # the last element in the list is the 0.2kmx0.2km mesh, which we don't want to use.
load(covgrid.file)
cov1grid.list <- cov1grid.list[1:4]
cov2grid.list <- cov2grid.list[1:4]


# Locations for the prediction, currently on the finest resolution, but can be altered accordingly, maybe on the finest, but an in between?
l.dat <- length(quad.list)
final.loc.ind <- l.dat

N.f <- sapply(1:length(quad.list),function(i){quad.list[[i]]$n})

print(paste0("N = ",N))
print(paste0("L = ",L))

# Pre-run data set with required details about the indices for Simulation-Grid-Mesh and Time Error, created from TimeErrorProcessandDataGeneration.R
load("TimingErrorDataFrames.rda")

# Set-up which nodes will be running which re-runs, set-up manually in order to make it easier to change when certain runs finish, with an example of such a case below.
proc.list <- list("N1"=c(2,7,9),"N2"=c(5),"N3"=c(6,10,17),"N4"=c(8),"N5"=c(14),"N6"=c(16),"N7"=c(18),"N8"=c(19,20),"N9"=c(23),"N10"=c(27),"N11"=c(29,32),"N12"=c(30),"N13"=c(33),"N14"=c(35),"N15"=c(38))
# # Just to remove P2 and P9 from J1 as they have completed, no need to re-load and save the exact same data set.
# proc.list <- list("N1"=c(7),"N2"=c(5),"N3"=c(6,10,17),"N4"=c(8),"N5"=c(14),"N6"=c(16),"N7"=c(18),"N8"=c(19,20),"N9"=c(23),"N10"=c(27),"N11"=c(29,32),"N12"=c(30),"N13"=c(33),"N14"=c(35),"N15"=c(38))

# Set-up list for each node for running.
err.df.order <- vector(mode="list",length=length(proc.list))

regexp <- "[[:digit:]]+"
tab.err.lab.ind <- data.frame(ind.ord=1:length(err.df.list),proc=as.numeric(str_extract(names(err.df.list),"[[:digit:]]+")))
for (i in 1:length(err.df.order)){
  err.df.order[[i]] <- tab.err.lab.ind$ind[match(proc.list[[i]],tab.err.lab.ind$proc)]
}
proc.node <- proc.list[[this.node]]
print(proc.node)
print(err.df.order[[this.node]])
print(match(proc.node,err.procs)) # this should be the same as the above!

for (tt in proc.node){
  tdirloc.k <- paste0("/local/intmpproc-",tt)
  print(paste0("Temp Directory ",tdirloc.k, " Created"))
  td.c <- try(dir.create(tdirloc.k))
  if (inherits(td.c, "try-error") || !td.c){
    stop(paste("Fail to create directory [", tdirloc.k, "]. Stop.", sep=""))
  }
}

# Want to run each on 16 cores, no need for parallelising, leave that to INLA which can use all available processors (16) on the compute node.
for (p in 1:length(proc.node)){
  
  k <- proc.node[p]
  err.df.ind <- err.df.order[[this.node]][p]
  
  
  # Set-Up Temp folder for each process, which was already created above and just assign it as the temporary folder for process k (=1:40)
  tdirloc.k <- paste0("/local/intmpproc-",k)
  unixtools::set.tempdir(tdirloc.k)
  
  print(paste0("Set Temp Directory to ",tdirloc.k))
  
  
  if (!file.exists(paste0("Process",k,"ErrorDf.rda"))){
    err.df.pk.orig <- err.df.list[[err.df.ind]]
    save(err.df.pk.orig,file=paste0("Process",k,"ErrorDf.rda"))
  } else {
    load(paste0("Process",k,"ErrorDf.rda"))
  }
  
  
  # Saving the output
  save.file <- paste0("GridMeshIrregPolLGCPSBCSS",k,".rda")
  print(save.file)
  
  if (file.exists(save.file)&(sum(err.df.pk.orig$rerun)>0)){
    print(paste0("Already started for Process ",k," so re-load data and continue."))
    load(save.file)
  } else {
    print(paste0("Have yet to start for Process ",k," so start new."))
    load.file <- paste0("GridMeshIrregPolLGCPSBCSS",k,"_TIMEERRORFINAL.rda")
    load(load.file)
  }
  
  print(err.df.pk.orig)
  
  for (t in 1:nrow(err.df.pk.orig)){
    if (err.df.pk.orig$rerun[t]==0){ # 0 if not re-run yet, 1 if re-run and completed, 2 if re-run and error produced (for example space error), or possibly the re-run took > 12 hours at which point it is considered an error, with no re-run considered to try to complete.
      time.err <- err.df.pk.orig$err.time[t]
      data.temp <- (paste0("TIMEERROR",time.err,"/temp_data",k,"irregpolsbc_timeerror",time.err,".rda"))
      if (!file.exists(data.temp)){
        stop("The data for this timing error does not exist!")
      } else { # Should have the required data set-up in the following folder for each re-run required.
        print(paste0("Loading ",data.temp))
        load(data.temp)
        print(paste0("Total Count: ",sum(data.sim$data[[1]]$count)))
      }
      
      lambda.im <- data.sim$lambda
      lambda.fun <- as.function.im(lambda.im,W)
      log.lambda.list <- vector(mode="list",length=length(data.sim$data))
      for (l.ind in 1:length(data.sim$data)){
        final.loc <- data.sim$data[[l.ind]][,c("x","y")]
        llist <- sapply(1:dim(final.loc)[1],function(ll,loc){lambda.fun(loc[ll,1],loc[ll,2])},final.loc)
        log.lambda.list[[l.ind]] <- log(llist)
      }
      
      # Get Simulation-Grid-Mesh needed to replace!
      sim.ind.err <- err.df.pk.orig$sim[t]
      
      grid.ind.err <- err.df.pk.orig$grid[t]
      mesh.ind.err <- err.df.pk.orig$mesh[t]
      
      print(paste0("Process ",k))
      print(paste0("Simulation ",sim.ind.err))
      print(paste0("Grid ",grid.ind.err))
      print(paste0("Mesh ",mesh.ind.err))
      
      
      i <- sim.ind.err
      print(i)
      
      for (j in grid.ind.err){
        data <- data.sim$data[[j]]
        
        print(N.gridx[j])
        print(N.gridy[j])
        
        for (l in mesh.ind.err){
          mesh <- mesh.list[[l]]
          print(mesh.edge[l])
          ind <- (j-1)*N.m + l
          
          print("Run INLA for the above settings.")
          str <- A_stack.gen(data.gm = data,mesh = mesh,sigma.star = sigma.star,rho.star = rho.star)
          start.time <- proc.time()
          fit.inla <- try(inla(resp ~ 0 + offset(larea) + int + cov1 + cov2 + f(field,model=str$spde), family="poisson", data=inla.stack.data(str$stack.est),control.predictor=list(A=inla.stack.A(str$stack.est),link=1,compute=TRUE),control.fixed=list(mean=list(int=3, cov1=0.75, cov2=-0.5),prec=list(int=1, cov1=4, cov2=4)),control.compute = list(config=TRUE,cpo=FALSE,waic=TRUE,dic=TRUE))) # include cpo=TRUE if we also want to consider the CPO output
          end.time <- proc.time()
          
          if (class(fit.inla)=="try-error"){
            # If there is an error, print the value of the offset that caused the error, otherwise, carry on.
            run.out[[j]][[l]]$mess.ls$error[i] <- "ERROR"
          } else if (length(grep('Fail to factorize',fit.inla$logfile)) > fft.threshold) {
            run.out[[j]][[l]]$mess.ls$FFT[i] <- length(grep('Fail to factorize',fit.inla$logfile))
            if (length(grep('WARNING',fit.inla$logfile)) > 0){
              run.out[[j]][[l]]$mess.ls$warning[i] <- "WARNING"
              run.out[[j]][[l]]$mess.ls$message[[i]] <- fit.inla$logfile[(grep('WARNING',fit.inla$logfile))]
            }
          } else if (length(grep('WARNING',fit.inla$logfile)) > 0) {
            run.out[[j]][[l]]$mess.ls$warning[i] <- "WARNING"
            run.out[[j]][[l]]$mess.ls$message[[i]] <- fit.inla$logfile[(grep('WARNING',fit.inla$logfile))]
          } else {
            run.out[[j]][[l]]$mess.ls$FFT[i] <- length(grep('Fail to factorize',fit.inla$logfile)) # incase there were some messages, but below the threshold, want to keep track of any messages.
          }  
          if (class(fit.inla)!="try-error"){
            time.taken <- unname(end.time[3] - start.time[3])
            
            
            print(paste0("Placing Results in Position ",i," Grid ",j," and Mesh ",l))
            
            # Put results of approximations into the output data set.
            run.out[[j]][[l]]$run.df$time[i] <- time.taken
            # run.out[[j]][[l]]$run.df$cpo[[i]] <- fit.inla$cpo # Need to tell inla to calculate this, uncomment if you want to output CPO
            run.out[[j]][[l]]$run.df$waic[i] <- fit.inla$waic$waic # Need to tell inla to calculate this too!
            run.out[[j]][[l]]$run.df$dic[i] <- fit.inla$dic$dic # Need to tell inla to calculate this too!
            # Posterior Mean
            run.out[[j]][[l]]$est.df$beta0[i] <- fit.inla$summary.fixed$mean[1]
            run.out[[j]][[l]]$est.df$beta1[i] <- fit.inla$summary.fixed$mean[2]
            run.out[[j]][[l]]$est.df$beta2[i] <- fit.inla$summary.fixed$mean[3]
            run.out[[j]][[l]]$est.df$sigma[i] <- fit.inla$summary.hyperpar$mean[2]
            run.out[[j]][[l]]$est.df$rho[i] <- fit.inla$summary.hyperpar$mean[1]
            # Posterior SD
            run.out[[j]][[l]]$est.df$beta0.sd[i] <- fit.inla$summary.fixed$sd[1]
            run.out[[j]][[l]]$est.df$beta1.sd[i] <- fit.inla$summary.fixed$sd[2]
            run.out[[j]][[l]]$est.df$beta2.sd[i] <- fit.inla$summary.fixed$sd[3]
            run.out[[j]][[l]]$est.df$sigma.sd[i] <- fit.inla$summary.hyperpar$sd[2]
            run.out[[j]][[l]]$est.df$rho.sd[i] <- fit.inla$summary.hyperpar$sd[1]
            # Mode for Sigma and Rho
            run.out[[j]][[l]]$est.df$sigma.mode[i] <- fit.inla$summary.hyperpar$mode[2]
            run.out[[j]][[l]]$est.df$rho.mode[i] <- fit.inla$summary.hyperpar$mode[1]
            # Posterior 2.5%
            run.out[[j]][[l]]$est.df$beta0.cil[i] <- fit.inla$summary.fixed$`0.025quant`[1]
            run.out[[j]][[l]]$est.df$beta1.cil[i] <- fit.inla$summary.fixed$`0.025quant`[2]
            run.out[[j]][[l]]$est.df$beta2.cil[i] <- fit.inla$summary.fixed$`0.025quant`[3]
            run.out[[j]][[l]]$est.df$sigma.cil[i] <- fit.inla$summary.hyperpar$`0.025quant`[2]
            run.out[[j]][[l]]$est.df$rho.cil[i] <- fit.inla$summary.hyperpar$`0.025quant`[1]
            # Posterior 97.5%
            run.out[[j]][[l]]$est.df$beta0.ciu[i] <- fit.inla$summary.fixed$`0.975quant`[1]
            run.out[[j]][[l]]$est.df$beta1.ciu[i] <- fit.inla$summary.fixed$`0.975quant`[2]
            run.out[[j]][[l]]$est.df$beta2.ciu[i] <- fit.inla$summary.fixed$`0.975quant`[3]
            run.out[[j]][[l]]$est.df$sigma.ciu[i] <- fit.inla$summary.hyperpar$`0.975quant`[2]
            run.out[[j]][[l]]$est.df$rho.ciu[i] <- fit.inla$summary.hyperpar$`0.975quant`[1]
            
            # Shift error message to warning along with a message about the timing issues.
            print(run.out[[j]][[l]]$mess.ls$error[i])
            run.out[[j]][[l]]$mess.ls$warning[i] <- run.out[[j]][[l]]$mess.ls$error[i]
            run.out[[j]][[l]]$mess.ls$message[[i]] <- paste0("This combination of grid and mesh for this simulation has been run with 16 processors available on a single node due to potential/noted timing errors. This was noted with an error: ",run.out[[j]][[l]]$mess.ls$error[i]," which has now been replaced.")
            run.out[[j]][[l]]$mess.ls$error[i] <- NA
            print(run.out[[j]][[l]]$mess.ls$error[i])
            print(run.out[[j]][[l]]$mess.ls$warning[i])
            
            # SBC
            
            print("Sampling now.")
            # Labelling for the latent field and parameter output samples to select from posterior sample function outputs
            contents <- fit.inla$misc$configs$contents
            # int indices
            id.int <- which(contents$tag=="int")
            ind.int <- contents$start[id.int] - 1 + (1:contents$length[id.int])
            # beta1 indices
            id.cov1 <- which(contents$tag=="cov1")
            ind.cov1 <- contents$start[id.cov1] - 1 + (1:contents$length[id.cov1])
            # beta2 indices
            id.cov2 <- which(contents$tag=="cov2")
            ind.cov2 <- contents$start[id.cov2] - 1 + (1:contents$length[id.cov2])
            
            theta.K <- inla.posterior.sample(L,fit.inla)
            thetahyperpar.K <- inla.hyperpar.sample(L,fit.inla)
            sum.int <- 0; sum.beta1 <- 0; sum.beta2 <- 0
            
            for (jj in 1:L){
              sum.int <- sum.int + as.numeric(theta.K[[jj]]$latent[ind.int,1] < theta.tilde$beta0.tilde)
              sum.beta1 <- sum.beta1 + as.numeric(theta.K[[jj]]$latent[ind.cov1,1] < theta.tilde$beta1.tilde)
              sum.beta2 <- sum.beta2 + as.numeric(theta.K[[jj]]$latent[ind.cov2,1] < theta.tilde$beta2.tilde)
            }
            
            gm[[ind]]$ranks.param[i,1] <- sum.int
            gm[[ind]]$ranks.param[i,2] <- sum.beta1
            gm[[ind]]$ranks.param[i,3] <- sum.beta2
            gm[[ind]]$ranks.param[i,4] <- sum(thetahyperpar.K[,2] < theta.tilde$sigma.tilde)
            gm[[ind]]$ranks.param[i,5] <- sum(thetahyperpar.K[,1] < theta.tilde$rho.tilde)
            
            
            mf.effect <- "APredictor";
            pred.ind <- inla.stack.index(str$stack.est, tag='est')$data
            
            log.lambda.true <- log.lambda.list[[j]]
            
            id.mfeffect <- which(contents$tag==mf.effect)
            ind.mfeffect <- (contents$start[id.mfeffect] - 1 + (1:contents$length[id.mfeffect]))[pred.ind]
            
            # NOTE: the below code has been changed after the simulation study was run, this is because a mistake was later found: the x$latent[ind.meffect] produces the linear predictor which is not what we have in log.lambda.true. The difference between the two is the inclusion of the offset of the log cell area in the lienar predictor (which is negative due to cell areas<1), however that can be easily rectified.
            # The commented out code for gm[[ind]]$ranks.mf[i,] is the code with the TYPO, and the un-commented code is the corrected version (which can be approached either via adding the log-areas to the rhs (loglambda) or subtracting it from the lhs (lin.predictor))
            # gm[[ind]]$ranks.mf[i,] <- rowSums(sapply(theta.K,function(x){x$latent[ind.mfeffect]<log.lambda.true}))
            gm[[ind]]$ranks.mf[i,] <- rowSums(sapply(theta.K,function(x){x$latent[ind.mfeffect]<log.lambda.true + log(data$area)}))
          }
          save(run.out,gm,true.theta,data.err.tracker,seed.vec,file=save.file)
        }
      }
      
      err.df.pk.orig$rerun[t] <- 1
      print(err.df.pk.orig)
      save(err.df.pk.orig,file=paste0("Process",k,"ErrorDf.rda")) # output updated error file with new information about completed re-runs.
    }
  }
  save(run.out,gm,true.theta,data.err.tracker,seed.vec,file=save.file)
}
# Stop the clock
print(proc.time() - ptm)

#################################################################################################

sessionInfo()

for (tt in proc.node){
  tdirloc.k <- paste0("/local/intmpproc-",tt)
  print(paste0("Unlinking Temp Directory ", tdirloc.k))
  unlink(tdirloc.k,recursive=T)
}


#Define arrays for storing result
rm(list=ls()) # Must finish with this.

