
# Plotting Simulation Study Data ------------------------------------------

# This script takes the data sets simulated and plots the covariates as well as the count data and count data for simulated data sets in comparison.
# We will use tmap for these plots rather than base R.
# While meshes were not plotted here, a few quick lines can plot them and they have been included within the section for plotting the covariates and grid but left commented.

# Author: Nadeen Khaleel

# Setwd and Load Libraries ------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))

library(spatstat)
library(raster)
library(sf)
library(sp)
library(maptools)
library(rgdal)
library(lwgeom)
library(RandomFields)
library(tmap)
library(tmaptools)


# Load Data ---------------------------------------------------------------

# LA Census Tract
load("../../../../DATA/PROCESSED_DATA/SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")
lacity_boundary <- st_union(ct_LA)
lacity_proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611") # changed as prev. is deprecated, no need for +init=epsg...
lacity_sp <- as(lacity_proj,"Spatial") #st_transform for crs...
lacity_win.proj <- as.owin.SpatialPolygons((lacity_sp))
## Transform the window
bbox <- lacity_sp@bbox
lacity_spshift <- elide(lacity_sp,shift=-c(bbox[1,1],bbox[2,1]))
bbox_shift <- lacity_spshift@bbox
lacity_spscale <- elide(lacity_spshift,scale=max(bbox_shift)/1e4)
W <- as.owin.SpatialPolygons((lacity_spscale))

## Get covariates
cov.name <- "GridMeshIrregPolLGCPSSCov.rda"
load(cov.name)
quad.name <- "QuadratsIrregPolLGCP.rda"
load(quad.name)


# Real Data
la_hom_sf <- readRDS("../../../../DATA/PROCESSED_DATA/CRIME/POINT_PATTERN/la_hom_sf.rds")
la_gta_sf <- readRDS("../../../../DATA/PROCESSED_DATA/CRIME/POINT_PATTERN/la_gta_sf.rds")

# Project to UTM
la_hom_sf.proj <- lwgeom::st_transform_proj(la_hom_sf,"epsg:32611")
la_gta_sf.proj <- lwgeom::st_transform_proj(la_gta_sf,"epsg:32611")

# Extract 2015 data
la_hom15_sf.proj <- la_hom_sf.proj[which(la_hom_sf.proj$Y==2015),]
la_gta15_sf.proj <- la_gta_sf.proj[which(la_gta_sf.proj$Y==2015),]

la_hom15_sp.proj <- as(la_hom15_sf.proj,"Spatial")
la_gta15_sp.proj <- as(la_gta15_sf.proj,"Spatial")

la_hom15_loc.proj <- la_hom15_sp.proj@coords
colnames(la_hom15_loc.proj) <- c("x","y")
la_gta15_loc.proj <- la_gta15_sp.proj@coords
colnames(la_gta15_loc.proj) <- c("x","y")

# Turn into ppp
la_hom15.proj.ppp <- as.ppp(la_hom15_loc.proj,lacity_win.proj) # 1 point lying outside window?
la_gta15.proj.ppp <- as.ppp(la_gta15_loc.proj,lacity_win.proj) # 35 points lying outside window?

# Aggregate homicide and motor vehicle point patterns to 2km and 1km grid resolutions for plot comparisons
quad2436 <- quadrats(lacity_win.proj,24,36)
h1 <- quadratcount(la_hom15.proj.ppp,tess=quad2436)
h1.sp <- as(quad2436,"SpatialPolygons")
h1.sf <- st_as_sf(h1.sp)
h1.sf$count <- h1

quad4872 <- quadrats(lacity_win.proj,48,72)
h2 <- quadratcount(la_hom15.proj.ppp,tess=quad4872)
h2.sp <- as(quad4872,"SpatialPolygons")
h2.sf <- st_as_sf(h2.sp)
h2.sf$count <- h2

quad2436 <- quadrats(lacity_win.proj,24,36)
g1 <- quadratcount(la_gta15.proj.ppp,tess=quad2436)
g1.sp <- as(quad2436,"SpatialPolygons")
g1.sf <- st_as_sf(g1.sp)
g1.sf$count <- g1

quad4872 <- quadrats(lacity_win.proj,48,72)
g2 <- quadratcount(la_gta15.proj.ppp,tess=quad4872)
g2.sp <- as(quad4872,"SpatialPolygons")
g2.sf <- st_as_sf(g2.sp)
g2.sf$count <- g2


# Set Parameter Values
theta <- list("beta0"=3,"beta1"=0.75,"beta2"=-0.5,"sigma"=1,"rho"=2)
i.im <- intb.im; c1.im <- popb.im; c2.im <- incb.im

# Generate fixed mean field
beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]]; sigma <- theta[[4]]; rho <- theta[[5]];
mu <- beta.0*i.im + beta.1*c1.im + beta.2*c2.im


# Label resolutions
h1.sf$res <- rep("2km-by-2km",nrow(h1.sf))
h2.sf$res <- rep("1km-by-1km",nrow(h2.sf))
g1.sf$res <- rep("2km-by-2km",nrow(g1.sf))
g2.sf$res <- rep("1km-by-1km",nrow(g2.sf))

# Combine into one for plotting
lahom.all <- rbind(h1.sf,h2.sf)
lagta.all <- rbind(g1.sf,g2.sf)

# Plot different resolutions for aggregated crimes, separately
la.hom.2konly <- tm_shape(h1.sf) + tm_fill("count",size=0.5,palette="viridis") + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title="LA Homicides Aggregated\n on a 2km by 2km Grid",main.title.position = c("center","top"))
la.hom.1konly <- tm_shape(h2.sf) + tm_fill("count",size=0.5,palette="viridis") + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title="LA Homicides Aggregated\n on a 1km by 1km Grid",main.title.position = c("center","top"))


tmap_save(la.hom.2konly,filename = "LAHom2436IrregPolSimStudy.pdf")
tmap_save(la.hom.1konly,filename = "LAHom4872IrregPolSimStudy.pdf")

la.gta.2konly <- tm_shape(g1.sf) + tm_fill("count",size=0.5,palette="viridis") + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title="LA Motor Vehicle Thefts\n Aggregated on a 2km by 2km Grid",main.title.position = c("center","top"))
la.gta.1konly <- tm_shape(g2.sf) + tm_fill("count",size=0.5,palette="viridis") + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title="LA Motor Vehicle Thefts\n Aggregated on a 1km by 1km Grid",main.title.position = c("center","top"))

tmap_save(la.gta.2konly,filename = "LAGTA2436IrregPolSimStudy.pdf")
tmap_save(la.gta.1konly,filename = "LAGTA4872IrregPolSimStudy.pdf")

# Plot both resolutions but each crime separately
la.hom.both <- tm_shape(lahom.all) + tm_fill("count",size=0.2,palette="viridis") + tm_facets(by="res") + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title="LA Homicides Aggregated at Different Resolutions",main.title.position = c("center","top"))

la.gta.both <- tm_shape(lagta.all) + tm_fill("count",size=0.2,palette="viridis") + tm_facets(by="res") + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title="LA Motor Vehicle Thefts Aggregated at Different Resolutions",main.title.position = c("center","top"))

tmap_save(la.hom.both,filename = "LAHom2436and4872IrregPolSimStudy.pdf")
tmap_save(la.gta.both,filename = "LAGTA2436and4872IrregPolSimStudy.pdf")


# Simulated Data Sets -----------------------------------------------------

i <- 1
set.seed(125*i+1)
lgcp.ppp <- try(rLGCP(model="matern",mu,var=(sigma)^2,scale=rho/2,nu=1,win = W))

# Aggregate the point patterns and plot the different resolutions separately and together
s1_1 <- quadratcount(lgcp.ppp,tess=quad.list$quad2436)
s1_1.sp <- as(quad.list$quad2436,"SpatialPolygons")
s1_1.sf <- st_as_sf(s1_1.sp)
s1_1.sf$count <- s1_1
s1_1.sf$res <- "2km-by-2km"
s1_1.sf$sim <- "Simulated Data Set 1"

t1_1 <- tm_shape(s1_1.sf) + tm_fill("count",palette = "viridis") + tm_layout(main.title=paste0("Simulated Data Set 1\n on 2km-by-2km Grid"),main.title.position = c("center","top"))

s2_1 <- quadratcount(lgcp.ppp,tess=quad.list$quad4872)
s2_1.sp <- as(quad.list$quad4872,"SpatialPolygons")
s2_1.sf <- st_as_sf(s2_1.sp)
s2_1.sf$count <- s2_1
s2_1.sf$res <- "1km-by-1km"
s2_1.sf$sim <- "Simulated Data Set 1"

t2_1 <- tm_shape(s2_1.sf) + tm_fill("count",palette = "viridis") + tm_layout(main.title=paste0("Simulated Data Set 1\n on 1km-by-1km Grid"),main.title.position = c("center","top"))

s_1.sf <- rbind(s1_1.sf,s2_1.sf)
t_1 <- tm_shape(s_1.sf) + tm_fill("count",palette = "viridis") + tm_facets(by="res") + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title=paste0("Simulated Data Set 1"),main.title.position = c("center","top"))

tmap_save(t1_1,filename = paste0("IrregPolSimStudySimulatedDataSet",i,"2436.pdf"))
tmap_save(t2_1,filename = paste0("IrregPolSimStudySimulatedDataSet",i,"4872.pdf"))
tmap_save(t_1,filename = paste0("IrregPolSimStudySimulatedDataSet",i,"All.pdf"))

# 
i <- 2
set.seed(125*i+1)
lgcp.ppp <- try(rLGCP(model="matern",mu,var=(sigma)^2,scale=rho/2,nu=1,win = W))

# Aggregate the point patterns and plot the different resolutions separately and together
s1_2 <- quadratcount(lgcp.ppp,tess=quad.list$quad2436)
s1_2.sp <- as(quad.list$quad2436,"SpatialPolygons")
s1_2.sf <- st_as_sf(s1_2.sp)
s1_2.sf$count <- s1_2
s1_2.sf$res <- "2km-by-2km"
s1_2.sf$sim <- "Simulated Data Set 2"

t1_2 <- tm_shape(s1_2.sf) + tm_fill("count",palette = "viridis") + tm_layout(main.title=paste0("Simulated Data Set 2\n on 2km-by-2km Grid"),main.title.position = c("center","top"))

s2_2 <- quadratcount(lgcp.ppp,tess=quad.list$quad4872)
s2_2.sp <- as(quad.list$quad4872,"SpatialPolygons")
s2_2.sf <- st_as_sf(s2_2.sp)
s2_2.sf$count <- s2_2
s2_2.sf$count <- s2_2
s2_2.sf$res <- "1km-by-1km"
s2_2.sf$sim <- "Simulated Data Set 2"

t2_2 <- tm_shape(s2_2.sf) + tm_fill("count",palette = "viridis") + tm_layout(main.title=paste0("Simulated Data Set 2\n on 1km-by-1km Grid"),main.title.position = c("center","top"))

s_2.sf <- rbind(s1_2.sf,s2_2.sf)
t_2 <- tm_shape(s_2.sf) + tm_fill("count",palette = "viridis") + tm_facets(by="res") + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title=paste0("Simulated Data Set 2"),main.title.position = c("center","top"))


tmap_save(t1_2,filename = paste0("IrregPolSimStudySimulatedDataSet",i,"2436.pdf"))
tmap_save(t2_2,filename = paste0("IrregPolSimStudySimulatedDataSet",i,"4872.pdf"))
tmap_save(t_2,filename = paste0("IrregPolSimStudySimulatedDataSet",i,"All.pdf"))

i <- 3
set.seed(125*i+1)
lgcp.ppp <- try(rLGCP(model="matern",mu,var=(sigma)^2,scale=rho/2,nu=1,win = W))

# Aggregate the point patterns and plot the different resolutions separately and together
s1_3 <- quadratcount(lgcp.ppp,tess=quad.list$quad2436)
s1_3.sp <- as(quad.list$quad2436,"SpatialPolygons")
s1_3.sf <- st_as_sf(s1_3.sp)
s1_3.sf$count <- s1_3
s1_3.sf$res <- "2km-by-2km"
s1_3.sf$sim <- "Simulated Data Set 3"

t1_3 <- tm_shape(s1_3.sf) + tm_fill("count",palette = "viridis") + tm_layout(main.title=paste0("Simulated Data Set 3\n on 2km-by-2km Grid"),main.title.position = c("center","top"))

s2_3 <- quadratcount(lgcp.ppp,tess=quad.list$quad4872)
s2_3.sp <- as(quad.list$quad4872,"SpatialPolygons")
s2_3.sf <- st_as_sf(s2_3.sp)
s2_3.sf$count <- s2_3
s2_3.sf$res <- "1km-by-1km"
s2_3.sf$sim <- "Simulated Data Set 3"

t2_3 <- tm_shape(s2_3.sf) + tm_fill("count",palette = "viridis") + tm_layout(main.title=paste0("Simulated Data Set 3\n on 1km-by-1km Grid"),main.title.position = c("center","top"))

s_3.sf <- rbind(s1_3.sf,s2_3.sf)
t_3 <- tm_shape(s_3.sf) + tm_fill("count",palette = "viridis") + tm_facets(by="res") + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title=paste0("Simulated Data Set 3"),main.title.position = c("center","top"))


tmap_save(t1_3,filename = paste0("IrregPolSimStudySimulatedDataSet",i,"2436.pdf"))
tmap_save(t2_3,filename = paste0("IrregPolSimStudySimulatedDataSet",i,"4872.pdf"))
tmap_save(t_3,filename = paste0("IrregPolSimStudySimulatedDataSet",i,"All.pdf"))

# Now to plot the true data along with the simulated data

# "Simulation" labels for facetting
h1.sf$sim <- "True Homicide"
h2.sf$sim <- "True Homicide"
g1.sf$sim <- "True MVT"
g2.sf$sim <- "True MVT"

# 2km Grid for true data and first two simulated data sets
combined.1k.sf <- rbind(h1.sf,g1.sf,s1_1.sf,s1_2.sf)
lev.sim <- unique(combined.1k.sf$sim)
combined.1k.sf$sim <- factor(combined.1k.sf$sim,levels = lev.sim)
hgt1 <- tm_shape(combined.1k.sf) + tm_fill("count",n=25,palette="viridis") + tm_facets(by="sim",ncol=2) + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title=paste0("True and Simulated Data Sets on 2km-by-2km Grid"),main.title.position = c("center","top"))
pb.2k <- c(0:6,11,21,31,41,51,76,101,126,151,201,251,300)
hgt1.pb <- tm_shape(combined.1k.sf) + tm_fill("count",n=25,palette="viridis",breaks=pb.2k) + tm_facets(by="sim",ncol=2) + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title=paste0("True and Simulated Data Sets on 2km-by-2km Grid"),main.title.position = c("center","top"))

# 2km Grid for true homicide data and two simulated data sets
combined.1k.sf.hom <- rbind(h1.sf,s1_1.sf,s1_2.sf,s1_3.sf)
lev.sim <- unique(combined.1k.sf.hom$sim)
combined.1k.sf.hom$sim <- factor(combined.1k.sf.hom$sim,levels = lev.sim)
ht1 <- tm_shape(combined.1k.sf.hom) + tm_fill("count",n=10,palette="viridis") + tm_facets(by="sim",ncol=2) + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title=paste0("True and Simulated Data Sets on 2km-by-2km Grid"),main.title.position = c("center","top"))


# 1km Grid for true data and first two simulated data sets
combined.2k.sf <- rbind(h2.sf,g2.sf,s2_1.sf,s2_2.sf)
lev.sim <- unique(combined.2k.sf$sim)
combined.2k.sf$sim <- factor(combined.2k.sf$sim,levels = lev.sim)
hgt2 <- tm_shape(combined.2k.sf) + tm_fill("count",n=25,palette="viridis") + tm_facets(by="sim",ncol=2) + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title=paste0("True and Simulated Data Sets on 1km-by-1km Grid"),main.title.position = c("center","top"))
pb.1k <- c(0:11,21,31,41,51,76,101,125)
hgt2.pb <- tm_shape(combined.2k.sf) + tm_fill("count",n=25,palette="viridis",breaks=pb.1k) + tm_facets(by="sim",ncol=2) + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title=paste0("True and Simulated Data Sets on 1km-by-1km Grid"),main.title.position = c("center","top"))

# 1km Grid for true homicide data and two simulated data sets
combined.2k.sf.hom <- rbind(h2.sf,s2_1.sf,s2_2.sf,s2_3.sf)
lev.sim <- unique(combined.2k.sf.hom$sim)
combined.2k.sf.hom$sim <- factor(combined.2k.sf.hom$sim,levels = lev.sim)
ht2 <- tm_shape(combined.2k.sf.hom) + tm_fill("count",n=10,palette="viridis") + tm_facets(by="sim",ncol=2) + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title=paste0("True and Simulated Data Sets on 1km-by-1km Grid"),main.title.position = c("center","top"))

# Save
tmap_save(hgt1,filename="IrregPolSimStudyTrueandSimulated2436.pdf")
tmap_save(hgt1.pb,filename="IrregPolSimStudyTrueandSimulated2436Breaks.pdf")
tmap_save(ht1,filename="IrregPolSimStudyTrueHomandSimulated2436.pdf")
tmap_save(hgt2,filename="IrregPolSimStudyTrueandSimulated4872.pdf")
tmap_save(hgt2.pb,filename="IrregPolSimStudyTrueandSimulated4872Breaks.pdf")
tmap_save(ht2,filename="IrregPolSimStudyTrueHomandSimulated4872.pdf")


# Plot Covariates , Grids and Gridded Covariates --------------------------

cov.name <- "GridMeshIrregPolLGCPSSCov.rda"
load(cov.name)
quad.name <- "QuadratsIrregPolLGCP.rda"
load(quad.name)
covgrid.name <- "CovAggGridIrregPolLGCP.rda"
load(covgrid.name)

# Grids
pdf("QuadratsIrregPolLGCP.pdf",h=15,w=15,pointsize=14)
for (i in 1:length(quad.list)){
  plot(quad.list[[i]],main="")
}
dev.off()

# Covariates (as rasters)
pdf("IrregPolLGCPCovariates.pdf",h=15,w=15,pointsize=14)
plot(pop.ras,main="")
plot(inc.ras,main="")
dev.off()

# Meshes
mesh.name <- "MeshesIrregPolLGCP.rda"
load(mesh.name)
pdf("MeshesIrregPolLGCP.pdf",h=15,w=15,pointsize=14)
for (i in 1:4){
  plot(mesh.list[[i]],main="")
}
dev.off()

agg.names <- c("5km-by5km","2km-by-2km","1km-by-1km","0.5km-by-0.5km")
grid.names <- c("1015","2436","4872","95144")
for (i in 1:4){
  g <- quad.list[[i]]
  g.sp <- as(g,"SpatialPolygons")
  g.sp$cov1 <- as.numeric(cov1grid.list[[i]])
  g.sp$cov2 <- as.numeric(cov2grid.list[[i]])
  g.sf <- st_as_sf(g.sp)
  
  c1.map <- tm_shape(g.sf) + tm_fill("cov1",palette="viridis",title="Covariate 1") + tm_layout(main.title=paste0("Covariate 1 Aggregated\n on a ",agg.names[i]," Grid"),main.title.position = c("center","top"))
  c2.map <- tm_shape(g.sf) + tm_fill("cov2",palette="viridis",title="Covariate 2") + tm_layout(main.title=paste0("Covariate 2 Aggregated\n on a ",agg.names[i]," Grid"),main.title.position = c("center","top"))
  
  tmap_save(c1.map,filename=paste0("IrregPolSimStudyCov1Grid",grid.names[i],".pdf"))
  tmap_save(c2.map,filename=paste0("IrregPolSimStudyCov2Grid",grid.names[i],".pdf"))
}

# All 4 resolutions for one covariate together
i <- 1
g <- quad.list[[i]]
g.sp <- as(g,"SpatialPolygons")
g.sp$cov1 <- as.numeric(cov1grid.list[[i]])
g.sp$cov2 <- as.numeric(cov2grid.list[[i]])
g1.sf <- st_as_sf(g.sp)
g1.sf$res <- agg.names[i]

i <- 2
g <- quad.list[[i]]
g.sp <- as(g,"SpatialPolygons")
g.sp$cov1 <- as.numeric(cov1grid.list[[i]])
g.sp$cov2 <- as.numeric(cov2grid.list[[i]])
g2.sf <- st_as_sf(g.sp)
g2.sf$res <- agg.names[i]

i <- 3
g <- quad.list[[i]]
g.sp <- as(g,"SpatialPolygons")
g.sp$cov1 <- as.numeric(cov1grid.list[[i]])
g.sp$cov2 <- as.numeric(cov2grid.list[[i]])
g3.sf <- st_as_sf(g.sp)
g3.sf$res <- agg.names[i]

i <- 4
g <- quad.list[[i]]
g.sp <- as(g,"SpatialPolygons")
g.sp$cov1 <- as.numeric(cov1grid.list[[i]])
g.sp$cov2 <- as.numeric(cov2grid.list[[i]])
g4.sf <- st_as_sf(g.sp)
g4.sf$res <- agg.names[i]

g.allres <- rbind(g1.sf,g2.sf,g3.sf,g4.sf)
g.allres$res <- factor(g.allres$res,level=agg.names)

c1.allres <- tm_shape(g.allres) + tm_fill("cov1",palette="viridis",n=10,title="Covariate 1") + tm_facets(by="res",ncol=2) + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title=paste0("Covariate 1 Aggregated at\n Different Resolutions"),main.title.position = c("center","top"))
c2.allres <- tm_shape(g.allres) + tm_fill("cov2",palette="viridis",n=10,title="Covariate 2") + tm_facets(by="res",ncol=2) + tm_layout(legend.outside.position = "right" , legend.outside.size = .2,asp = 1,main.title=paste0("Covariate 2 Aggregated at\n Different Resolutions"),main.title.position = c("center","top"))

tmap_save(c1.allres,filename = "IrregPolSimStudyCov1AllGrid.pdf")
tmap_save(c2.allres,filename = "IrregPolSimStudyCov2AllGrid.pdf")


# sessionInfo -------------------------------------------------------------

sessionInfo() 