
# Testing Priors for Irregular Polygon (c) --------------------------------

# This R script will test the number of points simulated as well as the proportion of cells with zero counts for varying priors for the fixed effects over the Los Angeles polygon, aiming to match proportions of zeros with the LA data (as much as reasonable) as well as reasonable mean and median n simulated points.
# Note: this file is labelled differently and highlights the outputs with the suffix below because the prior for rho used in these simulations is different!!
# Outputs from this have the suffix *_Finalic.rda with i taking several integer values.

# Author: Nadeen Khaleel


# Setwd and Load Libraries ------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))

library(doParallel)
library(foreach)


# INLA GLM Results --------------------------------------------------------
# Fit GLM with INLA to finest grid for LA homicide and motor vehicle theft, using the GMO data, so that the socio-economic variables match those used to simulate the LGCP.

library(INLA)
# load("../../../../DATA/PROCESSED_DATA/CRIME/COUNT_DATA_GMO/LA/LA2015CT236359CountData_proj.rda")
# hom_countdf$zpop <- (hom_countdf$pop - mean(hom_countdf$pop))/sd(hom_countdf$pop)
# hom_countdf$zinc <- (hom_countdf$inc - mean(hom_countdf$inc))/sd(hom_countdf$inc)
# # fit.glm <- glm(hom ~ 1+ zpop + zinc,data=hom_countdf,family="poisson")
# fit.glm <- inla(hom ~ 1+ zpop + zinc,data=hom_countdf,family="poisson")
# fit.glm$summary.fixed
# # mean         sd 0.025quant   0.5quant 0.975quant       mode          kld
# # (Intercept) -5.0379608 0.06838331 -5.1747257 -5.0370934 -4.9061282 -5.0353703 1.812007e-09
# # zpop         0.7699032 0.04832476  0.6751325  0.7698644  0.8648026  0.7697908 8.726262e-07
# # zinc        -0.4731952 0.08678917 -0.6484224 -0.4714974 -0.3075248 -0.4681270 2.716181e-07
# # 
# gta_countdf$zpop <- (gta_countdf$pop - mean(gta_countdf$pop))/sd(gta_countdf$pop)
# gta_countdf$zinc <- (gta_countdf$inc - mean(gta_countdf$inc))/sd(gta_countdf$inc)
# # fit.glm <- glm(gta ~ 1+ zpop + zinc,data=gta_countdf,family="poisson")
# fit.glm <- inla(gta ~ 1+ zpop + zinc,data=gta_countdf,family="poisson")
# fit.glm$summary.fixed
# fit.glm$summary.fixed
# # mean          sd 0.025quant   0.5quant  0.975quant       mode          kld
# # (Intercept) -0.9097305 0.008753028 -0.9269574 -0.9097164 -0.89259854 -0.9096876 1.023746e-06
# # zpop         0.5241291 0.006287413  0.5117815  0.5241301  0.53645944  0.5241327 4.545158e-07
# # zinc        -0.1019944 0.009264106 -0.1202739 -0.1019636 -0.08390399 -0.1019013 8.980831e-07


# Including area as an offset
# fit.glm <- inla(hom ~ 1 + offset(log(area)) + zpop + zinc,data=hom_countdf,family="poisson")
# fit.glm$summary.fixed
# mean         sd  0.025quant    0.5quant  0.975quant        mode          kld
# (Intercept) -15.5969175 0.06847258 -15.7338558 -15.5960509 -15.4649078 -15.5943291 2.024769e-09
# zpop          0.7724722 0.04857581   0.6772023   0.7724356   0.8678585   0.7723663 8.535231e-07
# zinc         -0.4956648 0.08807677  -0.6734560  -0.4939548  -0.3274998  -0.4905606 2.605868e-07
# 
# 
# fit.glm <- inla(hom ~ 1 + offset(log(area/1e8)) + zpop + zinc,data=hom_countdf,family="poisson")
# fit.glm$summary.fixed
# mean         sd 0.025quant   0.5quant 0.975quant       mode
# (Intercept)  2.8237737 0.06847232  2.6868359  2.8246403  2.9557829  2.8263620
# zpop         0.7724685 0.04857571  0.6771986  0.7724319  0.8678544  0.7723627
# zinc        -0.4956603 0.08807651 -0.6734506 -0.4939504 -0.3274954 -0.4905564
# kld
# (Intercept) 2.029426e-09
# zpop        8.531061e-07

# fit.glm <- inla(gta ~ 1 + offset(log(area)) + zpop + zinc,data=gta_countdf,family="poisson")
# fit.glm$summary.fixed
# mean          sd  0.025quant    0.5quant   0.975quant        mode          kld
# (Intercept) -11.4693401 0.008773817 -11.4866079 -11.4693260 -11.45216745 -11.4692972 1.023547e-06
# zpop          0.5235963 0.006326989   0.5111706   0.5235975   0.53600393   0.5236003 4.790313e-07
# zinc         -0.1165730 0.009404844  -0.1351294  -0.1165419  -0.09820709  -0.1164792 9.015960e-07
# 
# fit.glm <- inla(gta ~ 1 + offset(log(area/1e8)) + zpop + zinc,data=gta_countdf,family="poisson")
# fit.glm$summary.fixed
# mean          sd 0.025quant   0.5quant  0.975quant       mode
# (Intercept)  6.9513408 0.008773817  6.9340730  6.9513549  6.96851344  6.9513837
# zpop         0.5235963 0.006326989  0.5111706  0.5235975  0.53600391  0.5236003
# zinc        -0.1165730 0.009404844 -0.1351294 -0.1165419 -0.09820709 -0.1164792
# kld
# (Intercept) 1.023662e-06
# zpop        4.790306e-07
# zinc        9.015955e-07



# Simulation Set-Up -------------------------------------------------------
# Select parameter values for the priors, with the index of the outputs also attached. Then set-up the combinations and parallelsation.
# The label out i matches the i in the output filename

# mean.beta0 <- c(-0.5,0.5); mean.beta1 <- c(1.5); mean.beta2 <- c(-1) # out 1c
# sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1),sqrt(0.25)); sd.beta2 <- c(sqrt(0.1),sqrt(0.25))
# sigma.0 <- c(1,2); alpha.sigma <- c(0.1) # out 1c
# mean.beta0 <- c(0.5); mean.beta1 <- c(-2); mean.beta2 <- c(0.5) # out 2c
# sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(1)); sd.beta2 <- c(sqrt(0.5))
# sigma.0 <- c(0.5,0.75,1,2); alpha.sigma <- c(0.1) # out 2c
# mean.beta0 <- c(1); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 3c
# sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1),sqrt(0.25)); sd.beta2 <- c(sqrt(0.1),sqrt(0.25))
# sigma.0 <- c(0.75,1,2); alpha.sigma <- c(0.1) # out 3c
# mean.beta0 <- c(1); mean.beta1 <- c(1.5); mean.beta2 <- c(-1) # out 4c
# sd.beta0 <- c(sqrt(0.5)); sd.beta1 <- c(sqrt(0.1),sqrt(0.25)); sd.beta2 <- c(sqrt(0.1),sqrt(0.25))
# sigma.0 <- c(1,2,3); alpha.sigma <- c(0.1) # out 4c
# mean.beta0 <- c(1.5); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 5c
# sd.beta0 <- c(sqrt(0.25),sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
# sigma.0 <- c(0.75,1,2); alpha.sigma <- c(0.1) # out 5c
# mean.beta0 <- c(1); mean.beta1 <- c(-2); mean.beta2 <- c(0.5) # out 6c
# sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(1)); sd.beta2 <- c(sqrt(0.5))
# sigma.0 <- c(0.75,1,1.5); alpha.sigma <- c(0.01,0.1,0.25) # out 6c
# mean.beta0 <- c(-5); mean.beta1 <- c(0.5); mean.beta2 <- c(-0.5) # out 7c
# sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
# sigma.0 <- c(6); alpha.sigma <- c(0.01,0.1) # out 7c
# mean.beta0 <- c(-1); mean.beta1 <- c(0.5); mean.beta2 <- c(-0.5) # out 8c
# sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
# sigma.0 <- c(6); alpha.sigma <- c(0.01,0.1) # out 8c
# mean.beta0 <- c(-1); mean.beta1 <- c(1); mean.beta2 <- c(-0.5) # out 9c
# sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
# sigma.0 <- c(6); alpha.sigma <- c(0.01,0.1) # out 9c
# mean.beta0 <- c(-0.5,-0.25); mean.beta1 <- c(1); mean.beta2 <- c(-0.5,-0.25) # out 10c
# sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
# # sigma.0 <- c(6); alpha.sigma <- c(0.1) # out 10c
# mean.beta0 <- c(-1); mean.beta1 <- c(1); mean.beta2 <- c(-0.1) # out 11c
# sd.beta0 <- c(sqrt(0.1),sqrt(0.25)); sd.beta1 <- c(sqrt(0.1),sqrt(0.25)); sd.beta2 <- c(sqrt(0.1))
# sigma.0 <- c(6); alpha.sigma <- c(0.1) # out 11c
# mean.beta0 <- c(-1,-0.5); mean.beta1 <- c(1); mean.beta2 <- c(-0.1,-0.5) # out 12c
# sd.beta0 <- c(sqrt(0.25)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.1))
# sigma.0 <- c(4); alpha.sigma <- c(0.1) # out 12c
# mean.beta0 <- c(2.5); mean.beta1 <- c(0.5); mean.beta2 <- c(-0.5) # out 13c
# sd.beta0 <- c(sqrt(0.25),sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
# sigma.0 <- c(0.75,1,2); alpha.sigma <- c(0.1) # out 13c
# mean.beta0 <- c(2.5); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 14c
# sd.beta0 <- c(sqrt(0.25),sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
# sigma.0 <- c(0.75,1,2); alpha.sigma <- c(0.1) # out 14c
# mean.beta0 <- c(2); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 15c
# sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
# sigma.0 <- c(0.75,1,2); alpha.sigma <- c(0.1) # out 15c
# mean.beta0 <- c(2); mean.beta1 <- c(0.75); mean.beta2 <- c(-0.5) # out 16c
# sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.5)); sd.beta2 <- c(sqrt(0.5))
# sigma.0 <- c(1,2,3); alpha.sigma <- c(0.1) # out 16c
# mean.beta0 <- c(3); mean.beta1 <- c(0.75); mean.beta2 <- c(-0.5) # out 17c
# sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
# sigma.0 <- c(1,2,3); alpha.sigma <- c(0.1) # out 17c
# mean.beta0 <- c(2); mean.beta1 <- c(0.75); mean.beta2 <- c(-0.5) # out 18c
# sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
# sigma.0 <- c(1,2,3); alpha.sigma <- c(0.1) # out 18c
# mean.beta0 <- c(2); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 19c
# sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.5)); sd.beta2 <- c(sqrt(0.5))
# sigma.0 <- c(0.75,1,2); alpha.sigma <- c(0.1) # out 19c
mean.beta0 <- c(3); mean.beta1 <- c(1); mean.beta2 <- c(1) # out 20c
sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
sigma.0 <- c(1,2); alpha.sigma <- c(0.1) # out 20c

# mean.beta0 <- c(1.5); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 5c - test larger numbers
# sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
# sigma.0 <- c(2); alpha.sigma <- c(0.1) # out 5c - test larger numbers

# mean.beta0 <- c(2); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 15c - test larger numbers
# sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
# sigma.0 <- c(2); alpha.sigma <- c(0.1) # out 15c - test larger numbers

# mean.beta0 <- c(2.5); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 14c - test larger numbers
# sd.beta0 <- c(sqrt(0.25)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
# sigma.0 <- c(2); alpha.sigma <- c(0.1) # out 14c - test larger numbers

# mean.beta0 <- c(3); mean.beta1 <- c(0.75); mean.beta2 <- c(-0.5) # out 17c - test larger numbers
# sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
# sigma.0 <- c(2); alpha.sigma <- c(0.1) # out 17c - test larger numbers

# List for parameters which are then turned into the final combinations
param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)

param.combos <- expand.grid(param.list)

# Set-up parallelisation
x <- 1:nrow(param.combos)
Nproc <- 4
cl <- makeCluster(Nproc)
registerDoParallel(cl)
indexes <- split(x, sort(rep_len(1:Nproc, length(x))))


# Implement Simulations ---------------------------------------------------

ptm.start <- proc.time()
foreach(i=1:Nproc) %dopar% {
  
  library(spatstat)
  library(raster)
  library(sf)
  library(sp)
  library(maptools)
  library(rgdal)
  library(lwgeom)
  library(RandomFields)
  
  
  # Set-up ------------------------------------------------------------------
  # Need to set-up the parameters again for the each core - due to the parallelisation, we cannot fork so we have to set- things up again, make sure to match the parameters!
  
  # LA Census Tract
  load("../../../../DATA/PROCESSED_DATA/SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")
  lacity_boundary <- st_union(ct_LA)
  lacity_proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611") # changed as prev. is deprecated, no need for +init=epsg...
  lacity_sp <- as(lacity_proj,"Spatial") #st_transform for crs...

  # Transform the window
  bbox <- lacity_sp@bbox
  lacity_spshift <- elide(lacity_sp,shift=-c(bbox[1,1],bbox[2,1]))
  bbox_shift <- lacity_spshift@bbox
  lacity_spscale <- elide(lacity_spshift,scale=max(bbox_shift)/1e4)
  W <- as.owin.SpatialPolygons((lacity_spscale))

  # Get covariates
  cov.name <- "GridMeshIrregPolLGCPSSCov.rda"
  load(cov.name)
  
  
  # Need to include parameters again - make sure they match! The label out i matches the i in the output filename
  # mean.beta0 <- c(-0.5,0.5); mean.beta1 <- c(1.5); mean.beta2 <- c(-1) # out 1c
  # sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1),sqrt(0.25)); sd.beta2 <- c(sqrt(0.1),sqrt(0.25))
  # sigma.0 <- c(1,2); alpha.sigma <- c(0.1) # out 1c
  # mean.beta0 <- c(0.5); mean.beta1 <- c(-2); mean.beta2 <- c(0.5) # out 2c
  # sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(1)); sd.beta2 <- c(sqrt(0.5))
  # sigma.0 <- c(0.5,0.75,1,2); alpha.sigma <- c(0.1) # out 2c
  # mean.beta0 <- c(1); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 3c
  # sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1),sqrt(0.25)); sd.beta2 <- c(sqrt(0.1),sqrt(0.25))
  # sigma.0 <- c(0.75,1,2); alpha.sigma <- c(0.1) # out 3c
  # mean.beta0 <- c(1); mean.beta1 <- c(1.5); mean.beta2 <- c(-1) # out 4c
  # sd.beta0 <- c(sqrt(0.5)); sd.beta1 <- c(sqrt(0.1),sqrt(0.25)); sd.beta2 <- c(sqrt(0.1),sqrt(0.25))
  # sigma.0 <- c(1,2,3); alpha.sigma <- c(0.1) # out 4c
  # mean.beta0 <- c(1.5); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 5c
  # sd.beta0 <- c(sqrt(0.25),sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
  # sigma.0 <- c(0.75,1,2); alpha.sigma <- c(0.1) # out 5c
  # mean.beta0 <- c(1); mean.beta1 <- c(-2); mean.beta2 <- c(0.5) # out 6c
  # sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(1)); sd.beta2 <- c(sqrt(0.5))
  # sigma.0 <- c(0.75,1,1.5); alpha.sigma <- c(0.01,0.1,0.25) # out 6c
  # mean.beta0 <- c(-5); mean.beta1 <- c(0.5); mean.beta2 <- c(-0.5) # out 7c
  # sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
  # sigma.0 <- c(6); alpha.sigma <- c(0.01,0.1) # out 7c
  # mean.beta0 <- c(-1); mean.beta1 <- c(0.5); mean.beta2 <- c(-0.5) # out 8c
  # sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
  # sigma.0 <- c(6); alpha.sigma <- c(0.01,0.1) # out 8c
  # mean.beta0 <- c(-1); mean.beta1 <- c(1); mean.beta2 <- c(-0.5) # out 9c
  # sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
  # sigma.0 <- c(6); alpha.sigma <- c(0.01,0.1) # out 9c
  # mean.beta0 <- c(-0.5,-0.25); mean.beta1 <- c(1); mean.beta2 <- c(-0.5,-0.25) # out 10c
  # sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
  # sigma.0 <- c(6); alpha.sigma <- c(0.1) # out 10c
  # mean.beta0 <- c(-1); mean.beta1 <- c(1); mean.beta2 <- c(-0.1) # out 11c
  # sd.beta0 <- c(sqrt(0.1),sqrt(0.25)); sd.beta1 <- c(sqrt(0.1),sqrt(0.25)); sd.beta2 <- c(sqrt(0.1))
  # sigma.0 <- c(6); alpha.sigma <- c(0.1) # out 11c
  # mean.beta0 <- c(-1,-0.5); mean.beta1 <- c(1); mean.beta2 <- c(-0.1,-0.5) # out 12c
  # sd.beta0 <- c(sqrt(0.25)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.1))
  # sigma.0 <- c(4); alpha.sigma <- c(0.1) # out 12c
  # mean.beta0 <- c(2.5); mean.beta1 <- c(0.5); mean.beta2 <- c(-0.5) # out 13c
  # sd.beta0 <- c(sqrt(0.25),sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
  # sigma.0 <- c(0.75,1,2); alpha.sigma <- c(0.1) # out 13c
  # mean.beta0 <- c(2.5); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 14c
  # sd.beta0 <- c(sqrt(0.25),sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
  # sigma.0 <- c(0.75,1,2); alpha.sigma <- c(0.1) # out 14c
  # mean.beta0 <- c(2); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 15c
  # sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
  # sigma.0 <- c(0.75,1,2); alpha.sigma <- c(0.1) # out 15c
  # mean.beta0 <- c(2); mean.beta1 <- c(0.75); mean.beta2 <- c(-0.5) # out 16c
  # sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.5)); sd.beta2 <- c(sqrt(0.5))
  # sigma.0 <- c(1,2,3); alpha.sigma <- c(0.1) # out 16c
  # mean.beta0 <- c(3); mean.beta1 <- c(0.75); mean.beta2 <- c(-0.5) # out 17c
  # sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
  # sigma.0 <- c(1,2,3); alpha.sigma <- c(0.1) # out 17c
  # mean.beta0 <- c(2); mean.beta1 <- c(0.75); mean.beta2 <- c(-0.5) # out 18c
  # sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
  # sigma.0 <- c(1,2,3); alpha.sigma <- c(0.1) # out 18c
  # mean.beta0 <- c(2); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 19c
  # sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.5)); sd.beta2 <- c(sqrt(0.5))
  # sigma.0 <- c(0.75,1,2); alpha.sigma <- c(0.1) # out 19c
  mean.beta0 <- c(3); mean.beta1 <- c(1); mean.beta2 <- c(1) # out 20c
  sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
  sigma.0 <- c(1,2); alpha.sigma <- c(0.1) # out 20c
  
  # mean.beta0 <- c(1.5); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 5c - test larger numbers
  # sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
  # sigma.0 <- c(2); alpha.sigma <- c(0.1) # out 5c - test larger numbers
  
  # mean.beta0 <- c(2); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 15c - test larger numbers
  # sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
  # sigma.0 <- c(2); alpha.sigma <- c(0.1) # out 15c - test larger numbers
  
  # mean.beta0 <- c(2.5); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 14c - test larger numbers
  # sd.beta0 <- c(sqrt(0.25)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
  # sigma.0 <- c(2); alpha.sigma <- c(0.1) # out 14c - test larger numbers
  
  # mean.beta0 <- c(3); mean.beta1 <- c(0.75); mean.beta2 <- c(-0.5) # out 17c - test larger numbers
  # sd.beta0 <- c(sqrt(1)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
  # sigma.0 <- c(2); alpha.sigma <- c(0.1) # out 17c - test larger numbers

  param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)
  
  param.combos <- expand.grid(param.list)
  
  # Load grids to check proportions of zero
  load("QuadratsIrregPolLGCP.rda")
  g <- quad.list[2:3] # 24x36 and 48x72 grid

  N <- 50 # number of simulations for standard information seeking test
  # N <- 1e2 # final long test run with additional "*_Long_ic.rda" in suffix
  # N <- 5e2 # final final long test run, with additional suffix "*_Long_ic.rda" AND "2" or "Take 2" included in the filename
  
  # Change output filename as needed, don't forget to change accordingly in the rest of the code!
  out.filename <- "LACovFixedEffectsPrior_Final20c.rda"
  if ((!file.exists(out.filename)) & (i==1)){
    out <- vector(mode = "list", length = nrow(param.combos))
    save(out,file=out.filename)
  }
  
  # Functions ---------------------------------------------------------------
  
  # Simulate priors from hyper-parameters provided
  prior.sim <- function(n=1,mu.beta0,sigma.beta0,mu.beta1,sigma.beta1,mu.beta2,sigma.beta2,sigma.0,alpha.sigma){
    # n is the number of samples of the parameters you want sampled from their priors.  
    # mu.*: prior mean for fixed effects
    # sigma.beta*: standard deviation for fixed effects
    # sigma.0, alpha.sigma: prior parameters for sigma pc prior
    
    beta.0.sim <- rnorm(n,mu.beta0,sigma.beta0); beta.1.sim <- rnorm(n,mu.beta1,sigma.beta1); beta.2.sim <- rnorm(n,mu.beta2,sigma.beta2)
    
    # For the covariance parameters of the latent Gaussian field, I have a joint prior on the marginal standard deviation and the range, however, in order to simulate from this I need the inverse CDF to generate this from a Uniform RV. However, as this is a joint distribution I cannot quite get the correct arrangement for the inverse. Therefore, I will look at the the prior on kappa (and turn this into a pdf for rho) and then look at the conditional distribution of tau given kappa (and turn this into a pdf for sigma).
    # (https://www.tandfonline.com/doi/pdf/10.1080/01621459.2017.1415907?needAccess=true) 
    # (https://arxiv.org/pdf/1503.00256.pdf)
    
    alpha.rho <- 0.01; rho.0 <- 0.35; # ***c suffix for this rho prior***
    u1 <- runif(n,0,1); u2 <- runif(n,0,1)
    rho.sim <- rho.0*log(alpha.rho)/log(u1)
    sigma.sim <- sigma.0*log(1-u2)/log(alpha.sigma)
    return(list(beta0.tilde=beta.0.sim,beta1.tilde=beta.1.sim,beta2.tilde=beta.2.sim,sigma.tilde=sigma.sim,rho.tilde=rho.sim))
  }
  
  # Simulate the LGCPs and output data frame for the number of points simulated as well as the proportions of zeros etc.
  res.gen <- function(N,W,prior.param,g.tess,c1.im,c2.im,i.im,seed){
    # N is the number of LGCP simulations we want
    # W is the window
    # prior.param is the set of hyper-parameters for the priors ro simulate the parameter for the generation of the LGCP
    # g.tess are the list of quadrats for the discretisations to test the proportion of zeros
    # i.im, c*.im are the pixel images for the intercept and covariates over the study region
    # seed is to set the simulation seed
    
    set.seed(seed)
    # simulated parameter values
    thetas <- prior.sim(n=N,mu.beta0 = prior.param$mu.beta0,sigma.beta0 = prior.param$sd.beta0,mu.beta1 = prior.param$mu.beta1,sigma.beta1 = prior.param$sd.beta1,mu.beta2 = prior.param$mu.beta2,sigma.beta2 = prior.param$sd.beta2,sigma.0=prior.param$sigma.0,alpha.sigma = prior.param$alpha.sigma)
    
    # Create output storage for lgcp.ppp$n and proportion of zeros for different grids.
    grid.names <- names(g.tess)
    output.df <- data.frame(matrix(rep(NA,(length(grid.names)+3)*N),nrow=N))
    colnames(output.df) <- c("n","max.lambda","med.lambda",grid.names)
    
    # Simulated N LGCPs
    for (j in 1:N){
      print(paste0("Simulation ", j))
      # Extract parameter values simulated from priors
      theta <- lapply(1:length(names(thetas)),function(k,j){thetas[[k]][j]},j)
      names(theta) <- names(thetas)
      beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]]; sigma <- theta[[4]]; rho <- theta[[5]];
      
      mu <- beta.0*i.im + beta.1*c1.im + beta.2*c2.im
      # Produce the point pattern from LGCP
      lgcp.ppp <- try(rLGCP(model="matern",mu,var=(sigma)^2,scale=rho/2,nu=1,win = W,saveLambda = TRUE)) 
      
      if (class(lgcp.ppp)!="try-error"){
        output.df$n[j] <- lgcp.ppp$n
        att.ppp <- attributes(lgcp.ppp)
        output.df$max.lambda[j] <- max(att.ppp$Lambda)
        output.df$med.lambda[j] <- median(att.ppp$Lambda)
        
        for (i in 1:length(g.tess)){
          col.df <- grid.names[i] # for the grid proportion of zeros in the output.df need to account for the "n" column
          q <- quadratcount(lgcp.ppp,tess=g.tess[[i]])
          count.df <- data.frame(count=as.vector(unname(q)))
          
          output.df[j,col.df] <- sum(count.df$count==0)/length(count.df$count)
        }
      }
    }
    return(list("results"=output.df))
  }
  
  
  # Run All -----------------------------------------------------------------
  # Run the simulations using the parameter set-up and the functions above
  
  seeds <- 5*(1:nrow(param.combos))
  
  # Label output, change file name accordingly, with additional suffixes for long (N=100) runs and longer (N=500) runs.
  out.filename <- "LACovFixedEffectsPrior_Final20c.rda"
  load(out.filename)
  
  x <- which(unlist(sapply(1:length(out),function(i){(is.null(out[[i]]))}))) # for re-setting
  Nproc <- 4
  indexes <- split(x, sort(rep_len(1:Nproc, length(x))))
  for (k in indexes[[i]]){
    print(k)
    out.new <- res.gen(N,W,param.combos[k,],g.tess=g,popb.im,incb.im,intb.im,seeds[k])
    load(out.filename)
    out[[k]] <- out.new
    save(out,file=out.filename)
  }
}

ptm.end <- proc.time()
stopCluster(cl)
ptm.end-ptm.start

# Change file name accordingly
out.filename <- "LACovFixedEffectsPrior_Final20c.rda"

# Get the file and label the outputs so that we can more easily tell which parameters resulted in which results
load(out.filename)
combo.names <- sapply(1:nrow(param.combos),function(i){paste0("(",paste0(param.combos[i,],collapse=","),")")})
names(out) <- combo.names

out

# Save labelled output
save(out,file=out.filename)



# Check Output ------------------------------------------------------------
# Load outputs and generate tables, these can then be placed in a .tex file for easy reading.
# Note that the logner run outputs are not included here, but table output was copied directly into a .tex file.

# 
# load("LACovFixedEffectsPrior_Final1c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Wed Apr 21 10:39:05 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & -0.5 & 0.32 & 1.5 & 0.32 &  -1 & 0.32 &   1 & 0.1 & $7.3 \times 10^{2}$ &  78 &   0 & 0.43 & $1.8 \times 10^{5}$ &   1 & $2.1 \times 10^{5}$ & 0.94 & 0.98 \\ 
# # 2 & 0.5 & 0.32 & 1.5 & 0.32 &  -1 & 0.32 &   1 & 0.1 & $3.2 \times 10^{3}$ & $2.8 \times 10^{2}$ &   0 & 0.43 & $1 \times 10^{6}$ & 1.1 & $1 \times 10^{6}$ & 0.89 & 0.96 \\ 
# # 3 & -0.5 & 0.32 & 1.5 & 0.5 &  -1 & 0.32 &   1 & 0.1 & $1.5 \times 10^{4}$ &  34 &   0 & 0.43 & $5 \times 10^{6}$ & 0.91 & $6.2 \times 10^{6}$ & 0.94 & 0.98 \\ 
# # 4 & 0.5 & 0.32 & 1.5 & 0.5 &  -1 & 0.32 &   1 & 0.1 & $8.5 \times 10^{4}$ & $3.8 \times 10^{2}$ &   0 & 0.43 & $3.2 \times 10^{7}$ & 1.2 & $3 \times 10^{7}$ & 0.88 & 0.95 \\ 
# # 5 & -0.5 & 0.32 & 1.5 & 0.32 &  -1 & 0.5 &   1 & 0.1 & $2.3 \times 10^{3}$ &  42 &   0 & 0.43 & $6.1 \times 10^{5}$ & 0.91 & $7.1 \times 10^{5}$ & 0.95 & 0.98 \\ 
# # 6 & 0.5 & 0.32 & 1.5 & 0.32 &  -1 & 0.5 &   1 & 0.1 & $2.8 \times 10^{4}$ & $1.2 \times 10^{2}$ &   0 & 0.43 & $8.3 \times 10^{6}$ & 1.2 & $5.6 \times 10^{6}$ & 0.88 & 0.96 \\ 
# # 7 & -0.5 & 0.32 & 1.5 & 0.5 &  -1 & 0.5 &   1 & 0.1 & $5.6 \times 10^{5}$ &  94 &   0 & 0.43 & $2.3 \times 10^{8}$ & 0.92 & $4 \times 10^{8}$ & 0.93 & 0.97 \\ 
# # 8 & 0.5 & 0.32 & 1.5 & 0.5 &  -1 & 0.5 &   1 & 0.1 & $2.9 \times 10^{4}$ & $3 \times 10^{2}$ &   0 & 0.43 & $9.7 \times 10^{6}$ & 1.1 & $9.7 \times 10^{6}$ & 0.88 & 0.95 \\ 
# # 9 & -0.5 & 0.32 & 1.5 & 0.32 &  -1 & 0.32 &   2 & 0.1 & $2.6 \times 10^{4}$ & $1.7 \times 10^{2}$ &   0 & 0.87 & $9.3 \times 10^{6}$ & 1.1 & $1.1 \times 10^{7}$ & 0.92 & 0.97 \\ 
# # 10 & 0.5 & 0.32 & 1.5 & 0.32 &  -1 & 0.32 &   2 & 0.1 & $1.3 \times 10^{3}$ & $1.8 \times 10^{2}$ &   0 & 0.87 & $2.8 \times 10^{5}$ & 1.4 & $2.5 \times 10^{5}$ & 0.88 & 0.95 \\ 
# # 11 & -0.5 & 0.32 & 1.5 & 0.5 &  -1 & 0.32 &   2 & 0.1 & $1.4 \times 10^{5}$ &  90 &   0 & 0.87 & $4.8 \times 10^{7}$ &   1 & $3.5 \times 10^{7}$ & 0.92 & 0.97 \\ 
# # 12 & 0.5 & 0.32 & 1.5 & 0.5 &  -1 & 0.32 &   2 & 0.1 & $8.7 \times 10^{4}$ & $3.9 \times 10^{2}$ &   0 & 0.87 & $3.6 \times 10^{7}$ & 1.4 & $3.6 \times 10^{7}$ & 0.86 & 0.94 \\ 
# # 13 & -0.5 & 0.32 & 1.5 & 0.32 &  -1 & 0.5 &   2 & 0.1 & $9.4 \times 10^{3}$ & $1.3 \times 10^{2}$ &   0 & 0.87 & $2.5 \times 10^{6}$ & 2.4 & $1.9 \times 10^{6}$ & 0.91 & 0.96 \\ 
# # 14 & 0.5 & 0.32 & 1.5 & 0.32 &  -1 & 0.5 &   2 & 0.1 & $1.2 \times 10^{4}$ & $1.1 \times 10^{2}$ &   0 & 0.87 & $4.6 \times 10^{6}$ & 1.1 & $5.1 \times 10^{6}$ & 0.89 & 0.96 \\ 
# # 15 & -0.5 & 0.32 & 1.5 & 0.5 &  -1 & 0.5 &   2 & 0.1 & $2.1 \times 10^{4}$ &  98 &   0 & 0.87 & $7.3 \times 10^{6}$ & 0.89 & $7.6 \times 10^{6}$ & 0.93 & 0.97 \\ 
# # 16 & 0.5 & 0.32 & 1.5 & 0.5 &  -1 & 0.5 &   2 & 0.1 & $1.4 \times 10^{5}$ & $2 \times 10^{2}$ &   0 & 0.87 & $5.4 \times 10^{7}$ & 1.1 & $4.9 \times 10^{7}$ & 0.88 & 0.95 \\ 
# # \hline
# # \end{tabular}
# # \end{table}

# load("LACovFixedEffectsPrior_Final2c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Wed Apr 21 18:34:31 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & 0.5 &   1 &  -2 &   1 & 0.5 & 0.71 & 0.5 & 0.1 & $6.4 \times 10^{2}$ &  74 &   0 & 0.22 & $7.9 \times 10^{4}$ & 1.1 & $9 \times 10^{4}$ & 0.84 & 0.94 \\ 
# # 2 & 0.5 &   1 &  -2 &   1 & 0.5 & 0.71 & 0.75 & 0.1 & $1.6 \times 10^{2}$ &  48 &   0 & 0.33 & $2.3 \times 10^{4}$ &   1 & $2.1 \times 10^{4}$ & 0.9 & 0.96 \\ 
# # 3 & 0.5 &   1 &  -2 &   1 & 0.5 & 0.71 &   1 & 0.1 & $1.2 \times 10^{3}$ &  96 &   0 & 0.43 & $1.7 \times 10^{5}$ & 1.1 & $2.5 \times 10^{5}$ & 0.81 & 0.93 \\ 
# # 4 & 0.5 &   1 &  -2 &   1 & 0.5 & 0.71 &   2 & 0.1 & $2.4 \times 10^{3}$ &  56 &   0 & 0.87 & $9 \times 10^{5}$ & 1.1 & $4.4 \times 10^{5}$ & 0.86 & 0.94 \\ 
# # \hline
# # \end{tabular}
# # \end{table}

# load("LACovFixedEffectsPrior_Final3c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Wed Apr 21 10:43:18 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &   1 & 0.32 &   1 & 0.32 &  -1 & 0.32 & 0.75 & 0.1 & $1.2 \times 10^{2}$ &  60 &   0 & 0.33 & $1.3 \times 10^{4}$ & 1.3 & $1.3 \times 10^{4}$ & 0.87 & 0.96 \\ 
# # 2 &   1 & 0.32 &   1 & 0.5 &  -1 & 0.32 & 0.75 & 0.1 & $1.4 \times 10^{3}$ &  69 &   0 & 0.33 & $3.7 \times 10^{5}$ & 1.2 & $4.1 \times 10^{5}$ & 0.88 & 0.96 \\ 
# # 3 &   1 & 0.32 &   1 & 0.32 &  -1 & 0.5 & 0.75 & 0.1 & $2.4 \times 10^{2}$ &  68 &   0 & 0.33 & $3.4 \times 10^{4}$ & 1.3 & $2.9 \times 10^{4}$ & 0.87 & 0.96 \\ 
# # 4 &   1 & 0.32 &   1 & 0.5 &  -1 & 0.5 & 0.75 & 0.1 & $3.5 \times 10^{3}$ &  78 &   0 & 0.33 & $1.1 \times 10^{6}$ & 1.1 & $9.1 \times 10^{5}$ & 0.87 & 0.96 \\ 
# # 5 &   1 & 0.32 &   1 & 0.32 &  -1 & 0.32 &   1 & 0.1 & $2.3 \times 10^{2}$ &  57 &   0 & 0.43 & $3.2 \times 10^{4}$ & 1.3 & $2.8 \times 10^{4}$ & 0.88 & 0.96 \\ 
# # 6 &   1 & 0.32 &   1 & 0.5 &  -1 & 0.32 &   1 & 0.1 & $7.6 \times 10^{3}$ &  61 &   0 & 0.43 & $1.7 \times 10^{6}$ & 1.3 & $1.8 \times 10^{6}$ & 0.87 & 0.95 \\ 
# # 7 &   1 & 0.32 &   1 & 0.32 &  -1 & 0.5 &   1 & 0.1 & $1.6 \times 10^{3}$ &  73 &   0 & 0.43 & $3.7 \times 10^{5}$ & 1.6 & $3.1 \times 10^{5}$ & 0.85 & 0.95 \\ 
# # 8 &   1 & 0.32 &   1 & 0.5 &  -1 & 0.5 &   1 & 0.1 & $8.6 \times 10^{2}$ &  84 &   0 & 0.43 & $2.1 \times 10^{5}$ & 1.2 & $2.1 \times 10^{5}$ & 0.86 & 0.95 \\ 
# # 9 &   1 & 0.32 &   1 & 0.32 &  -1 & 0.32 &   2 & 0.1 & $1.7 \times 10^{3}$ &  93 &   0 & 0.87 & $4.8 \times 10^{5}$ & 1.7 & $4.3 \times 10^{5}$ & 0.84 & 0.94 \\ 
# # 10 &   1 & 0.32 &   1 & 0.5 &  -1 & 0.32 &   2 & 0.1 & $6.6 \times 10^{2}$ &  74 &   0 & 0.87 & $1.1 \times 10^{5}$ & 3.2 & $6.7 \times 10^{4}$ & 0.84 & 0.93 \\ 
# # 11 &   1 & 0.32 &   1 & 0.32 &  -1 & 0.5 &   2 & 0.1 & $3.1 \times 10^{3}$ &  76 &   0 & 0.87 & $3.5 \times 10^{5}$ & 1.3 & $4.3 \times 10^{5}$ & 0.85 & 0.94 \\ 
# # 12 &   1 & 0.32 &   1 & 0.5 &  -1 & 0.5 &   2 & 0.1 & $6.2 \times 10^{3}$ &  85 &   0 & 0.87 & $1.7 \times 10^{6}$ & 3.3 & $9.7 \times 10^{5}$ & 0.83 & 0.93 \\ 
# # \hline
# # \end{tabular}
# # \end{table}

# load("LACovFixedEffectsPrior_Final4c.rda")
# out <- out[which(lengths(out)!=0)]
# param.combos <- param.combos[which(lengths(out)!=0),]
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Wed Apr 21 15:25:57 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &   1 & 0.71 & 1.5 & 0.32 &  -1 & 0.32 &   1 & 0.1 & $4.1 \times 10^{3}$ & $4.3 \times 10^{2}$ &   0 & 0.43 & $1.4 \times 10^{6}$ & 1.2 & $1.1 \times 10^{6}$ & 0.83 & 0.93 \\ 
# # 2 &   1 & 0.71 & 1.5 & 0.5 &  -1 & 0.32 &   1 & 0.1 & $6.1 \times 10^{4}$ & $4.6 \times 10^{2}$ &   0 & 0.43 & $2.3 \times 10^{7}$ & 1.2 & $2.8 \times 10^{7}$ & 0.85 & 0.94 \\ 
# # 3 &   1 & 0.71 & 1.5 & 0.32 &  -1 & 0.5 &   1 & 0.1 & $1.7 \times 10^{4}$ & $2 \times 10^{2}$ &   0 & 0.43 & $5.1 \times 10^{6}$ & 1.3 & $3.7 \times 10^{6}$ & 0.81 & 0.93 \\ 
# # 4 &   1 & 0.71 & 1.5 & 0.5 &  -1 & 0.5 &   1 & 0.1 & $2.9 \times 10^{5}$ & $8.4 \times 10^{2}$ &   0 & 0.43 & $6.1 \times 10^{7}$ & 1.3 & $4.3 \times 10^{7}$ & 0.83 & 0.93 \\ 
# # 5 &   1 & 0.71 & 1.5 & 0.32 &  -1 & 0.32 &   2 & 0.1 & $3.1 \times 10^{5}$ & $2.3 \times 10^{2}$ &   0 & 0.87 & $1.3 \times 10^{8}$ & 1.2 & $1.3 \times 10^{8}$ & 0.83 & 0.93 \\ 
# # 6 &   1 & 0.71 & 1.5 & 0.5 &  -1 & 0.32 &   2 & 0.1 & $3.4 \times 10^{5}$ & $2.9 \times 10^{2}$ &   0 & 0.87 & $1.1 \times 10^{8}$ & 1.4 & $8.2 \times 10^{7}$ & 0.82 & 0.92 \\ 
# # 7 &   1 & 0.71 & 1.5 & 0.32 &  -1 & 0.5 &   2 & 0.1 & $4.3 \times 10^{4}$ & $1.2 \times 10^{3}$ &   0 & 0.87 & $1.1 \times 10^{7}$ & 9.8 & $3.1 \times 10^{7}$ & 0.74 & 0.86 \\ 
# # 8 &   1 & 0.71 & 1.5 & 0.5 &  -1 & 0.5 &   2 & 0.1 & $4.6 \times 10^{4}$ & $1.1 \times 10^{3}$ &   0 & 0.87 & $1.4 \times 10^{7}$ & 1.2 & $1.5 \times 10^{7}$ & 0.81 & 0.92 \\ 
# # 9 &   1 & 0.71 & 1.5 & 0.32 &  -1 & 0.32 &   3 & 0.1 & $1.2 \times 10^{5}$ & $8.2 \times 10^{2}$ &   0 & 1.3 & $4.5 \times 10^{7}$ & 1.3 & $3.8 \times 10^{7}$ & 0.8 & 0.91 \\ 
# # 10 &   1 & 0.71 & 1.5 & 0.5 &  -1 & 0.32 &   3 & 0.1 & $8.2 \times 10^{4}$ & $3.3 \times 10^{2}$ &   0 & 1.3 & $1.4 \times 10^{7}$ & 1.4 & $4.8 \times 10^{6}$ & 0.82 & 0.93 \\ 
# # 11 &   1 & 0.71 & 1.5 & 0.32 &  -1 & 0.5 &   3 & 0.1 & $3.7 \times 10^{4}$ & $8.1 \times 10^{2}$ &   2 & 1.3 & $1 \times 10^{7}$ & 2.5 & $2.7 \times 10^{6}$ & 0.78 & 0.9 \\ 
# # \hline
# # \end{tabular}
# # \end{table}

# load("LACovFixedEffectsPrior_Final5c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Wed Apr 21 15:27:11 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & 1.5 & 0.5 &   1 & 0.5 &  -1 & 0.5 & 0.75 & 0.1 & $2.2 \times 10^{3}$ & $1.2 \times 10^{2}$ &   0 & 0.33 & $5.6 \times 10^{5}$ & 1.4 & $4.7 \times 10^{5}$ & 0.8 & 0.93 \\ 
# # 2 & 1.5 &   1 &   1 & 0.5 &  -1 & 0.5 & 0.75 & 0.1 & $2.3 \times 10^{3}$ & $1.2 \times 10^{2}$ &   0 & 0.33 & $5.7 \times 10^{5}$ & 1.2 & $5.9 \times 10^{5}$ & 0.82 & 0.93 \\ 
# # 3 & 1.5 & 0.5 &   1 & 0.5 &  -1 & 0.5 &   1 & 0.1 & $4.3 \times 10^{3}$ & $1.4 \times 10^{2}$ &   0 & 0.43 & $1 \times 10^{6}$ & 1.4 & $7.5 \times 10^{5}$ & 0.79 & 0.92 \\ 
# # 4 & 1.5 &   1 &   1 & 0.5 &  -1 & 0.5 &   1 & 0.1 & $1.2 \times 10^{4}$ & $1.2 \times 10^{2}$ &   0 & 0.43 & $3.3 \times 10^{6}$ & 1.2 & $1.9 \times 10^{6}$ & 0.8 & 0.92 \\ 
# # 5 & 1.5 & 0.5 &   1 & 0.5 &  -1 & 0.5 &   2 & 0.1 & $4.2 \times 10^{5}$ & $1 \times 10^{2}$ &   0 & 0.87 & $3 \times 10^{7}$ & 2.7 & $6.4 \times 10^{7}$ & 0.77 & 0.89 \\ 
# # 6 & 1.5 &   1 &   1 & 0.5 &  -1 & 0.5 &   2 & 0.1 & $4.6 \times 10^{4}$ & $1.2 \times 10^{2}$ &   0 & 0.87 & $9.9 \times 10^{6}$ & 1.8 & $7.5 \times 10^{6}$ & 0.78 & 0.9 \\ 
# # \hline
# # \end{tabular}
# # \end{table}

# load("LACovFixedEffectsPrior_Final6c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Wed Apr 21 19:50:04 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\
# # \hline
# # 1 &   1 &   1 &  -2 &   1 & 0.5 & 0.71 & 0.75 & 0.01 & $1.8 \times 10^{3}$ & $1.2 \times 10^{2}$ &   0 & 0.16 & $2.9 \times 10^{5}$ &   1 & $2.6 \times 10^{5}$ & 0.78 & 0.91 \\
# # 2 &   1 &   1 &  -2 &   1 & 0.5 & 0.71 &   1 & 0.01 & $2.4 \times 10^{2}$ &  82 &   0 & 0.22 & $3.2 \times 10^{4}$ & 1.1 & $2.6 \times 10^{4}$ & 0.85 & 0.94 \\
# # 3 &   1 &   1 &  -2 &   1 & 0.5 & 0.71 & 1.5 & 0.01 & $2.2 \times 10^{3}$ & $1.6 \times 10^{2}$ &   0 & 0.33 & $3.6 \times 10^{5}$ & 1.2 & $3.6 \times 10^{5}$ & 0.74 & 0.89 \\
# # 4 &   1 &   1 &  -2 &   1 & 0.5 & 0.71 & 0.75 & 0.1 & $1.8 \times 10^{3}$ &  76 &   0 & 0.33 & $6 \times 10^{5}$ & 1.1 & $5.2 \times 10^{5}$ & 0.82 & 0.93 \\
# # 5 &   1 &   1 &  -2 &   1 & 0.5 & 0.71 &   1 & 0.1 & $6.8 \times 10^{2}$ &  98 &   0 & 0.43 & $1.1 \times 10^{5}$ & 1.5 & $9.2 \times 10^{4}$ & 0.78 & 0.9 \\
# # 6 &   1 &   1 &  -2 &   1 & 0.5 & 0.71 & 1.5 & 0.1 & $3.6 \times 10^{3}$ &  88 &   0 & 0.65 & $1.4 \times 10^{6}$ & 1.7 & $1.4 \times 10^{6}$ & 0.79 & 0.9 \\
# # 7 &   1 &   1 &  -2 &   1 & 0.5 & 0.71 & 0.75 & 0.25 & $2.9 \times 10^{3}$ & $1.9 \times 10^{2}$ &   1 & 0.54 & $8.7 \times 10^{5}$ & 1.4 & $7.9 \times 10^{5}$ & 0.74 & 0.89 \\
# # 8 &   1 &   1 &  -2 &   1 & 0.5 & 0.71 &   1 & 0.25 & $9.5 \times 10^{3}$ & $1.6 \times 10^{2}$ &   1 & 0.72 & $2.8 \times 10^{6}$ & 1.2 & $2.6 \times 10^{6}$ & 0.78 & 0.9 \\
# # 9 &   1 &   1 &  -2 &   1 & 0.5 & 0.71 & 1.5 & 0.25 & $5.9 \times 10^{4}$ & $1.6 \times 10^{2}$ &   0 & 1.1 & $2.3 \times 10^{7}$ & 3.2 & $1.3 \times 10^{7}$ & 0.72 & 0.86 \\
# # \hline
# # \end{tabular}
# # \end{table}

# load("LACovFixedEffectsPrior_Final7c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Fri Apr 23 01:57:11 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &  -5 & 0.32 & 0.5 & 0.32 & -0.5 & 0.32 &   6 & 0.01 &  26 &   0 &   0 & 1.3 & $1.7 \times 10^{4}$ & 0.69 & $8.1 \times 10^{4}$ & 0.99 &   1 \\ 
# # 2 &  -5 & 0.32 & 0.5 & 0.32 & -0.5 & 0.32 &   6 & 0.1 & $8.3 \times 10^{4}$ &   1 &   2 & 2.6 & $5.4 \times 10^{6}$ & 0.53 & $5 \times 10^{6}$ & 0.98 & 0.99 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final8c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Fri Apr 23 01:58:39 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &  -1 & 0.32 & 0.5 & 0.32 & -0.5 & 0.32 &   6 & 0.01 & $1.4 \times 10^{2}$ &  10 &   0 & 1.3 & $1 \times 10^{4}$ & 0.75 & $1.4 \times 10^{4}$ & 0.95 & 0.98 \\ 
# # 2 &  -1 & 0.32 & 0.5 & 0.32 & -0.5 & 0.32 &   6 & 0.1 & $3.3 \times 10^{4}$ &  13 &   3 & 2.6 & $5.6 \times 10^{6}$ & 1.3 & $8.3 \times 10^{9}$ & 0.92 & 0.96 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final9c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Fri Apr 23 01:59:10 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &  -1 & 0.32 &   1 & 0.32 & -0.5 & 0.32 &   6 & 0.01 & $2 \times 10^{2}$ &  18 &   0 & 1.3 & $4.9 \times 10^{4}$ & 0.83 & $9.3 \times 10^{4}$ & 0.94 & 0.98 \\ 
# # 2 &  -1 & 0.32 &   1 & 0.32 & -0.5 & 0.32 &   6 & 0.1 & $5.7 \times 10^{5}$ &  40 &   2 & 2.6 & $1.1 \times 10^{8}$ & 2.6 & $3.3 \times 10^{10}$ & 0.91 & 0.95 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final10c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Fri Apr 23 02:03:26 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & -0.5 & 0.32 &   1 & 0.32 & -0.5 & 0.32 &   6 & 0.1 & $1.3 \times 10^{5}$ &  88 &   2 & 2.6 & $1.8 \times 10^{7}$ &   3 & $3.2 \times 10^{7}$ & 0.84 & 0.91 \\ 
# # 2 & -0.25 & 0.32 &   1 & 0.32 & -0.5 & 0.32 &   6 & 0.1 & $9.3 \times 10^{2}$ &  62 &   3 & 2.6 & $2.3 \times 10^{5}$ & 1.1 & $2.6 \times 10^{8}$ & 0.9 & 0.95 \\ 
# # 3 & -0.5 & 0.32 &   1 & 0.32 & -0.25 & 0.32 &   6 & 0.1 & $5.1 \times 10^{5}$ & $2 \times 10^{2}$ &   6 & 2.6 & $9.1 \times 10^{7}$ & 3.5 & $6.2 \times 10^{7}$ & 0.84 & 0.91 \\ 
# # 4 & -0.25 & 0.32 &   1 & 0.32 & -0.25 & 0.32 &   6 & 0.1 & $2.9 \times 10^{5}$ & $2.8 \times 10^{2}$ &   0 & 2.6 & $4.2 \times 10^{7}$ & 2.1 & $7.9 \times 10^{8}$ & 0.83 & 0.9 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final11c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Fri Apr 23 13:52:22 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &  -1 & 0.32 &   1 & 0.32 & -0.1 & 0.32 &   6 & 0.1 & $9.4 \times 10^{4}$ & $1.6 \times 10^{2}$ &   4 & 2.6 & $1.1 \times 10^{7}$ &  15 & $1.2 \times 10^{9}$ & 0.86 & 0.92 \\ 
# # 2 &  -1 & 0.5 &   1 & 0.32 & -0.1 & 0.32 &   6 & 0.1 & $3.4 \times 10^{3}$ &  75 &   3 & 2.6 & $1.5 \times 10^{6}$ & 1.1 & $4.9 \times 10^{7}$ & 0.92 & 0.97 \\ 
# # 3 &  -1 & 0.32 &   1 & 0.5 & -0.1 & 0.32 &   6 & 0.1 & $6.8 \times 10^{5}$ & $1.9 \times 10^{2}$ &   4 & 2.6 & $9.4 \times 10^{7}$ & 3.2 & $3.8 \times 10^{8}$ & 0.86 & 0.92 \\ 
# # 4 &  -1 & 0.5 &   1 & 0.5 & -0.1 & 0.32 &   6 & 0.1 & $1.9 \times 10^{5}$ & $4.8 \times 10^{2}$ &   1 & 2.6 & $4.3 \times 10^{7}$ & 2.1 & $4.6 \times 10^{7}$ & 0.86 & 0.92 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final12c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Fri Apr 23 12:24:19 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &  -1 & 0.5 &   1 & 0.5 & -0.1 & 0.32 &   4 & 0.1 & $7.4 \times 10^{3}$ &  50 &   2 & 1.7 & $2 \times 10^{6}$ & 0.76 & $3 \times 10^{6}$ & 0.92 & 0.96 \\ 
# # 2 & -0.5 & 0.5 &   1 & 0.5 & -0.1 & 0.32 &   4 & 0.1 & $3.7 \times 10^{4}$ &  77 &   1 & 1.7 & $2.9 \times 10^{6}$ & 0.89 & $3.4 \times 10^{6}$ & 0.92 & 0.97 \\ 
# # 3 &  -1 & 0.5 &   1 & 0.5 & -0.5 & 0.32 &   4 & 0.1 & $5.8 \times 10^{4}$ &  42 &   1 & 1.7 & $3.9 \times 10^{7}$ & 0.96 & $9.1 \times 10^{7}$ & 0.91 & 0.96 \\ 
# # 4 & -0.5 & 0.5 &   1 & 0.5 & -0.5 & 0.32 &   4 & 0.1 & $1.8 \times 10^{4}$ &  73 &   0 & 1.7 & $3.9 \times 10^{6}$ & 1.2 & $4.4 \times 10^{6}$ & 0.9 & 0.96 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final13c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Thu Apr 29 09:22:20 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & 2.5 & 0.5 & 0.5 & 0.5 & -0.5 & 0.5 & 0.75 & 0.1 & $1.2 \times 10^{3}$ & $2.6 \times 10^{2}$ &   0 & 0.33 & $2 \times 10^{5}$ & 1.7 & $1.7 \times 10^{5}$ & 0.62 & 0.84 \\ 
# # 2 & 2.5 &   1 & 0.5 & 0.5 & -0.5 & 0.5 & 0.75 & 0.1 & $1.1 \times 10^{3}$ & $1.6 \times 10^{2}$ &   0 & 0.33 & $1.5 \times 10^{5}$ & 1.3 & $1.2 \times 10^{5}$ & 0.7 & 0.88 \\ 
# # 3 & 2.5 & 0.5 & 0.5 & 0.5 & -0.5 & 0.5 &   1 & 0.1 & $1.7 \times 10^{3}$ & $2.5 \times 10^{2}$ &   0 & 0.43 & $1.7 \times 10^{5}$ & 1.4 & $1.2 \times 10^{5}$ & 0.62 & 0.85 \\ 
# # 4 & 2.5 &   1 & 0.5 & 0.5 & -0.5 & 0.5 &   1 & 0.1 & $4.6 \times 10^{3}$ & $2.1 \times 10^{2}$ &   0 & 0.43 & $9.7 \times 10^{5}$ & 1.5 & $6 \times 10^{5}$ & 0.66 & 0.85 \\ 
# # 5 & 2.5 & 0.5 & 0.5 & 0.5 & -0.5 & 0.5 &   2 & 0.1 & $1.1 \times 10^{5}$ & $3.4 \times 10^{2}$ &   0 & 0.87 & $2.9 \times 10^{7}$ &   2 & $1.9 \times 10^{7}$ & 0.63 & 0.83 \\ 
# # 6 & 2.5 &   1 & 0.5 & 0.5 & -0.5 & 0.5 &   2 & 0.1 & $3.3 \times 10^{4}$ & $2.3 \times 10^{2}$ &   0 & 0.87 & $1.2 \times 10^{7}$ &   2 & $4.9 \times 10^{6}$ & 0.64 & 0.82 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final14c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Thu Apr 29 09:39:20 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & 2.5 & 0.5 &   1 & 0.5 &  -1 & 0.5 & 0.75 & 0.1 & $7.6 \times 10^{3}$ & $3.5 \times 10^{2}$ &   0 & 0.33 & $2.5 \times 10^{6}$ & 1.3 & $1.9 \times 10^{6}$ & 0.62 & 0.84 \\ 
# # 2 & 2.5 &   1 &   1 & 0.5 &  -1 & 0.5 & 0.75 & 0.1 & $1 \times 10^{4}$ & $3.3 \times 10^{2}$ &   0 & 0.33 & $3.2 \times 10^{6}$ & 1.3 & $2.3 \times 10^{6}$ & 0.68 & 0.87 \\ 
# # 3 & 2.5 & 0.5 &   1 & 0.5 &  -1 & 0.5 &   1 & 0.1 & $1.2 \times 10^{4}$ & $3.5 \times 10^{2}$ &   0 & 0.43 & $2.9 \times 10^{6}$ & 1.5 & $1.9 \times 10^{6}$ & 0.61 & 0.83 \\ 
# # 4 & 2.5 &   1 &   1 & 0.5 &  -1 & 0.5 &   1 & 0.1 & $3.2 \times 10^{4}$ & $3.9 \times 10^{2}$ &   0 & 0.43 & $9.2 \times 10^{6}$ & 1.4 & $4.1 \times 10^{6}$ & 0.64 & 0.83 \\ 
# # 5 & 2.5 & 0.5 &   1 & 0.5 &  -1 & 0.5 &   2 & 0.1 & $2 \times 10^{4}$ & $2.6 \times 10^{2}$ &   0 & 0.87 & $4 \times 10^{6}$ & 1.6 & $2.2 \times 10^{6}$ & 0.67 & 0.85 \\ 
# # 6 & 2.5 &   1 &   1 & 0.5 &  -1 & 0.5 &   2 & 0.1 & $1.3 \times 10^{5}$ & $3.9 \times 10^{2}$ &   0 & 0.87 & $2.5 \times 10^{7}$ &   2 & $1.1 \times 10^{7}$ & 0.61 & 0.8 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final15c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Thu Apr 29 09:44:12 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &   2 &   1 &   1 & 0.5 &  -1 & 0.5 & 0.75 & 0.1 & $2.4 \times 10^{3}$ & $2.4 \times 10^{2}$ &   0 & 0.33 & $5.4 \times 10^{5}$ & 1.3 & $4.4 \times 10^{5}$ & 0.69 & 0.87 \\ 
# # 2 &   2 &   1 &   1 & 0.5 &  -1 & 0.5 &   1 & 0.1 & $4.4 \times 10^{3}$ & $1.6 \times 10^{2}$ &   0 & 0.43 & $1.2 \times 10^{6}$ & 1.4 & $1.1 \times 10^{6}$ & 0.76 & 0.9 \\ 
# # 3 &   2 &   1 &   1 & 0.5 &  -1 & 0.5 &   2 & 0.1 & $1.6 \times 10^{4}$ & $5.2 \times 10^{2}$ &   0 & 0.87 & $5.6 \times 10^{6}$ & 2.3 & $3.1 \times 10^{6}$ & 0.6 & 0.8 \\ 
# # \hline
# # \end{tabular}
# # \end{table}
# % latex table generated in R 4.0.2 by xtable 1.8-4 package
# % Fri Apr 30 08:51:38 2021
# \begin{table}[ht]
# \centering
# \begin{tabular}{cccccccccccccccccc}
# \hline
# & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# \hline
# 1 &   2 &   1 &   1 & 0.5 &  -1 & 0.5 & 0.75 & 0.1 & $2.4 \times 10^{3}$ & $2.4 \times 10^{2}$ &   0 & 0.33 & $5.4 \times 10^{5}$ & 1.3 & $4.4 \times 10^{5}$ & 0.69 & 0.87 \\ 
# 2 &   2 &   1 &   1 & 0.5 &  -1 & 0.5 &   1 & 0.1 & $4.4 \times 10^{3}$ & $1.6 \times 10^{2}$ &   0 & 0.43 & $1.2 \times 10^{6}$ & 1.4 & $1.1 \times 10^{6}$ & 0.76 & 0.9 \\ 
# 3 &   2 &   1 &   1 & 0.5 &  -1 & 0.5 &   2 & 0.1 & $1.6 \times 10^{4}$ & $5.2 \times 10^{2}$ &   0 & 0.87 & $5.6 \times 10^{6}$ & 2.3 & $3.1 \times 10^{6}$ & 0.6 & 0.8 \\ 
# \hline
# \end{tabular}
# \end{table}


# load("LACovFixedEffectsPrior_Final16c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Sat May 01 23:00:46 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &   2 &   1 & 0.75 & 0.71 & -0.5 & 0.71 &   1 & 0.1 & $1.4 \times 10^{5}$ & $3.3 \times 10^{2}$ &   0 & 0.43 & $5.4 \times 10^{7}$ & 1.3 & $4 \times 10^{7}$ & 0.68 & 0.86 \\ 
# # 2 &   2 &   1 & 0.75 & 0.71 & -0.5 & 0.71 &   2 & 0.1 & $3.7 \times 10^{5}$ & $3.1 \times 10^{2}$ &   0 & 0.87 & $1 \times 10^{8}$ & 1.4 & $6.5 \times 10^{7}$ & 0.75 & 0.89 \\ 
# # 3 &   2 &   1 & 0.75 & 0.71 & -0.5 & 0.71 &   3 & 0.1 & $5.7 \times 10^{5}$ & $6.9 \times 10^{2}$ &   0 & 1.3 & $1.2 \times 10^{8}$ & 2.4 & $7.8 \times 10^{7}$ & 0.62 & 0.8 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final17c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Sun May 02 00:25:22 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &   3 &   1 & 0.75 & 0.5 & -0.5 & 0.5 &   1 & 0.1 & $7.7 \times 10^{3}$ & $7.7 \times 10^{2}$ &   0 & 0.43 & $1.5 \times 10^{6}$ & 1.5 & $9 \times 10^{5}$ & 0.5 & 0.74 \\ 
# # 2 &   3 &   1 & 0.75 & 0.5 & -0.5 & 0.5 &   2 & 0.1 & $1.6 \times 10^{4}$ & $5.2 \times 10^{2}$ &   0 & 0.87 & $2.6 \times 10^{6}$ &   2 & $1.6 \times 10^{6}$ & 0.58 & 0.78 \\ 
# # 3 &   3 &   1 & 0.75 & 0.5 & -0.5 & 0.5 &   3 & 0.1 & $4.6 \times 10^{5}$ & $1.5 \times 10^{3}$ &   0 & 1.3 & $3.2 \times 10^{7}$ & 2.5 & $1.1 \times 10^{7}$ & 0.48 & 0.68 \\ 
# # \hline
# # \end{tabular}
# # \end{table}
# % latex table generated in R 4.0.2 by xtable 1.8-4 package
# % Sun May 02 10:19:20 2021
# \begin{table}[ht]
# \centering
# \begin{tabular}{cccccccccccccccccc}
# \hline
# & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# \hline
# 1 &   3 &   1 & 0.75 & 0.5 & -0.5 & 0.5 &   1 & 0.1 & $7.7 \times 10^{3}$ & $7.7 \times 10^{2}$ &   0 & 0.43 & $1.5 \times 10^{6}$ & 1.5 & $9 \times 10^{5}$ & 0.5 & 0.74 \\ 
# 2 &   3 &   1 & 0.75 & 0.5 & -0.5 & 0.5 &   2 & 0.1 & $1.6 \times 10^{4}$ & $5.2 \times 10^{2}$ &   0 & 0.87 & $2.6 \times 10^{6}$ &   2 & $1.6 \times 10^{6}$ & 0.58 & 0.78 \\ 
# 3 &   3 &   1 & 0.75 & 0.5 & -0.5 & 0.5 &   3 & 0.1 & $4.6 \times 10^{5}$ & $1.5 \times 10^{3}$ &   0 & 1.3 & $3.2 \times 10^{7}$ & 2.5 & $1.1 \times 10^{7}$ & 0.48 & 0.68 \\ 
# \hline
# \end{tabular}
# \end{table}


# load("LACovFixedEffectsPrior_Final18c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Sat May 01 21:00:54 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &   2 &   1 & 0.75 & 0.5 & -0.5 & 0.5 &   1 & 0.1 & $2.4 \times 10^{3}$ & $2.7 \times 10^{2}$ &   0 & 0.43 & $5.5 \times 10^{5}$ & 1.4 & $4.6 \times 10^{5}$ & 0.7 & 0.87 \\ 
# # 2 &   2 &   1 & 0.75 & 0.5 & -0.5 & 0.5 &   2 & 0.1 & $6.5 \times 10^{3}$ & $2.3 \times 10^{2}$ &   0 & 0.87 & $7.4 \times 10^{5}$ & 3.2 & $3.4 \times 10^{5}$ & 0.73 & 0.87 \\ 
# # 3 &   2 &   1 & 0.75 & 0.5 & -0.5 & 0.5 &   3 & 0.1 & $1.7 \times 10^{4}$ & $7 \times 10^{2}$ &   0 & 1.3 & $3.3 \times 10^{6}$ & 2.1 & $1.2 \times 10^{6}$ & 0.62 & 0.8 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final19c.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Sat May 01 22:55:49 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &   2 &   1 &   1 & 0.71 &  -1 & 0.71 & 0.75 & 0.1 & $1.9 \times 10^{5}$ & $3.9 \times 10^{2}$ &   0 & 0.33 & $7.4 \times 10^{7}$ & 1.4 & $6.8 \times 10^{7}$ & 0.67 & 0.85 \\ 
# # 2 &   2 &   1 &   1 & 0.71 &  -1 & 0.71 &   1 & 0.1 & $3 \times 10^{5}$ & $3.2 \times 10^{2}$ &   0 & 0.43 & $9.5 \times 10^{7}$ & 1.3 & $7.7 \times 10^{7}$ & 0.75 & 0.89 \\ 
# # 3 &   2 &   1 &   1 & 0.71 &  -1 & 0.71 &   2 & 0.1 & $3.6 \times 10^{5}$ & $7.1 \times 10^{2}$ &   0 & 0.87 & $9.9 \times 10^{7}$ & 2.6 & $9.3 \times 10^{7}$ & 0.59 & 0.8 \\ 
# # \hline
# # \end{tabular}
# # \end{table}



# sessionInfo() -----------------------------------------------------------

sessionInfo()