
# Testing Priors for Irregular Polygon ------------------------------------

# This R script will test the number of points simulated as well as the proportion of cells with zero counts for varying priors for the fixed effects over the Los Angeles polygon, aiming to match proportions of zeros with the LA data (as much as reasonable) as well as reasonable mean and median n simulated points.
# Outputs from this have the suffix *_Finali.rda with i taking several integer values.

# Author: Nadeen Khaleel


# Setwd and Load Libraries ------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))

library(doParallel)
library(foreach)


# INLA GLM Results --------------------------------------------------------
# Fit GLM with INLA to finest grid for LA homicide and motor vehicle theft, using the GMO data, so that the socio-economic variables match those used to simulate the LGCP.

library(INLA)
# load("../../../../DATA/PROCESSED_DATA/CRIME/COUNT_DATA_GMO/LA/LA2015CT236359CountData_proj.rda")
# hom_countdf$zpop <- (hom_countdf$pop - mean(hom_countdf$pop))/sd(hom_countdf$pop)
# hom_countdf$zinc <- (hom_countdf$inc - mean(hom_countdf$inc))/sd(hom_countdf$inc)
# # fit.glm <- glm(hom ~ 1+ zpop + zinc,data=hom_countdf,family="poisson")
# fit.glm <- inla(hom ~ 1+ zpop + zinc,data=hom_countdf,family="poisson")
# fit.glm$summary.fixed
# # mean         sd 0.025quant   0.5quant 0.975quant       mode          kld
# # (Intercept) -5.0379608 0.06838331 -5.1747257 -5.0370934 -4.9061282 -5.0353703 1.812007e-09
# # zpop         0.7699032 0.04832476  0.6751325  0.7698644  0.8648026  0.7697908 8.726262e-07
# # zinc        -0.4731952 0.08678917 -0.6484224 -0.4714974 -0.3075248 -0.4681270 2.716181e-07
# # 
# gta_countdf$zpop <- (gta_countdf$pop - mean(gta_countdf$pop))/sd(gta_countdf$pop)
# gta_countdf$zinc <- (gta_countdf$inc - mean(gta_countdf$inc))/sd(gta_countdf$inc)
# # fit.glm <- glm(gta ~ 1+ zpop + zinc,data=gta_countdf,family="poisson")
# fit.glm <- inla(gta ~ 1+ zpop + zinc,data=gta_countdf,family="poisson")
# fit.glm$summary.fixed
# fit.glm$summary.fixed
# # mean          sd 0.025quant   0.5quant  0.975quant       mode          kld
# # (Intercept) -0.9097305 0.008753028 -0.9269574 -0.9097164 -0.89259854 -0.9096876 1.023746e-06
# # zpop         0.5241291 0.006287413  0.5117815  0.5241301  0.53645944  0.5241327 4.545158e-07
# # zinc        -0.1019944 0.009264106 -0.1202739 -0.1019636 -0.08390399 -0.1019013 8.980831e-07


# Including area as an offset
# fit.glm <- inla(hom ~ 1 + offset(log(area)) + zpop + zinc,data=hom_countdf,family="poisson")
# fit.glm$summary.fixed
# mean         sd  0.025quant    0.5quant  0.975quant        mode          kld
# (Intercept) -15.5969175 0.06847258 -15.7338558 -15.5960509 -15.4649078 -15.5943291 2.024769e-09
# zpop          0.7724722 0.04857581   0.6772023   0.7724356   0.8678585   0.7723663 8.535231e-07
# zinc         -0.4956648 0.08807677  -0.6734560  -0.4939548  -0.3274998  -0.4905606 2.605868e-07
# 
# 
# fit.glm <- inla(hom ~ 1 + offset(log(area/1e8)) + zpop + zinc,data=hom_countdf,family="poisson")
# fit.glm$summary.fixed
# mean         sd 0.025quant   0.5quant 0.975quant       mode
# (Intercept)  2.8237737 0.06847232  2.6868359  2.8246403  2.9557829  2.8263620
# zpop         0.7724685 0.04857571  0.6771986  0.7724319  0.8678544  0.7723627
# zinc        -0.4956603 0.08807651 -0.6734506 -0.4939504 -0.3274954 -0.4905564
# kld
# (Intercept) 2.029426e-09
# zpop        8.531061e-07

# fit.glm <- inla(gta ~ 1 + offset(log(area)) + zpop + zinc,data=gta_countdf,family="poisson")
# fit.glm$summary.fixed
# mean          sd  0.025quant    0.5quant   0.975quant        mode          kld
# (Intercept) -11.4693401 0.008773817 -11.4866079 -11.4693260 -11.45216745 -11.4692972 1.023547e-06
# zpop          0.5235963 0.006326989   0.5111706   0.5235975   0.53600393   0.5236003 4.790313e-07
# zinc         -0.1165730 0.009404844  -0.1351294  -0.1165419  -0.09820709  -0.1164792 9.015960e-07
# 
# fit.glm <- inla(gta ~ 1 + offset(log(area/1e8)) + zpop + zinc,data=gta_countdf,family="poisson")
# fit.glm$summary.fixed
# mean          sd 0.025quant   0.5quant  0.975quant       mode
# (Intercept)  6.9513408 0.008773817  6.9340730  6.9513549  6.96851344  6.9513837
# zpop         0.5235963 0.006326989  0.5111706  0.5235975  0.53600391  0.5236003
# zinc        -0.1165730 0.009404844 -0.1351294 -0.1165419 -0.09820709 -0.1164792
# kld
# (Intercept) 1.023662e-06
# zpop        4.790306e-07
# zinc        9.015955e-07



# Simulation Set-Up -------------------------------------------------------
# Select parameter values for the priors, with the index of the outputs also attached. Then set-up the combinations and parallelsation.
# The label out i matches the i in the output filename

# mean.beta0 <- c(-1); mean.beta1 <- c(1); mean.beta2 <- c(-0.5) # out 1
# sd.beta0 <- c(sqrt(0.5)); sd.beta1 <- c(sqrt(0.5)); sd.beta2 <- c(sqrt(0.5))
# sigma.0 <- c(0.1,1,2); alpha.sigma <- c(0.01,0.1,0.25) # out 1
# mean.beta0 <- c(-0.5); mean.beta1 <- c(1); mean.beta2 <- c(-0.5) # out 2
# sd.beta0 <- c(sqrt(0.5)); sd.beta1 <- c(sqrt(0.5)); sd.beta2 <- c(sqrt(0.5))
# sigma.0 <- c(0.1,0.5,1); alpha.sigma <- c(0.01,0.1,0.25) # out 2
# mean.beta0 <- c(-1); mean.beta1 <- c(1); mean.beta2 <- c(-0.5) # out 3
# sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
# sigma.0 <- c(0.1,0.5,0.75,1); alpha.sigma <- c(0.01,0.1) # out 3
# mean.beta0 <- c(-0.5); mean.beta1 <- c(1); mean.beta2 <- c(-0.5) # out 4
# sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
# sigma.0 <- c(0.5,0.75,1,2); alpha.sigma <- c(0.01,0.1) # out 4
# mean.beta0 <- c(-0.5); mean.beta1 <- c(1.5); mean.beta2 <- c(-0.5) # out 5
# sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
# sigma.0 <- c(0.5,0.75,1,2); alpha.sigma <- c(0.01,0.1) # out 5
# mean.beta0 <- c(-0.5); mean.beta1 <- c(1.5); mean.beta2 <- c(-0.5) # out 6
# sd.beta0 <- c(sqrt(0.25)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
# sigma.0 <- c(0.5,0.75,1,2); alpha.sigma <- c(0.01,0.1) # out 6
# mean.beta0 <- c(-0.25); mean.beta1 <- c(1); mean.beta2 <- c(-0.25) # out 7
# sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
# sigma.0 <- c(1,2,3); alpha.sigma <- c(0.01,0.1) # out 7
# mean.beta0 <- c(-0.5); mean.beta1 <- c(2); mean.beta2 <- c(-2) # out 8
# sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
# sigma.0 <- c(0.5,1,2); alpha.sigma <- c(0.01,0.1) # out 8
# mean.beta0 <- c(0.5); mean.beta1 <- c(0.5); mean.beta2 <- c(-0.5) # out 9
# sd.beta0 <- c(sqrt(0.5)); sd.beta1 <- c(sqrt(0.5)); sd.beta2 <- c(sqrt(0.5))
# sigma.0 <- c(0.5,1,2); alpha.sigma <- c(0.01,0.1) # out 9
# mean.beta0 <- c(-0.5); mean.beta1 <- c(1); mean.beta2 <- c(-0.5) # out 10
# sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
# sigma.0 <- c(1,2,5); alpha.sigma <- c(0.1) # out 10
mean.beta0 <- c(-0.5); mean.beta1 <- c(2); mean.beta2 <- c(-2) # out 11
sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
sigma.0 <- c(1,2,5); alpha.sigma <- c(0.1) # out 11

# List for parameters which are then turned into the final combinations
param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)

param.combos <- expand.grid(param.list)

# Set-up parallelisation
x <- 1:nrow(param.combos)
Nproc <- 4
cl <- makeCluster(Nproc)
registerDoParallel(cl)
indexes <- split(x, sort(rep_len(1:Nproc, length(x))))


# Implement Simulations ---------------------------------------------------

ptm.start <- proc.time()
foreach(i=1:Nproc) %dopar% {
  
  library(spatstat)
  library(raster)
  library(sf)
  library(sp)
  library(maptools)
  library(rgdal)
  library(lwgeom)
  library(RandomFields)
  
  
  # Set-up ------------------------------------------------------------------
  # Need to set-up the parameters again for the each core - due to the parallelisation, we cannot fork so we have to set- things up again, make sure to match the parameters!
  
  # LA Census Tract
  load("../../../../DATA/PROCESSED_DATA/SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")
  lacity_boundary <- st_union(ct_LA)
  lacity_proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611")
  lacity_sp <- as(lacity_proj,"Spatial")
  
  # Transform the window
  bbox <- lacity_sp@bbox
  lacity_spshift <- elide(lacity_sp,shift=-c(bbox[1,1],bbox[2,1]))
  bbox_shift <- lacity_spshift@bbox
  lacity_spscale <- elide(lacity_spshift,scale=max(bbox_shift)/1e4)
  W <- as.owin.SpatialPolygons((lacity_spscale))
  
  # Get covariates
  cov.name <- "GridMeshIrregPolLGCPSSCov.rda"
  load(cov.name)
  
  # Need to include parameters again - make sure they match! the label out i matches the i in the output filename
  # mean.beta0 <- c(-1); mean.beta1 <- c(1); mean.beta2 <- c(-0.5) # out 1
  # sd.beta0 <- c(sqrt(0.5)); sd.beta1 <- c(sqrt(0.5)); sd.beta2 <- c(sqrt(0.5))
  # sigma.0 <- c(0.1,1,2); alpha.sigma <- c(0.01,0.1,0.25) # out 1
  # mean.beta0 <- c(-0.5); mean.beta1 <- c(1); mean.beta2 <- c(-0.5) # out 2
  # sd.beta0 <- c(sqrt(0.5)); sd.beta1 <- c(sqrt(0.5)); sd.beta2 <- c(sqrt(0.5))
  # sigma.0 <- c(0.1,0.5,1); alpha.sigma <- c(0.01,0.1,0.25) # out 2
  # mean.beta0 <- c(-1); mean.beta1 <- c(1); mean.beta2 <- c(-0.5) # out 3
  # sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
  # sigma.0 <- c(0.1,0.5,0.75,1); alpha.sigma <- c(0.01,0.1) # out 3
  # mean.beta0 <- c(-0.5); mean.beta1 <- c(1); mean.beta2 <- c(-0.5) # out 4
  # sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
  # sigma.0 <- c(0.5,0.75,1,2); alpha.sigma <- c(0.01,0.1) # out 4
  # mean.beta0 <- c(-0.5); mean.beta1 <- c(1.5); mean.beta2 <- c(-0.5) # out 5
  # sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
  # sigma.0 <- c(0.5,0.75,1,2); alpha.sigma <- c(0.01,0.1) # out 5
  # mean.beta0 <- c(-0.5); mean.beta1 <- c(1.5); mean.beta2 <- c(-0.5) # out 6
  # sd.beta0 <- c(sqrt(0.25)); sd.beta1 <- c(sqrt(0.25)); sd.beta2 <- c(sqrt(0.25))
  # sigma.0 <- c(0.5,0.75,1,2); alpha.sigma <- c(0.01,0.1) # out 6
  # mean.beta0 <- c(-0.25); mean.beta1 <- c(1); mean.beta2 <- c(-0.25) # out 7
  # sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
  # sigma.0 <- c(1,2,3); alpha.sigma <- c(0.01,0.1) # out 7
  # mean.beta0 <- c(-0.5); mean.beta1 <- c(2); mean.beta2 <- c(-2) # out 8
  # sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
  # sigma.0 <- c(0.5,1,2); alpha.sigma <- c(0.01,0.1) # out 8
  # mean.beta0 <- c(0.5); mean.beta1 <- c(0.5); mean.beta2 <- c(-0.5) # out 9
  # sd.beta0 <- c(sqrt(0.5)); sd.beta1 <- c(sqrt(0.5)); sd.beta2 <- c(sqrt(0.5))
  # sigma.0 <- c(0.5,1,2); alpha.sigma <- c(0.01,0.1) # out 9
  # mean.beta0 <- c(-0.5); mean.beta1 <- c(1); mean.beta2 <- c(-0.5) # out 10
  # sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
  # sigma.0 <- c(1,2,5); alpha.sigma <- c(0.1) # out 10
  mean.beta0 <- c(-0.5); mean.beta1 <- c(2); mean.beta2 <- c(-2) # out 11
  sd.beta0 <- c(sqrt(0.1)); sd.beta1 <- c(sqrt(0.1)); sd.beta2 <- c(sqrt(0.1))
  sigma.0 <- c(1,2,5); alpha.sigma <- c(0.1) # out 11
  
  param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)
  
  param.combos <- expand.grid(param.list)
  
  # Load grids to check proportions of zero
  load("QuadratsIrregPolLGCP.rda")
  g <- quad.list[2:3] # 24x36 and 48x72 grid
  
  N <- 50  # number of simulations for standard information seeking test, if there are any of interest run with N <- 100
  
  # Change output filename as needed, don't forget to change accordingly in the rest of the code!
  out.filename <- "LACovFixedEffectsPrior_Final11.rda"
  if ((!file.exists(out.filename)) & (i==1)){
    out <- vector(mode = "list", length = nrow(param.combos))
    save(out,file=out.filename)
  }
  
  # Functions ---------------------------------------------------------------
  
  # Simulate priors from hyper-parameters provided
  prior.sim <- function(n=1,mu.beta0,sigma.beta0,mu.beta1,sigma.beta1,mu.beta2,sigma.beta2,sigma.0,alpha.sigma){
    # n is the number of samples of the parameters you want sampled from their priors.  
    # mu.*: prior mean for fixed effects
    # sigma.beta*: standard deviation for fixed effects
    # sigma.0, alpha.sigma: prior parameters for sigma pc prior
    
    beta.0.sim <- rnorm(n,mu.beta0,sigma.beta0); beta.1.sim <- rnorm(n,mu.beta1,sigma.beta1); beta.2.sim <- rnorm(n,mu.beta2,sigma.beta2)
    
    # For the covariance parameters of the latent Gaussian field, I have a joint prior on the marginal standard deviation and the range, however, in order to simulate from this I need the inverse CDF to generate this from a Uniform RV. However, as this is a joint distribution I cannot quite get the correct arrangement for the inverse. Therefore, I will look at the the prior on kappa (and turn this into a pdf for rho) and then look at the conditional distribution of tau given kappa (and turn this into a pdf for sigma).
    # (https://www.tandfonline.com/doi/pdf/10.1080/01621459.2017.1415907?needAccess=true) 
    # (https://arxiv.org/pdf/1503.00256.pdf)
    
    alpha.rho <- 0.1; rho.0 <- 0.15; # 
    u1 <- runif(n,0,1); u2 <- runif(n,0,1)
    rho.sim <- rho.0*log(alpha.rho)/log(u1)
    sigma.sim <- sigma.0*log(1-u2)/log(alpha.sigma)
    return(list(beta0.tilde=beta.0.sim,beta1.tilde=beta.1.sim,beta2.tilde=beta.2.sim,sigma.tilde=sigma.sim,rho.tilde=rho.sim))
  }
  
  # Simulate the LGCPs and output data frame for the number of points simulated as well as the proportions of zeros etc.
  res.gen <- function(N,W,prior.param,g.tess,c1.im,c2.im,i.im,seed){
    # N is the number of LGCP simulations we want
    # W is the window
    # prior.param is the set of hyper-parameters for the priors ro simulate the parameter for the generation of the LGCP
    # g.tess are the list of quadrats for the discretisations to test the proportion of zeros
    # i.im, c*.im are the pixel images for the intercept and covariates over the study region
    # seed is to set the simulation seed
    
    set.seed(seed)
    # simulated parameter values
    thetas <- prior.sim(n=N,mu.beta0 = prior.param$mu.beta0,sigma.beta0 = prior.param$sd.beta0,mu.beta1 = prior.param$mu.beta1,sigma.beta1 = prior.param$sd.beta1,mu.beta2 = prior.param$mu.beta2,sigma.beta2 = prior.param$sd.beta2,sigma.0=prior.param$sigma.0,alpha.sigma = prior.param$alpha.sigma)
    
    # Create output storage for lgcp.ppp$n and proportion of zeros for different grids.
    grid.names <- names(g.tess)
    output.df <- data.frame(matrix(rep(NA,(length(grid.names)+3)*N),nrow=N))
    colnames(output.df) <- c("n","max.lambda","med.lambda",grid.names)
    
    # Simulated N LGCPs
    for (j in 1:N){
      print(paste0("Simulation ", j))
      # Extract parameter values simulated from priors
      theta <- lapply(1:length(names(thetas)),function(k,j){thetas[[k]][j]},j)
      names(theta) <- names(thetas)
      beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]]; sigma <- theta[[4]]; rho <- theta[[5]];
      
      mu <- beta.0*i.im + beta.1*c1.im + beta.2*c2.im
      # Produce the point pattern from LGCP
      lgcp.ppp <- try(rLGCP(model="matern",mu,var=(sigma)^2,scale=rho/2,nu=1,win = W,saveLambda = TRUE))
      
      if (class(lgcp.ppp)!="try-error"){
        output.df$n[j] <- lgcp.ppp$n
        att.ppp <- attributes(lgcp.ppp)
        output.df$max.lambda[j] <- max(att.ppp$Lambda)
        output.df$med.lambda[j] <- median(att.ppp$Lambda)
        
        for (i in 1:length(g.tess)){
          col.df <- grid.names[i] # for the grid proportion of zeros in the output.df need to account for the "n" column
          q <- quadratcount(lgcp.ppp,tess=g.tess[[i]])
          count.df <- data.frame(count=as.vector(unname(q)))
          
          output.df[j,col.df] <- sum(count.df$count==0)/length(count.df$count)
        }
      }
    }
    return(list("results"=output.df))
  }
  
  
  # Run All -----------------------------------------------------------------
  # Run the simulations using the parameter set-up and the functions above
  
  seeds <- 5*(1:nrow(param.combos))
  
  # Label output, change file name accordingly
  out.filename <- "LACovFixedEffectsPrior_Final11.rda"
  load(out.filename)
  
  x <- which(unlist(sapply(1:length(out),function(i){(is.null(out[[i]]))}))) # for re-setting
  Nproc <- 4
  indexes <- split(x, sort(rep_len(1:Nproc, length(x))))
  for (k in indexes[[i]]){
    print(k)
    out.new <- res.gen(N,W,param.combos[k,],g.tess=g,popb.im,incb.im,intb.im,seeds[k])
    load(out.filename)
    out[[k]] <- out.new
    save(out,file=out.filename)
  }
}

ptm.end <- proc.time()
stopCluster(cl)
ptm.end-ptm.start

# Change file name accordingly
out.filename <- "LACovFixedEffectsPrior_Final11.rda"

# Get the file and label the outputs so that we can more easily tell which parameters resulted in which results
load(out.filename)
combo.names <- sapply(1:nrow(param.combos),function(i){paste0("(",paste0(param.combos[i,],collapse=","),")")})
names(out) <- combo.names

out

# Save labelled output
save(out,file=out.filename)



# Check Output ------------------------------------------------------------
# Load outputs and generate tables, these can then be placed in a .tex file for easy reading

# load("LACovFixedEffectsPrior_Final1.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Tue Apr 20 13:49:42 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &  -1 & 0.71 &   1 & 0.71 & -0.5 & 0.71 & 0.1 & 0.01 & $5.8 \times 10^{4}$ &  20 &   0 & 0.022 & $2 \times 10^{7}$ & 0.99 & $2 \times 10^{7}$ & 0.96 & 0.98 \\ 
# # 2 &  -1 & 0.71 &   1 & 0.71 & -0.5 & 0.71 &   1 & 0.01 & $9.6 \times 10^{4}$ &  14 &   0 & 0.22 & $2.7 \times 10^{7}$ & 0.89 & $3.1 \times 10^{7}$ & 0.96 & 0.99 \\ 
# # 3 &  -1 & 0.71 &   1 & 0.71 & -0.5 & 0.71 &   2 & 0.01 & $1.2 \times 10^{5}$ &  18 &   0 & 0.43 & $2.2 \times 10^{7}$ & 0.88 & $2.1 \times 10^{7}$ & 0.95 & 0.98 \\ 
# # 4 &  -1 & 0.71 &   1 & 0.71 & -0.5 & 0.71 & 0.1 & 0.1 & $2.5 \times 10^{5}$ &  30 &   0 & 0.043 & $5.9 \times 10^{7}$ & 0.97 & $6 \times 10^{7}$ & 0.95 & 0.98 \\ 
# # 5 &  -1 & 0.71 &   1 & 0.71 & -0.5 & 0.71 &   1 & 0.1 & $1.3 \times 10^{5}$ &   9 &   0 & 0.43 & $3.9 \times 10^{7}$ & 0.81 & $6.7 \times 10^{7}$ & 0.96 & 0.98 \\ 
# # 6 &  -1 & 0.71 &   1 & 0.71 & -0.5 & 0.71 &   2 & 0.1 & $1 \times 10^{6}$ &  27 &   0 & 0.87 & $2.5 \times 10^{8}$ & 0.89 & $3.5 \times 10^{8}$ & 0.93 & 0.97 \\ 
# # 7 &  -1 & 0.71 &   1 & 0.71 & -0.5 & 0.71 & 0.1 & 0.25 & $7.4 \times 10^{5}$ &  27 &   1 & 0.072 & $2.6 \times 10^{8}$ & 0.96 & $2.8 \times 10^{8}$ & 0.95 & 0.98 \\ 
# # 8 &  -1 & 0.71 &   1 & 0.71 & -0.5 & 0.71 &   1 & 0.25 & $2.8 \times 10^{5}$ &  56 &   0 & 0.72 & $1.3 \times 10^{8}$ & 0.85 & $2.4 \times 10^{8}$ & 0.94 & 0.98 \\ 
# # 9 &  -1 & 0.71 &   1 & 0.71 & -0.5 & 0.71 &   2 & 0.25 & $1.8 \times 10^{6}$ & $1.8 \times 10^{2}$ &   1 & 1.4 & $4.3 \times 10^{8}$ & 0.85 & $5.2 \times 10^{8}$ & 0.9 & 0.95 \\ 
# # \hline
# # \end{tabular}
# # \end{table}

# load("LACovFixedEffectsPrior_Final2.rda")
# out <- out[which(lengths(out)!=0)]
# param.combos <- param.combos[which(lengths(out)!=0),]
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Tue Apr 20 13:52:32 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & -0.5 & 0.71 &   1 & 0.71 & -0.5 & 0.71 & 0.1 & 0.01 & $9.3 \times 10^{4}$ &  35 &   0 & 0.022 & $3.1 \times 10^{7}$ & 0.99 & $3.1 \times 10^{7}$ & 0.94 & 0.98 \\ 
# # 2 & -0.5 & 0.71 &   1 & 0.71 & -0.5 & 0.71 & 0.5 & 0.01 & $1.6 \times 10^{5}$ &  22 &   0 & 0.11 & $4.9 \times 10^{7}$ & 0.95 & $5.1 \times 10^{7}$ & 0.95 & 0.98 \\ 
# # 3 & -0.5 & 0.71 &   1 & 0.71 & -0.5 & 0.71 &   1 & 0.01 & $2.1 \times 10^{5}$ &  18 &   0 & 0.22 & $4.4 \times 10^{7}$ & 0.95 & $3.8 \times 10^{7}$ & 0.94 & 0.97 \\ 
# # 4 & -0.5 & 0.71 &   1 & 0.71 & -0.5 & 0.71 & 0.1 & 0.1 & $3.9 \times 10^{5}$ &  44 &   0 & 0.043 & $8.6 \times 10^{7}$ & 0.98 & $8.6 \times 10^{7}$ & 0.93 & 0.97 \\ 
# # 5 & -0.5 & 0.71 &   1 & 0.71 & -0.5 & 0.71 & 0.5 & 0.1 & $1.5 \times 10^{5}$ &  15 &   0 & 0.22 & $3.8 \times 10^{7}$ & 0.91 & $4.8 \times 10^{7}$ & 0.95 & 0.98 \\ 
# # 6 & -0.5 & 0.71 &   1 & 0.71 & -0.5 & 0.71 &   1 & 0.1 & $1.1 \times 10^{6}$ &  29 &   0 & 0.43 & $2.1 \times 10^{8}$ & 0.93 & $2.3 \times 10^{8}$ & 0.94 & 0.98 \\ 
# # 7 & -0.5 & 0.71 &   1 & 0.71 & -0.5 & 0.71 & 0.1 & 0.25 & $5.2 \times 10^{4}$ &  36 &   2 & 0.072 & $7.9 \times 10^{6}$ & 0.99 & $7.9 \times 10^{6}$ & 0.93 & 0.98 \\ 
# # 8 & -0.5 & 0.71 &   1 & 0.71 & -0.5 & 0.71 & 0.5 & 0.25 & $7.9 \times 10^{5}$ &  37 &   0 & 0.36 & $3.8 \times 10^{8}$ & 0.97 & $4.3 \times 10^{8}$ & 0.94 & 0.98 \\ 
# # \hline
# # \end{tabular}
# # \end{table}



# 
# load("LACovFixedEffectsPrior_Final3.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Tue Apr 20 13:53:52 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 &  -1 & 0.32 &   1 & 0.32 & -0.5 & 0.32 & 0.1 & 0.01 &  50 &  12 &   0 & 0.022 & $8.6 \times 10^{3}$ & 0.98 & $8.7 \times 10^{3}$ & 0.97 & 0.99 \\ 
# # 2 &  -1 & 0.32 &   1 & 0.32 & -0.5 & 0.32 & 0.5 & 0.01 &  71 &  10 &   0 & 0.11 & $1.4 \times 10^{4}$ & 0.93 & $1.5 \times 10^{4}$ & 0.98 & 0.99 \\ 
# # 3 &  -1 & 0.32 &   1 & 0.32 & -0.5 & 0.32 & 0.75 & 0.01 &  71 &   8 &   0 & 0.16 & $1.2 \times 10^{4}$ & 0.9 & $1.4 \times 10^{4}$ & 0.97 & 0.99 \\ 
# # 4 &  -1 & 0.32 &   1 & 0.32 & -0.5 & 0.32 &   1 & 0.01 & $1 \times 10^{2}$ &  12 &   0 & 0.22 & $2.1 \times 10^{4}$ & 0.88 & $2.3 \times 10^{4}$ & 0.97 & 0.99 \\ 
# # 5 &  -1 & 0.32 &   1 & 0.32 & -0.5 & 0.32 & 0.1 & 0.1 &  80 & 7.5 &   0 & 0.043 & $1.4 \times 10^{4}$ & 0.97 & $1.5 \times 10^{4}$ & 0.98 & 0.99 \\ 
# # 6 &  -1 & 0.32 &   1 & 0.32 & -0.5 & 0.32 & 0.5 & 0.1 & $1.7 \times 10^{2}$ &   9 &   0 & 0.22 & $4 \times 10^{4}$ & 0.89 & $4.6 \times 10^{4}$ & 0.98 & 0.99 \\ 
# # 7 &  -1 & 0.32 &   1 & 0.32 & -0.5 & 0.32 & 0.75 & 0.1 & $3.7 \times 10^{2}$ &  14 &   0 & 0.33 & $8.4 \times 10^{4}$ & 0.84 & $9.8 \times 10^{4}$ & 0.97 & 0.99 \\ 
# # 8 &  -1 & 0.32 &   1 & 0.32 & -0.5 & 0.32 &   1 & 0.1 &  68 &  17 &   0 & 0.43 & $1 \times 10^{4}$ & 0.86 & $1.2 \times 10^{4}$ & 0.97 & 0.99 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final4.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Tue Apr 20 13:55:45 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & -0.5 & 0.32 &   1 & 0.32 & -0.5 & 0.32 & 0.5 & 0.01 &  82 &  20 &   0 & 0.11 & $1.5 \times 10^{4}$ & 0.95 & $1.6 \times 10^{4}$ & 0.96 & 0.99 \\ 
# # 2 & -0.5 & 0.32 &   1 & 0.32 & -0.5 & 0.32 & 0.75 & 0.01 & $1.3 \times 10^{2}$ &  17 &   0 & 0.16 & $2.5 \times 10^{4}$ & 0.94 & $2.6 \times 10^{4}$ & 0.96 & 0.99 \\ 
# # 3 & -0.5 & 0.32 &   1 & 0.32 & -0.5 & 0.32 &   1 & 0.01 & $1.2 \times 10^{2}$ &  17 &   0 & 0.22 & $2 \times 10^{4}$ & 0.93 & $2.1 \times 10^{4}$ & 0.96 & 0.99 \\ 
# # 4 & -0.5 & 0.32 &   1 & 0.32 & -0.5 & 0.32 &   2 & 0.01 & $2.3 \times 10^{2}$ &  20 &   0 & 0.43 & $6.4 \times 10^{4}$ & 0.89 & $7 \times 10^{4}$ & 0.96 & 0.99 \\ 
# # 5 & -0.5 & 0.32 &   1 & 0.32 & -0.5 & 0.32 & 0.5 & 0.1 & $1.3 \times 10^{2}$ &  14 &   0 & 0.22 & $2.7 \times 10^{4}$ & 0.9 & $3 \times 10^{4}$ & 0.96 & 0.99 \\ 
# # 6 & -0.5 & 0.32 &   1 & 0.32 & -0.5 & 0.32 & 0.75 & 0.1 & $3.1 \times 10^{2}$ &  14 &   0 & 0.33 & $8.2 \times 10^{4}$ & 0.91 & $9.3 \times 10^{4}$ & 0.96 & 0.99 \\ 
# # 7 & -0.5 & 0.32 &   1 & 0.32 & -0.5 & 0.32 &   1 & 0.1 & $7 \times 10^{2}$ &  24 &   0 & 0.43 & $1.7 \times 10^{5}$ & 0.91 & $2.3 \times 10^{5}$ & 0.95 & 0.98 \\ 
# # 8 & -0.5 & 0.32 &   1 & 0.32 & -0.5 & 0.32 &   2 & 0.1 & $2.5 \times 10^{2}$ &  37 &   0 & 0.87 & $6.2 \times 10^{4}$ & 0.9 & $6.2 \times 10^{4}$ & 0.93 & 0.97 \\ 
# # \hline
# # \end{tabular}
# # \end{table}

# 
# load("LACovFixedEffectsPrior_Final5.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Tue Apr 20 14:04:24 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & -0.5 & 0.32 & 1.5 & 0.32 & -0.5 & 0.32 & 0.5 & 0.01 & $4.2 \times 10^{3}$ & $2.9 \times 10^{2}$ &   0 & 0.11 & $1.2 \times 10^{6}$ & 0.95 & $1.3 \times 10^{6}$ & 0.93 & 0.97 \\ 
# # 2 & -0.5 & 0.32 & 1.5 & 0.32 & -0.5 & 0.32 & 0.75 & 0.01 & $6.7 \times 10^{3}$ & $3.3 \times 10^{2}$ &   0 & 0.16 & $1.8 \times 10^{6}$ & 0.95 & $1.9 \times 10^{6}$ & 0.94 & 0.98 \\ 
# # 3 & -0.5 & 0.32 & 1.5 & 0.32 & -0.5 & 0.32 &   1 & 0.01 & $6.5 \times 10^{3}$ & $1.4 \times 10^{2}$ &   0 & 0.22 & $1.8 \times 10^{6}$ & 0.92 & $2 \times 10^{6}$ & 0.93 & 0.97 \\ 
# # 4 & -0.5 & 0.32 & 1.5 & 0.32 & -0.5 & 0.32 &   2 & 0.01 & $2 \times 10^{4}$ & $6.2 \times 10^{2}$ &   0 & 0.43 & $7.8 \times 10^{6}$ & 0.87 & $7.9 \times 10^{6}$ & 0.93 & 0.97 \\ 
# # 5 & -0.5 & 0.32 & 1.5 & 0.32 & -0.5 & 0.32 & 0.5 & 0.1 & $7.3 \times 10^{3}$ & $1 \times 10^{2}$ &   0 & 0.22 & $2 \times 10^{6}$ & 0.95 & $2.2 \times 10^{6}$ & 0.93 & 0.98 \\ 
# # 6 & -0.5 & 0.32 & 1.5 & 0.32 & -0.5 & 0.32 & 0.75 & 0.1 & $1.6 \times 10^{4}$ & $1.2 \times 10^{2}$ &   0 & 0.33 & $3.9 \times 10^{6}$ & 0.92 & $4.3 \times 10^{6}$ & 0.93 & 0.97 \\ 
# # 7 & -0.5 & 0.32 & 1.5 & 0.32 & -0.5 & 0.32 &   1 & 0.1 & $3.8 \times 10^{4}$ & $3.9 \times 10^{2}$ &   0 & 0.43 & $8.6 \times 10^{6}$ & 0.92 & $9.5 \times 10^{6}$ & 0.92 & 0.97 \\ 
# # 8 & -0.5 & 0.32 & 1.5 & 0.32 & -0.5 & 0.32 &   2 & 0.1 & $8.6 \times 10^{3}$ & $7.1 \times 10^{2}$ &   0 & 0.87 & $3.1 \times 10^{6}$ & 0.87 & $3.6 \times 10^{6}$ & 0.9 & 0.96 \\ 
# # \hline
# # \end{tabular}
# # \end{table}



# load("LACovFixedEffectsPrior_Final6.rda")
# out <- out[which(lengths(out)!=0)]
# param.combos <- param.combos[which(lengths(out)!=0),]
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Tue Apr 20 14:06:48 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & -0.5 & 0.5 & 1.5 & 0.5 & -0.5 & 0.5 & 0.5 & 0.01 & $1.1 \times 10^{5}$ & $2.8 \times 10^{2}$ &   0 & 0.11 & $3.8 \times 10^{7}$ & 0.96 & $4.1 \times 10^{7}$ & 0.92 & 0.97 \\ 
# # 2 & -0.5 & 0.5 & 1.5 & 0.5 & -0.5 & 0.5 & 0.75 & 0.01 & $1.9 \times 10^{5}$ & $3.2 \times 10^{2}$ &   0 & 0.16 & $5.9 \times 10^{7}$ & 0.94 & $6.3 \times 10^{7}$ & 0.93 & 0.97 \\ 
# # 3 & -0.5 & 0.5 & 1.5 & 0.5 & -0.5 & 0.5 &   1 & 0.01 & $1.8 \times 10^{5}$ &  94 &   0 & 0.22 & $5.2 \times 10^{7}$ & 0.94 & $5.2 \times 10^{7}$ & 0.92 & 0.97 \\ 
# # 4 & -0.5 & 0.5 & 1.5 & 0.5 & -0.5 & 0.5 &   2 & 0.01 & $3.8 \times 10^{5}$ & $7.1 \times 10^{2}$ &   0 & 0.43 & $8.9 \times 10^{7}$ & 0.9 & $9.6 \times 10^{7}$ & 0.91 & 0.96 \\ 
# # 5 & -0.5 & 0.5 & 1.5 & 0.5 & -0.5 & 0.5 & 0.5 & 0.1 & $1.7 \times 10^{5}$ &  66 &   0 & 0.22 & $4.2 \times 10^{7}$ & 0.96 & $4.6 \times 10^{7}$ & 0.93 & 0.97 \\ 
# # 6 & -0.5 & 0.5 & 1.5 & 0.5 & -0.5 & 0.5 & 0.75 & 0.1 & $1.1 \times 10^{6}$ &  86 &   0 & 0.33 & $3.7 \times 10^{8}$ & 0.93 & $4.3 \times 10^{8}$ & 0.93 & 0.97 \\ 
# # 7 & -0.5 & 0.5 & 1.5 & 0.5 & -0.5 & 0.5 &   1 & 0.1 & $7.7 \times 10^{4}$ & $1.3 \times 10^{3}$ &   0 & 0.43 & $2.3 \times 10^{7}$ & 0.87 & $3.3 \times 10^{7}$ & 0.9 & 0.96 \\ 
# # \hline
# # \end{tabular}
# # \end{table}




# load("LACovFixedEffectsPrior_Final7.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Tue Apr 20 14:09:15 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & -0.25 & 0.32 &   1 & 0.32 & -0.25 & 0.32 &   1 & 0.01 & $2.8 \times 10^{2}$ &  36 &   0 & 0.22 & $4.9 \times 10^{4}$ & 0.94 & $5.3 \times 10^{4}$ & 0.94 & 0.98 \\ 
# # 2 & -0.25 & 0.32 &   1 & 0.32 & -0.25 & 0.32 &   2 & 0.01 & $3.8 \times 10^{2}$ &  36 &   0 & 0.43 & $7.1 \times 10^{4}$ & 0.94 & $7.3 \times 10^{4}$ & 0.94 & 0.98 \\ 
# # 3 & -0.25 & 0.32 &   1 & 0.32 & -0.25 & 0.32 &   3 & 0.01 & $4.3 \times 10^{2}$ &  38 &   0 & 0.65 & $8 \times 10^{4}$ & 0.93 & $8.8 \times 10^{4}$ & 0.93 & 0.97 \\ 
# # 4 & -0.25 & 0.32 &   1 & 0.32 & -0.25 & 0.32 &   1 & 0.1 & $4.8 \times 10^{2}$ &  66 &   0 & 0.43 & $7.3 \times 10^{4}$ & 0.92 & $7.3 \times 10^{4}$ & 0.94 & 0.98 \\ 
# # 5 & -0.25 & 0.32 &   1 & 0.32 & -0.25 & 0.32 &   2 & 0.1 & $8.7 \times 10^{4}$ &  37 &   0 & 0.87 & $1.4 \times 10^{7}$ & 0.97 & $1.7 \times 10^{7}$ & 0.92 & 0.97 \\ 
# # 6 & -0.25 & 0.32 &   1 & 0.32 & -0.25 & 0.32 &   3 & 0.1 & $6.3 \times 10^{5}$ &  39 &   1 & 1.3 & $2 \times 10^{8}$ & 0.88 & $3.8 \times 10^{9}$ & 0.89 & 0.95 \\ 
# # \hline
# # \end{tabular}
# # \end{table}




# load("LACovFixedEffectsPrior_Final8.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Tue Apr 20 14:11:27 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & -0.5 & 0.32 &   2 & 0.32 &  -2 & 0.32 & 0.5 & 0.01 & $1.7 \times 10^{3}$ & $1.7 \times 10^{2}$ &   0 & 0.11 & $6 \times 10^{5}$ & 0.97 & $6.3 \times 10^{5}$ & 0.93 & 0.98 \\ 
# # 2 & -0.5 & 0.32 &   2 & 0.32 &  -2 & 0.32 &   1 & 0.01 & $3.2 \times 10^{3}$ & $1.8 \times 10^{2}$ &   0 & 0.22 & $1.1 \times 10^{6}$ & 0.95 & $1.2 \times 10^{6}$ & 0.93 & 0.98 \\ 
# # 3 & -0.5 & 0.32 &   2 & 0.32 &  -2 & 0.32 &   2 & 0.01 & $2.6 \times 10^{3}$ &  92 &   0 & 0.43 & $9.1 \times 10^{5}$ & 0.94 & $1 \times 10^{6}$ & 0.92 & 0.97 \\ 
# # 4 & -0.5 & 0.32 &   2 & 0.32 &  -2 & 0.32 & 0.5 & 0.1 & $4.7 \times 10^{3}$ & $2.4 \times 10^{2}$ &   0 & 0.22 & $1.7 \times 10^{6}$ & 0.94 & $1.9 \times 10^{6}$ & 0.93 & 0.98 \\ 
# # 5 & -0.5 & 0.32 &   2 & 0.32 &  -2 & 0.32 &   1 & 0.1 & $1.1 \times 10^{4}$ &  86 &   0 & 0.43 & $2.7 \times 10^{6}$ & 0.97 & $3.3 \times 10^{6}$ & 0.93 & 0.97 \\ 
# # 6 & -0.5 & 0.32 &   2 & 0.32 &  -2 & 0.32 &   2 & 0.1 & $1.2 \times 10^{4}$ & $1.5 \times 10^{2}$ &   0 & 0.87 & $3.1 \times 10^{6}$ & 0.92 & $2.8 \times 10^{6}$ & 0.9 & 0.96 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final9.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Tue Apr 20 14:12:56 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero2436 & prop.zero4872 \\ 
# # \hline
# # 1 & 0.5 & 0.71 & 0.5 & 0.71 & -0.5 & 0.71 & 0.5 & 0.01 & $4.2 \times 10^{3}$ &  35 &   0 & 0.11 & $1.3 \times 10^{6}$ &   1 & $1.3 \times 10^{6}$ & 0.9 & 0.97 \\ 
# # 2 & 0.5 & 0.71 & 0.5 & 0.71 & -0.5 & 0.71 &   1 & 0.01 & $6.9 \times 10^{3}$ &  32 &   0 & 0.22 & $1.4 \times 10^{6}$ & 1.1 & $1.5 \times 10^{6}$ & 0.92 & 0.97 \\ 
# # 3 & 0.5 & 0.71 & 0.5 & 0.71 & -0.5 & 0.71 &   2 & 0.01 & $6.5 \times 10^{3}$ &  52 &   0 & 0.43 & $1.1 \times 10^{6}$ & 1.2 & $1 \times 10^{6}$ & 0.88 & 0.96 \\ 
# # 4 & 0.5 & 0.71 & 0.5 & 0.71 & -0.5 & 0.71 & 0.5 & 0.1 & $1.8 \times 10^{4}$ &  46 &   0 & 0.22 & $2.7 \times 10^{6}$ & 1.1 & $2.3 \times 10^{6}$ & 0.9 & 0.97 \\ 
# # 5 & 0.5 & 0.71 & 0.5 & 0.71 & -0.5 & 0.71 &   1 & 0.1 & $1.5 \times 10^{4}$ &  35 &   0 & 0.43 & $4 \times 10^{6}$ & 1.2 & $4.3 \times 10^{6}$ & 0.9 & 0.96 \\ 
# # 6 & 0.5 & 0.71 & 0.5 & 0.71 & -0.5 & 0.71 &   2 & 0.1 & $1.7 \times 10^{5}$ &  36 &   0 & 0.87 & $5.6 \times 10^{7}$ & 1.2 & $4.5 \times 10^{7}$ & 0.87 & 0.95 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# load("LACovFixedEffectsPrior_Final10.rda")
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero2436 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2436,na.rm=TRUE)})
# df$prop.zero4872 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad4872,na.rm=TRUE)})
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g","g")), math.style.exponents = TRUE)


# sessionInfo() -----------------------------------------------------------

sessionInfo()