
# Testing Fixed Theta Values: Irregular Polygon ---------------------------

# This R script takes the set-up for the Grid-Mesh Optimisation over the Los Angeles polygon with the goal of assessing the behaviour of the number of points simulated for a fixed set of values so that when we run the tradition simulation study for these values we get a reasonable range of point patterns simulated.

# Author: Nadeen Khaleel


# Setwd and Load Libraries ------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))

library(spatstat)
library(raster)
library(sf)
library(sp)
library(maptools)
library(rgdal)
library(lwgeom)
library(RandomFields)


# Set-up ------------------------------------------------------------------
# We will set up the window and covariates and then create a small function that will take these and user-provided parameter values to simulate lgcps over the window with the intention of testing the number of point simulated and the behaviour of the field.

load("../../../../DATA/PROCESSED_DATA/SHAPEFILES/CENSUS_TRACTS/LACityCT.rda")
lacity_boundary <- st_union(ct_LA)
lacity_proj <- lwgeom::st_transform_proj(lacity_boundary,"epsg:32611")
lacity_sp <- as(lacity_proj,"Spatial")

# Transform the window
bbox <- lacity_sp@bbox
lacity_spshift <- elide(lacity_sp,shift=-c(bbox[1,1],bbox[2,1]))
bbox_shift <- lacity_spshift@bbox
lacity_spscale <- elide(lacity_spshift,scale=max(bbox_shift)/1e4)
W <- as.owin.SpatialPolygons((lacity_spscale))

# Get covariates - these are the same as those for the main Grid-Mesh Optimisation, but moved over into this directory.
cov.name <- paste0("GridMeshIrregPolLGCPSSCov.rda")
load(cov.name)
i.im <- intb.im; c1.im <- popb.im; c2.im <- incb.im


# Function to simulate lgcps using the above covariates and window for a given set of parameter values, theta.
n.theta <- function(theta,n=10,i.im,c1.im,c2.im){
  beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]]; sigma <- theta[[4]]; rho <- theta[[5]];
  
  mu <- beta.0*i.im + beta.1*c1.im + beta.2*c2.im
  
  n.vec <- rep(NA,n)
  for (i in 1:n){
    lgcp.ppp <- try(rLGCP(model="matern",mu,var=(sigma)^2,scale=rho/2,nu=1,win = W))
    n.vec[i] <- lgcp.ppp$n
  }
  
  return(list("n"=n.vec,"mu"=mu$v))
}


# Testing Values ----------------------------------------------------------
# We will now use the set-up above to test different values for the parameters, checking the behaviour of the number of points simulated.
# Note: Final parameter choice is theta34i

theta1 <- data.frame(beta0=-1,beta1=1,beta2=-1,sigma=1,rho=0.75)
theta.n1 <- n.theta(theta=theta1,n=5e2,i.im,c1.im,c2.im)
quantile(theta.n1$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%   10%   25%   50%   75%   90%   99% 
# 3.00  6.00  8.00 11.00 14.00 17.00 25.02 
quantile(theta.n1$mu)
# 0%        25%        50%        75%       100% 
# -9.3227230 -0.8490983  0.0000000  0.0000000  5.3592188 

theta2 <- data.frame(beta0=-1,beta1=1,beta2=-0.5,sigma=1,rho=0.75)
theta.n2 <- n.theta(theta=theta2,n=5e2,i.im,c1.im,c2.im)
quantile(theta.n2$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1% 10% 25% 50% 75% 90% 99% 
# 5   8  11  15  20  25  36
quantile(theta.n2$mu)   
# 0%       25%       50%       75%      100% 
# -4.259101 -1.008911  0.000000  0.000000  6.723051 


theta3 <- data.frame(beta0=-0.5,beta1=1,beta2=-0.5,sigma=1,rho=0.75)
theta.n3 <- n.theta(theta=theta3,n=5e2,i.im,c1.im,c2.im)
quantile(theta.n3$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%   10%   25%   50%   75%   90%   99% 
# 9.00 15.00 19.00 24.00 31.00 41.00 55.05 
quantile(theta.n3$mu)
# 0%        25%        50%        75%       100% 
# -3.7591008 -0.5089114  0.0000000  0.0000000  7.2230512 

theta4 <- data.frame(beta0=-0.5,beta1=1.5,beta2=-0.5,sigma=1,rho=0.75)
theta.n4 <- n.theta(theta=theta4,n=5e2,i.im,c1.im,c2.im)
quantile(theta.n4$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 99.96  176.90  250.00  380.50  552.50  765.20 1537.86 
quantile(theta.n4$mu)
# 0%       25%       50%       75%      100% 
# -3.647087 -0.491809  0.000000  0.000000 11.898712 


theta4 <- data.frame(beta0=-0.5,beta1=1.5,beta2=-0.5,sigma=1,rho=0.75)
theta.n4 <- n.theta(theta=theta4,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n4$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 108.98  173.00  234.00  356.00  544.00  775.30 1622.12 
quantile(theta.n4$mu)
# 0%       25%       50%       75%      100% 
# -3.647087 -0.491809  0.000000  0.000000 11.898712 


theta5 <- data.frame(beta0=-0.5,beta1=1.5,beta2=-0.5,sigma=0.75,rho=0.75)
theta.n5 <- n.theta(theta=theta5,n=5e2,i.im,c1.im,c2.im)
quantile(theta.n5$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 131.88 182.00 243.50 316.50 430.25 579.20 922.46 
quantile(theta.n5$mu)
# 0%       25%       50%       75%      100% 
# -3.647087 -0.491809  0.000000  0.000000 11.898712 

theta5 <- data.frame(beta0=-0.5,beta1=1.5,beta2=-0.5,sigma=0.75,rho=0.75)
theta.n5 <- n.theta(theta=theta5,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n5$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 114.99 184.90 244.75 327.00 449.25 592.10 980.09 
quantile(theta.n5$mu)
# 0%       25%       50%       75%      100% 
# -3.647087 -0.491809  0.000000  0.000000 11.898712 



theta6 <- data.frame(beta0=-5,beta1=2,beta2=-0.5,sigma=1,rho=0.75)
theta.n6 <- n.theta(theta=theta6,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n6$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 50.96   94.00  134.75  212.00  315.00  482.30 1149.92 
quantile(theta.n6$mu)
# 0%       25%       50%       75%      100% 
# -8.150571 -4.947123  0.000000  0.000000 12.712706 


theta7 <- data.frame(beta0=-5,beta1=2,beta2=-0.25,sigma=1,rho=0.75)
theta.n7 <- n.theta(theta=theta7,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n7$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 147.99  249.80  354.00  533.50  836.75 1214.70 3174.31 
quantile(theta.n7$mu)
# 0%       25%       50%       75%      100% 
# -6.663395 -4.872472  0.000000  0.000000 14.484341 

theta8 <- data.frame(beta0=-5,beta1=2,beta2=-0.5,sigma=0.75,rho=0.75)
theta.n8 <- n.theta(theta=theta8,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n8$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 61.00 100.00 131.00 177.00 250.25 346.00 585.36 
quantile(theta.n8$mu)

theta9 <- data.frame(beta0=-1,beta1=1,beta2=-0.5,sigma=0.75,rho=0.75)
theta.n9 <- n.theta(theta=theta9,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n9$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
#   1%   10%   25%   50%   75%   90%   99% 
# 4.00  7.00  9.00 12.00 16.00 19.00 25.02 
quantile(theta.n9$mu)


theta10 <- data.frame(beta0=-0.5,beta1=1,beta2=-0.5,sigma=3,rho=0.75)
theta.n10 <- n.theta(theta=theta10,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n10$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%      10%      25%      50%      75%      90%      99% 
# 56.96   135.90   238.75   483.00   986.00  2170.20 10632.68 
quantile(theta.n10$mu)

theta11 <- data.frame(beta0=-0.5,beta1=0.5,beta2=-0.5,sigma=3,rho=0.75)
theta.n11 <- n.theta(theta=theta11,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n11$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 41.97  102.90  172.75  329.00  628.00 1299.50 6688.85 
quantile(theta.n11$mu)

theta12 <- data.frame(beta0=-0.25,beta1=0.5,beta2=-0.25,sigma=3,rho=0.75)
theta.n12 <- n.theta(theta=theta12,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n12$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%      10%      25%      50%      75%      90%      99% 
# 63.99   134.00   225.75   450.50   899.50  1816.00 10580.75 
quantile(theta.n12$mu)


theta13 <- data.frame(beta0=-0.25,beta1=0.5,beta2=-0.25,sigma=2,rho=0.75)
theta.n13 <- n.theta(theta=theta13,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n13$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 18.99  31.90  45.00  65.00  94.25 138.00 309.01 
quantile(theta.n13$mu)


theta14 <- data.frame(beta0=-0.5,beta1=0.5,beta2=-0.5,sigma=2,rho=0.75)
theta.n14 <- n.theta(theta=theta14,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n14$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 12.00  24.00  34.00  47.00  70.00 102.10 214.01 
quantile(theta.n14$mu)


theta15 <- data.frame(beta0=-0.25,beta1=0.5,beta2=-0.5,sigma=2,rho=0.75)
theta.n15 <- n.theta(theta=theta15,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n15$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 18.00  31.90  44.00  67.00 100.00 141.10 312.16 
quantile(theta.n15$mu)



theta16 <- data.frame(beta0=-0.25,beta1=0.5,beta2=-0.25,sigma=2,rho=0.75)
theta.n16 <- n.theta(theta=theta16,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n16$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 17.00  30.00  44.00  63.00  96.00 145.10 322.05 
quantile(theta.n16$mu)


theta17 <- data.frame(beta0=-0.25,beta1=0.75,beta2=-0.25,sigma=2,rho=0.75)
theta.n17 <- n.theta(theta=theta17,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n17$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 18.99  36.00  53.00  79.00 119.00 189.10 564.02 
quantile(theta.n17$mu)


theta18 <- data.frame(beta0=-0.25,beta1=1,beta2=-0.25,sigma=2,rho=0.75)
theta.n18 <- n.theta(theta=theta18,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n18$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 28.00   58.90   85.00  133.00  240.00  427.90 1562.26 
quantile(theta.n18$mu)



theta19 <- data.frame(beta0=-0.25,beta1=2,beta2=-2,sigma=1,rho=0.75)
theta.n19 <- n.theta(theta=theta19,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n19$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 79.97  125.00  173.00  261.50  395.00  579.10 1150.17 
quantile(theta.n19$mu)
# 0%       25%       50%       75%      100% 
# -16.89545   0.00000   0.00000   0.00000  12.46844 



theta20 <- data.frame(beta0=-0.25,beta1=2,beta2=-1.5,sigma=1,rho=0.75)
theta.n20 <- n.theta(theta=theta20,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n20$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 247.87  429.80  608.00  951.00 1507.25 2398.70 4125.73 
quantile(theta.n20$mu)


theta21 <- data.frame(beta0=-0.25,beta1=1.5,beta2=-1.5,sigma=1,rho=0.75)
theta.n21 <- n.theta(theta=theta21,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n21$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 17.00  28.00  35.00  46.00  63.00  83.10 132.02 
quantile(theta.n21$mu)

theta22 <- data.frame(beta0=-0.25,beta1=1.5,beta2=-1.5,sigma=2,rho=0.75)
theta.n22 <- n.theta(theta=theta22,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n22$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 34.99   65.90   92.00  138.00  232.25  390.10 1258.27 
quantile(theta.n22$mu)



theta23 <- data.frame(beta0=-1,beta1=1,beta2=-1,sigma=3,rho=0.75)
theta.n23 <- n.theta(theta=theta23,n=5e2,i.im,c1.im,c2.im)
quantile(theta.n23$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 25.97   73.90  137.00  278.50  577.00 1047.80 4723.57 
quantile(theta.n23$mu)
# 0%        25%        50%        75%       100% 
# -9.3227230 -0.8490983  0.0000000  0.0000000  5.3592188 


theta24 <- data.frame(beta0=-1,beta1=0.5,beta2=-2,sigma=1,rho=0.75)
theta.n24 <- n.theta(theta=theta24,n=5e2,i.im,c1.im,c2.im)
quantile(theta.n24$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1% 10% 25% 50% 75% 90% 99% 
# 5   9  11  14  18  22  32 
quantile(theta.n24$mu)

theta25 <- data.frame(beta0=-1,beta1=0.5,beta2=-2,sigma=2,rho=0.75)
theta.n25 <- n.theta(theta=theta25,n=5e2,i.im,c1.im,c2.im)
quantile(theta.n25$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 14.99  27.00  36.00  53.50  80.00 116.10 315.21 
quantile(theta.n25$mu)

theta26 <- data.frame(beta0=-1,beta1=1,beta2=-2,sigma=1,rho=0.75)
theta.n26 <- n.theta(theta=theta26,n=5e2,i.im,c1.im,c2.im)
quantile(theta.n26$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1% 10% 25% 50% 75% 90% 99% 
# 4   7   9  12  16  19  25 
quantile(theta.n26$mu)


theta27 <- data.frame(beta0=-0.5,beta1=2,beta2=-2,sigma=1,rho=0.75)
theta.n27 <- n.theta(theta=theta27,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n27$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 50.99  96.00 136.00 204.00 315.00 475.00 847.37 
quantile(theta.n27$mu)
# 0%         25%         50%         75%        100% 
# -17.1454461  -0.1981967   0.0000000   0.0000000  12.2184376 



theta28 <- data.frame(beta0=0.5,beta1=2,beta2=-2,sigma=1,rho=0.75)
theta.n28 <- n.theta(theta=theta28,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n28$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 162.98  268.00  377.00  552.00  800.75 1154.50 2299.10 
quantile(theta.n28$mu)
# 0%         25%         50%         75%        100% 
# -16.1454461   0.0000000   0.0000000   0.2689768  13.2184376 


theta29 <- data.frame(beta0=0.5,beta1=1.5,beta2=-1,sigma=1,rho=0.75)
theta.n29 <- n.theta(theta=theta29,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n29$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 90.97 129.00 170.75 240.00 343.00 502.00 982.10 
quantile(theta.n29$mu)
# 0%        25%        50%        75%       100% 
# -6.9204624  0.0000000  0.0000000  0.1496954 11.4026606  



theta30 <- data.frame(beta0=1,beta1=1.5,beta2=-1,sigma=1,rho=0.75)
theta.n30 <- n.theta(theta=theta30,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n30$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 136.00  224.00  283.00  387.50  555.25  781.60 1419.39 
quantile(theta.n30$mu)
# 0%        25%        50%        75%       100% 
# -6.4204624  0.0000000  0.0000000  0.6496954 11.9026606 


theta31 <- data.frame(beta0=-0.1,beta1=1,beta2=-1,sigma=1,rho=2)
theta.n31 <- n.theta(theta=theta31,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n31$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%   10%   25%   50%   75%   90%   99% 
# 6.00 12.00 16.00 23.00 33.00 44.00 75.03 
quantile(theta.n31$mu)
# 0%       25%       50%       75%      100% 
# -8.422723  0.000000  0.000000  0.000000  6.259219 

theta32 <- data.frame(beta0=-0.5,beta1=1.5,beta2=-1,sigma=1,rho=2)
theta.n32 <- n.theta(theta=theta32,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n32$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 20.00  38.00  53.00  85.00 126.00 186.10 373.02 
quantile(theta.n32$mu)
# 0%        25%        50%        75%       100% 
# -7.9204624 -0.5168778  0.0000000  0.0000000 10.4026606 


theta33 <- data.frame(beta0=-0.5,beta1=1.5,beta2=-0.5,sigma=1,rho=2)
theta.n33 <- n.theta(theta=theta33,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n33$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 69.99  139.90  212.50  333.50  528.25  844.30 1904.17 
quantile(theta.n33$mu)
# 0%       25%       50%       75%      100% 
# -3.647087 -0.491809  0.000000  0.000000 11.898712 





theta34 <- data.frame(beta0=1,beta1=1.5,beta2=-1,sigma=1,rho=2)
theta.n34 <- n.theta(theta=theta34,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n34$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 86.00  168.00  236.00  359.00  579.00  877.10 1882.43 
quantile(theta.n34$mu)
# 0%        25%        50%        75%       100% 
# -6.4204624  0.0000000  0.0000000  0.6496954 11.9026606 



theta34a <- data.frame(beta0=1.5,beta1=1.5,beta2=-1,sigma=1,rho=2)
theta.n34a <- n.theta(theta=theta34a,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n34a$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 140.00  283.00  425.00  627.50  997.50 1522.90 2959.77 
quantile(theta.n34a$mu)
# 0%       25%       50%       75%      100% 
# -5.920462  0.000000  0.000000  1.149695 12.402661


theta34b <- data.frame(beta0=1.5,beta1=1.5,beta2=-1,sigma=2,rho=2) 
theta.n34b <- n.theta(theta=theta34b,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n34b$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
quantile(theta.n34b$mu)
# 0%       25%       50%       75%      100% 
# -5.920462  0.000000  0.000000  1.149695 12.402661 


theta34c <- data.frame(beta0=1.5,beta1=1.25,beta2=-1,sigma=1,rho=2)
theta.n34c <- n.theta(theta=theta34c,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n34c$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 56.96  97.00 133.00 193.00 278.00 400.20 754.22 
quantile(theta.n34c$mu)
# 0%       25%       50%       75%      100% 
# -6.371593  0.000000  0.000000  1.285466 10.130940 


theta34d <- data.frame(beta0=2,beta1=1.25,beta2=-1,sigma=1,rho=2)
theta.n34d <- n.theta(theta=theta34d,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n34d$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 96.93  161.90  216.75  321.00  461.00  662.20 1242.88 
quantile(theta.n34d$mu)
# 0%       25%       50%       75%      100% 
# -5.871593  0.000000  0.000000  1.785466 10.630940 


theta34e <- data.frame(beta0=2,beta1=1.25,beta2=-0.75,sigma=1,rho=2)
theta.n34e <- n.theta(theta=theta34e,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n34e$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 121.00  219.90  303.00  464.50  709.25 1043.90 2021.22 
quantile(theta.n34e$mu)
# 0%       25%       50%       75%      100% 
# -3.339782  0.000000  0.000000  1.643165 11.312856 


theta34f <- data.frame(beta0=2,beta1=1,beta2=-1,sigma=1,rho=2)
theta.n34f <- n.theta(theta=theta34f,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n34f$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
#     1%    10%    25%    50%    75%    90%    99% 
# 59.00 106.00 143.00 195.50 270.00 373.10 577.05 
quantile(theta.n34f$mu)
# 0%       25%       50%       75%      100% 
# -6.322723  0.000000  0.000000  1.884488  8.359219 
# 


theta34g <- data.frame(beta0=2.5,beta1=1,beta2=-0.5,sigma=1,rho=2)
theta.n34g <- n.theta(theta=theta34g,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n34g$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 140.99  227.00  302.00  435.00  618.25  895.60 1642.01 
# 1%     10%     25%     50%     75%     90%     99% 
# 119.98  211.90  288.75  428.50  622.25  895.40 1450.89 
quantile(theta.n34g$mu)
# 0%        25%        50%        75%       100% 
# -0.7591008  0.0000000  0.0000000  2.1389767 10.2230512 

theta34h <- data.frame(beta0=2.5,beta1=1,beta2=-1,sigma=1,rho=2)
theta.n34h <- n.theta(theta=theta34h,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n34h$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 96.00  169.00  224.75  314.50  450.25  609.10 1038.00 
quantile(theta.n34h$mu)
# 0%       25%       50%       75%      100% 
# -5.822723  0.000000  0.000000  2.384488  8.859219 


theta34i <- data.frame(beta0=3,beta1=0.75,beta2=-0.5,sigma=1,rho=2) # Take these values ****
theta.n34i <- n.theta(theta=theta34i,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n34i$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 146.95  238.00  321.75  456.50  636.00  851.20 1477.17 
# 1%     10%     25%     50%     75%     90%     99% 
# 152.91  245.90  333.00  451.00  610.25  807.20 1501.20 
# 1%     10%     25%     50%     75%     90%     99% 
# 157.98  248.90  331.00  457.00  639.00  841.30 1332.19 
quantile(theta.n34i$mu)
# 0%        25%        50%        75%       100% 
# -0.7102312  0.0000000  0.0000000  2.8248477  8.4513303 


theta35 <- data.frame(beta0=1.5,beta1=1,beta2=-1,sigma=1,rho=2)
theta.n35 <- n.theta(theta=theta35,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n35$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 31.00  59.00  81.00 114.00 162.00 219.00 361.04 
quantile(theta.n35$mu)
# 0%       25%       50%       75%      100% 
# -6.822723  0.000000  0.000000  1.384488  7.859219 

theta36 <- data.frame(beta0=1.5,beta1=1,beta2=-1,sigma=1.5,rho=2)
theta.n36 <- n.theta(theta=theta36,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n36$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 32.99  73.00 113.75 178.00 295.00 464.00 954.27 
quantile(theta.n36$mu)
# 0%       25%       50%       75%      100% 
# -6.822723  0.000000  0.000000  1.384488  7.859219 


theta37 <- data.frame(beta0=1.5,beta1=1.5,beta2=-1,sigma=0.75,rho=2) 
theta.n37 <- n.theta(theta=theta37,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n37$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 193.99  298.90  400.50  550.00  742.00 1013.50 1716.76 
quantile(theta.n37$mu)
# 0%       25%       50%       75%      100% 
# -5.920462  0.000000  0.000000  1.149695 12.402661 




theta38 <- data.frame(beta0=1.5,beta1=1.5,beta2=-1,sigma=0.75,rho=1)
theta.n38 <- n.theta(theta=theta38,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n38$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 238.97  335.80  424.00  559.50  755.50  993.00 1653.04 
quantile(theta.n38$mu)
# 0%       25%       50%       75%      100% 
# -5.920462  0.000000  0.000000  1.149695 12.402661 



theta39 <- data.frame(beta0=-0.5,beta1=1,beta2=-0.5,sigma=2,rho=1)
theta.n39 <- n.theta(theta=theta39,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n39$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 12.99  29.00  43.00  72.00 124.00 209.20 651.05
quantile(theta.n39$mu)
# 0%        25%        50%        75%       100% 
# -3.7591008 -0.5089114  0.0000000  0.0000000  7.2230512


theta40 <- data.frame(beta0=1.5,beta1=1,beta2=-1,sigma=1.5,rho=1)
theta.n40 <- n.theta(theta=theta40,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n40$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 65.99 112.00 150.75 210.50 297.25 409.10 758.03 
quantile(theta.n40$mu)
# 0%       25%       50%       75%      100% 
# -6.822723  0.000000  0.000000  1.384488  7.859219 


theta41 <- data.frame(beta0=-0.5,beta1=1,beta2=-0.5,sigma=2,rho=2)
theta.n41 <- n.theta(theta=theta41,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n41$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 5.00  15.00  27.00  54.00 116.00 255.20 918.04 
quantile(theta.n41$mu)
# 0%        25%        50%        75%       100% 
# -3.7591008 -0.5089114  0.0000000  0.0000000  7.2230512 


theta42 <- data.frame(beta0=-0.1,beta1=1,beta2=-0.1,sigma=2,rho=2)
theta.n42 <- n.theta(theta=theta42,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n42$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%    10%    25%    50%    75%    90%    99% 
# 9.0   34.0   64.0  156.5  360.0  796.3 3882.9 
quantile(theta.n42$mu)
# 0%         25%         50%         75%        100% 
# -0.95014374 -0.01792859  0.00000000  0.00000000  9.81933393 




theta43 <- data.frame(beta0=-1,beta1=1,beta2=-0.1,sigma=2,rho=2)
theta.n43 <- n.theta(theta=theta43,n=1e3,i.im,c1.im,c2.im)
quantile(theta.n43$n,c(0.01,0.1,0.25,0.5,0.75,0.9,0.99))
# 1%     10%     25%     50%     75%     90%     99% 
# 4.00   14.00   27.00   60.00  130.25  282.10 1528.16 
quantile(theta.n43$mu)
# 0%        25%        50%        75%       100% 
# -1.8501437 -0.9179286  0.0000000  0.0000000  8.9193339


# sessionInfo() -----------------------------------------------------------

sessionInfo()