
# Traditional Simulation Study Reg Pol Output -----------------------------

# This R script will take in the output from the traditional simulation study to output plots and a summary table for the timings and parameter recovery for the LGCP example on the regular polygon for Chapter 3, where there are four grid resolutions and four mesh resolutions to consider.

# Author: Nadeen Khaleel


# Libraries ---------------------------------------------------------------

library(purrr)
library(ggplot2)
library(rlist)
library(magrittr)
library(grid)
library(gridExtra)
library(stringr)
library(xtable)
library(tidyverse)
library(scales)


# Functions ---------------------------------------------------------------

time.meansd.mesh <- function(grid.dat){
  N <- length(grid.dat[[1]]$run.df$time)
  meshes <- names(grid.dat); N.m <- length(meshes)
  means <- sapply(1:N.m,function(i){mean(grid.dat[[i]]$run.df$time)})
  sds <- sapply(1:N.m,function(i){sd(grid.dat[[i]]$run.df$time)})
  q <- qnorm(0.975)
  lower.c <- (as.vector(means) - q*as.vector(sds)/sqrt(N)); upper.c <- (as.vector(means) + q*as.vector(sds)/sqrt(N))
  meshes.s <- str_extract(meshes, "\\d+\\.*\\d*")
  mesh.lab <- paste0("Mesh ",meshes.s)
  out.time.df <- data.frame(Mesh=mesh.lab,mean=as.vector(means),sd=as.vector(sds),ci.l=lower.c,ci.u=upper.c)
  out.time.df$Mesh <- factor(as.character(out.time.df$Mesh),levels=unique(out.time.df$Mesh)[order(unique(out.time.df$Mesh),decreasing=TRUE)])
  return(out.time.df)
}

param.meansd.mesh <- function(grid.dat){
  N <- nrow(grid.dat[[1]]$est.df)
  meshes <- names(grid.dat); m <- as.factor(meshes); levels(m) <- meshes; N.m <- length(meshes)
  means <- sapply(1:N.m,function(i){colMeans(grid.dat[[i]]$est.df)})
  sds <- sapply(1:N.m,function(i){sapply(grid.dat[[i]]$est.df,sd,2)})
  params <- rownames(means); N.p <- length(params)
  q <- qnorm(0.975)
  lower.c <- (as.vector(means) - q*as.vector(sds)/sqrt(N)); upper.c <- (as.vector(means) + q*as.vector(sds)/sqrt(N))
  meshes.s <- str_extract(meshes, "\\d+\\.*\\d*")
  mesh.lab <- paste0("Mesh ",meshes.s)
  out.est.df <- data.frame(Mesh=rep(mesh.lab,each=N.p),Param=rep(params,N.m),mean=as.vector(means),sd=as.vector(sds),ci.l=lower.c,ci.u=upper.c)
  out.est.df$Mesh <- factor(as.character(out.est.df$Mesh),levels=unique(out.est.df$Mesh)[order(unique(out.est.df$Mesh),decreasing=TRUE)])
  out.est.df$Param <- factor(as.character(out.est.df$Param),levels=unique(out.est.df$Param))
  
  # Setting up labels for the plots
  var.lab.orig <- as.character(out.est.df$Param)
  num <- as.numeric(str_extract(var.lab.orig, "[0-9]+"))
  char <- str_extract(var.lab.orig, "[aA-zZ]+")
  
  plot.x.lab <- char
  plot.x.lab
  for (i in which(!is.na(num))){
    plot.x.lab[i] <- paste0(plot.x.lab[i],"[",num[i],"]")
  }
  plot.x.lab
  
  if (sum(plot.x.lab=="Int")>0){
    plot.x.lab[plot.x.lab=="Int"] <- "beta[0]"
  }
  out.est.df$Label <- plot.x.lab
  
  return(out.est.df)
}

combined.meanquantsd.mesh <- function(grid.dat,param){
  N <- length(grid.dat[[1]]$run.df$time)
  grid.dat.sub <- lapply(1:length(grid.dat),function(i){grid.dat[[i]]$est.df[,param]})
  names(grid.dat.sub) <- names(grid.dat)
  meshes <- names(grid.dat); m <- as.factor(meshes); levels(m) <- meshes; N.m <- length(meshes)
  means <- sapply(1:N.m,function(i){colMeans(grid.dat.sub[[i]])})
  means.t <- sapply(1:N.m,function(i){mean(grid.dat[[i]]$run.df$time)})
  means.full <- rbind(means.t,means)
  rownames(means.full)[1] <- "time"
  sel.25 <- sapply(1:N.m,function(i){sapply(1:length(param),function(j,g){sort(g[,j])[25]},grid.dat.sub[[i]])})
  colnames(sel.25) <- names(grid.dat.sub)
  rownames(sel.25) <- param
  sel.25t <- rep(NA,length(N.m))
  sel.25.full <- rbind(sel.25t,sel.25)
  rownames(sel.25.full)[1] <- "time"
  sel.975 <- sapply(1:N.m,function(i){sapply(1:length(param),function(j,g){sort(g[,j])[975]},grid.dat.sub[[i]])})
  colnames(sel.975) <- names(grid.dat.sub)
  rownames(sel.975) <- param
  sel.975t <- rep(NA,length(N.m))
  sel.975.full <- rbind(sel.975t,sel.975)
  sds <- sapply(1:N.m,function(i){sapply(grid.dat.sub[[i]],sd,2)})
  sds.t <- sapply(1:N.m,function(i){sd(grid.dat[[i]]$run.df$time)})
  sds.full <- rbind(sds.t,sds)
  rownames(sds.full)[1] <- "time"
  params <- rownames(means.full); N.p <- length(params) # Now includes time!!! - so technically N.p + 1
  q <- qnorm(0.975)
  lower.c <- (as.vector(means.full) - q*as.vector(sds.full)); upper.c <- (as.vector(means.full) + q*as.vector(sds.full))
  meshes.s <- str_extract(meshes, "\\d+\\.*\\d*")
  mesh.lab <- paste0("Mesh ",meshes.s)
  out.df <- data.frame(Mesh=rep(mesh.lab,each=N.p),Param=rep(params,N.m),mean=as.vector(means.full),q0.025=as.vector(sel.25.full),q0.975=as.vector(sel.975.full),sd=as.vector(sds.full),ci.l=lower.c,ci.u=upper.c)
  out.df$Mesh <- factor(as.character(out.df$Mesh),levels=unique(out.df$Mesh)[order(unique(out.df$Mesh),decreasing=TRUE)])
  out.df$Param <- factor(as.character(out.df$Param),levels=unique(out.df$Param))
  
  # Setting up labels for the plots
  var.lab.orig <- as.character(out.df$Param)
  num <- as.numeric(str_extract(var.lab.orig, "[0-9]+"))
  char <- str_extract(var.lab.orig, "[aA-zZ]+")
  
  plot.x.lab <- char
  plot.x.lab
  for (i in which(!is.na(num))){
    plot.x.lab[i] <- paste0(plot.x.lab[i],"[",num[i],"]")
  }
  plot.x.lab
  
  if (sum(plot.x.lab=="Int")>0){
    plot.x.lab[plot.x.lab=="Int"] <- "beta[0]"
  }
  out.df$Label <- plot.x.lab
  
  return(out.df)
}

# Coverage of the 95% Credible Intervals
coverageparam.grid <- function(grid.dat,param,true.theta){
  param.cil <- paste0(param,".cil")
  param.ciu <- paste0(param,".ciu")
  grid.dat.sub <- lapply(1:length(grid.dat),function(i){grid.dat[[i]]$est.df[,c(param,param.cil,param.ciu)]})
  names(grid.dat.sub) <- names(grid.dat)

  cov <- sapply(1:length(grid.dat.sub),function(i,true.theta,grid.dat.sub){sapply(1:length(param),function(j,true.theta,data){sum(data[,param.cil[j]] <=true.theta[j] & true.theta[j] <= data[,param.ciu[j]])},true.theta,grid.dat.sub[[i]])},true.theta,grid.dat.sub)
  colnames(cov) <- names(grid.dat.sub)
  rownames(cov) <- param
  cov <- cov/length(grid.dat.sub[[1]][[1]])
  cov.df <- as.data.frame(cov)
  cov.df <- tibble::rownames_to_column(cov.df,var="Parameter")
  cov.df <- cov.df %>% gather(Mesh,Coverage,Mesh0.5:Mesh0.1)
  cov.df$Parameter <- as.factor(cov.df$Parameter)
  cov.df$Mesh <- as.factor(cov.df$Mesh)
  
  # Setting up labels for the plots
  var.lab.orig <- as.character(cov.df$Parameter)
  num <- as.numeric(str_extract(var.lab.orig, "[0-9]+"))
  char <- str_extract(var.lab.orig, "[aA-zZ]+")
  
  plot.x.lab <- char
  plot.x.lab
  for (i in which(!is.na(num))){
    plot.x.lab[i] <- paste0(plot.x.lab[i],"[",num[i],"]")
  }
  plot.x.lab
  
  if (sum(plot.x.lab=="Int")>0){
    plot.x.lab[plot.x.lab=="Int"] <- "beta[0]"
  }
  cov.df$Label <- plot.x.lab

  return(cov.df)
}


table.grid <- function(grid.dat,latex.table=TRUE,round.vals=3){
  msd.time <- paste0(round(grid.dat$mean[grid.dat$Param=="time"],round.vals)," (",round(grid.dat$ci.l[grid.dat$Param=="time"],round.vals),",",round(grid.dat$ci.u[grid.dat$Param=="time"],round.vals),")")
  msd.param <- paste0(round(grid.dat$mean[grid.dat$Param!="time"],round.vals)," (",round(grid.dat$q0.025[grid.dat$Param!="time"],round.vals),",",round(grid.dat$q0.975[grid.dat$Param!="time"],round.vals),")")
  msd <- vector(mode="character",length=(length(msd.time)+length(msd.param)))
  msd[grid.dat$Param=="time"]<- msd.time
  msd[grid.dat$Param!="time"] <- msd.param
  rn <- unique(grid.dat$Param); cn <- unique(grid.dat$Mesh)
  full.matrix <- matrix(msd,length(rn),length(cn),dimnames = list(rn,cn))
  full.table <- as.table(full.matrix)
  
  if (latex.table==TRUE){
    full.table.latex <- xtable(full.matrix)
    align(full.table.latex) <- "c|c|c|c|c|"
    digits(full.table.latex) <- 4
    
    param.ind <- str_extract(rownames(full.table.latex), "\\d+\\.*\\d*")
    sub.ind <- which(!is.na(str_extract(rownames(full.table.latex), "\\d+\\.*\\d*")))
    rownames(full.table.latex)[sub.ind] <- paste0(str_extract(rownames(full.table.latex)[sub.ind],"[a-z]+"),"_",param.ind[sub.ind])
    rownames(full.table.latex)[2:nrow(full.table.latex)] <- paste0("$ ","\\",rownames(full.table.latex)[2:nrow(full.table.latex)]," $")
    
    return(list(full.table,full.table.latex))
  } else {
    return(full.table)
  }
}


table.mesh <- function(mesh.dat,latex.table=TRUE,round.vals=3){
  msd.time <- paste0(round(mesh.dat$mean[mesh.dat$Param=="time"],round.vals)," (",round(mesh.dat$ci.l[mesh.dat$Param=="time"],round.vals),",",round(mesh.dat$ci.u[mesh.dat$Param=="time"],round.vals),")")
  msd.param <- paste0(round(mesh.dat$mean[mesh.dat$Param!="time"],round.vals)," (",round(mesh.dat$q0.025[mesh.dat$Param!="time"],round.vals),",",round(mesh.dat$q0.975[mesh.dat$Param!="time"],round.vals),")")
  msd <- vector(mode="character",length=(length(msd.time)+length(msd.param)))
  msd[mesh.dat$Param=="time"]<- msd.time
  msd[mesh.dat$Param!="time"] <- msd.param
  rn <- unique(mesh.dat$Param); cn <- unique(mesh.dat$Grid)
  full.matrix <- matrix(msd,length(rn),length(cn),dimnames = list(rn,cn))
  full.table <- as.table(full.matrix)
  
  if (latex.table==TRUE){
    full.table.latex <- xtable(full.matrix)
    align(full.table.latex) <- "c|c|c|c|c|"
    digits(full.table.latex) <- 4
    
    param.ind <- str_extract(rownames(full.table.latex), "\\d+\\.*\\d*")
    sub.ind <- which(!is.na(str_extract(rownames(full.table.latex), "\\d+\\.*\\d*")))
    rownames(full.table.latex)[sub.ind] <- paste0(str_extract(rownames(full.table.latex)[sub.ind],"[a-z]+"),"_",param.ind[sub.ind])
    rownames(full.table.latex)[2:nrow(full.table.latex)] <- paste0("$ ","\\",rownames(full.table.latex)[2:nrow(full.table.latex)]," $")
    
    return(list(full.table,full.table.latex))
  } else {
    return(full.table)
  }
}

# Main Function -----------------------------------------------------------

trad_sim_out <- function(data,grids,meshes,by,true.theta,param,table=TRUE,round.vals=3,time.marker=60,latex.table=FALSE,plots=TRUE,plots.save=FALSE,plot.eachp=TRUE){
  # data - data from traditional simulation study
  # meshes - character string for names of the meshes for each list item of rank.values
  # true.theta - values of the fixed parameters for the simulations
  # param - vector of character strings for each parameter
  # table - generate tables for results?
  # round.vals - how many significant figures?
  # time.marker - only time plots, what time do we want to include as a horizontal red line?
  # latex.table - generate latex code for table of results?
  # plots - generate plots of data?
  # plots.save - save plots or only print?
  # plot.eachp - plot each parameter parameter plot individually?
  
  # Create mean and standard deviation data frames.
  l.g <- length(unique(grids))
  l.m <- length(unique(meshes))
  
  full.df <- lapply(1:length(data),function(i){combined.meanquantsd.mesh(data[[i]],param)})
  names(full.df) <- names(data)
  
  full.df.plot <- data.frame(matrix(rep(NA,length(full.df)*nrow(full.df[[1]])*(1+ncol(full.df[[1]]))),nrow=length(full.df)*nrow(full.df[[1]])))
  colnames(full.df.plot) <- c("Grid",colnames(full.df[[1]]))
  full.df.plot$Grid <- rep(names(full.df),each=nrow(full.df[[1]]))
  for (i in 1:4){
    df.ind <- (i-1)*nrow(full.df[[i]])
    full.df[[i]]$Param <- as.character(full.df[[i]]$Param)
    full.df[[i]]$Mesh <- as.character(full.df[[i]]$Mesh)
    full.df.plot[((df.ind+1):(df.ind+nrow(full.df[[i]]))),2:ncol(full.df.plot)] <- full.df[[i]]
  }
  
  grids.names <- as.character(full.df.plot$Grid)
  ord.grids.names <- unique(grids.names)
  for (i in 1:l.g){
    grids.names <- str_replace(grids.names,ord.grids.names[i],paste0("Grid ",i))
  }
  meshes.names <- as.character(full.df.plot$Mesh)
  ord.meshes.names <- unique(meshes.names)[order(unique(meshes.names),decreasing = T)]
  for (i in 1:l.m){
    meshes.names <- str_replace(meshes.names,ord.meshes.names[i],paste0("Mesh ",i)) # assume grid and mesh labels are in order
  }
  full.df.plot$Grid <- grids.names
  full.df.plot$Grid <- factor(as.character(full.df.plot$Grid),levels=unique(full.df.plot$Grid)[order(unique(full.df.plot$Grid))])
  full.df.plot$Mesh <- meshes.names
  full.df.plot$Mesh <- factor(as.character(full.df.plot$Mesh),levels=unique(full.df.plot$Mesh)[order(unique(full.df.plot$Mesh),decreasing=FALSE)])
  
  
  cov.df <- lapply(1:length(data),function(i){coverageparam.grid(data[[i]],param,true.theta)})
  names(cov.df) <- names(data)
  
  cov.df.plot <- data.frame(matrix(rep(NA,length(cov.df)*nrow(cov.df[[1]])*(1+ncol(cov.df[[1]]))),nrow=length(cov.df)*nrow(cov.df[[1]])))
  colnames(cov.df.plot) <- c("Grid",colnames(cov.df[[1]]))
  cov.df.plot$Grid <- rep(names(cov.df),each=nrow(cov.df[[1]]))
  for (i in 1:4){
    ind.r <- (i-1)*nrow(cov.df[[i]])
    cov.df[[i]]$Parameter <- as.character(cov.df[[i]]$Param)
    cov.df[[i]]$Mesh <- as.character(cov.df[[i]]$Mesh)
    cov.df.plot[(ind.r+1):(ind.r+nrow(cov.df[[i]])),2:ncol(cov.df.plot)] <- cov.df[[i]][,1:ncol(cov.df[[i]])]
  }
  
  grids.names <- as.character(cov.df.plot$Grid)
  ord.grids.names <- unique(grids.names)
  for (i in 1:l.g){
    grids.names <- str_replace(grids.names,ord.grids.names[i],paste0("Grid ",i))
  }
  meshes.names <- as.character(cov.df.plot$Mesh)
  ord.meshes.names <- unique(meshes.names)[order(unique(meshes.names),decreasing = T)]
  for (i in 1:l.m){
    meshes.names <- str_replace(meshes.names,ord.meshes.names[i],paste0("Mesh ",i)) # assume grid and mesh labels are in order
  }
  cov.df.plot$Grid <- grids.names
  cov.df.plot$Grid <- factor(as.character(cov.df.plot$Grid),levels=unique(cov.df.plot$Grid)[order(unique(cov.df.plot$Grid))])
  cov.df.plot$Mesh <- meshes.names
  cov.df.plot$Mesh <- factor(as.character(cov.df.plot$Mesh),levels=unique(cov.df.plot$Mesh)[order(unique(cov.df.plot$Mesh),decreasing=FALSE)])
  
  if (table==TRUE){
    bold <- function(x){
      paste0('{\\bfseries ', x, '}')
    }
    if (by=="grid"){ # for each grid look across meshes
      gg <- as.character(unique(full.df.plot$Grid))
      tables.grid <- lapply(1:length(gg),function(i,lt,rv){table.grid(full.df.plot[full.df.plot$Grid==gg[i],],lt,rv)},lt=latex.table,rv=round.vals)
      names(tables.grid) <- unique(full.df.plot$Grid)
    } else { # for each mesh, look across grids
      mm <- as.character(unique(full.df.plot$Mesh))
      tables.mesh <- lapply(1:length(mm),function(i,lt,rv){table.mesh(full.df.plot[full.df.plot$Mesh==mm[i],],lt,rv)},lt=latex.table,rv=round.vals)
      names(tables.mesh) <- unique(full.df.plot$Mesh)
    }
    
  }
  
  N <- nrow(run.out.final[[1]][[1]]$est.df)
  
  grids.names <- grids
  ord.grids.names <- unique(grids.names)
  for (i in 1:l.g){
    grids.names <- str_replace(grids.names,ord.grids.names[i],paste0("Grid ",i))
  }
  meshes.names <- meshes
  ord.meshes.names <- unique(meshes.names)[order(unique(meshes.names),decreasing = T)]
  for (i in 1:l.m){
    meshes.names <- str_replace(meshes.names,ord.meshes.names[i],paste0("Mesh ",i)) # assume grid and mesh labels are in order
  }
  
  crit.df <- data.frame(Grid=rep(unique(grids.names),each=8*N),Mesh=rep(rep(unique(meshes.names),each=2*N),4),Criterion=rep(rep(c("WAIC","DIC"),each=N),16))
  start.ind <- 1
  for (i in 1:length(run.out.final)){
    for (j in 1:length(run.out.final[[i]])){
      crit.df$Value[start.ind:(start.ind+length(run.out.final[[i]][[j]]$run.df$waic)-1)] <- run.out.final[[i]][[j]]$run.df$waic
      start.ind <- start.ind + length(run.out.final[[i]][[j]]$run.df$waic)
      crit.df$Value[start.ind:(start.ind+length(run.out.final[[i]][[j]]$run.df$dic)-1)] <- run.out.final[[i]][[j]]$run.df$dic
      start.ind <- start.ind + length(run.out.final[[i]][[j]]$run.df$dic)
    }
  }
  crit.df$Grid <- as.factor(crit.df$Grid)
  crit.df$Mesh <- factor(as.character(crit.df$Mesh),levels=unique(crit.df$Mesh)[order(unique(crit.df$Mesh),decreasing=FALSE)]) # as.factor(crit.df$Mesh)
  crit.df$Criterion <- as.factor(crit.df$Criterion)
  crit.df$Value <- as.numeric(crit.df$Value)
  
  critsum.df <- crit.df
  critsum.df %<>%
    group_by(Grid,Mesh,Criterion) %>%
    summarise(mean = mean(Value),sd=sd(Value), n = n())
  
  
  err.df <- data.frame(Grid=rep(unique(grids.names),each=12*N),Mesh=rep(rep(unique(meshes.names),each=3*N),4),Criterion=rep(rep(c("Error","Warning","FFT"),each=N),16),Run=rep(1:N,(4*4*3)))
  start.ind <- 1
  for (i in 1:length(run.out.final)){
    for (j in 1:length(run.out.final[[i]])){
      err <- run.out.final[[i]][[j]]$mess.ls$error
      err[!is.na(err)] <- 1
      err[is.na(err)] <- 0
      warn <- run.out.final[[i]][[j]]$mess.ls$warning
      warn[!is.na(warn)] <- 1
      warn[is.na(warn)] <- 0
      fft <- run.out.final[[i]][[j]]$mess.ls$FFT
      err.df$Value[start.ind:(start.ind+length(err)-1)] <- err
      start.ind <- start.ind + length(err)
      err.df$Value[start.ind:(start.ind+length(warn)-1)] <- warn
      start.ind <- start.ind + length(warn)
      err.df$Value[start.ind:(start.ind+length(fft)-1)] <- fft
      start.ind <- start.ind + length(fft)
    }
  }
  err.df$Grid <- as.factor(err.df$Grid)
  err.df$Mesh <- factor(as.character(err.df$Mesh),levels=unique(err.df$Mesh)[order(unique(err.df$Mesh),decreasing=FALSE)]) # as.factor(err.df$Mesh)
  err.df$Criterion <- as.factor(err.df$Criterion)
  err.df$Value <- as.numeric(err.df$Value)
  
  errsum.df <- err.df
  errsum.df %<>%
    group_by(Grid,Mesh,Criterion) %>%
    summarise(mean = mean(Value),sd=sd(Value), n = n())
  
  
    if (plots==TRUE){
      if (plots.save==TRUE){
        if (by=="grid"){
          # Time Plot
          p1 <- ggplot(full.df.plot[full.df.plot$Param=="time",],aes(x=Grid,y=mean,colour=Mesh)) + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=ci.l,ymax=ci.u,size=1),width=0.1) + geom_hline(yintercept=time.marker,color="red",linetype="dashed",size=1.5) + ggtitle("Average Time Taken for INLA Run") + scale_x_discrete(limits = levels(full.df.plot$Grid)) + ylab("Time") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.text = element_text(size=20),legend.title = element_text(size=25)) + guides(size=FALSE, color = guide_legend(override.aes = list(size = 3)))
          print(p1)
          ggsave("RegPolLGCPTradTimebyGrid.pdf",plot=p1,width = 15,height = 10,device="pdf")
          
          p1 <- ggplot(full.df.plot[full.df.plot$Param=="time",],aes(x=Mesh,y=mean,colour=Mesh)) + facet_wrap(~Grid,scales="free_y") + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=ci.l,ymax=ci.u,size=1),width=0.1) + geom_hline(yintercept=time.marker,color="red",linetype="dashed",size=1.5) + ggtitle("Average Time Taken for INLA Run") + scale_x_discrete(limits = levels(full.df.plot$Mesh)) + ylab("Time") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
          print(p1)
          ggsave("RegPolLGCPTradTimevMeshfacetGrid.pdf",plot=p1,width = 15,height = 10,device="pdf")
          
          # Parameter Estimates
          params <- as.character(unique(full.df.plot[full.df.plot$Param!="time",]$Param)); N.p <- length(params);
          
          param.ind <- str_extract(params, "\\d+\\.*\\d*")
          sub.ind <- which(!is.na(param.ind))
          param.lab <- params
          param.lab[sub.ind] <- paste0(str_extract(param.lab[sub.ind],"[a-z]+"),"[",param.ind[sub.ind],"]")
          plots <- lapply(1:N.p,function(i,names,data,lab,vals){ggplot(data[data$Param==names[i],],aes(x=Mesh,y=mean,colour=Mesh)) + facet_wrap(~Grid,scales = "free_y") + scale_x_discrete(limits = levels(data$Mesh)) + xlab("Mesh") + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=q0.025,ymax=q0.975,size=1),width=0.1) + geom_hline(yintercept=true.theta[i],color="red",linetype="dashed",size=1.5) + ggtitle(parse(text=paste0("Parameter~Estimates~of~", lab[i]))) + ylab("Estimate") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")},params,full.df.plot,param.lab,true.theta)
          
          all.param <- do.call(grid.arrange,list(grobs=plots,ncol=1))
          
          ggsave("RegPolLGCPTradParameterEstimateswQuantbyGrid.pdf",plot=all.param,width=15,height=10,device="pdf")
          
          if (plot.eachp==TRUE){
            for (i in 1:N.p){
              ggsave(paste0("RegPolLGCPTrad",params[i],"EstimateswQuantbyGrid.pdf"),plot=plots[[i]],width=15,height=10,device="pdf")
            }
          }
          
          p6 <- ggplot(cov.df.plot) + geom_point(aes(x=Label,y=Coverage,col=Mesh,size=2)) + scale_x_discrete(labels = label_parse()) + facet_wrap(~Grid) + xlab("Parameter") + geom_hline(yintercept=0.95,color="red",linetype="dashed",size=1.5) + ggtitle("Coverage of 95% Credible Intervals") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.text = element_text(size=20),legend.title = element_text(size=25)) + guides(size = FALSE, color = guide_legend(override.aes = list(size = 3)))
          print(p6)
          
          ggsave("RegPolLGCPCoveragebyGrid.pdf",plot=p6,width = 15,height = 10,device="pdf")
          
          p5 <- ggplot(errsum.df) + geom_point(aes(x=Mesh,y=mean,col=Criterion,size=2)) + scale_x_discrete(limits = levels(full.df.plot$Mesh)) + xlab("Mesh") + facet_wrap(~Grid+Criterion,scales="free_x",ncol=3) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text.x = element_text(size=20),axis.text.y = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
          
          print(p5)
          
          ggsave("RegPolLGCPTradErrMeanbyGrid.pdf",plot=p5,width = 15,height = 10,device="pdf")
        } else {
          # Time Plot
          p1 <- ggplot(full.df.plot[full.df.plot$Param=="time",],aes(x=Mesh,y=mean,colour=Grid)) + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=ci.l,ymax=ci.u,size=1),width=0.1) + geom_hline(yintercept=time.marker,color="red",linetype="dashed",size=1.5) + ggtitle("Average Time Taken for INLA Run") + scale_x_discrete(limits = levels(full.df.plot$Mesh)) + ylab("Time") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.text = element_text(size=20),legend.title = element_text(size=25)) + guides(size=FALSE, color = guide_legend(override.aes = list(size = 3)))
          print(p1)
          ggsave("RegPolLGCPTradTimebyMesh.pdf",plot=p1,width = 15,height = 10,device="pdf")
          
          p1 <- ggplot(full.df.plot[full.df.plot$Param=="time",],aes(x=Grid,y=mean,colour=Grid)) + facet_wrap(~Mesh,scales="free_y") + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=ci.l,ymax=ci.u,size=1),width=0.1) + geom_hline(yintercept=time.marker,color="red",linetype="dashed",size=1.5) + ggtitle("Average Time Taken for INLA Run") + scale_x_discrete(limits = levels(full.df.plot$Grid)) + ylab("Time") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
          print(p1)
          ggsave("RegPolLGCPTradTimevGridfacetMesh.pdf",plot=p1,width = 15,height = 10,device="pdf")
          
          # Parameter Estimates
          params <- as.character(unique(full.df.plot[full.df.plot$Param!="time",]$Param)); N.p <- length(params);
          
          param.ind <- str_extract(params, "\\d+\\.*\\d*")
          sub.ind <- which(!is.na(param.ind))
          param.lab <- params
          param.lab[sub.ind] <- paste0(str_extract(param.lab[sub.ind],"[a-z]+"),"[",param.ind[sub.ind],"]")
          plots <- lapply(1:N.p,function(i,names,data,lab,vals){ggplot(data[data$Param==names[i],],aes(x=Grid,y=mean,colour=Grid)) + facet_wrap(~Mesh,scales = "free_y") + scale_x_discrete(limits = levels(data$Grid)) + xlab("Grid") + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=q0.025,ymax=q0.975,size=1),width=0.1) + geom_hline(yintercept=true.theta[i],color="red",linetype="dashed",size=1.5) + ggtitle(parse(text=paste0("Parameter~Estimates~of~", lab[i]))) + ylab("Estimate") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")},params,full.df.plot,param.lab,true.theta)
          
          all.param <- do.call(grid.arrange,list(grobs=plots,ncol=1))
          
          ggsave("RegPolLGCPTradParameterEstimateswQuantbyMesh.pdf",plot=all.param,width=15,height=15,device="pdf")
          
          if (plot.eachp==TRUE){
            for (i in 1:N.p){
              ggsave(paste0("RegPolLGCPTrad",params[i],"EstimateswQuantbyMesh.pdf"),plot=plots[[i]],width=15,height=10,device="pdf")
            }
          }
          
          
          p6 <- ggplot(cov.df.plot) + geom_point(aes(x=Label,y=Coverage,colour=Grid,size=2)) + scale_x_discrete(labels = label_parse()) + facet_wrap(~Mesh,scales="free_y") + xlab("Parameter") + geom_hline(yintercept=0.95,color="red",linetype="dashed",size=1.5) + ggtitle("Coverage of 95% Credible Intervals") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.text = element_text(size=20),legend.title = element_text(size=25)) + guides(size = FALSE, color = guide_legend(override.aes = list(size = 3)))
          print(p6)
          
          ggsave("RegPolLGCPCoveragebyMesh.pdf",plot=p6,width = 15,height = 10,device="pdf")
          
          p5 <- ggplot(errsum.df) + geom_point(aes(x=Grid,y=mean,col=Criterion,size=2)) + scale_x_discrete(limits = levels(full.df.plot$Grid)) + xlab("Grid") + facet_wrap(~Mesh+Criterion,scales="free_x",ncol=3) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text.x = element_text(size=20),axis.text.y = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
          
          print(p5)
          
          ggsave("RegPolLGCPTradErrMeanbyMesh.pdf",plot=p5,width = 15,height = 10,device="pdf")
        }
        
        p2 <- ggplot(crit.df) + geom_histogram(aes(Value,fill=Criterion),binwidth = 25) + facet_wrap(~Grid+Mesh,scales = "free_x") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=15),strip.text.x = element_text(size=20),legend.text = element_text(size=20),legend.title = element_text(size=25)) + guides(color = guide_legend(override.aes = list(size = 3)))
        print(p2)
        
        ggsave("RegPolLGCPTradDICWAIC.pdf",plot=p2,width = 15,height = 10,device="pdf")
        
        
        p3 <- ggplot(critsum.df) + geom_point(aes(x=Mesh,y=mean,col=Grid,size=2)) + xlab("Mesh") + facet_wrap(~Grid+Criterion,scales = "free_y",ncol=2) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.text = element_text(size=20),legend.title = element_text(size=25)) + guides(size=FALSE, color = guide_legend(override.aes = list(size = 3)))
        print(p3)
        
        ggsave("RegPolLGCPTradDICWAICMean.pdf",plot=p3,width = 15,height = 10,device="pdf")
        
      } else {
        if (by=="grid"){
          # Time Plot
          p1 <- ggplot(full.df.plot[full.df.plot$Param=="time",],aes(x=Grid,y=mean,colour=Mesh)) + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=ci.l,ymax=ci.u,size=1),width=0.1) + geom_hline(yintercept=time.marker,color="red",linetype="dashed",size=1.5) + ggtitle("Average Time Taken for INLA Run") + scale_x_discrete(limits = levels(full.df.plot$Grid)) + ylab("Time") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.text = element_text(size=20),legend.title = element_text(size=25)) + guides(size=FALSE, color = guide_legend(override.aes = list(size = 3)))
          print(p1)
          
          p1 <- ggplot(full.df.plot[full.df.plot$Param=="time",],aes(x=Mesh,y=mean,colour=Mesh)) + facet_wrap(~Grid,scales="free_y") + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=ci.l,ymax=ci.u,size=1),width=0.1) + geom_hline(yintercept=time.marker,color="red",linetype="dashed",size=1.5) + ggtitle("Average Time Taken for INLA Run") + scale_x_discrete(limits = levels(full.df.plot$Mesh)) + ylab("Time") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
          print(p1)
          
          # Parameter Estimates
          params <- as.character(unique(full.df.plot[full.df.plot$Param!="time",]$Param)); N.p <- length(params);

          param.ind <- str_extract(params, "\\d+\\.*\\d*")
          sub.ind <- which(!is.na(param.ind))
          param.lab <- params
          param.lab[sub.ind] <- paste0(str_extract(param.lab[sub.ind],"[a-z]+"),"[",param.ind[sub.ind],"]")
          plots <- lapply(1:N.p,function(i,names,data,lab,vals){ggplot(data[data$Param==names[i],],aes(x=Mesh,y=mean,colour=Mesh)) + facet_wrap(~Grid,scales = "free_y") + scale_x_discrete(limits = levels(data$Mesh)) + xlab("Mesh") + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=q0.025,ymax=q0.975,size=1),width=0.1) + geom_hline(yintercept=true.theta[i],color="red",linetype="dashed",size=1.5) + ggtitle(parse(text=paste0("Parameter~Estimates~of~", lab[i]))) + ylab("Estimate") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")},params,full.df.plot,param.lab,true.theta)

          all.param <- do.call(grid.arrange,list(grobs=plots,ncol=1))

          print(all.param)
          
          if (plot.eachp==TRUE){
            for (i in 1:N.p){
              print(plots[[i]])
            }
          }

          p6 <- ggplot(cov.df.plot) + geom_point(aes(x=Label,y=Coverage,col=Mesh,size=2)) + scale_x_discrete(labels = label_parse()) + facet_wrap(~Grid) + xlab("Parameter") + geom_hline(yintercept=0.95,color="red",linetype="dashed",size=1.5) + ggtitle("Coverage of 95% Credible Intervals") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.text = element_text(size=20),legend.title = element_text(size=25)) + guides(size = FALSE, color = guide_legend(override.aes = list(size = 3)))
          print(p6)
          
          p5 <- ggplot(errsum.df) + geom_point(aes(x=Mesh,y=mean,col=Criterion,size=2)) + scale_x_discrete(limits = levels(full.df.plot$Mesh)) + xlab("Mesh") + facet_wrap(~Grid+Criterion,scales="free_x",ncol=3) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text.x = element_text(size=20),axis.text.y = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
          
          print(p5)
          

        } else {
          # Time Plot
          p1 <- ggplot(full.df.plot[full.df.plot$Param=="time",],aes(x=Mesh,y=mean,colour=Grid)) + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=ci.l,ymax=ci.u,size=1),width=0.1) + geom_hline(yintercept=time.marker,color="red",linetype="dashed",size=1.5) + ggtitle("Average Time Taken for INLA Run") + scale_x_discrete(limits = levels(full.df.plot$Mesh)) + ylab("Time") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.text = element_text(size=20),legend.title = element_text(size=25)) + guides(size=FALSE, color = guide_legend(override.aes = list(size = 3)))
          print(p1)
          
          p1 <- ggplot(full.df.plot[full.df.plot$Param=="time",],aes(x=Mesh,y=mean,colour=Grid)) + facet_wrap(~Mesh,scales="free_y") + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=ci.l,ymax=ci.u,size=1),width=0.1) + geom_hline(yintercept=time.marker,color="red",linetype="dashed",size=1.5) + ggtitle("Average Time Taken for INLA Run") + scale_x_discrete(limits = levels(full.df.plot$Grid)) + ylab("Time") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
          print(p1)
          
          # Parameter Estimates
          params <- as.character(unique(full.df.plot[full.df.plot$Param!="time",]$Param)); N.p <- length(params);

          param.ind <- str_extract(params, "\\d+\\.*\\d*")
          sub.ind <- which(!is.na(param.ind))
          param.lab <- params
          param.lab[sub.ind] <- paste0(str_extract(param.lab[sub.ind],"[a-z]+"),"[",param.ind[sub.ind],"]")
          plots <- lapply(1:N.p,function(i,names,data,lab,vals){ggplot(data[data$Param==names[i],],aes(x=Grid,y=mean,colour=Grid)) + facet_wrap(~Mesh,scales = "free_y") + scale_x_discrete(limits = levels(data$Grid)) + xlab("Grid") + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=q0.025,ymax=q0.975,size=1),width=0.1) + geom_hline(yintercept=true.theta[i],color="red",linetype="dashed",size=1.5) + ggtitle(parse(text=paste0("Parameter~Estimates~of~", lab[i]))) + ylab("Estimate") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")},params,full.df.plot,param.lab,true.theta)

          all.param <- do.call(grid.arrange,list(grobs=plots,ncol=1))

          print(all.param)
          
          if (plot.eachp==TRUE){
            for (i in 1:N.p){
              print(plots[[i]])
            }
          }
          
          
          p6 <- ggplot(cov.df.plot) + geom_point(aes(x=Label,y=Coverage,col=Grid,size=2)) + scale_x_discrete(labels = label_parse()) + facet_wrap(~Mesh) + xlab("Parameter") + geom_hline(yintercept=0.95,color="red",linetype="dashed",size=1.5) + ggtitle("Coverage of 95% Credible Intervals") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.text = element_text(size=20),legend.title = element_text(size=25)) + guides(size = FALSE,color = guide_legend(override.aes = list(size = 3)))
          print(p6)
          
          p5 <- ggplot(errsum.df) + geom_point(aes(x=Grid,y=mean,col=Criterion,size=2)) + scale_x_discrete(limits = levels(full.df.plot$Grid)) + xlab("Grid") + facet_wrap(~Mesh+Criterion,scales="free_x",ncol=3) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text.x = element_text(size=20),axis.text.y = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
          
          print(p5)
          
        }

        p2 <- ggplot(crit.df) + geom_histogram(aes(Value,fill=Criterion),binwidth = 25) + facet_wrap(~Grid+Mesh,scales = "free_x") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.text = element_text(size=20),legend.title = element_text(size=25))
        p3 <- ggplot(critsum.df) + geom_point(aes(x=Mesh,y=mean,col=Grid,size=2)) + xlab("Mesh") + facet_wrap(~Grid+Criterion,scales="free_y",ncol=2) + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")

        print(p2)
        print(p3)


        # Time Plot
        p1 <- ggplot(full.df.plot[full.df.plot$Param=="time",],aes(x=Mesh,y=mean,colour=Mesh)) + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=ci.l,ymax=ci.u,size=1),width=0.1) + geom_hline(yintercept=time.marker,color="red",linetype="dashed",size=1.5) + ggtitle("Average Time Taken for INLA Run") + scale_x_discrete(limits = levels(full.df.plot$Mesh)) + ylab("Time") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")
        print(p1)
        
        # Parameter Estimates
        params <- as.character(unique(full.df.plot[full.df.plot$Param!="time",]$Param)); N.p <- length(params);
        
        param.ind <- str_extract(params, "\\d+\\.*\\d*")
        sub.ind <- which(!is.na(param.ind))
        param.lab <- params
        param.lab[sub.ind] <- paste0(str_extract(param.lab[sub.ind],"[a-z]+"),"[",param.ind[sub.ind],"]")
        
        plots <- lapply(1:N.p,function(i,names,data,lab){ggplot(data[data$Param==names[i],],aes(x=as.factor(str_extract(Mesh,"[-+.e0-9]*\\d")),y=mean,colour=Mesh)) + xlab("Mesh") + geom_point(aes(size=2)) + geom_errorbar(aes(ymin=q0.025,ymax=q0.975,size=1),width=0.1) + geom_hline(yintercept=true.theta[i],color="red",linetype="dashed",size=1.5) + scale_x_discrete(limits = levels(full.df.plot$Mesh)) + ggtitle(parse(text=paste0("Parameter~Estimates~of~", lab[i]))) + ylab("Estimate") + theme(plot.title = element_text(size=35,hjust = 0.5),axis.title = element_text(size=25),axis.text = element_text(size=25),strip.text.x = element_text(size=20),legend.position = "none")},params,full.df,param.lab)
        
        all.param <- do.call(grid.arrange,plots)
        
      }
    }
    
  if (table==TRUE){
    if (by=="grid"){
      return(list("results.table"=tables.grid))
    } else {
      return(list("results.table"=tables.mesh))
    }
  }
}
