# Grid-Mesh: LGCP Example: SBC Simulation Study ---------------------------

# This script will implement the SBC simulation study for the LGCP Regular Polygon example to check how the algorithm behaves when we use different grid and mesh resolutions for both the parameters and the mean field.
# The section for the simulation of the covariates and meshes are run before the code is moved to Balena, with the outputs saved in order to ensure there is no need to re-simulate for each iteration.
# We have  sub-functions that generates the data and the A matrix for the INLA runs to keep the implementation of the simulation study as compact as possible.

# Author: Nadeen Khaleel

# Arguments Read In -------------------------------------------------------

args=(commandArgs(TRUE))
print(args)

if (length(args)==0){
  print("No arguments supplied.")
  # Set default values.
  this.node = 1 # which "node" am I on - change per job
  total.nodes = 1 # how many nodes am I using?
  Nprocs.vec = 1 # vector where each element contains the number of processors for a particular node
  Nprocs.total = 1 # total number of processors across ALL nodes (/jobs)
  N = 1 # how many simulations?
  L = 1
  sim = 0 # start new or re-starting at last saved simulation?
}else{
  for (i in 1:length(args)){
    eval(parse(text=args[[i]]))
  }
}

# Example of R CMD BATCH command from job slurm scripts for implementing this simulation study:
# R CMD BATCH --vanilla '--args this.node=1 total.nodes=15 Nprocs.vec=rep(8,15) Nprocs.total=120 N=1000 L=100 sim=1' GridMeshOptim_final.R gm_lgcp1.out
# We had 15 jobs each run across a separate node using similar commands to the above.


# Libraries ---------------------------------------------------------------

# Local Directory is the Temp Directory in Compute Nodes
library(unixtools) # in r_packages
unixtools::set.tempdir("/local/")


# Set up parallel computing
Nprocs <- Nprocs.vec[this.node]

library(doParallel)
library(foreach)
parallelCluster <- parallel::makeCluster(Nprocs)
print(parallelCluster)
registerDoParallel(parallelCluster)


cs <- c(0,cumsum(Nprocs.vec))

ind.procs.all <- 1:Nprocs.total

ii <- (this.node-1)*8
ind.procs <- ind.procs.all[(ii+1):(ii+8)]

for (tt in ind.procs){
  tdirloc.k <- paste0("/local/intmpproc-",tt)
  print(paste0("Temp Directory ",tdirloc.k, " Created"))
  td.c <- try(dir.create(tdirloc.k))
  if (inherits(td.c, "try-error") || !td.c){
    stop(paste("Fail to create directory [", tdirloc.k, "]. Stop.", sep=""))
  }
}

ptm <- proc.time()

foreach(k = ind.procs) %dopar% {
  
  library(unixtools) # r_packages
  
  # Set-Up Temp folder for each process, which was already created above and just assign it as the temporary folder for process k (=1:40)
  tdirloc.k <- paste0("/local/intmpproc-",k)
  unixtools::set.tempdir(tdirloc.k)
  
  print(paste0("Set Temp Directory to ",tdirloc.k))
  
  
  library(INLA)
  library(mvtnorm)
  library(sp)
  library(sf)
  library(spatstat)
  library(raster)
  library(maptools)
  library(stringr)
  
  par.lic.filepath <- "./pardiso.lic" # file path for pardiso licence if in use
  inla.setOption(pardiso.license = par.lic.filepath)
  
  inla.setOption(num.threads=2) # 8 procs using 2 cores each per node
  
  # Functions ---------------------------------------------------------------
  
  prior.sim <- function(n=1,rho.star,sigma.star){
    # n is the number of samples of the parameters you want sampled from their priors.  
    # sigma.star, rho.star: pc prior for covariance parameters will be provided, but will keep the priors for the fixed effects in place here.
    
    # Using results from LGCPCovarianceandFixedPriorTest.R and the tables output, I get the following priors for beta and sigma.
    
    beta.0.sim <- rnorm(n,2,2); beta.1.sim <- rnorm(n,1,2); beta.2.sim <- rnorm(n,-2,2)
    
    # For the covariance parameters of the latent Gaussian field, I have a joint prior on the marginal standard deviation and the range, however, in order to simulate from this I need the inverse CDF to generate this from a Uniform RV. We can also use the priors fpr kappa and tau given kappa to generate the inverses.
    # (https://www.tandfonline.com/doi/pdf/10.1080/01621459.2017.1415907?needAccess=true) 
    # (https://projecteuclid.org/journals/statistical-science/volume-32/issue-1/Penalising-Model-Component-Complexity--A-Principled-Practical-Approach-to/10.1214/16-STS576.full)
    
    alpha.rho <- rho.star[2]; alpha.sigma <- sigma.star[2]; rho.0 <- rho.star[1]; sigma.0 <- sigma.star[1]
    u1 <- runif(n,0,1); u2 <- runif(n,0,1)
    rho.sim <- rho.0*log(alpha.rho)/log(u1)
    sigma.sim <- sigma.0*log(1-u2)/log(alpha.sigma)
    return(list(beta0.tilde=beta.0.sim,beta1.tilde=beta.1.sim,beta2.tilde=beta.2.sim,sigma.tilde=sigma.sim,rho.tilde=rho.sim))
  }
  
  cov.surface.gen <- function(W,theta,cov1.im,cov2.im,int.im){
    beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]];
    
    mu <- beta.0*int.im + beta.1*cov1.im + beta.2*cov2.im
    
    return(mu)
  }
  
  
  
  data.gen <- function(W,n.max,theta,disc,quads,coord,area,ord,c1.list,c2.list,mu){
    # W: the window
    # n.max: the largest discretisation which will be aggregated up for the disc < n.max values
    # theta: the parameters required to generate the data.
    # disc: the vector of discretisations across the window for which the data will be produced at. For each data frame, produce at the finest, n.max > disc
    # quads: list of grid resolutions, increases efficiency of point aggregation as it does not need to regenerate grids every time
    # coord: list of grid centres for quads, removes another unnecessary calculation
    # area: list of areas of the grid cells (should be the same in this example but transferrable to irregular polygon)
    # ord: list of index orderings so that the data goes down y before going across in x
    # ci.list: list of covariates aggregated to matching grid resolution
    # mu: pre-calculated mu pixel image for simulation
    
    # Simulate from this process with the parameters as defined above.
    beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]]; sigma <- theta[[4]]; rho <- theta[[5]];
    
    
    # Produce the point pattern from LGCP, error produced if the number of points is TOO large
    lgcp.ppp <- try(rLGCP(model="matern",mu,var=(sigma)^2,scale=rho/2,nu=1,win = W,saveLambda = TRUE)) # Note: default resolution of the simulated data is 128x128, if this is too coarse for the data aggregations (if one of the data aggregation requires a grid finer than 128x128), then also set dimyx=c(ny,nx).
    
    
    if (class(lgcp.ppp)!="try-error"){
      
      if (lgcp.ppp$n!=0){
        grid.names <- c(paste0("grid",disc[,1],disc[,2]),paste0("grid",n.max,n.max))
        data.list <- vector(mode="list",length=length(grid.names))
        names(data.list) <- grid.names
        
        disc.full <- rbind(disc,c(n.max,n.max))
        
        for (i in 1:dim(disc.full)[1]){
          # Count Data Generation
          dist <- disc.full[i,]
          M <- dist[1]; N <- dist[2]
          g <- quads[[i]]
          coord.df <- coord[[i]]
          a <- area[[i]]
          ord.df <- ord[[i]]
          
          
          x.ord <- coord.df$x[ord.df$ind]; y.ord <- coord.df$y[ord.df$ind]; cell.area <- a[ord.df$ind]
          
          q <- quadratcount(lgcp.ppp,tess=g)
          count.df <- data.frame(x=x.ord,y=y.ord,count=as.vector(unname(q)),area=cell.area)
          
          count.df$cov1 <- c1.list[[i]][ord.df$ind]
          count.df$cov2 <- c2.list[[i]][ord.df$ind]
          
          data.list[[i]] <- count.df
        }
        
        return(list("data"=data.list,"lambda"=attributes(lgcp.ppp)$Lambda,"err"=FALSE,"zero"=FALSE))
      } else {
        return(list("err"=TRUE,"zero"=TRUE,"lgcp"=lgcp.ppp)) # err=TRUE for while loop
      }
    } else {
      return(list("err"=TRUE,"zero"=FALSE))
    }
    
  }
  
  # Generate A matrix and stack data
  A_stack.gen <- function(data.gm,mesh,sigma.star,rho.star){
    the_spde <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[1],rho.star[2]),prior.sigma = c(sigma.star[1],sigma.star[2]))
    
    s.index <- inla.spde.make.index("field",n.spde=the_spde$n.spde)
    coords <- data.gm[,c("x","y")]
    coordinates(coords) <- ~ x + y
    A <- inla.spde.make.A(mesh, loc=coords)
    stk <- inla.stack(data=list(resp=data.gm$count),A=list(A,1),effects=list(c(s.index,list(int=1)),list(cov1=data.gm$cov1,cov2=data.gm$cov2,larea=log(data.gm$area))),tag='est')
    
    return(list("spde"=the_spde,"A"=A,"stack.est"=stk))
  }
  
  
  # Covariate and Mesh Set-up -----------------------------------------------
  
  # Set working directory to save the outputs before moving them over to Balena
  # # Either setwd() to the source file location, or run the following:
  # library("rstudioapi")
  # setwd(dirname(getActiveDocumentContext()$path))
  
  # W <- owin(c(0,5),c(0,5))
  # n.max <- 50 # (0.1x0.1)
  # disc <- matrix(c(10,10,20,20,25,25),ncol=2,byrow=T)
  # disc.full <- rbind(disc,c(n.max,n.max))
  # 
  # quad.gen <- function(window,cell.n,save.name){
  #   quad.names <- c(paste0("quad",cell.n[,1],cell.n[,2]))
  #   quad.list <- vector(mode="list",length=length(quad.names))
  #   names(quad.list) <- quad.names
  #   W <- window # added 04/2021 for re-run
  #   for (i in 1:dim(cell.n)[1]){
  #     print(i)
  #     n.cell <- cell.n[i,]
  #     M <- n.cell[1]; N <- n.cell[2]
  #     g <- quadrats(W,M,N)
  #     
  #     quad.list[[i]] <- g
  #     
  #   }
  #   save("quad"=quad.list,file=save.name)
  # }
  # # 
  quad.file <- "QuadratsRegPolLGCP.rda"
  # quad.gen(window=W,disc.full,quad.file)
  # 
  # coordareaord.gen <- function(quad,save.name){
  #   coord.list <- vector(mode="list",length=length(quad))
  #   names(coord.list) <- str_replace_all(names(quad),"quad","coord")
  #   area.list <- vector(mode="list",length=length(quad))
  #   names(area.list) <- str_replace_all(names(quad),"quad","cellarea")
  #   ord.list <- vector(mode="list",length=length(quad))
  #   names(ord.list) <- str_replace_all(names(quad),"quad","ord")
  #   for (i in 1:length(quad)){
  #     g <- quad[[i]]
  #     g.sp <- as(g,"SpatialPolygons")
  #     ord <- sapply(1:length(g.sp),function(ii){unlist(centroid.owin(g.sp[ii]))})
  #     a <- unname(rgeos::gArea(g.sp,byid = T))
  #     
  #     m <- matrix(as.numeric(unlist(str_extract_all(names(g.sp),"\\d*\\d"))),ncol=2,byrow=TRUE)
  #     ord.df <- data.frame(ind=1:g$n,row=m[,1],col=m[,2]) # col matches the raster definition of col=x, and row=y which makes sense visually from the plots as x<--> and y ^inc(^)
  #     ord.list[[i]] <- ord.df[order(ord.df$col),]
  #     
  #     coord.list[[i]] <- data.frame(x=ord[1,],y=ord[2,])
  #     area.list[[i]] <- a
  #   }
  #   save(coord.list,area.list,ord.list,file=save.name)
  # }
  # 
  # load(quad.file)
  coord.file <- "CoordsRegPolLGCP.rda"
  # Re-ordering the coordinates and area values so that they change y for a fixed x before moving along x.
  # coordareaord.gen(quad=quad.list,coord.file)
  # #
  # set.seed(625)
  # reg.poly <- W$type=="rectangle" # True if rectangluar window/false if different polygon.
  # 
  # # Points at which to simulate the data
  # nx <- 250; ny <- 250;
  # 
  # # Create (fine) regular grid over the window in order to be able to simulate the covariate. Want to use intersect function to make sure that the overlayed grid points retained lie within the boundary, this is not needed for the rectangular grid, the function gridcentres should work fine there.
  # # However, we can use the function (in spatstat), inside.owin to create a logical vector for which points lie inside/outside the grid.
  # 
  # W.gridc <- gridcentres(W, nx, ny)
  # keep.c <- inside.owin(W.gridc$x,W.gridc$y,W)
  # 
  # W.gridc$x <- W.gridc$x[keep.c]; W.gridc$y <- W.gridc$y[keep.c];
  # 
  # 
  # # Simulate the covariate data (and save covariates too!)
  # 
  # max.y <- max(W.gridc$y)
  # cov1.df <- data.frame(x=W.gridc$x,y=W.gridc$y)
  # cov1.df$val <- W.gridc$y/max.y + rnorm(length(W.gridc$x),mean = 0,sd=0.5)
  # 
  # max.norm <- sqrt(max(W.gridc$x)^2+max(W.gridc$y)^2)
  # cov2.df <- data.frame(x=W.gridc$x,y=W.gridc$y)
  # cov2.df$val <- sqrt(W.gridc$x^2+W.gridc$y^2)/max.norm + rnorm(length(W.gridc$x),mean = 0,sd = 0.5)
  # 
  # 
  # int.df <- data.frame(x=W.gridc$x,y=W.gridc$y,val=rep(1,length(W.gridc$x)))
  # 
  # cov1.ras <- rasterFromXYZ(cov1.df)
  # cov2.ras <- rasterFromXYZ(cov2.df)
  # int.ras <- rasterFromXYZ(int.df,digits = 10)
  # 
  # cov1.im <- as.im.RasterLayer(cov1.ras)
  # cov2.im <- as.im.RasterLayer(cov2.ras)
  # int.im <- as.im.RasterLayer(int.ras)
  # 
  # 
  cov.name <- paste0("GridMeshRegPolLGCPSSCov.rda")
  # save(cov1.im,cov2.im,int.im,cov1.ras,cov2.ras,file=cov.name)
  # 
  # covgrid <- function(cov1.ras,cov2.ras,quad,covgrid.file){
  #   cov1grid.list <- vector(mode="list",length=length(quad))
  #   cov1.names <- str_replace_all(names(quad),"quad","cov1")
  #   names(cov1grid.list) <- cov1.names
  #   cov2grid.list <- vector(mode="list",length=length(quad))
  #   cov2.names <- str_replace_all(names(quad),"quad","cov2")
  #   names(cov2grid.list) <- cov2.names
  #   
  #   for (i in 1:length(quad)){
  #     print(i)
  #     g <- quad[[i]]
  #     g.sp <- as(g,"SpatialPolygons")
  #     cov1grid.list[[i]] <- raster::extract(cov1.ras,g.sp,weights=T,fun=mean)
  #     cov2grid.list[[i]] <- raster::extract(cov2.ras,g.sp,weights=T,fun=mean)
  #   }
  #   save(cov1grid.list,cov2grid.list,file=covgrid.file)
  # }
  # 
  covgrid.file <- "CovAggGridRegPolLGCP.rda"
  # load(cov.name)
  # load(quad.file)
  # covgrid(cov1.ras,cov2.ras,quad.list,covgrid.file)
  # 
  # # Meshes
  # # Run this once then load up the meshes at the beginning.
  # set.seed(1250)
  # n.max <- 50 # (0.1x0.1)
  # disc <- matrix(c(10,10,20,20,25,25),ncol=2,byrow=T)
  # disc.full <- rbind(disc,c(n.max,n.max))
  # 
  # mesh.gen <- function(W,disc,quad,coord,save.name){
  #   mesh.size <- matrix(c(round((W$xrange[2]-W$xrange[1])/disc[,1],3),round((W$yrange[2]-W$yrange[1])/disc[,2],3)),ncol=2)
  #   mesh.names <- paste0("mesh",mesh.size[,1],mesh.size[,2])
  #   mesh.list <- vector(mode="list",length=length(mesh.names))
  #   names(mesh.list) <- mesh.names
  #   for (i in 1:dim(disc)[1]){
  #     M <- disc[i,1]; N <- disc[i,2]
  #     cellsize <- c((W$xrange[2]-W$xrange[1])/M,(W$yrange[2]-W$yrange[1])/N)
  #     # q <- quadrats(W,M,N)
  #     g <- quad[[i]]
  #     ord <- coord[[i]]
  #     
  #     coords <- ord[,c("x","y")]
  #     coordinates(coords) <- ~ x + y
  #     boundary <- as(W,"SpatialPolygons") # For the meshes
  #     mesh <- inla.mesh.2d(loc=coords, boundary=boundary, max.edge=c(max(cellsize), max(cellsize)+0.5), min.angle=c(30, 21),
  #                          max.n=c(48000, 16000), ## Safeguard against large meshes.
  #                          max.n.strict=c(128000, 128000), ## Don't build a huge mesh!
  #                          cutoff=0.01, ## Filter away adjacent points.
  #                          offset=c(0.1, 1)) ## Offset for extra boundaries, if needed.
  #     mesh.list[[i]] <- mesh
  #   }
  #   save("mesh"=mesh.list,file=save.name)
  # }
  # 
  # load(quad.file)
  # load(coord.file)
  meshes.file <- "MeshesRegPolLGCP.rda"
 
  # Simulations -------------------------------------------------------------
  
  
  # SBC Parameters for Parallelisation
  # Assign simulations per processors and then extract the required processors and simulations for each node/job
  M.it.total <- rep(N%/%Nprocs.total,length=Nprocs.total)
  if (N%%Nprocs.total!=0){M.it.total[1:(N%%Nprocs.total)] <- M.it.total[1:(N%%Nprocs.total)] + 1}
  
  # For the seed, so can more easily run another N=1000 if needed
  SEED.N <- 2*N
  SEED.M.it.total <- rep(SEED.N%/%Nprocs.total,length=Nprocs.total)
  if (SEED.N%%Nprocs.total!=0){SEED.M.it.total[1:(SEED.N%%Nprocs.total)] <- SEED.M.it.total[1:(SEED.N%%Nprocs.total)] + 1}
  
  
  fft.threshold <- 5 # how many "Fail to factorise Q" warnings accepted before a warning message is produced for user. This was set, however the FFT warnings are still placed in the output regardless.
  
  
  load(cov.name) # load the covariates
  load(meshes.file) # load the pre-created meshes
  load(quad.file) # load grid
  load(covgrid.file) # load aggregated covariates
  load(coord.file) # load coordinates of grid centres
  
  
  # Prior for the Gaussian latent field covariance parameters
  alpha.rho <- 0.01; alpha.sigma <- 0.1; rho.0 <- 0.35; sigma.0 <- 2
  rho.star <- c(rho.0,alpha.rho) ; sigma.star <- c(sigma.0,alpha.sigma)
  
  W <- owin(c(0,5),c(0,5))
  
  pred <- FALSE # Want to predict on the same resolution as the data. UNUSED except for the generation of the grid-mesh ranks for the latent field - in order to get the correct resolution for the data.
  
  # Saving the output
  save.file <- paste0("GridMeshRegPolLGCPSBCSS",k,".rda")
  print(save.file)
  
  n.max <- 50 # (0.1x0.1)
  disc <- matrix(c(10,10,20,20,25,25),ncol=2,byrow=T)
  param <- c("Int","Beta1","Beta2","Sigma","Rho")
  N.grid <- c(disc[,1],n.max); mesh.edge <- (W$xrange[2]-W$xrange[1])/N.grid
  N.g <- length(N.grid); N.m <- length(mesh.edge); N.p <- length(param)
  
  l.dat <- length(N.grid) # want grid centre locations from Grid 1,...,N.g
  final.loc.ind <- l.dat #  use this to produce the gm.ind[[]] list of locations for the rank values for each grid, then they are altered with the relevant length of locations
  
  N.f <- N.grid*N.grid
  
  grid.ind <- paste0("Grid",N.grid); mesh.ind <- paste0("Mesh",round(mesh.edge,3));
  
  
  if (sim==0){
    p.length <- 1
    # Grid-Mesh List Creation 
    gm <- vector(mode="list",length=N.g*N.m)
    names(gm) <- paste0(rep(paste0("Grid",N.grid),each=N.m),rep(paste0("Mesh",round(mesh.edge,3)),N.g))
    rank.list <- list()
    rank.list$ranks.param <- data.frame(matrix(rep(NA,N.p*M.it.total[k]),ncol=N.p))
    names(rank.list$ranks.param) <- param
    rank.list$ranks.mf <- data.frame(matrix(rep(NA,N.f[final.loc.ind]*M.it.total[k]),ncol=N.f[final.loc.ind]))
    names(rank.list$ranks.mf) <- paste0("loc",1:N.f[final.loc.ind])
    gm <- lapply(gm,function(x){x <- rank.list})
    if (pred==FALSE){
      for (i in 1:N.g){
        for (j in 1:N.m){
          gm.ind <- (i-1)*N.m + j
          gm[[gm.ind]]$ranks.mf <- data.frame(matrix(rep(NA,N.f[i]*M.it.total[k]),ncol=N.f[i]))
          names(gm[[gm.ind]]$ranks.mf) <- paste0("loc",1:N.f[i])
        }
      }
    }
    
    # Final Data list
    list.grid <- vector(mode="list",length=N.g)
    names(list.grid) <- grid.ind
    list.mesh <- vector(mode="list",length=N.m)
    names(list.mesh) <- mesh.ind
    list.param <- vector(mode="list",length=3)
    names(list.param) <- c("est.df","run.df","mess.ls")
    list.param$est.df <- data.frame(beta0=rep(NA,M.it.total[k]),beta0.sd=rep(NA,M.it.total[k]),beta0.cil=rep(NA,M.it.total[k]),beta0.ciu=rep(NA,M.it.total[k]),beta1=rep(NA,M.it.total[k]),beta1.sd=rep(NA,M.it.total[k]),beta1.cil=rep(NA,M.it.total[k]),beta1.ciu=rep(NA,M.it.total[k]),beta2=rep(NA,M.it.total[k]),beta2.sd=rep(NA,M.it.total[k]),beta2.cil=rep(NA,M.it.total[k]),beta2.ciu=rep(NA,M.it.total[k]),sigma=rep(NA,M.it.total[k]),sigma.sd=rep(NA,M.it.total[k]),sigma.cil=rep(NA,M.it.total[k]),sigma.ciu=rep(NA,M.it.total[k]),rho=rep(NA,M.it.total[k]),rho.sd=rep(NA,M.it.total[k]),rho.cil=rep(NA,M.it.total[k]),rho.ciu=rep(NA,M.it.total[k]))
    # list.param$run.df <- list(time=rep(NA,M.it.total[k]),cpo=vector(mode="list",length=M.it.total[k]),waic=rep(NA,M.it.total[k]),dic=rep(NA,M.it.total[k])) # if we wanted to include the cpo, use this line rather than the below.
    list.param$run.df <- list(time=rep(NA,M.it.total[k]),waic=rep(NA,M.it.total[k]),dic=rep(NA,M.it.total[k]))
    list.param$mess.ls <- list(error=rep(NA,M.it.total[k]),warning=rep(NA,M.it.total[k]),FFT=rep(NA,M.it.total[k]),message=vector(mode="list",length=M.it.total[k]))
    
    list.meshparam <- lapply(list.mesh,function(x){x <- list.param})
    run.out <- lapply(list.grid,function(x){x <- list.meshparam})
    
    true.theta <- data.frame(beta0=rep(NA,M.it.total[k]),beta1=rep(NA,M.it.total[k]),beta2=rep(NA,M.it.total[k]),sigma=rep(NA,M.it.total[k]),rho=rep(NA,M.it.total[k]))
    
    grid.start.ind <- 1
    mesh.start.ind <- 1
    
    
    data.err.tracker <- data.frame(matrix(vector(), ncol = (length(param)+4), nrow = 0))
    colnames(data.err.tracker) <- c("proc","iter","seed.init","type.err","beta0","beta1","beta2","sigma","rho")
    seed.vec <- rep(NA,M.it.total[k])
  } else{
    load(save.file)
    
    nn <- dim(gm[[N.g*N.m]]$ranks.mf)[2]
    p.length <- sum(!is.na(gm[[N.g*N.m]]$ranks.mf[,nn])) + sum(!is.na(run.out[[N.g]][[N.m]]$mess.ls$error)) + 1 # no ranks in trad ss
    
    # Extract the indices for the next grid and mesh resolution needed to run
    gm.s <- function(g){sapply(1:length(g),function(i){sum(!is.na(g[[i]]$ranks.mf[,dim(g[[i]]$ranks.mf)[2]]))})}
    s <- matrix(gm.s(gm),nrow=N.m) # fills in down the columns, so for each grid, fills in row i with mesh i, following the output from e.s below
    e.s <- function(g){sapply(1:length(g),function(i){sum(!is.na(g[[i]]$mess.ls$error))})}
    s <- s + sapply(1:length(run.out),function(i){e.s(run.out[[i]])})
    ds <- diff(s)
    if (sum(ds)!=0){
      w <- which(ds!=0,arr.ind = T)
      grid.start.ind <- unname(w)[2]
      mesh.start.ind <- unname(w)[1] + 1
    } else if (sum(diff(t(s))!=0)){
      w <- which(diff(t(s))!=0,arr.ind=TRUE) # should be easy to extract the common row, then +1 to get the required GRID that needs to begin running...
      grid.start.ind <- unname(w)[1,1] + 1
      mesh.start.ind <- unname(w)[1,2]
    } else {
      grid.start.ind <- 1
      mesh.start.ind <- 1
    }
  } 
  
  
  for (i in p.length:M.it.total[k]){
    
    if (i!=p.length){ # when we carry on, then we want to re-start at grid 1 and mesh 1 for all other iterations
      grid.start.ind <- 1
      mesh.start.ind <- 1
    }
    
    
    if (grid.start.ind==1 & mesh.start.ind==1){
      # seed <- (k-1)*sum(M.it.total[0:(k-1)]) + i # (k-1) not really necessary but...
      seed <- (k-1)*sum(SEED.M.it.total[0:(k-1)]) + i
      set.seed(5*seed)
      theta.tilde <- prior.sim(1,rho.star,sigma.star) # simulate theta from pi(theta)
      mu.true <- theta.tilde$beta0.tilde*int.im + theta.tilde$beta1.tilde*cov1.im + theta.tilde$beta2.tilde*cov2.im
      data.sim <- data.gen(W=owin(c(0,5),c(0,5)),n.max=n.max,theta=theta.tilde,disc=disc,quads=quad.list,coord=coord.list,area=area.list,ord=ord.list,c1.list=cov1grid.list,c2.list=cov2grid.list,mu.true) # simulate a dataset from pi(y|theta.tilde)
      
      while(data.sim$err==TRUE){
        
        err.count <- nrow(data.err.tracker)
        data.err.tracker[(err.count + 1),1] <- k
        data.err.tracker[(err.count + 1),2] <- i
        data.err.tracker[(err.count + 1),3] <- 5*seed
        if (data.sim$zero==FALSE){
          data.err.tracker[(err.count + 1),4] <- "err"
        } else {
          data.err.tracker[(err.count + 1),4] <- "zero"
        }
        data.err.tracker[(err.count + 1),5:(length(theta.tilde)+4)] <- unlist(theta.tilde)
        
        
        theta.tilde <- prior.sim(n = 1,rho.star = rho.star,sigma.star = sigma.star) # simulate theta from pi(theta)
        mu.true <- cov.surface.gen(W = W,theta = theta.tilde,cov1.im = cov1.im,cov2.im = cov2.im,int.im = int.im)
        data.sim <- data.gen(W=owin(c(0,5),c(0,5)),n.max=n.max,theta=theta.tilde,disc=disc,quads=quad.list,coord=coord.list,area=area.list,ord=ord.list,c1.list=cov1grid.list,c2.list=cov2grid.list,mu.true) # simulate a dataset from pi(y|theta.tilde)
        # 
      }
      save(data.sim,theta.tilde,mu.true,file=paste0("temp_data",k,"sbc.rda"))
      seed.vec[i] <- 5*seed
    } else {
      load(paste0("temp_data",k,"sbc.rda"))
    }
    
    
    true.theta[i,] <- unname(unlist(theta.tilde))
    
    
    lambda.im <- data.sim$lambda
    lambda.fun <- as.function.im(lambda.im,W)
    log.lambda.list <- vector(mode="list",length=length(data.sim$data))
    for (l.ind in 1:length(data.sim$data)){
      final.loc <- data.sim$data[[l.ind]][,c("x","y")]
      llist <- sapply(1:dim(final.loc)[1],function(ll,loc){lambda.fun(loc[ll,1],loc[ll,2])},final.loc)
      
      log.lambda.list[[l.ind]] <- log(llist)
    }
    
    # 
    for (j in grid.start.ind:N.g){
      
      data <- data.sim$data[[j]]
      print(N.grid[j])
      
      if (j==grid.start.ind & i==p.length){
        mesh.start.ind <- mesh.start.ind
      } else {
        mesh.start.ind <- 1
      }
      
      if (j!=grid.start.ind | i!=p.length){
        mesh.start.ind <- 1
      }
      
      for (l in mesh.start.ind:N.m){
        
        mesh <- mesh.list[[l]]
        print(mesh.edge[l])
        ind <- (j-1)*N.m + l
        
        str <- A_stack.gen(data.gm = data,mesh = mesh,sigma.star = sigma.star,rho.star = rho.star)
        start.time <- proc.time()
        fit.inla <- try(inla(resp ~ 0 + int + cov1 + cov2 + offset(larea) + f(field,model=str$spde), family="poisson", data=inla.stack.data(str$stack.est),control.predictor=list(A=inla.stack.A(str$stack.est),link=1,compute=TRUE),control.fixed=list(mean=list(int=2, cov1=1, cov2=-2),prec=list(int=0.25, cov1=0.25, cov2=0.25)),control.compute = list(config=TRUE,waic=TRUE,dic=TRUE))) # ,include cpo=TRUE in control.compute, if needed
        end.time <- proc.time()
        
        
        if (class(fit.inla)=="try-error"){
          # If there is an error, print the value of the offset that caused the error, otherwise, carry on.
          run.out[[j]][[l]]$mess.ls$error[i] <- "ERROR"
        } else if (length(grep('Fail to factorize',fit.inla$logfile)) > fft.threshold) {
          run.out[[j]][[l]]$mess.ls$FFT[i] <- length(grep('Fail to factorize',fit.inla$logfile))
          if (length(grep('WARNING',fit.inla$logfile)) > 0){
            run.out[[j]][[l]]$mess.ls$warning[i] <- "WARNING"
            run.out[[j]][[l]]$mess.ls$message[[i]] <- fit.inla$logfile[(grep('WARNING',fit.inla$logfile))]
          }
        } else if (length(grep('WARNING',fit.inla$logfile)) > 0) {
          run.out[[j]][[l]]$mess.ls$warning[i] <- "WARNING"
          run.out[[j]][[l]]$mess.ls$message[[i]] <- fit.inla$logfile[(grep('WARNING',fit.inla$logfile))]
        } else {
          run.out[[j]][[l]]$mess.ls$FFT[i] <- length(grep('Fail to factorize',fit.inla$logfile)) # incase there were some messages, but below the threshold, want to keep track of any messages.
        }  
        if (class(fit.inla)!="try-error"){
          time.taken <- unname(end.time[3] - start.time[3])
          
          # Put results of approximations into the output data set.
          run.out[[j]][[l]]$run.df$time[i] <- time.taken
          # run.out[[j]][[l]]$run.df$cpo[[i]] <- fit.inla$cpo # Need to tell inla to calculate this, uncomment if wanting to include this, also uncomment the altered version of run.df above
          run.out[[j]][[l]]$run.df$waic[i] <- fit.inla$waic$waic # Need to tell inla to calculate this too!
          run.out[[j]][[l]]$run.df$dic[i] <- fit.inla$dic$dic # Need to tell inla to calculate this too!
          # Posterior Mean
          run.out[[j]][[l]]$est.df$beta0[i] <- fit.inla$summary.fixed$mean[1]
          run.out[[j]][[l]]$est.df$beta1[i] <- fit.inla$summary.fixed$mean[2]
          run.out[[j]][[l]]$est.df$beta2[i] <- fit.inla$summary.fixed$mean[3]
          run.out[[j]][[l]]$est.df$sigma[i] <- fit.inla$summary.hyperpar$mean[2]
          run.out[[j]][[l]]$est.df$rho[i] <- fit.inla$summary.hyperpar$mean[1]
          # Posterior SD
          run.out[[j]][[l]]$est.df$beta0.sd[i] <- fit.inla$summary.fixed$sd[1]
          run.out[[j]][[l]]$est.df$beta1.sd[i] <- fit.inla$summary.fixed$sd[2]
          run.out[[j]][[l]]$est.df$beta2.sd[i] <- fit.inla$summary.fixed$sd[3]
          run.out[[j]][[l]]$est.df$sigma.sd[i] <- fit.inla$summary.hyperpar$sd[2]
          run.out[[j]][[l]]$est.df$rho.sd[i] <- fit.inla$summary.hyperpar$sd[1]
          # Posterior Mode
          run.out[[j]][[l]]$est.df$sigma.mode[i] <- fit.inla$summary.hyperpar$mode[2]
          run.out[[j]][[l]]$est.df$rho.mode[i] <- fit.inla$summary.hyperpar$mode[1]
          # Posterior 2.5%
          run.out[[j]][[l]]$est.df$beta0.cil[i] <- fit.inla$summary.fixed$`0.025quant`[1]
          run.out[[j]][[l]]$est.df$beta1.cil[i] <- fit.inla$summary.fixed$`0.025quant`[2]
          run.out[[j]][[l]]$est.df$beta2.cil[i] <- fit.inla$summary.fixed$`0.025quant`[3]
          run.out[[j]][[l]]$est.df$sigma.cil[i] <- fit.inla$summary.hyperpar$`0.025quant`[2]
          run.out[[j]][[l]]$est.df$rho.cil[i] <- fit.inla$summary.hyperpar$`0.025quant`[1]
          # Posterior 97.5%
          run.out[[j]][[l]]$est.df$beta0.ciu[i] <- fit.inla$summary.fixed$`0.975quant`[1]
          run.out[[j]][[l]]$est.df$beta1.ciu[i] <- fit.inla$summary.fixed$`0.975quant`[2]
          run.out[[j]][[l]]$est.df$beta2.ciu[i] <- fit.inla$summary.fixed$`0.975quant`[3]
          run.out[[j]][[l]]$est.df$sigma.ciu[i] <- fit.inla$summary.hyperpar$`0.975quant`[2]
          run.out[[j]][[l]]$est.df$rho.ciu[i] <- fit.inla$summary.hyperpar$`0.975quant`[1]
          
          
          # SBC
          
          # Labelling for the latent field and parameter output samples to select from posterior sample function outputs
          contents <- fit.inla$misc$configs$contents
          # int indices
          id.int <- which(contents$tag=="int")
          ind.int <- contents$start[id.int] - 1 + (1:contents$length[id.int])
          # beta1 indices
          id.cov1 <- which(contents$tag=="cov1")
          ind.cov1 <- contents$start[id.cov1] - 1 + (1:contents$length[id.cov1])
          # beta2 indices
          id.cov2 <- which(contents$tag=="cov2")
          ind.cov2 <- contents$start[id.cov2] - 1 + (1:contents$length[id.cov2])
          
          theta.K <- inla.posterior.sample(L,fit.inla)
          thetahyperpar.K <- inla.hyperpar.sample(L,fit.inla)
          sum.int <- 0; sum.beta1 <- 0; sum.beta2 <- 0
          
          for (jj in 1:L){
            sum.int <- sum.int + as.numeric(theta.K[[jj]]$latent[ind.int,1] < theta.tilde$beta0.tilde)
            sum.beta1 <- sum.beta1 + as.numeric(theta.K[[jj]]$latent[ind.cov1,1] < theta.tilde$beta1.tilde)
            sum.beta2 <- sum.beta2 + as.numeric(theta.K[[jj]]$latent[ind.cov2,1] < theta.tilde$beta2.tilde)
          }
          
          gm[[ind]]$ranks.param[i,1] <- sum.int
          gm[[ind]]$ranks.param[i,2] <- sum.beta1
          gm[[ind]]$ranks.param[i,3] <- sum.beta2
          gm[[ind]]$ranks.param[i,4] <- sum(thetahyperpar.K[,2] < theta.tilde$sigma.tilde)
          gm[[ind]]$ranks.param[i,5] <- sum(thetahyperpar.K[,1] < theta.tilde$rho.tilde)
          
          
          # Using same grid resolution for data in model fit and for the mean field.
          mf.effect <- "APredictor";
          pred.ind <- inla.stack.index(str$stack.est, tag='est')$data
          
          log.lambda.true <- log.lambda.list[[j]]
          
          id.mfeffect <- which(contents$tag==mf.effect)
          ind.mfeffect <- (contents$start[id.mfeffect] - 1 + (1:contents$length[id.mfeffect]))[pred.ind]
          
          # NOTE: the below code has been changed after the simulation study was run, this is because a mistake was later found: the x$latent[ind.meffect] produces the linear predictor which is not what we have in log.lambda.true. The difference between the two is the inclusion of the offset of the log cell area in the lienar predictor (which is negative due to cell areas<1), however that can be easily rectified.
          # The commented out code for gm[[ind]]$ranks.mf[i,] is the code with the TYPO, and the un-commented code is the corrected version (which can be approached either via adding the log-areas to the rhs (loglambda) or subtracting it from the lhs (lin.predictor))
          # gm[[ind]]$ranks.mf[i,] <- rowSums(sapply(theta.K,function(x){x$latent[ind.mfeffect]<log.lambda.true})) # TYPO
          gm[[ind]]$ranks.mf[i,] <- rowSums(sapply(theta.K,function(x){x$latent[ind.mfeffect]<log.lambda.true + log(data$area)}))
        }
        save(run.out,gm,true.theta,data.err.tracker,seed.vec,file=save.file)
      }
    }
  }
  save(run.out,gm,true.theta,data.err.tracker,seed.vec,file=save.file)
  
}
# Stop the clock
print(proc.time() - ptm)

stopCluster(parallelCluster)
#################################################################################################

sessionInfo()

for (tt in ind.procs){
  tdirloc.k <- paste0("/local/intmpproc-",tt)
  print(paste0("Unlinking Temp Directory ", tdirloc.k))
  unlink(tdirloc.k,recursive=T)
}


#Define arrays for storing result
rm(list=ls()) # Must finish with this.

