
# Testing Priors for Regular Polygon LGCP ---------------------------------

# This R script will test the number of points simulated as well as the proportion of cells with zero counts for varying priors for the fixed effects over the Regular Polygon.

# Author: Nadeen Khaleel


# Setwd and Load Libraries ------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))

library(doParallel)
library(foreach)


# Set-Up ------------------------------------------------------------------

# Parameters for priors,the out i next to the hyperparmeter values matches the i in the output file name

# mean.beta0 <- c(1); mean.beta1 <- c(2); mean.beta2 <- c(-2) # out 1
# sd.beta0 <- c(2,4); sd.beta1 <- c(2,4); sd.beta2 <- c(2,4)
# sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 1

# mean.beta0 <- c(1); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 2
# sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
# sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 2

# mean.beta0 <- c(1,2); mean.beta1 <- c(1,2); mean.beta2 <- c(-1,2) # out 3
# sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
# sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 3

# mean.beta0 <- c(1,2); mean.beta1 <- c(1,2); mean.beta2 <- c(-1,-2) # out 4
# sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
# sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 4

# # Out 4 long test
mean.beta0 <- c(1,2); mean.beta1 <- c(1); mean.beta2 <- c(-2) # out 4 long
sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
sigma.0 <- c(2); alpha.sigma <- c(0.1) # out 4 long

# Set up list of parameters to generate the combinations
param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)

# Create the combinations to test of the above prior parameters
param.combos <- expand.grid(param.list)

# Set-up parallelisation
x <- 1:nrow(param.combos)
Nproc <- 4 # however many processes you would like to split these simulations across
cl <- makeCluster(Nproc)
registerDoParallel(cl)
indexes <- split(x, sort(rep_len(1:Nproc, length(x))))


# Implement Simulations ---------------------------------------------------

ptm.start <- proc.time()
foreach(i=1:Nproc) %dopar% {
  
  library(spatstat)
  library(raster)
  library(sf)
  library(sp)
  library(maptools)
  library(rgdal)
  library(lwgeom)
  library(RandomFields)
  
  
  # Set-up ------------------------------------------------------------------
  # Need to set-up the data again for the each core - due to the parallelisation, we cannot fork so we have to set- things up again, make sure to match the parameters!
  
  # Create window
  W <- owin(c(0,5),c(0,5))
  
  # Get covariates
  cov.name <- paste0("GridMeshRegPolLGCPSSCov.rda")
  load(cov.name)
  
  # Need to include parameters again - make sure they match! the out i next to the hyperparmeter values matches the i in the output file name
  # mean.beta0 <- c(1); mean.beta1 <- c(2); mean.beta2 <- c(-2) # out 1
  # sd.beta0 <- c(2,4); sd.beta1 <- c(2,4); sd.beta2 <- c(2,4)
  # sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 1
  # 
  # mean.beta0 <- c(1); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 2
  # sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
  # sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 2
  
  # mean.beta0 <- c(1,2); mean.beta1 <- c(1,2); mean.beta2 <- c(-1,2) # out 3
  # sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
  # sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 3
  
  
  # mean.beta0 <- c(1,2); mean.beta1 <- c(1,2); mean.beta2 <- c(-1,-2) # out 4
  # sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
  # sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 4
  
  # # Out 4 long test
  mean.beta0 <- c(1,2); mean.beta1 <- c(1); mean.beta2 <- c(-2) # out 4 long
  sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
  sigma.0 <- c(2); alpha.sigma <- c(0.1) # out 4 long

  param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)
  
  # Create combinations again
  param.combos <- expand.grid(param.list)
  
  load("QuadratsRegPolLGCP.rda")
  g <- quad.list
  
  # N <- 50 # number of simulations for standard information seeking test
  N <- 100 # number of simulations for long test, with output saved under *_LongTest.rda
  
  # Change output filename as needed, don't forget to change accordingly in the rest of the code!
  out.filename <- "RegPolCovFixedEffectsPrior4_LongTest.rda"
  if ((!file.exists(out.filename)) & (i==1)){
    out <- vector(mode = "list", length = nrow(param.combos))
    save(out,file=out.filename)
  }
  
  # Functions ---------------------------------------------------------------
  
  prior.sim <- function(n=1,mu.beta0,sigma.beta0,mu.beta1,sigma.beta1,mu.beta2,sigma.beta2,sigma.0,alpha.sigma){
    # n is the number of samples of the parameters you want sampled from their priors.  
    # mu.*: prior mean for fixed effects
    # sigma.beta*: standard deviation for fixed effects
    # sigma.0, alpha.sigma: prior parameters for sigma pc prior
    
    beta.0.sim <- rnorm(n,mu.beta0,sigma.beta0); beta.1.sim <- rnorm(n,mu.beta1,sigma.beta1); beta.2.sim <- rnorm(n,mu.beta2,sigma.beta2)
    
    # For the covariance parameters of the latent Gaussian field, I have a joint prior on the marginal standard deviation and the range, however, in order to simulate from this I need the inverse CDF to generate this from a Uniform RV. We can also use the priors fpr kappa and tau given kappa to generate the inverses.
    # (https://www.tandfonline.com/doi/pdf/10.1080/01621459.2017.1415907?needAccess=true) 
    # (https://projecteuclid.org/journals/statistical-science/volume-32/issue-1/Penalising-Model-Component-Complexity--A-Principled-Practical-Approach-to/10.1214/16-STS576.full)
    
    alpha.rho <- 0.01; rho.0 <- 0.35;
    u1 <- runif(n,0,1); u2 <- runif(n,0,1)
    rho.sim <- rho.0*log(alpha.rho)/log(u1)
    sigma.sim <- sigma.0*log(1-u2)/log(alpha.sigma)
    return(list(beta0.tilde=beta.0.sim,beta1.tilde=beta.1.sim,beta2.tilde=beta.2.sim,sigma.tilde=sigma.sim,rho.tilde=rho.sim))
  }
  
  res.gen <- function(N,W,prior.param,g.tess,c1.im,c2.im,i.im,seed){
    # N: number of data sets to simulate
    # W: the window
    # prior.param: the set of hyper-parameters for the priors ro simulate the parameter for the generation of the LGCP
    # g.tess: different grid resolutions
    # i.im: pixel image for intercept for data generation
    # ci.im: pixel image for covariates for data simulation
    # seed: set the seed for the data simulations
    
    set.seed(seed)
    # simulate the parameters
    thetas <- prior.sim(n=N,mu.beta0 = prior.param$mu.beta0,sigma.beta0 = prior.param$sd.beta0,mu.beta1 = prior.param$mu.beta1,sigma.beta1 = prior.param$sd.beta1,mu.beta2 = prior.param$mu.beta2,sigma.beta2 = prior.param$sd.beta2,sigma.0=prior.param$sigma.0,alpha.sigma = prior.param$alpha.sigma)
    
    # Create output storage for lgcp.ppp$n and proportion of zeros for different grids.
    grid.names <- names(g.tess)
    output.df <- data.frame(matrix(rep(NA,(length(grid.names)+3)*N),nrow=N))
    colnames(output.df) <- c("n","max.lambda","med.lambda",grid.names)
    
    # Simulated N LGCPs
    for (j in 1:N){
      print(paste0("Simulation ", j))
      # Extract parameter values simulated from priors
      theta <- lapply(1:length(names(thetas)),function(k,j){thetas[[k]][j]},j)
      names(theta) <- names(thetas)
      beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]]; sigma <- theta[[4]]; rho <- theta[[5]];
      
      mu <- beta.0*i.im + beta.1*c1.im + beta.2*c2.im
      # Produce the point pattern from LGCP
      lgcp.ppp <- try(rLGCP(model="matern",mu,var=(sigma)^2,scale=rho/2,nu=1,win = W,saveLambda = TRUE))
      
      if (class(lgcp.ppp)!="try-error"){
        output.df$n[j] <- lgcp.ppp$n
        att.ppp <- attributes(lgcp.ppp)
        output.df$max.lambda[j] <- max(att.ppp$Lambda)
        output.df$med.lambda[j] <- median(att.ppp$Lambda)
        
        for (i in 1:length(g.tess)){
          col.df <- grid.names[i] # for the grid proportion of zeros in the output.df need to account for the "n" column
          q <- quadratcount(lgcp.ppp,tess=g.tess[[i]])
          count.df <- data.frame(count=as.vector(unname(q)))
          
          output.df[j,col.df] <- sum(count.df$count==0)/length(count.df$count)
        }
      }
      
    }
    
    return(list("results"=output.df))
  }
  
  
  # Run All -----------------------------------------------------------------
  
  seeds <- 5*(1:nrow(param.combos))
  
  # Change the file name below accordingly
  out.filename <- "RegPolCovFixedEffectsPrior4_LongTest.rda"
  load(out.filename)
  
  x <- which(unlist(sapply(1:length(out),function(i){(is.null(out[[i]]))}))) # for re-setting
  Nproc <- 4 # make sure this matches all other values of Nproc above
  indexes <- split(x, sort(rep_len(1:Nproc, length(x))))
  
  for (k in indexes[[i]]){
    print(k)
    out.new <- res.gen(N,W,param.combos[k,],g.tess=g,cov1.im,cov2.im,int.im,seeds[k])
    load(out.filename)
    out[[k]] <- out.new
    save(out,file=out.filename)
  }
}
ptm.end <- proc.time()
stopCluster(cl)
ptm.end-ptm.start

# Change file name accordingly
out.filename <- "RegPolCovFixedEffectsPrior4_LongTest.rda" # use of *_LongTest.rda indicates that we considered N=100 simulations

# Get the file and label the outputs so that we can more easily tell which parameters resulted in which results
load(out.filename)
combo.names <- sapply(1:nrow(param.combos),function(i){paste0("(",paste0(param.combos[i,],collapse=","),")")})
names(out) <- combo.names

# Save labelled output
save(out,file=out.filename)



# Check Output ------------------------------------------------------------
# Load outputs and generate tables, these can then be placed in a .tex file for easy reading

# # Run 1
# load("RegPolCovFixedEffectsPrior1.rda")
# mean.beta0 <- c(1); mean.beta1 <- c(2); mean.beta2 <- c(-2) # out 1
# sd.beta0 <- c(2,4); sd.beta1 <- c(2,4); sd.beta2 <- c(2,4)
# sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 1
# 
# param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)
# 
# param.combos <- expand.grid(param.list)
# 
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero1010 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad1010,na.rm=TRUE)})
# df$prop.zero2020 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2020,na.rm=TRUE)})
# df$prop.zero2525 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2525,na.rm=TRUE)})
# df$prop.zero5050 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad5050,na.rm=TRUE)})
# 
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Sat Jun 12 00:51:17 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero1010 & prop.zero2020 & prop.zero2525 & prop.zero5050 \\ 
# # \hline
# # 1 &   1 &   2 &   2 &   2 &  -2 &   2 &   2 & 0.01 & $2.6 \times 10^{4}$ & $8.2 \times 10^{2}$ &   0 & 0.43 & $5.1 \times 10^{6}$ &  40 & $1.1 \times 10^{6}$ & 0.27 & 0.45 & 0.51 & 0.69 \\ 
# # 2 &   1 &   4 &   2 &   2 &  -2 &   2 &   2 & 0.01 & $3.4 \times 10^{4}$ & $1.2 \times 10^{2}$ &   0 & 0.43 & $2.5 \times 10^{6}$ &  54 & $5.8 \times 10^{5}$ & 0.35 & 0.49 & 0.53 & 0.65 \\ 
# # 3 &   1 &   2 &   2 &   4 &  -2 &   2 &   2 & 0.01 & $3.7 \times 10^{6}$ & $5.1 \times 10^{2}$ &   5 & 0.43 & $1.8 \times 10^{9}$ &  60 & $1.8 \times 10^{8}$ & 0.33 & 0.51 & 0.55 & 0.7 \\ 
# # 4 &   1 &   4 &   2 &   4 &  -2 &   2 &   2 & 0.01 & $1.1 \times 10^{7}$ & $4 \times 10^{2}$ &   4 & 0.43 & $3.8 \times 10^{9}$ & $7.1 \times 10^{2}$ & $2.8 \times 10^{7}$ & 0.32 & 0.44 & 0.48 & 0.59 \\ 
# # 5 &   1 &   2 &   2 &   2 &  -2 &   4 &   2 & 0.01 & $3.8 \times 10^{5}$ & $5.4 \times 10^{2}$ &   0 & 0.43 & $3.1 \times 10^{7}$ &  69 & $3.6 \times 10^{6}$ & 0.34 & 0.51 & 0.56 & 0.73 \\ 
# # 6 &   1 &   4 &   2 &   2 &  -2 &   4 &   2 & 0.01 & $2.6 \times 10^{6}$ & $1.6 \times 10^{3}$ &   1 & 0.43 & $7.9 \times 10^{8}$ & $5.7 \times 10^{2}$ & $1 \times 10^{8}$ & 0.32 & 0.44 & 0.48 & 0.61 \\ 
# # 7 &   1 &   2 &   2 &   4 &  -2 &   4 &   2 & 0.01 & $5.2 \times 10^{5}$ & $9.1 \times 10^{3}$ &   8 & 0.43 & $1.8 \times 10^{8}$ & $2.8 \times 10^{2}$ & $4.6 \times 10^{7}$ & 0.13 & 0.27 & 0.32 & 0.54 \\ 
# # 8 &   1 &   4 &   2 &   4 &  -2 &   4 &   2 & 0.01 & $6 \times 10^{6}$ & $6.1 \times 10^{3}$ &   7 & 0.43 & $3.6 \times 10^{9}$ & $3.2 \times 10^{2}$ & $3 \times 10^{8}$ & 0.18 & 0.29 & 0.33 & 0.49 \\ 
# # 9 &   1 &   2 &   2 &   2 &  -2 &   2 &   5 & 0.01 & $7.7 \times 10^{5}$ & $1.3 \times 10^{3}$ &   1 & 1.1 & $1.6 \times 10^{8}$ & $1.6 \times 10^{2}$ & $1.2 \times 10^{6}$ & 0.33 & 0.48 & 0.53 & 0.69 \\ 
# # 10 &   1 &   4 &   2 &   2 &  -2 &   2 &   5 & 0.01 & $9.2 \times 10^{4}$ & $4 \times 10^{2}$ &   0 & 1.1 & $7.9 \times 10^{6}$ & $1.8 \times 10^{3}$ & $4.6 \times 10^{5}$ & 0.33 & 0.5 & 0.56 & 0.72 \\ 
# # 11 &   1 &   2 &   2 &   4 &  -2 &   2 &   5 & 0.01 & $6.2 \times 10^{4}$ & $5.7 \times 10^{2}$ &   6 & 1.1 & $2.2 \times 10^{7}$ &  31 & $1.5 \times 10^{7}$ & 0.29 & 0.5 & 0.57 & 0.77 \\ 
# # 12 &   1 &   4 &   2 &   4 &  -2 &   2 &   5 & 0.01 & $2.1 \times 10^{6}$ & $4.4 \times 10^{3}$ &   5 & 1.1 & $4.1 \times 10^{8}$ & $1.5 \times 10^{3}$ & $9.7 \times 10^{8}$ & 0.19 & 0.3 & 0.35 & 0.51 \\ 
# # 13 &   1 &   2 &   2 &   2 &  -2 &   4 &   5 & 0.01 & $1.6 \times 10^{6}$ & $1.2 \times 10^{3}$ &   3 & 1.1 & $8.3 \times 10^{8}$ &  92 & $4.2 \times 10^{8}$ & 0.3 & 0.45 & 0.5 & 0.66 \\ 
# # 14 &   1 &   4 &   2 &   2 &  -2 &   4 &   5 & 0.01 & $3.7 \times 10^{6}$ & $7.8 \times 10^{2}$ &   2 & 1.1 & $2.1 \times 10^{9}$ & $1.3 \times 10^{2}$ & $4.6 \times 10^{8}$ & 0.34 & 0.49 & 0.53 & 0.67 \\ 
# # 15 &   1 &   2 &   2 &   4 &  -2 &   4 &   5 & 0.01 & $1.8 \times 10^{5}$ & $1.5 \times 10^{3}$ &   6 & 1.1 & $2.3 \times 10^{7}$ &  19 & $2.4 \times 10^{7}$ & 0.24 & 0.44 & 0.51 & 0.69 \\ 
# # 16 &   1 &   4 &   2 &   4 &  -2 &   4 &   5 & 0.01 & $1.6 \times 10^{6}$ & $4.6 \times 10^{3}$ &   5 & 1.1 & $6.7 \times 10^{8}$ & $2 \times 10^{2}$ & $8.8 \times 10^{8}$ & 0.17 & 0.3 & 0.36 & 0.55 \\ 
# # 17 &   1 &   2 &   2 &   2 &  -2 &   2 &   2 & 0.1 & $1.8 \times 10^{4}$ & $1.5 \times 10^{2}$ &   0 & 0.87 & $2.3 \times 10^{6}$ &  14 & $8.1 \times 10^{4}$ & 0.36 & 0.58 & 0.64 & 0.78 \\ 
# # 18 &   1 &   4 &   2 &   2 &  -2 &   2 &   2 & 0.1 & $4.9 \times 10^{6}$ & $5.8 \times 10^{2}$ &   0 & 0.87 & $9.3 \times 10^{8}$ & $1.7 \times 10^{4}$ & $1 \times 10^{8}$ & 0.3 & 0.44 & 0.47 & 0.58 \\ 
# # 19 &   1 &   2 &   2 &   4 &  -2 &   2 &   2 & 0.1 & $1.8 \times 10^{7}$ & $1.5 \times 10^{4}$ &   4 & 0.87 & $9.3 \times 10^{9}$ & $2.3 \times 10^{2}$ & $2.7 \times 10^{8}$ & 0.22 & 0.33 & 0.37 & 0.51 \\ 
# # 20 &   1 &   4 &   2 &   4 &  -2 &   2 &   2 & 0.1 & $4.6 \times 10^{6}$ & $5.1 \times 10^{2}$ &   7 & 0.87 & $9.6 \times 10^{8}$ & $1.7 \times 10^{4}$ & $4.6 \times 10^{7}$ & 0.32 & 0.48 & 0.53 & 0.67 \\ 
# # 21 &   1 &   2 &   2 &   2 &  -2 &   4 &   2 & 0.1 & $7.4 \times 10^{5}$ & $2.1 \times 10^{3}$ &   0 & 0.87 & $5.9 \times 10^{7}$ & $2.9 \times 10^{2}$ & $2 \times 10^{8}$ & 0.24 & 0.4 & 0.45 & 0.64 \\ 
# # 22 &   1 &   4 &   2 &   2 &  -2 &   4 &   2 & 0.1 & $2.2 \times 10^{6}$ & $8.2 \times 10^{3}$ &   0 & 0.87 & $7.9 \times 10^{8}$ & $3.1 \times 10^{2}$ & $7.1 \times 10^{7}$ & 0.18 & 0.3 & 0.34 & 0.48 \\ 
# # 23 &   1 &   2 &   2 &   4 &  -2 &   4 &   2 & 0.1 & $7.5 \times 10^{6}$ & $6.7 \times 10^{3}$ &   6 & 0.87 & $4.2 \times 10^{9}$ &  80 & $1 \times 10^{10}$ & 0.18 & 0.31 & 0.36 & 0.54 \\ 
# # 24 &   1 &   4 &   2 &   4 &  -2 &   4 &   2 & 0.1 & $9.6 \times 10^{6}$ & $1.7 \times 10^{4}$ &   3 & 0.87 & $2.7 \times 10^{9}$ & $1 \times 10^{4}$ & $5.4 \times 10^{9}$ & 0.18 & 0.32 & 0.36 & 0.54 \\ 
# # 25 &   1 &   2 &   2 &   2 &  -2 &   2 &   5 & 0.1 & $6.9 \times 10^{6}$ & $1.9 \times 10^{3}$ &   2 & 2.2 & $1.9 \times 10^{9}$ &  67 & $4 \times 10^{7}$ & 0.27 & 0.43 & 0.49 & 0.66 \\ 
# # 26 &   1 &   4 &   2 &   2 &  -2 &   2 &   5 & 0.1 & $8.3 \times 10^{6}$ & $1.4 \times 10^{3}$ &   5 & 2.2 & $1.3 \times 10^{9}$ & $1.2 \times 10^{5}$ & $2 \times 10^{8}$ & 0.31 & 0.47 & 0.51 & 0.63 \\ 
# # 27 &   1 &   2 &   2 &   4 &  -2 &   2 &   5 & 0.1 & $1.7 \times 10^{6}$ & $1.8 \times 10^{3}$ &   5 & 2.2 & $9.2 \times 10^{8}$ &  14 & $8.4 \times 10^{7}$ & 0.28 & 0.45 & 0.51 & 0.69 \\ 
# # 28 &   1 &   4 &   2 &   4 &  -2 &   2 &   5 & 0.1 & $7.9 \times 10^{6}$ & $8.1 \times 10^{3}$ &   4 & 2.2 & $1.8 \times 10^{9}$ & $3.2 \times 10^{2}$ & $2.3 \times 10^{8}$ & 0.18 & 0.3 & 0.34 & 0.49 \\ 
# # 29 &   1 &   2 &   2 &   2 &  -2 &   4 &   5 & 0.1 & $5 \times 10^{6}$ & $2.4 \times 10^{4}$ &   4 & 2.2 & $8.9 \times 10^{8}$ &  63 & $6.4 \times 10^{9}$ & 0.14 & 0.28 & 0.34 & 0.55 \\ 
# # 30 &   1 &   4 &   2 &   2 &  -2 &   4 &   5 & 0.1 & $1.6 \times 10^{7}$ & $7.9 \times 10^{3}$ &   3 & 2.2 & $1.4 \times 10^{9}$ & $2.2 \times 10^{4}$ & $1.9 \times 10^{9}$ & 0.21 & 0.31 & 0.35 & 0.49 \\ 
# # 31 &   1 &   2 &   2 &   4 &  -2 &   4 &   5 & 0.1 & $2.5 \times 10^{7}$ & $8.1 \times 10^{3}$ &   4 & 2.2 & $7.4 \times 10^{9}$ & $1.1 \times 10^{2}$ & $1.4 \times 10^{9}$ & 0.25 & 0.41 & 0.46 & 0.61 \\ 
# # 32 &   1 &   4 &   2 &   4 &  -2 &   4 &   5 & 0.1 & $2.2 \times 10^{7}$ & $2.2 \times 10^{4}$ &   7 & 2.2 & $4.6 \times 10^{9}$ & $2.3 \times 10^{2}$ & $1.3 \times 10^{9}$ & 0.27 & 0.38 & 0.42 & 0.57 \\ 
# # \hline
# # \end{tabular}
# # \end{table}



# # Run 2
# load("RegPolCovFixedEffectsPrior2.rda")
# 
# mean.beta0 <- c(1); mean.beta1 <- c(1); mean.beta2 <- c(-1) # out 2
# sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
# sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 2
# 
# param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)
# 
# param.combos <- expand.grid(param.list)
# 
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero1010 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad1010,na.rm=TRUE)})
# df$prop.zero2020 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2020,na.rm=TRUE)})
# df$prop.zero2525 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2525,na.rm=TRUE)})
# df$prop.zero5050 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad5050,na.rm=TRUE)})
# 
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Fri Jun 11 21:55:26 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero1010 & prop.zero2020 & prop.zero2525 & prop.zero5050 \\ 
# # \hline
# # 1 &   1 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.01 & $7.1 \times 10^{3}$ & $3.4 \times 10^{2}$ &   0 & 0.43 & $2.6 \times 10^{5}$ &  42 & $4 \times 10^{4}$ & 0.36 & 0.52 & 0.56 & 0.73 \\ 
# # 2 &   1 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.01 & $1.7 \times 10^{4}$ & $1.7 \times 10^{2}$ &   0 & 1.1 & $7.6 \times 10^{5}$ &  14 & $1.8 \times 10^{5}$ & 0.4 & 0.59 & 0.65 & 0.8 \\ 
# # 3 &   1 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.1 & $6.1 \times 10^{5}$ & $5.6 \times 10^{2}$ &   0 & 0.87 & $2.6 \times 10^{8}$ &  49 & $2.4 \times 10^{6}$ & 0.33 & 0.52 & 0.58 & 0.72 \\ 
# # 4 &   1 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.1 & $2.5 \times 10^{6}$ & $3.3 \times 10^{3}$ &   1 & 2.2 & $2.3 \times 10^{8}$ &  38 & $8.8 \times 10^{6}$ & 0.37 & 0.52 & 0.56 & 0.7 \\ 
# # \hline
# # \end{tabular}
# # \end{table}

# # # Run 3
# load("RegPolCovFixedEffectsPrior3.rda")
# 
# mean.beta0 <- c(1,2); mean.beta1 <- c(1,2); mean.beta2 <- c(-1,2) # out 3
# sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
# sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 3
# 
# param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)
# 
# param.combos <- expand.grid(param.list)
# 
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero1010 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad1010,na.rm=TRUE)})
# df$prop.zero2020 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2020,na.rm=TRUE)})
# df$prop.zero2525 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2525,na.rm=TRUE)})
# df$prop.zero5050 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad5050,na.rm=TRUE)})
# 
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Fri Jun 11 23:37:03 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero1010 & prop.zero2020 & prop.zero2525 & prop.zero5050 \\ 
# # \hline
# # 1 &   1 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.01 & $7.1 \times 10^{3}$ & $3.4 \times 10^{2}$ &   0 & 0.43 & $2.6 \times 10^{5}$ &  42 & $4 \times 10^{4}$ & 0.36 & 0.52 & 0.56 & 0.73 \\ 
# # 2 &   2 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.01 & $9.4 \times 10^{3}$ & $4.7 \times 10^{2}$ &   0 & 0.43 & $3 \times 10^{5}$ &  36 & $2.1 \times 10^{4}$ & 0.28 & 0.48 & 0.54 & 0.74 \\ 
# # 3 &   1 &   2 &   2 &   2 &  -1 &   2 &   2 & 0.01 & $2.9 \times 10^{5}$ & $5.1 \times 10^{2}$ &   1 & 0.43 & $8.3 \times 10^{7}$ &  79 & $4.3 \times 10^{6}$ & 0.25 & 0.45 & 0.51 & 0.69 \\ 
# # 4 &   2 &   2 &   2 &   2 &  -1 &   2 &   2 & 0.01 & $1.1 \times 10^{5}$ & $1.6 \times 10^{3}$ &   1 & 0.43 & $9.3 \times 10^{6}$ & $1.4 \times 10^{2}$ & $3.7 \times 10^{5}$ & 0.18 & 0.33 & 0.38 & 0.56 \\ 
# # 5 &   1 &   2 &   1 &   2 &   2 &   2 &   2 & 0.01 & $7 \times 10^{5}$ & $5.4 \times 10^{2}$ &   0 & 0.43 & $1.4 \times 10^{8}$ & $1 \times 10^{2}$ & $3.2 \times 10^{5}$ & 0.28 & 0.44 & 0.49 & 0.66 \\ 
# # 6 &   2 &   2 &   1 &   2 &   2 &   2 &   2 & 0.01 & $2.8 \times 10^{5}$ & $1.3 \times 10^{3}$ &   3 & 0.43 & $4.3 \times 10^{7}$ & $1.2 \times 10^{2}$ & $2.7 \times 10^{5}$ & 0.18 & 0.35 & 0.41 & 0.6 \\ 
# # 7 &   1 &   2 &   2 &   2 &   2 &   2 &   2 & 0.01 & $1.4 \times 10^{6}$ & $3.7 \times 10^{4}$ &   2 & 0.43 & $2.5 \times 10^{8}$ & $1.2 \times 10^{3}$ & $1.5 \times 10^{6}$ & 0.056 & 0.13 & 0.16 & 0.33 \\ 
# # 8 &   2 &   2 &   2 &   2 &   2 &   2 &   2 & 0.01 & $5.8 \times 10^{5}$ & $3 \times 10^{4}$ &   2 & 0.43 & $1.5 \times 10^{8}$ & $3.2 \times 10^{2}$ & $1.3 \times 10^{6}$ & 0.059 & 0.13 & 0.16 & 0.32 \\ 
# # 9 &   1 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.01 & $1.4 \times 10^{5}$ & $9.1 \times 10^{2}$ &   1 & 1.1 & $1.5 \times 10^{7}$ & $1.2 \times 10^{2}$ & $8.4 \times 10^{4}$ & 0.29 & 0.46 & 0.52 & 0.68 \\ 
# # 10 &   2 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.01 & $1 \times 10^{4}$ & $6.3 \times 10^{2}$ &   0 & 1.1 & $5.3 \times 10^{5}$ &  53 & $5.5 \times 10^{4}$ & 0.26 & 0.47 & 0.54 & 0.72 \\ 
# # 11 &   1 &   2 &   2 &   2 &  -1 &   2 &   5 & 0.01 & $9.5 \times 10^{5}$ & $5.4 \times 10^{2}$ &   1 & 1.1 & $3 \times 10^{8}$ &  37 & $1.3 \times 10^{8}$ & 0.26 & 0.46 & 0.53 & 0.72 \\ 
# # 12 &   2 &   2 &   2 &   2 &  -1 &   2 &   5 & 0.01 & $9.3 \times 10^{5}$ & $3.9 \times 10^{3}$ &   0 & 1.1 & $9.5 \times 10^{7}$ & $2.2 \times 10^{2}$ & $4.4 \times 10^{7}$ & 0.14 & 0.26 & 0.32 & 0.52 \\ 
# # 13 &   1 &   2 &   1 &   2 &   2 &   2 &   5 & 0.01 & $7.8 \times 10^{5}$ & $5.4 \times 10^{3}$ &   2 & 1.1 & $9.1 \times 10^{7}$ & $1.8 \times 10^{2}$ & $2.6 \times 10^{6}$ & 0.24 & 0.35 & 0.4 & 0.55 \\ 
# # 14 &   2 &   2 &   1 &   2 &   2 &   2 &   5 & 0.01 & $1.3 \times 10^{6}$ & $2 \times 10^{3}$ &   1 & 1.1 & $2.5 \times 10^{8}$ & $2.1 \times 10^{2}$ & $5.1 \times 10^{5}$ & 0.1 & 0.27 & 0.33 & 0.53 \\ 
# # 15 &   1 &   2 &   2 &   2 &   2 &   2 &   5 & 0.01 & $2.4 \times 10^{6}$ & $7.3 \times 10^{3}$ &   0 & 1.1 & $5 \times 10^{8}$ & $1.6 \times 10^{2}$ & $2.7 \times 10^{6}$ & 0.13 & 0.27 & 0.31 & 0.5 \\ 
# # 16 &   2 &   2 &   2 &   2 &   2 &   2 &   5 & 0.01 & $6.2 \times 10^{5}$ & $1.3 \times 10^{4}$ &   3 & 1.1 & $1.4 \times 10^{8}$ & $2.9 \times 10^{2}$ & $4.1 \times 10^{5}$ & 0.084 & 0.17 & 0.2 & 0.38 \\ 
# # 17 &   1 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.1 & $4.3 \times 10^{3}$ & $1.5 \times 10^{2}$ &   0 & 0.87 & $1 \times 10^{5}$ &  21 & $5.9 \times 10^{4}$ & 0.44 & 0.64 & 0.69 & 0.83 \\ 
# # 18 &   2 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.1 & $5.7 \times 10^{5}$ & $3.1 \times 10^{2}$ &   1 & 0.87 & $1.9 \times 10^{8}$ & $3.4 \times 10^{2}$ & $2 \times 10^{7}$ & 0.27 & 0.44 & 0.5 & 0.65 \\ 
# # 19 &   1 &   2 &   2 &   2 &  -1 &   2 &   2 & 0.1 & $1.7 \times 10^{5}$ & $5.1 \times 10^{3}$ &   0 & 0.87 & $9.7 \times 10^{6}$ & $6.7 \times 10^{2}$ & $5.2 \times 10^{6}$ & 0.22 & 0.37 & 0.42 & 0.6 \\ 
# # 20 &   2 &   2 &   2 &   2 &  -1 &   2 &   2 & 0.1 & $4.5 \times 10^{5}$ & $6.5 \times 10^{2}$ &   2 & 0.87 & $6.6 \times 10^{7}$ & $1.6 \times 10^{2}$ & $3.9 \times 10^{5}$ & 0.19 & 0.38 & 0.44 & 0.63 \\ 
# # 21 &   1 &   2 &   1 &   2 &   2 &   2 &   2 & 0.1 & $1.2 \times 10^{6}$ & $6.4 \times 10^{3}$ &   0 & 0.87 & $3.6 \times 10^{8}$ & $2.9 \times 10^{2}$ & $1.3 \times 10^{5}$ & 0.15 & 0.29 & 0.34 & 0.54 \\ 
# # 22 &   2 &   2 &   1 &   2 &   2 &   2 &   2 & 0.1 & $1.7 \times 10^{6}$ & $5.6 \times 10^{3}$ &   0 & 0.87 & $5.5 \times 10^{8}$ & $2 \times 10^{2}$ & $4.1 \times 10^{6}$ & 0.093 & 0.22 & 0.27 & 0.45 \\ 
# # 23 &   1 &   2 &   2 &   2 &   2 &   2 &   2 & 0.1 & $8.2 \times 10^{5}$ & $2.9 \times 10^{4}$ &   3 & 0.87 & $2.6 \times 10^{8}$ & $2.9 \times 10^{2}$ & $2.7 \times 10^{6}$ & 0.097 & 0.2 & 0.24 & 0.41 \\ 
# # 24 &   2 &   2 &   2 &   2 &   2 &   2 &   2 & 0.1 & $5 \times 10^{6}$ & $1.6 \times 10^{4}$ &   0 & 0.87 & $8.7 \times 10^{8}$ & $2 \times 10^{3}$ & $7.9 \times 10^{5}$ & 0.039 & 0.11 & 0.15 & 0.33 \\ 
# # 25 &   1 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.1 & $3.2 \times 10^{5}$ & $6.1 \times 10^{2}$ &   4 & 2.2 & $2.7 \times 10^{7}$ &  46 & $1.5 \times 10^{7}$ & 0.3 & 0.48 & 0.55 & 0.74 \\ 
# # 26 &   2 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.1 & $1.5 \times 10^{6}$ & $1.3 \times 10^{3}$ &   5 & 2.2 & $4 \times 10^{8}$ & $4.5 \times 10^{2}$ & $1.1 \times 10^{7}$ & 0.2 & 0.4 & 0.46 & 0.64 \\ 
# # 27 &   1 &   2 &   2 &   2 &  -1 &   2 &   5 & 0.1 & $2.5 \times 10^{5}$ & $1.8 \times 10^{3}$ &   2 & 2.2 & $1 \times 10^{8}$ &  38 & $2.3 \times 10^{8}$ & 0.26 & 0.44 & 0.49 & 0.69 \\ 
# # 28 &   2 &   2 &   2 &   2 &  -1 &   2 &   5 & 0.1 & $2.2 \times 10^{6}$ & $5.7 \times 10^{3}$ &   1 & 2.2 & $3.9 \times 10^{8}$ & $1.4 \times 10^{2}$ & $1.3 \times 10^{7}$ & 0.14 & 0.29 & 0.34 & 0.54 \\ 
# # 29 &   1 &   2 &   1 &   2 &   2 &   2 &   5 & 0.1 & $1.5 \times 10^{6}$ & $2.8 \times 10^{4}$ &   4 & 2.2 & $3.7 \times 10^{8}$ & $2.5 \times 10^{2}$ & $4.1 \times 10^{7}$ & 0.15 & 0.3 & 0.34 & 0.49 \\ 
# # 30 &   2 &   2 &   1 &   2 &   2 &   2 &   5 & 0.1 & $2.4 \times 10^{6}$ & $2 \times 10^{4}$ &   4 & 2.2 & $3.2 \times 10^{8}$ & $9.1 \times 10^{2}$ & $6.8 \times 10^{5}$ & 0.14 & 0.24 & 0.27 & 0.43 \\ 
# # 31 &   1 &   2 &   2 &   2 &   2 &   2 &   5 & 0.1 & $1 \times 10^{6}$ & $3.9 \times 10^{3}$ &   6 & 2.2 & $1.2 \times 10^{8}$ & $4.1 \times 10^{2}$ & $1.6 \times 10^{6}$ & 0.19 & 0.34 & 0.39 & 0.56 \\ 
# # 32 &   2 &   2 &   2 &   2 &   2 &   2 &   5 & 0.1 & $1.4 \times 10^{6}$ & $1.4 \times 10^{4}$ &   8 & 2.2 & $1.4 \times 10^{8}$ & $2.6 \times 10^{2}$ & $5.1 \times 10^{5}$ & 0.097 & 0.21 & 0.25 & 0.43 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# # Run 4
# load("RegPolCovFixedEffectsPrior4.rda")
# 
# mean.beta0 <- c(1,2); mean.beta1 <- c(1,2); mean.beta2 <- c(-1,-2) # out 4
# sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
# sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 4
# 
# param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)
# 
# param.combos <- expand.grid(param.list)
# 
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero1010 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad1010,na.rm=TRUE)})
# df$prop.zero2020 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2020,na.rm=TRUE)})
# df$prop.zero2525 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2525,na.rm=TRUE)})
# df$prop.zero5050 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad5050,na.rm=TRUE)})
# 
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Sat Jun 12 11:30:50 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero1010 & prop.zero2020 & prop.zero2525 & prop.zero5050 \\ 
# # \hline
# # 1 &   1 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.01 & $7.1 \times 10^{3}$ & $3.4 \times 10^{2}$ &   0 & 0.43 & $2.6 \times 10^{5}$ &  42 & $4 \times 10^{4}$ & 0.36 & 0.52 & 0.56 & 0.73 \\ 
# # 2 &   2 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.01 & $9.4 \times 10^{3}$ & $4.7 \times 10^{2}$ &   0 & 0.43 & $3 \times 10^{5}$ &  36 & $2.1 \times 10^{4}$ & 0.28 & 0.48 & 0.54 & 0.74 \\ 
# # 3 &   1 &   2 &   2 &   2 &  -1 &   2 &   2 & 0.01 & $3 \times 10^{5}$ & $5.2 \times 10^{2}$ &   1 & 0.43 & $8.3 \times 10^{7}$ &  80 & $4.2 \times 10^{6}$ & 0.25 & 0.45 & 0.51 & 0.69 \\ 
# # 4 &   2 &   2 &   2 &   2 &  -1 &   2 &   2 & 0.01 & $9.5 \times 10^{5}$ & $1.8 \times 10^{3}$ &   0 & 0.43 & $8.7 \times 10^{7}$ & $2.8 \times 10^{2}$ & $3.8 \times 10^{5}$ & 0.17 & 0.32 & 0.37 & 0.54 \\ 
# # 5 &   1 &   2 &   1 &   2 &  -2 &   2 &   2 & 0.01 & $3.1 \times 10^{3}$ & $1 \times 10^{2}$ &   0 & 0.43 & $1.7 \times 10^{5}$ &   9 & $4.9 \times 10^{4}$ & 0.47 & 0.65 & 0.7 & 0.84 \\ 
# # 6 &   2 &   2 &   1 &   2 &  -2 &   2 &   2 & 0.01 & $1.3 \times 10^{5}$ & $4.2 \times 10^{2}$ &   0 & 0.43 & $5.1 \times 10^{7}$ &  29 & $1.6 \times 10^{6}$ & 0.31 & 0.5 & 0.57 & 0.74 \\ 
# # 7 &   1 &   2 &   2 &   2 &  -2 &   2 &   2 & 0.01 & $7.6 \times 10^{5}$ & $8 \times 10^{2}$ &   0 & 0.43 & $6.9 \times 10^{7}$ & $1.9 \times 10^{2}$ & $6.8 \times 10^{6}$ & 0.19 & 0.37 & 0.43 & 0.64 \\ 
# # 8 &   2 &   2 &   2 &   2 &  -2 &   2 &   2 & 0.01 & $4.9 \times 10^{4}$ & $2.8 \times 10^{3}$ &   0 & 0.43 & $9.5 \times 10^{6}$ &  48 & $8.6 \times 10^{5}$ & 0.14 & 0.28 & 0.33 & 0.56 \\ 
# # 9 &   1 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.01 & $2.5 \times 10^{6}$ & $9.3 \times 10^{2}$ &   0 & 1.1 & $7.8 \times 10^{8}$ & $1.3 \times 10^{2}$ & $2.2 \times 10^{6}$ & 0.28 & 0.45 & 0.51 & 0.67 \\ 
# # 10 &   2 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.01 & $1 \times 10^{4}$ & $6.3 \times 10^{2}$ &   0 & 1.1 & $5.3 \times 10^{5}$ &  53 & $5.5 \times 10^{4}$ & 0.26 & 0.47 & 0.54 & 0.72 \\ 
# # 11 &   1 &   2 &   2 &   2 &  -1 &   2 &   5 & 0.01 & $1.3 \times 10^{5}$ & $6.9 \times 10^{2}$ &   1 & 1.1 & $2.5 \times 10^{7}$ &  40 & $1.8 \times 10^{8}$ & 0.26 & 0.46 & 0.53 & 0.72 \\ 
# # 12 &   2 &   2 &   2 &   2 &  -1 &   2 &   5 & 0.01 & $9.3 \times 10^{5}$ & $3.9 \times 10^{3}$ &   0 & 1.1 & $9.5 \times 10^{7}$ & $2.2 \times 10^{2}$ & $4.4 \times 10^{7}$ & 0.14 & 0.26 & 0.32 & 0.52 \\ 
# # 13 &   1 &   2 &   1 &   2 &  -2 &   2 &   5 & 0.01 & $7.2 \times 10^{5}$ & $2.8 \times 10^{2}$ &   1 & 1.1 & $6.2 \times 10^{7}$ &  27 & $3.5 \times 10^{6}$ & 0.32 & 0.49 & 0.54 & 0.7 \\ 
# # 14 &   2 &   2 &   1 &   2 &  -2 &   2 &   5 & 0.01 & $4.7 \times 10^{5}$ & $5.8 \times 10^{2}$ &   0 & 1.1 & $2.3 \times 10^{8}$ &  56 & $1.2 \times 10^{7}$ & 0.31 & 0.5 & 0.56 & 0.74 \\ 
# # 15 &   1 &   2 &   2 &   2 &  -2 &   2 &   5 & 0.01 & $1.5 \times 10^{6}$ & $5.3 \times 10^{2}$ &   1 & 1.1 & $8.8 \times 10^{8}$ &  13 & $9.3 \times 10^{6}$ & 0.28 & 0.5 & 0.57 & 0.76 \\ 
# # 16 &   2 &   2 &   2 &   2 &  -2 &   2 &   5 & 0.01 & $2 \times 10^{4}$ & $1.6 \times 10^{3}$ &   2 & 1.1 & $5.3 \times 10^{6}$ &  23 & $2.5 \times 10^{5}$ & 0.18 & 0.36 & 0.43 & 0.68 \\ 
# # 17 &   1 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.1 & $4.3 \times 10^{3}$ & $1.5 \times 10^{2}$ &   0 & 0.87 & $1 \times 10^{5}$ &  21 & $5.9 \times 10^{4}$ & 0.44 & 0.64 & 0.69 & 0.83 \\ 
# # 18 &   2 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.1 & $5.6 \times 10^{5}$ & $2.2 \times 10^{2}$ &   1 & 0.87 & $1.9 \times 10^{8}$ & $2.9 \times 10^{2}$ & $2 \times 10^{7}$ & 0.28 & 0.46 & 0.52 & 0.67 \\ 
# # 19 &   1 &   2 &   2 &   2 &  -1 &   2 &   2 & 0.1 & $1.7 \times 10^{5}$ & $5.1 \times 10^{3}$ &   0 & 0.87 & $9.7 \times 10^{6}$ & $6.7 \times 10^{2}$ & $5.2 \times 10^{6}$ & 0.22 & 0.37 & 0.42 & 0.6 \\ 
# # 20 &   2 &   2 &   2 &   2 &  -1 &   2 &   2 & 0.1 & $2 \times 10^{6}$ & $7.6 \times 10^{2}$ &   1 & 0.87 & $7.1 \times 10^{8}$ & $4.2 \times 10^{2}$ & $1.6 \times 10^{6}$ & 0.2 & 0.38 & 0.44 & 0.63 \\ 
# # 21 &   1 &   2 &   1 &   2 &  -2 &   2 &   2 & 0.1 & $6.5 \times 10^{3}$ & $1.7 \times 10^{2}$ &   0 & 0.87 & $9.6 \times 10^{5}$ &  34 & $2.3 \times 10^{5}$ & 0.37 & 0.54 & 0.6 & 0.77 \\ 
# # 22 &   2 &   2 &   1 &   2 &  -2 &   2 &   2 & 0.1 & $8.7 \times 10^{3}$ & $8.4 \times 10^{2}$ &   0 & 0.87 & $5.1 \times 10^{5}$ &  25 & $7.7 \times 10^{4}$ & 0.21 & 0.4 & 0.46 & 0.67 \\ 
# # 23 &   1 &   2 &   2 &   2 &  -2 &   2 &   2 & 0.1 & $4.9 \times 10^{5}$ & $1.6 \times 10^{3}$ &   0 & 0.87 & $2.3 \times 10^{8}$ &  35 & $1.6 \times 10^{7}$ & 0.24 & 0.4 & 0.47 & 0.67 \\ 
# # 24 &   2 &   2 &   2 &   2 &  -2 &   2 &   2 & 0.1 & $4.9 \times 10^{4}$ & $3.5 \times 10^{3}$ &   0 & 0.87 & $3 \times 10^{6}$ & $1.7 \times 10^{2}$ & $7.8 \times 10^{5}$ & 0.13 & 0.29 & 0.36 & 0.6 \\ 
# # 25 &   1 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.1 & $1.4 \times 10^{6}$ & $7.1 \times 10^{2}$ &   3 & 2.2 & $1 \times 10^{8}$ &  49 & $1.5 \times 10^{7}$ & 0.26 & 0.46 & 0.53 & 0.72 \\ 
# # 26 &   2 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.1 & $1.1 \times 10^{6}$ & $1.2 \times 10^{3}$ &   5 & 2.2 & $2.4 \times 10^{8}$ & $2.7 \times 10^{2}$ & $1.3 \times 10^{7}$ & 0.21 & 0.41 & 0.48 & 0.65 \\ 
# # 27 &   1 &   2 &   2 &   2 &  -1 &   2 &   5 & 0.1 & $2.5 \times 10^{5}$ & $1.8 \times 10^{3}$ &   2 & 2.2 & $1 \times 10^{8}$ &  38 & $2.3 \times 10^{8}$ & 0.26 & 0.44 & 0.49 & 0.69 \\ 
# # 28 &   2 &   2 &   2 &   2 &  -1 &   2 &   5 & 0.1 & $5.7 \times 10^{5}$ & $4.9 \times 10^{3}$ &   2 & 2.2 & $5.8 \times 10^{7}$ & $1.3 \times 10^{2}$ & $1.2 \times 10^{7}$ & 0.15 & 0.29 & 0.35 & 0.55 \\ 
# # 29 &   1 &   2 &   1 &   2 &  -2 &   2 &   5 & 0.1 & $5.2 \times 10^{6}$ & $1.4 \times 10^{3}$ &   2 & 2.2 & $1.7 \times 10^{8}$ & $1.8 \times 10^{3}$ & $5.7 \times 10^{6}$ & 0.25 & 0.43 & 0.49 & 0.66 \\ 
# # 30 &   2 &   2 &   1 &   2 &  -2 &   2 &   5 & 0.1 & $1 \times 10^{7}$ & $1.8 \times 10^{3}$ &   1 & 2.2 & $1.5 \times 10^{9}$ & $1.5 \times 10^{2}$ & $2.1 \times 10^{7}$ & 0.19 & 0.37 & 0.43 & 0.61 \\ 
# # 31 &   1 &   2 &   2 &   2 &  -2 &   2 &   5 & 0.1 & $1.1 \times 10^{7}$ & $9.3 \times 10^{2}$ &   0 & 2.2 & $4 \times 10^{9}$ & $1.7 \times 10^{2}$ & $1.3 \times 10^{10}$ & 0.32 & 0.49 & 0.54 & 0.7 \\ 
# # 32 &   2 &   2 &   2 &   2 &  -2 &   2 &   5 & 0.1 & $1.6 \times 10^{6}$ & $9.5 \times 10^{3}$ &   1 & 2.2 & $1.2 \times 10^{8}$ & $4 \times 10^{2}$ & $1.6 \times 10^{8}$ & 0.18 & 0.32 & 0.37 & 0.56 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# # Run 4 LONG
# load("RegPolCovFixedEffectsPrior4_LongTest.rda")
# 
# mean.beta0 <- c(1,2); mean.beta1 <- c(1); mean.beta2 <- c(-2) # out 4
# sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
# sigma.0 <- c(2); alpha.sigma <- c(0.1) # out 4
# 
# param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)
# 
# param.combos <- expand.grid(param.list)
# 
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero1010 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad1010,na.rm=TRUE)})
# df$prop.zero2020 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2020,na.rm=TRUE)})
# df$prop.zero2525 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2525,na.rm=TRUE)})
# df$prop.zero5050 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad5050,na.rm=TRUE)})
# 
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Sun Jun 13 13:29:00 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero1010 & prop.zero2020 & prop.zero2525 & prop.zero5050 \\ 
# # \hline
# # 1 &   1 &   2 &   1 &   2 &  -2 &   2 &   2 & 0.1 & $6.1 \times 10^{5}$ & $3.2 \times 10^{2}$ &   0 & 0.87 & $1 \times 10^{8}$ &  61 & $2.4 \times 10^{6}$ & 0.34 & 0.54 & 0.59 & 0.76 \\ 
# # 2 &   2 &   2 &   1 &   2 &  -2 &   2 &   2 & 0.1 & $7.4 \times 10^{5}$ & $6.9 \times 10^{2}$ &   0 & 0.87 & $4 \times 10^{8}$ &  44 & $6.4 \times 10^{6}$ & 0.27 & 0.48 & 0.55 & 0.75 \\ 
# # \hline
# # \end{tabular}
# # \end{table}




# sessionInfo() -----------------------------------------------------------

sessionInfo()
