
# Testing Priors for Regular Polygon LGCP (b) -----------------------------

# This R script will test the number of points simulated as well as the proportion of cells with zero counts for varying priors for the fixed effects over the Regular Polygon.

# This contains the code running with rho.start <- c(0.35,0.1) and data output will have a suffix *Prior*b.rda

# Author: Nadeen Khaleel


# Setwd and Load Libraries ------------------------------------------------

library("rstudioapi")
# Either setwd() to the source file location, or run the following:
setwd(dirname(getActiveDocumentContext()$path))

library(doParallel)
library(foreach)


# Set-Up ------------------------------------------------------------------

# Set up Parameter values for priors, the out i next to the hyperparmeter values matches the i in the output file name

# mean.beta0 <- c(1,2); mean.beta1 <- c(1,2); mean.beta2 <- c(-1,-2) # out 1b
# sd.beta0 <- c(2,4); sd.beta1 <- c(2,4); sd.beta2 <- c(2,4)
# sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 1b

# For longer run to test
mean.beta0 <- c(1,2); mean.beta1 <- c(1); mean.beta2 <- c(--2) # out 1b long
sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
sigma.0 <- c(5); alpha.sigma <- c(0.01) # out 1b long

# Set up list of parameters to generate the combinations
param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)

# Create the combinations to test of the above prior parameters
param.combos <- expand.grid(param.list)

# Set-up parallelisation
x <- 1:nrow(param.combos)
Nproc <- 4 # how many processors to spread the simulations across
cl <- makeCluster(Nproc)
registerDoParallel(cl)
indexes <- split(x, sort(rep_len(1:Nproc, length(x))))


# Implement Simulations ---------------------------------------------------

ptm.start <- proc.time()
foreach(i=1:Nproc) %dopar% {
  
  library(spatstat)
  library(raster)
  library(sf)
  library(sp)
  library(maptools)
  library(rgdal)
  library(lwgeom)
  library(RandomFields)
  
  
  # Set-up ------------------------------------------------------------------
  # Need to set-up the data again for the each core - due to the parallelisation, we cannot fork so we have to set- things up again, make sure to match the parameters!
  
  # Create window
  W <- owin(c(0,5),c(0,5))
  
  # Get covariates
  cov.name <- paste0("GridMeshRegPolLGCPSSCov.rda")
  load(cov.name)
  
  # Need to include parameters again - make sure they match!
  # mean.beta0 <- c(1,2); mean.beta1 <- c(1,2); mean.beta2 <- c(-1,-2) # out 1b
  # sd.beta0 <- c(2,4); sd.beta1 <- c(2,4); sd.beta2 <- c(2,4)
  # sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 1b
  # 
  # For longer run to test
  mean.beta0 <- c(1,2); mean.beta1 <- c(1); mean.beta2 <- c(--2) # out 1b Long
  sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
  sigma.0 <- c(5); alpha.sigma <- c(0.01) # out 1b Long
  
  
  param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)
  
  # Create the combinations to test of the above prior parameters
  param.combos <- expand.grid(param.list)
  
  load("QuadratsRegPolLGCP.rda")
  g <- quad.list
  
  # N <- 50 # for shorter more investigative runs
  N <- 100 # for longer runs, saved as *_LongTest.rda
  
  # Change name for files accordingly and make sure to change the following names in the below code
  out.filename <- "RegPolCovFixedEffectsPrior1b_LongTest.rda"
  if ((!file.exists(out.filename)) & (i==1)){
    out <- vector(mode = "list", length = nrow(param.combos))
    save(out,file=out.filename)
  }
  
  # Functions ---------------------------------------------------------------
  
  prior.sim <- function(n=1,mu.beta0,sigma.beta0,mu.beta1,sigma.beta1,mu.beta2,sigma.beta2,sigma.0,alpha.sigma){
    # n is the number of samples of the parameters you want sampled from their priors.  
    # mu.*: prior mean for fixed effects
    # sigma.beta*: standard deviation for fixed effects
    # sigma.0, alpha.sigma: prior parameters for sigma pc prior
    
    beta.0.sim <- rnorm(n,mu.beta0,sigma.beta0); beta.1.sim <- rnorm(n,mu.beta1,sigma.beta1); beta.2.sim <- rnorm(n,mu.beta2,sigma.beta2)
    
    # For the covariance parameters of the latent Gaussian field, I have a joint prior on the marginal standard deviation and the range, however, in order to simulate from this I need the inverse CDF to generate this from a Uniform RV. We can also use the priors fpr kappa and tau given kappa to generate the inverses.
    # (https://www.tandfonline.com/doi/pdf/10.1080/01621459.2017.1415907?needAccess=true) 
    # (https://projecteuclid.org/journals/statistical-science/volume-32/issue-1/Penalising-Model-Component-Complexity--A-Principled-Practical-Approach-to/10.1214/16-STS576.full)
    
    alpha.rho <- 0.1; rho.0 <- 0.35; # suffix b for saved data sets
    u1 <- runif(n,0,1); u2 <- runif(n,0,1)
    rho.sim <- rho.0*log(alpha.rho)/log(u1)
    sigma.sim <- sigma.0*log(1-u2)/log(alpha.sigma)
    return(list(beta0.tilde=beta.0.sim,beta1.tilde=beta.1.sim,beta2.tilde=beta.2.sim,sigma.tilde=sigma.sim,rho.tilde=rho.sim))
  }
  
  res.gen <- function(N,W,prior.param,g.tess,c1.im,c2.im,i.im,seed){
    # N: number of data sets to simulate
    # W: the window
    # prior.param: the set of hyper-parameters for the priors ro simulate the parameter for the generation of the LGCP
    # g.tess: different grid resolutions
    # i.im: pixel image for intercept for data generation
    # ci.im: pixel image for covariates for data simulation
    # seed: set the seed for the data simulations
    
    set.seed(seed)
    # simulate the parameters
    thetas <- prior.sim(n=N,mu.beta0 = prior.param$mu.beta0,sigma.beta0 = prior.param$sd.beta0,mu.beta1 = prior.param$mu.beta1,sigma.beta1 = prior.param$sd.beta1,mu.beta2 = prior.param$mu.beta2,sigma.beta2 = prior.param$sd.beta2,sigma.0=prior.param$sigma.0,alpha.sigma = prior.param$alpha.sigma)
    
    # Create output storage for lgcp.ppp$n and proportion of zeros for different grids.
    grid.names <- names(g.tess)
    output.df <- data.frame(matrix(rep(NA,(length(grid.names)+3)*N),nrow=N))
    colnames(output.df) <- c("n","max.lambda","med.lambda",grid.names)
    
    # Simulated N LGCPs
    for (j in 1:N){
      print(paste0("Simulation ", j))
      # Extract parameter values simulated from priors
      theta <- lapply(1:length(names(thetas)),function(k,j){thetas[[k]][j]},j)
      names(theta) <- names(thetas)
      beta.0 <- theta[[1]]; beta.1 <- theta[[2]]; beta.2 <- theta[[3]]; sigma <- theta[[4]]; rho <- theta[[5]];
      
      mu <- beta.0*i.im + beta.1*c1.im + beta.2*c2.im
      # Produce the point pattern from LGCP
      lgcp.ppp <- try(rLGCP(model="matern",mu,var=(sigma)^2,scale=rho/2,nu=1,win = W,saveLambda = TRUE))
      
      if (class(lgcp.ppp)!="try-error"){
        output.df$n[j] <- lgcp.ppp$n
        att.ppp <- attributes(lgcp.ppp)
        output.df$max.lambda[j] <- max(att.ppp$Lambda)
        output.df$med.lambda[j] <- median(att.ppp$Lambda)
        
        for (i in 1:length(g.tess)){
          col.df <- grid.names[i] # for the grid proportion of zeros in the output.df need to account for the "n" column
          q <- quadratcount(lgcp.ppp,tess=g.tess[[i]])
          count.df <- data.frame(count=as.vector(unname(q)))
          
          output.df[j,col.df] <- sum(count.df$count==0)/length(count.df$count)
        }
      }
    }
    return(list("results"=output.df))
  }
  
  
  # Run All -----------------------------------------------------------------
  
  seeds <- 5*(1:nrow(param.combos))
  
  # Change the filename below accordingly
  out.filename <- "RegPolCovFixedEffectsPrior1b_LongTest.rda"
  load(out.filename)
  
  x <- which(unlist(sapply(1:length(out),function(i){(is.null(out[[i]]))}))) # for re-setting
  Nproc <- 4
  indexes <- split(x, sort(rep_len(1:Nproc, length(x))))
  
  for (k in indexes[[i]]){
    print(k)
    out.new <- res.gen(N,W,param.combos[k,],g.tess=g,cov1.im,cov2.im,int.im,seeds[k])
    load(out.filename)
    out[[k]] <- out.new
    save(out,file=out.filename)
  }
}
ptm.end <- proc.time()
stopCluster(cl)
ptm.end-ptm.start

# Change the file name below accordignly
out.filename <- "RegPolCovFixedEffectsPrior1b_LongTest.rda"

# Get the file and label the outputs so that we can more easily tell which parameters resulted in which results
load(out.filename)
combo.names <- sapply(1:nrow(param.combos),function(i){paste0("(",paste0(param.combos[i,],collapse=","),")")})
names(out) <- combo.names

# Save labelled output
save(out,file=out.filename)



# Check Output ------------------------------------------------------------
# Load outputs and generate tables, these can then be placed in a .tex file for easy reading


# # Run 1
# load("RegPolCovFixedEffectsPrior1b.rda")
# mean.beta0 <- c(1,2); mean.beta1 <- c(1,2); mean.beta2 <- c(-1,-2) # out 1b
# sd.beta0 <- c(2,4); sd.beta1 <- c(2,4); sd.beta2 <- c(2,4)
# sigma.0 <- c(2,5); alpha.sigma <- c(0.01,0.1) # out 1b
# 
# param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)
# 
# param.combos <- expand.grid(param.list)
# 
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero1010 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad1010,na.rm=TRUE)})
# df$prop.zero2020 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2020,na.rm=TRUE)})
# df$prop.zero2525 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2525,na.rm=TRUE)})
# df$prop.zero5050 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad5050,na.rm=TRUE)})
# 
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Sun Jun 13 11:44:44 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero1010 & prop.zero2020 & prop.zero2525 & prop.zero5050 \\ 
# # \hline
# # 1 &   1 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.01 & $7.5 \times 10^{3}$ & $4.3 \times 10^{2}$ &   0 & 0.43 & $2.7 \times 10^{5}$ &  47 & $4.8 \times 10^{4}$ & 0.35 & 0.5 & 0.55 & 0.72 \\ 
# # 2 &   2 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.01 & $9 \times 10^{3}$ & $5.8 \times 10^{2}$ &   0 & 0.43 & $2.7 \times 10^{5}$ &  35 & $2.4 \times 10^{4}$ & 0.27 & 0.47 & 0.54 & 0.73 \\ 
# # 3 &   1 &   4 &   1 &   2 &  -1 &   2 &   2 & 0.01 & $2.6 \times 10^{5}$ & $3.5 \times 10^{2}$ &   1 & 0.43 & $4.1 \times 10^{7}$ & $7.4 \times 10^{2}$ & $5.7 \times 10^{5}$ & 0.28 & 0.42 & 0.46 & 0.58 \\ 
# # 4 &   2 &   4 &   1 &   2 &  -1 &   2 &   2 & 0.01 & $4.2 \times 10^{6}$ & $5.3 \times 10^{2}$ &   0 & 0.43 & $3.4 \times 10^{8}$ & $3.3 \times 10^{3}$ & $4.5 \times 10^{4}$ & 0.29 & 0.39 & 0.43 & 0.54 \\ 
# # 5 &   1 &   2 &   2 &   2 &  -1 &   2 &   2 & 0.01 & $5.6 \times 10^{4}$ & $1.5 \times 10^{2}$ &   0 & 0.43 & $8 \times 10^{6}$ &  27 & $2.2 \times 10^{6}$ & 0.39 & 0.55 & 0.6 & 0.74 \\ 
# # 6 &   2 &   2 &   2 &   2 &  -1 &   2 &   2 & 0.01 & $1.4 \times 10^{6}$ & $1 \times 10^{3}$ &   0 & 0.43 & $6.1 \times 10^{8}$ &  77 & $6.8 \times 10^{6}$ & 0.2 & 0.38 & 0.45 & 0.64 \\ 
# # 7 &   1 &   4 &   2 &   2 &  -1 &   2 &   2 & 0.01 & $1.4 \times 10^{6}$ & $2.7 \times 10^{3}$ &   2 & 0.43 & $2 \times 10^{8}$ & $2.5 \times 10^{3}$ & $2.2 \times 10^{6}$ & 0.2 & 0.31 & 0.35 & 0.49 \\ 
# # 8 &   2 &   4 &   2 &   2 &  -1 &   2 &   2 & 0.01 & $1.6 \times 10^{6}$ & $9.1 \times 10^{3}$ &   0 & 0.43 & $7 \times 10^{8}$ & $1.7 \times 10^{3}$ & $4.9 \times 10^{5}$ & 0.19 & 0.27 & 0.3 & 0.43 \\ 
# # 9 &   1 &   2 &   1 &   4 &  -1 &   2 &   2 & 0.01 & $4.8 \times 10^{4}$ & $9.5 \times 10^{2}$ &   4 & 0.43 & $1.7 \times 10^{7}$ &  62 & $3.3 \times 10^{6}$ & 0.33 & 0.47 & 0.52 & 0.67 \\ 
# # 10 &   2 &   2 &   1 &   4 &  -1 &   2 &   2 & 0.01 & $4.2 \times 10^{5}$ & $9.6 \times 10^{2}$ &   1 & 0.43 & $1.6 \times 10^{8}$ &  46 & $3.3 \times 10^{7}$ & 0.2 & 0.39 & 0.45 & 0.64 \\ 
# # 11 &   1 &   4 &   1 &   4 &  -1 &   2 &   2 & 0.01 & $3.1 \times 10^{6}$ & $8.2 \times 10^{2}$ &   1 & 0.43 & $1.2 \times 10^{9}$ & $2.5 \times 10^{3}$ & $1.2 \times 10^{7}$ & 0.28 & 0.43 & 0.48 & 0.62 \\ 
# # 12 &   2 &   4 &   1 &   4 &  -1 &   2 &   2 & 0.01 & $2.6 \times 10^{5}$ & $2.1 \times 10^{3}$ &   5 & 0.43 & $1.3 \times 10^{7}$ & $2.4 \times 10^{3}$ & $8.1 \times 10^{6}$ & 0.22 & 0.32 & 0.36 & 0.53 \\ 
# # 13 &   1 &   2 &   2 &   4 &  -1 &   2 &   2 & 0.01 & $2.4 \times 10^{6}$ & $1.5 \times 10^{3}$ &   4 & 0.43 & $6.8 \times 10^{8}$ &  96 & $2.6 \times 10^{7}$ & 0.33 & 0.44 & 0.48 & 0.63 \\ 
# # 14 &   2 &   2 &   2 &   4 &  -1 &   2 &   2 & 0.01 & $1.7 \times 10^{6}$ & $9.5 \times 10^{2}$ &   5 & 0.43 & $6.9 \times 10^{8}$ &  42 & $1.9 \times 10^{8}$ & 0.24 & 0.44 & 0.5 & 0.68 \\ 
# # 15 &   1 &   4 &   2 &   4 &  -1 &   2 &   2 & 0.01 & $3.3 \times 10^{6}$ & $7.3 \times 10^{2}$ &   2 & 0.43 & $1.4 \times 10^{9}$ & $1.7 \times 10^{3}$ & $1.1 \times 10^{9}$ & 0.26 & 0.42 & 0.47 & 0.63 \\ 
# # 16 &   2 &   4 &   2 &   4 &  -1 &   2 &   2 & 0.01 & $1.9 \times 10^{6}$ & $4.1 \times 10^{3}$ &   4 & 0.43 & $7.5 \times 10^{8}$ & $4 \times 10^{2}$ & $7.7 \times 10^{7}$ & 0.17 & 0.3 & 0.34 & 0.5 \\ 
# # 17 &   1 &   2 &   1 &   2 &  -2 &   2 &   2 & 0.01 & $1.8 \times 10^{3}$ &  53 &   0 & 0.43 & $1.4 \times 10^{5}$ &  11 & $1.3 \times 10^{4}$ & 0.5 & 0.69 & 0.74 & 0.85 \\ 
# # 18 &   2 &   2 &   1 &   2 &  -2 &   2 &   2 & 0.01 & $4 \times 10^{4}$ & $3.6 \times 10^{2}$ &   0 & 0.43 & $1.4 \times 10^{6}$ & $2 \times 10^{2}$ & $6.7 \times 10^{4}$ & 0.27 & 0.45 & 0.5 & 0.67 \\ 
# # 19 &   1 &   4 &   1 &   2 &  -2 &   2 &   2 & 0.01 & $3.9 \times 10^{5}$ & $7.7 \times 10^{2}$ &   0 & 0.43 & $7.8 \times 10^{6}$ & $3.8 \times 10^{3}$ & $1.5 \times 10^{6}$ & 0.29 & 0.4 & 0.44 & 0.55 \\ 
# # 20 &   2 &   4 &   1 &   2 &  -2 &   2 &   2 & 0.01 & $9.1 \times 10^{6}$ & $3.1 \times 10^{2}$ &   0 & 0.43 & $4.5 \times 10^{9}$ & $5.6 \times 10^{3}$ & $6.2 \times 10^{6}$ & 0.28 & 0.46 & 0.51 & 0.63 \\ 
# # 21 &   1 &   2 &   2 &   2 &  -2 &   2 &   2 & 0.01 & $2.9 \times 10^{4}$ & $2.5 \times 10^{2}$ &   0 & 0.43 & $5.7 \times 10^{6}$ &  59 & $8.7 \times 10^{5}$ & 0.31 & 0.5 & 0.56 & 0.73 \\ 
# # 22 &   2 &   2 &   2 &   2 &  -2 &   2 &   2 & 0.01 & $2.5 \times 10^{4}$ & $1.2 \times 10^{3}$ &   0 & 0.43 & $3.2 \times 10^{6}$ &  37 & $1 \times 10^{5}$ & 0.17 & 0.34 & 0.41 & 0.64 \\ 
# # 23 &   1 &   4 &   2 &   2 &  -2 &   2 &   2 & 0.01 & $1.2 \times 10^{6}$ & $1.9 \times 10^{3}$ &   0 & 0.43 & $1.6 \times 10^{8}$ & $1.8 \times 10^{3}$ & $4.7 \times 10^{6}$ & 0.24 & 0.35 & 0.4 & 0.58 \\ 
# # 24 &   2 &   4 &   2 &   2 &  -2 &   2 &   2 & 0.01 & $1.4 \times 10^{6}$ & $2 \times 10^{3}$ &   0 & 0.43 & $6.7 \times 10^{7}$ & $1.6 \times 10^{4}$ & $5.2 \times 10^{5}$ & 0.18 & 0.34 & 0.39 & 0.55 \\ 
# # 25 &   1 &   2 &   1 &   4 &  -2 &   2 &   2 & 0.01 & $3.2 \times 10^{6}$ & $2.2 \times 10^{2}$ &   3 & 0.43 & $1.4 \times 10^{9}$ &  62 & $1.9 \times 10^{11}$ & 0.31 & 0.48 & 0.53 & 0.69 \\ 
# # 26 &   2 &   2 &   1 &   4 &  -2 &   2 &   2 & 0.01 & $1.6 \times 10^{6}$ & $3 \times 10^{3}$ &   1 & 0.43 & $6.8 \times 10^{8}$ & $1.4 \times 10^{2}$ & $1.9 \times 10^{7}$ & 0.17 & 0.34 & 0.39 & 0.58 \\ 
# # 27 &   1 &   4 &   1 &   4 &  -2 &   2 &   2 & 0.01 & $3.2 \times 10^{6}$ & $5.4 \times 10^{2}$ &   0 & 0.43 & $1.9 \times 10^{9}$ &  99 & $7.1 \times 10^{7}$ & 0.27 & 0.41 & 0.46 & 0.62 \\ 
# # 28 &   2 &   4 &   1 &   4 &  -2 &   2 &   2 & 0.01 & $3.3 \times 10^{6}$ & $4 \times 10^{3}$ &   3 & 0.43 & $1.6 \times 10^{9}$ & $4.3 \times 10^{2}$ & $6.6 \times 10^{9}$ & 0.16 & 0.27 & 0.31 & 0.46 \\ 
# # 29 &   1 &   2 &   2 &   4 &  -2 &   2 &   2 & 0.01 & $6.6 \times 10^{6}$ & $1.1 \times 10^{3}$ &   3 & 0.43 & $2.1 \times 10^{9}$ & $3.3 \times 10^{2}$ & $1.6 \times 10^{7}$ & 0.22 & 0.39 & 0.46 & 0.65 \\ 
# # 30 &   2 &   2 &   2 &   4 &  -2 &   2 &   2 & 0.01 & $2.6 \times 10^{6}$ & $1.2 \times 10^{3}$ &   4 & 0.43 & $4.9 \times 10^{8}$ & $2.3 \times 10^{2}$ & $2 \times 10^{7}$ & 0.17 & 0.33 & 0.39 & 0.58 \\ 
# # 31 &   1 &   4 &   2 &   4 &  -2 &   2 &   2 & 0.01 & $2.8 \times 10^{6}$ & $5.7 \times 10^{2}$ &   5 & 0.43 & $1.2 \times 10^{9}$ & $2.8 \times 10^{3}$ & $3.5 \times 10^{7}$ & 0.3 & 0.43 & 0.47 & 0.6 \\ 
# # 32 &   2 &   4 &   2 &   4 &  -2 &   2 &   2 & 0.01 & $1.9 \times 10^{6}$ & $4.4 \times 10^{3}$ &   4 & 0.43 & $6 \times 10^{8}$ & $1.5 \times 10^{4}$ & $3.7 \times 10^{7}$ & 0.19 & 0.29 & 0.33 & 0.46 \\ 
# # 33 &   1 &   2 &   1 &   2 &  -1 &   4 &   2 & 0.01 & $3 \times 10^{6}$ & $2.8 \times 10^{3}$ &   2 & 0.43 & $8.7 \times 10^{8}$ & $5.5 \times 10^{2}$ & $2.4 \times 10^{8}$ & 0.15 & 0.33 & 0.39 & 0.61 \\ 
# # 34 &   2 &   2 &   1 &   2 &  -1 &   4 &   2 & 0.01 & $4.1 \times 10^{5}$ & $1.2 \times 10^{3}$ &   1 & 0.43 & $2.1 \times 10^{7}$ & $3.5 \times 10^{2}$ & $6.3 \times 10^{5}$ & 0.15 & 0.35 & 0.42 & 0.64 \\ 
# # 35 &   1 &   4 &   1 &   2 &  -1 &   4 &   2 & 0.01 & $1.9 \times 10^{6}$ & $2.3 \times 10^{3}$ &   0 & 0.43 & $4.3 \times 10^{8}$ & $5.3 \times 10^{2}$ & $8.7 \times 10^{6}$ & 0.27 & 0.36 & 0.39 & 0.51 \\ 
# # 36 &   2 &   4 &   1 &   2 &  -1 &   4 &   2 & 0.01 & $9.3 \times 10^{5}$ & $7.4 \times 10^{3}$ &   3 & 0.43 & $1.9 \times 10^{8}$ & $1.2 \times 10^{3}$ & $2.5 \times 10^{7}$ & 0.2 & 0.31 & 0.36 & 0.51 \\ 
# # 37 &   1 &   2 &   2 &   2 &  -1 &   4 &   2 & 0.01 & $5.4 \times 10^{5}$ & $4.2 \times 10^{3}$ &   2 & 0.43 & $2.2 \times 10^{8}$ & $1.1 \times 10^{2}$ & $2.1 \times 10^{7}$ & 0.12 & 0.31 & 0.37 & 0.59 \\ 
# # 38 &   2 &   2 &   2 &   2 &  -1 &   4 &   2 & 0.01 & $1.7 \times 10^{6}$ & $5.7 \times 10^{3}$ &   0 & 0.43 & $5.9 \times 10^{8}$ & $1.4 \times 10^{2}$ & $2.3 \times 10^{7}$ & 0.16 & 0.28 & 0.32 & 0.52 \\ 
# # 39 &   1 &   4 &   2 &   2 &  -1 &   4 &   2 & 0.01 & $6 \times 10^{5}$ & $2.6 \times 10^{3}$ &   2 & 0.43 & $2.3 \times 10^{8}$ & $5.4 \times 10^{2}$ & $7.9 \times 10^{7}$ & 0.2 & 0.33 & 0.38 & 0.55 \\ 
# # 40 &   2 &   4 &   2 &   2 &  -1 &   4 &   2 & 0.01 & $2.8 \times 10^{5}$ & $2.4 \times 10^{3}$ &   3 & 0.43 & $3.6 \times 10^{7}$ & $2.5 \times 10^{2}$ & $4 \times 10^{6}$ & 0.17 & 0.31 & 0.36 & 0.54 \\ 
# # 41 &   1 &   2 &   1 &   4 &  -1 &   4 &   2 & 0.01 & $1.7 \times 10^{6}$ & $2.1 \times 10^{3}$ &   1 & 0.43 & $7 \times 10^{8}$ &  58 & $4.6 \times 10^{7}$ & 0.22 & 0.39 & 0.45 & 0.64 \\ 
# # 42 &   2 &   2 &   1 &   4 &  -1 &   4 &   2 & 0.01 & $1.1 \times 10^{7}$ & $2.6 \times 10^{3}$ &   1 & 0.43 & $4 \times 10^{9}$ & $3.4 \times 10^{2}$ & $3.8 \times 10^{8}$ & 0.18 & 0.31 & 0.36 & 0.54 \\ 
# # 43 &   1 &   4 &   1 &   4 &  -1 &   4 &   2 & 0.01 & $5.5 \times 10^{6}$ & $1.5 \times 10^{3}$ &   3 & 0.43 & $1.7 \times 10^{9}$ & $2.9 \times 10^{3}$ & $1.1 \times 10^{9}$ & 0.25 & 0.39 & 0.43 & 0.56 \\ 
# # 44 &   2 &   4 &   1 &   4 &  -1 &   4 &   2 & 0.01 & $8.2 \times 10^{6}$ & $9 \times 10^{2}$ &   3 & 0.43 & $1.4 \times 10^{9}$ & $8.7 \times 10^{2}$ & $6.9 \times 10^{7}$ & 0.27 & 0.38 & 0.42 & 0.56 \\ 
# # 45 &   1 &   2 &   2 &   4 &  -1 &   4 &   2 & 0.01 & $8.2 \times 10^{6}$ & $1 \times 10^{4}$ &   6 & 0.43 & $4.9 \times 10^{9}$ &  87 & $2.1 \times 10^{9}$ & 0.19 & 0.34 & 0.39 & 0.57 \\ 
# # 46 &   2 &   2 &   2 &   4 &  -1 &   4 &   2 & 0.01 & $5.3 \times 10^{6}$ & $6.9 \times 10^{3}$ &   3 & 0.43 & $1.6 \times 10^{9}$ & $2.3 \times 10^{2}$ & $1.2 \times 10^{7}$ & 0.14 & 0.28 & 0.33 & 0.53 \\ 
# # 47 &   1 &   4 &   2 &   4 &  -1 &   4 &   2 & 0.01 & $3.4 \times 10^{6}$ & $6.2 \times 10^{3}$ &   6 & 0.43 & $7.8 \times 10^{8}$ & $2.7 \times 10^{3}$ & $2 \times 10^{6}$ & 0.23 & 0.35 & 0.37 & 0.48 \\ 
# # 48 &   2 &   4 &   2 &   4 &  -1 &   4 &   2 & 0.01 & $1.4 \times 10^{6}$ & $1.2 \times 10^{4}$ &   7 & 0.43 & $4.2 \times 10^{8}$ & $4.5 \times 10^{2}$ & $1.6 \times 10^{7}$ & 0.17 & 0.27 & 0.32 & 0.49 \\ 
# # 49 &   1 &   2 &   1 &   2 &  -2 &   4 &   2 & 0.01 & $4.8 \times 10^{4}$ & $8.7 \times 10^{2}$ &   1 & 0.43 & $4.6 \times 10^{6}$ &  34 & $3.2 \times 10^{6}$ & 0.28 & 0.46 & 0.51 & 0.69 \\ 
# # 50 &   2 &   2 &   1 &   2 &  -2 &   4 &   2 & 0.01 & $3.2 \times 10^{5}$ & $1.5 \times 10^{3}$ &   0 & 0.43 & $9 \times 10^{7}$ &  48 & $6.5 \times 10^{7}$ & 0.17 & 0.36 & 0.43 & 0.66 \\ 
# # 51 &   1 &   4 &   1 &   2 &  -2 &   4 &   2 & 0.01 & $3.4 \times 10^{5}$ & $4.9 \times 10^{2}$ &   0 & 0.43 & $4.1 \times 10^{7}$ & $5.4 \times 10^{2}$ & $1.3 \times 10^{7}$ & 0.32 & 0.47 & 0.51 & 0.65 \\ 
# # 52 &   2 &   4 &   1 &   2 &  -2 &   4 &   2 & 0.01 & $2.4 \times 10^{6}$ & $9 \times 10^{2}$ &   3 & 0.43 & $1 \times 10^{9}$ & $2.3 \times 10^{2}$ & $5.8 \times 10^{8}$ & 0.23 & 0.37 & 0.41 & 0.54 \\ 
# # 53 &   1 &   2 &   2 &   2 &  -2 &   4 &   2 & 0.01 & $1.4 \times 10^{6}$ & $9.6 \times 10^{2}$ &   0 & 0.43 & $3.9 \times 10^{8}$ &  26 & $1 \times 10^{8}$ & 0.25 & 0.43 & 0.5 & 0.7 \\ 
# # 54 &   2 &   2 &   2 &   2 &  -2 &   4 &   2 & 0.01 & $3.6 \times 10^{5}$ & $8.1 \times 10^{3}$ &   0 & 0.43 & $8.5 \times 10^{7}$ &  98 & $4.7 \times 10^{6}$ & 0.14 & 0.28 & 0.33 & 0.52 \\ 
# # 55 &   1 &   4 &   2 &   2 &  -2 &   4 &   2 & 0.01 & $5.8 \times 10^{5}$ & $3.7 \times 10^{3}$ &   1 & 0.43 & $1.2 \times 10^{8}$ & $6.5 \times 10^{2}$ & $1.3 \times 10^{8}$ & 0.22 & 0.34 & 0.38 & 0.55 \\ 
# # 56 &   2 &   4 &   2 &   2 &  -2 &   4 &   2 & 0.01 & $1.5 \times 10^{6}$ & $1.3 \times 10^{3}$ &   1 & 0.43 & $1.1 \times 10^{8}$ & $1 \times 10^{3}$ & $4.1 \times 10^{9}$ & 0.24 & 0.38 & 0.43 & 0.57 \\ 
# # 57 &   1 &   2 &   1 &   4 &  -2 &   4 &   2 & 0.01 & $4.2 \times 10^{5}$ & $1.2 \times 10^{3}$ &   2 & 0.43 & $1.1 \times 10^{8}$ &  49 & $1.7 \times 10^{7}$ & 0.31 & 0.46 & 0.52 & 0.7 \\ 
# # 58 &   2 &   2 &   1 &   4 &  -2 &   4 &   2 & 0.01 & $5.7 \times 10^{6}$ & $1.4 \times 10^{4}$ &   4 & 0.43 & $1.5 \times 10^{9}$ & $2.8 \times 10^{2}$ & $1.5 \times 10^{8}$ & 0.11 & 0.24 & 0.3 & 0.49 \\ 
# # 59 &   1 &   4 &   1 &   4 &  -2 &   4 &   2 & 0.01 & $2.7 \times 10^{6}$ & $6.4 \times 10^{2}$ &   4 & 0.43 & $1 \times 10^{9}$ & $2.3 \times 10^{2}$ & $2.5 \times 10^{8}$ & 0.28 & 0.43 & 0.48 & 0.63 \\ 
# # 60 &   2 &   4 &   1 &   4 &  -2 &   4 &   2 & 0.01 & $1.1 \times 10^{7}$ & $2 \times 10^{3}$ &   7 & 0.43 & $3 \times 10^{9}$ & $7 \times 10^{2}$ & $4.3 \times 10^{7}$ & 0.16 & 0.31 & 0.36 & 0.55 \\ 
# # 61 &   1 &   2 &   2 &   4 &  -2 &   4 &   2 & 0.01 & $8 \times 10^{6}$ & $2.6 \times 10^{4}$ &   4 & 0.43 & $1.7 \times 10^{9}$ &  41 & $1.4 \times 10^{10}$ & 0.14 & 0.29 & 0.34 & 0.53 \\ 
# # 62 &   2 &   2 &   2 &   4 &  -2 &   4 &   2 & 0.01 & $4.6 \times 10^{6}$ & $5.8 \times 10^{3}$ &   4 & 0.43 & $1.7 \times 10^{9}$ &  72 & $4.7 \times 10^{9}$ & 0.19 & 0.34 & 0.39 & 0.59 \\ 
# # 63 &   1 &   4 &   2 &   4 &  -2 &   4 &   2 & 0.01 & $1.4 \times 10^{6}$ & $1.2 \times 10^{3}$ &   4 & 0.43 & $5.8 \times 10^{8}$ &  63 & $4.4 \times 10^{8}$ & 0.33 & 0.47 & 0.52 & 0.65 \\ 
# # 64 &   2 &   4 &   2 &   4 &  -2 &   4 &   2 & 0.01 & $9.6 \times 10^{6}$ & $1 \times 10^{4}$ &   6 & 0.43 & $3.4 \times 10^{9}$ & $7.2 \times 10^{2}$ & $2.4 \times 10^{9}$ & 0.21 & 0.31 & 0.35 & 0.49 \\ 
# # 65 &   1 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.01 & $1.5 \times 10^{4}$ & $3.8 \times 10^{2}$ &   0 & 1.1 & $2.5 \times 10^{6}$ &  17 & $1.1 \times 10^{5}$ & 0.31 & 0.54 & 0.62 & 0.8 \\ 
# # 66 &   2 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.01 & $6.4 \times 10^{5}$ & $9.8 \times 10^{2}$ &   0 & 1.1 & $1.8 \times 10^{8}$ & $1.1 \times 10^{2}$ & $2.8 \times 10^{7}$ & 0.2 & 0.39 & 0.46 & 0.66 \\ 
# # 67 &   1 &   4 &   1 &   2 &  -1 &   2 &   5 & 0.01 & $2.6 \times 10^{4}$ & $3.1 \times 10^{2}$ &   2 & 1.1 & $1.4 \times 10^{6}$ & $1 \times 10^{2}$ & $1.1 \times 10^{6}$ & 0.34 & 0.5 & 0.56 & 0.69 \\ 
# # 68 &   2 &   4 &   1 &   2 &  -1 &   2 &   5 & 0.01 & $5.8 \times 10^{6}$ & $8.1 \times 10^{2}$ &   0 & 1.1 & $1.3 \times 10^{9}$ & $4.3 \times 10^{2}$ & $1.2 \times 10^{7}$ & 0.22 & 0.35 & 0.4 & 0.53 \\ 
# # 69 &   1 &   2 &   2 &   2 &  -1 &   2 &   5 & 0.01 & $2.1 \times 10^{5}$ & $1.3 \times 10^{3}$ &   1 & 1.1 & $3.5 \times 10^{7}$ &  60 & $1 \times 10^{9}$ & 0.3 & 0.47 & 0.53 & 0.7 \\ 
# # 70 &   2 &   2 &   2 &   2 &  -1 &   2 &   5 & 0.01 & $1.8 \times 10^{6}$ & $2.8 \times 10^{3}$ &   0 & 1.1 & $6.3 \times 10^{8}$ &  71 & $7.3 \times 10^{6}$ & 0.13 & 0.3 & 0.37 & 0.59 \\ 
# # 71 &   1 &   4 &   2 &   2 &  -1 &   2 &   5 & 0.01 & $1.4 \times 10^{6}$ & $9.1 \times 10^{2}$ &   3 & 1.1 & $5.9 \times 10^{7}$ & $3.8 \times 10^{3}$ & $4.9 \times 10^{5}$ & 0.32 & 0.42 & 0.45 & 0.56 \\ 
# # 72 &   2 &   4 &   2 &   2 &  -1 &   2 &   5 & 0.01 & $5.5 \times 10^{6}$ & $5.2 \times 10^{3}$ &   1 & 1.1 & $7.2 \times 10^{8}$ & $3.9 \times 10^{3}$ & $4.6 \times 10^{5}$ & 0.23 & 0.32 & 0.35 & 0.46 \\ 
# # 73 &   1 &   2 &   1 &   4 &  -1 &   2 &   5 & 0.01 & $1.5 \times 10^{5}$ & $3.4 \times 10^{2}$ &   1 & 1.1 & $6.6 \times 10^{7}$ &  26 & $3.1 \times 10^{6}$ & 0.34 & 0.5 & 0.55 & 0.7 \\ 
# # 74 &   2 &   2 &   1 &   4 &  -1 &   2 &   5 & 0.01 & $3.1 \times 10^{6}$ & $1.8 \times 10^{3}$ &   5 & 1.1 & $8.8 \times 10^{8}$ &  61 & $1.5 \times 10^{8}$ & 0.18 & 0.38 & 0.44 & 0.63 \\ 
# # 75 &   1 &   4 &   1 &   4 &  -1 &   2 &   5 & 0.01 & $3 \times 10^{6}$ & $3.4 \times 10^{3}$ &   3 & 1.1 & $7.9 \times 10^{8}$ & $2.8 \times 10^{2}$ & $4.9 \times 10^{8}$ & 0.21 & 0.3 & 0.34 & 0.48 \\ 
# # 76 &   2 &   4 &   1 &   4 &  -1 &   2 &   5 & 0.01 & $1.1 \times 10^{7}$ & $1.3 \times 10^{3}$ &   3 & 1.1 & $3 \times 10^{9}$ & $5.2 \times 10^{3}$ & $4.1 \times 10^{8}$ & 0.28 & 0.42 & 0.47 & 0.61 \\ 
# # 77 &   1 &   2 &   2 &   4 &  -1 &   2 &   5 & 0.01 & $4.6 \times 10^{6}$ & $2.1 \times 10^{3}$ &   5 & 1.1 & $2.3 \times 10^{9}$ &  31 & $2.1 \times 10^{8}$ & 0.28 & 0.45 & 0.51 & 0.67 \\ 
# # 78 &   2 &   2 &   2 &   4 &  -1 &   2 &   5 & 0.01 & $1.1 \times 10^{7}$ & $1.2 \times 10^{4}$ &   3 & 1.1 & $5.2 \times 10^{9}$ & $1.6 \times 10^{2}$ & $5.1 \times 10^{8}$ & 0.13 & 0.25 & 0.29 & 0.47 \\ 
# # 79 &   1 &   4 &   2 &   4 &  -1 &   2 &   5 & 0.01 & $4.2 \times 10^{6}$ & $4.7 \times 10^{3}$ &   0 & 1.1 & $1 \times 10^{9}$ & $7.3 \times 10^{2}$ & $8.8 \times 10^{7}$ & 0.19 & 0.29 & 0.33 & 0.49 \\ 
# # 80 &   2 &   4 &   2 &   4 &  -1 &   2 &   5 & 0.01 & $9.8 \times 10^{6}$ & $6.4 \times 10^{3}$ &   4 & 1.1 & $2.4 \times 10^{9}$ & $1.8 \times 10^{3}$ & $1.8 \times 10^{8}$ & 0.24 & 0.37 & 0.41 & 0.54 \\ 
# # 81 &   1 &   2 &   1 &   2 &  -2 &   2 &   5 & 0.01 & $4.4 \times 10^{5}$ & $2.6 \times 10^{2}$ &   0 & 1.1 & $1.1 \times 10^{8}$ &  17 & $7.2 \times 10^{8}$ & 0.38 & 0.6 & 0.66 & 0.81 \\ 
# # 82 &   2 &   2 &   1 &   2 &  -2 &   2 &   5 & 0.01 & $4.9 \times 10^{3}$ & $6.9 \times 10^{2}$ &   0 & 1.1 & $5.2 \times 10^{5}$ &  20 & $2.3 \times 10^{5}$ & 0.25 & 0.44 & 0.51 & 0.73 \\ 
# # 83 &   1 &   4 &   1 &   2 &  -2 &   2 &   5 & 0.01 & $1 \times 10^{5}$ & $2.6 \times 10^{2}$ &   2 & 1.1 & $4.5 \times 10^{6}$ & $1.6 \times 10^{2}$ & $9.3 \times 10^{7}$ & 0.37 & 0.5 & 0.54 & 0.66 \\ 
# # 84 &   2 &   4 &   1 &   2 &  -2 &   2 &   5 & 0.01 & $2.9 \times 10^{6}$ & $1.1 \times 10^{3}$ &   0 & 1.1 & $5 \times 10^{7}$ & $3.4 \times 10^{3}$ & $3 \times 10^{5}$ & 0.29 & 0.4 & 0.45 & 0.58 \\ 
# # 85 &   1 &   2 &   2 &   2 &  -2 &   2 &   5 & 0.01 & $1.2 \times 10^{5}$ & $6.1 \times 10^{2}$ &   0 & 1.1 & $1.6 \times 10^{7}$ &  21 & $2.4 \times 10^{6}$ & 0.28 & 0.48 & 0.54 & 0.73 \\ 
# # 86 &   2 &   2 &   2 &   2 &  -2 &   2 &   5 & 0.01 & $3 \times 10^{5}$ & $1.7 \times 10^{3}$ &   0 & 1.1 & $6.7 \times 10^{7}$ & $1.7 \times 10^{2}$ & $2.6 \times 10^{6}$ & 0.14 & 0.31 & 0.38 & 0.58 \\ 
# # 87 &   1 &   4 &   2 &   2 &  -2 &   2 &   5 & 0.01 & $1.8 \times 10^{6}$ & $2.6 \times 10^{3}$ &   1 & 1.1 & $3.3 \times 10^{8}$ & $1.2 \times 10^{3}$ & $4.5 \times 10^{6}$ & 0.24 & 0.38 & 0.43 & 0.59 \\ 
# # 88 &   2 &   4 &   2 &   2 &  -2 &   2 &   5 & 0.01 & $5 \times 10^{6}$ & $4.8 \times 10^{3}$ &   1 & 1.1 & $2.2 \times 10^{9}$ & $5.6 \times 10^{3}$ & $3.4 \times 10^{6}$ & 0.14 & 0.28 & 0.32 & 0.46 \\ 
# # 89 &   1 &   2 &   1 &   4 &  -2 &   2 &   5 & 0.01 & $1.1 \times 10^{6}$ & $5.5 \times 10^{2}$ &   0 & 1.1 & $3.1 \times 10^{8}$ &  17 & $2.5 \times 10^{8}$ & 0.38 & 0.55 & 0.61 & 0.76 \\ 
# # 90 &   2 &   2 &   1 &   4 &  -2 &   2 &   5 & 0.01 & $6.6 \times 10^{6}$ & $2.2 \times 10^{3}$ &   0 & 1.1 & $1.9 \times 10^{9}$ &  66 & $1.1 \times 10^{11}$ & 0.22 & 0.43 & 0.5 & 0.7 \\ 
# # 91 &   1 &   4 &   1 &   4 &  -2 &   2 &   5 & 0.01 & $3.9 \times 10^{5}$ & $3.2 \times 10^{3}$ &   2 & 1.1 & $1.2 \times 10^{8}$ & $7.9 \times 10^{2}$ & $2.5 \times 10^{6}$ & 0.19 & 0.31 & 0.35 & 0.53 \\ 
# # 92 &   2 &   4 &   1 &   4 &  -2 &   2 &   5 & 0.01 & $8.6 \times 10^{6}$ & $3.5 \times 10^{3}$ &   3 & 1.1 & $2.2 \times 10^{9}$ & $5.9 \times 10^{3}$ & $2.9 \times 10^{8}$ & 0.29 & 0.4 & 0.44 & 0.59 \\ 
# # 93 &   1 &   2 &   2 &   4 &  -2 &   2 &   5 & 0.01 & $8.4 \times 10^{6}$ & $8.7 \times 10^{2}$ &   1 & 1.1 & $2.6 \times 10^{9}$ & $1.1 \times 10^{2}$ & $3.5 \times 10^{8}$ & 0.24 & 0.45 & 0.51 & 0.69 \\ 
# # 94 &   2 &   2 &   2 &   4 &  -2 &   2 &   5 & 0.01 & $2 \times 10^{6}$ & $3.1 \times 10^{3}$ &   4 & 1.1 & $7.7 \times 10^{8}$ & $1 \times 10^{2}$ & $2.5 \times 10^{9}$ & 0.17 & 0.34 & 0.4 & 0.6 \\ 
# # 95 &   1 &   4 &   2 &   4 &  -2 &   2 &   5 & 0.01 & $3.9 \times 10^{5}$ & $9.6 \times 10^{2}$ &   7 & 1.1 & $2.1 \times 10^{8}$ & $1.6 \times 10^{2}$ & $7.3 \times 10^{7}$ & 0.32 & 0.44 & 0.48 & 0.64 \\ 
# # 96 &   2 &   4 &   2 &   4 &  -2 &   2 &   5 & 0.01 & $3.8 \times 10^{6}$ & $5.7 \times 10^{3}$ &   5 & 1.1 & $7.8 \times 10^{8}$ & $7.5 \times 10^{2}$ & $2.5 \times 10^{7}$ & 0.16 & 0.3 & 0.34 & 0.5 \\ 
# # 97 &   1 &   2 &   1 &   2 &  -1 &   4 &   5 & 0.01 & $2.3 \times 10^{5}$ & $8.2 \times 10^{2}$ &   0 & 1.1 & $6.4 \times 10^{7}$ &  38 & $6 \times 10^{6}$ & 0.31 & 0.49 & 0.54 & 0.7 \\ 
# # 98 &   2 &   2 &   1 &   2 &  -1 &   4 &   5 & 0.01 & $3.5 \times 10^{5}$ & $1.9 \times 10^{3}$ &   2 & 1.1 & $6.4 \times 10^{7}$ & $1.3 \times 10^{2}$ & $1.2 \times 10^{6}$ & 0.23 & 0.41 & 0.47 & 0.66 \\ 
# # 99 &   1 &   4 &   1 &   2 &  -1 &   4 &   5 & 0.01 & $1 \times 10^{7}$ & $8.4 \times 10^{2}$ &   1 & 1.1 & $3 \times 10^{9}$ & $2.5 \times 10^{3}$ & $1.5 \times 10^{8}$ & 0.29 & 0.44 & 0.49 & 0.65 \\ 
# # 100 &   2 &   4 &   1 &   2 &  -1 &   4 &   5 & 0.01 & $4 \times 10^{6}$ & $1.7 \times 10^{3}$ &   2 & 1.1 & $5.7 \times 10^{8}$ & $5.1 \times 10^{3}$ & $5 \times 10^{7}$ & 0.27 & 0.4 & 0.44 & 0.57 \\ 
# # 101 &   1 &   2 &   2 &   2 &  -1 &   4 &   5 & 0.01 & $8.5 \times 10^{5}$ & $8.7 \times 10^{2}$ &   1 & 1.1 & $2.1 \times 10^{8}$ & $1.3 \times 10^{2}$ & $5.1 \times 10^{11}$ & 0.26 & 0.44 & 0.5 & 0.69 \\ 
# # 102 &   2 &   2 &   2 &   2 &  -1 &   4 &   5 & 0.01 & $4.7 \times 10^{6}$ & $7.7 \times 10^{3}$ &   2 & 1.1 & $7.9 \times 10^{8}$ & $3.2 \times 10^{2}$ & $1.4 \times 10^{10}$ & 0.077 & 0.22 & 0.28 & 0.5 \\ 
# # 103 &   1 &   4 &   2 &   2 &  -1 &   4 &   5 & 0.01 & $1.4 \times 10^{7}$ & $3.4 \times 10^{3}$ &   6 & 1.1 & $1.4 \times 10^{9}$ & $4.9 \times 10^{4}$ & $7.4 \times 10^{7}$ & 0.22 & 0.35 & 0.4 & 0.56 \\ 
# # 104 &   2 &   4 &   2 &   2 &  -1 &   4 &   5 & 0.01 & $3.1 \times 10^{6}$ & $3.2 \times 10^{3}$ &   2 & 1.1 & $2.8 \times 10^{8}$ & $1.7 \times 10^{3}$ & $2 \times 10^{7}$ & 0.21 & 0.35 & 0.4 & 0.56 \\ 
# # 105 &   1 &   2 &   1 &   4 &  -1 &   4 &   5 & 0.01 & $7.3 \times 10^{6}$ & $5.3 \times 10^{3}$ &   7 & 1.1 & $3.7 \times 10^{9}$ &  31 & $1.3 \times 10^{10}$ & 0.25 & 0.41 & 0.46 & 0.64 \\ 
# # 106 &   2 &   2 &   1 &   4 &  -1 &   4 &   5 & 0.01 & $1.1 \times 10^{6}$ & $4.8 \times 10^{3}$ &   5 & 1.1 & $1.4 \times 10^{8}$ &  59 & $2.2 \times 10^{7}$ & 0.15 & 0.32 & 0.39 & 0.61 \\ 
# # 107 &   1 &   4 &   1 &   4 &  -1 &   4 &   5 & 0.01 & $1.4 \times 10^{6}$ & $1.9 \times 10^{3}$ &   9 & 1.1 & $3.8 \times 10^{8}$ & $1.6 \times 10^{2}$ & $2.4 \times 10^{8}$ & 0.3 & 0.47 & 0.52 & 0.66 \\ 
# # 108 &   2 &   4 &   1 &   4 &  -1 &   4 &   5 & 0.01 & $2.8 \times 10^{6}$ & $1.9 \times 10^{4}$ &   6 & 1.1 & $6.1 \times 10^{8}$ & $9.6 \times 10^{2}$ & $7.6 \times 10^{7}$ & 0.18 & 0.31 & 0.35 & 0.51 \\ 
# # 109 &   1 &   2 &   2 &   4 &  -1 &   4 &   5 & 0.01 & $1.1 \times 10^{6}$ & $2.2 \times 10^{3}$ &   7 & 1.1 & $2.3 \times 10^{8}$ &  98 & $1.2 \times 10^{8}$ & 0.28 & 0.44 & 0.49 & 0.66 \\ 
# # 110 &   2 &   2 &   2 &   4 &  -1 &   4 &   5 & 0.01 & $8 \times 10^{5}$ & $7 \times 10^{3}$ &   5 & 1.1 & $2.1 \times 10^{8}$ & $1.4 \times 10^{2}$ & $8.1 \times 10^{6}$ & 0.11 & 0.26 & 0.32 & 0.51 \\ 
# # 111 &   1 &   4 &   2 &   4 &  -1 &   4 &   5 & 0.01 & $2.4 \times 10^{6}$ & $3.9 \times 10^{4}$ &   4 & 1.1 & $5.5 \times 10^{8}$ & $6.8 \times 10^{2}$ & $1.1 \times 10^{9}$ & 0.16 & 0.29 & 0.34 & 0.53 \\ 
# # 112 &   2 &   4 &   2 &   4 &  -1 &   4 &   5 & 0.01 & $3.5 \times 10^{6}$ & $2 \times 10^{4}$ &   4 & 1.1 & $2.9 \times 10^{8}$ & $1.8 \times 10^{3}$ & $1.7 \times 10^{7}$ & 0.19 & 0.31 & 0.36 & 0.51 \\ 
# # 113 &   1 &   2 &   1 &   2 &  -2 &   4 &   5 & 0.01 & $1.7 \times 10^{6}$ & $1.1 \times 10^{3}$ &   1 & 1.1 & $1.9 \times 10^{8}$ & $3.4 \times 10^{2}$ & $2.3 \times 10^{8}$ & 0.28 & 0.48 & 0.54 & 0.72 \\ 
# # 114 &   2 &   2 &   1 &   2 &  -2 &   4 &   5 & 0.01 & $2 \times 10^{6}$ & $9.4 \times 10^{3}$ &   0 & 1.1 & $5.8 \times 10^{8}$ & $3 \times 10^{2}$ & $5.9 \times 10^{7}$ & 0.16 & 0.29 & 0.34 & 0.54 \\ 
# # 115 &   1 &   4 &   1 &   2 &  -2 &   4 &   5 & 0.01 & $7.4 \times 10^{5}$ & $3.1 \times 10^{2}$ &   4 & 1.1 & $1.6 \times 10^{8}$ & $3.2 \times 10^{2}$ & $1.8 \times 10^{7}$ & 0.34 & 0.48 & 0.52 & 0.63 \\ 
# # 116 &   2 &   4 &   1 &   2 &  -2 &   4 &   5 & 0.01 & $1.6 \times 10^{6}$ & $2.7 \times 10^{3}$ &   2 & 1.1 & $2.5 \times 10^{8}$ & $3.7 \times 10^{2}$ & $3.6 \times 10^{11}$ & 0.21 & 0.34 & 0.39 & 0.56 \\ 
# # 117 &   1 &   2 &   2 &   2 &  -2 &   4 &   5 & 0.01 & $2.6 \times 10^{6}$ & $1.3 \times 10^{3}$ &   1 & 1.1 & $4.6 \times 10^{8}$ &  66 & $1.1 \times 10^{9}$ & 0.23 & 0.41 & 0.47 & 0.65 \\ 
# # 118 &   2 &   2 &   2 &   2 &  -2 &   4 &   5 & 0.01 & $2.7 \times 10^{6}$ & $5.2 \times 10^{3}$ &   3 & 1.1 & $4.8 \times 10^{8}$ & $4.3 \times 10^{2}$ & $3.1 \times 10^{9}$ & 0.15 & 0.29 & 0.35 & 0.57 \\ 
# # 119 &   1 &   4 &   2 &   2 &  -2 &   4 &   5 & 0.01 & $5.4 \times 10^{6}$ & $5.8 \times 10^{3}$ &   3 & 1.1 & $1.9 \times 10^{9}$ & $1 \times 10^{3}$ & $5.4 \times 10^{8}$ & 0.2 & 0.32 & 0.36 & 0.51 \\ 
# # 120 &   2 &   4 &   2 &   2 &  -2 &   4 &   5 & 0.01 & $3 \times 10^{6}$ & $1.2 \times 10^{4}$ &   2 & 1.1 & $8.4 \times 10^{8}$ & $1.1 \times 10^{4}$ & $1.3 \times 10^{7}$ & 0.22 & 0.31 & 0.35 & 0.51 \\ 
# # 121 &   1 &   2 &   1 &   4 &  -2 &   4 &   5 & 0.01 & $6.5 \times 10^{6}$ & $5.3 \times 10^{3}$ &   6 & 1.1 & $2.2 \times 10^{9}$ &  66 & $2.6 \times 10^{9}$ & 0.24 & 0.4 & 0.45 & 0.63 \\ 
# # 122 &   2 &   2 &   1 &   4 &  -2 &   4 &   5 & 0.01 & $5.7 \times 10^{6}$ & $3.8 \times 10^{3}$ &   4 & 1.1 & $2 \times 10^{9}$ & $3.6 \times 10^{2}$ & $2.2 \times 10^{8}$ & 0.2 & 0.4 & 0.46 & 0.64 \\ 
# # 123 &   1 &   4 &   1 &   4 &  -2 &   4 &   5 & 0.01 & $4.8 \times 10^{5}$ & $1.2 \times 10^{3}$ &   3 & 1.1 & $6.2 \times 10^{7}$ & $2.2 \times 10^{2}$ & $3.3 \times 10^{7}$ & 0.35 & 0.43 & 0.46 & 0.57 \\ 
# # 124 &   2 &   4 &   1 &   4 &  -2 &   4 &   5 & 0.01 & $2.4 \times 10^{6}$ & $1.4 \times 10^{4}$ &   7 & 1.1 & $7.1 \times 10^{8}$ & $3.7 \times 10^{2}$ & $6.2 \times 10^{8}$ & 0.23 & 0.36 & 0.4 & 0.55 \\ 
# # 125 &   1 &   2 &   2 &   4 &  -2 &   4 &   5 & 0.01 & $9.2 \times 10^{5}$ & $5.9 \times 10^{2}$ &   3 & 1.1 & $4.9 \times 10^{8}$ & $1 \times 10^{2}$ & $2.1 \times 10^{9}$ & 0.3 & 0.48 & 0.53 & 0.72 \\ 
# # 126 &   2 &   2 &   2 &   4 &  -2 &   4 &   5 & 0.01 & $4.5 \times 10^{6}$ & $2.7 \times 10^{4}$ &   5 & 1.1 & $9.1 \times 10^{8}$ & $3 \times 10^{2}$ & $1.9 \times 10^{8}$ & 0.11 & 0.24 & 0.29 & 0.48 \\ 
# # 127 &   1 &   4 &   2 &   4 &  -2 &   4 &   5 & 0.01 & $3.5 \times 10^{6}$ & $1.4 \times 10^{4}$ &   8 & 1.1 & $6.7 \times 10^{8}$ & $9.4 \times 10^{2}$ & $1.3 \times 10^{8}$ & 0.18 & 0.27 & 0.31 & 0.46 \\ 
# # 128 &   2 &   4 &   2 &   4 &  -2 &   4 &   5 & 0.01 & $5.6 \times 10^{6}$ & $1.7 \times 10^{4}$ &   6 & 1.1 & $1.3 \times 10^{9}$ & $8 \times 10^{3}$ & $2.9 \times 10^{10}$ & 0.14 & 0.26 & 0.31 & 0.49 \\ 
# # 129 &   1 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.1 & $6.5 \times 10^{5}$ & $2.6 \times 10^{2}$ &   0 & 0.87 & $1.9 \times 10^{8}$ &  30 & $4.8 \times 10^{6}$ & 0.37 & 0.55 & 0.6 & 0.74 \\ 
# # 130 &   2 &   2 &   1 &   2 &  -1 &   2 &   2 & 0.1 & $3.6 \times 10^{5}$ & $1.3 \times 10^{3}$ &   0 & 0.87 & $1.9 \times 10^{8}$ &  82 & $6.1 \times 10^{6}$ & 0.25 & 0.4 & 0.46 & 0.64 \\ 
# # 131 &   1 &   4 &   1 &   2 &  -1 &   2 &   2 & 0.1 & $1.6 \times 10^{5}$ & $8 \times 10^{2}$ &   1 & 0.87 & $8.3 \times 10^{6}$ & $3.3 \times 10^{2}$ & $5.4 \times 10^{4}$ & 0.19 & 0.3 & 0.34 & 0.48 \\ 
# # 132 &   2 &   4 &   1 &   2 &  -1 &   2 &   2 & 0.1 & $1.2 \times 10^{6}$ & $1.6 \times 10^{3}$ &   0 & 0.87 & $9.8 \times 10^{7}$ & $9.9 \times 10^{2}$ & $3.6 \times 10^{5}$ & 0.17 & 0.31 & 0.37 & 0.56 \\ 
# # 133 &   1 &   2 &   2 &   2 &  -1 &   2 &   2 & 0.1 & $1.2 \times 10^{5}$ & $7.2 \times 10^{2}$ &   0 & 0.87 & $2 \times 10^{7}$ &  60 & $6.2 \times 10^{6}$ & 0.23 & 0.41 & 0.47 & 0.67 \\ 
# # 134 &   2 &   2 &   2 &   2 &  -1 &   2 &   2 & 0.1 & $2.9 \times 10^{6}$ & $1.2 \times 10^{3}$ &   0 & 0.87 & $9.9 \times 10^{8}$ & $1.2 \times 10^{2}$ & $3.2 \times 10^{6}$ & 0.2 & 0.38 & 0.44 & 0.62 \\ 
# # 135 &   1 &   4 &   2 &   2 &  -1 &   2 &   2 & 0.1 & $3.4 \times 10^{6}$ &  99 &   1 & 0.87 & $1.2 \times 10^{9}$ & $7.9 \times 10^{2}$ & $2.3 \times 10^{7}$ & 0.36 & 0.49 & 0.53 & 0.65 \\ 
# # 136 &   2 &   4 &   2 &   2 &  -1 &   2 &   2 & 0.1 & $4.4 \times 10^{6}$ & $1.4 \times 10^{4}$ &   1 & 0.87 & $1.2 \times 10^{9}$ & $1.9 \times 10^{3}$ & $2.1 \times 10^{6}$ & 0.22 & 0.32 & 0.34 & 0.42 \\ 
# # 137 &   1 &   2 &   1 &   4 &  -1 &   2 &   2 & 0.1 & $5.7 \times 10^{6}$ & $1.4 \times 10^{3}$ &   2 & 0.87 & $1.1 \times 10^{9}$ & $6.5 \times 10^{2}$ & $1.7 \times 10^{7}$ & 0.26 & 0.42 & 0.47 & 0.65 \\ 
# # 138 &   2 &   2 &   1 &   4 &  -1 &   2 &   2 & 0.1 & $4.2 \times 10^{6}$ & $6.6 \times 10^{2}$ &   3 & 0.87 & $2.2 \times 10^{9}$ &  69 & $3.5 \times 10^{7}$ & 0.3 & 0.46 & 0.51 & 0.66 \\ 
# # 139 &   1 &   4 &   1 &   4 &  -1 &   2 &   2 & 0.1 & $2.1 \times 10^{6}$ & $8.2 \times 10^{2}$ &   2 & 0.87 & $1 \times 10^{9}$ & $1.5 \times 10^{2}$ & $3.1 \times 10^{7}$ & 0.32 & 0.44 & 0.48 & 0.62 \\ 
# # 140 &   2 &   4 &   1 &   4 &  -1 &   2 &   2 & 0.1 & $3.6 \times 10^{6}$ & $2 \times 10^{3}$ &   4 & 0.87 & $4.6 \times 10^{8}$ & $2.3 \times 10^{3}$ & $1 \times 10^{8}$ & 0.23 & 0.37 & 0.42 & 0.57 \\ 
# # 141 &   1 &   2 &   2 &   4 &  -1 &   2 &   2 & 0.1 & $4.3 \times 10^{5}$ & $1.5 \times 10^{3}$ &   3 & 0.87 & $5.3 \times 10^{7}$ & $1 \times 10^{2}$ & $1.5 \times 10^{7}$ & 0.25 & 0.4 & 0.45 & 0.63 \\ 
# # 142 &   2 &   2 &   2 &   4 &  -1 &   2 &   2 & 0.1 & $1.4 \times 10^{7}$ & $7.2 \times 10^{2}$ &   7 & 0.87 & $5.4 \times 10^{9}$ & $2.8 \times 10^{2}$ & $5.2 \times 10^{6}$ & 0.22 & 0.39 & 0.44 & 0.57 \\ 
# # 143 &   1 &   4 &   2 &   4 &  -1 &   2 &   2 & 0.1 & $1.5 \times 10^{6}$ & $2.2 \times 10^{3}$ &   5 & 0.87 & $1.3 \times 10^{8}$ & $9.5 \times 10^{2}$ & $6.4 \times 10^{6}$ & 0.26 & 0.36 & 0.4 & 0.51 \\ 
# # 144 &   2 &   4 &   2 &   4 &  -1 &   2 &   2 & 0.1 & $1.2 \times 10^{6}$ & $1.6 \times 10^{3}$ &   2 & 0.87 & $4.8 \times 10^{8}$ &  69 & $7.2 \times 10^{7}$ & 0.27 & 0.36 & 0.4 & 0.53 \\ 
# # 145 &   1 &   2 &   1 &   2 &  -2 &   2 &   2 & 0.1 & $2.3 \times 10^{4}$ & $3.4 \times 10^{2}$ &   0 & 0.87 & $3.2 \times 10^{6}$ &  23 & $1.7 \times 10^{5}$ & 0.38 & 0.58 & 0.64 & 0.8 \\ 
# # 146 &   2 &   2 &   1 &   2 &  -2 &   2 &   2 & 0.1 & $3.5 \times 10^{5}$ & $6.2 \times 10^{2}$ &   0 & 0.87 & $1 \times 10^{8}$ &  24 & $1.3 \times 10^{6}$ & 0.2 & 0.42 & 0.51 & 0.74 \\ 
# # 147 &   1 &   4 &   1 &   2 &  -2 &   2 &   2 & 0.1 & $6.5 \times 10^{4}$ & $5.1 \times 10^{2}$ &   1 & 0.87 & $4.9 \times 10^{6}$ & $2.1 \times 10^{2}$ & $2 \times 10^{5}$ & 0.35 & 0.47 & 0.51 & 0.66 \\ 
# # 148 &   2 &   4 &   1 &   2 &  -2 &   2 &   2 & 0.1 & $3.3 \times 10^{6}$ & $1.7 \times 10^{3}$ &   0 & 0.87 & $1.2 \times 10^{8}$ & $9.1 \times 10^{3}$ & $2.2 \times 10^{6}$ & 0.27 & 0.38 & 0.42 & 0.54 \\ 
# # 149 &   1 &   2 &   2 &   2 &  -2 &   2 &   2 & 0.1 & $3.1 \times 10^{5}$ & $6.2 \times 10^{2}$ &   1 & 0.87 & $1.2 \times 10^{8}$ &  17 & $3 \times 10^{6}$ & 0.33 & 0.51 & 0.57 & 0.76 \\ 
# # 150 &   2 &   2 &   2 &   2 &  -2 &   2 &   2 & 0.1 & $5.1 \times 10^{5}$ & $2.2 \times 10^{3}$ &   0 & 0.87 & $5.1 \times 10^{7}$ & $1.3 \times 10^{2}$ & $1 \times 10^{7}$ & 0.22 & 0.38 & 0.43 & 0.61 \\ 
# # 151 &   1 &   4 &   2 &   2 &  -2 &   2 &   2 & 0.1 & $3.8 \times 10^{4}$ & $7.2 \times 10^{2}$ &   1 & 0.87 & $2.8 \times 10^{6}$ & $3.7 \times 10^{2}$ & $7.5 \times 10^{5}$ & 0.27 & 0.41 & 0.46 & 0.6 \\ 
# # 152 &   2 &   4 &   2 &   2 &  -2 &   2 &   2 & 0.1 & $3 \times 10^{6}$ & $2.1 \times 10^{3}$ &   0 & 0.87 & $2.1 \times 10^{8}$ & $1.6 \times 10^{3}$ & $7.9 \times 10^{5}$ & 0.29 & 0.4 & 0.44 & 0.57 \\ 
# # 153 &   1 &   2 &   1 &   4 &  -2 &   2 &   2 & 0.1 & $4.5 \times 10^{5}$ & $8.3 \times 10^{2}$ &   3 & 0.87 & $2 \times 10^{8}$ &  27 & $2.9 \times 10^{7}$ & 0.31 & 0.47 & 0.52 & 0.69 \\ 
# # 154 &   2 &   2 &   1 &   4 &  -2 &   2 &   2 & 0.1 & $8.7 \times 10^{6}$ & $2.7 \times 10^{3}$ &   0 & 0.87 & $4.1 \times 10^{9}$ & $1 \times 10^{2}$ & $7.7 \times 10^{7}$ & 0.15 & 0.3 & 0.36 & 0.57 \\ 
# # 155 &   1 &   4 &   1 &   4 &  -2 &   2 &   2 & 0.1 & $6.7 \times 10^{5}$ & $1.3 \times 10^{3}$ &   1 & 0.87 & $5.9 \times 10^{7}$ &  77 & $5.4 \times 10^{8}$ & 0.3 & 0.43 & 0.48 & 0.63 \\ 
# # 156 &   2 &   4 &   1 &   4 &  -2 &   2 &   2 & 0.1 & $5.5 \times 10^{6}$ & $8.9 \times 10^{2}$ &   5 & 0.87 & $1.9 \times 10^{9}$ & $6.2 \times 10^{2}$ & $1.4 \times 10^{7}$ & 0.25 & 0.35 & 0.39 & 0.5 \\ 
# # 157 &   1 &   2 &   2 &   4 &  -2 &   2 &   2 & 0.1 & $5 \times 10^{6}$ & $2 \times 10^{3}$ &   3 & 0.87 & $2.7 \times 10^{9}$ & $1.5 \times 10^{2}$ & $1.4 \times 10^{8}$ & 0.26 & 0.42 & 0.47 & 0.64 \\ 
# # 158 &   2 &   2 &   2 &   4 &  -2 &   2 &   2 & 0.1 & $3 \times 10^{6}$ & $4.3 \times 10^{3}$ &   6 & 0.87 & $1.8 \times 10^{9}$ &  49 & $1.3 \times 10^{8}$ & 0.19 & 0.32 & 0.37 & 0.56 \\ 
# # 159 &   1 &   4 &   2 &   4 &  -2 &   2 &   2 & 0.1 & $2.5 \times 10^{6}$ & $2.5 \times 10^{2}$ &   6 & 0.87 & $1.1 \times 10^{9}$ &  50 & $1.7 \times 10^{9}$ & 0.38 & 0.5 & 0.54 & 0.67 \\ 
# # 160 &   2 &   4 &   2 &   4 &  -2 &   2 &   2 & 0.1 & $1.3 \times 10^{7}$ & $7 \times 10^{3}$ &   1 & 0.87 & $6.3 \times 10^{9}$ & $7.1 \times 10^{2}$ & $3.9 \times 10^{8}$ & 0.2 & 0.31 & 0.35 & 0.48 \\ 
# # 161 &   1 &   2 &   1 &   2 &  -1 &   4 &   2 & 0.1 & $4.6 \times 10^{5}$ & $8.9 \times 10^{2}$ &   0 & 0.87 & $9 \times 10^{7}$ & $1.3 \times 10^{2}$ & $2.4 \times 10^{8}$ & 0.22 & 0.4 & 0.46 & 0.63 \\ 
# # 162 &   2 &   2 &   1 &   2 &  -1 &   4 &   2 & 0.1 & $1.5 \times 10^{6}$ & $1.3 \times 10^{3}$ &   2 & 0.87 & $1.2 \times 10^{8}$ & $6.9 \times 10^{2}$ & $1.1 \times 10^{7}$ & 0.21 & 0.37 & 0.42 & 0.59 \\ 
# # 163 &   1 &   4 &   1 &   2 &  -1 &   4 &   2 & 0.1 & $6.8 \times 10^{5}$ & $7 \times 10^{2}$ &   1 & 0.87 & $7.6 \times 10^{7}$ & $5.7 \times 10^{2}$ & $3.7 \times 10^{6}$ & 0.2 & 0.34 & 0.39 & 0.55 \\ 
# # 164 &   2 &   4 &   1 &   2 &  -1 &   4 &   2 & 0.1 & $4.3 \times 10^{6}$ & $1.8 \times 10^{3}$ &   1 & 0.87 & $1.5 \times 10^{9}$ & $5.9 \times 10^{2}$ & $8.8 \times 10^{6}$ & 0.26 & 0.38 & 0.43 & 0.59 \\ 
# # 165 &   1 &   2 &   2 &   2 &  -1 &   4 &   2 & 0.1 & $3 \times 10^{6}$ & $1.5 \times 10^{3}$ &   1 & 0.87 & $3.3 \times 10^{8}$ &  64 & $7.5 \times 10^{6}$ & 0.25 & 0.44 & 0.49 & 0.67 \\ 
# # 166 &   2 &   2 &   2 &   2 &  -1 &   4 &   2 & 0.1 & $1.9 \times 10^{6}$ & $1.5 \times 10^{4}$ &   2 & 0.87 & $5 \times 10^{8}$ & $1.5 \times 10^{2}$ & $2.1 \times 10^{8}$ & 0.085 & 0.22 & 0.27 & 0.48 \\ 
# # 167 &   1 &   4 &   2 &   2 &  -1 &   4 &   2 & 0.1 & $5.8 \times 10^{5}$ & $1.6 \times 10^{3}$ &   5 & 0.87 & $2.1 \times 10^{8}$ & $1.9 \times 10^{2}$ & $1.6 \times 10^{7}$ & 0.26 & 0.38 & 0.42 & 0.56 \\ 
# # 168 &   2 &   4 &   2 &   2 &  -1 &   4 &   2 & 0.1 & $4.1 \times 10^{6}$ & $6 \times 10^{3}$ &   1 & 0.87 & $1.3 \times 10^{9}$ & $1.4 \times 10^{3}$ & $1.1 \times 10^{8}$ & 0.21 & 0.32 & 0.36 & 0.48 \\ 
# # 169 &   1 &   2 &   1 &   4 &  -1 &   4 &   2 & 0.1 & $9.4 \times 10^{4}$ & $2.9 \times 10^{3}$ &   4 & 0.87 & $2.6 \times 10^{7}$ &  55 & $1.7 \times 10^{7}$ & 0.23 & 0.4 & 0.46 & 0.66 \\ 
# # 170 &   2 &   2 &   1 &   4 &  -1 &   4 &   2 & 0.1 & $1.6 \times 10^{7}$ & $1.7 \times 10^{4}$ &   8 & 0.87 & $4 \times 10^{9}$ & $1.2 \times 10^{2}$ & $6.2 \times 10^{9}$ & 0.13 & 0.28 & 0.33 & 0.52 \\ 
# # 171 &   1 &   4 &   1 &   4 &  -1 &   4 &   2 & 0.1 & $3.4 \times 10^{6}$ & $2.1 \times 10^{3}$ &  12 & 0.87 & $1.8 \times 10^{9}$ & $3.4 \times 10^{2}$ & $3.3 \times 10^{8}$ & 0.29 & 0.43 & 0.47 & 0.6 \\ 
# # 172 &   2 &   4 &   1 &   4 &  -1 &   4 &   2 & 0.1 & $1.7 \times 10^{7}$ & $2.3 \times 10^{4}$ &   5 & 0.87 & $8.1 \times 10^{9}$ & $3 \times 10^{3}$ & $2.2 \times 10^{8}$ & 0.16 & 0.28 & 0.32 & 0.45 \\ 
# # 173 &   1 &   2 &   2 &   4 &  -1 &   4 &   2 & 0.1 & $4.4 \times 10^{6}$ & $2.5 \times 10^{3}$ &   7 & 0.87 & $1.3 \times 10^{9}$ &  67 & $3.1 \times 10^{7}$ & 0.2 & 0.36 & 0.42 & 0.61 \\ 
# # 174 &   2 &   2 &   2 &   4 &  -1 &   4 &   2 & 0.1 & $2.3 \times 10^{6}$ & $5.1 \times 10^{3}$ &   6 & 0.87 & $4.8 \times 10^{8}$ & $1.4 \times 10^{2}$ & $5.4 \times 10^{7}$ & 0.14 & 0.3 & 0.36 & 0.58 \\ 
# # 175 &   1 &   4 &   2 &   4 &  -1 &   4 &   2 & 0.1 & $1.5 \times 10^{7}$ & $7.1 \times 10^{2}$ &  10 & 0.87 & $2.9 \times 10^{9}$ & $2 \times 10^{3}$ & $3 \times 10^{9}$ & 0.39 & 0.52 & 0.56 & 0.68 \\ 
# # 176 &   2 &   4 &   2 &   4 &  -1 &   4 &   2 & 0.1 & $6.4 \times 10^{6}$ & $8.2 \times 10^{3}$ &   6 & 0.87 & $2.4 \times 10^{9}$ & $4.2 \times 10^{2}$ & $7.5 \times 10^{7}$ & 0.27 & 0.36 & 0.39 & 0.52 \\ 
# # 177 &   1 &   2 &   1 &   2 &  -2 &   4 &   2 & 0.1 & $5.3 \times 10^{4}$ & $8.4 \times 10^{2}$ &   1 & 0.87 & $7 \times 10^{6}$ &  25 & $1.5 \times 10^{7}$ & 0.25 & 0.45 & 0.52 & 0.73 \\ 
# # 178 &   2 &   2 &   1 &   2 &  -2 &   4 &   2 & 0.1 & $4.5 \times 10^{5}$ & $3.2 \times 10^{3}$ &   2 & 0.87 & $1.7 \times 10^{8}$ &  80 & $3.4 \times 10^{8}$ & 0.14 & 0.29 & 0.35 & 0.57 \\ 
# # 179 &   1 &   4 &   1 &   2 &  -2 &   4 &   2 & 0.1 & $2.8 \times 10^{6}$ & $8.5 \times 10^{2}$ &   0 & 0.87 & $7.7 \times 10^{8}$ & $5 \times 10^{3}$ & $2.6 \times 10^{9}$ & 0.26 & 0.41 & 0.46 & 0.61 \\ 
# # 180 &   2 &   4 &   1 &   2 &  -2 &   4 &   2 & 0.1 & $5.1 \times 10^{5}$ & $7 \times 10^{2}$ &   0 & 0.87 & $7.3 \times 10^{7}$ & $1.8 \times 10^{2}$ & $1.1 \times 10^{7}$ & 0.27 & 0.4 & 0.45 & 0.6 \\ 
# # 181 &   1 &   2 &   2 &   2 &  -2 &   4 &   2 & 0.1 & $1.8 \times 10^{5}$ & $1.3 \times 10^{3}$ &   1 & 0.87 & $3.5 \times 10^{7}$ &  76 & $9.2 \times 10^{7}$ & 0.26 & 0.45 & 0.52 & 0.71 \\ 
# # 182 &   2 &   2 &   2 &   2 &  -2 &   4 &   2 & 0.1 & $1.1 \times 10^{6}$ & $2.8 \times 10^{3}$ &   2 & 0.87 & $5.1 \times 10^{8}$ & $1.2 \times 10^{2}$ & $4.2 \times 10^{6}$ & 0.21 & 0.36 & 0.42 & 0.61 \\ 
# # 183 &   1 &   4 &   2 &   2 &  -2 &   4 &   2 & 0.1 & $3.4 \times 10^{5}$ & $3 \times 10^{3}$ &   1 & 0.87 & $2.5 \times 10^{7}$ & $4.2 \times 10^{2}$ & $4 \times 10^{6}$ & 0.23 & 0.36 & 0.4 & 0.56 \\ 
# # 184 &   2 &   4 &   2 &   2 &  -2 &   4 &   2 & 0.1 & $2.3 \times 10^{6}$ & $4.1 \times 10^{3}$ &   3 & 0.87 & $4.5 \times 10^{8}$ & $1.2 \times 10^{3}$ & $8.9 \times 10^{6}$ & 0.25 & 0.38 & 0.42 & 0.56 \\ 
# # 185 &   1 &   2 &   1 &   4 &  -2 &   4 &   2 & 0.1 & $2.7 \times 10^{6}$ & $7 \times 10^{2}$ &   3 & 0.87 & $8.6 \times 10^{8}$ &  94 & $3.1 \times 10^{7}$ & 0.34 & 0.54 & 0.6 & 0.75 \\ 
# # 186 &   2 &   2 &   1 &   4 &  -2 &   4 &   2 & 0.1 & $2 \times 10^{6}$ & $8.7 \times 10^{2}$ &   8 & 0.87 & $6.4 \times 10^{8}$ &  66 & $1.6 \times 10^{10}$ & 0.27 & 0.43 & 0.49 & 0.66 \\ 
# # 187 &   1 &   4 &   1 &   4 &  -2 &   4 &   2 & 0.1 & $1.2 \times 10^{7}$ & $9.9 \times 10^{2}$ &   4 & 0.87 & $3.3 \times 10^{9}$ & $1.6 \times 10^{2}$ & $6.9 \times 10^{8}$ & 0.33 & 0.44 & 0.47 & 0.57 \\ 
# # 188 &   2 &   4 &   1 &   4 &  -2 &   4 &   2 & 0.1 & $1.1 \times 10^{6}$ & $1.5 \times 10^{3}$ &   4 & 0.87 & $2.9 \times 10^{8}$ & $2.7 \times 10^{3}$ & $5.7 \times 10^{9}$ & 0.23 & 0.38 & 0.43 & 0.59 \\ 
# # 189 &   1 &   2 &   2 &   4 &  -2 &   4 &   2 & 0.1 & $8 \times 10^{5}$ & $1.2 \times 10^{3}$ &   7 & 0.87 & $2.3 \times 10^{8}$ &  25 & $1.6 \times 10^{8}$ & 0.32 & 0.49 & 0.54 & 0.7 \\ 
# # 190 &   2 &   2 &   2 &   4 &  -2 &   4 &   2 & 0.1 & $1.8 \times 10^{6}$ & $6.2 \times 10^{3}$ &   5 & 0.87 & $4.7 \times 10^{8}$ & $2.2 \times 10^{2}$ & $2 \times 10^{8}$ & 0.18 & 0.34 & 0.39 & 0.59 \\ 
# # 191 &   1 &   4 &   2 &   4 &  -2 &   4 &   2 & 0.1 & $3.3 \times 10^{6}$ & $1.3 \times 10^{3}$ &   5 & 0.87 & $8.9 \times 10^{8}$ & $2.1 \times 10^{3}$ & $1.9 \times 10^{10}$ & 0.32 & 0.44 & 0.48 & 0.61 \\ 
# # 192 &   2 &   4 &   2 &   4 &  -2 &   4 &   2 & 0.1 & $1.2 \times 10^{6}$ & $5.3 \times 10^{3}$ &   9 & 0.87 & $6.4 \times 10^{7}$ & $9.6 \times 10^{2}$ & $3.6 \times 10^{8}$ & 0.17 & 0.31 & 0.35 & 0.5 \\ 
# # 193 &   1 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.1 & $4.3 \times 10^{6}$ & $1.5 \times 10^{3}$ &   3 & 2.2 & $4.7 \times 10^{8}$ & $1.9 \times 10^{2}$ & $4.9 \times 10^{8}$ & 0.35 & 0.51 & 0.56 & 0.72 \\ 
# # 194 &   2 &   2 &   1 &   2 &  -1 &   2 &   5 & 0.1 & $3 \times 10^{6}$ & $1.1 \times 10^{3}$ &   0 & 2.2 & $7.4 \times 10^{8}$ &  78 & $3.1 \times 10^{6}$ & 0.24 & 0.42 & 0.48 & 0.66 \\ 
# # 195 &   1 &   4 &   1 &   2 &  -1 &   2 &   5 & 0.1 & $1.2 \times 10^{6}$ & $5.4 \times 10^{2}$ &   5 & 2.2 & $2.3 \times 10^{8}$ & $1.7 \times 10^{2}$ & $1.8 \times 10^{8}$ & 0.34 & 0.47 & 0.51 & 0.64 \\ 
# # 196 &   2 &   4 &   1 &   2 &  -1 &   2 &   5 & 0.1 & $6.4 \times 10^{6}$ & $6.2 \times 10^{3}$ &   1 & 2.2 & $3 \times 10^{8}$ & $2.6 \times 10^{3}$ & $8.9 \times 10^{6}$ & 0.22 & 0.33 & 0.38 & 0.52 \\ 
# # 197 &   1 &   2 &   2 &   2 &  -1 &   2 &   5 & 0.1 & $2.5 \times 10^{6}$ & $3.2 \times 10^{3}$ &   3 & 2.2 & $9.3 \times 10^{8}$ & $1.9 \times 10^{2}$ & $2.8 \times 10^{8}$ & 0.27 & 0.46 & 0.51 & 0.68 \\ 
# # 198 &   2 &   2 &   2 &   2 &  -1 &   2 &   5 & 0.1 & $2.1 \times 10^{6}$ & $4.1 \times 10^{4}$ &   5 & 2.2 & $2 \times 10^{8}$ & $4.6 \times 10^{2}$ & $4.1 \times 10^{6}$ & 0.067 & 0.17 & 0.21 & 0.39 \\ 
# # 199 &   1 &   4 &   2 &   2 &  -1 &   2 &   5 & 0.1 & $6 \times 10^{6}$ & $5.2 \times 10^{3}$ &   3 & 2.2 & $7.3 \times 10^{8}$ & $6.1 \times 10^{2}$ & $1.4 \times 10^{7}$ & 0.26 & 0.35 & 0.39 & 0.51 \\ 
# # 200 &   2 &   4 &   2 &   2 &  -1 &   2 &   5 & 0.1 & $6.1 \times 10^{6}$ & $1.3 \times 10^{4}$ &   6 & 2.2 & $6.4 \times 10^{8}$ & $5.8 \times 10^{3}$ & $3.1 \times 10^{8}$ & 0.25 & 0.37 & 0.41 & 0.56 \\ 
# # 201 &   1 &   2 &   1 &   4 &  -1 &   2 &   5 & 0.1 & $1.2 \times 10^{6}$ & $4.3 \times 10^{3}$ &   4 & 2.2 & $3 \times 10^{8}$ & $1.1 \times 10^{2}$ & $6.4 \times 10^{8}$ & 0.27 & 0.41 & 0.46 & 0.6 \\ 
# # 202 &   2 &   2 &   1 &   4 &  -1 &   2 &   5 & 0.1 & $3.6 \times 10^{6}$ & $1.3 \times 10^{3}$ &  11 & 2.2 & $1.3 \times 10^{9}$ &  71 & $4.1 \times 10^{9}$ & 0.3 & 0.49 & 0.55 & 0.7 \\ 
# # 203 &   1 &   4 &   1 &   4 &  -1 &   2 &   5 & 0.1 & $3 \times 10^{6}$ & $1.1 \times 10^{3}$ &   8 & 2.2 & $1.4 \times 10^{8}$ & $1.2 \times 10^{4}$ & $2.1 \times 10^{8}$ & 0.41 & 0.53 & 0.57 & 0.69 \\ 
# # 204 &   2 &   4 &   1 &   4 &  -1 &   2 &   5 & 0.1 & $2.9 \times 10^{6}$ & $6.5 \times 10^{2}$ &   8 & 2.2 & $1.3 \times 10^{9}$ & $2 \times 10^{3}$ & $1.3 \times 10^{9}$ & 0.32 & 0.44 & 0.47 & 0.58 \\ 
# # 205 &   1 &   2 &   2 &   4 &  -1 &   2 &   5 & 0.1 & $4.1 \times 10^{6}$ & $6.6 \times 10^{3}$ &   6 & 2.2 & $1.5 \times 10^{9}$ & $1.5 \times 10^{2}$ & $9.3 \times 10^{7}$ & 0.25 & 0.42 & 0.48 & 0.65 \\ 
# # 206 &   2 &   2 &   2 &   4 &  -1 &   2 &   5 & 0.1 & $5.7 \times 10^{6}$ & $7.6 \times 10^{3}$ &   9 & 2.2 & $1.2 \times 10^{9}$ & $6.6 \times 10^{2}$ & $1.3 \times 10^{9}$ & 0.17 & 0.34 & 0.4 & 0.58 \\ 
# # 207 &   1 &   4 &   2 &   4 &  -1 &   2 &   5 & 0.1 & $2.3 \times 10^{6}$ & $3.2 \times 10^{3}$ &  14 & 2.2 & $5.7 \times 10^{8}$ & $5.1 \times 10^{2}$ & $3.8 \times 10^{8}$ & 0.18 & 0.31 & 0.36 & 0.5 \\ 
# # 208 &   2 &   4 &   2 &   4 &  -1 &   2 &   5 & 0.1 & $1.9 \times 10^{6}$ & $1.5 \times 10^{4}$ &   5 & 2.2 & $6 \times 10^{8}$ & $2.8 \times 10^{2}$ & $1.9 \times 10^{9}$ & 0.21 & 0.35 & 0.41 & 0.58 \\ 
# # 209 &   1 &   2 &   1 &   2 &  -2 &   2 &   5 & 0.1 & $8.2 \times 10^{4}$ & $3.6 \times 10^{2}$ &   0 & 2.2 & $8.9 \times 10^{6}$ &  34 & $5.3 \times 10^{6}$ & 0.36 & 0.58 & 0.64 & 0.8 \\ 
# # 210 &   2 &   2 &   1 &   2 &  -2 &   2 &   5 & 0.1 & $1.7 \times 10^{6}$ & $1.9 \times 10^{3}$ &   5 & 2.2 & $2.2 \times 10^{8}$ &  51 & $3.3 \times 10^{7}$ & 0.25 & 0.43 & 0.49 & 0.67 \\ 
# # 211 &   1 &   4 &   1 &   2 &  -2 &   2 &   5 & 0.1 & $1.5 \times 10^{6}$ & $4.4 \times 10^{2}$ &   6 & 2.2 & $5.9 \times 10^{8}$ &  43 & $2.9 \times 10^{11}$ & 0.36 & 0.53 & 0.58 & 0.73 \\ 
# # 212 &   2 &   4 &   1 &   2 &  -2 &   2 &   5 & 0.1 & $3.4 \times 10^{6}$ & $2 \times 10^{3}$ &   3 & 2.2 & $7.5 \times 10^{8}$ & $5.6 \times 10^{3}$ & $2.6 \times 10^{6}$ & 0.21 & 0.32 & 0.36 & 0.49 \\ 
# # 213 &   1 &   2 &   2 &   2 &  -2 &   2 &   5 & 0.1 & $4.6 \times 10^{5}$ & $7.3 \times 10^{2}$ &   5 & 2.2 & $8.2 \times 10^{7}$ &  29 & $1.2 \times 10^{7}$ & 0.31 & 0.48 & 0.53 & 0.68 \\ 
# # 214 &   2 &   2 &   2 &   2 &  -2 &   2 &   5 & 0.1 & $4.3 \times 10^{6}$ & $3.9 \times 10^{3}$ &   3 & 2.2 & $1.1 \times 10^{9}$ & $2 \times 10^{2}$ & $2 \times 10^{8}$ & 0.24 & 0.43 & 0.48 & 0.64 \\ 
# # 215 &   1 &   4 &   2 &   2 &  -2 &   2 &   5 & 0.1 & $8.5 \times 10^{5}$ & $7.4 \times 10^{2}$ &   5 & 2.2 & $3.8 \times 10^{7}$ & $1.7 \times 10^{4}$ & $4.3 \times 10^{5}$ & 0.31 & 0.45 & 0.5 & 0.63 \\ 
# # 216 &   2 &   4 &   2 &   2 &  -2 &   2 &   5 & 0.1 & $1.7 \times 10^{6}$ & $1.7 \times 10^{3}$ &   2 & 2.2 & $1.8 \times 10^{8}$ & $9.2 \times 10^{2}$ & $1.3 \times 10^{10}$ & 0.27 & 0.43 & 0.48 & 0.63 \\ 
# # 217 &   1 &   2 &   1 &   4 &  -2 &   2 &   5 & 0.1 & $3.7 \times 10^{6}$ & $3.6 \times 10^{3}$ &   1 & 2.2 & $6.6 \times 10^{8}$ &  15 & $3.1 \times 10^{8}$ & 0.31 & 0.48 & 0.54 & 0.7 \\ 
# # 218 &   2 &   2 &   1 &   4 &  -2 &   2 &   5 & 0.1 & $4.5 \times 10^{6}$ & $9.9 \times 10^{3}$ &   7 & 2.2 & $8 \times 10^{8}$ &  64 & $1.9 \times 10^{9}$ & 0.2 & 0.37 & 0.42 & 0.6 \\ 
# # 219 &   1 &   4 &   1 &   4 &  -2 &   2 &   5 & 0.1 & $1.5 \times 10^{6}$ & $1.9 \times 10^{4}$ &   8 & 2.2 & $4.3 \times 10^{8}$ & $1.1 \times 10^{2}$ & $4.7 \times 10^{9}$ & 0.28 & 0.41 & 0.44 & 0.57 \\ 
# # 220 &   2 &   4 &   1 &   4 &  -2 &   2 &   5 & 0.1 & $2.7 \times 10^{6}$ & $8 \times 10^{3}$ &   6 & 2.2 & $3.3 \times 10^{8}$ & $8.8 \times 10^{2}$ & $2.3 \times 10^{7}$ & 0.23 & 0.38 & 0.43 & 0.57 \\ 
# # 221 &   1 &   2 &   2 &   4 &  -2 &   2 &   5 & 0.1 & $6.6 \times 10^{6}$ & $8.2 \times 10^{3}$ &   9 & 2.2 & $3.8 \times 10^{9}$ &  39 & $3.3 \times 10^{10}$ & 0.22 & 0.38 & 0.43 & 0.62 \\ 
# # 222 &   2 &   2 &   2 &   4 &  -2 &   2 &   5 & 0.1 & $2.2 \times 10^{6}$ & $2 \times 10^{4}$ &  10 & 2.2 & $5.2 \times 10^{8}$ & $7.1 \times 10^{2}$ & $7.8 \times 10^{8}$ & 0.14 & 0.29 & 0.34 & 0.51 \\ 
# # 223 &   1 &   4 &   2 &   4 &  -2 &   2 &   5 & 0.1 & $6.7 \times 10^{6}$ & $6.8 \times 10^{3}$ &   7 & 2.2 & $2.2 \times 10^{9}$ & $1.9 \times 10^{2}$ & $2.2 \times 10^{9}$ & 0.26 & 0.43 & 0.48 & 0.65 \\ 
# # 224 &   2 &   4 &   2 &   4 &  -2 &   2 &   5 & 0.1 & $8.6 \times 10^{6}$ & $3 \times 10^{3}$ &   5 & 2.2 & $1.6 \times 10^{9}$ & $3.4 \times 10^{2}$ & $2 \times 10^{11}$ & 0.3 & 0.43 & 0.47 & 0.61 \\ 
# # 225 &   1 &   2 &   1 &   2 &  -1 &   4 &   5 & 0.1 & $4.2 \times 10^{6}$ & $1.5 \times 10^{4}$ &   6 & 2.2 & $2.1 \times 10^{9}$ & $1.3 \times 10^{2}$ & $1.7 \times 10^{8}$ & 0.15 & 0.3 & 0.36 & 0.54 \\ 
# # 226 &   2 &   2 &   1 &   2 &  -1 &   4 &   5 & 0.1 & $5.8 \times 10^{6}$ & $3.6 \times 10^{4}$ &   4 & 2.2 & $1.5 \times 10^{9}$ & $2.1 \times 10^{2}$ & $1.7 \times 10^{9}$ & 0.19 & 0.35 & 0.4 & 0.57 \\ 
# # 227 &   1 &   4 &   1 &   2 &  -1 &   4 &   5 & 0.1 & $4.9 \times 10^{6}$ & $5.6 \times 10^{3}$ &   7 & 2.2 & $7.9 \times 10^{8}$ & $9.3 \times 10^{2}$ & $1.3 \times 10^{7}$ & 0.19 & 0.29 & 0.33 & 0.47 \\ 
# # 228 &   2 &   4 &   1 &   2 &  -1 &   4 &   5 & 0.1 & $3.8 \times 10^{6}$ & $1.5 \times 10^{4}$ &   6 & 2.2 & $2.8 \times 10^{8}$ & $1.6 \times 10^{3}$ & $9.6 \times 10^{8}$ & 0.17 & 0.31 & 0.36 & 0.52 \\ 
# # 229 &   1 &   2 &   2 &   2 &  -1 &   4 &   5 & 0.1 & $6.8 \times 10^{6}$ & $8.1 \times 10^{3}$ &   5 & 2.2 & $1.2 \times 10^{9}$ &  66 & $1.3 \times 10^{10}$ & 0.16 & 0.37 & 0.44 & 0.65 \\ 
# # 230 &   2 &   2 &   2 &   2 &  -1 &   4 &   5 & 0.1 & $8.2 \times 10^{6}$ & $3.6 \times 10^{4}$ &   4 & 2.2 & $1.7 \times 10^{9}$ & $2.7 \times 10^{2}$ & $4.8 \times 10^{11}$ & 0.12 & 0.24 & 0.29 & 0.47 \\ 
# # 231 &   1 &   4 &   2 &   2 &  -1 &   4 &   5 & 0.1 & $2.7 \times 10^{6}$ & $1.4 \times 10^{4}$ &   1 & 2.2 & $4.9 \times 10^{8}$ & $9.3 \times 10^{2}$ & $2.8 \times 10^{10}$ & 0.27 & 0.39 & 0.43 & 0.56 \\ 
# # 232 &   2 &   4 &   2 &   2 &  -1 &   4 &   5 & 0.1 & $3.2 \times 10^{6}$ & $5.3 \times 10^{4}$ &   4 & 2.2 & $5.3 \times 10^{8}$ & $8.5 \times 10^{2}$ & $3.2 \times 10^{8}$ & 0.21 & 0.3 & 0.33 & 0.46 \\ 
# # 233 &   1 &   2 &   1 &   4 &  -1 &   4 &   5 & 0.1 & $8.3 \times 10^{6}$ & $5.9 \times 10^{3}$ &   5 & 2.2 & $2.4 \times 10^{9}$ &  36 & $5.3 \times 10^{8}$ & 0.24 & 0.43 & 0.48 & 0.67 \\ 
# # 234 &   2 &   2 &   1 &   4 &  -1 &   4 &   5 & 0.1 & $9.9 \times 10^{6}$ & $8.9 \times 10^{3}$ &   4 & 2.2 & $1.4 \times 10^{9}$ & $2.9 \times 10^{2}$ & $3.2 \times 10^{9}$ & 0.27 & 0.44 & 0.49 & 0.64 \\ 
# # 235 &   1 &   4 &   1 &   4 &  -1 &   4 &   5 & 0.1 & $2 \times 10^{6}$ & $2.4 \times 10^{3}$ &  11 & 2.2 & $3 \times 10^{8}$ & $4.6 \times 10^{3}$ & $8.7 \times 10^{8}$ & 0.34 & 0.46 & 0.49 & 0.61 \\ 
# # 236 &   2 &   4 &   1 &   4 &  -1 &   4 &   5 & 0.1 & $3.8 \times 10^{6}$ & $2.4 \times 10^{4}$ &  11 & 2.2 & $6 \times 10^{8}$ & $2.4 \times 10^{3}$ & $4.7 \times 10^{10}$ & 0.2 & 0.32 & 0.36 & 0.48 \\ 
# # 237 &   1 &   2 &   2 &   4 &  -1 &   4 &   5 & 0.1 & $1.3 \times 10^{6}$ & $5.6 \times 10^{2}$ &  11 & 2.2 & $5.4 \times 10^{8}$ & $1.1 \times 10^{2}$ & $7.9 \times 10^{6}$ & 0.33 & 0.51 & 0.57 & 0.72 \\ 
# # 238 &   2 &   2 &   2 &   4 &  -1 &   4 &   5 & 0.1 & $1.1 \times 10^{7}$ & $2.3 \times 10^{4}$ &   8 & 2.2 & $4.1 \times 10^{9}$ & $3.1 \times 10^{2}$ & $5.6 \times 10^{8}$ & 0.13 & 0.27 & 0.32 & 0.5 \\ 
# # 239 &   1 &   4 &   2 &   4 &  -1 &   4 &   5 & 0.1 & $5.8 \times 10^{6}$ & $3.3 \times 10^{4}$ &  10 & 2.2 & $1.7 \times 10^{9}$ & $3.8 \times 10^{2}$ & $5.9 \times 10^{9}$ & 0.24 & 0.4 & 0.45 & 0.6 \\ 
# # 240 &   2 &   4 &   2 &   4 &  -1 &   4 &   5 & 0.1 & $5.8 \times 10^{6}$ & $7.5 \times 10^{3}$ &  15 & 2.2 & $1.1 \times 10^{9}$ & $1.4 \times 10^{2}$ & $2.1 \times 10^{9}$ & 0.26 & 0.38 & 0.43 & 0.6 \\ 
# # 241 &   1 &   2 &   1 &   2 &  -2 &   4 &   5 & 0.1 & $3 \times 10^{6}$ & $1.2 \times 10^{3}$ &   1 & 2.2 & $5.5 \times 10^{8}$ &  72 & $7.3 \times 10^{9}$ & 0.3 & 0.47 & 0.52 & 0.71 \\ 
# # 242 &   2 &   2 &   1 &   2 &  -2 &   4 &   5 & 0.1 & $5.1 \times 10^{6}$ & $2.4 \times 10^{4}$ &   5 & 2.2 & $1.2 \times 10^{9}$ & $2.2 \times 10^{2}$ & $3.8 \times 10^{8}$ & 0.15 & 0.3 & 0.35 & 0.53 \\ 
# # 243 &   1 &   4 &   1 &   2 &  -2 &   4 &   5 & 0.1 & $1.1 \times 10^{7}$ & $7.6 \times 10^{3}$ &   8 & 2.2 & $2.8 \times 10^{9}$ & $1.2 \times 10^{3}$ & $7.8 \times 10^{8}$ & 0.24 & 0.39 & 0.45 & 0.61 \\ 
# # 244 &   2 &   4 &   1 &   2 &  -2 &   4 &   5 & 0.1 & $7.8 \times 10^{6}$ & $5.2 \times 10^{3}$ &   5 & 2.2 & $9.9 \times 10^{8}$ & $1.1 \times 10^{3}$ & $1 \times 10^{7}$ & 0.23 & 0.36 & 0.4 & 0.54 \\ 
# # 245 &   1 &   2 &   2 &   2 &  -2 &   4 &   5 & 0.1 & $1.6 \times 10^{6}$ & $1.7 \times 10^{4}$ &   0 & 2.2 & $4.1 \times 10^{8}$ & $1.8 \times 10^{2}$ & $1.8 \times 10^{9}$ & 0.21 & 0.4 & 0.46 & 0.62 \\ 
# # 246 &   2 &   2 &   2 &   2 &  -2 &   4 &   5 & 0.1 & $5.4 \times 10^{6}$ & $2 \times 10^{4}$ &   9 & 2.2 & $4.2 \times 10^{8}$ & $1.6 \times 10^{2}$ & $1.7 \times 10^{8}$ & 0.081 & 0.2 & 0.25 & 0.46 \\ 
# # 247 &   1 &   4 &   2 &   2 &  -2 &   4 &   5 & 0.1 & $1 \times 10^{7}$ & $2.8 \times 10^{4}$ &   8 & 2.2 & $1.1 \times 10^{9}$ & $7 \times 10^{3}$ & $1.1 \times 10^{11}$ & 0.21 & 0.34 & 0.38 & 0.53 \\ 
# # 248 &   2 &   4 &   2 &   2 &  -2 &   4 &   5 & 0.1 & $8.1 \times 10^{6}$ & $2 \times 10^{4}$ &   6 & 2.2 & $3 \times 10^{9}$ & $2.2 \times 10^{3}$ & $2.9 \times 10^{8}$ & 0.22 & 0.35 & 0.39 & 0.52 \\ 
# # 249 &   1 &   2 &   1 &   4 &  -2 &   4 &   5 & 0.1 & $7.7 \times 10^{6}$ & $2.9 \times 10^{3}$ &   5 & 2.2 & $3 \times 10^{9}$ & $1.6 \times 10^{2}$ & $2.7 \times 10^{9}$ & 0.29 & 0.48 & 0.53 & 0.69 \\ 
# # 250 &   2 &   2 &   1 &   4 &  -2 &   4 &   5 & 0.1 & $5.4 \times 10^{5}$ & $8.9 \times 10^{3}$ &   8 & 2.2 & $1.4 \times 10^{8}$ &  73 & $7.6 \times 10^{8}$ & 0.18 & 0.32 & 0.38 & 0.57 \\ 
# # 251 &   1 &   4 &   1 &   4 &  -2 &   4 &   5 & 0.1 & $3.5 \times 10^{6}$ & $7.5 \times 10^{3}$ &   5 & 2.2 & $7 \times 10^{8}$ & $1.3 \times 10^{3}$ & $8.8 \times 10^{9}$ & 0.31 & 0.43 & 0.47 & 0.63 \\ 
# # 252 &   2 &   4 &   1 &   4 &  -2 &   4 &   5 & 0.1 & $3.1 \times 10^{6}$ & $1.2 \times 10^{3}$ &   7 & 2.2 & $6.4 \times 10^{8}$ & $2.4 \times 10^{2}$ & $2.4 \times 10^{10}$ & 0.3 & 0.45 & 0.5 & 0.66 \\ 
# # 253 &   1 &   2 &   2 &   4 &  -2 &   4 &   5 & 0.1 & $7.7 \times 10^{6}$ & $5.4 \times 10^{4}$ &  11 & 2.2 & $1.4 \times 10^{9}$ & $1.4 \times 10^{2}$ & $3.2 \times 10^{9}$ & 0.24 & 0.38 & 0.43 & 0.6 \\ 
# # 254 &   2 &   2 &   2 &   4 &  -2 &   4 &   5 & 0.1 & $3.6 \times 10^{6}$ & $2.2 \times 10^{4}$ &   7 & 2.2 & $4 \times 10^{8}$ & $1.7 \times 10^{2}$ & $3.2 \times 10^{7}$ & 0.16 & 0.3 & 0.35 & 0.51 \\ 
# # 255 &   1 &   4 &   2 &   4 &  -2 &   4 &   5 & 0.1 & $1.2 \times 10^{7}$ & $5.1 \times 10^{4}$ &   7 & 2.2 & $2 \times 10^{9}$ & $2 \times 10^{3}$ & $1.1 \times 10^{10}$ & 0.17 & 0.28 & 0.32 & 0.49 \\ 
# # 256 &   2 &   4 &   2 &   4 &  -2 &   4 &   5 & 0.1 & $1.2 \times 10^{7}$ & $5.8 \times 10^{4}$ &   9 & 2.2 & $2.4 \times 10^{9}$ & $1.6 \times 10^{3}$ & $8.2 \times 10^{9}$ & 0.23 & 0.36 & 0.4 & 0.56 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# # Run 1 Long
# load("RegPolCovFixedEffectsPrior1b_LongTest.rda")
# mean.beta0 <- c(1,2); mean.beta1 <- c(1); mean.beta2 <- c(-2) # out 1b
# sd.beta0 <- c(2); sd.beta1 <- c(2); sd.beta2 <- c(2)
# sigma.0 <- c(5); alpha.sigma <- c(0.01) # out 1b
# 
# param.list <- list(mu.beta0=mean.beta0,sd.beta0=sd.beta0,mu.beta1=mean.beta1,sd.beta1=sd.beta1,mu.beta2=mean.beta2,sd.beta2=sd.beta2,sigma.0=sigma.0,alpha.sigma=alpha.sigma)
# 
# param.combos <- expand.grid(param.list)
# 
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   # exp.sigma <-
#   print(paste0("Expectation of Prior Distribution: ", param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))))
#   print(paste0("Mean n: ", mean(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Median n: ", median(out[[i]]$results$n,na.rm=TRUE)))
#   print(paste0("Average Max Lambda: ", mean(out[[i]]$results$max.lambda,na.rm=TRUE)))
#   print(paste0("Average Median Lambda: ", mean(out[[i]]$results$med.lambda,na.rm=TRUE)))
#   print(paste0("Average Ratio of Max Lambda to Median Lambda: ", mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)))
# }
# 
# for (i in 1:length(out)){
#   print(paste0("Combination ",i,":",names(out)[i]))
#   print(paste0("Number of NAs: ", sum(is.na(out[[i]]$results$n))))
# }
# 
# # Want to place in a table:
# library(xtable)
# # First, put all the above values and turn into a data frame
# df <- data.frame("mu.beta0"=param.combos$mu.beta0,"sd.beta0"=param.combos$sd.beta0,"mu.beta1"=param.combos$mu.beta1,"sd.beta1"=param.combos$sd.beta1,"mu.beta2"=param.combos$mu.beta2,"sd.beta2"=param.combos$sd.beta2,"sigma.0"=param.combos$sigma.0,"alpha.sigma"=param.combos$alpha.sigma)
# df$ave.n <- sapply(1:length(out),function(i){mean(out[[i]]$results$n,na.rm=TRUE)})
# df$med.n <- sapply(1:length(out),function(i){median(out[[i]]$results$n,na.rm=TRUE)})
# df$nas <- sapply(1:length(out),function(i){sum(is.na(out[[i]]$results$n))})
# df$expectation <- sapply(1:length(out),function(i){param.combos$sigma.0[i]/(-log(param.combos$alpha.sigma[i]))})
# df$ave.maxLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda,na.rm=TRUE)})
# df$ave.medLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$ave.ratioLambda <- sapply(1:length(out),function(i){mean(out[[i]]$results$max.lambda/out[[i]]$results$med.lambda,na.rm=TRUE)})
# df$prop.zero1010 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad1010,na.rm=TRUE)})
# df$prop.zero2020 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2020,na.rm=TRUE)})
# df$prop.zero2525 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad2525,na.rm=TRUE)})
# df$prop.zero5050 <- sapply(1:length(out),function(i){mean(out[[i]]$results$quad5050,na.rm=TRUE)})
# 
# print(xtable(df,align=c("c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c","c"),display=c("g","g","g","g","g","g","g", "g","g","g","g","g","g","g","g","g","g", "g","g","g")), math.style.exponents = TRUE)
# # % latex table generated in R 4.0.2 by xtable 1.8-4 package
# # % Sun Jun 13 13:10:57 2021
# # \begin{table}[ht]
# # \centering
# # \begin{tabular}{cccccccccccccccccccc}
# # \hline
# # & mu.beta0 & sd.beta0 & mu.beta1 & sd.beta1 & mu.beta2 & sd.beta2 & sigma.0 & alpha.sigma & ave.n & med.n & nas & expectation & ave.maxLambda & ave.medLambda & ave.ratioLambda & prop.zero1010 & prop.zero2020 & prop.zero2525 & prop.zero5050 \\ 
# # \hline
# # 1 &   1 &   2 &   1 &   2 &  -2 &   2 &   5 & 0.01 & $1.6 \times 10^{6}$ & $5.5 \times 10^{3}$ &   1 & 1.1 & $2.5 \times 10^{8}$ & $2 \times 10^{2}$ & $2 \times 10^{6}$ & 0.16 & 0.31 & 0.36 & 0.54 \\ 
# # 2 &   2 &   2 &   1 &   2 &  -2 &   2 &   5 & 0.01 & $3 \times 10^{6}$ & $6.5 \times 10^{3}$ &   1 & 1.1 & $5.5 \times 10^{8}$ & $4.4 \times 10^{2}$ & $1.3 \times 10^{9}$ & 0.099 & 0.24 & 0.3 & 0.49 \\ 
# # \hline
# # \end{tabular}
# # \end{table}


# sessionInfo() -----------------------------------------------------------

sessionInfo()