# Multivariate INLA within MCMC Functions ---------------------------------

# Functions that will implement the Multivariate INLA within MCMC algorithm from Gomez-Rubio and Rue (Gmez-Rubio, V. and Rue, H., 2018. Markov chain Monte Carlo with the integrated nested Laplace approximation. Statistics and Computing, 28(5), pp.1033-1051.).
# lik.inla_base() - fit inla model with the sampled covariate effects held fixed and extract the INLA marginals and likelihood in order to calculate the log-posterior.
# lik.inla_bparam() - fit inla model with the sampled covariate effects held fixed and additional covariate effects calculated which are the covariate contrast between the sampled value and the optimal value through the INLA algorithm and extract the INLA marginals and likelihood in order to calculate the log-posterior.
# lik.inla_b_wrapper() - calls the correct lik.inla_*() function for each study region, depending on whether it belongs to the `base' study region or otherwise.
# log.post.fun.inla.multivar() -  calculates the log-posterior (unnormalised) for the state of the Markov Chain when sharing parameters with no contrast parameters required. Also outputs the INLA marginals and the likelihoods.
# log.post.fun.inla.multivar_base() - calculates the log-posterior (unnormalised) for the state of the Markov Chain when sharing parameters and also requiring covariate contrasts approximated. Also outputs the INLA marginals and likelihoods.
# mh.inlawmcmc.multivar() - performs the MH step of the INLA within MCMC algorithm, takes in values such as total iterations (which can also be extended when re-run), with proposal standard deviations, number of errors allowed before the function stops and returns an error, there are also indicators for whether or not we want to approximate the covariate contrasts and which study region should be the `base' study region as well as many other options described below, and outputs information on the run such as the likelihood and log-posterior, acceptance at each iteration as well as the states of the chains and the marginals from the INLA run at each iteration.
# lhyptohyp() - transform the internal hyperpar, from log() transform. Used within the BMA function in order to alter the internal.hyperparmeter output if it is needed.
# bma.inlawmcmc.multivar() - implement the BMA step of the Multivariate INLA within MCMC algorithm, with burn-in and thinning variables taken for processing the MH output before the BMA step begins. Performs the BMA step for each study region separately, with the list outputs for these posterior marginals found in a within a list for each region.
# bma.inlawmcmc.multivar.combined() - implement the BMA step of the Multivariate INLA within MCMC algorithm, with burn-in and thining variables as for the above. In addition to the parameters that would have posteriors approximated, this function accounts for the additional covariate contrast parameters, not only approximating their posterior marginals but also combining each MCMC state for a covariate effect with its respective posterior marginal for the contrast in order to approximate the posterior marginal for the non-`base' study region, that we did not previously have from considering the posterior of the shared parameter and the posterior marginal of the covariate contrasts individually.

# Author: Nadeen Khaleel

library(INLA)
library(INLABMA)
library(mvtnorm)
library(mcmcse)
library(sp)
library(spatstat)
library(RandomFields)
library(rgeos)
library(maptools)
library(raster)
library(dplyr)
library(purrr) # map function

par.lic.filepath <- "./pardiso.lic" # file path for pardiso licence if in use
inla.setOption(pardiso.license = par.lic.filepath)


#################################################################################################
#FUNCTIONS
#################################################################################################

# Sub-functions -----------------------------------------------------------


lik.inla_base <- function(data,mesh,spde,param){
  # Calculate the conditional marginal likelihood and INLA marginals for the `base' study region, so any covariate effect for the shared parameters are described wholly by the proposed shared parameter from the Metropolis-Hastings algorithm.
  
  data.temp <- data # re-label data to prevent any potential mix-ups
  data.temp$OFF <- param[1]*data.temp$cov1 + param[2]*data.temp$cov2
  
  s.index <- inla.spde.make.index("field",n.spde=spde$n.spde)
  coords <- data.temp[,c("x","y")]
  coordinates(coords) <- ~ x + y
  A <- inla.spde.make.A(mesh, loc=coords)
  stk <- inla.stack(data=list(resp=data.temp$count),A=list(A,1),effects=list(c(s.index,list(int=1)),list(OFF=data.temp$OFF,larea=log(data.temp$area))),tag='est')
  
  fit.inla <- try(inla(resp ~ 0 + offset(larea) + offset(OFF) + int + f(field,model=spde), family="poisson", data=inla.stack.data(stk),control.predictor=list(A=inla.stack.A(stk),compute=TRUE),control.fixed=list(mean=list(int=0),prec=list(int=1/100))))
  
  if (class(fit.inla)!="try-error"){
    lik.val <- fit.inla$mlik[1]
    
    # INLA output
    list.names <- c("marginals.fixed","marginals.hyperpar","internal.marginals.hyperpar")
    
    temp.mod.marg <- vector(mode="list",length=length(list.names))
    names(temp.mod.marg) <- list.names  
    
    nlist.fixed <- names(fit.inla[["marginals.fixed"]])
    nlist.hyperpar <- names(fit.inla[["marginals.hyperpar"]])
    nlist.int.hyperpar <- names(fit.inla[["internal.marginals.hyperpar"]])
    fixed.marg <- lapply(nlist.fixed,function(nl){fit.inla[["marginals.fixed"]][[nl]]})
    names(fixed.marg) <- nlist.fixed
    hyperpar.marg <- lapply(nlist.hyperpar,function(nl){fit.inla[["marginals.hyperpar"]][[nl]]})
    names(hyperpar.marg) <- nlist.hyperpar
    internal.hyperpar.marg <- lapply(nlist.int.hyperpar,function(nl){fit.inla[["internal.marginals.hyperpar"]][[nl]]})
    names(internal.hyperpar.marg) <- nlist.int.hyperpar
    
    temp.mod.marg[["marginals.fixed"]] <- fixed.marg
    temp.mod.marg[["marginals.hyperpar"]] <- hyperpar.marg
    temp.mod.marg[["internal.marginals.hyperpar"]] <- internal.hyperpar.marg
    
    return(list(lik.val=lik.val,tempmod=temp.mod.marg))
  } else {
    lik.val <- "fail"
    return(list(lik.val=lik.val))
  }
}

lik.inla_bparam <- function(data,mesh,spde,param,b.prior.mean,b.prior.sd){
  # Calculate the conditional marginal likelihood and INLA marginals for the non-`base' regions, where the inclusion of the offset shared parameter values and the estimate of the covariate effects allows the approximation of the conditional posterior marginal for the covariate contrast between this study region and the `base' study region.
  
  data.temp <- data # re-label data to prevent any potential mix-ups
  data.temp$OFF <- param[1]*data.temp$cov1 + param[2]*data.temp$cov2
  
  s.index <- inla.spde.make.index("field",n.spde=spde$n.spde)
  coords <- data.temp[,c("x","y")]
  coordinates(coords) <- ~ x + y
  A <- inla.spde.make.A(mesh, loc=coords)
  stk <- inla.stack(data=list(resp=data.temp$count),A=list(A,1),effects=list(c(s.index,list(int=1)),list(cov1=data.temp$cov1,cov2=data.temp$cov2,OFF=data.temp$OFF,larea=log(data.temp$area))),tag='est')
  
  # Include cov1 and cov2, whose estimated effect can inform us whether or not there is evidence of a difference in the effects of cov1 and cov2 between the cities.
  fit.inla <- try(inla(resp ~ 0 + offset(larea) + offset(OFF) + int + cov1 + cov2 + f(field,model=spde), family="poisson", data=inla.stack.data(stk),control.predictor=list(A=inla.stack.A(stk),compute=TRUE),control.fixed=list(mean=list(int=0,cov1=b.prior.mean[1],cov2=b.prior.mean[2]),prec=list(int=1/100,cov1=1/(b.prior.sd[1]^2),cov2=1/(b.prior.sd[2]^2)))))
  
  if (class(fit.inla)!="try-error"){
    lik.val <- fit.inla$mlik[1]
    
    # INLA output
    list.names <- c("marginals.fixed","marginals.hyperpar","internal.marginals.hyperpar")
    
    temp.mod.marg <- vector(mode="list",length=length(list.names))
    names(temp.mod.marg) <- list.names  
    
    nlist.fixed <- names(fit.inla[["marginals.fixed"]])
    nlist.hyperpar <- names(fit.inla[["marginals.hyperpar"]])
    nlist.int.hyperpar <- names(fit.inla[["internal.marginals.hyperpar"]])
    fixed.marg <- lapply(nlist.fixed,function(nl){fit.inla[["marginals.fixed"]][[nl]]})
    names(fixed.marg) <- nlist.fixed
    hyperpar.marg <- lapply(nlist.hyperpar,function(nl){fit.inla[["marginals.hyperpar"]][[nl]]})
    names(hyperpar.marg) <- nlist.hyperpar
    internal.hyperpar.marg <- lapply(nlist.int.hyperpar,function(nl){fit.inla[["internal.marginals.hyperpar"]][[nl]]})
    names(internal.hyperpar.marg) <- nlist.int.hyperpar
    
    temp.mod.marg[["marginals.fixed"]] <- fixed.marg
    temp.mod.marg[["marginals.hyperpar"]] <- hyperpar.marg
    temp.mod.marg[["internal.marginals.hyperpar"]] <- internal.hyperpar.marg
    
    return(list(lik.val=lik.val,tempmod=temp.mod.marg))
  } else {
    lik.val <- "fail"
    return(list(lik.val=lik.val))
  }
}

lik.inla_b_wrapper <- function(data,mesh,spde,param,b.prior.mean,b.prior.sd,base.ind){
  # Wrapper function, to split the data between `base' and non-`base' study regions in order to implement the correct INLA model for the data. This allows the mclapply() to call only one function and improve the parallelisation of this code, where base.ind indication whether this study region belongs to a `base' study region or otherwise.
  
  if (base.ind==1){
    out <- lik.inla_base(data=data,mesh=mesh,spde=spde,param=param)
  } else {
    out <- lik.inla_bparam(data=data,mesh=mesh,spde=spde,param=param,b.prior.mean=b.prior.mean,b.prior.sd=b.prior.sd)
  }
}

log.post.fun.inla.multivar <- function(data.list,param,mesh.list,spde.list,prior.mean,prior.sd){
  # Calculate the log-posterior (unnormalised) for acceptance probability in MH algorithm, when we are only interested in the shared parameters, and do not want to calculate the covariate contrasts between the study regions.
  
  lik.inla.out <- mclapply(1:length(data.list),function(i,dl,ml,sl,p){lik.inla_base(dl[[i]],ml[[i]],sl[[i]],p)},dl=data.list,ml=mesh.list,sl=spde.list,p=param,mc.cores=length(data.list))
  lik.val.vec <- unlist(map(lik.inla.out,"lik.val"))
  if (sum(lik.val.vec=="fail")>0){
    return(list(type="fail"))
  } else {
    log.beta.prior <- dmvnorm(param,mean=prior.mean,sigma=diag(prior.sd^2,length(prior.mean)),log=TRUE)
    log.post.fun <- sum(lik.val.vec) + log.beta.prior
    temp.mod.marg.list <- map(lik.inla.out,"tempmod")
    if (is.null(names(data.list))){
      names(temp.mod.marg.list) <- paste0("Area",1:length(data.list))
    } else {
      names(temp.mod.marg.list) <- names(data.list)
    }
    return(list(type="success",likval=lik.val.vec,logpost=log.post.fun,marginals=temp.mod.marg.list))
  }
}

log.post.fun.inla.multivar_base <- function(data.list,param,mesh.list,spde.list,prior.mean,prior.sd,which.base=1,b.prior.mean,b.prior.sd){
  # Calculate the log-posterior (unnormalised) for acceptance probability in MH algorithm, where we have one `base' sutyd region indicated by which.base and the rest are the non-base study regions.
  
  base.indicator <- rep(0,length(data.list));
  base.indicator[which.base] <- 1
  lik.inla.out <- mclapply(1:length(data.list),function(i,dl,ml,sl,p,bm,bs,b.ind){lik.inla_b_wrapper(data=dl[[i]],mesh=ml[[i]],spde=sl[[i]],param=p,b.prior.mean=bm,b.prior.sd=bs,base.ind=b.ind[i])},dl=data.list,ml=mesh.list,sl=spde.list,p=param,bm=b.prior.mean,bs=b.prior.sd,b.ind=base.indicator,mc.cores=length(data.list))
  
  lik.val.vec <- unlist(map(lik.inla.out,"lik.val"))
  if (sum(lik.val.vec=="fail")>0){
    return(list(type="fail"))
  } else {
    log.beta.prior <- dmvnorm(param,mean=prior.mean,sigma=diag(prior.sd^2,length(prior.mean)),log=TRUE)
    log.post.fun <- sum(lik.val.vec) + log.beta.prior
    temp.mod.marg.list <- map(lik.inla.out,"tempmod")
    if (is.null(names(data.list))){
      names(temp.mod.marg.list) <- paste0("Area",1:length(data.list))
    } else {
      names(temp.mod.marg.list) <- names(data.list)
    }
    return(list(type="success",likval=lik.val.vec,logpost=log.post.fun,marginals=temp.mod.marg.list))
  }
}



####################################################################

# Main Functions ----------------------------------------------------------


# MCMC --------------------------------------------------------------------

mh.inlawmcmc.multivar <- function(data.list,spde.list,mesh.list,its,init=NULL,prior.mean,prior.sd,prop.sd,param.names,restart=0,form="orig",which.base=NULL,b.prior.mean=NULL,b.prior.sd=NULL,save.name="IwMMultiMH.rda",lb.iterr=100,lb.buff=50,it.err.lim=2,tot.err.lim=10)
{
  # This is for the model with covariates and therefore we want to perform the MCMC on the betas.
  
  # INPUTS:
  # data.list - list of count data frame of the point pattern for each city
  # spde - the spde set up for INLA on the input mesh with the pre-chosen values of for the covariance priors
  # mesh - pre-produced mesh for INLA run, to ensure consistency with every INLA run
  # its - iterations for MCMC algorithm (not total iterations, so if more iterations are needed, e.g. have already run 500, but want 1000, set its = 500, essentially how many more iterations)
  # init - initial values for the parameters of MCMC.
  # prior.mean - mean for the priors of the covariate effects being sampled.
  # prior.sd - sd for the priors of the covariate effects being sampled.
  # prop.sd - standard deviation for the proposal distribution for sampling.
  # param.names - character vector for parameter names
  # save.name - file to save output (inc. ".rda")
  # restart - 0 if starting MCMC run afresh, otherwise set to 1 re-starting with a previous run and MCMC output
  # lb.iterr - limit before which any INLA errors will cause the entire function to stop (essentially should be at least as large as the burn-in), otherwise we can proceed by sampling from previous parameter values to move past (while storing information) the error.
  # form - "orig" if no extra parameters included for covariate effects to assess difference between cities (using this for full v. sparse), or "base" to include extra parameters, for which the which.base must be an integer between 1 and the total number of cities.
  # which.base - if form="base" then which city is the base and so does not have the extra covariate effects.
  # b.prior.mean - prior mean for non-base covariate effects.
  # b.prior.sd - prior standard deviation for non-base covariate effects.
  # it.err.lim - number of errors for a single iterations for which if err>=it.err.lim for iteration i the function will stop and spit out an error.
  # tot.err.lim - the total number of errors within the MH run for which the function will stop and spit out an error.
  
  # OUTPUTS:
  # out - a list of length 2:
  # run - information from the MCMC: (NB: FIRST VALUES ARE THE INITIAL VALUES!!!)?
  # theta (data frame) - parameter values for each state of the chain for each iteration
  # acc.rej (binary vector) - whether each step resulted in an acceptance or rejection
  # logpost.lik (data frame) - log-posterior and log likelihood for the respective state of the chain
  # ess (numeric) - effective sample size of the chain so far
  # error (data frame) - data frame containing any iteration of which there was an error and the (proposed?) parameter values for which there was an error
  # inla - inla output: the posterior marginals for BMA
  
  if (length(data.list)!=length(spde.list)|length(data.list)!=length(mesh.list)){
    stop("Differing length of data sets and SPDE or Meshes.")
  }
  if(form=="base"&is.null(which.base)){
    stop("Need an integer value denoting the city that is the 'base' which does not include extra covariate effects.")
  }
  
  if (restart==0){ # not re-starting, set up new list
    out <- vector(mode="list",length=2)
    names(out) <- c("run","inla")
    out$run <- vector(mode="list",length=5)
    names(out$run) <- c("theta","acc.rej","logpost.lik","ess","error")
    out$run$theta <- data.frame(matrix(rep(NA,length(init)*its), ncol = length(init), nrow = (its)))
    colnames(out$run$theta) <- param.names
    out$run$acc.rej <- rep(NA,(its))
    out$run$logpost.lik <- data.frame(matrix(rep(NA,its*(length(data.list)+1)),ncol=(length(data.list)+1))) # lik.val=rep(NA,(its)),log.post=rep(NA,(its))
    colnames(out$run$logpost.lik) <- c(paste0("lik.val",1:length(data.list)),"log.post")
    out$run$ess <- rep(NA, length(init))
    out$run$error <-  data.frame(matrix(vector(), ncol = (2*length(init)+2), nrow = 0)) # how many errors in INLA for the algorithm and which iteration did it result from and what were the parameter values
    colnames(out$run$error) <- c("iteration",paste0(param.names,"_curr"),paste0(param.names,"_prop"),"replacement iterations")
    out$inla <- vector(mode="list",length=(its))
    names(out$inla) <- paste0("Model",1:(its))
    
    
    theta.c <- init
    if (form=="orig"){
      log.post.c <- log.post.fun.inla.multivar(data.list,theta.c,mesh.list,spde.list,prior.mean,prior.sd)
    } else {
      log.post.c <- log.post.fun.inla.multivar_base(data.list,theta.c,mesh.list,spde.list,prior.mean,prior.sd,which.base=which.base,b.prior.mean=b.prior.mean,b.prior.sd=b.prior.sd)
    }
    
    start.it <- 1
    
  } else {
    load(save.name)
    out.old <- out
    l.old <- sum(!is.na(out.old$run$theta[,1]))
    
    # Set-up new output
    out <- vector(mode="list",length=2)
    names(out) <- c("run","inla")
    out$run <- vector(mode="list",length=5)
    names(out$run) <- c("theta","acc.rej","logpost.lik","ess","error")
    out$run$theta <- data.frame(matrix(rep(NA,ncol(out.old$run$theta)*(l.old+its)), ncol = ncol(out.old$run$theta), nrow = (l.old+its)))
    colnames(out$run$theta) <- param.names
    out$run$acc.rej <- rep(NA,(l.old+its))
    out$run$logpost.lik <- data.frame(matrix(rep(NA,(l.old+its)*(length(data.list)+1)),ncol=(length(data.list)+1))) # lik.val=rep(NA,(its)),log.post=rep(NA,(its))
    colnames(out$run$logpost.lik) <- c(paste0("lik.val",1:length(data.list)),"log.post")
    out$inla <- vector(mode="list",length=(l.old+its))
    
    out$run$theta[1:l.old,] <- out.old$run$theta[1:l.old,]
    out$run$acc.rej[1:l.old] <- out.old$run$acc.rej[1:l.old]
    out$run$logpost.lik[1:l.old,] <- out.old$run$logpost.lik[1:l.old,]
    out$run$ess <- out.old$run$ess
    out$run$error <- out.old$run$error
    out$inla[1:l.old] <- out.old$inla[1:l.old]
    
    names(out$inla) <- paste0("Model",1:(l.old+its))
    
    theta.c <- as.numeric(as.vector(out$run$theta[l.old,]))
    log.post.c <- list(likval=out$run$logpost.lik[l.old,1:length(data.list)],logpost=out$run$logpost.lik$log.post[l.old],marginals=out$inla[[l.old]])
    
    start.it <- l.old + 1
  }
  
  rep.iteration.min <- lb.iterr + lb.buff # - don't want to just be sampling from 1 or 2 values? burn-in + some for lower bound
#  #######
#  # create progress bar
#  if (restart==0){
#    total.its.bar <- its
#  } else {
#    total.its.bar <- l.old + its  
#  }
#  # pb <- txtProgressBar(min = 0, max = its, style = 3)
#  pb <- txtProgressBar(min = 0, max = total.its.bar, style = 3)
#  Sys.sleep(0.1)
#  # update progress bar
#  setTxtProgressBar(pb, (start.it-1))

  for (i in start.it:(start.it + its - 1)){
    
    it.err <- 0
    next.state <- 0
    # While loop, if there is no INLA error, we move onto the next iteration, next.step=1, and break out of the while loop and move onto i+1.
    # However, if there is an INLA error we use a method from the INLABMA package, in particular the INLAMH code (Roger S. Bivand, Virgilio Gomez-Rubio, Havard Rue (2015). Spatial Data Analysis with R-INLA with Some Extensions. Journal of Statistical Software, 63(20), 1-31. URL http://www.jstatsoft.org/v63/i20/.), where either:
    # (a) - we are too early in our simulations, stop function and print error message.
    # (b) - we are `far` enough in our iterations that we can (as in INLAMH function) replace our current state of the chain by some randomly selected older parameter values and re-propose.
    # In the case of (b) we limit the number of times we allow this to happen per iteration (maybe even 2) - while also noting and saving the error table which contains the iteration at which the error occurred and the state of the chain where the error occurred. So either, too many errors (WARNING) or INLA ran fine and we move onto the next iteration.
    while (it.err < it.err.lim & next.state == 0){
      
      theta.p <- theta.c + rmvnorm(1,mean=rep(0,length(theta.c)),sigma=diag(prop.sd^2,length(theta.c)))
      
      if (form=="orig"){
        log.post.p <- log.post.fun.inla.multivar(data.list,theta.p,mesh.list,spde.list,prior.mean,prior.sd)
      } else {
        log.post.p <- log.post.fun.inla.multivar_base(data.list,theta.p,mesh.list,spde.list,prior.mean,prior.sd,which.base=which.base,b.prior.mean=b.prior.mean,b.prior.sd=b.prior.sd)
      }
      
      
      if (log.post.p$type=="success"){
        alpha <- exp((log.post.p$logpost)-(log.post.c$logpost)) # Using Random Walk MH
        
        if (alpha > runif(1)){
          out$run$acc.rej[i] <- 1
          out$run$theta[i,] <- theta.p
          out$run$logpost.lik$log.post[i] <- log.post.p$logpost
          out$run$logpost.lik[i,1:length(data.list)] <- log.post.p$likval
          out$inla[[i]] <- log.post.p$marginals
          
          # Proposed state is now current state
          theta.c <- theta.p
          log.post.c <- log.post.p
        } else {
          out$run$acc.rej[i] <- 0
          out$run$theta[i,] <- theta.c
          out$run$logpost.lik$log.post[i] <- log.post.c$logpost
          out$run$logpost.lik[i,1:length(data.list)] <- log.post.c$likval
          out$inla[[i]] <- log.post.c$marginals
        }
        if (i >= 1e3){
          out$run$ess <- sapply(1:ncol(out$run$theta),function(j){ess(out$run$theta[(1:i),j])})
        }
        print(i)
        next.state <- 1
        save(out,file=save.name)
        
#        Sys.sleep(0.1)
#        # update progress bar
#        setTxtProgressBar(pb, i)
        
      } else {
        
        # WHAT TO DO IF INLA HAS ISSUES
        # Using method in INLAMH function from the INLABMA package (cited below), replace current state by older state and re-propose new state with accept/reject decision
        if (i <= rep.iteration.min){
          err.count <- nrow(out$run$error)
          out$run$error[(err.count + 1),1] <- i
          out$run$error[(err.count + 1),2:(length(param.names)+1)] <- theta.c
          out$run$error[(err.count + 1),(length(param.names)+2):(2*length(param.names)+1)] <- theta.p
          stop(paste0("INLA error occured in iteration ",i ," of MH run. Too early in the chain to replace current value."))
        } else {
          print(paste0("INLA run with parameter values ", theta.p, " at iteration ", i," resulted in error"))
          
          err.count <- nrow(out$run$error)
          out$run$error[(err.count + 1),1] <- i
          out$run$error[(err.count + 1),2:(length(param.names)+1)] <- theta.c
          out$run$error[(err.count + 1),(length(param.names)+2):(2*length(param.names)+1)] <- theta.p
          
          if (nrow(out$run$error) >= tot.err.lim){
            print(out$run$error)
            stop(paste0("There have been ",tot.err.lim," errors, reaching the maximum limit of INLA errors for this MCMC run."))
          }
          
          # Idea borrowed from INLAMH in INLABMA package: replace this with the (non-burnin) older samples..
          rep.i <- sample((rep.iteration.min:(i-2)),1) # in INLAMH: only look at "save" values -they don't keep all simulations
          
          out$run$error[(err.count + 1),(2*length(param.names)+2)] <- rep.i
          
          theta.c <- as.numeric(as.vector(out$run$theta[rep.i,]))
          log.post.c <- list(likval=out$run$logpost.lik[rep.i,1:length(data.list)],logpost=out$run$logpost.lik$log.post[rep.i],marginals=out$inla[[rep.i]])
          
          if ((i-1)>=1e3){
            out$run$ess <- sapply(1:ncol(out$run$theta),function(j){ess(out$run$theta[(1:(i-1)),j])})
          }
          it.err <- it.err + 1
        }
        if (it.err >= it.err.lim){
          stop(paste0("Repeatedly (",it.err,") received errors for iteration ",i))
        }
      }
    }
  }
#  close(pb)
#  ########
  
  return(out)
}



# BMA ---------------------------------------------------------------------


# Transform the internal.marginals.hyperpar to the output parameter forms (exp(log(x))).
lhyptohyp <- function(marg){inla.tmarginal(function(x){exp(x)},as.matrix(marg))}

bma.inlawmcmc.multivar <- function(mh.inlawmcmc.out,burnin=0,thin=1,save.name="IwMMultiBMA.rda"){
  # Runs the Bayesian Model Averaging step of the INLA with MCMC algorithm, using the INLABMA function fitmargBMA2.
  # 
  # INPUTS:
  # mh.inlawmcmc.out - output from the MH step of the INLA with MCMC algorithm (mh.inlawmcmc function), taking in the posterior marginal outputs saved from the MH run at each state as well as the states of the chain.
  # burnin - how much of a burn in is required, no burn in is implemented in the mh.inlawmcm function.
  # thin - how much thinning is required to the full MCMC output.
  # save.name - file name for saving the output.
  # 
  # OUTPUTS:
  # theta - the thinned theta chain.
  # ess - the effective sample size of the thinned theta chain.
  # marginals - the approximation posterior marginals for the remaining parameters.
  
  # Thin the theta chains
  step_thin <- seq(from=(burnin+1),to=nrow(mh.inlawmcmc.out$run$theta),by=thin)
  theta_thin <- mh.inlawmcmc.out$run$theta[step_thin,]
  ess_thin <- sapply(1:ncol(theta_thin),function(j){ess(theta_thin[,j])})
  
  # Output list
  out <- vector(mode="list",length=3)
  names(out) <- c("run","inla.mh","inla.bma")
  out$run <- vector(mode="list",length=3)
  names(out$run) <- c("theta","logpost.lik","ess")
  out$run$theta <- theta_thin
  colnames(out$run$theta) <- colnames(mh.inlawmcmc.out$run$theta)
  out$run$logpost.lik <- out$run$logpost.lik[step_thin,]
  out$run$ess <- ess_thin
  out$inla.mh <- mh.inlawmcmc.out$inla[step_thin] # thinned list of marginals, note that each i in step_thin, contains marginals for the multiple cities
  
  # How many cities/data sets do we have?
  n.c <- length(out$inla.mh[[1]])
  
  # BMA Approximated Posterior Marginals
  out$inla.bma <- mclapply(1:n.c,function(i){
    # Extract the marginals
    listmarg <- c("marginals.fixed", "marginals.hyperpar","internal.marginals.hyperpar")
    marg.all <- map(mh.inlawmcmc.out$inla[step_thin],i)
    
    ws <- rep(1/length(marg.all),length(marg.all))
    margeff <- mclapply(listmarg, function(X){INLABMA:::fitmargBMA2(marg.all, ws, X)})
    names(margeff) <- listmarg
    
    # BMA Approximated Posterior Marginals
    return(margeff)
  })
  names(out$inla.bma) <- names(out$inla.mh[[1]])
  
  for (i in 1:length(out$inla.bma)){
    for (j in 1:length(out$inla.bma[[i]]$internal.marginals.hyperpar)){
      out$inla.bma[[i]]$transformed.internal.marginals.hyperpar[[j]] <- lhyptohyp(out$inla.bma[[i]]$internal.marginals.hyperpar[[j]])
    }
    names(out$inla.bma[[i]]$transformed.internal.marginals.hyperpar) <- paste0("Transformed ",names(out$inla.bma[[i]]$internal.marginals.hyperpar))
  }
  
  save(out,file=save.name)
  
  return(out)
}

# The below code, is an addition and was used for the Multivarite INLA withing MCMC algorithm implementations for my thesis results when covariates contrasts are calculated as this also combines the samples from the MCMC chain with the respective posterior marginals for the covariate effects in order to approximate the posterior marginal of the true covariate effect for the non-base study regions.
bma.inlawmcmc.multivar.combined <- function(mh.inlawmcmc.out,which.comb=2,burnin=0,thin=1,save.name="IwMMultiBMACombined.rda"){
  # Runs the Bayesian Model Averaging step of the INLA with MCMC algorithm, using the INLABMA function fitmargBMA2.
  # 
  # INPUTS:
  # mh.inlawmcmc.out - output from the MH step of the INLA with MCMC algorithm (mh.inlawmcmc function), taking in the posterior marginal outputs saved from the MH run at each state as well as the states of the chain.
  # which.comb - which (index) study region do we want to combine the MCMC parameters to the marginals b_i, usually window 2
  # burnin - how much of a burn in is required, no burn in is implemented in the mh.inlawmcm function.
  # thin - how much thinning is required to the full MCMC output.
  # save.name - file name for saving the output.
  # 
  # OUTPUTS:
  # theta - the thinned theta chain.
  # ess - the effective sample size of the thinned theta chain.
  # marginals - the approximation posterior marginals for the remaining parameters.
  
  # Thin the theta chains
  step_thin <- seq(from=(burnin+1),to=nrow(mh.inlawmcmc.out$run$theta),by=thin)
  theta_thin <- mh.inlawmcmc.out$run$theta[step_thin,]
  ess_thin <- sapply(1:ncol(theta_thin),function(j){ess(theta_thin[,j])})
  
  # Output list
  out <- vector(mode="list",length=3)
  names(out) <- c("run","inla.mh","inla.bma")
  out$run <- vector(mode="list",length=3)
  names(out$run) <- c("theta","logpost.lik","ess")
  out$run$theta <- theta_thin
  colnames(out$run$theta) <- colnames(mh.inlawmcmc.out$run$theta)
  out$run$logpost.lik <- out$run$logpost.lik[step_thin,]
  out$run$ess <- ess_thin
  out$inla.mh <- mh.inlawmcmc.out$inla[step_thin] # thinned list of marginals, note that each i in step_thin, contains marginals for the multiple cities
  
  # How many cities/data sets do we have?
  n.c <- length(out$inla.mh[[1]])
  
  # BMA Approximated Posterior Marginals
  out$inla.bma <- mclapply(1:n.c,function(i,wc){
    # Extract the marginals
    listmarg <- c("marginals.fixed", "marginals.hyperpar","internal.marginals.hyperpar")
    marg.all <- map(mh.inlawmcmc.out$inla[step_thin],i)
    
    ws <- rep(1/length(marg.all),length(marg.all))
    margeff <- mclapply(listmarg, function(X){INLABMA:::fitmargBMA2(marg.all, ws, X)})
    names(margeff) <- listmarg
    
    if (i==wc){
      fixed.all <- map(marg.all,"marginals.fixed")
      fixed.cov <- map(fixed.all,`[`,c("cov1","cov2"))
      
      
      for (kk in 1:length(fixed.cov)){
        ll <- nrow(fixed.cov[[kk]]$cov1)
        fixed.cov[[kk]]$cov1[,1] <- fixed.cov[[kk]]$cov1[,1] + rep(theta_thin$beta1[kk],ll)
        ll <- nrow(fixed.cov[[kk]]$cov2)
        fixed.cov[[kk]]$cov2[,1] <- fixed.cov[[kk]]$cov2[,1] + rep(theta_thin$beta2[kk],ll)
      }
      
      comb.marg <- "combined.marginals.fixed"
      fixed.list <- vector(mode="list",length=length(fixed.cov))
      names(fixed.list) <- names(fixed.cov)
      for (kk in 1:length(fixed.list)){
        fixed.list[[kk]] <- vector(mode="list",length=1)
        names(fixed.list[[kk]]) <- comb.marg
        fixed.list[[kk]]$combined.marginals.fixed <- fixed.cov[[kk]]
      }
      
      margeff.comb <- INLABMA:::fitmargBMA2(fixed.list, ws,comb.marg)
      
      margeff.all <- margeff
      margeff.all$combined.marginals.fixed <- margeff.comb
    } else {
      margeff.all <- margeff
    }
    
    # BMA Approximated Posterior Marginals
    return(margeff.all)
  },wc=which.comb)
  names(out$inla.bma) <- names(out$inla.mh[[1]])
  
  for (i in 1:length(out$inla.bma)){
    for (j in 1:length(out$inla.bma[[i]]$internal.marginals.hyperpar)){
      out$inla.bma[[i]]$transformed.internal.marginals.hyperpar[[j]] <- lhyptohyp(out$inla.bma[[i]]$internal.marginals.hyperpar[[j]])
    }
    names(out$inla.bma[[i]]$transformed.internal.marginals.hyperpar) <- paste0("Transformed ",names(out$inla.bma[[i]]$internal.marginals.hyperpar))
  }
  
  save(out,file=save.name)
  return(out)
}


# INLABMA package
# Roger S. Bivand, Virgilio Gomez-Rubio, Havard Rue (2015). Spatial Data Analysis with R-INLA with Some Extensions. Journal of Statistical Software, 63(20), 1-31. URL http://www.jstatsoft.org/v63/i20/.
#' @Article{,
#'   title = {Spatial Data Analysis with {R}-{INLA} with Some
#'     Extensions},
#'   author = {Roger S. Bivand and Virgilio G\'omez-Rubio and H{\aa}vard
#'       Rue},
#'     journal = {Journal of Statistical Software},
#'     year = {2015},
#'     volume = {63},
#'     number = {20},
#'     pages = {1--31},
#'     url = {http://www.jstatsoft.org/v63/i20/},
#'   }