# Balena Timing Tests for Multivariate INLA within MCMC -------------------

# This script is for quick tests for the timings of IwM runs when we compare running the Multivariate INLA with MCMC for 2 vs 3 cities, in particular LA and NYC/Portland vs LA and NYC AND Portland.
# I will consider Grid 3 and Mesh 3 for all of the cities (~1km-by-1km) and will be using the version of the Multivariate INLA with MCMC where we want to test the difference between covariate effects between some base city and the other cities.

# Author: Nadeen Khaleel


# Loading Libraries and Functions -----------------------------------------

source("lgcp_inla_w_mcmc_multivar_functions_wtiming_final.R") # this should load up all relevant libraries

library(unixtools)
unixtools::set.tempdir("/local/")

# Loading Data ------------------------------------------------------------

# Los Angeles
LA.file <- "LA2015CT4872CountData_projFinalScale.rda"
LA.window <- "LAWindowProjScale.rda"
LA.mesh <- "LAMesh4872_projFinalScale.rda"

# New York
NY.file <- "NY2015CT4748CountData_projFinalScale.rda"
NY.window <- "NYWindowProjScale.rda"
NY.mesh <- "NYMesh4748_projFinalScale.rda"

# Portland
P.file <- "P2015CT3826CountData_projFinalScale.rda"
P.window <- "PWindowProjScale.rda"
P.mesh <- "PMesh3826_projFinalScale.rda"


# Set-Up ------------------------------------------------------------------

# Priors
# These priors are old and not necessarily what would be implemented within the actual model fits, these values are mainly just for testing.
rho.star <- vector(mode="list",length=3)
sigma.star <- vector(mode="list",length=3)
for (i in 1:3){
  rho.star[[i]] <- c(0.35,0.01)
  sigma.star[[i]] <- c(2,0.1)
}

beta.prior.means <- rep(0,2); beta.prior.sd <- rep(sqrt(1000),2)
init.vals <- rep(0,2)

b.prior.mean <- c(0,0)
b.prior.sd <- c(0.5,0.5)

# Main Data Set of Three Cities (Restrict to two if necessary)
threecities <- vector(mode="list",length=3)
names(threecities) <-  paste0(c("LA","NY","P"),"_City")
data.files <- c(LA.file,NY.file,P.file)
mesh.files <- c(LA.mesh,NY.mesh,P.mesh)

for (i in 1:3){
  load(data.files[i])
  count.df <- gta_countdf # Want to fit to Motor Vehicle Theft
  # Change names of Covariates "pop"->"cov1" and "inc"->"cov2" and "gta"->"count"
  colnames(count.df)[colnames(count.df)=="gta"] <- "count"
  
  # Want to scale the variables, 10,000 population and 100,000 income and create "cov1" <- "pop" and "cov2" <- "inc"
  count.df$cov1 <- count.df$pop/1e4
  count.df$cov2 <- count.df$inc/1e5
  count.df$area <- as.numeric(count.df$area)/1e8
  
  threecities[[i]]$count <- count.df
  
  load(mesh.files[i])
  threecities[[i]]$mesh <- mesh
  
  threecities[[i]]$spde.prior <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[[i]][1],rho.star[[i]][2]),prior.sigma = c(sigma.star[[i]][1],sigma.star[[i]][2]))
}




# Function Inputs
param.names <- paste0("beta",1:2)
lower.bound.iteration <- 100 # beyond which, we can resample if INLA has errors
lower.bound.buff <- 50 # if INLA has errors, want to a have a sample size of around 50 to sample from, so need atleast 150 iterations otherwise the algorithm will output an error
tot.err.lim <- 10 # how many times will we accept errors before stopping the run
it.err.lim <- 2 # how many errors per iteration will we allow

total.it <- 10
prop.sd <- 0.5

total.cities <- 2


if (total.cities==3){
  count.list <- map(threecities,"count")
  mesh.list <- map(threecities,"mesh")
  spde.list <- map(threecities,"spde.prior")
  
  save.name <- "MultiCityTimings3Cities.rda"

  restart <- 0
  if (restart==1){
    load(save.name)
    comp.it <- sum(!is.na(out$run$theta$beta1))
  } else {
    comp.it <- 0
    set.seed(1*1e4+1)
  }
  
  start.time <- proc.time()
  mh.out.3 <- mh.inlawmcmc.multivar(data.list=count.list,spde.list=spde.list,mesh.list=mesh.list,its=total.it-comp.it,
                                    init=init.vals,prior.mean=beta.prior.means,prior.sd=beta.prior.sd,prop.sd=prop.sd,
                                    param.names=param.names,restart=restart,form="base",which.base=1,b.prior.mean=b.prior.mean,
                                    b.prior.sd=b.prior.sd,save.name=save.name,lb.iterr=lower.bound.iteration,
                                    lb.buff=lower.bound.buff,it.err.lim=it.err.lim,tot.err.lim=tot.err.lim)
  end.time <- proc.time()
  time.taken <- end.time-start.time
  save(time.taken,file=paste0("TimeTaken3Cities",total.it,"Iterations.rda"))
} else {
  twocities <- threecities[1:2]
  count.list <- map(twocities,"count")
  mesh.list <- map(twocities,"mesh")
  spde.list <- map(twocities,"spde.prior")
  
  save.name <- "MultiCityTimings2Cities.rda"
  
  restart <- 0
  if (restart==1){
    load(save.name)
    comp.it <- sum(!is.na(out$run$theta$beta1))
  } else {
    comp.it <- 0
    set.seed(5*1e4+1)
  }
  
  start.time <- proc.time()
  mh.out.2 <- mh.inlawmcmc.multivar(data.list=count.list,spde.list=spde.list,mesh.list=mesh.list,its=total.it-comp.it,
                                    init=init.vals,prior.mean=beta.prior.means,prior.sd=beta.prior.sd,prop.sd=prop.sd,
                                    param.names=param.names,restart=restart,form="base",which.base=1,b.prior.mean=b.prior.mean,
                                    b.prior.sd=b.prior.sd,save.name=save.name,lb.iterr=lower.bound.iteration,
                                    lb.buff=lower.bound.buff,it.err.lim=it.err.lim,tot.err.lim=tot.err.lim)
  end.time <- proc.time()
  time.taken <- end.time-start.time
  save(time.taken,file=paste0("TimeTaken2Cities",total.it,"Iterations.rda"))
}



