
# INLA with MCMC Plots ----------------------------------------------------

# This script contains the functions that will produce the summary tables and plots for the output of INLA with MCMC runs as well as functions that will plot the results of pure INLA output along with them.

# Author: Nadeen Khaleel


# Set Up and Libraries ----------------------------------------------------

library(INLA)
library(mcmcse)
library(ggplot2)
library(gridExtra)
library(scales)
library(bayesplot)

fill.blue <- "grey"
col.blue <- "black"
param.lab.df <- c("beta[0]","beta[1]","beta[2]","sigma","rho","b[1]","b[2]") 
param.lab.plot <- parse(text=param.lab.df)
param.lab.html <- c("$\\beta_0$","$\\beta_1$","$\\beta_2$","$\\sigma$","$\\rho$","$b_1$","$b_2$")



# Table Summary -----------------------------------------------------------

# Just INLA within MCMC, or with INLA outputs
tab.summ <- function(iwmbma.out,bma.param=NULL,bma.ind=NULL,param.mh=NULL,param.bma=NULL,multi=FALSE,windows.l=1,w.lab=NULL,cov=FALSE,cov.param=NULL,cov.ind=NULL,comb.ind=NULL,param.cov=NULL,true.vals=NULL,inla.out=NULL,mh.inla.param=NULL,mh.inla.ind=NULL,bma.inla.ind=NULL){
  # Inputs:
  # iwmbma.out - the output from the INLA with MCMC algorithm, mainly the FINAL output after the BMA has been computed, with the correct burn-in and thinning, so that summaries are not based on the full chains
  # bma.param - location of marginals, "fixed" or "hyperpar" for each BMA'd parameter, this should be the same for the INLA output (if it is included).
  # bma.ind - if the ordering of the BMA outputs need to be different, for example we may want sigma (Stdev) to be placed ahead of the rho (Range) and so will need to alter the order of the hyperparameters. Also the number of BMA outputs are different from those in the INLA summary output as some parameters are separate for the MH step.
  # multi - multivariate INLA with MCMC output, or univariate? Default FALSE (univariate).
  # windows.l - number of windows, deafult=1.
  # w.lab - labels for the windows, if NULL, will set as indices of number of windows.
  # cov - if multi=TRUE, did we model the other study regions accounting for different fixed effects? Default FALSE.
  # cov.param - location of extra parameters, similar to bma.param.
  # cov.ind - which element of the list belongs to the extra parameters.
  # param.* - parameter labels for the table, so can be printed as data frame and latex or html, if latex=TRUE, this must be a list, first in the list must be data frame labels, then html then latexparam.cov (if all 3 are TRUE)
  # true.vals - if there are true values to compare
  # inla.out - want to compare the results from a pure INLA model, if NULL, then no, otherwise need a list of INLA models with length according to the number of study regions.
  # mh.inla.param - where to find the inla results for the parameters from the MH step, "fixed" or "hyperpar" for each parameter.
  # mh.inla.ind - where to find parameters within mh.inla.param that correspond to the MH parameters
  # bma.inla.ind - where to find parameters within bma.param that correspond to the BMA parameters as indexed within the INLA output which could differ from the IwM BMA output.
  
  
  mh.df.lab <- param.mh
  bma.df.lab <- param.bma
  mh.l <- ncol(iwmbma.out$run$theta)
  bma.l <- length(bma.inla.ind)
  if (cov==TRUE){
    bcov.df.lab <- param.cov
    cov.l <- length(cov.ind)
  }
  
  if (is.null(w.lab)){
    w.lab <- 1:windows.l
  }
  
  # MCMC Results
  if (multi==FALSE){ # univariate
    mean.vec <- sapply(1:mh.l,function(i){mean(iwmbma.out$run$theta[,i])})
    sd.vec <- sapply(1:mh.l,function(i){sd(iwmbma.out$run$theta[,i])})
    ess.vec <- sapply(1:mh.l,function(i){ess(iwmbma.out$run$theta[,i])})
    mcmc.tab <- data.frame("Algorithm"=rep("INLAwMCMC-MH",mh.l),"Window"=rep(w.lab,mh.l),"Parameter"=mh.df.lab,"Mean"=mean.vec,"SD"=sd.vec,"ESS"=ess.vec)
    
    if (!is.null(inla.out)){
      # In case, inla.out is place as a list
      inla.out.mh <- inla.out[[1]]
      
      mean.inla.vec <- sapply(1:mh.l,function(i,fix.hyper,ind){inla.out.mh[paste0("summary.",fix.hyper[i])][[1]]$mean[ind[i]]},fix.hyper=mh.inla.param,ind=mh.inla.ind)
      sd.inla.vec <- sapply(1:mh.l,function(i,fix.hyper,ind){inla.out.mh[paste0("summary.",fix.hyper[i])][[1]]$sd[ind[i]]},fix.hyper=mh.inla.param,ind=mh.inla.ind)
      inla.tab <- data.frame("Algorithm"=rep("INLA",mh.l),"Window"=rep(w.lab,mh.l),"Parameter"=mh.df.lab,"Mean"=mean.inla.vec,"SD"=sd.inla.vec,"ESS"=rep(NA,mh.l))
      mcmc_inla.tab <- rbind(mcmc.tab,inla.tab)
      rownames(mcmc_inla.tab) <- NULL
      mh.tab <- mcmc_inla.tab
    } else {
      rownames(mcmc.tab) <- NULL
      mh.tab <- mcmc.tab
    }
    
  } else { # multivariate
    mean.vec <- sapply(1:mh.l,function(i){mean(iwmbma.out$run$theta[,i])})
    sd.vec <- sapply(1:mh.l,function(i){sd(iwmbma.out$run$theta[,i])})
    ess.vec <- sapply(1:mh.l,function(i){ess(iwmbma.out$run$theta[,i])})
    mcmc.tab <- data.frame("Algorithm"=rep("INLAwMCMC-MH",mh.l),"Window"=rep("Combined",mh.l),"Parameter"=mh.df.lab,"Mean"=mean.vec,"SD"=sd.vec,"ESS"=ess.vec)
    
    if (!is.null(inla.out)){
      mean.inla.vec <- function(inla.out.w,mh.inla.p,mh.inla.i){sapply(1:mh.l,function(i,fix.hyper,ind){inla.out.w[paste0("summary.",fix.hyper[i])][[1]]$mean[ind[i]]},fix.hyper=mh.inla.p,ind=mh.inla.i)}
      mean.inla.list <- lapply(inla.out,function(i,mh.inla.p,mh.inla.i){mean.inla.vec(i,mh.inla.p,mh.inla.i)},mh.inla.p=mh.inla.param,mh.inla.i=mh.inla.ind)
      sd.inla.vec <- function(inla.out.w,mh.inla.p,mh.inla.i){sapply(1:mh.l,function(i,fix.hyper,ind){inla.out.w[paste0("summary.",fix.hyper[i])][[1]]$sd[ind[i]]},fix.hyper=mh.inla.p,ind=mh.inla.i)}
      sd.inla.list <- lapply(inla.out,function(i,mh.inla.p,mh.inla.i){sd.inla.vec(i,mh.inla.p,mh.inla.i)},mh.inla.p=mh.inla.param,mh.inla.i=mh.inla.ind)
      inla.tab <- data.frame("Algorithm"=rep("INLA",mh.l),"Window"=rep(names(inla.out),each=mh.l),"Parameter"=rep(mh.df.lab,windows.l),"Mean"=unname(unlist(mean.inla.list)),"SD"=unname(unlist(sd.inla.list)),"ESS"=rep(NA,windows.l*mh.l))
      
      mcmc_inla.tab <- rbind(mcmc.tab,inla.tab)
      rownames(mcmc_inla.tab) <- NULL
      mh.tab <- mcmc_inla.tab
      
    } else {
      rownames(mcmc.tab) <- NULL
      mh.tab <- mcmc.tab
    }
    if (cov==TRUE){
      mean.bma.bcov.vec <- function(window.iwmbma,cov.p,cov.i){sapply(1:cov.l,function(i,w.bma,hyper.fix.cov,cov.ind){a <- inla.zmarginal(w.bma[paste0("marginals.",hyper.fix.cov[i])][[1]][cov.ind[i]][[1]],silent = TRUE); return(a$mean)},w.bma=window.iwmbma,hyper.fix.cov=cov.p,cov.ind=cov.i)}
      
      sd.bma.bcov.vec <- function(window.iwmbma,cov.p,cov.i){sapply(1:cov.l,function(i,w.bma,hyper.fix.cov,cov.ind){a <- inla.zmarginal(w.bma[paste0("marginals.",hyper.fix.cov[i])][[1]][cov.ind[i]][[1]],silent = TRUE); return(a$sd)},w.bma=window.iwmbma,hyper.fix.cov=cov.p,cov.ind=cov.i)}
      
      cov.bma.win <- iwmbma.out$inla.bma[2:windows.l]
      mean.bma.bcov.list <- lapply(cov.bma.win,function(ii,cov.p,cov.i){mean.bma.bcov.vec(window.iwmbma=ii,cov.p,cov.i)},cov.p=cov.param,cov.i=cov.ind)
      sd.bma.bcov.list <- lapply(cov.bma.win,function(ii,cov.p,cov.i){sd.bma.bcov.vec(window.iwmbma=ii,cov.p,cov.i)},cov.p=cov.param,cov.i=cov.ind)
      
      bma.bcov.tab <- data.frame("Algorithm"=rep("INLAwMCMC-BMA",(windows.l-1)*cov.l),"Window"=rep(w.lab[2:windows.l],each=cov.l),"Parameter"=rep(bcov.df.lab,(windows.l-1)),"Mean"=unname(unlist(mean.bma.bcov.list)),"SD"=unname(unlist(sd.bma.bcov.list)),"ESS"=c(NA,NA))
      
      mean.bma.combcov.vec <- function(window.iwmbma,cov.p,cov.i){sapply(1:cov.l,function(i,w.bma,hyper.fix.cov,cov.ind){a <- inla.zmarginal(w.bma[paste0("combined.marginals.",hyper.fix.cov[i])][[1]][i][[1]],silent = TRUE); return(a$mean)},w.bma=window.iwmbma,hyper.fix.cov=cov.p,cov.ind=cov.i)}
      
      sd.bma.combcov.vec <- function(window.iwmbma,cov.p,cov.i){sapply(1:cov.l,function(i,w.bma,hyper.fix.cov,cov.ind){a <- inla.zmarginal(w.bma[paste0("combined.marginals.",hyper.fix.cov[i])][[1]][i][[1]],silent = TRUE); return(a$sd)},w.bma=window.iwmbma,hyper.fix.cov=cov.param,cov.ind=cov.ind)}
      
      cov.bma.win <- iwmbma.out$inla.bma[2:windows.l]
      mean.bma.combcov.list <- lapply(cov.bma.win,function(ii,cov.p,cov.i){mean.bma.combcov.vec(window.iwmbma=ii,cov.p,cov.i)},cov.p=cov.param,cov.i=comb.ind)
      sd.bma.combcov.list <- lapply(cov.bma.win,function(ii,cov.p,cov.i){sd.bma.combcov.vec(window.iwmbma=ii,cov.p,cov.i)},cov.p=cov.param,cov.i=comb.ind)
      
      bma.combcov.tab <- data.frame("Algorithm"=rep("INLAwMCMC",(windows.l-1)*cov.l),"Window"=rep(w.lab[2:windows.l],each=cov.l),"Parameter"=rep(paste0(mh.df.lab,"+",bcov.df.lab),(windows.l-1)),"Mean"=unname(unlist(mean.bma.combcov.list)),"SD"=unname(unlist(sd.bma.combcov.list)),"ESS"=c(NA,NA))
      
      bma.cov.tab <- rbind(bma.bcov.tab,bma.combcov.tab)
      rownames(bma.cov.tab) <- NULL
      
      
      # Combine with main IwM MCMC Output table.
      
      mh.tab <- rbind(mh.tab,bma.cov.tab)
    }
    
  }
  
  # BMA Results
  if (multi==FALSE){ # univariate
    mean.bma.vec <- sapply(1:bma.l,function(i,fix.hyper,ind){a <- inla.zmarginal(iwmbma.out$inla.bma[paste0("marginals.",fix.hyper[i])][[1]][[ind[i]]],silent=TRUE); return(a$mean)},fix.hyper=bma.param,ind=bma.ind)
    sd.bma.vec <- sapply(1:bma.l,function(i,fix.hyper,ind){a <- inla.zmarginal(iwmbma.out$inla.bma[paste0("marginals.",fix.hyper[i])][[1]][[ind[i]]],silent=TRUE); return(a$sd)},fix.hyper=bma.param,ind=bma.ind)
    bma.tab <- data.frame("Algorithm"=rep("INLAwMCMC-BMA",bma.l),"Window"=rep(w.lab,bma.l),"Parameter"=bma.df.lab,"Mean"=mean.bma.vec,"SD"=sd.bma.vec)
    
    if (!is.null(inla.out)){
      
      # In case, inla.out is place as a list
      inla.out.bma <- inla.out[[1]]
      
      bma.inla.param <- bma.param
      
      mean.inlabma.vec <- sapply(1:bma.l,function(i,fix.hyper,ind){inla.out.bma[paste0("summary.",fix.hyper[i])][[1]]$mean[ind[i]]},fix.hyper=bma.inla.param,ind=bma.inla.ind)
      sd.inlabma.vec <- sapply(1:bma.l,function(i,fix.hyper,ind){inla.out.bma[paste0("summary.",fix.hyper[i])][[1]]$sd[ind[i]]},fix.hyper=bma.inla.param,ind=bma.inla.ind)
      
      inla.tab <- data.frame("Algorithm"=rep("INLA",bma.l),"Window"=rep(w.lab,bma.l),"Parameter"=bma.df.lab,"Mean"=mean.inlabma.vec,"SD"=sd.inlabma.vec)
      
      
      bma_inla.tab <- rbind(bma.tab,inla.tab)
      rownames(bma_inla.tab) <- NULL
      bma.tab <- bma_inla.tab
    } else {
      rownames(bma.tab) <- NULL
    }
    
  } else { # multivariate
    mean.bma.vec <- function(window.iwmbma,bma.inla.p,bma.inla.i){sapply(1:bma.l,function(i,iwm.bma.w,fix.hyper,ind){a <- inla.zmarginal(iwm.bma.w[paste0("marginals.",fix.hyper[i])][[1]][[ind[i]]],silent=TRUE); return(a$mean)},iwm.bma.w=window.iwmbma,fix.hyper=bma.inla.p,ind=bma.inla.i)}
    sd.bma.vec <- function(window.iwmbma,bma.inla.p,bma.inla.i){sapply(1:bma.l,function(i,iwm.bma.w,fix.hyper,ind){a <- inla.zmarginal(iwm.bma.w[paste0("marginals.",fix.hyper[i])][[1]][[ind[i]]],silent=TRUE); return(a$sd)},iwm.bma.w=window.iwmbma,fix.hyper=bma.inla.p,ind=bma.inla.i)}
    
    
    bma.win <- iwmbma.out$inla.bma
    mean.bma.list <- lapply(bma.win,function(i,bma.inla.p,bma.inla.i){mean.bma.vec(window.iwmbma=i,bma.inla.p,bma.inla.i)},bma.inla.p=bma.param,bma.inla.i=bma.ind)
    sd.bma.list <- lapply(bma.win,function(i,bma.inla.p,bma.inla.i){sd.bma.vec(window.iwmbma=i,bma.inla.p,bma.inla.i)},bma.inla.p=bma.param,bma.inla.i=bma.ind)
    
    bma.tab <- data.frame("Algorithm"=rep("INLAwMCMC-BMA",bma.l),"Window"=rep(w.lab,each=bma.l),"Parameter"=bma.df.lab,"Mean"=unname(unlist(mean.bma.list)),"SD"=unname(unlist(sd.bma.list)))
    
    if (!is.null(inla.out)){
      bma.inla.param <- bma.param
      
      mean.inlabma.vec <- function(inla.out.w,bma.inla.p,bma.inla.i){sapply(1:bma.l,function(i,inla.out.w,fix.hyper,ind){inla.out.w[paste0("summary.",fix.hyper[i])][[1]]$mean[ind[i]]},inla.out.w=inla.out.w,fix.hyper=bma.inla.p,ind=bma.inla.i)}
      sd.inlabma.vec <- function(inla.out.w,bma.inla.p,bma.inla.i){sapply(1:bma.l,function(i,inla.out.w,fix.hyper,ind){inla.out.w[paste0("summary.",fix.hyper[i])][[1]]$sd[ind[i]]},inla.out.w=inla.out.w,fix.hyper=bma.inla.p,ind=bma.inla.i)}
      
      mean.inlabma.list <- lapply(inla.out,function(i,bma.inla.p,bma.inla.i){mean.inlabma.vec(i,bma.inla.p,bma.inla.i)},bma.inla.p=bma.inla.param,bma.inla.i=bma.inla.ind)
      sd.inlabma.list <- lapply(inla.out,function(i,bma.inla.p,bma.inla.i){sd.inlabma.vec(i,bma.inla.p,bma.inla.i)},bma.inla.p=bma.inla.param,bma.inla.i=bma.inla.ind)
      
      inla.tab <- data.frame("Algorithm"=rep("INLA",bma.l),"Window"=rep(w.lab,each=bma.l),"Parameter"=bma.df.lab,"Mean"=unname(unlist(mean.inlabma.list)),"SD"=unname(unlist(sd.inlabma.list)))
      
      bma_inla.tab <- rbind(bma.tab,inla.tab)
      rownames(bma_inla.tab) <- NULL
      bma.tab <- bma_inla.tab
    } else {
      rownames(bma.tab) <- NULL
    }
    
  }
  
  return(list("mcmc.table"=mh.tab,"bma.table"=bma.tab))
}




# Plot Summary ------------------------------------------------------------

# Functions to produce the different plots of interest for the INLA with MCMC output, with different functions for adding INLA results.

# MCMC Chains Outputs
# Will have functions for the following:
# MCMC Parameter Chains
# Plot of the chains for all parameters against each other - scatter and density
# Log-likelihood and log-posterior
# Density Plots (with additional values like true values of parameters, posterior marginals from INLA outputs)

iwm_mcmc_chains <- function(iwmbma_mh=NULL,which.param=NULL,param.lab=NULL,group=FALSE,ncol.group=NULL,print=TRUE){
  # Inputs:
  # iwmbma_mh - the matrix of chains for the different parameters from the MH run, columns must be each parameter.
  # which.param - which parameter chain do we want a plot for? Default is NULL which means all parameters.
  # param.lab - vector of parameter labels for the plots, will be parsed if special labels needed, like sub/superscript or Greek letters.
  # group - arrange the plots together, through grid.arrange().
  # ncol.group - if group=TRUE, how many columns for grid.arrange()?
  # print - print the plots as well?
  
  if (is.null(which.param)){
    which.param <- 1:ncol(iwmbma_mh)
  }
  total.plots <- length(which.param)
  
  if (is.null(colnames(iwmbma_mh))&is.null(param.lab)){
    colnames(iwmbma_mh) <- paste0("parameter_",total.plots)
    which.plot.param <- colnames(iwmbma_mh)[which.param]
    parsed.which.plot.param <- parse(text=which.plot.param)
  } else if (is.null(colnames(iwmbma_mh))&!is.null(param.lab)){
    colnames(iwmbma_mh)[which.param] <- param.lab
    which.plot.param <- colnames(iwmbma_mh)[which.param]
    parsed.which.plot.param <- parse(text=which.plot.param)
  } else {
    which.plot.param <- colnames(iwmbma_mh)[which.param]
    parsed.which.plot.param <- parse(text=param.lab)
  }
  
  chain.plots <- vector(mode="list",length=total.plots)
  
  l.its <- sum(!is.na(iwmbma_mh[,1]))
  
  for (i in 1:total.plots){
    # param.ind <- which.param[i]
    chain.plots[[i]] <- ggplot(iwmbma_mh) + geom_line(aes_string(x=1:l.its,y=(which.plot.param[i]))) + ylab(parsed.which.plot.param[i]) + xlab("Iterations") + theme_bw() + theme(axis.title = element_text(size = 25),axis.text = element_text(size = 25))
    if (print==TRUE){
      print(chain.plots[[i]])
    }
  }
  names(chain.plots) <- which.plot.param
  
  if (group==TRUE){
    if (is.null(ncol.group)){
      ncol.group <- floor(sqrt(total.plots))
    }
    group.plots <- do.call("grid.arrange", c(chain.plots, ncol=ncol.group))
    return(list("all.plots"=chain.plots,"grid.plots"=group.plots))
  } else {
    return(list("all.plots"=chain.plots))
  }
}

iwm_mcmc_scatterdens <- function(iwmbma_mh=NULL,which.param=NULL,param.lab=NULL,print=TRUE){ # may not want to include group variable rn, ,group=FALSE,ncol.group=NULL
  # Inputs:
  # iwmbma_mh - the matrix of chains for the different parameters from the MH run, columns must be each parameter.
  # which.param - which parameter chain do we want a plot for? Default is NULL which means all parameters.
  # param.lab - vector of parameter labels for the plots, will be parsed if special labels needed, like sub/superscript or Greek letters.
  if (is.null(which.param)){
    which.param <- 1:ncol(iwmbma_mh)
  }
  # total.combos <- length(which.param)*(length(which.param)-1)/2
  combos.param.ind <- combn(which.param,m=2)
  total.combos <- ncol(combos.param.ind)
  lab.param.ind <- combn(1:length(which.param),m=2)
  
  if (is.null(colnames(iwmbma_mh))&is.null(param.lab)){
    colnames(iwmbma_mh) <- paste0("parameter_",total.plots)
    which.plot.param.r1 <- colnames(iwmbma_mh)[combos.param.ind[1,]]
    which.plot.param.r2 <- colnames(iwmbma_mh)[combos.param.ind[2,]]
    parsed.which.plot.param.r1 <- parse(text=which.plot.param.r1)
    parsed.which.plot.param.r2 <- parse(text=which.plot.param.r2)
    combo.names <- paste0(which.plot.param.r1,"_",which.plot.param.r2)
  } else if (is.null(colnames(iwmbma_mh))&!is.null(param.lab)){
    colnames(iwmbma_mh)[which.param] <- param.lab
    which.plot.param.r1 <- colnames(iwmbma_mh)[combos.param.ind[1,]]
    which.plot.param.r2 <- colnames(iwmbma_mh)[combos.param.ind[2,]]
    parsed.which.plot.param.r1 <- parse(text=which.plot.param.r1)
    parsed.which.plot.param.r2 <- parse(text=which.plot.param.r2)
    combo.names <- paste0(which.plot.param.r1,"_",which.plot.param.r2)
  } else {
    which.plot.param.r1 <- colnames(iwmbma_mh)[combos.param.ind[1,]]
    which.plot.param.r2 <- colnames(iwmbma_mh)[combos.param.ind[2,]]
    param.lab.r1 <- param.lab[lab.param.ind[1,]]
    param.lab.r2 <- param.lab[lab.param.ind[2,]]
    parsed.which.plot.param.r1 <- parse(text=param.lab.r1)
    parsed.which.plot.param.r2 <- parse(text=param.lab.r2)
    combo.names <- paste0(which.plot.param.r1,"_",which.plot.param.r2)
  }
  
  scatter.plots <- vector(mode="list",length=total.combos)
  density.plots <- vector(mode="list",length=total.combos)
  
  l.its <- sum(!is.na(iwmbma_mh[,1]))
  
  for (i in 1:total.combos){
    scatter.plots[[i]] <- ggplot(iwmbma_mh) + geom_point(aes(x=get(which.plot.param.r1[i]),y=get(which.plot.param.r2[i]))) + ylab(parsed.which.plot.param.r2[i]) + xlab(parsed.which.plot.param.r1[i]) + theme_bw() + theme(axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.title = element_text(size = 25),legend.text = element_text(size = 25))
    density.plots[[i]] <- ggplot(iwmbma_mh) + geom_density2d_filled(aes(x=get(which.plot.param.r1[i]),y=get(which.plot.param.r2[i]))) + ylab(parsed.which.plot.param.r2[i]) + xlab(parsed.which.plot.param.r1[i]) + theme_bw() + theme(axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.title = element_text(size = 25),legend.text = element_text(size = 25))
    if (print==TRUE){
      print(scatter.plots[[i]])
      print(density.plots[[i]])
    }
  }
  names(scatter.plots) <- combo.names
  names(density.plots) <- combo.names
  
  return(list("scatter"=scatter.plots,"density"=density.plots))
}



cred.df.iwm <- function(iwmbma.out=NULL,param.lab.mh=NULL,which.bma=NULL,bma.ind=NULL,param.lab.bma=NULL,int.q=c(0.95,0.99),multivar=FALSE,window.total=1,window.lab=NULL,which.w=NULL,cov=FALSE,which.cov=NULL,cov.ind=NULL,param.lab.cov=NULL,comb=FALSE,which.comb=NULL,comb.ind=NULL,param.lab.comb=NULL){
  # Create a data frame with all of the parameters of interest as well as their credible intervals at the user-specified level
  # Inputs:
  # iwmbma.out - Output from the univariate or multivariate IwM, including the MH chain and BMA.
  # param.lab.mh - vector of labels for the parameters of interest in the MH chain
  # which.bma - which type of parameter for each parameter of interest from the BMA, vector matching length of parameters we want from the BMA step and labelled with "fixed" or "hyperpar"
  # bma.ind - indices of the parameters for the BMA step to match the labels "fixed" and "hyperpar", e.g. for the range we would take "hyperpar" in which.bma and match with bma.ind 1, as it is the first parameter in the hyperpar section of the approximate posterior marginals in the BMA step
  # param.lab.bma - vector of labels for the BMA parameters of interest, e.g. "rho"
  # int.q - the levels for the quantiles, e.g. 0.95 would give an upper and lower limit for the 95% CI, while c(0.95,0.99) would give upper and lower limits for both the 95% and 99% CI.
  # multivar - is the output multivariate or not? default=FALSE (so univariate input)
  # window.total - how many windows iw multivar=TRUE
  # window.lab - labels for the windows, if multivar=TRUE and window.total=n, then this should be a vector of length n
  # which.w - which windows (if multivar=TRUE) are not the base study region and so have the additional parameters to consider.
  # cov - are there additional parameters for the which.w study regions? (e.g. if we had a base/non-base study region)
  # which.cov - which type of parameter for the additional parameters, e.g. for the shared covariate effects to calculate the contrast parameters, we would have "fixed" as the contrast parameters are fixed effects
  # cov.ind - which indices in the "fixed" BMA output are these additional variables
  # param.lab.cov - vector of labels for these additional parameters
  # comb - do we want to get the credible intervals for the combined the parameters, e.g. the contrast parameters with the base study region shared paramters, beta[1] + b[1] for example - if they are output in the BMA
  # which.comb - types of parameters, e.g. "fixed"
  # comb.ind - which indices in the combined.marginals.* list element do we want?
  # param.lab.comb - vector of labels for the combined parameters
  
  if (is.null(window.lab)){
    window.lab <- paste0("Window ",1:window.total)
  }
  
  # MCMC Output
  mcmc.int <- mcmc_intervals_data(iwmbma.out$run$theta,point_est = "mean",prob=int.q[1],prob_outer = int.q[2])
  ll <- 0.5*(1-int.q[2])
  l <- 0.5*(1-int.q[1])
  h <- int.q[1]+0.5*(1-int.q[1])
  hh <- int.q[2]+0.5*(1-int.q[2])
  quant.lev <- paste0(c(ll,l,h,hh),"quant")
  mcmc.int$parameter <- param.lab.mh
  colnames(mcmc.int)[colnames(mcmc.int)=="ll"] <- quant.lev[1]
  colnames(mcmc.int)[colnames(mcmc.int)=="l"] <- quant.lev[2]
  colnames(mcmc.int)[colnames(mcmc.int)=="h"] <- quant.lev[3]
  colnames(mcmc.int)[colnames(mcmc.int)=="hh"] <- quant.lev[4]
  colnames(mcmc.int)[colnames(mcmc.int)=="m"] <- "mean"
  mcmc.int$window <- rep("all",nrow(mcmc.int))
  
  cred.df.mcmc <- mcmc.int[,c("parameter","window","mean",quant.lev)]
  
  # BMA Output
  if (multivar==FALSE){
    par.fixed <- length(iwmbma.out$inla.bma$marginals.fixed)
    par.hyperpar <- length(iwmbma.out$inla.bma$marginals.hyperpar)
    par.total <- par.fixed + par.hyperpar
    
    marg.list <- vector(mode="list",length=par.total)
    
    for (i in 1:length(which.bma)){
      marg.list[[i]] <- iwmbma.out$inla.bma[paste0("marginals.",which.bma[i])][[1]][[bma.ind[i]]]
    }
    marg.exp <- sapply(marg.list,function(i){inla.emarginal(function(x) x,i)})
    ci.95 <- sapply(marg.list,function(i){inla.qmarginal(c(0.5*(1-int.q[1]),int.q[1]+0.5*(1-int.q[1])),i)})
    ci.99 <- sapply(marg.list,function(i){inla.qmarginal(c(0.5*(1-int.q[2]),int.q[2]+0.5*(1-int.q[2])),i)})
    
    cred.df.bma <- data.frame("parameter"=param.lab.bma,"window"=rep(window.lab[1],length(param.lab.bma)),"mean"=marg.exp,"ll"=ci.99[1,],"l"=ci.95[1,],"h"=ci.95[2,],"hh"=ci.99[2,])
    colnames(cred.df.bma)[colnames(cred.df.bma)=="ll"] <- quant.lev[1]
    colnames(cred.df.bma)[colnames(cred.df.bma)=="l"] <- quant.lev[2]
    colnames(cred.df.bma)[colnames(cred.df.bma)=="h"] <- quant.lev[3]
    colnames(cred.df.bma)[colnames(cred.df.bma)=="hh"] <- quant.lev[4]
    
    cred.df <- rbind(cred.df.mcmc,cred.df.bma)
  } else {
    param.lab.bma.multi <- sapply(1:length(iwmbma.out$inla.bma),function(i){paste0(window.lab[i]," ",param.lab.bma)})
    cred.df.bma.list <- vector(mode="list",length=length(iwmbma.out$inla.bma))
    for (k in 1:length(iwmbma.out$inla.bma)){
      if (k!=which.w){
        par.fixed <- length(iwmbma.out$inla.bma[[k]]$marginals.fixed)
        par.hyperpar <- length(iwmbma.out$inla.bma[[k]]$marginals.hyperpar)
      } else {
        par.fixed <- sum(which.bma=="fixed")
        par.hyperpar <- sum(which.bma=="hyperpar")
      }
      par.total <- par.fixed + par.hyperpar
      
      marg.list <- vector(mode="list",length=par.total)
      
      for (i in 1:length(which.bma)){
        marg.list[[i]] <- iwmbma.out$inla.bma[[k]][paste0("marginals.",which.bma[i])][[1]][[bma.ind[i]]]
      }
      marg.exp <- sapply(marg.list,function(i){inla.emarginal(function(x) x,i)})
      ci.95 <- sapply(marg.list,function(i){inla.qmarginal(c(0.5*(1-int.q[1]),int.q[1]+0.5*(1-int.q[1])),i)})
      ci.99 <- sapply(marg.list,function(i){inla.qmarginal(c(0.5*(1-int.q[2]),int.q[2]+0.5*(1-int.q[2])),i)})
      
      cred.df.bma <- data.frame("parameter"=param.lab.bma.multi[,k],"window"=rep(window.lab[k],length(param.lab.bma.multi[,k])),"mean"=marg.exp,"ll"=ci.99[1,],"l"=ci.95[1,],"h"=ci.95[2,],"hh"=ci.99[2,])
      colnames(cred.df.bma)[colnames(cred.df.bma)=="ll"] <- quant.lev[1]
      colnames(cred.df.bma)[colnames(cred.df.bma)=="l"] <- quant.lev[2]
      colnames(cred.df.bma)[colnames(cred.df.bma)=="h"] <- quant.lev[3]
      colnames(cred.df.bma)[colnames(cred.df.bma)=="hh"] <- quant.lev[4]
      
      cred.df.bma.list[[k]] <- cred.df.bma
    }
    cred.df <- rbind(cred.df.mcmc,cred.df.bma.list[[1]])
    for (k in 2:length(iwmbma.out$inla.bma)){
      cred.df <- rbind(cred.df,cred.df.bma.list[[k]])
    }
    # 
    if (cov==TRUE){
      cred.df.cov.list <- vector(mode="list",length=length(which.w))
      for (k in 1:length(which.w)){
        par.total <- length(which.cov)
        
        marg.list <- vector(mode="list",length=par.total)
        list.ind <- 1
        for (i in 1:par.total){
          marg.list[[list.ind]] <- iwmbma.out$inla.bma[[which.w[k]]][paste0("marginals.",which.cov[i])][[1]][[cov.ind[i]]]
          list.ind <- list.ind + 1
        }
        marg.exp <- sapply(marg.list,function(i){inla.emarginal(function(x) x,i)})
        ci.95 <- sapply(marg.list,function(i){inla.qmarginal(c(0.5*(1-int.q[1]),int.q[1]+0.5*(1-int.q[1])),i)})
        ci.99 <- sapply(marg.list,function(i){inla.qmarginal(c(0.5*(1-int.q[2]),int.q[2]+0.5*(1-int.q[2])),i)})
        
        cred.df.cov <- data.frame("parameter"=param.lab.cov,"window"=rep(window.lab[which.w[k]],length(param.lab.cov)),"mean"=marg.exp,"ll"=ci.99[1,],"l"=ci.95[1,],"h"=ci.95[2,],"hh"=ci.99[2,])
        colnames(cred.df.cov)[colnames(cred.df.cov)=="ll"] <- quant.lev[1]
        colnames(cred.df.cov)[colnames(cred.df.cov)=="l"] <- quant.lev[2]
        colnames(cred.df.cov)[colnames(cred.df.cov)=="h"] <- quant.lev[3]
        colnames(cred.df.cov)[colnames(cred.df.cov)=="hh"] <- quant.lev[4]
        
        cred.df.cov.list[[k]] <- cred.df.cov
      }
      for (k in 1:length(which.w)){
        cred.df <- rbind(cred.df,cred.df.cov.list[[k]])
      }
    }
    if (comb==TRUE){
      cred.df.comb.list <- vector(mode="list",length=length(which.w))
      for (k in 1:length(which.w)){
        par.total <- length(which.comb)
        
        marg.list <- vector(mode="list",length=par.total)
        list.ind <- 1
        for (i in 1:par.total){
          marg.list[[list.ind]] <- iwmbma.out$inla.bma[[which.w[k]]][paste0("combined.marginals.",which.comb[i])][[1]][[comb.ind[i]]]
          list.ind <- list.ind + 1
        }
        marg.exp <- sapply(marg.list,function(i){inla.emarginal(function(x) x,i)})
        ci.95 <- sapply(marg.list,function(i){inla.qmarginal(c(0.5*(1-int.q[1]),int.q[1]+0.5*(1-int.q[1])),i)})
        ci.99 <- sapply(marg.list,function(i){inla.qmarginal(c(0.5*(1-int.q[2]),int.q[2]+0.5*(1-int.q[2])),i)})
        
        cred.df.comb <- data.frame("parameter"=param.lab.comb,"window"=rep(window.lab[which.w[k]],length(param.lab.comb)),"mean"=marg.exp,"ll"=ci.99[1,],"l"=ci.95[1,],"h"=ci.95[2,],"hh"=ci.99[2,])
        colnames(cred.df.comb)[colnames(cred.df.comb)=="ll"] <- quant.lev[1]
        colnames(cred.df.comb)[colnames(cred.df.comb)=="l"] <- quant.lev[2]
        colnames(cred.df.comb)[colnames(cred.df.comb)=="h"] <- quant.lev[3]
        colnames(cred.df.comb)[colnames(cred.df.comb)=="hh"] <- quant.lev[4]
        
        cred.df.comb.list[[k]] <- cred.df.comb
      }
      for (k in 1:length(which.w)){
        cred.df <- rbind(cred.df,cred.df.comb.list[[k]])
      }
    }
  }
  return("cred.df"=cred.df)
}


iwm_mcmc_cred <- function(cred.df,quant.lev,title){
  # Plot the credible intervals of interest
  # Inputs
  # cred.df - the credible interval data frame as output from cred.df.iwm
  # quant.lev - quantile levels, in particular the limits, e.g. 0.005 and 0.995 for the 99%, should range from the lowest quantile to the highest.
  # title - title for the plot
  
  parse.param <- str_replace_all(cred.df$parameter," ","~")
  cred.df$Parameter <- parse.param
  cred.int <- ggplot(cred.df,aes(mean,Parameter)) + geom_linerange(aes_string(xmin = quant.lev[1], xmax = quant.lev[4]),colour="#6497b1") + geom_linerange(aes_string(xmin = quant.lev[2], xmax = quant.lev[3]),size=2,colour="#005b96") + geom_point(col="#6497b1",size=2.5) + geom_vline(xintercept=0,col="red",linetype="dashed") + scale_y_discrete(labels=scales::parse_format()) + xlab("Posterior Quantiles") + ylab("Parameter") + ggtitle(title) + theme(plot.title = element_text(size = 32.5,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25))
}

# Posterior Density: Multivarite+Univariate MH plots

iwm_mcmc_post <- function(iwmbma_mh=NULL,post.lab="IwM MH",which.param=NULL,param.lab=NULL,hist.bins=30,prior_fun=NULL,true.value=NULL,true.lab=NULL,inla.out=NULL,inla.param=NULL,inla.param.ind=NULL,inla.lab,iwmbma_bma=NULL,which.window=NULL,which.param_combo=NULL,combo.param=NULL,combo.param.ind=NULL,combo.param.lab=NULL,col.vec=NULL,print=TRUE){ # 
  # Plots the posterior densities for the MH step outputs from an INLA within MCMC run, univariate or multivariate with the possibility of toverlaying the priors and INLA outputs and even any true values of the parameters if they are known.
  # Inputs
  # iwmbma_mh - the matrix of chains for the different parameters from the MH run, columns must be each parameter.
  # post.lab - label for the histogram in the legend
  # which.param - which parameters in the MH matrix are we plotting
  # param.lab - vector of labels for the parameters
  # hist.bins - how many bins in the histogram of the chains
  # prior_fun - function for the priors to overlay, this should be a list as long as the number of parameters we want to plot
  # true.value - true values we also want to plot? should be a list as long as the number of parameters, if there are multiple true values for a single parameter, these are a vector in the list.
  # true.lab - labels for the true values, can just be of length 1, "True Value" if there is only one true value per parameters, otherwise needs to be a vector to match the length of the longest element of true.value.
  # inla.out - INLA model outputs, of type "inla", list as long as the number of parameters we want to plot
  # inla.param - which type of parameters, "fixed" or "hyperpar", should be a vector as long as the number of parameters
  # inla.param.ind - which index for the parameters for each "fixed" or "hyperpar"
  # inla.lab - vector of labels for INLA marginals to place in the legend
  # iwmbma_bma - any BMA output, for example combined parameters that we want to overlay on the MH sample histograms
  # which.window - which window (if we have non-NULL iwmbma_bma included, as we want to overlay additional marginals)
  # which.param_combo - index for the parameters we are plotting, particularly which of the plotted parameters do we also have combo parameters for overlay
  # combo.param - vector of type of parameters for each combination, e.g. beta[1] + b[1] would be "fixed"
  # combo.param.ind - index of the combo parameters in the "combo.param" sections highlighted
  # combo.param.lab - labels for the combo parameters, if included. this will be placed in the legend
  # col.vec - vector of colours, one for the prior, one for each true value for a single parameter
  # print - print the plots as the function runs?
  
  
  if (is.null(which.param)){
    which.param <- 1:ncol(iwmbma_mh)
  }
  total.plots <- length(which.param)
  
  if (is.null(colnames(iwmbma_mh))&is.null(param.lab)){
    colnames(iwmbma_mh) <- paste0("parameter_",total.plots)
    which.plot.param <- colnames(iwmbma_mh)[which.param]
    parsed.which.plot.param <- parse(text=which.plot.param)
  } else if (is.null(colnames(iwmbma_mh))&!is.null(param.lab)){
    colnames(iwmbma_mh)[which.param] <- param.lab
    which.plot.param <- colnames(iwmbma_mh)[which.param]
    parsed.which.plot.param <- parse(text=which.plot.param)
  } else {
    which.plot.param <- colnames(iwmbma_mh)[which.param]
    parsed.which.plot.param <- parse(text=param.lab)
  }
  
  post.lab.parse <- str_replace_all(post.lab," ","~")
  
  if (!is.null(true.value)){
    tv.sum <- max(lengths(true.value))
    tv.lab.parse <- str_replace_all(true.lab," ","~")
  } else {
    tv.sum <- 0
  }
  
  postdens.plots <- vector(mode="list",length=total.plots)
  if (!is.null(inla.out)){
    inla.lab.parse <- str_replace_all(inla.lab," ","~")
    if (length(inla.out)==1){
      col.check <- c(0,0,rep(0,tv.sum),0)
    } else {
      col.check <- rep(0,(2+tv.sum+length(inla.out)))
    }
  } else {
    col.check <- c(0,0,rep(0,tv.sum))
  }
  if (!is.null(iwmbma_bma)){
    combo.param.lab.parse <- str_replace_all(combo.param.lab," ","~")
    col.check <- c(col.check,rep(0,length(which.param_combo)))
    postdens.plots_cov <- vector(mode="list",length=total.plots)
  }
  
  if (is.null(col.vec)){
    col.vec <- c(fill.blue,"magenta")
    if (length(col.check) >= 3){
      rcols <- RColorBrewer::brewer.pal(n=(length(col.check)-2),name="Dark2")
      col.vec <- c(col.vec,rcols)
    }
  }
  
  if (!is.null(iwmbma_bma)&!is.null(inla.out)&!is.null(true.value)){
    col.vec <- col.vec[1:length(col.check)]
    postdens.plots_cov <- vector(mode="list",length=length(which.param_combo))
    lev <- c(post.lab.parse,"Prior",inla.lab.parse,combo.param.lab.parse,tv.lab.parse)
    names(col.vec) <- lev
    names(col.vec) <- levels(factor(lev,levels=lev))
    col.vec <- col.vec[1:length(col.check)]
  } else if (!is.null(iwmbma_bma)&is.null(inla.out)&!is.null(true.value)) {
    col.vec <- col.vec[1:length(col.check)]
    lev <- c(post.lab.parse,"Prior",combo.param.lab.parse,tv.lab.parse)
    names(col.vec) <- lev
    names(col.vec) <- levels(factor(lev,levels=lev))
  }  else if (!is.null(iwmbma_bma)&!is.null(inla.out)&is.null(true.value)) {
    col.vec <- col.vec[1:length(col.check)]
    lev <- c(post.lab.parse,"Prior",inla.lab.parse,combo.param.lab.parse)
    names(col.vec)[col.vec==col.vec[1]] <- post.lab.parse
    names(col.vec)[col.vec==col.vec[2]] <- "Prior"
    names(col.vec)[is.na(names(col.vec))] <- c(inla.lab.parse,combo.param.lab.parse)
    names(col.vec) <- levels(factor(lev,levels=lev))
  }else if (is.null(iwmbma_bma)&!is.null(inla.out)&!is.null(true.value)) {
    col.vec <- col.vec[1:length(col.check)]
    lev <- c(post.lab.parse,"Prior",inla.lab.parse,tv.lab.parse)
    names(col.vec) <- lev
    names(col.vec) <- levels(factor(lev,levels=lev))
    col.vec <- col.vec[1:length(col.check)]
  } else if (!is.null(iwmbma_bma)&is.null(inla.out)&is.null(true.value)) {
    col.vec <- col.vec[1:length(col.check)]
    lev <- c(post.lab.parse,"Prior",combo.param.lab.parse)
    names(col.vec) <- lev
    names(col.vec) <- levels(factor(lev,levels=lev))
  } else if (is.null(iwmbma_bma)&!is.null(inla.out)&is.null(true.value)) {
    col.vec <- col.vec[1:length(col.check)]
    lev <- c(post.lab.parse,"Prior",inla.lab.parse)
    names(col.vec) <- lev
    names(col.vec) <- levels(factor(lev,levels=lev))
    col.vec <- col.vec[1:length(col.check)]
  } else if (is.null(iwmbma_bma)&is.null(inla.out)&!is.null(true.value)) {
    col.vec <- col.vec[1:length(col.check)]
    lev <- c(post.lab.parse,"Prior",tv.lab.parse)
    names(col.vec) <- lev
    names(col.vec) <- levels(factor(lev,levels=lev))
    col.vec <- col.vec[1:length(col.check)]
  } else {
    col.vec <- col.vec[1:length(col.check)]
    lev <- c(post.lab.parse,"Prior")
    names(col.vec) <- lev
    names(col.vec) <- levels(factor(lev,levels=lev))
    col.vec <- col.vec[1:length(col.check)]
  }
  
  
  
  base <- vector(mode = "list",length=length(which.param))
  
  for (i in 1:total.plots){
    grey.line <- data.frame(yint=-5,"lab"=post.lab.parse)
    base[[i]] <- ggplot(iwmbma_mh) + geom_histogram(aes_string(x=which.plot.param[i],y = "..density.."),bins=hist.bins,fill=fill.blue,col=col.blue) + geom_hline(data=grey.line,aes(yintercept=yint,col=lab)) + ylim(c(0,NA)) + xlab(parsed.which.plot.param[i]) + ggtitle(parse(text=paste0("Posterior~Distribution~'for'~",as.character(parsed.which.plot.param[i])))) + theme_bw() + theme(plot.title = element_text(size = 30,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.title = element_text(size = 25),legend.text = element_text(size = 25),legend.text.align = 0,legend.title.align = 0.5)
    col.check[1] <- 1
    
    if (!is.null(prior_fun)){
      col.check[2] <- 1
      p.fun <- prior_fun[[i]]
      base[[i]] <- base[[i]] + stat_function(fun=p.fun,geom="line",aes(colour="Prior"),size=1.25)
      
    }
    if (!is.null(true.value)){
      if (lengths(true.value)[i]==1){
        col.check[3+length(inla.out)+length(which.param_combo)] <- 1
        tv.df <- data.frame(xint=true.value[[i]],"lab"=tv.lab.parse[1])
        base[[i]] <- base[[i]] + geom_vline(data=tv.df,aes(xintercept=xint,col=lab),size=1.25,show.legend=FALSE)
      } else {
        col.check[(3+length(inla.out)+length(which.param_combo)):(2+length(inla.out)+length(which.param_combo)+lengths(true.value)[i])] <- rep(1,lengths(true.value)[i])
        for (j in 1:lengths(true.value)[i]){
          tv.df <- data.frame(xint=true.value[[i]],"lab"=tv.lab.parse)
          base[[i]] <- base[[i]] + geom_vline(data=tv.df,aes(xintercept=xint,col=lab),size=1.25,show.legend=FALSE)
        }
      }
    }
    if (!is.null(inla.out)){
      if (length(inla.out)==1){ # unviar
        col.check[3] <- 1
        marg.df <- data.frame(inla.out[[1]][paste0("marginals.",inla.param[i])][[1]][[inla.param.ind[i]]])
        inla.lab.parse <- str_replace_all(inla.lab," ","~")
        marg.df$inla.lab <- rep(inla.lab.parse[1],nrow(marg.df))
        
        base[[i]] <- base[[i]] + geom_line(data=marg.df,aes(x=x,y=y,col=inla.lab),size=1.25)
      } else { # multivar
        marg.df <- data.frame(inla.out[[1]][paste0("marginals.",inla.param[i])][[1]][[inla.param.ind[i]]])
        inla.lab.parse <- str_replace_all(inla.lab," ","~")
        marg.df$inla.lab <- rep(inla.lab.parse[1],nrow(marg.df))
        for (j in 1:length(inla.out)){
          col.check[2+j] <- 1
          marg.df.sub <- data.frame(inla.out[[j]][paste0("marginals.",inla.param[i])][[1]][[inla.param.ind[i]]])
          marg.df.sub$inla.lab <- rep(inla.lab.parse[j],nrow(marg.df.sub))
          marg.df <- rbind(marg.df,marg.df.sub)
          
        }
        base[[i]] <- base[[i]] + geom_line(data=marg.df,aes(x=x,y=y,col=inla.lab),linetype="dotdash",size=1.25)
      }
      
      
      postdens.plots[[i]] <- base[[i]] + scale_colour_manual(name="Distribution",values=col.vec[as.logical(col.check)],labels = parse_format())
    } else {
      postdens.plots[[i]] <- base[[i]] + scale_colour_manual(name="Distribution",values=col.vec[as.logical(col.check)],labels = parse_format())
    }
    if (print==TRUE){
      print(postdens.plots[[i]])
    }
  }
  if (!is.null(iwmbma_bma)){
    combo.param.lab.parse <- str_replace_all(combo.param.lab," ","~")
    for (i in 1:length(which.param_combo)){
      col.check[(2+length(inla.out)+i)] <- 1
      col.check.sub.cov <- col.check
      if (!is.null(true.value)){
        true.val.vec.ind <- (3+length(inla.out)+length(which.param_combo)):(2+length(inla.out)+length(which.param_combo)+max(lengths(true.value)))
        curr.true.val.vec.ind <- (3+length(inla.out)+length(which.param_combo)):(2+length(inla.out)+length(which.param_combo)+lengths(true.value)[i])
        which.inc <- curr.true.val.vec.ind==true.val.vec.ind
        if (sum(which.inc)<length(true.val.vec.ind)){
          col.check.sub.cov[true.val.vec.ind[!which.inc]] <- 0
        }
      }
      if (i > 1){
        col.check.sub.cov[(2+length(inla.out)+1):(2+length(inla.out)+i-1)] <- 0
      }
      j <- which.param_combo[i]
      combo.marg <- data.frame(iwmbma_bma[[which.window]][paste0("combined.marginals.",combo.param[i])][[1]][[combo.param.ind[i]]])
      combo.marg$lab <- rep(combo.param.lab.parse[i],nrow(combo.marg))
      postdens.plots_cov[[i]] <- base[[j]] + geom_line(data=combo.marg,aes(x=xx,y=auxbma,col=lab),linetype="dotted",size=1.25) + scale_colour_manual(name="Distribution",values=col.vec[as.logical(col.check.sub.cov)],labels = parse_format())
      if (print==TRUE){
        print(postdens.plots_cov[[i]])
      }
    }
  }
  if (!is.null(iwmbma_bma)){
    return(list("postdens"=postdens.plots,"postdens.cov"=postdens.plots_cov))
  } else {
    return(list("postdens"=postdens.plots))
  }
}



# Posterior Density: BMA plots
iwm_bma_post <- function(iwmbma_bma=NULL,post.lab="IwM BMA",which.param=NULL,param.ind=NULL,param.lab=NULL,prior_fun=NULL,true.value=NULL,true.lab=NULL,inla.out=NULL,inla.param=NULL,inla.param.ind=NULL,inla.lab=NULL,col.vec=NULL,print=TRUE){ # 
  # This function plots the IwM output from the BMA step and can overlay true values, priors and INLA-only approximated posterior marginals.
  # Altered after initial run to use aes_ instead of aes for geom_vline() for the true values, because otherwise the final true value is printed for ALL plots instead, which was the case for the Regular Polygon Simulation Study. Therefore, I will re-run the BMA plots for these results, but not for the City data, as these were untouched as they do not have true values included.
  # Inputs
  # iwmbma_bma - BMA output for a single window from the IwM BMA step output
  # post.lab - label for the IwM posterior marginals, will be used in the legend
  # which.param - vector of types of parameters we are plotting, either "fixed" or "hyperpar"
  # param.ind - index for the which.param sub-section of the BMA output, e.g. for the range we would take "hyperpar" in which.param and match with param.ind 1, as it is the first parameter in the hyperpar section of the approximate posterior marginals in the BMA step
  # param.lab - vector of labels for the parameters being plotted, for the title
  # prior_fun - function for the priors to overlay, this should be a list as long as the number of parameters we want to plot
  # true.value - list with length of number of parameters we are plotting, each element should be of length one, as we would expect a single true value for these parameters 
  # true.lab - label for the true values, in legend, plotted on our BMA marginals, can just be of length one and will produce the same label for ALL parameter plots in legend
  # inla.out - INLA output, of type "inla" in order to overlay INLA-only approximated posterior marginals
  # inla.param - vector of types of parameters from the INLA run that we want to plot, e.g. "fixed" or "hyperpar"
  # inla.param.ind - index for the parameter posterior marginals within inla.param, e.g. if we want rho then inla.param is "hyperpar" and inla.param.ind is 1. Should be the same length of inla.param
  # inla.lab - label for the INLA marginal, to be placed in the legend
  # col.vec - vector of colours for the different parts of the plot, e.g. BMA marginal, prior, INLA marginal, true value
  # print - print plots as the function runs?
  
  total.plots <- length(which.param)
  
  if (!is.null(param.lab)){
    parsed.which.plot.param <- parse(text=param.lab)
  } else {
    which.plot.param <- rep(NA,length(which.param))
    for (i in 1:length(which.param)){
      which.plot.param[i] <- names(iwmbma_bma[paste0("marginals.",which.param[i])][[1]])[param.ind[i]]
    }
    parsed.which.plot.param <- parse(text=which.plot.param)
  }
  
  post.lab.parse <- str_replace_all(post.lab," ","~")
  
  if (!is.null(true.value)){
    tv.sum <- max(lengths(true.value))
    tv.lab.parse <- str_replace_all(true.lab," ","~")
  } else {
    tv.sum <- 0
  }
  
  postdens.plots <- vector(mode="list",length=total.plots)
  if (!is.null(inla.out)){
    inla.lab.parse <- str_replace_all(inla.lab," ","~")
    if (length(inla.out)==1){
      col.check <- c(1,0,rep(0,tv.sum),0)
    } else {
      col.check <- rep(1,0,(1+tv.sum+length(inla.out)))
    }
  } else {
    col.check <- c(1,0,rep(0,tv.sum))
  }
  
  if (is.null(col.vec)){
    create.col <- 1
    col.vec <- c(col.blue,"magenta")
    if (length(col.check) > 2){
      rcols <- RColorBrewer::brewer.pal(n=(length(col.check)-2),name="Dark2")
      col.vec <- c(col.vec,rcols)
    }
  } else {
    create.col <- 0
  }
  
  if (!is.null(inla.out)&!is.null(true.value)){
    col.vec <- col.vec[1:length(col.check)]
    inla.lab.parse <- str_replace_all(inla.lab," ","~")
    lev <- c(post.lab.parse,"Prior",inla.lab.parse,tv.lab.parse)
    if (create.col==1){
      names(col.vec)[col.vec==col.vec[1]] <- post.lab.parse
      names(col.vec)[col.vec==col.vec[2]] <- "Prior"
      names(col.vec)[is.na(names(col.vec))] <- c(inla.lab.parse,tv.lab.parse)
    } else {
      names(col.vec) <- c(post.lab.parse,"Prior",inla.lab.parse,tv.lab.parse)
    }
    names(col.vec) <- levels(factor(lev,levels=lev))
    col.vec <- col.vec[1:length(col.check)]
  } else if (is.null(inla.out)&!is.null(true.value)){
    col.vec <- col.vec[1:length(col.check)]
    lev <- c(post.lab.parse,"Prior",tv.lab.parse)
    if (create.col==1){
      names(col.vec)[col.vec==col.vec[1]] <- post.lab.parse
      names(col.vec)[col.vec==col.vec[2]] <- "Prior"
      names(col.vec) <- c(tv.lab.parse)
    } else {
      names(col.vec) <- c(post.lab.parse,"Prior",tv.lab.parse)
    }
    names(col.vec) <- levels(factor(lev,levels=lev))
    col.vec <- col.vec[1:length(col.check)]
  } else if (!is.null(inla.out)&is.null(true.value)){
    col.vec <- col.vec[1:length(col.check)]
    inla.lab.parse <- str_replace_all(inla.lab," ","~")
    lev <- c(post.lab.parse,"Prior",inla.lab.parse)
    if (create.col==1){
      names(col.vec)[col.vec==col.vec[1]] <- post.lab.parse
      names(col.vec)[col.vec==col.vec[2]] <- "Prior"
      names(col.vec) <- c(inla.lab.parse)
    } else {
      names(col.vec) <- c(post.lab.parse,"Prior",inla.lab.parse)
    }
    names(col.vec) <- levels(factor(lev,levels=lev))
    col.vec <- col.vec[1:length(col.check)]
  } else {
    col.vec <- col.vec[1:length(col.check)]
    lev <- c(post.lab.parse,"Prior")
    names(col.vec) <- c(post.lab.parse,"Prior")
    names(col.vec) <- levels(factor(lev,levels=lev))
    col.vec <- col.vec[1:length(col.check)]
  }
  
  
  base <- vector(mode = "list",length=length(which.param))
  
  for (i in 1:total.plots){
    
    base.marg <- data.frame(iwmbma_bma[paste0("marginals.",which.param[i])][[1]][[param.ind[i]]])
    base[[i]] <- ggplot(base.marg) + geom_line(aes(x=xx,y = auxbma,col=post.lab.parse),size=1.25) + xlab(parsed.which.plot.param[i]) + ylab("Density") + ggtitle(parse(text=paste0("Posterior~Distribution~'for'~",as.character(parsed.which.plot.param[i])))) + theme_bw() + theme(plot.title = element_text(size = 30,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.title = element_text(size = 25),legend.text = element_text(size = 25),legend.text.align = 0,legend.title.align = 0.5)
    
    if (!is.null(prior_fun)){
      col.check[2] <- 1
      p.fun <- prior_fun[[i]]
      base[[i]] <- base[[i]] + stat_function(fun=p.fun,geom="line",aes(colour="Prior"),size=1.25)
      
    }
    if (!is.null(true.value)){
      if (lengths(true.value)[i]==1){
        col.check[2+length(inla.out)+1] <- 1
        base[[i]] <- base[[i]] + geom_vline(aes_(xintercept=true.value[i][[1]],col=tv.lab.parse[1]),size=1.25,show.legend=FALSE) # OLD VERSION (USED FOR US CITY PLOTS): geom_vline(aes_(xintercept=true.value[i][[1]],col=tv.lab.parse[1]),size=1.25,show.legend=FALSE)
      } else {
        col.check[(2+length(inla.out)):(2+length(inla.out)+lengths(true.value)[i])] <- rep(1,lengths(true.value)[i])
        for (j in 1:lengths(true.value)[i]){
          tv.df <- data.frame(xint=true.value[[i]],"lab"=tv.lab.parse)
          base[[i]] <- base[[i]] + geom_vline(data=tv.df,aes_(xintercept=xint,col=lab),size=1.25,show.legend=FALSE) # OLD VERSION (USED FOR US CITY PLOTS): geom_vline(data=tv.df,aes_(xintercept=xint,col=lab),size=1.25,show.legend=FALSE)
        }
      }
    }
    if (!is.null(inla.out)){
      col.check[2+length(inla.out)] <- 1
      marg.df <- data.frame(inla.out[[1]][paste0("marginals.",inla.param[i])][[1]][[inla.param.ind[i]]])
      inla.lab.parse <- str_replace_all(inla.lab," ","~")
      marg.df$inla.lab <- rep(inla.lab.parse[1],nrow(marg.df))
      
      
      base[[i]] <- base[[i]] + geom_line(data=marg.df,aes(x=x,y=y,col=inla.lab),linetype="dotdash",size=1.25)
      
      postdens.plots[[i]] <- base[[i]] + scale_colour_manual(name="Distribution",values=col.vec[as.logical(col.check)],labels = parse_format())
    } else {
      postdens.plots[[i]] <- base[[i]] + scale_colour_manual(name="Distribution",values=col.vec[as.logical(col.check)],labels = parse_format())
    }
    if (print==TRUE){
      print(postdens.plots[[i]])
    }
  }
  return(list("postdens"=postdens.plots))
}
