
# IwMSummary --------------------------------------------------------------

# This R script contains the calls to the IwMSummaryFunctions.R script to produce plots for the outputs from the INLA with MCMC implementations from Balena, both the Regular Polygon Simulation Study results and the US Cities results.

# NOTE - Re-plot the BMA plots for the Regular Polygon Simulation Study as these plots had issues, this involved a slight change to the iwm_bma_post() function for the true values geom_vline() where aes_ was used instead of aes, however as this should have only affected the simulation study plots, as there were no true values to plot for the US Cities, these were the only plots re-run.

# Author: Nadeen Khaleel

# Set-Up ------------------------------------------------------------------

# Either setwd() to the source file location, or run the following:
library("rstudioapi")
setwd(dirname(getActiveDocumentContext()$path))
# Then as we want to store the results in the IwM_OUTPUTS directory, also run
setwd("./IwM_OUTPUTS/")
source("../IwMSummary_Functions.R")

library(tidyverse)
library(ggplot2)
library(xtable)
library(INLA)
library(ggpubr)

# Regular Polygon Simulation Study ----------------------------------------

setwd("./REGPOLSIMSTUDY")

load("../../../IwMSIMSTUDY/Window1FullINLA.rda")
load("../../../IwMSIMSTUDY/Window2FullINLA.rda")
load("../../../IwMSIMSTUDY/Window2SparseINLA.rda")
load("../../../IwMSIMSTUDY/Window2CovINLA.rda")
load("../../../IwMSIMSTUDY/Window2DSCovINLA.rda")

load("../../../IwMSIMSTUDY/IwMMultivarMHFullbyFull.rda")
fullmh.out <- out

load("../../../IwMSIMSTUDY/IwMMultivarBMAFullbyFull.rda")
fullbma.out <- out

load("../../../IwMSIMSTUDY/IwMMultivarBMAFullbySparse.rda")
sparsebma.out <- out

load("../../../IwMSIMSTUDY/IwMMultivarMHFullbySparse.rda")
sparsemh.out <- out

load("../../../IwMSIMSTUDY/IwMMultivarBMAFullbyCovComb.rda")
covbma.out <- out

load("../../../IwMSIMSTUDY/IwMMultivarMHFullbyCov.rda")
covmh.out <- out

load("../../../IwMSIMSTUDY/IwMMultivarBMAFullbyDifferentSignCovComb.rda")
dscovbma.out <- out

load("../../../IwMSIMSTUDY/IwMMultivarMHFullbyDifferentSignCov.rda")
dscovmh.out <- out


pc_denrho <- function(rho,alpha_rho,rho_0){
  lambda_1 <- -log(alpha_rho)*rho_0;
  pi = lambda_1*rho^(-2)*exp(-lambda_1*rho^(-1))
}

pc_densig <- function(sigma,alpha_sigma,sigma_0){
  lambda_2 <- -log(alpha_sigma)/sigma_0
  pi = lambda_2*exp(-lambda_2*sigma)
}

gauss_prior_inla <- function(x,mean,sd){
  pi = dnorm(x,mean=mean,sd=sd)
}

beta_prior <- function(x){
  pi = gauss_prior_inla(x,mean=0,sd=sqrt(1/0.01))
}

b_prior <- function(x){
  pi = gauss_prior_inla(x,mean=0,sd=sqrt(4))
}

rho.star <- c(0.4,0.1); sigma.star <- c(4,0.1)
pc_den_rho <- function(x){pc_denrho(x,rho_0=rho.star[1],alpha_rho=rho.star[2])}
pc_den_sig <- function(x){pc_densig(x,sigma_0=sigma.star[1],alpha_sigma=sigma.star[2])}

col.vec.mcmc.noncov <- c("grey","magenta","red","blue","purple") # hist post, prior, window 1, window 2, 1 true value
col.vec.mcmc.cov <- c("grey","magenta","red","blue","darkgreen","darkorange3","purple","violetred") #  hist post, prior, window 1, window 2, window 1 par 1 combo, window 2 par 2 combo, 2 true values
col.vec.bma.w1 <- c("grey","magenta","red","purple") # post, prior, INLA w1 - match mcmc INLA, 1 true value
col.vec.bma.w2 <- c("grey","magenta","blue","purple") # post, prior, INLA w2 - match mcmc INLA, 1 true value

# Full v Full
full.multivar.tab <- tab.summ(iwmbma.out=fullbma.out,bma.param=c("fixed","hyperpar","hyperpar"),bma.ind=c(1,2,1),param.mh=param.lab.df[c(2,3)],param.bma=param.lab.df[c(1,4,5)],multi=TRUE,windows.l=2,w.lab=c("Window 1","Window 2"),cov=FALSE,inla.out=list("Window 1"=full1.fit,"Window 2"=full2.fit),mh.inla.param=c("fixed","fixed"),mh.inla.ind=c(2,3),bma.inla.ind=c(1,2,1))
save(full.multivar.tab,file="RegularPolygonSimStudyFullFullTableDF.rda")

full.table.df <- rbind(full.multivar.tab$mcmc.table[1:5],full.multivar.tab$bma.table)
full.table.df$Mean <- format(full.table.df$Mean,digits=3)
full.table.df$SD <- format(full.table.df$SD,digits=3)
full.table.df <- unite(full.table.df,"Mean (SD)",Mean:SD,sep=" (")
full.table.df$`Mean (SD)` <- paste0(full.table.df$`Mean (SD)`,")")
full.compact.table.df <- full.table.df %>% spread(Parameter,`Mean (SD)`)
full.tex.table <- full.compact.table.df %>% 
  pivot_longer(`beta[0]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

full.tex.table  <- full.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

full.tex.table$`INLAwMCMC-MH_Combined`[!is.na(full.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(full.tex.table$`INLAwMCMC-MH_Combined`[!is.na(full.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
full.tex.table <- unite(full.tex.table,"INLAwMCMC_Window1",c(`INLAwMCMC-BMA_Window 1`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
full.tex.table <- unite(full.tex.table,"INLAwMCMC_Window2",c(`INLAwMCMC-BMA_Window 2`,`INLAwMCMC-MH_Combined`),sep="",remove = TRUE,na.rm=TRUE)
full.pretex.table.final <- full.tex.table %>% select(.,-`INLAwMCMC-BMA_Window 1`)

full.pretex.table.final$Parameter <- full.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")

colnames(full.pretex.table.final) <- str_replace_all(colnames(full.pretex.table.final),"\\_","\\\\\\_")
rownames(full.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & Window 1 & Window 2 & Window 1 & Window 2 \\\\\n")

full.tex.table.final <- xtable(full.pretex.table.final,align = c("cc|cccc"))
save(full.tex.table.final,file="RegularPolygonSimStudyFullFullTableTex.rda")

print(full.tex.table.final, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'RegularPolygonSimStudyFullFullTableTex.tex')

# Change significant figures
load("RegularPolygonSimStudyFullFullTableDF.rda")
full.table.df <- rbind(full.multivar.tab$mcmc.table[1:5],full.multivar.tab$bma.table)
full.table.df$Mean <- signif(full.table.df$Mean,digits=3)
full.table.df$SD <- signif(full.table.df$SD,digits=3)
full.table.df <- unite(full.table.df,"Mean (SD)",Mean:SD,sep=" (")
full.table.df$`Mean (SD)` <- paste0(full.table.df$`Mean (SD)`,")")
full.compact.table.df <- full.table.df %>% spread(Parameter,`Mean (SD)`)
full.tex.table <- full.compact.table.df %>% 
  pivot_longer(`beta[0]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

full.tex.table  <- full.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

full.tex.table$`INLAwMCMC-MH_Combined`[!is.na(full.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(full.tex.table$`INLAwMCMC-MH_Combined`[!is.na(full.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
full.tex.table <- unite(full.tex.table,"INLAwMCMC_Window1",c(`INLAwMCMC-BMA_Window 1`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
full.tex.table <- unite(full.tex.table,"INLAwMCMC_Window2",c(`INLAwMCMC-BMA_Window 2`,`INLAwMCMC-MH_Combined`),sep="",remove = TRUE,na.rm=TRUE)
full.pretex.table.final <- full.tex.table %>% select(.,-`INLAwMCMC-BMA_Window 1`)

full.pretex.table.final$Parameter <- full.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")

colnames(full.pretex.table.final) <- str_replace_all(colnames(full.pretex.table.final),"\\_","\\\\\\_")
rownames(full.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & Window 1 & Window 2 & Window 1 & Window 2 \\\\\n")

full.tex.table.final_3sf <- xtable(full.pretex.table.final,align = c("cc|cccc"))
save(full.tex.table.final_3sf,file="RegularPolygonSimStudyFullFullTableTex_3sf.rda")

print(full.tex.table.final_3sf, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'RegularPolygonSimStudyFullFullTableTex_3sf.tex') # , only.contents=TRUE

# Different significant figures for latex table - when rounding up perfect 5's, it may be the case that they don't round correctly in signif (they round to the nearest even number), so will repeat by adding .Machine$double.eps
load("RegularPolygonSimStudyFullFullTableDF.rda")
full.table.df <- rbind(full.multivar.tab$mcmc.table[1:5],full.multivar.tab$bma.table)
full.table.df$Mean <- signif(full.table.df$Mean + .Machine$double.eps,digits=3)
full.table.df$SD <- signif(full.table.df$SD + .Machine$double.eps,digits=3)
full.table.df <- unite(full.table.df,"Mean (SD)",Mean:SD,sep=" (")
full.table.df$`Mean (SD)` <- paste0(full.table.df$`Mean (SD)`,")")
full.compact.table.df <- full.table.df %>% spread(Parameter,`Mean (SD)`)
full.tex.table <- full.compact.table.df %>% 
  pivot_longer(`beta[0]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

full.tex.table  <- full.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

full.tex.table$`INLAwMCMC-MH_Combined`[!is.na(full.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(full.tex.table$`INLAwMCMC-MH_Combined`[!is.na(full.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
full.tex.table <- unite(full.tex.table,"INLAwMCMC_Window1",c(`INLAwMCMC-BMA_Window 1`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
full.tex.table <- unite(full.tex.table,"INLAwMCMC_Window2",c(`INLAwMCMC-BMA_Window 2`,`INLAwMCMC-MH_Combined`),sep="",remove = TRUE,na.rm=TRUE)
full.pretex.table.final <- full.tex.table %>% select(.,-`INLAwMCMC-BMA_Window 1`)

full.pretex.table.final$Parameter <- full.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")

colnames(full.pretex.table.final) <- str_replace_all(colnames(full.pretex.table.final),"\\_","\\\\\\_")
rownames(full.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & Window 1 & Window 2 & Window 1 & Window 2 \\\\\n")

full.tex.table.final_3sfTake2 <- xtable(full.pretex.table.final,align = c("cc|cccc"))
save(full.tex.table.final_3sfTake2,file="RegularPolygonSimStudyFullFullTableTex_3sfTake2.rda")

print(full.tex.table.final_3sfTake2, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'RegularPolygonSimStudyFullFullTableTex_3sfTake2.tex') # , only.contents=TRUE


full.chains.full <- iwm_mcmc_chains(iwmbma_mh = fullmh.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),group = TRUE,ncol.group = 2,print=TRUE)
ggsave(plot = full.chains.full$all.plots$beta1,file="RegularPolygonSimStudyFullFullBeta1Chain_Full.pdf",height = 7.5, width = 10)
ggsave(plot = full.chains.full$all.plots$beta2,file="RegularPolygonSimStudyFullFullBeta2Chain_Full.pdf",height = 7.5, width = 10)
ggsave(plot = full.chains.full$grid.plots,file="RegularPolygonSimStudyFullFullChains_Full.pdf",height = 7.5, width = 20)
full.chains.burnin <- iwm_mcmc_chains(iwmbma_mh = fullbma.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),group = TRUE,ncol.group = 2,print=TRUE)
ggsave(plot = full.chains.burnin$all.plots$beta1,file="RegularPolygonSimStudyFullFullBeta1Chain_Burnin.pdf",height = 7.5, width = 10)
ggsave(plot = full.chains.burnin$all.plots$beta2,file="RegularPolygonSimStudyFullFullBeta2Chain_Burnin.pdf",height = 7.5, width = 10)
ggsave(plot = full.chains.burnin$grid.plots,file="RegularPolygonSimStudyFullFullChains_Burnin.pdf",height = 7.5, width = 20)

full.scatterdens.full <- iwm_mcmc_scatterdens(iwmbma_mh = fullmh.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),print=TRUE)
ggsave(plot = full.scatterdens.full$scatter$beta1_beta2,file="RegularPolygonSimStudyFullFullBeta1Beta2Scatter_Full.pdf", height = 7.5, width=7.5)
ggsave(plot = full.scatterdens.full$density$beta1_beta2,file="RegularPolygonSimStudyFullFullBeta1Beta2Density_Full.pdf",height = 7.5, width = 10)
full.scatterdens.burnin <- iwm_mcmc_scatterdens(iwmbma_mh = fullbma.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),print=TRUE)
ggsave(plot = full.scatterdens.burnin$scatter$beta1_beta2,file="RegularPolygonSimStudyFullFullBeta1Beta2Scatter_Burnin.pdf",height = 7.5, width = 7.5)
ggsave(plot = full.scatterdens.burnin$density$beta1_beta2,file="RegularPolygonSimStudyFullFullBeta1Beta2Density_Burnin.pdf",height = 7.5, width = 10)

full.postdens.mcmc <- iwm_mcmc_post(iwmbma_mh = fullbma.out$run$theta,post.lab="Multivar IwM MH",which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),hist.bins=50,prior_fun=list(beta_prior,beta_prior),true.value=list(2,-2),true.lab="True Value",inla.out = list("Window 1"=full1.fit,"Window 2"=full2.fit),inla.param = c("fixed","fixed"),inla.param.ind = c(2,3),inla.lab = c("Window 1 INLA","Window 2 INLA"),col.vec=col.vec.mcmc.noncov,print=TRUE)

ggsave(plot = full.postdens.mcmc$postdens[[1]],file="RegularPolygonSimStudyFullFullBeta1PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = full.postdens.mcmc$postdens[[2]],file="RegularPolygonSimStudyFullFullBeta2PostDens.pdf",height = 7.5, width = 10)

full.postdens.bma.w1 <- iwm_bma_post(iwmbma_bma = fullbma.out$inla.bma[[1]],post.lab="Multivar IwM BMA",which.param=c("fixed","hyperpar","hyperpar"),param.ind = c(1,2,1),param.lab = c("beta[0]","sigma","rho"),prior_fun = list(beta_prior,pc_den_sig,pc_den_rho),true.value = c(2,2,0.75),true.lab=c("True Value"),inla.out = list("Window 1"=full1.fit),inla.param = c("fixed","hyperpar","hyperpar"),inla.param.ind = c(1,2,1),inla.lab = c("Window 1 INLA"),col.vec=col.vec.bma.w1,print=TRUE)
ggsave(plot = full.postdens.bma.w1$postdens[[1]],file="RegularPolygonSimStudyFullFullW1Beta0PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = full.postdens.bma.w1$postdens[[2]],file="RegularPolygonSimStudyFullFullW1SigmaPostDens.pdf",height = 7.5, width = 10)
ggsave(plot = full.postdens.bma.w1$postdens[[3]],file="RegularPolygonSimStudyFullFullW1RhoPostDens.pdf",height = 7.5, width = 10)

mylegend <- ggpubr::get_legend(full.postdens.bma.w1$postdens[[1]] + theme(legend.title = element_text(size = 50),legend.text = element_text(size=45)) + guides(colour = guide_legend(override.aes = list(size=5))))

beta0.scaled <- full.postdens.bma.w1$postdens[[1]] + theme(legend.position="none")
sigma.scaled <- full.postdens.bma.w1$postdens[[2]] + theme(legend.position="none")
rho.scaled <- full.postdens.bma.w1$postdens[[3]] + theme(legend.position="none")

fullw1bma.postdens.grid <- grid.arrange(beta0.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=fullw1bma.postdens.grid,filename = "RegularPolygonSimStudyFullW1BMAPostDensAll.pdf",height = 20, width = 25)

full.postdens.bma.w2 <- iwm_bma_post(iwmbma_bma = fullbma.out$inla.bma[[2]],post.lab="Multivar IwM BMA",which.param=c("fixed","hyperpar","hyperpar"),param.ind = c(1,2,1),param.lab = c("beta[0]","sigma","rho"),prior_fun = list(beta_prior,pc_den_sig,pc_den_rho),true.value = c(2,2,0.75),true.lab=c("True Value"),inla.out = list("Window 2"=full2.fit),inla.param = c("fixed","hyperpar","hyperpar"),inla.param.ind = c(1,2,1),inla.lab = c("Window 2 INLA"),col.vec=col.vec.bma.w2,print=TRUE)
ggsave(plot = full.postdens.bma.w2$postdens[[1]],file="RegularPolygonSimStudyFullFullW2Beta0PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = full.postdens.bma.w2$postdens[[2]],file="RegularPolygonSimStudyFullFullW2SigmaPostDens.pdf",height = 7.5, width = 10)
ggsave(plot = full.postdens.bma.w2$postdens[[3]],file="RegularPolygonSimStudyFullFullW2RhoPostDens.pdf",height = 7.5, width = 10)

mylegend <- ggpubr::get_legend(full.postdens.bma.w2$postdens[[1]] + theme(legend.title = element_text(size = 50),legend.text = element_text(size=45)) + guides(colour = guide_legend(override.aes = list(size=5))))

beta0.scaled <- full.postdens.bma.w2$postdens[[1]] + theme(legend.position="none")
sigma.scaled <- full.postdens.bma.w2$postdens[[2]] + theme(legend.position="none")
rho.scaled <- full.postdens.bma.w2$postdens[[3]] + theme(legend.position="none")

fullw2bma.postdens.grid <- grid.arrange(beta0.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=fullw2bma.postdens.grid,filename = "RegularPolygonSimStudyFullW2BMAPostDensAll.pdf",height = 20, width = 25)

rm(full2.fit)
rm(fullbma.out)

# Full v Sparse
sparse.multivar.tab <- tab.summ(iwmbma.out=sparsebma.out,bma.param=c("fixed","hyperpar","hyperpar"),bma.ind=c(1,2,1),param.mh=param.lab.df[c(2,3)],param.bma=param.lab.df[c(1,4,5)],multi=TRUE,windows.l=2,w.lab=c("Window 1","Window 2"),cov=FALSE,inla.out=list("Window 1"=full1.fit,"Window 2"=sparse.fit),mh.inla.param=c("fixed","fixed"),mh.inla.ind=c(2,3),bma.inla.ind=c(1,2,1))
save(sparse.multivar.tab,file="RegularPolygonSimStudyFullSparseTableDF.rda")

sparse.table.df <- rbind(sparse.multivar.tab$mcmc.table[1:5],sparse.multivar.tab$bma.table)
sparse.table.df$Mean <- format(sparse.table.df$Mean,digits=3)
sparse.table.df$SD <- format(sparse.table.df$SD,digits=3)
sparse.table.df <- unite(sparse.table.df,"Mean (SD)",Mean:SD,sep=" (")
sparse.table.df$`Mean (SD)` <- paste0(sparse.table.df$`Mean (SD)`,")")
sparse.compact.table.df <- sparse.table.df %>% spread(Parameter,`Mean (SD)`)
sparse.tex.table <- sparse.compact.table.df %>% 
  pivot_longer(`beta[0]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

sparse.tex.table  <- sparse.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

sparse.tex.table$`INLAwMCMC-MH_Combined`[!is.na(sparse.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(sparse.tex.table$`INLAwMCMC-MH_Combined`[!is.na(sparse.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
sparse.tex.table <- unite(sparse.tex.table,"INLAwMCMC_Window1",c(`INLAwMCMC-BMA_Window 1`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
sparse.tex.table <- unite(sparse.tex.table,"INLAwMCMC_Window2",c(`INLAwMCMC-BMA_Window 2`,`INLAwMCMC-MH_Combined`),sep="",remove = TRUE,na.rm=TRUE)
sparse.pretex.table.final <- sparse.tex.table %>% select(.,-`INLAwMCMC-BMA_Window 1`)

sparse.pretex.table.final$Parameter <- sparse.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")

colnames(sparse.pretex.table.final) <- str_replace_all(colnames(sparse.pretex.table.final),"\\_","\\\\\\_")
rownames(sparse.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & Window 1 & Window 2 & Window 1 & Window 2 \\\\\n")

sparse.tex.table.final <- xtable(sparse.pretex.table.final,align = c("cc|cccc"))
save(sparse.tex.table.final,file="RegularPolygonSimStudyFullSparseTableTex.rda")

print(sparse.tex.table.final, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'RegularPolygonSimStudyFullSparseTableTex.tex')

# Change significant figures
load("RegularPolygonSimStudyFullSparseTableDF.rda")
sparse.table.df <- rbind(sparse.multivar.tab$mcmc.table[1:5],sparse.multivar.tab$bma.table)
sparse.table.df$Mean <- signif(sparse.table.df$Mean,digits=3)
sparse.table.df$SD <- signif(sparse.table.df$SD,digits=3)
sparse.table.df <- unite(sparse.table.df,"Mean (SD)",Mean:SD,sep=" (")
sparse.table.df$`Mean (SD)` <- paste0(sparse.table.df$`Mean (SD)`,")")
sparse.compact.table.df <- sparse.table.df %>% spread(Parameter,`Mean (SD)`)
sparse.tex.table <- sparse.compact.table.df %>% 
  pivot_longer(`beta[0]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

sparse.tex.table  <- sparse.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

sparse.tex.table$`INLAwMCMC-MH_Combined`[!is.na(sparse.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(sparse.tex.table$`INLAwMCMC-MH_Combined`[!is.na(sparse.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
sparse.tex.table <- unite(sparse.tex.table,"INLAwMCMC_Window1",c(`INLAwMCMC-BMA_Window 1`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
sparse.tex.table <- unite(sparse.tex.table,"INLAwMCMC_Window2",c(`INLAwMCMC-BMA_Window 2`,`INLAwMCMC-MH_Combined`),sep="",remove = TRUE,na.rm=TRUE)
sparse.pretex.table.final <- sparse.tex.table %>% select(.,-`INLAwMCMC-BMA_Window 1`)

sparse.pretex.table.final$Parameter <- sparse.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")

colnames(sparse.pretex.table.final) <- str_replace_all(colnames(sparse.pretex.table.final),"\\_","\\\\\\_")
rownames(sparse.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & Window 1 & Window 2 & Window 1 & Window 2 \\\\\n")

sparse.tex.table.final_3sf <- xtable(sparse.pretex.table.final,align = c("cc|cccc"))
save(sparse.tex.table.final_3sf,file="RegularPolygonSimStudyFullSparseTableTex_3sf.rda")

print(sparse.tex.table.final_3sf, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'RegularPolygonSimStudyFullSparseTableTex_3sf.tex') # , only.contents=TRUE

# # Smaller significant figures for tables - Take 2 for signif() potential rounding issues with 5's
load("RegularPolygonSimStudyFullSparseTableDF.rda")
sparse.table.df <- rbind(sparse.multivar.tab$mcmc.table[1:5],sparse.multivar.tab$bma.table)
sparse.table.df$Mean <- signif(sparse.table.df$Mean + .Machine$double.eps,digits=3)
sparse.table.df$SD <- signif(sparse.table.df$SD + .Machine$double.eps,digits=3)
sparse.table.df <- unite(sparse.table.df,"Mean (SD)",Mean:SD,sep=" (")
sparse.table.df$`Mean (SD)` <- paste0(sparse.table.df$`Mean (SD)`,")")
sparse.compact.table.df <- sparse.table.df %>% spread(Parameter,`Mean (SD)`)
sparse.tex.table <- sparse.compact.table.df %>% 
  pivot_longer(`beta[0]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

sparse.tex.table  <- sparse.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

sparse.tex.table$`INLAwMCMC-MH_Combined`[!is.na(sparse.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(sparse.tex.table$`INLAwMCMC-MH_Combined`[!is.na(sparse.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
sparse.tex.table <- unite(sparse.tex.table,"INLAwMCMC_Window1",c(`INLAwMCMC-BMA_Window 1`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
sparse.tex.table <- unite(sparse.tex.table,"INLAwMCMC_Window2",c(`INLAwMCMC-BMA_Window 2`,`INLAwMCMC-MH_Combined`),sep="",remove = TRUE,na.rm=TRUE)
sparse.pretex.table.final <- sparse.tex.table %>% select(.,-`INLAwMCMC-BMA_Window 1`)

sparse.pretex.table.final$Parameter <- sparse.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")

colnames(sparse.pretex.table.final) <- str_replace_all(colnames(sparse.pretex.table.final),"\\_","\\\\\\_")
rownames(sparse.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & Window 1 & Window 2 & Window 1 & Window 2 \\\\\n")

sparse.tex.table.final_3sfTake2 <- xtable(sparse.pretex.table.final,align = c("cc|cccc"))
save(sparse.tex.table.final_3sfTake2,file="RegularPolygonSimStudyFullSparseTableTex_3sfTake2.rda")

print(sparse.tex.table.final_3sfTake2, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'RegularPolygonSimStudyFullSparseTableTex_3sfTake2.tex') # , only.contents=TRUE



sparse.chains.full <- iwm_mcmc_chains(iwmbma_mh = sparsemh.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),group = TRUE,ncol.group = 2,print=TRUE)
ggsave(plot = sparse.chains.full$all.plots$beta1,file="RegularPolygonSimStudyFullSparseBeta1Chain_Full.pdf",height = 7.5, width = 10)
ggsave(plot = sparse.chains.full$all.plots$beta2,file="RegularPolygonSimStudyFullSparseBeta2Chain_Full.pdf",height = 7.5, width = 10)
ggsave(plot = sparse.chains.full$grid.plots,file="RegularPolygonSimStudyFullSparseChains_Full.pdf",height = 7.5, width = 20)
sparse.chains.burnin <- iwm_mcmc_chains(iwmbma_mh = sparsebma.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),group = TRUE,ncol.group = 2,print=TRUE)
ggsave(plot = sparse.chains.burnin$all.plots$beta1,file="RegularPolygonSimStudyFullSparseBeta1Chain_Burnin.pdf",height = 7.5, width = 10)
ggsave(plot = sparse.chains.burnin$all.plots$beta2,file="RegularPolygonSimStudyFullSparseBeta2Chain_Burnin.pdf",height = 7.5, width = 10)
ggsave(plot = sparse.chains.burnin$grid.plots,file="RegularPolygonSimStudyFullSparseChains_Burnin.pdf",height = 7.5, width = 20)

sparse.scatterdens.full <- iwm_mcmc_scatterdens(iwmbma_mh = sparsemh.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),print=TRUE)
ggsave(plot = sparse.scatterdens.full$scatter$beta1_beta2,file="RegularPolygonSimStudyFullSparseBeta1Beta2Scatter_Full.pdf",height = 7.5, width = 7.5)
ggsave(plot = sparse.scatterdens.full$density$beta1_beta2,file="RegularPolygonSimStudyFullSparseBeta1Beta2Density_Full.pdf",height = 7.5, width = 10)
sparse.scatterdens.burnin <- iwm_mcmc_scatterdens(iwmbma_mh = sparsebma.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),print=TRUE)
ggsave(plot = sparse.scatterdens.burnin$scatter$beta1_beta2,file="RegularPolygonSimStudyFullSparseBeta1Beta2Scatter_Burnin.pdf",height = 7.5, width = 7.5)
ggsave(plot = sparse.scatterdens.burnin$density$beta1_beta2,file="RegularPolygonSimStudyFullSparseBeta1Beta2Density_Burnin.pdf",height = 7.5, width = 10)

sparse.postdens.mcmc <- iwm_mcmc_post(iwmbma_mh = sparsebma.out$run$theta,post.lab="Multivar IwM MH",which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),hist.bins=50,prior_fun=list(beta_prior,beta_prior),true.value=list(2,-2),true.lab=c("True Value"),inla.out = list("Window 1"=full1.fit,"Window 2"=sparse.fit),inla.param = c("fixed","fixed"),inla.param.ind = c(2,3),inla.lab = c("Window 1 INLA","Window 2 INLA"),col.vec=col.vec.mcmc.noncov,print=TRUE)

ggsave(plot = sparse.postdens.mcmc$postdens[[1]],file="RegularPolygonSimStudyFullSparseBeta1PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = sparse.postdens.mcmc$postdens[[2]],file="RegularPolygonSimStudyFullSparseBeta2PostDens.pdf",height = 7.5, width = 10)

sparse.postdens.bma.w1 <- iwm_bma_post(iwmbma_bma = sparsebma.out$inla.bma[[1]],post.lab="Multivar IwM BMA",which.param=c("fixed","hyperpar","hyperpar"),param.ind = c(1,2,1),param.lab = c("beta[0]","sigma","rho"),prior_fun = list(beta_prior,pc_den_sig,pc_den_rho),true.value = c(2,2,0.75),true.lab=c("True Value"),inla.out = list("Window 1"=full1.fit),inla.param = c("fixed","hyperpar","hyperpar"),inla.param.ind = c(1,2,1),inla.lab = c("Window 1 INLA"),col.vec=col.vec.bma.w1,print=TRUE)
ggsave(plot = sparse.postdens.bma.w1$postdens[[1]],file="RegularPolygonSimStudyFullSparseW1Beta0PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = sparse.postdens.bma.w1$postdens[[2]],file="RegularPolygonSimStudyFullSparseW1SigmaPostDens.pdf",height = 7.5, width = 10)
ggsave(plot = sparse.postdens.bma.w1$postdens[[3]],file="RegularPolygonSimStudyFullSparseW1RhoPostDens.pdf",height = 7.5, width = 10)

mylegend <- ggpubr::get_legend(sparse.postdens.bma.w1$postdens[[1]] + theme(legend.title = element_text(size = 50),legend.text = element_text(size=45)) + guides(colour = guide_legend(override.aes = list(size=5))))

beta0.scaled <- sparse.postdens.bma.w1$postdens[[1]] + theme(legend.position="none")
sigma.scaled <- sparse.postdens.bma.w1$postdens[[2]] + theme(legend.position="none")
rho.scaled <- sparse.postdens.bma.w1$postdens[[3]] + theme(legend.position="none")

sparsew1bma.postdens.grid <- grid.arrange(beta0.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=sparsew1bma.postdens.grid,filename = "RegularPolygonSimStudySparseW1BMAPostDensAll.pdf",height = 20, width = 25)

sparse.postdens.bma.w2 <- iwm_bma_post(iwmbma_bma = sparsebma.out$inla.bma[[2]],post.lab="Multivar IwM BMA",which.param=c("fixed","hyperpar","hyperpar"),param.ind = c(1,2,1),param.lab = c("beta[0]","sigma","rho"),prior_fun = list(beta_prior,pc_den_sig,pc_den_rho),true.value = c(2,2,0.75),true.lab=c("True Value"),inla.out = list("Window 2"=sparse.fit),inla.param = c("fixed","hyperpar","hyperpar"),inla.param.ind = c(1,2,1),inla.lab = c("Window 2 INLA"),col.vec=col.vec.bma.w2,print=TRUE)
ggsave(plot = sparse.postdens.bma.w2$postdens[[1]],file="RegularPolygonSimStudyFullSparseW2Beta0PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = sparse.postdens.bma.w2$postdens[[2]],file="RegularPolygonSimStudyFullSparseW2SigmaPostDens.pdf",height = 7.5, width = 10)
ggsave(plot = sparse.postdens.bma.w2$postdens[[3]],file="RegularPolygonSimStudyFullSparseW2RhoPostDens.pdf",height = 7.5, width = 10)

mylegend <- ggpubr::get_legend(sparse.postdens.bma.w2$postdens[[1]] + theme(legend.title = element_text(size = 50),legend.text = element_text(size=45)) + guides(colour = guide_legend(override.aes = list(size=5))))

beta0.scaled <- sparse.postdens.bma.w2$postdens[[1]] + theme(legend.position="none")
sigma.scaled <- sparse.postdens.bma.w2$postdens[[2]] + theme(legend.position="none")
rho.scaled <- sparse.postdens.bma.w2$postdens[[3]] + theme(legend.position="none")

sparsew2bma.postdens.grid <- grid.arrange(beta0.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=sparsew2bma.postdens.grid,filename = "RegularPolygonSimStudySparseW2BMAPostDensAll.pdf",height = 20, width = 25)

rm(sparse.fit)
rm(sparsebma.out)


# Full v Cov
cov.multivar.tab <- tab.summ(iwmbma.out=covbma.out,bma.param=c("fixed","hyperpar","hyperpar"),bma.ind=c(1,2,1),param.mh=param.lab.df[c(2,3)],param.bma=param.lab.df[c(1,4,5)],multi=TRUE,windows.l=2,w.lab=c("Window 1","Window 2"),cov=TRUE,cov.param=c("fixed","fixed"),cov.ind = c(2,3),comb.ind=c(1,2),param.cov=c("b[1]","b[2]"),inla.out=list("Window 1"=full1.fit,"Window 2"=cov.fit),mh.inla.param=c("fixed","fixed"),mh.inla.ind=c(2,3),bma.inla.ind=c(1,2,1))
save(cov.multivar.tab,file="RegularPolygonSimStudyFullCovTableDF.rda")


cov.table.df <- rbind(cov.multivar.tab$mcmc.table[1:5],cov.multivar.tab$bma.table)
cov.table.df$Mean <- format(cov.table.df$Mean,digits=3)
cov.table.df$SD <- format(cov.table.df$SD,digits=3)
cov.table.df[cov.table.df$Algorithm=="INLA"&cov.table.df$Window=="Window 2"&cov.table.df$Parameter=="beta[1]",]$Parameter <- "beta[1]+b[1]"
cov.table.df[cov.table.df$Algorithm=="INLA"&cov.table.df$Window=="Window 2"&cov.table.df$Parameter=="beta[2]",]$Parameter <- "beta[2]+b[2]"
cov.table.df <- unite(cov.table.df,"Mean (SD)",Mean:SD,sep=" (")
cov.table.df$`Mean (SD)` <- paste0(cov.table.df$`Mean (SD)`,")")
cov.compact.table.df <- cov.table.df %>% spread(Parameter,`Mean (SD)`)
cov.tex.table <- cov.compact.table.df %>% 
  pivot_longer(`b[1]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

cov.tex.table  <- cov.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

cov.tex.table$`INLAwMCMC-MH_Combined`[!is.na(cov.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(cov.tex.table$`INLAwMCMC-MH_Combined`[!is.na(cov.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
cov.tex.table$`INLAwMCMC_Window 2`[!is.na(cov.tex.table$`INLAwMCMC_Window 2`)] <- paste0(cov.tex.table$`INLAwMCMC_Window 2`[!is.na(cov.tex.table$`INLAwMCMC_Window 2`)],"$^\\star$")
cov.tex.table <- unite(cov.tex.table,"IwM_Window 1",c(`INLAwMCMC-BMA_Window 1`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
cov.tex.table <- unite(cov.tex.table,"IwM_Window 2",c(`INLAwMCMC-BMA_Window 2`,`INLAwMCMC-MH_Combined`,`INLAwMCMC_Window 2`),sep="",remove = TRUE,na.rm=TRUE)
cov.tex.table <- cov.tex.table %>% select(.,-`INLAwMCMC-BMA_Window 1`)
cov.pretex.table.final <- cov.tex.table[,c(1:3,5,4)] # IwM w1 and w2 are wrong way round for labels...


cov.pretex.table.final$Parameter <- cov.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
cov.pretex.table.final$Parameter <- cov.pretex.table.final$Parameter %>% str_replace_all(.,"\\\\b\\_","\\b\\_")

colnames(cov.pretex.table.final) <- str_replace_all(colnames(cov.pretex.table.final),"\\_","\\\\\\_")
rownames(cov.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & Window 1 & Window 2 & Window 1 & Window 2 \\\\\n")

cov.tex.table.final <- xtable(cov.pretex.table.final,align = c("cc|cccc"))
save(cov.tex.table.final,file="RegularPolygonSimStudyFullCovTableTex.rda")

print(cov.tex.table.final, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'RegularPolygonSimStudyFullCovTableTex.tex')

# Change significant figures
load("RegularPolygonSimStudyFullCovTableDF.rda")
cov.table.df <- rbind(cov.multivar.tab$mcmc.table[1:5],cov.multivar.tab$bma.table)
cov.table.df$Mean <- signif(cov.table.df$Mean,digits=3)
cov.table.df$SD <- signif(cov.table.df$SD,digits=3)
cov.table.df[cov.table.df$Algorithm=="INLA"&cov.table.df$Window=="Window 2"&cov.table.df$Parameter=="beta[1]",]$Parameter <- "beta[1]+b[1]"
cov.table.df[cov.table.df$Algorithm=="INLA"&cov.table.df$Window=="Window 2"&cov.table.df$Parameter=="beta[2]",]$Parameter <- "beta[2]+b[2]"
cov.table.df <- unite(cov.table.df,"Mean (SD)",Mean:SD,sep=" (")
cov.table.df$`Mean (SD)` <- paste0(cov.table.df$`Mean (SD)`,")")
cov.compact.table.df <- cov.table.df %>% spread(Parameter,`Mean (SD)`)
cov.tex.table <- cov.compact.table.df %>% 
  pivot_longer(`b[1]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

cov.tex.table  <- cov.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

cov.tex.table$`INLAwMCMC-MH_Combined`[!is.na(cov.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(cov.tex.table$`INLAwMCMC-MH_Combined`[!is.na(cov.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
cov.tex.table$`INLAwMCMC_Window 2`[!is.na(cov.tex.table$`INLAwMCMC_Window 2`)] <- paste0(cov.tex.table$`INLAwMCMC_Window 2`[!is.na(cov.tex.table$`INLAwMCMC_Window 2`)],"$^\\star$")
cov.tex.table <- unite(cov.tex.table,"IwM_Window 1",c(`INLAwMCMC-BMA_Window 1`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
cov.tex.table <- unite(cov.tex.table,"IwM_Window 2",c(`INLAwMCMC-BMA_Window 2`,`INLAwMCMC-MH_Combined`,`INLAwMCMC_Window 2`),sep="",remove = TRUE,na.rm=TRUE)
cov.tex.table <- cov.tex.table %>% select(.,-`INLAwMCMC-BMA_Window 1`)
cov.pretex.table.final <- cov.tex.table[,c(1:3,5,4)] # IwM w1 and w2 are wrong way round for labels...


cov.pretex.table.final$Parameter <- cov.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
cov.pretex.table.final$Parameter <- cov.pretex.table.final$Parameter %>% str_replace_all(.,"\\\\b\\_","\\b\\_")

colnames(cov.pretex.table.final) <- str_replace_all(colnames(cov.pretex.table.final),"\\_","\\\\\\_")
rownames(cov.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & Window 1 & Window 2 & Window 1 & Window 2 \\\\\n")

cov.tex.table.final_3sf <- xtable(cov.pretex.table.final,align = c("cc|cccc"))
save(cov.tex.table.final_3sf,file="RegularPolygonSimStudyFullCovTableTex_3sf.rda")

print(cov.tex.table.final_3sf, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'RegularPolygonSimStudyFullCovTableTex_3sf.tex')

# Smaller significant figures for tables - Take 2 for signif() potential rounding issues
load("RegularPolygonSimStudyFullCovTableDF.rda")
cov.table.df <- rbind(cov.multivar.tab$mcmc.table[1:5],cov.multivar.tab$bma.table)
cov.table.df$Mean <- signif(cov.table.df$Mean + .Machine$double.eps,digits=3)
cov.table.df$SD <- signif(cov.table.df$SD + .Machine$double.eps,digits=3)
cov.table.df[cov.table.df$Algorithm=="INLA"&cov.table.df$Window=="Window 2"&cov.table.df$Parameter=="beta[1]",]$Parameter <- "beta[1]+b[1]"
cov.table.df[cov.table.df$Algorithm=="INLA"&cov.table.df$Window=="Window 2"&cov.table.df$Parameter=="beta[2]",]$Parameter <- "beta[2]+b[2]"
cov.table.df <- unite(cov.table.df,"Mean (SD)",Mean:SD,sep=" (")
cov.table.df$`Mean (SD)` <- paste0(cov.table.df$`Mean (SD)`,")")
cov.compact.table.df <- cov.table.df %>% spread(Parameter,`Mean (SD)`)
cov.tex.table <- cov.compact.table.df %>% 
  pivot_longer(`b[1]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

cov.tex.table  <- cov.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

cov.tex.table$`INLAwMCMC-MH_Combined`[!is.na(cov.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(cov.tex.table$`INLAwMCMC-MH_Combined`[!is.na(cov.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
cov.tex.table$`INLAwMCMC_Window 2`[!is.na(cov.tex.table$`INLAwMCMC_Window 2`)] <- paste0(cov.tex.table$`INLAwMCMC_Window 2`[!is.na(cov.tex.table$`INLAwMCMC_Window 2`)],"$^\\star$")
cov.tex.table <- unite(cov.tex.table,"IwM_Window 1",c(`INLAwMCMC-BMA_Window 1`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
cov.tex.table <- unite(cov.tex.table,"IwM_Window 2",c(`INLAwMCMC-BMA_Window 2`,`INLAwMCMC-MH_Combined`,`INLAwMCMC_Window 2`),sep="",remove = TRUE,na.rm=TRUE)
cov.tex.table <- cov.tex.table %>% select(.,-`INLAwMCMC-BMA_Window 1`)
cov.pretex.table.final <- cov.tex.table[,c(1:3,5,4)] # IwM w1 and w2 are wrong way round for labels...


cov.pretex.table.final$Parameter <- cov.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
cov.pretex.table.final$Parameter <- cov.pretex.table.final$Parameter %>% str_replace_all(.,"\\\\b\\_","\\b\\_")

colnames(cov.pretex.table.final) <- str_replace_all(colnames(cov.pretex.table.final),"\\_","\\\\\\_")
rownames(cov.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & Window 1 & Window 2 & Window 1 & Window 2 \\\\\n")

cov.tex.table.final_3sfTake2 <- xtable(cov.pretex.table.final,align = c("cc|cccc"))
save(cov.tex.table.final_3sfTake2,file="RegularPolygonSimStudyFullCovTableTex_3sfTake2.rda")

print(cov.tex.table.final_3sfTake2, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'RegularPolygonSimStudyFullCovTableTex_3sfTake2.tex') # , only.contents=TRUE



cov.chains.full <- iwm_mcmc_chains(iwmbma_mh = covmh.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),group = TRUE,ncol.group = 2,print=TRUE)
ggsave(plot = cov.chains.full$all.plots$beta1,file="RegularPolygonSimStudyFullCovBeta1Chain_Full.pdf",height = 7.5, width = 10)
ggsave(plot = cov.chains.full$all.plots$beta2,file="RegularPolygonSimStudyFullCovBeta2Chain_Full.pdf",height = 7.5, width = 10)
ggsave(plot = cov.chains.full$grid.plots,file="RegularPolygonSimStudyFullCovChains_Full.pdf",height = 7.5, width = 20)
cov.chains.burnin <- iwm_mcmc_chains(iwmbma_mh = covbma.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),group = TRUE,ncol.group = 2,print=TRUE)
ggsave(plot = cov.chains.burnin$all.plots$beta1,file="RegularPolygonSimStudyFullCovBeta1Chain_Burnin.pdf",height = 7.5, width = 10)
ggsave(plot = cov.chains.burnin$all.plots$beta2,file="RegularPolygonSimStudyFullCovBeta2Chain_Burnin.pdf",height = 7.5, width = 10)
ggsave(plot = cov.chains.burnin$grid.plots,file="RegularPolygonSimStudyFullCovChains_Burnin.pdf",height = 7.5, width = 20)

cov.scatterdens.full <- iwm_mcmc_scatterdens(iwmbma_mh = covmh.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),print=TRUE)
ggsave(plot = cov.scatterdens.full$scatter$beta1_beta2,file="RegularPolygonSimStudyFullCovBeta1Beta2Scatter_Full.pdf",height = 7.5, width = 7.5)
ggsave(plot = cov.scatterdens.full$density$beta1_beta2,file="RegularPolygonSimStudyFullCovBeta1Beta2Density_Full.pdf",height = 7.5, width = 10)
cov.scatterdens.burnin <- iwm_mcmc_scatterdens(iwmbma_mh = covbma.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),print=TRUE)
ggsave(plot = cov.scatterdens.burnin$scatter$beta1_beta2,file="RegularPolygonSimStudyFullCovBeta1Beta2Scatter_Burnin.pdf",height = 7.5, width = 7.5)
ggsave(plot = cov.scatterdens.burnin$density$beta1_beta2,file="RegularPolygonSimStudyFullCovBeta1Beta2Density_Burnin.pdf",height = 7.5, width = 10)

cov.postdens.mcmc <- iwm_mcmc_post(iwmbma_mh = covbma.out$run$theta,post.lab="Multivar IwM MH",which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),hist.bins=50,prior_fun=list(beta_prior,beta_prior),true.value=list(2,c(-2,-1)),true.lab=c("True Window 1","True Window 2"),inla.out = list("Window 1"=full1.fit,"Window 2"=cov.fit),inla.param = c("fixed","fixed"),inla.param.ind = c(2,3),inla.lab = c("Window 1 INLA","Window 2 INLA"),iwmbma_bma = covbma.out$inla.bma,which.window = 2,which.param_combo=c(1,2),combo.param = c("fixed","fixed"),combo.param.ind = c(1,2),combo.param.lab = c("beta[1] + b[1]","beta[2] + b[2]"),col.vec=col.vec.mcmc.cov,print=TRUE)

ggsave(plot = cov.postdens.mcmc$postdens[[1]],file="RegularPolygonSimStudyFullCovBeta1PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = cov.postdens.mcmc$postdens.cov[[1]],file="RegularPolygonSimStudyFullCovBeta1PostDenswComb.pdf",height = 7.5, width = 10)
ggsave(plot = cov.postdens.mcmc$postdens[[2]],file="RegularPolygonSimStudyFullCovBeta2PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = cov.postdens.mcmc$postdens.cov[[2]],file="RegularPolygonSimStudyFullCovBeta2PostDenswComb.pdf",height = 7.5, width = 10)

cov.postdens.bma.w1 <- iwm_bma_post(iwmbma_bma = covbma.out$inla.bma[[1]],post.lab="Multivar IwM BMA",which.param=c("fixed","hyperpar","hyperpar"),param.ind = c(1,2,1),param.lab = c("beta[0]","sigma","rho"),prior_fun = list(beta_prior,pc_den_sig,pc_den_rho),true.value = c(2,2,0.75),true.lab=c("True Value"),inla.out = list("Window 1"=full1.fit),inla.param = c("fixed","hyperpar","hyperpar"),inla.param.ind = c(1,2,1),inla.lab = c("Window 1 INLA"),col.vec=col.vec.bma.w1,print=TRUE)
ggsave(plot = cov.postdens.bma.w1$postdens[[1]],file="RegularPolygonSimStudyFullCovW1Beta0PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = cov.postdens.bma.w1$postdens[[2]],file="RegularPolygonSimStudyFullCovW1SigmaPostDens.pdf",height = 7.5, width = 10)
ggsave(plot = cov.postdens.bma.w1$postdens[[3]],file="RegularPolygonSimStudyFullCovW1RhoPostDens.pdf",height = 7.5, width = 10)

mylegend <- ggpubr::get_legend(cov.postdens.bma.w1$postdens[[1]] + theme(legend.title = element_text(size = 50),legend.text = element_text(size=45)) + guides(colour = guide_legend(override.aes = list(size=5))))

beta0.scaled <- cov.postdens.bma.w1$postdens[[1]] + theme(legend.position="none")
sigma.scaled <- cov.postdens.bma.w1$postdens[[2]] + theme(legend.position="none")
rho.scaled <- cov.postdens.bma.w1$postdens[[3]] + theme(legend.position="none")

covw1bma.postdens.grid <- grid.arrange(beta0.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=covw1bma.postdens.grid,filename = "RegularPolygonSimStudyCovW1BMAPostDensAll.pdf",height = 20, width = 25)

cov.postdens.bma.w2 <- iwm_bma_post(iwmbma_bma = covbma.out$inla.bma[[2]],post.lab="Multivar IwM BMA",which.param=c("fixed","hyperpar","hyperpar"),param.ind = c(1,2,1),param.lab = c("beta[0]","sigma","rho"),prior_fun = list(beta_prior,pc_den_sig,pc_den_rho),true.value = c(2,2,0.75),true.lab=c("True Value"),inla.out = list("Window 2"=cov.fit),inla.param = c("fixed","hyperpar","hyperpar"),inla.param.ind = c(1,2,1),inla.lab = c("Window 2 INLA"),col.vec=col.vec.bma.w2,print=TRUE)
ggsave(plot = cov.postdens.bma.w2$postdens[[1]],file="RegularPolygonSimStudyFullCovW2Beta0PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = cov.postdens.bma.w2$postdens[[2]],file="RegularPolygonSimStudyFullCovW2SigmaPostDens.pdf",height = 7.5, width = 10)
ggsave(plot = cov.postdens.bma.w2$postdens[[3]],file="RegularPolygonSimStudyFullCovW2RhoPostDens.pdf",height = 7.5, width = 10)

mylegend <- ggpubr::get_legend(cov.postdens.bma.w2$postdens[[1]] + theme(legend.title = element_text(size = 50),legend.text = element_text(size=45)) + guides(colour = guide_legend(override.aes = list(size=5))))

beta0.scaled <- cov.postdens.bma.w2$postdens[[1]] + theme(legend.position="none")
sigma.scaled <- cov.postdens.bma.w2$postdens[[2]] + theme(legend.position="none")
rho.scaled <- cov.postdens.bma.w2$postdens[[3]] + theme(legend.position="none")

covw2bma.postdens.grid <- grid.arrange(beta0.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=covw2bma.postdens.grid,filename = "RegularPolygonSimStudyCovW2BMAPostDensAll.pdf",height = 20, width = 25)

# Full v DS Cov
dscov.multivar.tab <- tab.summ(iwmbma.out=dscovbma.out,bma.param=c("fixed","hyperpar","hyperpar"),bma.ind=c(1,2,1),param.mh=param.lab.df[c(2,3)],param.bma=param.lab.df[c(1,4,5)],multi=TRUE,windows.l=2,w.lab=c("Window 1","Window 2"),cov=TRUE,cov.param=c("fixed","fixed"),cov.ind = c(2,3),comb.ind=c(1,2),param.cov=c("b[1]","b[2]"),inla.out=list("Window 1"=full1.fit,"Window 2"=dscov.fit),mh.inla.param=c("fixed","fixed"),mh.inla.ind=c(2,3),bma.inla.ind=c(1,2,1))
save(dscov.multivar.tab,file="RegularPolygonSimStudyFullDSCovTableDF.rda")


dscov.table.df <- rbind(dscov.multivar.tab$mcmc.table[1:5],dscov.multivar.tab$bma.table)
dscov.table.df$Mean <- format(dscov.table.df$Mean,digits=3)
dscov.table.df$SD <- format(dscov.table.df$SD,digits=3)
dscov.table.df[dscov.table.df$Algorithm=="INLA"&dscov.table.df$Window=="Window 2"&dscov.table.df$Parameter=="beta[1]",]$Parameter <- "beta[1]+b[1]"
dscov.table.df[dscov.table.df$Algorithm=="INLA"&dscov.table.df$Window=="Window 2"&dscov.table.df$Parameter=="beta[2]",]$Parameter <- "beta[2]+b[2]"
dscov.table.df <- unite(dscov.table.df,"Mean (SD)",Mean:SD,sep=" (")
dscov.table.df$`Mean (SD)` <- paste0(dscov.table.df$`Mean (SD)`,")")
dscov.compact.table.df <- dscov.table.df %>% spread(Parameter,`Mean (SD)`)
dscov.tex.table <- dscov.compact.table.df %>% 
  pivot_longer(`b[1]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

dscov.tex.table  <- dscov.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

dscov.tex.table$`INLAwMCMC-MH_Combined`[!is.na(dscov.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(dscov.tex.table$`INLAwMCMC-MH_Combined`[!is.na(dscov.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
dscov.tex.table$`INLAwMCMC_Window 2`[!is.na(dscov.tex.table$`INLAwMCMC_Window 2`)] <- paste0(dscov.tex.table$`INLAwMCMC_Window 2`[!is.na(dscov.tex.table$`INLAwMCMC_Window 2`)],"$^\\star$")
dscov.tex.table <- unite(dscov.tex.table,"IwM_Window 1",c(`INLAwMCMC-BMA_Window 1`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
dscov.tex.table <- unite(dscov.tex.table,"IwM_Window 2",c(`INLAwMCMC-BMA_Window 2`,`INLAwMCMC-MH_Combined`,`INLAwMCMC_Window 2`),sep="",remove = TRUE,na.rm=TRUE)
dscov.tex.table <- dscov.tex.table %>% select(.,-`INLAwMCMC-BMA_Window 1`)
dscov.pretex.table.final <- dscov.tex.table[,c(1:3,5,4)] # IwM w1 and w2 are wrong way round for labels...



dscov.pretex.table.final$Parameter <- dscov.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
dscov.pretex.table.final$Parameter <- dscov.pretex.table.final$Parameter %>% str_replace_all(.,"\\\\b\\_","\\b\\_")

colnames(dscov.pretex.table.final) <- str_replace_all(colnames(dscov.pretex.table.final),"\\_","\\\\\\_")
rownames(dscov.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & Window 1 & Window 2 & Window 1 & Window 2 \\\\\n")

dscov.tex.table.final <- xtable(dscov.pretex.table.final,align = c("cc|cccc"))
save(dscov.tex.table.final,file="RegularPolygonSimStudyFullDSCovTableTex.rda")

print(dscov.tex.table.final, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'RegularPolygonSimStudyFullDSCovTableTex.tex')

# Change significant figures
load("RegularPolygonSimStudyFullDSCovTableDF.rda")
dscov.table.df <- rbind(dscov.multivar.tab$mcmc.table[1:5],dscov.multivar.tab$bma.table)
dscov.table.df$Mean <- signif(dscov.table.df$Mean,digits=3)
dscov.table.df$SD <- signif(dscov.table.df$SD,digits=3)
dscov.table.df[dscov.table.df$Algorithm=="INLA"&dscov.table.df$Window=="Window 2"&dscov.table.df$Parameter=="beta[1]",]$Parameter <- "beta[1]+b[1]"
dscov.table.df[dscov.table.df$Algorithm=="INLA"&dscov.table.df$Window=="Window 2"&dscov.table.df$Parameter=="beta[2]",]$Parameter <- "beta[2]+b[2]"
dscov.table.df <- unite(dscov.table.df,"Mean (SD)",Mean:SD,sep=" (")
dscov.table.df$`Mean (SD)` <- paste0(dscov.table.df$`Mean (SD)`,")")
dscov.compact.table.df <- dscov.table.df %>% spread(Parameter,`Mean (SD)`)
dscov.tex.table <- dscov.compact.table.df %>% 
  pivot_longer(`b[1]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

dscov.tex.table  <- dscov.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

dscov.tex.table$`INLAwMCMC-MH_Combined`[!is.na(dscov.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(dscov.tex.table$`INLAwMCMC-MH_Combined`[!is.na(dscov.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
dscov.tex.table$`INLAwMCMC_Window 2`[!is.na(dscov.tex.table$`INLAwMCMC_Window 2`)] <- paste0(dscov.tex.table$`INLAwMCMC_Window 2`[!is.na(dscov.tex.table$`INLAwMCMC_Window 2`)],"$^\\star$")
dscov.tex.table <- unite(dscov.tex.table,"IwM_Window 1",c(`INLAwMCMC-BMA_Window 1`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
dscov.tex.table <- unite(dscov.tex.table,"IwM_Window 2",c(`INLAwMCMC-BMA_Window 2`,`INLAwMCMC-MH_Combined`,`INLAwMCMC_Window 2`),sep="",remove = TRUE,na.rm=TRUE)
dscov.tex.table <- dscov.tex.table %>% select(.,-`INLAwMCMC-BMA_Window 1`)
dscov.pretex.table.final <- dscov.tex.table[,c(1:3,5,4)] # IwM w1 and w2 are wrong way round for labels...


dscov.pretex.table.final$Parameter <- dscov.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
dscov.pretex.table.final$Parameter <- dscov.pretex.table.final$Parameter %>% str_replace_all(.,"\\\\b\\_","\\b\\_")

colnames(dscov.pretex.table.final) <- str_replace_all(colnames(dscov.pretex.table.final),"\\_","\\\\\\_")
rownames(dscov.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & Window 1 & Window 2 & Window 1 & Window 2 \\\\\n")

dscov.tex.table.final_3sf <- xtable(dscov.pretex.table.final,align = c("cc|cccc"))
save(dscov.tex.table.final_3sf,file="RegularPolygonSimStudyFullDSCovTableTex_3sf.rda")

print(dscov.tex.table.final_3sf, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'RegularPolygonSimStudyFullDSCovTableTex_3sf.tex')

# Smaller significant figures for tables - Take 2 for signif() potential rounding issues
load("RegularPolygonSimStudyFullDSCovTableDF.rda")
dscov.table.df <- rbind(dscov.multivar.tab$mcmc.table[1:5],dscov.multivar.tab$bma.table)
dscov.table.df$Mean <- signif(dscov.table.df$Mean + .Machine$double.eps,digits=3)
dscov.table.df$SD <- signif(dscov.table.df$SD + .Machine$double.eps,digits=3)
dscov.table.df[dscov.table.df$Algorithm=="INLA"&dscov.table.df$Window=="Window 2"&dscov.table.df$Parameter=="beta[1]",]$Parameter <- "beta[1]+b[1]"
dscov.table.df[dscov.table.df$Algorithm=="INLA"&dscov.table.df$Window=="Window 2"&dscov.table.df$Parameter=="beta[2]",]$Parameter <- "beta[2]+b[2]"
dscov.table.df <- unite(dscov.table.df,"Mean (SD)",Mean:SD,sep=" (")
dscov.table.df$`Mean (SD)` <- paste0(dscov.table.df$`Mean (SD)`,")")
dscov.compact.table.df <- dscov.table.df %>% spread(Parameter,`Mean (SD)`)
dscov.tex.table <- dscov.compact.table.df %>% 
  pivot_longer(`b[1]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

dscov.tex.table  <- dscov.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

dscov.tex.table$`INLAwMCMC-MH_Combined`[!is.na(dscov.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(dscov.tex.table$`INLAwMCMC-MH_Combined`[!is.na(dscov.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
dscov.tex.table$`INLAwMCMC_Window 2`[!is.na(dscov.tex.table$`INLAwMCMC_Window 2`)] <- paste0(dscov.tex.table$`INLAwMCMC_Window 2`[!is.na(dscov.tex.table$`INLAwMCMC_Window 2`)],"$^\\star$")
dscov.tex.table <- unite(dscov.tex.table,"IwM_Window 1",c(`INLAwMCMC-BMA_Window 1`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
dscov.tex.table <- unite(dscov.tex.table,"IwM_Window 2",c(`INLAwMCMC-BMA_Window 2`,`INLAwMCMC-MH_Combined`,`INLAwMCMC_Window 2`),sep="",remove = TRUE,na.rm=TRUE)
dscov.tex.table <- dscov.tex.table %>% select(.,-`INLAwMCMC-BMA_Window 1`)
dscov.pretex.table.final <- dscov.tex.table[,c(1:3,5,4)] # IwM w1 and w2 are wrong way round for labels...


dscov.pretex.table.final$Parameter <- dscov.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
dscov.pretex.table.final$Parameter <- dscov.pretex.table.final$Parameter %>% str_replace_all(.,"\\\\b\\_","\\b\\_")

colnames(dscov.pretex.table.final) <- str_replace_all(colnames(dscov.pretex.table.final),"\\_","\\\\\\_")
rownames(dscov.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & Window 1 & Window 2 & Window 1 & Window 2 \\\\\n")

dscov.tex.table.final_3sfTake2 <- xtable(dscov.pretex.table.final,align = c("cc|cccc"))
save(dscov.tex.table.final_3sfTake2,file="RegularPolygonSimStudyFullDSCovTableTex_3sfTake2.rda")

print(dscov.tex.table.final_3sfTake2, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'RegularPolygonSimStudyFullDSCovTableTex_3sfTake2.tex')


dscov.chains.full <- iwm_mcmc_chains(iwmbma_mh = dscovmh.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),group = TRUE,ncol.group = 2,print=TRUE)
ggsave(plot = dscov.chains.full$all.plots$beta1,file="RegularPolygonSimStudyFullDSCovBeta1Chain_Full.pdf",height = 7.5, width = 10)
ggsave(plot = dscov.chains.full$all.plots$beta2,file="RegularPolygonSimStudyFullDSCovBeta2Chain_Full.pdf",height = 7.5, width = 10)
ggsave(plot = dscov.chains.full$grid.plots,file="RegularPolygonSimStudyFullDSCovChains_Full.pdf",height = 7.5, width = 20)
dscov.chains.burnin <- iwm_mcmc_chains(iwmbma_mh = dscovbma.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),group = TRUE,ncol.group = 2,print=TRUE)
ggsave(plot = dscov.chains.burnin$all.plots$beta1,file="RegularPolygonSimStudyFullDSCovBeta1Chain_Burnin.pdf",height = 7.5, width = 10)
ggsave(plot = dscov.chains.burnin$all.plots$beta2,file="RegularPolygonSimStudyFullDSCovBeta2Chain_Burnin.pdf",height = 7.5, width = 10)
ggsave(plot = dscov.chains.burnin$grid.plots,file="RegularPolygonSimStudyFullDSCovChains_Burnin.pdf",height = 7.5, width = 20)

dscov.scatterdens.full <- iwm_mcmc_scatterdens(iwmbma_mh = dscovmh.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),print=TRUE)
ggsave(plot = dscov.scatterdens.full$scatter$beta1_beta2,file="RegularPolygonSimStudyFullDSCovBeta1Beta2Scatter_Full.pdf",height = 7.5, width = 7.5)
ggsave(plot = dscov.scatterdens.full$density$beta1_beta2,file="RegularPolygonSimStudyFullDSCovBeta1Beta2Density_Full.pdf",height = 7.5, width = 10)
dscov.scatterdens.burnin <- iwm_mcmc_scatterdens(iwmbma_mh = dscovbma.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),print=TRUE)
ggsave(plot = dscov.scatterdens.burnin$scatter$beta1_beta2,file="RegularPolygonSimStudyFullDSCovBeta1Beta2Scatter_Burnin.pdf",height = 7.5, width = 7.5)
ggsave(plot = dscov.scatterdens.burnin$density$beta1_beta2,file="RegularPolygonSimStudyFullDSCovBeta1Beta2Density_Burnin.pdf",height = 7.5, width = 10)

dscov.postdens.mcmc <- iwm_mcmc_post(iwmbma_mh = dscovbma.out$run$theta,post.lab="Multivar IwM MH",which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),hist.bins=50,prior_fun=list(beta_prior,beta_prior),true.value=list(2,c(-2,2)),true.lab=c("True Window 1","True Window 2"),inla.out = list("Window 1"=full1.fit,"Window 2"=dscov.fit),inla.param = c("fixed","fixed"),inla.param.ind = c(2,3),inla.lab = c("Window 1 INLA","Window 2 INLA"),iwmbma_bma = dscovbma.out$inla.bma,which.window = 2,which.param_combo=c(1,2),combo.param = c("fixed","fixed"),combo.param.ind = c(1,2),combo.param.lab = c("beta[1] + b[1]","beta[2] + b[2]"),col.vec=col.vec.mcmc.cov,print=TRUE)

ggsave(plot = dscov.postdens.mcmc$postdens[[1]],file="RegularPolygonSimStudyFullDSCovBeta1PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = dscov.postdens.mcmc$postdens.cov[[1]],file="RegularPolygonSimStudyFullDSCovBeta1PostDenswComb.pdf",height = 7.5, width = 10) 
ggsave(plot = dscov.postdens.mcmc$postdens[[2]],file="RegularPolygonSimStudyFullDSCovBeta2PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = dscov.postdens.mcmc$postdens.cov[[2]],file="RegularPolygonSimStudyFullDSCovBeta2PostDenswComb.pdf",height = 7.5, width = 10)

dscov.postdens.bma.w1 <- iwm_bma_post(iwmbma_bma = dscovbma.out$inla.bma[[1]],post.lab="Multivar IwM BMA",which.param=c("fixed","hyperpar","hyperpar"),param.ind = c(1,2,1),param.lab = c("beta[0]","sigma","rho"),prior_fun = list(beta_prior,pc_den_sig,pc_den_rho),true.value = list(2,2,0.75),true.lab=c("True Value"),inla.out = list("Window 1"=full1.fit),inla.param = c("fixed","hyperpar","hyperpar"),inla.param.ind = c(1,2,1),inla.lab = c("Window 1 INLA"),col.vec=col.vec.bma.w1,print=TRUE)
ggsave(plot = dscov.postdens.bma.w1$postdens[[1]],file="RegularPolygonSimStudyFullDSCovW1Beta0PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = dscov.postdens.bma.w1$postdens[[2]],file="RegularPolygonSimStudyFullDSCovW1SigmaPostDens.pdf",height = 7.5, width = 10)
ggsave(plot = dscov.postdens.bma.w1$postdens[[3]],file="RegularPolygonSimStudyFullDSCovW1RhoPostDens.pdf",height = 7.5, width = 10) 

mylegend <- ggpubr::get_legend(dscov.postdens.bma.w1$postdens[[1]] + theme(legend.title = element_text(size = 50),legend.text = element_text(size=45)) + guides(colour = guide_legend(override.aes = list(size=5))))

beta0.scaled <- dscov.postdens.bma.w1$postdens[[1]] + theme(legend.position="none")
sigma.scaled <- dscov.postdens.bma.w1$postdens[[2]] + theme(legend.position="none")
rho.scaled <- dscov.postdens.bma.w1$postdens[[3]] + theme(legend.position="none")

dscovw1bma.postdens.grid <- grid.arrange(beta0.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=dscovw1bma.postdens.grid,filename = "RegularPolygonSimStudyDSCovW1BMAPostDensAll.pdf",height = 20, width = 25)

dscov.postdens.bma.w2 <- iwm_bma_post(iwmbma_bma = dscovbma.out$inla.bma[[2]],post.lab="Multivar IwM BMA",which.param=c("fixed","hyperpar","hyperpar"),param.ind = c(1,2,1),param.lab = c("beta[0]","sigma","rho"),prior_fun = list(beta_prior,pc_den_sig,pc_den_rho),true.value = list(2,2,0.75),true.lab=c("True Value"),inla.out = list("Window 2"=dscov.fit),inla.param = c("fixed","hyperpar","hyperpar"),inla.param.ind = c(1,2,1),inla.lab = c("Window 2 INLA"),col.vec=col.vec.bma.w2,print=TRUE)
ggsave(plot = dscov.postdens.bma.w2$postdens[[1]],file="RegularPolygonSimStudyFullDSCovW2Beta0PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = dscov.postdens.bma.w2$postdens[[2]],file="RegularPolygonSimStudyFullDSCovW2SigmaPostDens.pdf",height = 7.5, width = 10)
ggsave(plot = dscov.postdens.bma.w2$postdens[[3]],file="RegularPolygonSimStudyFullDSCovW2RhoPostDens.pdf",height = 7.5, width = 10)

mylegend <- ggpubr::get_legend(dscov.postdens.bma.w2$postdens[[1]] + theme(legend.title = element_text(size = 50),legend.text = element_text(size=45)) + guides(colour = guide_legend(override.aes = list(size=5))))

beta0.scaled <- dscov.postdens.bma.w2$postdens[[1]] + theme(legend.position="none")
sigma.scaled <- dscov.postdens.bma.w2$postdens[[2]] + theme(legend.position="none")
rho.scaled <- dscov.postdens.bma.w2$postdens[[3]] + theme(legend.position="none")

dscovw2bma.postdens.grid <- grid.arrange(beta0.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=dscovw2bma.postdens.grid,filename = "RegularPolygonSimStudyDSCovW2BMAPostDensAll.pdf",height = 20, width = 25)


mylegend <- ggpubr::get_legend(dscov.postdens.mcmc$postdens.cov[[2]] + theme(legend.title = element_text(size = 45),legend.text = element_text(size=40)) + guides(colour = guide_legend(override.aes = list(size=5))))


fullbeta2.scaled <- full.postdens.mcmc$postdens[[2]] + ggtitle(parse(text="Posterior~Distribution~'for'~beta[2]~'for'~Scenario~1")) + theme(legend.position="none")
sparsebeta2.scaled <- sparse.postdens.mcmc$postdens[[2]] + ggtitle(parse(text="Posterior~Distribution~'for'~beta[2]~'for'~Scenario~2")) + theme(legend.position="none")
covbeta2.scaled <- cov.postdens.mcmc$postdens.cov[[2]] + ggtitle(parse(text="Posterior~Distribution~'for'~beta[2]~'for'~Scenario~3")) + theme(legend.position="none")
dscovbeta2.scaled <- dscov.postdens.mcmc$postdens.cov[[2]] + ggtitle(parse(text="Posterior~Distribution~'for'~beta[2]~'for'~Scenario~4")) + theme(legend.position="none")

beta2.postdens.grid <- grid.arrange(arrangeGrob(fullbeta2.scaled,sparsebeta2.scaled,covbeta2.scaled,dscovbeta2.scaled,ncol=2),
                                    mylegend,ncol=2,widths=c(10,2.5))

ggsave(plot=beta2.postdens.grid,filename = "RegularPolygonSimStudyBeta2PostDensAll.pdf",height = 20, width = 25)

# US Cities ---------------------------------------------------------------

setwd(dirname(getActiveDocumentContext()$path))
# Then as we want to store the results in the IwM_OUTPUTS directory, also run
setwd("./IwM_OUTPUTS/")
source("../IwMSummary_Functions.R")

setwd("./USCITIES")

load("../../../USCITIESIwM/LAMVTINLA.rda")
load("../../../USCITIESIwM/NYCMVTINLA.rda")


pc_denrho <- function(rho,alpha_rho,rho_0){
  lambda_1 <- -log(alpha_rho)*rho_0;
  pi = lambda_1*rho^(-2)*exp(-lambda_1*rho^(-1))
}

pc_densig <- function(sigma,alpha_sigma,sigma_0){
  lambda_2 <- -log(alpha_sigma)/sigma_0
  pi = lambda_2*exp(-lambda_2*sigma)
}

gauss_prior_inla <- function(x,mean,sd){
  pi = dnorm(x,mean=mean,sd=sd)
}

beta_prior <- function(x){
  pi = gauss_prior_inla(x,mean=0,sd=sqrt(1/0.01))
}

b_prior <- function(x){
  pi = gauss_prior_inla(x,mean=0,sd=sqrt(4))
}

rho.star <- c(0.4,0.1); sigma.star <- c(2,0.1)
pc_den_rho <- function(x){pc_denrho(x,rho_0=rho.star[1],alpha_rho=rho.star[2])}
pc_den_sig <- function(x){pc_densig(x,sigma_0=sigma.star[1],alpha_sigma=sigma.star[2])}

col.vec.mcmc <- c("grey","magenta","red","blue","darkgreen","darkorange") # prior, window 1, window 2, window 1 par 1 combo, window 2 par 2 combo
col.vec.bma <- c("grey","magenta","red") # post, prior, INLA
col.vec.bma.w1 <- c("grey","magenta","red") # post, prior, INLA w1 - match mcmc INLA
col.vec.bma.w2 <- c("grey","magenta","blue") # post, prior, INLA w2 - match mcmc INLA

# Univariate LA City ------------------------------------------------------

load("../../../USCITIESIwM/MultiCityUniIwM_MH_11000.rda")
mh.out <- out
load("../../../USCITIESIwM/MultiCityUniIwM_BMA_11000_2000.rda")
iwmbma.out <- out
inla.out <- list(la.gta.fit)

univar.tab <- tab.summ(iwmbma.out=iwmbma.out,bma.param=c("fixed","hyperpar","hyperpar"),bma.ind=c(1,2,1),param.mh=param.lab.df[c(2,3)],param.bma=param.lab.df[c(1,4,5)],multi=FALSE,windows.l=1,w.lab=c("LA"),cov=FALSE,inla.out=inla.out,mh.inla.param=c("fixed","fixed"),mh.inla.ind=c(2,3),bma.inla.ind=c(1,2,1))
save(univar.tab,file="UnivarIwMLATableDF.rda")

univar.table.df <- rbind(univar.tab$mcmc.table[1:5],univar.tab$bma.table)
univar.table.df$Mean <- format(univar.table.df$Mean,digits=3)
univar.table.df$SD <- format(univar.table.df$SD,digits=3)
univar.table.df <- unite(univar.table.df,"Mean (SD)",Mean:SD,sep=" (")
univar.table.df$`Mean (SD)` <- paste0(univar.table.df$`Mean (SD)`,")")
univar.compact.table.df <- univar.table.df %>% spread(Parameter,`Mean (SD)`)
univar.tex.table <- univar.compact.table.df %>% 
  pivot_longer(`beta[0]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

univar.tex.table  <- univar.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

univar.tex.table$`INLAwMCMC-MH_LA`[!is.na(univar.tex.table$`INLAwMCMC-MH_LA`)] <- paste0(univar.tex.table$`INLAwMCMC-MH_LA`[!is.na(univar.tex.table$`INLAwMCMC-MH_LA`)],"$^\\dagger$")
univar.pretex.table.final <- unite(univar.tex.table,"INLAwMCMC_LA",c(`INLAwMCMC-BMA_LA`,`INLAwMCMC-MH_LA`),sep="",remove = TRUE,na.rm=TRUE)

univar.pretex.table.final$Parameter <- univar.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")

colnames(univar.pretex.table.final) <- str_replace_all(colnames(univar.pretex.table.final),"\\_","\\\\\\_")
rownames(univar.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{1}{c}{INLA} & \\multicolumn{1}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & LA & LA \\\\\n")

univar.tex.table.final <- xtable(univar.pretex.table.final,align = c("cc|cc"))
save(univar.tex.table.final,file="UnivarIwMLATableTex.rda")

print(univar.tex.table.final, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'UnivarIwMLATableTex.tex')

# Change significant figures
load("UnivarIwMLATableDF.rda")
univar.table.df_3sf <- rbind(univar.tab$mcmc.table[1:5],univar.tab$bma.table)
univar.table.df_3sf$Mean <- signif(univar.table.df_3sf$Mean,digits=3)
univar.table.df_3sf$SD <- signif(univar.table.df_3sf$SD,digits=3)
univar.table.df_3sf <- unite(univar.table.df_3sf,"Mean (SD)",Mean:SD,sep=" (")
univar.table.df_3sf$`Mean (SD)` <- paste0(univar.table.df_3sf$`Mean (SD)`,")")
univar.compact.table.df <- univar.table.df_3sf %>% spread(Parameter,`Mean (SD)`)
univar.tex.table <- univar.compact.table.df %>% 
  pivot_longer(`beta[0]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

univar.tex.table  <- univar.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

univar.tex.table$`INLAwMCMC-MH_LA`[!is.na(univar.tex.table$`INLAwMCMC-MH_LA`)] <- paste0(univar.tex.table$`INLAwMCMC-MH_LA`[!is.na(univar.tex.table$`INLAwMCMC-MH_LA`)],"$^\\dagger$")
univar.pretex.table.final <- unite(univar.tex.table,"INLAwMCMC_LA",c(`INLAwMCMC-BMA_LA`,`INLAwMCMC-MH_LA`),sep="",remove = TRUE,na.rm=TRUE)

univar.pretex.table.final$Parameter <- univar.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")

colnames(univar.pretex.table.final) <- str_replace_all(colnames(univar.pretex.table.final),"\\_","\\\\\\_")
rownames(univar.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{1}{c}{INLA} & \\multicolumn{1}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & LA & LA \\\\\n")

univar.tex.table.final_3sf <- xtable(univar.pretex.table.final,align = c("cc|cc"))
save(univar.tex.table.final_3sf,file="UnivarIwMLATableTex_3sf.rda")

print(univar.tex.table.final_3sf, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'UnivarIwMLATableTex_3sf.tex')

# Smaller significant figures for tables - Take 2 for signif() potential rounding issues
load("UnivarIwMLATableDF.rda")
univar.table.df_3sfTake2 <- rbind(univar.tab$mcmc.table[1:5],univar.tab$bma.table)
univar.table.df_3sfTake2$Mean <- signif(univar.table.df_3sfTake2$Mean + .Machine$double.eps,digits=3)
univar.table.df_3sfTake2$SD <- signif(univar.table.df_3sfTake2$SD + .Machine$double.eps,digits=3)
univar.table.df_3sfTake2 <- unite(univar.table.df_3sfTake2,"Mean (SD)",Mean:SD,sep=" (")
univar.table.df_3sfTake2$`Mean (SD)` <- paste0(univar.table.df_3sfTake2$`Mean (SD)`,")")
univar.compact.table.df <- univar.table.df_3sfTake2 %>% spread(Parameter,`Mean (SD)`)
univar.tex.table <- univar.compact.table.df %>% 
  pivot_longer(`beta[0]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

univar.tex.table  <- univar.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

univar.tex.table$`INLAwMCMC-MH_LA`[!is.na(univar.tex.table$`INLAwMCMC-MH_LA`)] <- paste0(univar.tex.table$`INLAwMCMC-MH_LA`[!is.na(univar.tex.table$`INLAwMCMC-MH_LA`)],"$^\\dagger$")
univar.pretex.table.final <- unite(univar.tex.table,"INLAwMCMC_LA",c(`INLAwMCMC-BMA_LA`,`INLAwMCMC-MH_LA`),sep="",remove = TRUE,na.rm=TRUE)

univar.pretex.table.final$Parameter <- univar.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")

colnames(univar.pretex.table.final) <- str_replace_all(colnames(univar.pretex.table.final),"\\_","\\\\\\_")
rownames(univar.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{1}{c}{INLA} & \\multicolumn{1}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & LA & LA \\\\\n")

univar.tex.table.final_3sfTake2 <- xtable(univar.pretex.table.final,align = c("cc|cc"))
save(univar.tex.table.final_3sfTake2,file="UnivarIwMLATableTex_3sfTake2.rda")

print(univar.tex.table.final_3sfTake2, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'UnivarIwMLATableTex_3sfTake2.tex')



univar.chains.full <- iwm_mcmc_chains(iwmbma_mh = mh.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),group = TRUE,ncol.group = 2,print=TRUE)
ggsave(plot = univar.chains.full$all.plots$beta1,file="UnivarIwMLABeta1Chain_Full.pdf",height = 7.5, width = 10)
ggsave(plot = univar.chains.full$all.plots$beta2,file="UnivarIwMLABeta2Chain_Full.pdf",height = 7.5, width = 10)
ggsave(plot = univar.chains.full$grid.plots,file="UnivarIwMLACovChains_Full.pdf",height = 7.5, width = 20)
univar.chains.burnin <- iwm_mcmc_chains(iwmbma_mh = iwmbma.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),group = TRUE,ncol.group = 2,print=TRUE)
ggsave(plot = univar.chains.burnin$all.plots$beta1,file="UnivarIwMLABeta1Chain_Burnin.pdf",height = 7.5, width = 10)
ggsave(plot = univar.chains.burnin$all.plots$beta2,file="UnivarIwMLABeta2Chain_Burnin.pdf",height = 7.5, width = 10)
ggsave(plot = univar.chains.burnin$grid.plots,file="UnivarIwMLAChains_Burnin.pdf",height = 7.5, width = 20)

univar.scatterdens.full <- iwm_mcmc_scatterdens(iwmbma_mh = mh.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),print=TRUE)
ggsave(plot = univar.scatterdens.full$scatter$beta1_beta2,file="UnivarIwMLABeta1Beta2Scatter_Full.pdf",height = 7.5, width = 7.5)
ggsave(plot = univar.scatterdens.full$density$beta1_beta2,file="UnivarIwMLABeta1Beta2Density_Full.pdf",height = 7.5, width = 10)
univar.scatterdens.burnin <- iwm_mcmc_scatterdens(iwmbma_mh = iwmbma.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),print=TRUE)
ggsave(plot = univar.scatterdens.burnin$scatter$beta1_beta2,file="UnivarIwMLABeta1Beta2Scatter_Burnin.pdf",height = 7.5, width = 7.5)
ggsave(plot = univar.scatterdens.burnin$density$beta1_beta2,file="UnivarIwMLABeta1Beta2Density_Burnin.pdf",height = 7.5, width = 10)

univar.postdens.mcmc <- iwm_mcmc_post(iwmbma_mh = iwmbma.out$run$theta,post.lab="Univar IwM MH",which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),prior_fun=list(beta_prior,beta_prior),true.value=NULL,inla.out = list("LA"=la.gta.fit),inla.param = c("fixed","fixed"),inla.param.ind = c(2,3),inla.lab = "LA INLA",col.vec=col.vec.mcmc,print=TRUE)

ggsave(plot = univar.postdens.mcmc$postdens[[1]],file="UnivarIwMLABeta1PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = univar.postdens.mcmc$postdens[[2]],file="UnivarIwMLABeta2PostDens.pdf",height = 7.5, width = 10)

univar.postdens.bma <- iwm_bma_post(iwmbma_bma = iwmbma.out$inla.bma,post.lab="Univar IwM BMA",which.param=c("fixed","hyperpar","hyperpar"),param.ind = c(1,2,1),param.lab = c("beta[0]","sigma","rho"),prior_fun = list(beta_prior,pc_den_sig,pc_den_rho),true.value = NULL,inla.out = list("LA"=la.gta.fit),inla.param = c("fixed","hyperpar","hyperpar"),inla.param.ind = c(1,2,1),inla.lab = "LA INLA",col.vec=col.vec.bma,print=TRUE)
ggsave(plot = univar.postdens.bma$postdens[[1]],file="UnivarIwMLABeta0PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = univar.postdens.bma$postdens[[2]],file="UnivarIwMLASigmaPostDens.pdf",height = 7.5, width = 10)
ggsave(plot = univar.postdens.bma$postdens[[3]],file="UnivarIwMLARhoPostDens.pdf",height = 7.5, width = 10)


mylegend <- ggpubr::get_legend(univar.postdens.bma$postdens[[1]] + theme(legend.title = element_text(size = 50),legend.text = element_text(size=45)) + guides(colour = guide_legend(override.aes = list(size=5))))

beta0.scaled <- univar.postdens.bma$postdens[[1]] + theme(legend.position="none")
sigma.scaled <- univar.postdens.bma$postdens[[2]] + theme(legend.position="none")
rho.scaled <- univar.postdens.bma$postdens[[3]] + theme(legend.position="none")

labma.postdens.grid <- grid.arrange(beta0.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=labma.postdens.grid,filename = "UnivarIwMLABMAPostDensAll.pdf",height = 20, width = 25)


beta1.scaled <- univar.postdens.mcmc$postdens[[1]] + theme(legend.position="none")
beta2.scaled <- univar.postdens.mcmc$postdens[[2]] + theme(legend.position="none")
laall.postdens.grid <- grid.arrange(beta1.scaled,beta2.scaled,beta0.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=laall.postdens.grid,filename = "UnivarIwMLAPostDensAll.pdf",height = 30, width = 25)

# Credible Intervals
univar.cred.df <- cred.df.iwm(iwmbma.out=iwmbma.out,param.lab.mh=c("beta[1]","beta[2]"),which.bma=c("fixed","hyperpar","hyperpar"),bma.ind=c(1,2,1),param.lab.bma=c("beta[0]","sigma","rho"),int.q=c(0.95,0.99),multivar=FALSE,window.total=1,window.lab="LA")


univar.cred.int <- iwm_mcmc_cred(univar.cred.df,quant.lev=paste0(c("`0.005","`0.025","`0.975","`0.995"),"quant`"),title="95% and 99% Credible Intervals for\n Posterior Distributions from Univariate IwM\n for LA Motor Vehicle Theft")
ggsave(plot = univar.cred.int,filename = "UnivarIwMLACredInt.pdf",height = 7.5,width = 10)

# Multivariate LA and NY City ---------------------------------------------

load("../../../USCITIESIwM/MultiCityMultiIwM_MH_11000.rda")
l <- sum(!is.na(out$run$theta$beta1))
mh.out <- out
mh.out$run$theta <- mh.out$run$theta[1:l,]
load("../../../USCITIESIwM/MultiCityMultiIwM_BMA_11000_2000.rda")
iwmbma.out <- out
inla.out <- list("LA"=la.gta.fit,"NY"=ny.gta.fit)

multivar.tab <- tab.summ(iwmbma.out=iwmbma.out,bma.param=c("fixed","hyperpar","hyperpar"),bma.ind=c(1,2,1),param.mh=param.lab.df[c(2,3)],param.bma=param.lab.df[c(1,4,5)],multi=TRUE,windows.l=2,w.lab=c("LA","NY"),cov=TRUE,cov.param=c("fixed","fixed"),cov.ind = c(2,3),comb.ind=c(1,2),param.cov=c("b[1]","b[2]"),inla.out=inla.out,mh.inla.param=c("fixed","fixed"),mh.inla.ind=c(2,3),bma.inla.ind=c(1,2,1))
save(multivar.tab,file="MultivarIwMLANYTableDF.rda")

multivar.table.df <- rbind(multivar.tab$mcmc.table[1:5],multivar.tab$bma.table)
multivar.table.df$Mean <- format(multivar.table.df$Mean,digits=3)
multivar.table.df$SD <- format(multivar.table.df$SD,digits=3)
multivar.table.df <- unite(multivar.table.df,"Mean (SD)",Mean:SD,sep=" (")
multivar.table.df$`Mean (SD)` <- paste0(multivar.table.df$`Mean (SD)`,")")
multivar.compact.table.df <- multivar.table.df %>% spread(Parameter,`Mean (SD)`)
multivar.tex.table <- multivar.compact.table.df %>% 
  pivot_longer(`b[1]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

multivar.tex.table  <- multivar.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

multivar.tex.table$`INLAwMCMC-MH_Combined`[!is.na(multivar.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(multivar.tex.table$`INLAwMCMC-MH_Combined`[!is.na(multivar.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
multivar.tex.table$`INLAwMCMC_NY`[!is.na(multivar.tex.table$`INLAwMCMC_NY`)] <- paste0(multivar.tex.table$`INLAwMCMC_NY`[!is.na(multivar.tex.table$`INLAwMCMC_NY`)],"$^\\star$")
multivar.tex.table <- unite(multivar.tex.table,"IwM_LA",c(`INLAwMCMC-BMA_LA`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
multivar.tex.table <- unite(multivar.tex.table,"IwM_NY",c(`INLAwMCMC_NY`,`INLAwMCMC-BMA_NY`,`INLAwMCMC-MH_Combined`),sep="",remove = TRUE,na.rm=TRUE)
multivar.pretex.table.final <- multivar.tex.table %>% select(.,-`INLAwMCMC-BMA_LA`)
multivar.pretex.table.final <- multivar.pretex.table.final[,c(1:3,5,4)]

multivar.pretex.table.final$Parameter <- multivar.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
multivar.pretex.table.final$Parameter <- multivar.pretex.table.final$Parameter %>% str_replace_all(.,"\\\\b\\_","\\b\\_")

colnames(multivar.pretex.table.final) <- str_replace_all(colnames(multivar.pretex.table.final),"\\_","\\\\\\_")
rownames(multivar.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & LA & NY & LA & NY \\\\\n")

multivar.tex.table.final <- xtable(multivar.pretex.table.final,align = c("cc|cccc"))
save(multivar.tex.table.final,file="MultivarIwMLANYTableTex.rda")

print(multivar.tex.table.final, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'MultivarIwMLANYTableTex.tex')

# Change significant figures
load("MultivarIwMLANYTableDF.rda")
multivar.table.df_3sf <- rbind(multivar.tab$mcmc.table[1:5],multivar.tab$bma.table)
multivar.table.df_3sf$Mean <- signif(multivar.table.df_3sf$Mean,digits=3)
multivar.table.df_3sf$SD <- signif(multivar.table.df_3sf$SD,digits=3)
multivar.table.df_3sf <- unite(multivar.table.df_3sf,"Mean (SD)",Mean:SD,sep=" (")
multivar.table.df_3sf$`Mean (SD)` <- paste0(multivar.table.df_3sf$`Mean (SD)`,")")
multivar.compact.table.df <- multivar.table.df_3sf %>% spread(Parameter,`Mean (SD)`)
multivar.tex.table <- multivar.compact.table.df %>% 
  pivot_longer(`b[1]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

multivar.tex.table  <- multivar.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

multivar.tex.table$`INLAwMCMC-MH_Combined`[!is.na(multivar.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(multivar.tex.table$`INLAwMCMC-MH_Combined`[!is.na(multivar.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
multivar.tex.table$`INLAwMCMC_NY`[!is.na(multivar.tex.table$`INLAwMCMC_NY`)] <- paste0(multivar.tex.table$`INLAwMCMC_NY`[!is.na(multivar.tex.table$`INLAwMCMC_NY`)],"$^\\star$")
multivar.tex.table <- unite(multivar.tex.table,"IwM_LA",c(`INLAwMCMC-BMA_LA`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
multivar.tex.table <- unite(multivar.tex.table,"IwM_NY",c(`INLAwMCMC_NY`,`INLAwMCMC-BMA_NY`,`INLAwMCMC-MH_Combined`),sep="",remove = TRUE,na.rm=TRUE)
multivar.pretex.table.final <- multivar.tex.table %>% select(.,-`INLAwMCMC-BMA_LA`)
multivar.pretex.table.final <- multivar.pretex.table.final[,c(1:3,5,4)]

multivar.pretex.table.final$Parameter <- multivar.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
multivar.pretex.table.final$Parameter <- multivar.pretex.table.final$Parameter %>% str_replace_all(.,"\\\\b\\_","\\b\\_")

colnames(multivar.pretex.table.final) <- str_replace_all(colnames(multivar.pretex.table.final),"\\_","\\\\\\_")
rownames(multivar.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & LA & NY & LA & NY \\\\\n")

multivar.tex.table.final_3sf <- xtable(multivar.pretex.table.final,align = c("cc|cccc"))
save(multivar.tex.table.final_3sf,file="MultivarIwMLANYTableTex_3sf.rda")

print(multivar.tex.table.final_3sf, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'MultivarIwMLANYTableTex_3sf.tex')

# Smaller significant figures for tables - Take 2 for signif() potential rounding issues
load("MultivarIwMLANYTableDF.rda")
multivar.table.df_3sfTake2 <- rbind(multivar.tab$mcmc.table[1:5],multivar.tab$bma.table)
multivar.table.df_3sfTake2$Mean <- signif(multivar.table.df_3sfTake2$Mean + .Machine$double.eps,digits=3)
multivar.table.df_3sfTake2$SD <- signif(multivar.table.df_3sfTake2$SD + .Machine$double.eps,digits=3)
multivar.table.df_3sfTake2 <- unite(multivar.table.df_3sfTake2,"Mean (SD)",Mean:SD,sep=" (")
multivar.table.df_3sfTake2$`Mean (SD)` <- paste0(multivar.table.df_3sfTake2$`Mean (SD)`,")")
multivar.compact.table.df <- multivar.table.df_3sfTake2 %>% spread(Parameter,`Mean (SD)`)
multivar.tex.table <- multivar.compact.table.df %>% 
  pivot_longer(`b[1]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

multivar.tex.table  <- multivar.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

multivar.tex.table$`INLAwMCMC-MH_Combined`[!is.na(multivar.tex.table$`INLAwMCMC-MH_Combined`)] <- paste0(multivar.tex.table$`INLAwMCMC-MH_Combined`[!is.na(multivar.tex.table$`INLAwMCMC-MH_Combined`)],"$^\\dagger$")
multivar.tex.table$`INLAwMCMC_NY`[!is.na(multivar.tex.table$`INLAwMCMC_NY`)] <- paste0(multivar.tex.table$`INLAwMCMC_NY`[!is.na(multivar.tex.table$`INLAwMCMC_NY`)],"$^\\star$")
multivar.tex.table <- unite(multivar.tex.table,"IwM_LA",c(`INLAwMCMC-BMA_LA`,`INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
multivar.tex.table <- unite(multivar.tex.table,"IwM_NY",c(`INLAwMCMC_NY`,`INLAwMCMC-BMA_NY`,`INLAwMCMC-MH_Combined`),sep="",remove = TRUE,na.rm=TRUE)
multivar.pretex.table.final <- multivar.tex.table %>% select(.,-`INLAwMCMC-BMA_LA`)
multivar.pretex.table.final <- multivar.pretex.table.final[,c(1:3,5,4)]

multivar.pretex.table.final$Parameter <- multivar.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
multivar.pretex.table.final$Parameter <- multivar.pretex.table.final$Parameter %>% str_replace_all(.,"\\\\b\\_","\\b\\_")

colnames(multivar.pretex.table.final) <- str_replace_all(colnames(multivar.pretex.table.final),"\\_","\\\\\\_")
rownames(multivar.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{2}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & LA & NY & LA & NY \\\\\n")

multivar.tex.table.final_3sfTake2 <- xtable(multivar.pretex.table.final,align = c("cc|cccc"))
save(multivar.tex.table.final_3sfTake2,file="MultivarIwMLANYTableTex_3sfTake2.rda")

print(multivar.tex.table.final_3sfTake2, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'MultivarIwMLANYTableTex_3sfTake2.tex')

multivar.chains.full <- iwm_mcmc_chains(iwmbma_mh = mh.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),group = TRUE,ncol.group = 2,print=TRUE)
ggsave(plot = multivar.chains.full$all.plots$beta1,file="MultivarIwMLANYBeta1Chain_Full.pdf",height = 7.5, width = 10)
ggsave(plot = multivar.chains.full$all.plots$beta2,file="MultivarIwMLANYBeta2Chain_Full.pdf",height = 7.5, width = 10)
ggsave(plot = multivar.chains.full$grid.plots,file="MultivarIwMLANYChains_Full.pdf",height = 7.5, width = 20)
multivar.chains.burnin <- iwm_mcmc_chains(iwmbma_mh = iwmbma.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),group = TRUE,ncol.group = 2,print=TRUE)
ggsave(plot = multivar.chains.burnin$all.plots$beta1,file="MultivarIwMLANYBeta1Chain_Burnin.pdf",height = 7.5, width = 10)
ggsave(plot = multivar.chains.burnin$all.plots$beta2,file="MultivarIwMLANYBeta2Chain_Burnin.pdf",height = 7.5, width = 10)
ggsave(plot = multivar.chains.burnin$grid.plots,file="MultivarIwMLANYChains_Burnin.pdf",height = 7.5, width = 20)

multivar.scatterdens.full <- iwm_mcmc_scatterdens(iwmbma_mh = mh.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),print=TRUE)
ggsave(plot = multivar.scatterdens.full$scatter$beta1_beta2,file="MultivarIwMLANYBeta1Beta2Scatter_Full.pdf",height = 7.5, width = 7.5)
ggsave(plot = multivar.scatterdens.full$density$beta1_beta2,file="MultivarIwMLANYBeta1Beta2Density_Full.pdf",height = 7.5, width = 10)
multivar.scatterdens.burnin <- iwm_mcmc_scatterdens(iwmbma_mh = iwmbma.out$run$theta,which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),print=TRUE)
ggsave(plot = multivar.scatterdens.burnin$scatter$beta1_beta2,file="MultivarIwMLANYBeta1Beta2Scatter_Burnin.pdf",height = 7.5, width = 7.5)
ggsave(plot = multivar.scatterdens.burnin$density$beta1_beta2,file="MultivarIwMLANYBeta1Beta2Density_Burnin.pdf",height = 7.5, width = 10)

multivar.postdens.mcmc <- iwm_mcmc_post(iwmbma_mh = iwmbma.out$run$theta,post.lab="Multivar IwM MH",which.param = c(1,2),param.lab = c("beta[1]","beta[2]"),hist.bins=50,prior_fun=list(beta_prior,beta_prior),inla.out = inla.out,inla.param = c("fixed","fixed"),inla.param.ind = c(2,3),inla.lab = c("LA INLA","NY INLA"),iwmbma_bma = iwmbma.out$inla.bma,which.window = 2,which.param_combo=c(1,2),combo.param = c("fixed","fixed"),combo.param.ind = c(1,2),combo.param.lab = c("beta[1] + b[1]","beta[2] + b[2]"),col.vec=col.vec.mcmc,print=TRUE)

ggsave(plot = multivar.postdens.mcmc$postdens[[1]],file="MultivarIwMLANYBeta1PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = multivar.postdens.mcmc$postdens.cov[[1]],file="MultivarIwMLANYBeta1PostDenswComb.pdf",height = 7.5, width = 10)
ggsave(plot = multivar.postdens.mcmc$postdens[[2]],file="MultivarIwMLANYBeta2PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = multivar.postdens.mcmc$postdens.cov[[2]],file="MultivarIwMLANYBeta2PostDenswComb.pdf",height = 7.5, width = 10)

multivar.postdens.bma.w1 <- iwm_bma_post(iwmbma_bma = iwmbma.out$inla.bma[[1]],post.lab="LA Multivar IwM BMA",which.param=c("fixed","hyperpar","hyperpar"),param.ind = c(1,2,1),param.lab = c("beta[0]","sigma","rho"),prior_fun = list(beta_prior,pc_den_sig,pc_den_rho),inla.out = inla.out[1],inla.param = c("fixed","hyperpar","hyperpar"),inla.param.ind = c(1,2,1),inla.lab = c("LA INLA"),col.vec=col.vec.bma.w1,print=TRUE)
ggsave(plot = multivar.postdens.bma.w1$postdens[[1]],file="MultivarIwMLABeta0PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = multivar.postdens.bma.w1$postdens[[2]],file="MultivarIwMLASigmaPostDens.pdf",height = 7.5, width = 10)
ggsave(plot = multivar.postdens.bma.w1$postdens[[3]],file="MultivarIwMLARhoPostDens.pdf",height = 7.5, width = 10)



multivar.postdens.bma.w2 <- iwm_bma_post(iwmbma_bma = iwmbma.out$inla.bma[[2]],post.lab="NY Multivar IwM BMA",which.param=c("fixed","hyperpar","hyperpar"),param.ind = c(1,2,1),param.lab = c("beta[0]","sigma","rho"),prior_fun = list(beta_prior,pc_den_sig,pc_den_rho),inla.out = inla.out[2],inla.param = c("fixed","hyperpar","hyperpar"),inla.param.ind = c(1,2,1),inla.lab = c("NY INLA"),col.vec=col.vec.bma.w2,print=TRUE)
ggsave(plot = multivar.postdens.bma.w2$postdens[[1]],file="MultivarIwMNYBeta0PostDens.pdf",height = 7.5, width = 10)
ggsave(plot = multivar.postdens.bma.w2$postdens[[2]],file="MultivarIwMNYSigmaPostDens.pdf",height = 7.5, width = 10)
ggsave(plot = multivar.postdens.bma.w2$postdens[[3]],file="MultivarIwMNYRhoPostDens.pdf",height = 7.5, width = 10)


mylegend <- ggpubr::get_legend(multivar.postdens.bma.w1$postdens[[1]] + theme(legend.title = element_text(size = 50),legend.text = element_text(size=45)) + guides(colour = guide_legend(override.aes = list(size=10))))

beta0.scaled <- multivar.postdens.bma.w1$postdens[[1]] + theme(legend.position="none")
sigma.scaled <- multivar.postdens.bma.w1$postdens[[2]] + theme(legend.position="none")
rho.scaled <- multivar.postdens.bma.w1$postdens[[3]] + theme(legend.position="none")


labma.postdens.grid <- grid.arrange(beta0.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=labma.postdens.grid,filename = "MultivarIwMLABMAPostDensAll.pdf",height = 20, width = 25)


mylegend <- ggpubr::get_legend(multivar.postdens.bma.w2$postdens[[1]] + theme(legend.title = element_text(size = 50),legend.text = element_text(size=45)) + guides(colour = guide_legend(override.aes = list(size=10))))

beta0.scaled <- multivar.postdens.bma.w2$postdens[[1]] + theme(legend.position="none")
sigma.scaled <- multivar.postdens.bma.w2$postdens[[2]] + theme(legend.position="none")
rho.scaled <- multivar.postdens.bma.w2$postdens[[3]] + theme(legend.position="none")


nybma.postdens.grid <- grid.arrange(beta0.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=nybma.postdens.grid,filename = "MultivarIwMNYBMAPostDensAll.pdf",height = 20, width = 25)

# Want a plot for beta 1 and beta 2 combining the legends
leg <- get_legend(multivar.postdens.mcmc$postdens[[1]]) 

beta1.scaled <- multivar.postdens.mcmc$postdens[[1]] + theme(legend.position="none")
beta2.scaled <- multivar.postdens.mcmc$postdens[[2]] + theme(legend.position="none")
nymcmc.postdens.grid <- grid.arrange(beta1.scaled,beta2.scaled,leg,ncol=3,widths=c(15.5,15,8))
ggsave(plot=nymcmc.postdens.grid,filename = "MultivarIwMNYMCMCPostDensAll.pdf",height = 7.5, width = 15)

# Want a plot for beta 1 and beta 2 with combined beta and b posteriors so need to combined the slightly different legends
leg <- get_legend(multivar.postdens.mcmc$postdens.cov[[1]])
leg2 <- get_legend(multivar.postdens.mcmc$postdens.cov[[2]])
leg.comb <- rbind(leg$grobs[[1]],leg2$grobs[[1]])
# Now want to extract the parts of the legend that belong to the labels we want, not doubling up the prior or INLA distributions.
leg.comb.sub <- leg.comb[c(2,4:8,17)] # alter again for RemoteDesktop implementation

beta1.cov.scaled <- multivar.postdens.mcmc$postdens.cov[[1]] + theme(legend.position="none")
beta2.cov.scaled <- multivar.postdens.mcmc$postdens.cov[[2]] + theme(legend.position="none")
nymcmc.postdens.grid <- grid.arrange(beta1.cov.scaled,beta2.cov.scaled,leg.comb.sub,ncol=3,widths=c(15.5,15,8))

ggsave(plot=nymcmc.postdens.grid,filename = "MultivarIwMNYMCMCPostDensCovAll.pdf",height = 7.5, width = 15)

# 19/10/2021 - run marginal calculations
inla.pmarginal(q=0,marginal=iwmbma.out$inla.bma$NY_City$marginals.fixed$cov1)
inla.zmarginal(marginal=iwmbma.out$inla.bma$NY_City$marginals.fixed$cov1)
# > inla.pmarginal(q=0,marginal=iwmbma.out$inla.bma$NY_City$marginals.fixed$cov1)
# [1] 1
# > inla.zmarginal(marginal=iwmbma.out$inla.bma$NY_City$marginals.fixed$cov1)
# Mean            -0.94356 
# Stdev           0.136191 
# Quantile  0.025 -1.21673 
# Quantile  0.25  -1.03528 
# Quantile  0.5   -0.942873 
# Quantile  0.75  -0.851683 
# Quantile  0.975 -0.674698 

inla.pmarginal(q=0,marginal=iwmbma.out$inla.bma$NY_City$marginals.fixed$cov2)
inla.zmarginal(marginal=iwmbma.out$inla.bma$NY_City$marginals.fixed$cov2)
# > inla.pmarginal(q=0,marginal=iwmbma.out$inla.bma$NY_City$marginals.fixed$cov2)
# [1] 0.005281876
# > inla.zmarginal(marginal=iwmbma.out$inla.bma$NY_City$marginals.fixed$cov2)
# Mean            0.430892 
# Stdev           0.168718 
# Quantile  0.025 0.0996847 
# Quantile  0.25  0.316369 
# Quantile  0.5   0.430344 
# Quantile  0.75  0.544229 
# Quantile  0.975 0.761045 

multivar.cred.df <- cred.df.iwm(iwmbma.out=iwmbma.out,param.lab.mh=c("beta[1]","beta[2]"),which.bma=c("fixed","hyperpar","hyperpar"),bma.ind=c(1,2,1),param.lab.bma=c("beta[0]","sigma","rho"),int.q=c(0.95,0.99),multivar=TRUE,window.total=2,window.lab=c("LA","NY"),which.w=2,cov=TRUE,which.cov=c("fixed","fixed"),cov.ind=c(2,3),param.lab.cov=c("b[1]","b[2]"),comb=TRUE,which.comb=c("fixed","fixed"),comb.ind=c(1,2),param.lab.comb=c("beta[1] + b[1]","beta[2] + b[2]"))


multivar.cred.int <- iwm_mcmc_cred(multivar.cred.df,quant.lev=paste0(c("`0.005","`0.025","`0.975","`0.995"),"quant`"),title="95% and 99% Credible Intervals for\n Posterior Distributions from Multivariate \nIwM for LA and NYC Motor Vehicle Theft")
ggsave(plot = multivar.cred.int,filename = "MultivarIwMLANYCredInt.pdf",height = 7.5,width = 10)

multivar.cred.int.seg  <- multivar.cred.int + geom_hline(yintercept=c(2.5,4.5,6.5,9.5),size=1.5)
ggsave(plot = multivar.cred.int.seg,filename = "MultivarIwMLANYCredIntSegments.pdf",height = 7.5,width = 10)


# Combined Tables: Univar and Multivar
city.table.uni <- univar.table.df[univar.table.df$Algorithm!="INLA",]
city.table.uni$Algorithm <- paste0("Univariate-",city.table.uni$Algorithm)
city.table.multi <- multivar.table.df
city.table.multi$Algorithm[city.table.multi$Algorithm!="INLA"] <- paste0("Multivariate-",city.table.multi$Algorithm[city.table.multi$Algorithm!="INLA"])
city.table.multi[city.table.multi$Algorithm=="INLA"&city.table.multi$Window=="NY"&city.table.multi$Parameter=="beta[1]",]$Parameter <- "beta[1]+b[1]"
city.table.multi[city.table.multi$Algorithm=="INLA"&city.table.multi$Window=="NY"&city.table.multi$Parameter=="beta[2]",]$Parameter <- "beta[2]+b[2]"


city.table.df <- rbind(city.table.uni,city.table.multi)
city.compact.table.df <- city.table.df %>% spread(Parameter,`Mean (SD)`)
city.tex.table <- city.compact.table.df %>% 
  pivot_longer(`b[1]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

city.tex.table  <- city.tex.table %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

city.tex.table$`Multivariate-INLAwMCMC-MH_Combined`[!is.na(city.tex.table$`Multivariate-INLAwMCMC-MH_Combined`)] <- paste0(city.tex.table$`Multivariate-INLAwMCMC-MH_Combined`[!is.na(city.tex.table$`Multivariate-INLAwMCMC-MH_Combined`)],"$^\\dagger$")
city.tex.table$`Univariate-INLAwMCMC-MH_LA`[!is.na(city.tex.table$`Univariate-INLAwMCMC-MH_LA`)] <- paste0(city.tex.table$`Univariate-INLAwMCMC-MH_LA`[!is.na(city.tex.table$`Univariate-INLAwMCMC-MH_LA`)],"$^\\dagger$")
city.tex.table$`Multivariate-INLAwMCMC_NY`[!is.na(city.tex.table$`Multivariate-INLAwMCMC_NY`)] <- paste0(city.tex.table$`Multivariate-INLAwMCMC_NY`[!is.na(city.tex.table$`Multivariate-INLAwMCMC_NY`)],"$^\\star$")
city.tex.table <- unite(city.tex.table,"Univariate-IwM_LA",c(`Univariate-INLAwMCMC-MH_LA`,`Univariate-INLAwMCMC-BMA_LA`),sep="",remove = TRUE,na.rm=TRUE)
city.tex.table <- unite(city.tex.table,"Multivariate-IwM_LA",c(`Multivariate-INLAwMCMC-BMA_LA`,`Multivariate-INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
city.tex.table <- unite(city.tex.table,"Multivariate-IwM_NY",c(`Multivariate-INLAwMCMC_NY`,`Multivariate-INLAwMCMC-BMA_NY`,`Multivariate-INLAwMCMC-MH_Combined`),sep="",remove = TRUE,na.rm=TRUE)
city.pretex.table.final <- city.tex.table %>% select(.,-`Multivariate-INLAwMCMC-BMA_LA`)
city.pretex.table.final <- city.pretex.table.final[,c(1:3,6,5,4)]

city.pretex.table.final$Parameter <- city.pretex.table.final$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
city.pretex.table.final$Parameter <- city.pretex.table.final$Parameter %>% str_replace_all(.,"\\\\b\\_","\\b\\_")

colnames(city.pretex.table.final) <- str_replace_all(colnames(city.pretex.table.final),"\\_","\\\\\\_")
rownames(city.pretex.table.final) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{3}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & LA & NY & Univariate LA & Multivariate LA & Multivariate NY \\\\\n")

city.tex.table.final <- xtable(city.pretex.table.final,align = c("cc|ccccc"))
save(city.tex.table.final,file="AllIwMLANYTableTex.rda")

print(city.tex.table.final, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'AllIwMLANYTableTex.tex')


# 19/10/2021 - smaller significant figures for tables
# Combined Tables: Univar and Multivar
city.table.uni_3sf <- univar.table.df_3sf[univar.table.df_3sf$Algorithm!="INLA",]
city.table.uni_3sf$Algorithm <- paste0("Univariate-",city.table.uni_3sf$Algorithm)
city.table.multi_3sf <- multivar.table.df_3sf
city.table.multi_3sf$Algorithm[city.table.multi_3sf$Algorithm!="INLA"] <- paste0("Multivariate-",city.table.multi_3sf$Algorithm[city.table.multi_3sf$Algorithm!="INLA"])
city.table.multi_3sf[city.table.multi_3sf$Algorithm=="INLA"&city.table.multi_3sf$Window=="NY"&city.table.multi_3sf$Parameter=="beta[1]",]$Parameter <- "beta[1]+b[1]"
city.table.multi_3sf[city.table.multi_3sf$Algorithm=="INLA"&city.table.multi_3sf$Window=="NY"&city.table.multi_3sf$Parameter=="beta[2]",]$Parameter <- "beta[2]+b[2]"


city.table.df_3sf <- rbind(city.table.uni_3sf,city.table.multi_3sf)
city.compact.table.df_3sf <- city.table.df_3sf %>% spread(Parameter,`Mean (SD)`)
city.tex.table_3sf <- city.compact.table.df_3sf %>% 
  pivot_longer(`b[1]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

city.tex.table_3sf  <- city.tex.table_3sf %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

city.tex.table_3sf$`Multivariate-INLAwMCMC-MH_Combined`[!is.na(city.tex.table_3sf$`Multivariate-INLAwMCMC-MH_Combined`)] <- paste0(city.tex.table_3sf$`Multivariate-INLAwMCMC-MH_Combined`[!is.na(city.tex.table_3sf$`Multivariate-INLAwMCMC-MH_Combined`)],"$^\\dagger$")
city.tex.table_3sf$`Univariate-INLAwMCMC-MH_LA`[!is.na(city.tex.table_3sf$`Univariate-INLAwMCMC-MH_LA`)] <- paste0(city.tex.table_3sf$`Univariate-INLAwMCMC-MH_LA`[!is.na(city.tex.table_3sf$`Univariate-INLAwMCMC-MH_LA`)],"$^\\dagger$")
city.tex.table_3sf$`Multivariate-INLAwMCMC_NY`[!is.na(city.tex.table_3sf$`Multivariate-INLAwMCMC_NY`)] <- paste0(city.tex.table_3sf$`Multivariate-INLAwMCMC_NY`[!is.na(city.tex.table_3sf$`Multivariate-INLAwMCMC_NY`)],"$^\\star$")
city.tex.table_3sf <- unite(city.tex.table_3sf,"Univariate-IwM_LA",c(`Univariate-INLAwMCMC-MH_LA`,`Univariate-INLAwMCMC-BMA_LA`),sep="",remove = TRUE,na.rm=TRUE)
city.tex.table_3sf <- unite(city.tex.table_3sf,"Multivariate-IwM_LA",c(`Multivariate-INLAwMCMC-BMA_LA`,`Multivariate-INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
city.tex.table_3sf <- unite(city.tex.table_3sf,"Multivariate-IwM_NY",c(`Multivariate-INLAwMCMC_NY`,`Multivariate-INLAwMCMC-BMA_NY`,`Multivariate-INLAwMCMC-MH_Combined`),sep="",remove = TRUE,na.rm=TRUE)
city.pretex.table.final_3sf <- city.tex.table_3sf %>% select(.,-`Multivariate-INLAwMCMC-BMA_LA`)
city.pretex.table.final_3sf <- city.pretex.table.final_3sf[,c(1:3,6,5,4)]

city.pretex.table.final_3sf$Parameter <- city.pretex.table.final_3sf$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
city.pretex.table.final_3sf$Parameter <- city.pretex.table.final_3sf$Parameter %>% str_replace_all(.,"\\\\b\\_","\\b\\_")

colnames(city.pretex.table.final_3sf) <- str_replace_all(colnames(city.pretex.table.final_3sf),"\\_","\\\\\\_")
rownames(city.pretex.table.final_3sf) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{3}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & LA & NY & Univariate LA & Multivariate LA & Multivariate NY \\\\\n")

city.tex.table.final_3sf <- xtable(city.pretex.table.final_3sf,align = c("cc|ccccc"))
save(city.tex.table.final_3sf,file="AllIwMLANYTableTex_3sf.rda")

print(city.tex.table.final_3sf, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'AllIwMLANYTableTex_3sf.tex')

# Smaller significant figures for tables - Take 2 for signif() potential rounding issues
# Combined Tables: Univar and Multivar
city.table.uni_3sfTake2 <- univar.table.df_3sfTake2[univar.table.df_3sfTake2$Algorithm!="INLA",]
city.table.uni_3sfTake2$Algorithm <- paste0("Univariate-",city.table.uni_3sfTake2$Algorithm)
city.table.multi_3sfTake2 <- multivar.table.df_3sfTake2
city.table.multi_3sfTake2$Algorithm[city.table.multi_3sfTake2$Algorithm!="INLA"] <- paste0("Multivariate-",city.table.multi_3sfTake2$Algorithm[city.table.multi_3sfTake2$Algorithm!="INLA"])
city.table.multi_3sfTake2[city.table.multi_3sfTake2$Algorithm=="INLA"&city.table.multi_3sfTake2$Window=="NY"&city.table.multi_3sfTake2$Parameter=="beta[1]",]$Parameter <- "beta[1]+b[1]"
city.table.multi_3sfTake2[city.table.multi_3sfTake2$Algorithm=="INLA"&city.table.multi_3sfTake2$Window=="NY"&city.table.multi_3sfTake2$Parameter=="beta[2]",]$Parameter <- "beta[2]+b[2]"


city.table.df_3sfTake2 <- rbind(city.table.uni_3sfTake2,city.table.multi_3sfTake2)
city.compact.table.df_3sfTake2 <- city.table.df_3sfTake2 %>% spread(Parameter,`Mean (SD)`)
city.tex.table_3sfTake2 <- city.compact.table.df_3sfTake2 %>% 
  pivot_longer(`b[1]`:`sigma`, names_to = "Parameter", values_to = "Mean (SD)")

city.tex.table_3sfTake2  <- city.tex.table_3sfTake2 %>% 
  pivot_wider(names_from = c(Algorithm,Window), values_from = `Mean (SD)`)

city.tex.table_3sfTake2$`Multivariate-INLAwMCMC-MH_Combined`[!is.na(city.tex.table_3sfTake2$`Multivariate-INLAwMCMC-MH_Combined`)] <- paste0(city.tex.table_3sfTake2$`Multivariate-INLAwMCMC-MH_Combined`[!is.na(city.tex.table_3sfTake2$`Multivariate-INLAwMCMC-MH_Combined`)],"$^\\dagger$")
city.tex.table_3sfTake2$`Univariate-INLAwMCMC-MH_LA`[!is.na(city.tex.table_3sfTake2$`Univariate-INLAwMCMC-MH_LA`)] <- paste0(city.tex.table_3sfTake2$`Univariate-INLAwMCMC-MH_LA`[!is.na(city.tex.table_3sfTake2$`Univariate-INLAwMCMC-MH_LA`)],"$^\\dagger$")
city.tex.table_3sfTake2$`Multivariate-INLAwMCMC_NY`[!is.na(city.tex.table_3sfTake2$`Multivariate-INLAwMCMC_NY`)] <- paste0(city.tex.table_3sfTake2$`Multivariate-INLAwMCMC_NY`[!is.na(city.tex.table_3sfTake2$`Multivariate-INLAwMCMC_NY`)],"$^\\star$")
city.tex.table_3sfTake2 <- unite(city.tex.table_3sfTake2,"Univariate-IwM_LA",c(`Univariate-INLAwMCMC-MH_LA`,`Univariate-INLAwMCMC-BMA_LA`),sep="",remove = TRUE,na.rm=TRUE)
city.tex.table_3sfTake2 <- unite(city.tex.table_3sfTake2,"Multivariate-IwM_LA",c(`Multivariate-INLAwMCMC-BMA_LA`,`Multivariate-INLAwMCMC-MH_Combined`),sep="",remove = FALSE,na.rm=TRUE)
city.tex.table_3sfTake2 <- unite(city.tex.table_3sfTake2,"Multivariate-IwM_NY",c(`Multivariate-INLAwMCMC_NY`,`Multivariate-INLAwMCMC-BMA_NY`,`Multivariate-INLAwMCMC-MH_Combined`),sep="",remove = TRUE,na.rm=TRUE)
city.pretex.table.final_3sfTake2 <- city.tex.table_3sfTake2 %>% select(.,-`Multivariate-INLAwMCMC-BMA_LA`)
city.pretex.table.final_3sfTake2 <- city.pretex.table.final_3sfTake2[,c(1:3,6,5,4)]

city.pretex.table.final_3sfTake2$Parameter <- city.pretex.table.final_3sfTake2$Parameter %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
city.pretex.table.final_3sfTake2$Parameter <- city.pretex.table.final_3sfTake2$Parameter %>% str_replace_all(.,"\\\\b\\_","\\b\\_")

colnames(city.pretex.table.final_3sfTake2) <- str_replace_all(colnames(city.pretex.table.final_3sfTake2),"\\_","\\\\\\_")
rownames(city.pretex.table.final_3sfTake2) <- NULL
addtorow <- list()
addtorow$pos <- list(0, 0)
addtorow$command <- c(" & \\multicolumn{2}{c}{INLA} & \\multicolumn{3}{c}{INLAwMCMC} \\\\\n",
                      " Parameter & LA & NY & Univariate LA & Multivariate LA & Multivariate NY \\\\\n")

city.tex.table.final_3sfTake2 <- xtable(city.pretex.table.final_3sfTake2,align = c("cc|ccccc"))
save(city.tex.table.final_3sfTake2,file="AllIwMLANYTableTex_3sfTake2.rda")

print(city.tex.table.final_3sfTake2, add.to.row = addtorow, include.colnames = TRUE, include.rownames=FALSE, floating=T, sanitize.rownames.function = identity,sanitize.text.function = identity, latex.environments = "center", file = 'AllIwMLANYTableTex_3sfTake2.tex') # , only.contents=TRUE


# Also want plots for the log-likelihoods for the two cities as well as the log-posterior.
g.ll.df <- mh.out$run$logpost.lik %>% gather(Term,Approximation,lik.val1:log.post)
g.ll.df$x.ind <- rep((1:nrow(mh.out$run$logpost.lik)),3)
re.label <- as_labeller(c(`lik.val1`="Log-likelihood LA",`lik.val2`="Log-likelihood NY",`log.post`="Log-posterior"))
ll.plot <- ggplot(g.ll.df) + geom_line(aes(x=x.ind,y=Approximation)) + facet_wrap(vars(Term), scales="free", labeller=re.label) + xlab("Iterations") + theme_bw() + theme(axis.title = element_text(size = 25),axis.text = element_text(size = 15), strip.text = element_text(size = 25))
ggsave(plot = ll.plot,file="MultivarIwMLANYLikelihoodPosterior.pdf",height = 7.5, width = 15)

# re-run the summaries below on 19/10/2021
mean(mh.out$run$logpost.lik$lik.val1)
mean(mh.out$run$logpost.lik$lik.val2)
mean(mh.out$run$logpost.lik$log.post)
sd(mh.out$run$logpost.lik$lik.val1)
sd(mh.out$run$logpost.lik$lik.val2)
sd(mh.out$run$logpost.lik$log.post)
var(mh.out$run$logpost.lik$lik.val1)
var(mh.out$run$logpost.lik$lik.val2)
var(mh.out$run$logpost.lik$log.post)
# > mean(mh.out$run$logpost.lik$lik.val1)
# [1] -3835.644
# > mean(mh.out$run$logpost.lik$lik.val2)
# [1] -2374.587
# > mean(mh.out$run$logpost.lik$log.post)
# [1] -6216.686
# > sd(mh.out$run$logpost.lik$lik.val1)
# [1] 1.543369
# > sd(mh.out$run$logpost.lik$lik.val2)
# [1] 0.03044323
# > sd(mh.out$run$logpost.lik$log.post)
# [1] 1.542696
# > var(mh.out$run$logpost.lik$lik.val1)
# [1] 2.381988
# > var(mh.out$run$logpost.lik$lik.val2)
# [1] 0.0009267903
# > var(mh.out$run$logpost.lik$log.post)
# [1] 2.37991



# sessionInfo() -----------------------------------------------------------

sessionInfo()
