
# US Cities INLA Results --------------------------------------------------

# We want to plot the results from a pure INLA fit to the US cities with the grid and mesh resolution as selected in Chapter 4 of my thesis.
# These models were fit on the Balena HPC to ensure that they were run with the same versions of the software and packages as the INLA with MCMC implementations.

# This R script produces results for the INLA fit to the New York Motor Vehicle Theft data.

# Author: Nadeen Khaleel


# Set-Up and Libraries ----------------------------------------------------

# Either setwd() to the source file location, or run the following:
library("rstudioapi")
setwd(dirname(getActiveDocumentContext()$path))
# Then as we want to store the results in the INLA_OUTPUTS directory, also run
setwd("./INLA_OUTPUTS/USCITIES")

library(INLA)
library(sp)
library(sf)
library(maptools)
library(tibble)
library(tidyr)
library(stringr)
library(tmap)
library(gridExtra)
library(tmaptools)
library(ggplot2)
library(xtable)


# Load the Data -----------------------------------------------------------


load("../../../DATA/PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/NYC/NY2015CT4748SFCountData_projFinalScale.rda") 
ny.gta.sf <- sf_gtacount
load("../../USCITIESIwM/NYCMVTINLA.rda")


# Create Prior Functions --------------------------------------------------

# Create prior functions for plotting priors over the results from the INLA fit.


pc_denrho <- function(rho,alpha_rho,rho_0){
  lambda_1 <- -log(alpha_rho)*rho_0;
  pi = lambda_1*rho^(-2)*exp(-lambda_1*rho^(-1))
}

pc_densig <- function(sigma,alpha_sigma,sigma_0){
  lambda_2 <- -log(alpha_sigma)/sigma_0
  pi = lambda_2*exp(-lambda_2*sigma)
}

gauss_prior_inla <- function(x,mean,sd){
  pi = dnorm(x,mean=mean,sd=sd)
}

beta_prior <- function(x){
  pi = gauss_prior_inla(x,mean=0,sd=sqrt(1/0.01))
}

rho.star.ny <- c(0.4,0.1); sigma.star.ny <- c(2,0.1)
pc_den_rho <- function(x){pc_denrho(x,rho_0=rho.star.ny[1],alpha_rho=rho.star.ny[2])}
pc_den_sig <- function(x){pc_densig(x,sigma_0=sigma.star.ny[1],alpha_sigma=sigma.star.ny[2])}


# Maps for Fitted Values --------------------------------------------------

# The breaks below were created by looking at the maps and then re-arranging slightly, esp. to have 0 by itself in the colour scheme for the mean.
breaks.mean <- c(0,seq(1,50,by=5),50)
breaks.sd <- seq(0,8,by=2)
breaks.list <- list(breaks.mean,breaks.sd)

ind <- inla.stack.index(stk.ny,"est")$data
ny.gta.sf$Fitted_Mean <- ny.gta.fit$summary.fitted.values[ind,"mean"]
ny.gta.sf$Fitted_SD <- ny.gta.fit$summary.fitted.values[ind,"sd"]

fitted.ny.gta <- tm_shape(ny.gta.sf) + tm_dots(c("Fitted_Mean","Fitted_SD"),breaks=breaks.list,title=c("Mean","SD"),palette = "viridis",size = 0.225) + tm_layout(legend.position = c("left","top"),legend.title.size=1,legend.text.size = 1,panel.labels = c("Fitted Values Mean", "Fitted Values SD"))
tmap_save(fitted.ny.gta,filename="NY2015INLAMVTFitted.pdf", height = 5, width=10)


load("../../../DATA/PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/NYC/NYQuad4748_projFinal.rda")
load("../../../DATA/PROCESSED_DATA/CRIME/COUNT_DATA_FINAL/NYC/NYOrdDF4748_projFinalScale.rda")

g <- quad.ny
g.sp <- as(g,"SpatialPolygons")
g.sp <- g.sp[ord.df.ny$ind,]
g.sp$pop <- sf_homcount$pop
g.sp$inc <- sf_homcount$inc
g.sp$hom <- sf_homcount$hom
g.sp$gta <- sf_gtacount$gta

grid.ny <- st_as_sf(g.sp)

grid.ny$Fitted_Mean <- ny.gta.fit$summary.fitted.values[ind,"mean"]
grid.ny$Fitted_SD <- ny.gta.fit$summary.fitted.values[ind,"sd"]

fitted.ny.gta.grid <- tm_shape(grid.ny) + tm_fill(c("Fitted_Mean","Fitted_SD"),breaks=breaks.list,title=c("Mean","SD"),palette = "viridis",size = 0.225) + tm_layout(legend.position = c("left","top"),legend.title.size=1,legend.text.size = 1,panel.labels = c("Fitted Values Mean", "Fitted Values SD"))
tmap_save(fitted.ny.gta.grid,filename="NY2015INLAMVTFittedGrid.pdf", height = 5, width=10)



# Posterior Distributions for Parameters ----------------------------------

int.marg <- ny.gta.fit$marginals.fixed$int
beta1.marg <- ny.gta.fit$marginals.fixed$cov1
beta2.marg <- ny.gta.fit$marginals.fixed$cov2
sigma.marg <- ny.gta.fit$marginals.hyperpar$`Stdev for field`
rho.marg <- ny.gta.fit$marginals.hyperpar$`Range for field`

# Intercept
int.postprior <- ggplot() + geom_line(aes(x=int.marg[,1],y=int.marg[,2],colour="Posterior"),size=1.5) + stat_function(fun=beta_prior,geom="line",aes(colour="Prior"),size=1.5) + scale_colour_manual(values=c("black","magenta")) + xlab(~beta[0]) + ylab("Density") + ggtitle(expression(paste("Prior and Posterior Distributions for ",beta[0]),sep="")) + labs(colour = "Distribution") + theme_bw() + theme(plot.title = element_text(size = 30,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.title = element_text(size = 25),legend.text = element_text(size = 25))
int.postprior.noleg <- ggplot() + geom_line(aes(x=int.marg[,1],y=int.marg[,2],colour="Posterior"),size=1.5) + stat_function(fun=beta_prior,geom="line",aes(colour="Prior"),size=1.5) + scale_colour_manual(values=c("black","magenta")) + xlab(~beta[0]) + ylab("Density") + ggtitle(expression(paste("Prior and Posterior Distributions for ",beta[0]),sep="")) + labs(colour = "Distribution") + theme_bw() + theme(plot.title = element_text(size = 30,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.position = "none")

# Beta[1]
beta1.postprior <- ggplot() + geom_line(aes(x=beta1.marg[,1],y=beta1.marg[,2],colour="Posterior"),size=1.5) + stat_function(fun=beta_prior,geom="line",aes(colour="Prior"),size=1.5) + scale_colour_manual(values=c("black","magenta")) + xlab(~beta[1]) + ylab("Density") + ggtitle(expression(paste("Prior and Posterior Distributions for ",beta[1]),sep="")) + labs(colour = "Distribution") + theme_bw() + theme(plot.title = element_text(size = 30,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.title = element_text(size = 25),legend.text = element_text(size = 25))
beta1.postprior.noleg <- ggplot() + geom_line(aes(x=beta1.marg[,1],y=beta1.marg[,2],colour="Posterior"),size=1.5) + stat_function(fun=beta_prior,geom="line",aes(colour="Prior"),size=1.5) + scale_colour_manual(values=c("black","magenta")) + xlab(~beta[1]) + ylab("Density") + ggtitle(expression(paste("Prior and Posterior Distributions for ",beta[1]),sep="")) + labs(colour = "Distribution") + theme_bw() + theme(plot.title = element_text(size = 30,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.position = "none")

# Beta[2]
beta2.postprior <- ggplot() + geom_line(aes(x=beta2.marg[,1],y=beta2.marg[,2],colour="Posterior"),size=1.5) + stat_function(fun=beta_prior,geom="line",aes(colour="Prior"),size=1.5) + scale_colour_manual(values=c("black","magenta")) + xlab(~beta[2]) + ylab("Density") + ggtitle(expression(paste("Prior and Posterior Distributions for ",beta[2]),sep="")) + labs(colour = "Distribution") + theme_bw() + theme(plot.title = element_text(size = 30,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.title = element_text(size = 25),legend.text = element_text(size = 25))
beta2.postprior.noleg <- ggplot() + geom_line(aes(x=beta2.marg[,1],y=beta2.marg[,2],colour="Posterior"),size=1.5) + stat_function(fun=beta_prior,geom="line",aes(colour="Prior"),size=1.5) + scale_colour_manual(values=c("black","magenta")) + xlab(~beta[2]) + ylab("Density") + ggtitle(expression(paste("Prior and Posterior Distributions for ",beta[2]),sep="")) + labs(colour = "Distribution") + theme_bw() + theme(plot.title = element_text(size = 30,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.position = "none")

# Standard Deviation of GP
sigma.postprior <- ggplot() + geom_line(aes(x=sigma.marg[,1],y=sigma.marg[,2],colour="Posterior"),size=1.5) + stat_function(fun=pc_den_sig,geom="line",aes(colour="Prior"),size=1.5) + scale_colour_manual(values=c("black","magenta")) + xlab(~sigma) + ylab("Density") + ggtitle(expression(paste("Prior and Posterior Distributions for ",sigma),sep="")) + labs(colour = "Distribution") + theme_bw() + theme(plot.title = element_text(size = 30,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.title = element_text(size = 25),legend.text = element_text(size = 25))
sigma.postprior.noleg <- ggplot() + geom_line(aes(x=sigma.marg[,1],y=sigma.marg[,2],colour="Posterior"),size=1.5) + stat_function(fun=pc_den_sig,geom="line",aes(colour="Prior"),size=1.5) + scale_colour_manual(values=c("black","magenta")) + xlab(~sigma) + ylab("Density") + ggtitle(expression(paste("Prior and Posterior Distributions for ",sigma),sep="")) + labs(colour = "Distribution") + theme_bw() + theme(plot.title = element_text(size = 30,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.position = "none")

# Range of GP
rho.postprior <- ggplot() + geom_line(aes(x=rho.marg[,1],y=rho.marg[,2],colour="Posterior"),size=1.5) + stat_function(fun=pc_den_rho,geom="line",aes(colour="Prior"),size=1.5) + scale_colour_manual(values=c("black","magenta")) + xlab(~rho) + ylab("Density") + ggtitle(expression(paste("Prior and Posterior Distributions for ",rho),sep="")) + labs(colour = "Distribution") + theme_bw() + theme(plot.title = element_text(size = 30,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.title = element_text(size = 25),legend.text = element_text(size = 25))
rho.postprior.noleg <- ggplot() + geom_line(aes(x=rho.marg[,1],y=rho.marg[,2],colour="Posterior"),size=1.5) + stat_function(fun=pc_den_rho,geom="line",aes(colour="Prior"),size=1.5) + scale_colour_manual(values=c("black","magenta")) + xlab(~rho) + ylab("Density") + ggtitle(expression(paste("Prior and Posterior Distributions for ",rho),sep="")) + labs(colour = "Distribution") + theme_bw() + theme(plot.title = element_text(size = 30,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25),legend.position = "none")


ggsave(filename = "NYCMVTINLAInterceptPostPrior.pdf",plot=int.postprior, height = 7.5, width=10)
ggsave(filename = "NYCMVTINLABeta1PostPrior.pdf",plot=beta1.postprior, height = 7.5, width=10)
ggsave(filename = "NYCMVTINLABeta2PostPrior.pdf",plot=beta2.postprior, height = 7.5, width=10)
ggsave(filename = "NYCMVTINLASigmaPostPrior.pdf",plot=sigma.postprior, height = 7.5, width=10)
ggsave(filename = "NYCMVTINLARhoPostPrior.pdf",plot=rho.postprior, height = 7.5, width=10)
ggsave(filename = "NYCMVTINLAInterceptPostPriorNoLegend.pdf",plot=int.postprior.noleg, height = 7.5, width=10)
ggsave(filename = "NYCMVTINLABeta1PostPriorNoLegend.pdf",plot=beta1.postprior.noleg, height = 7.5, width=10)
ggsave(filename = "NYCMVTINLABeta2PostPriorNoLegend.pdf",plot=beta2.postprior.noleg, height = 7.5, width=10)
ggsave(filename = "NYCMVTINLASigmaPostPriorNoLegend.pdf",plot=sigma.postprior.noleg, height = 7.5, width=10)
ggsave(filename = "NYCMVTINLARhoPostPriorNoLegend.pdf",plot=rho.postprior.noleg, height = 7.5, width=10)



#extract legend (https://stackoverflow.com/questions/13649473/add-a-common-legend-for-combined-ggplots)
#https://github.com/hadley/ggplot2/wiki/Share-a-legend-between-two-ggplot2-graphs
g_legend<-function(a.gplot){
  a.gplot <- a.gplot + theme(legend.title = element_text(size = 50),legend.text = element_text(size=45)) + guides(colour = guide_legend(override.aes = list(size=10)))
  tmp <- ggplot_gtable(ggplot_build(a.gplot))
  leg <- which(sapply(tmp$grobs, function(x) x$name) == "guide-box")
  legend <- tmp$grobs[[leg]]
  return(legend)}
mylegend <- g_legend(int.postprior)

int.scaled <- int.postprior + theme(legend.position="none")
beta1.scaled <- beta1.postprior + theme(legend.position="none")
beta2.scaled <- beta2.postprior + theme(legend.position="none")
sigma.scaled <- sigma.postprior + theme(legend.position="none")
rho.scaled <- rho.postprior + theme(legend.position="none")

all.postdens.grid <- grid.arrange(int.scaled,beta1.scaled,beta2.scaled,sigma.scaled,rho.scaled,mylegend,ncol=2,widths=c(10,10))

ggsave(plot=all.postdens.grid,filename = "NYCMVTINLAPostDensAll.pdf",height = 20, width = 25)


marg.list <- list(int.marg,beta1.marg,beta2.marg,rho.marg,sigma.marg)

summ.df <- rbind(ny.gta.fit$summary.fixed[,1:6],ny.gta.fit$summary.hyperpar)
rownames(summ.df) <- c("beta[0]","beta[1]","beta[2]","rho","sigma")
ci.99 <- sapply(marg.list,function(i){inla.qmarginal(c(0.005,0.995),i)})
summ.df$`0.005quant` <- ci.99[1,]
summ.df$`0.995quant` <- ci.99[2,]
summ.df <- tibble::rownames_to_column(summ.df)
colnames(summ.df)[1] <- "parameter"
summ.df <- summ.df[c(1:3,5,4),] # want sigma before rho


# cred.int <- ggplot(summ.df,aes(mean,parameter)) + geom_linerange(aes(xmin = `0.005quant`, xmax = `0.995quant`),colour="#6497b1") + geom_linerange(aes(xmin = `0.025quant`, xmax = `0.975quant`),size=2,colour="#005b96") + geom_point(col="#6497b1",size=2.5) + geom_vline(xintercept=0,col="red",linetype="dashed") + scale_y_discrete(labels=scales::parse_format()) + xlab("Mean") + ylab("Parameter") + ggtitle("95% and 99% Credible Intervals for\n Posterior Distributions for NYC INLA Fit") + theme_bw() + theme(plot.title = element_text(size = 32.5,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25))
cred.int <- ggplot(summ.df,aes(mean,parameter)) + geom_linerange(aes(xmin = `0.005quant`, xmax = `0.995quant`),colour="#6497b1") + geom_linerange(aes(xmin = `0.025quant`, xmax = `0.975quant`),size=2,colour="#005b96") + geom_point(col="#6497b1",size=2.5) + geom_vline(xintercept=0,col="red",linetype="dashed") + scale_y_discrete(labels=scales::parse_format()) + xlab("Posterior Quantiles") + ylab("Parameter") + ggtitle("95% and 99% Credible Intervals for\n Posterior Distributions for NYC INLA Fit") + theme_bw() + theme(plot.title = element_text(size = 32.5,hjust = 0.5),axis.title = element_text(size = 25),axis.text = element_text(size = 25))
ggsave(filename="NYCMVTINLAPosteriorCI.pdf",plot=cred.int,height = 7.5,width = 10)



ny.gta.tab <- summ.df[,c("parameter","mean","sd","0.025quant","0.975quant")]
ny.gta.tab <- ny.gta.tab %>% remove_rownames %>% column_to_rownames(var="parameter")
colnames(ny.gta.tab) <- c("Mean","SD","2.5%","97.5%")
r.name <- rownames(ny.gta.tab)
rownames(ny.gta.tab) <- r.name %>% str_replace_all(.,"\\[","\\_") %>% str_replace_all(.,"\\]","") %>% paste0("$\\",.,"$")
ny.gta.xtab <- xtable(ny.gta.tab,caption = 'Results from INLA fit of LGCP to 2015 NYC Motor Vehicle Theft Data on 1km-by-1km Grid.',digits=c(0,3,3,3,3))
align(ny.gta.xtab) <- "|c|cccc|"
print(ny.gta.xtab, include.rownames=T,include.colnames=T, floating=T, sanitize.rownames.function = identity, sanitize.rcolnames.function = identity,latex.environments = "center", file = 'NYCMVTSummaryTable.tex') # , only.contents=TRUE



# sessionInfo() -----------------------------------------------------------

sessionInfo()