# INLA within MCMC Simulation Study ---------------------------------------

# This R script contains the code for running Univariate and Multivariate INLA within MCMC code for the Simulation Study on Regular Polygon windows with the following data set-up:
# D1. Full Data set on Window 1 with parameter set 1;
# D2. Full Data set on Window 2 with parameter set 1;
# D3. Sparse Data set on Window 2 with parameter set 1 calculated by thinning simulated point pattern to 20%;
# D4. Full Data set on Window 2 with parameter set 2 (smaller same sign covariate effect 2).
# D5. Full Data set on Window 2 with parameter set 3 (same size but different sign covariate effect 2).
# This will allow us to test the behaviour of the Multivariate INLA within MCMC algorithm in different scenarios and compare them to the results of the Univariate INLA within MCMC and/or an INLA-only implementation.
# We consider the result in the following scenarios with the above data sets:
# S1: Two full data sets with matching parameter values (D1+D2);
# S2: One full data set and one sparse data set created from matching parameter values (D1+D3);
# S3: Two full data sets created from slightly different parameter values for one of the covariates (D1+D4).
# S4: Two full data sets created from different sign parameter values for one of the covariates (D1+D5).

### Altered only on Balena, removing the parallelisation and running one IwM run at a time as we reach the output, as the parallelisation ensures that the outputs (if return()) is used are printed in the out file, which results in a very (unnecesarily) large .out file, as there is only less than 500 iterations left of the current iterations (1e4) for all but the FullvFull I do not want the output being printed in such a way, so will set k=1 and remove the foreach().

# Author: Nadeen Khaleel


# Temporary Directory
library(unixtools)
unixtools::set.tempdir("/local/")


# Arguments Read In -------------------------------------------------------
# 
args=(commandArgs(TRUE))
print(args)

if (length(args)==0){
  print("No arguments supplied.")
  # Set default values.
  method = "univar" # or "multivar" - which type of INLA w MCMC
  window.vec = 1 # if univar is selected, which window
  step = "MH" # or "BMA"
  type.vec = "full" # or "sparse" or "cov" if univar + window + full (window 1 or 2) or sparse (window 2 only) or cov (window 2 only) or multivar + full+full or full+sparse or full+cov
  restart.vec = 0 # or 1 to start from scratch
  total.it.vec = 1e2 # how many total iterations
  prop.sd.vec = 1 # proposal standard deviation for the MH step
  burnin = 0 # only of use if step = "BMA"
  thin = 1 # only of use if step = "BMA"
  sign.vec = "same" # only use if looking at different cov, "same" or "diff"
}else{
  for (i in 1:length(args)){
    eval(parse(text=args[[i]]))
  }
}

# Example of R CMD BATCH command from job slurm script for implementing the Multivariate INLA within MCMC algorithm for the RegularPolygon Simulation Study:

# for MH step (running two simulations concurrently- which requires the commented out foreach code and parallel set-up below):
# R CMD BATCH --vanilla '--args method="multivar" window.vec=NA step="MH" type.vec=c("full","sparse") restart.vec=c(1,1) total.it.vec=c(1e4,1e4) prop.sd.vec=c(2,3) burnin=NA thin=NA sign.vec=NA'  SimulationStudy_Balena_final.R iwm_simstudyfullsparse.out
# R CMD BATCH --vanilla '--args method="multivar" window.vec=NA step="MH" type.vec=c("cov","cov") restart.vec=c(1,1) total.it.vec=c(1e4,1e4) prop.sd.vec=c(2,2) burnin=NA thin=NA sign.vec=c("same","diff")'  SimulationStudy_Balena_final.R iwm_simstudycovcov.out

# for MH step (just one simulation at a time):
# R CMD BATCH --vanilla '--args method="multivar" window.vec=NA step="MH" type.vec=c("full") restart.vec=c(1) total.it.vec=c(1e4) prop.sd.vec=c(2) burnin=NA thin=NA sign.vec=NA'  SimulationStudy_Balena_final.R iwm_simstudyfull.out
# R CMD BATCH --vanilla '--args method="multivar" window.vec=NA step="MH" type.vec=c("sparse") restart.vec=c(1) total.it.vec=c(1e4) prop.sd.vec=c(3) burnin=NA thin=NA sign.vec=NA'  SimulationStudy_Balena_final.R iwm_simstudysparse.out
# R CMD BATCH --vanilla '--args method="multivar" window.vec=NA step="MH" type.vec=c("cov") restart.vec=c(1) total.it.vec=c(1e4) prop.sd.vec=c(2) burnin=NA thin=NA sign.vec=c("same")'  SimulationStudy_Balena_final.R iwm_simstudycovsame.out
# R CMD BATCH --vanilla '--args method="multivar" window.vec=NA step="MH" type.vec=c("cov") restart.vec=c(1) total.it.vec=c(1e4) prop.sd.vec=c(2) burnin=NA thin=NA sign.vec=c("diff")'  SimulationStudy_Balena_final.R iwm_simstudycovdiff.out

# for BMA step:
# R CMD BATCH --vanilla '--args method="multivar" window.vec=NA step="BMA" type.vec=c("full") restart.vec=c(1) total.it.vec=c(1e4) prop.sd.vec=c(2) burnin=500 thin=1 sign.vec=NA'  SimulationStudy_Balena_final.R iwm_simstudyfullbma.out
# R CMD BATCH --vanilla '--args method="multivar" window.vec=NA step="BMA" type.vec=c("sparse") restart.vec=c(1) total.it.vec=c(1e4) prop.sd.vec=c(3) burnin=500 thin=1 sign.vec=NA'  SimulationStudy_Balena_final.R iwm_simstudysparsebma.out
# R CMD BATCH --vanilla '--args method="multivar" window.vec=NA step="BMA" type.vec=c("cov") restart.vec=c(1) total.it.vec=c(1e4) prop.sd.vec=c(2) burnin=500 thin=1 sign.vec=c("same")'  SimulationStudy_Balena_final.R iwm_simstudycovsamebma.out
# R CMD BATCH --vanilla '--args method="multivar" window.vec=NA step="BMA" type.vec=c("cov") restart.vec=c(1) total.it.vec=c(1e4) prop.sd.vec=c(2) burnin=500 thin=1 sign.vec=c("diff")'  SimulationStudy_Balena_final.R iwm_simstudycovdiffbma.out


# Libraries ---------------------------------------------------------------

# Set up parallel computing (if needed)
Nprocs <- length(type.vec)

print(Nprocs)
print(type.vec)

library(INLA)
library(mvtnorm)
library(sp)
library(sf)
library(spatstat)
library(raster)
library(maptools)
library(stringr)
library(rgeos)

library(doParallel)
library(foreach)

# Commented out as parallelisation resulted in ALL output (inc. marginals for each iteration) being printed. Paralellisation was used for most of the iterations, but removed when there were only a few iterations left in order to prevent a large .out file.

#parallelCluster <- parallel::makeCluster(Nprocs)
#print(parallelCluster)
#registerDoParallel(parallelCluster)

ind.procs <- 1:Nprocs
ptm <- proc.time()
#foreach(k = ind.procs) %dopar% { # this was used (with the parallelCluster above) for running two implementations of the Multivariate INLA within MCMC algorithm together (e.g. Scenario 1 and Scenario 2, while Scenario 3 and Scenario 4 were also run on the same node. However, due to the entire output printed (all of the INLA marginals etc) the final runs were run using k=1 below to run separately each Scenario on a node to ensure that the outputs were not all printed out.
k <- 1 
  
  library(INLA)
  library(mvtnorm)
  library(sp)
  library(sf)
  library(spatstat)
  library(raster)
  library(maptools)
  library(stringr)
  library(rgeos)
  library(unixtools)

  tdirloc.k <- paste0("/local/intmp-",k)
  dir.create(tdirloc.k)
  unixtools::set.tempdir(tdirloc.k)


  par.lic.filepath <- "./pardiso.lic" # file path for pardiso licence if in use
  inla.setOption(pardiso.license = par.lic.filepath)
  inla.setOption(num.threads=8)


  type <- type.vec[k]
  if (method=="univar"){
    window <- window.vec[k]
  }
  restart <- restart.vec[k]
  total.it <- total.it.vec[k]
  print(type)  
  prop.sd <- prop.sd.vec[k]
  sign <- sign.vec[k]

  # Set-up Constanct Across All Below Steps
  
  # Windows
  W_1 <- owin(c(0,2.5),c(0,2.5)) # this will always have 100% data
  W_2 <- owin(c(2.5,5),c(2.5,5)) # this will have 100% or 20% data and will also have different covariates
  
  # Count Data
  data.name_1 <- "Window1FullData.rda"
  data.name_2 <- "Window2FullData.rda"
  data.name_3 <- "Window2SparseData.rda"
  data.name_4 <- "Window2DifferentBeta2Data.rda"
  data.name_5 <- "Window2DifferentBeta2SparseData.rda"
  data.name_6 <- "Window2DifferentSignBeta2Data.rda"
  
  # Meshes
  mesh.name_1 <- "MeshWindow1.rda"
  mesh.name_2 <- "MeshWindow2.rda"
  
  beta.prior.means <- rep(0,2); beta.prior.sd <- rep(sqrt(100),2)
  sigma.star <- c(4,0.1); rho.star <- c(0.4,0.1)
  init.vals <- rep(0,2)
  
  # Function Inputs
  param.names <- paste0("beta",1:2)
  lower.bound.iteration <- 500 # beyond which, we can resample if INLA has errors, essentially a burn-in, or at least an greater than the burn-in
  lower.bound.buff <- 100 # if INLA has errors, want to a have a sample size of around 100 to sample from, so need at least 600 (100+lower.bound.iteration) iterations otherwise the algorithm will output an error
  tot.err.lim <- 20 # how many times will we accept errors before stopping the run
  it.err.lim <- 2 # how many errors per iteration will we allow
  
  if (method=="univar"){
    source("../IwMFunctions/lgcp_inla_w_mcmc_functions_final.R")
    inla.setOption(pardiso.license = "/home/h/nk376/pardiso.lic")
    inla.setOption(num.threads=8)

    window <- window.vec[k]    

    if (step=="MH"){ # method="univar"-step="MH"
      
      if (type=="full"){ # method="univar"-step="MH"-type="full"
        
        if (window==1){ # method="univar"-step="MH"-type="full"-window=1
          save.name.MHfull1 <- "IwMUnivarMHFullWindow1.rda"
          # 
          load(data.name_1)
          load(mesh.name_1)
          
          if (restart==1){
            load(save.name.MHfull1)
            comp.it.full.1 <- sum(!is.na(out$run$theta$beta1))
          } else {
            comp.it.full.1 <- 0
            set.seed(1*1e4+1)
          }
          
          the_spde.prior <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[1],rho.star[2]),
                                                prior.sigma = c(sigma.star[1],sigma.star[2]))
          
          mh.out.full1 <- mh.inlawmcmc(data=count.df,spde=the_spde.prior,mesh=mesh,its=total.it-comp.it.full.1,
                                      init=rep(0,2),prior.mean=beta.prior.means,prior.sd=beta.prior.sd,
                                      prop.sd=prop.sd,param.names=param.names,save.name=save.name.MHfull1,
                                      restart=restart,lb.iterr=lower.bound.iteration,lb.buff=lower.bound.buff,
                                      it.err.lim=it.err.lim,tot.err.lim=tot.err.lim)
          
        } else { # method="univar"-step="MH"-type="full"-window=2
          save.name.MHfull2 <- "IwMUnivarMHFullWindow2.rda"
          # 
          load(data.name_2)
          load(mesh.name_2)
          
          if (restart==1){
            load(save.name.MHfull2)
            comp.it.full.2 <- sum(!is.na(out$run$theta$beta1))
          } else {
            comp.it.full.2 <- 0
            set.seed(1*1e4+1)
          }
          
          the_spde.prior <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[1],rho.star[2]),
                                                prior.sigma = c(sigma.star[1],sigma.star[2]))
          
          mh.out.full2 <- mh.inlawmcmc(data=count.df,spde=the_spde.prior,mesh=mesh,its=total.it-comp.it.full.2,
                                      init=rep(0,2),prior.mean=beta.prior.means,prior.sd=beta.prior.sd,
                                      prop.sd=prop.sd,param.names=param.names,save.name=save.name.MHfull2,
                                      restart=restart,lb.iterr=lower.bound.iteration,lb.buff=lower.bound.buff,
                                      it.err.lim=it.err.lim,tot.err.lim=tot.err.lim)
          
        } # method="univar"-step="MH"-type="full"-window END if
        
      } else if (type=="sparse"){ # method="univar"-step="MH"-type="sparse"
        save.name.MHSparse <- "IwMUnivarMHSparse.rda"
        # 
        load(data.name_3)
        load(mesh.name_2)
        
        if (restart==1){
          load(save.name.MHSparse)
          comp.it.sparse <- sum(!is.na(out$run$theta$beta1))
        } else {
          comp.it.sparse <- 0
          set.seed(1*1e4+1)
        }
        
        the_spde.prior <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[1],rho.star[2]),
                                              prior.sigma = c(sigma.star[1],sigma.star[2]))
        
        mh.out.sparse <- mh.inlawmcmc(data=count.df,spde=the_spde.prior,mesh=mesh,its=total.it-comp.it.sparse,
                                    init=rep(0,2),prior.mean=beta.prior.means,prior.sd=beta.prior.sd,
                                    prop.sd=prop.sd,param.names=param.names,save.name=save.name.MHSparse,
                                    restart=restart,lb.iterr=lower.bound.iteration,lb.buff=lower.bound.buff,
                                    it.err.lim=it.err.lim,tot.err.lim=tot.err.lim)
        
      } else { # method="univar"-step="MH"-type="cov"
        if (sign=="same"){
          save.name.MHCov <- "IwMUnivarMHCov.rda"
        } else {
          save.name.MHCov <- "IwMUnivarMHDifferentSignCov.rda"
        }
        # 
        load(data.name_4)
        load(mesh.name_2)
        
        if (restart==1){
          load(save.name.MHCov)
          comp.it.cov <- sum(!is.na(out$run$theta$beta1))
        } else {
          comp.it.cov <- 0
          set.seed(1*1e4+1)
        }
        
        the_spde.prior <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[1],rho.star[2]),
                                              prior.sigma = c(sigma.star[1],sigma.star[2]))
        
        mh.out.sparse <- mh.inlawmcmc(data=count.df,spde=the_spde.prior,mesh=mesh,its=total.it-comp.it.cov,
                                      init=rep(0,2),prior.mean=beta.prior.means,prior.sd=beta.prior.sd,
                                      prop.sd=prop.sd,param.names=param.names,save.name=save.name.MHCov,
                                      restart=restart,lb.iterr=lower.bound.iteration,lb.buff=lower.bound.buff,
                                      it.err.lim=it.err.lim,tot.err.lim=tot.err.lim)
        
      } # method="univar"-step="MH"-type END if
        
    } else { # method="univar"-step="BMA"
      
      if (type=="full"){ # method="univar"-step="BMA"-type="full"
        
        if (window==1){ # method="univar"-step="BMA"-type="full"-window=1
          
          save.name.MHFull.1 <- "IwMUnivarMHFullWindow1.rda"
          load(save.name.MHFull.1)
          mh.out.full1 <- out
          
          save.name.BMAFull.1 <- "IwMUnivarBMAFullWindow1.rda"
          
          bma.out.full1 <- bma.inlawmcmc(mh.inlawmcmc.out=mh.out.full1,burnin=burnin,thin=thin,save.name=save.name.BMAFull.1)
          
        } else { # method="univar"-step="BMA"-type="full"-window=2
          
          save.name.MHFull.2 <- "IwMUnivarMHFullWindow2.rda"
          load(save.name.MHFull.2)
          mh.out.full2 <- out
          
          save.name.BMAFull.2 <- "IwMUnivarBMAFullWindow2.rda"
          
          bma.out.full2 <- bma.inlawmcmc(mh.inlawmcmc.out=mh.out.full2,burnin=burnin,thin=thin,save.name=save.name.BMAFull.2)
          
        } # method="univar"-step="BMA"-type="full"-window END if
        
      } else if (type=="sparse"){ # method="univar"-step="BMA"-type="sparse"
        
        save.name.MHSparse <- "IwMUnivarMHSparse.rda"
        load(save.name.MHSparse)
        mh.out.sparse <- out
        
        save.name.BMASparse <- "IwMUnivarBMASparse.rda"
        
        bma.out.sparse <- bma.inlawmcmc(mh.inlawmcmc.out=mh.out.sparse,burnin=burnin,thin=thin,save.name=save.name.BMASparse)
        
      } else { # method="univar"-step="BMA"-type="cov"
        
        if (sign=="same"){
          save.name.MHCov <- "IwMUnivarMHCov.rda"
          save.name.BMACov <- "IwMUnivarBMACov.rda"
        } else {
          save.name.MHCov <- "IwMUnivarMHDifferentSignCov.rda"
          save.name.BMACov <- "IwMUnivarBMADifferentSignCov.rda"
        }
        load(save.name.MHCov)
        mh.out.cov <- out
        
        
        bma.out.cov <- bma.inlawmcmc(mh.inlawmcmc.out=mh.out.cov,burnin=burnin,thin=thin,save.name=save.name.BMACov)
        
      } # method="univar"-step="BMA"-type END if
      
    } # method="univar"-step END if
    
  } else { # method="multivar"
    source("../IwMFunctions/lgcp_inla_w_mcmc_multivar_functions_final.R")
    par.lic.filepath <- "./pardiso.lic" # file path for pardiso licence if in use
    inla.setOption(pardiso.license = par.lic.filepath)
    inla.setOption(num.threads=8)
    

    if (step=="MH"){ # method="multivar"-step="MH"
      
      if (type=="full"){ # method="multivar"-step="MH"-type="full"
        
        full_by_full <- vector(mode="list",length=2)
        names(full_by_full) <-  paste0("Window_",1:2)
        data.files <- c(data.name_1,data.name_2)
        mesh.files <- c(mesh.name_1,mesh.name_2)
        for (i in 1:2){
          load(data.files[i])
          full_by_full[[i]]$count <- count.df
          
          load(mesh.files[i])
          full_by_full[[i]]$mesh <- mesh
          
          full_by_full[[i]]$spde.prior <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[1],rho.star[2]),prior.sigma = c(sigma.star[1],sigma.star[2]))
        }
        
        save.name.MHFullFull <- paste0("IwMMultivarMHFullbyFull.rda")

        count.list <- map(full_by_full,"count")
        mesh.list <- map(full_by_full,"mesh")
        spde.list <- map(full_by_full,"spde.prior")
        
        if (restart==1){
          load(save.name.MHFullFull)
          comp.it <- sum(!is.na(out$run$theta$beta1))
        } else {
          comp.it <- 0
          set.seed(1*1e4)
        }
        
        mh.out.fullfull <- mh.inlawmcmc.multivar(data.list=count.list,spde.list=spde.list,mesh.list=mesh.list,its=total.it-comp.it,
                                        init=init.vals,prior.mean=beta.prior.means,prior.sd=beta.prior.sd,prop.sd=prop.sd,
                                        param.names=param.names,restart=restart,form="orig",which.base=NULL,b.prior.mean=NULL,b.prior.sd=NULL,
                                        save.name=save.name.MHFullFull,lb.iterr=lower.bound.iteration,lb.buff=lower.bound.buff,
                                        it.err.lim=it.err.lim,tot.err.lim=tot.err.lim)
        
        
      } else if (type=="sparse"){ # method="multivar"-step="MH"-type="sparse"
        
        full_by_sparse <- vector(mode="list",length=2)
        names(full_by_sparse) <-  paste0("Window_",1:2)
        data.files <- c(data.name_1,data.name_3)
        mesh.files <- c(mesh.name_1,mesh.name_2)
        for (i in 1:2){
          load(data.files[i])
          full_by_sparse[[i]]$count <- count.df
          
          load(mesh.files[i])
          full_by_sparse[[i]]$mesh <- mesh
          
          full_by_sparse[[i]]$spde.prior <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[1],rho.star[2]),prior.sigma = c(sigma.star[1],sigma.star[2]))
        }
        
        save.name.MHFullSparse <- paste0("IwMMultivarMHFullbySparse.rda")

        count.list <- map(full_by_sparse,"count")
        mesh.list <- map(full_by_sparse,"mesh")
        spde.list <- map(full_by_sparse,"spde.prior")
        
        
        if (restart==1){
          load(save.name.MHFullSparse)
          comp.it <- sum(!is.na(out$run$theta$beta1))
        } else {
          comp.it <- 0
          set.seed(2*1e4)
        }
        
        mh.out.fullsparse <- mh.inlawmcmc.multivar(data.list=count.list,spde.list=spde.list,mesh.list=mesh.list,its=total.it-comp.it,
                                        init=init.vals,prior.mean=beta.prior.means,prior.sd=beta.prior.sd,prop.sd=prop.sd,
                                        param.names=param.names,restart=restart,form="orig",which.base=NULL,b.prior.mean=NULL,b.prior.sd=NULL,
                                        save.name=save.name.MHFullSparse,lb.iterr=lower.bound.iteration,lb.buff=lower.bound.buff,
                                        it.err.lim=it.err.lim,tot.err.lim=tot.err.lim)
        
        
      } else { # method="multivar"-step="MH"-type="cov"
        
        full_by_cov <- vector(mode="list",length=2)
        names(full_by_cov) <-  paste0("Window_",1:2)
        if (sign=="same"){
          save.name.MHFullCov <- paste0("IwMMultivarMHFullbyCov.rda")
          data.files <- c(data.name_1,data.name_4)
        } else {
          save.name.MHFullCov <- paste0("IwMMultivarMHFullbyDifferentSignCov.rda")
          data.files <- c(data.name_1,data.name_6)
        }
        mesh.files <- c(mesh.name_1,mesh.name_2)
        for (i in 1:2){
          load(data.files[i])
          full_by_cov[[i]]$count <- count.df
          
          load(mesh.files[i])
          full_by_cov[[i]]$mesh <- mesh
          
          full_by_cov[[i]]$spde.prior <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[1],rho.star[2]),prior.sigma = c(sigma.star[1],sigma.star[2]))
        }

        count.list <- map(full_by_cov,"count")
        mesh.list <- map(full_by_cov,"mesh")
        spde.list <- map(full_by_cov,"spde.prior")
        
        if (restart==1){
          load(save.name.MHFullCov)
          comp.it <- sum(!is.na(out$run$theta$beta1))
        } else {
          comp.it <- 0
	  if (sign=="same"){
	     set.seed(3*1e4)
	  } else {
	     set.seed(3.5*1e4)
	  }
        }
        
        b.prior.mean <- c(0,0)
        b.prior.sd <- c(2,2)
        
        mh.out.fullcov <- mh.inlawmcmc.multivar(data.list=count.list,spde.list=spde.list,mesh.list=mesh.list,its=total.it-comp.it,
                                        init=init.vals,prior.mean=beta.prior.means,prior.sd=beta.prior.sd,prop.sd=prop.sd,
                                        param.names=param.names,restart=restart,form="base",which.base=1,b.prior.mean=b.prior.mean,
                                        b.prior.sd=b.prior.sd,save.name=save.name.MHFullCov,lb.iterr=lower.bound.iteration,
                                        lb.buff=lower.bound.buff,it.err.lim=it.err.lim,tot.err.lim=tot.err.lim)
        
        
      } # method="multivar"-step="MH"-type END if
      
    } else { # method="multivar"-step="BMA"
      
      if (type=="full"){ # method="multivar"-step="BMA"-type="full"
        
        save.name.MHFullFull <- paste0("IwMMultivarMHFullbyFull.rda")
        load(save.name.MHFullFull)
        mh.out.fullfull <- out
        
        save.name.BMAFullFull <- paste0("IwMMultivarBMAFullbyFull.rda")
        
        bma.out.fullfull <- bma.inlawmcmc.multivar(mh.inlawmcmc.out=mh.out.fullfull,burnin=burnin,thin=thin,save.name.BMAFullFull)
        
      } else if (type=="sparse"){ # method="multivar"-step="BMA"-type="sparse"
        
        save.name.MHFullSparse <- paste0("IwMMultivarMHFullbySparse.rda")
        load(save.name.MHFullSparse)
        mh.out.fullsparse <- out
        
        save.name.BMAFullSparse <- paste0("IwMMultivarBMAFullbySparse.rda")
        
        bma.out.fullsparse <- bma.inlawmcmc.multivar(mh.inlawmcmc.out=mh.out.fullsparse,burnin=burnin,thin=thin,save.name.BMAFullSparse)
        
      } else { # method="multivar"-step="BMA"-type="cov"
        
        if (sign=="same"){
          save.name.MHFullCov <- paste0("IwMMultivarMHFullbyCov.rda")
          save.name.BMAFullCov <- paste0("IwMMultivarBMAFullbyCovComb.rda")
        } else {
          save.name.MHFullCov <- paste0("IwMMultivarMHFullbyDifferentSignCov.rda")
          save.name.BMAFullCov <- paste0("IwMMultivarBMAFullbyDifferentSignCovComb.rda")
        }
        
        load(save.name.MHFullCov)
        mh.out.fullcov <- out
        
        bma.out.fullcov <- bma.inlawmcmc.multivar.combined(mh.inlawmcmc.out=mh.out.fullcov,which.comb=2,burnin=burnin,thin=thin,save.name.BMAFullCov)
        
      } # method="multivar"-step="BMA"-type END if
      
    } # method="multivar"-step END if
    
  } # method END if
  
#} # foreach END # Parallelisation commented out
 
# Stop the clock
print(proc.time() - ptm)

print(sessionInfo())
 
#stopCluster(parallelCluster) # Parallelisation commented out
#################################################################################################

#Define arrays for storing result
rm(list=ls()) # Must finish with this.

