# INLA within MCMC: Multivariate Case for Los Angeles and New York --------

# I will consider Grid 3 and Mesh 3 for all of the cities (~1km-by-1km) and will be using the version of the Multivariate INLA within MCMC where we want to test the difference between covariate effects between some base city (LA) and the other cities (NY).

# Author: Nadeen Khaleel

# Loading Libraries and Functions -----------------------------------------

library(purrr)
library(INLA)
library(mvtnorm)
library(sp)
library(sf)
library(spatstat)
library(raster)
library(maptools)
library(stringr)
library(rgeos)
library(unixtools)
unixtools::set.tempdir("/local/")

source("../IwMFunctions/lgcp_inla_w_mcmc_multivar_functions_final.R") # this should load up all relevant libraries

# Arguments Read In -------------------------------------------------------
 
args=(commandArgs(TRUE))
print(args)

if (length(args)==0){
  print("No arguments supplied.")
  # Set default values.
  step = "MH" # or "BMA"
  restart = 0 # or 1 to start from scratch
  total.it = 1e2 # how many total iterations
  burnin = 0 # only of use if step = "BMA"
  thin = 1 # only of use if step = "BMA"
}else{
  for (i in 1:length(args)){
    eval(parse(text=args[[i]]))
  }
}

# Example of R CMD BATCH command from job slurm script for implementing the Multivariate INLA within MCMC algorithm for the US Cities:
# for MH step:
# R CMD BATCH --vanilla '--args step="MH" restart=1 total.it=1.1e4 burnin=NA thin=NA'  MultiCityMultiIwM_final.R iwm_multivar.out
# for BMA step:
# R CMD BATCH --vanilla '--args step="BMA" burnin=2000 thin=1'  MultiCityMultiIwM_final.R iwm_multivarbma.out


if (step=="MH"){
  # Loading Data ------------------------------------------------------------
  
  # Los Angeles
  LA.file <- "LA2015CT4872CountData_projFinalScale.rda"
  LA.mesh <- "LAMesh4872_projFinalScale.rda"
  
  
  # New York
  NY.file <- "NY2015CT4748CountData_projFinalScale.rda"
  NY.mesh <- "NYMesh4748_projFinalScale.rda"
  
  
  
  # Set-Up ------------------------------------------------------------------
  
  # Priors
  rho.star <- list(c(0.4,0.1),c(0.4,0.1)); sigma.star <- list(c(2,0.1),c(2,0.1))
  beta.prior.means <- rep(0,2); beta.prior.sd <- rep(sqrt(100),2)
  init.vals <- rep(0,2)
  
  b.prior.mean <- c(0,0)
  b.prior.sd <- c(2,2)
  
  # Main Data Set of All (Two) Cities
  data.list <- vector(mode="list",length=2)
  names(data.list) <-  paste0(c("LA","NY"),"_City")
  data.files <- c(LA.file,NY.file)
  mesh.files <- c(LA.mesh,NY.mesh)
  
  for (i in 1:2){
    load(data.files[i])
    count.df <- gta_countdf # Want to fit to Motor Vehicle Theft
    # Change names of Covariates "pop"->"cov1" and "inc"->"cov2" and "gta"->"count"
    colnames(count.df)[colnames(count.df)=="gta"] <- "count"
    
    # Want to scale the variables, 10,000 population and 100,000 income and create "cov1" <- "pop" and "cov2" <- "inc"
    count.df$cov1 <- count.df$pop/1e4
    count.df$cov2 <- count.df$inc/1e5
    count.df$area <- as.numeric(count.df$area)/1e8
    
    print(paste("City ",names(data.list)[i]))
    print(min(count.df$count))
    print(max(count.df$count))
    print(min(count.df$cov1))
    print(max(count.df$cov1))
    print(min(count.df$cov2))
    print(max(count.df$cov2))
    print(min(count.df$area))
    print(max(count.df$area))
    
    
    data.list[[i]]$count <- count.df
    
    load(mesh.files[i])
    data.list[[i]]$mesh <- mesh
    
    data.list[[i]]$spde.prior <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[[i]][1],rho.star[[i]][2]),prior.sigma = c(sigma.star[[i]][1],sigma.star[[i]][2]))
  }
  
  
  
  
  # Function Inputs
  param.names <- paste0("beta",1:2)
  lower.bound.iteration <- 2e3 #  beyond which, we can resample if INLA has errors, at least must be the burn-in. After first 5 day run (25/07/2021), changed this from 5e2 to 7.5e2 as this seems to be a better burn-in minimum. On 30/08/2021, after 9500 iterations, 2e3 appears to be a better minimum for the burn-in, and so any further runs will have a new burn-in.
  lower.bound.buff <- 100 # if INLA has errors, want to a have a sample size of at least 100 to sample from, so need at least 100+lower.bound.iteration iterations otherwise the algorithm will output an error
  tot.err.lim <- 20 # how many times will we accept errors before stopping the run
  it.err.lim <- 2 # how many errors per iteration will we allow
  
  prop.sd <- c(0.25,0.15)
  
  count.list <- map(data.list,"count")
  mesh.list <- map(data.list,"mesh")
  spde.list <- map(data.list,"spde.prior")
  
  save.name <- "MultiCityMultiIwM_MH.rda"
  
  if (restart==1){
    load(save.name)
    comp.it <- sum(!is.na(out$run$theta$beta1))
  } else {
    comp.it <- 0
    set.seed(5*1e4+1)
  }
  
  start.time <- proc.time()
  mh.out <- mh.inlawmcmc.multivar(data.list=count.list,spde.list=spde.list,mesh.list=mesh.list,its=total.it-comp.it,
                                  init=init.vals,prior.mean=beta.prior.means,prior.sd=beta.prior.sd,prop.sd=prop.sd,
                                  param.names=param.names,restart=restart,form="base",which.base=1,b.prior.mean=b.prior.mean,
                                  b.prior.sd=b.prior.sd,save.name=save.name,lb.iterr=lower.bound.iteration,
                                  lb.buff=lower.bound.buff,it.err.lim=it.err.lim,tot.err.lim=tot.err.lim)
  end.time <- proc.time()
  time.taken <- end.time-start.time
  print(time.taken)
  
} else { # step = "BMA"
  mh.save.name <- "MultiCityMultiIwM_MH.rda"
  load(mh.save.name)
  mh.out <- out
  
  save.name <- "MultiCityMultiIwM_BMA.rda"
  
  bma.out <- bma.inlawmcmc.multivar.combined(mh.inlawmcmc.out=mh.out,burnin=burnin,thin=thin,save.name=save.name)
  
}


#############
sessionInfo()

#############

rm(list=ls())
