#  INLA within MCMC: Univariate Case for Los Angeles MVT ------------------


# This script takes the Los Angeles data at the 1km scale with the 1km mesh resolution and will implement the univariate INLA within MCMC algorithm in order to sample from the posterior distribution of the fixed effects and then perform BMA to approximate the posterior marginals for the intercept and covariance parameters.

# Author: Nadeen Khaleel

# Loading Libraries and Functions -----------------------------------------

library(purrr)
library(INLA)
library(mvtnorm)
library(sp)
library(sf)
library(spatstat)
library(raster)
library(maptools)
library(stringr)
library(rgeos)
library(unixtools)
unixtools::set.tempdir("/local/")

source("../IwMFunctions/lgcp_inla_w_mcmc_functions_final.R") # this should load up any other relevant libraries

# Arguments Read In -------------------------------------------------------
# 
args=(commandArgs(TRUE))
print(args)

if (length(args)==0){
  print("No arguments supplied.")
  # Set default values.
  step = "MH" # or "BMA"
  restart = 0 # or 1 to start from scratch
  total.it = 1e2 # how many total iterations
  burnin = 0 # only of use if step = "BMA"
  thin = 1 # only of use if step = "BMA"
}else{
  for (i in 1:length(args)){
    eval(parse(text=args[[i]]))
  }
}

# Example of R CMD BATCH command from job slurm script for implementing the Univariate INLA within MCMC algorithm for Los Angeles:
# for MH step:
# R CMD BATCH --vanilla '--args step="MH" restart=1 total.it=1.1e4 burnin=NA thin=NA'  MultiCityUniIwM_final.R iwm_univar.out
# for BMA step:
# R CMD BATCH --vanilla '--args step="BMA" burnin=2000 thin=1'  MultiCityUniIwM_final.R iwm_univarbma.out

if (step=="MH"){
  # Loading Data ------------------------------------------------------------
  
  # Los Angeles
  LA.file <- "LA2015CT4872CountData_projFinalScale.rda"
  LA.mesh <- "LAMesh4872_projFinalScale.rda"
  
  
  # Set-Up ------------------------------------------------------------------
  
  # Priors
  rho.star <- c(0.4,0.1); sigma.star <- c(2,0.1)
  beta.prior.means <- rep(0,2); beta.prior.sd <- rep(sqrt(100),2)
  init.vals <- rep(0,2)
  
  
  # Main Data Set
  
  load(LA.file)
  count.df <- gta_countdf # Want to fit to Motor Vehicle Theft
  # Change names of Covariates "pop"->"cov1" and "inc"->"cov2" and "gta"->"count"
  colnames(count.df)[colnames(count.df)=="gta"] <- "count"
  
  # Want to scale the variables, 10,000 population and 100,000 income and create "cov1" <- "pop" and "cov2" <- "inc"
  count.df$cov1 <- count.df$pop/1e4
  count.df$cov2 <- count.df$inc/1e5
  count.df$area <- as.numeric(count.df$area)/1e8
  
  
  load(LA.mesh)
  
  spde.prior <- inla.spde2.pcmatern(mesh,alpha=2,prior.range = c(rho.star[1],rho.star[2]),prior.sigma = c(sigma.star[1],sigma.star[2]))
  
  
  # Function Inputs
  param.names <- paste0("beta",1:2)
  lower.bound.iteration <- 7.5e2 # beyond which, we can resample if INLA has errors, at least must be the burn-in. After first run (5 Days finishing on 25/07/2021), increased this to 7.5e2 from 5e2 as this appears to be a preferrable burn-in minimum. Ended up with a burn-in of 2e3, instead.
  lower.bound.buff <- 100 # if INLA has errors, want to a have a sample size of at least 100 to sample from, so need at least 100+lower.bound.iteration iterations otherwise the algorithm will output an error
  tot.err.lim <- 20 # how many times will we accept errors before stopping the run
  it.err.lim <- 2 # how many errors per iteration will we allow
  
  prop.sd <- 0.25
  
  save.name <- "MultiCityUniIwM_MH.rda"
  
  if (restart==1){
    load(save.name)
    comp.it <- sum(!is.na(out$run$theta$beta1))
  } else {
    comp.it <- 0
    set.seed(4*1e4+1)
  }
  
  start.time <- proc.time()
  mh.out <- mh.inlawmcmc(data=count.df,spde=spde.prior,mesh=mesh,its=total.it-comp.it,
                                  init=init.vals,prior.mean=beta.prior.means,prior.sd=beta.prior.sd,prop.sd=prop.sd,
                                  param.names=param.names,restart=restart,save.name=save.name,lb.iterr=lower.bound.iteration,
                                  lb.buff=lower.bound.buff,it.err.lim=it.err.lim,tot.err.lim=tot.err.lim)
  end.time <- proc.time()
  time.taken <- end.time-start.time
  print(time.taken)
  
} else { # step = "BMA"
  mh.save.name <- "MultiCityUniIwM_MH.rda"
  load(mh.save.name)
  mh.out <- out
  
  save.name <- "MultiCityUniIwM_BMA.rda"
  
  bma.out <- bma.inlawmcmc(mh.inlawmcmc.out=mh.out,burnin=burnin,thin=thin,save.name=save.name)
  
}



#############
sessionInfo()

#############

rm(list=ls())

