% This script is used to plot the experimental centroid positions as shown
% in figure 3(c), 4(a) and 4(b). To select which data set is plotted change
% the value of datasetnum accordingly: 

% 1 - 1.7 arcminute steps (Figure 4(a)).
% 2 - 2.9 arcminute steps (Figure 3(c)).
% 3 - 5.7 arcminute steps (Figure 4(b)). 

% A brief summary of this Matlab script:
% First, the code imports the near-field images obtained for each of
% the 49 wavefront tilts and calculates a weighted centroid for each one.
% The script then calculates the centroid positions (rho,phi) of the 
% analytical grid using equations 2 and 3, acting on a perfectly square 
% input array (delta,alpha). It determines the centre offset and 
% orientation of this square input grid relative to the fibre end-face 
% by taking the mean position of the experimental values of rho and phi, 
% as well as the gradients of lines drawn through each row and column. 
% In calculating the delta component of the analytical centroid 
% coordinates, the code uses the measured values of pitch (24 um) and 
% wavelength (1.55 um), and assumed the sinusoidal term was equal 
% to 1 (i.e. z = L_C/4). The code then plots both the experimental and
% analytical grids. 

% clear the workspace, command window and close all open figures. 
clear 
close all
clc

% select which dataset to plot (1-3)
datasetnum = 3;

%folder names 
folder_path = {'lab20201117/','lab20201111/','lab20201110/'};

% colour variable for plotting 
C1 = linspecer(16); 
% max rho to plot
max_rho = [0.2 0.25 0.5];

% how far the device is from the light source [m]
distance_from_source = [0.6 0.6 0.6];
micrometer_step_size = [0.30e-3 0.50e-3 1e-3];

%% define some experimental parameters
% wavelength [m]
wavelength = 1.55e-6; 
% how far the device is from the light source [m]
source_offset = distance_from_source(datasetnum);
% micrometer step size in sweep [m] 
xystepsize = micrometer_step_size(datasetnum); 
% The measured fibre pitch of the tip and untapered fibre [m]. 
pitch_tip = 24e-6;
% the number of data points. 
pts_in_sweep = 49;
% the index of the datapoint corresponding to centre of the experimental grid
midp = mean(1:pts_in_sweep);

% minimum anticipated peak height in near-field image
peak_threshold = 1000;
% read in background noise
bg_name = strcat(string(folder_path(datasetnum)), 'bg.tif');
bg_grayImage = imread(bg_name);

%% pre allocation of some variables
COM_sweepx = zeros(1,pts_in_sweep);
COM_sweepy = zeros(1,pts_in_sweep);
cores_sweep = zeros(6,pts_in_sweep);
name_array = 1;
mean_x = zeros(1,1,pts_in_sweep);
mean_y = zeros(1,1,pts_in_sweep);

% create grids/coordinates to allow for a mask to remove noise from outside
% regions of interest. 
[imageSizeY, imageSizeX] =  size(bg_grayImage);
[columnsInImage, rowsInImage] = meshgrid(1:imageSizeX, 1:imageSizeY);

%% Load near-field data and calculate the weighted-centroid positions
% read columns of grid from right to left (1:49 would be left to right)
data_index = [43:49 36:42 29:35 22:28 15:21 8:14 1:7];
for kk = 1:pts_in_sweep
for k = 1:length(name_array) 

        %im_name = ['lab20201117/sweep1/y' num2str(data_index(kk)) '.tif'];
        im_name = strcat(string(folder_path(datasetnum)),'y', num2str(data_index(kk)),'.tif');
        grayImage = imread(im_name);
        grayImage_no_subtraction = grayImage;
        grayImage = grayImage - bg_grayImage;
        
        % finds centres of each of the three gaussians. 
        p=FastPeakFind(grayImage,peak_threshold);
        
        % occasionally one of the cores appears as a double peak - here
        % these additional peaks are removed (although additional
        % conditions in the peak finding function would be a more robust 
        % way to achieve the same result)
        if (length(p) == 8) && (abs(p(5) - p(7)) < 5) && (abs(p(6) - p(8)) < 5)
            p(7:8) = [];
        elseif (length(p) == 8)
            p(3:4) = [];
        else 
        end
        %identify average position of core - i.e. the centre of the fibre. 
        mean_x(1,k,kk) = mean(p(1:2:end));
        mean_y(1,k,kk) = mean(p(2:2:end));
        cores_sweep(:,kk) = p;
        
        % use core positions to draw regions of interest (ROIs) around
        % cores. 1.Create masks for each core -> 2.create cumulative mask ->
        % 3.apply cumulative mask to set all areas of the image that are
        % not the cores to zero. 
        
        % 1.Create circular mask for each core.
        radius =60;
        circlePixels1 = (rowsInImage - p(2)).^2 ...
            + (columnsInImage - p(1)).^2 <= radius.^2;
        circlePixels2 = (rowsInImage - p(4)).^2 ...
            + (columnsInImage - p(3)).^2 <= radius.^2;
        circlePixels3 = (rowsInImage - p(6)).^2 ...
            + (columnsInImage - p(5)).^2 <= radius.^2;
        % 2.Create cumulative mask
        mask = uint16(circlePixels1 + circlePixels2 + circlePixels3);
        % 3.apply cumulative mask to set all areas of the image that are
        % not the cores to zero
        grayImage = grayImage.*mask;
        
        % if the ROIs are a bit too large, use this line to truncate the
        % tails of the 2d Gaussians. 
        grayImage(grayImage <= 5) = 0;
        % Calculate weighted centroids (Centre-of-mass (COM) positions). 
        binaryImage = true(size(grayImage));
        labeledImage = bwlabel(binaryImage);
        measurements = regionprops(labeledImage, grayImage, 'WeightedCentroid');
        centerOfMass = measurements.WeightedCentroid;
        
        % strore centre-of-mass positions 
        COM_sweepx(kk) = centerOfMass(1);
        COM_sweepy(kk) = centerOfMass(2);
    end
end

%% Evaluate position of (0,0), core 1, core 2, & core 3 
% These are all evaluated using an average of core positions across all the
% near-field images. 

% x coordinates of core 1, 2 & 3. 
corex(1) = mean(cores_sweep(1,:));
corex(2) = mean(cores_sweep(3,:));
corex(3) = mean(cores_sweep(5,:));

% y coordinates of core 1, 2 & 3. 
corey(1) = mean(cores_sweep(2,:));
corey(2) = mean(cores_sweep(4,:));
corey(3) = mean(cores_sweep(6,:));

% origin (i.e. point at the centre of the three cores). 
originx = mean(mean(corex));
originy = mean(mean(corey));

% radial displacement of cores 1, 2 & 3 from the origin. 
rad_core_1 = sqrt((corex(1) - originx)^2 + (corey(1) - originy)^2);
rad_core_2 = sqrt((corex(2) - originx)^2 + (corey(2) - originy)^2);
rad_core_3 = sqrt((corex(3) - originx)^2 + (corey(3) - originy)^2);

% the mean radial displacement - useful for normalising data later. 
rad_mean = mean([rad_core_1 rad_core_2 rad_core_3]);

%% Find a (linear) fit of each of the Y sweeps.
% This is done to work out the orientation of Y axis relative to the 
% optical fibre

py1 = polyfit(COM_sweepx(1:7),COM_sweepy(1:7),1); 
py2 = polyfit(COM_sweepx(8:14),COM_sweepy(8:14),1); 
py3 = polyfit(COM_sweepx(15:21),COM_sweepy(15:21),1); 
py4 = polyfit(COM_sweepx(22:28),COM_sweepy(22:28),1); 
py5 = polyfit(COM_sweepx(29:35),COM_sweepy(29:35),1); 
py6 = polyfit(COM_sweepx(36:42),COM_sweepy(36:42),1); 
py7 = polyfit(COM_sweepx(43:49),COM_sweepy(43:49),1); 

fy1 = py1(1);
fy2 = py2(1);
fy3 = py3(1);
fy4 = py4(1);
fy5 = py5(1);
fy6 = py6(1);
fy7 = py7(1);

% the mean gradient of fits for data points running along the y axis. 
fym = mean([fy1 fy2 fy3 fy4 fy5 fy6 fy7]);

% clear up the workspace... 
clearvars py1 py2 py3 py4 py5 py6 py7;
clearvars fy1 fy2 fy3 fy4 fy5 fy6 fy7;

%% Find a (linear) fit of each of the X sweeps.
% This is done to work out the orientation of X axis relative to the 
% optical fibre

px1 = polyfit(COM_sweepx(1:7:43),COM_sweepy(1:7:43),1);
px2 = polyfit(COM_sweepx(2:7:44),COM_sweepy(2:7:44),1);
px3 = polyfit(COM_sweepx(3:7:45),COM_sweepy(3:7:45),1);
px4 = polyfit(COM_sweepx(4:7:46),COM_sweepy(4:7:46),1);
px5 = polyfit(COM_sweepx(5:7:47),COM_sweepy(5:7:47),1);
px6 = polyfit(COM_sweepx(6:7:48),COM_sweepy(6:7:48),1);
px7 = polyfit(COM_sweepx(7:7:49),COM_sweepy(7:7:49),1);

fx1 = px1(1);
fx2 = px2(1);
fx3 = px3(1);
fx4 = px4(1);
fx5 = px5(1);
fx6 = px6(1);
fx7 = px7(1);

% the mean gradient of fits for data points running along the x axis. 
fxm = mean([fx1 fx2 fx3 fx4 fx5 fx6 fx7]);

% clear up the workspace... 
clearvars px1 px2 px3 px4 px5 px6 px7;
clearvars fx1 fx2 fx3 fx4 fx5 fx6 fx7;

%% Offset the data so the centre fo the fibre is (0,0). 
COM_sweepx_off = COM_sweepx - originx;
COM_sweepy_off = COM_sweepy - originy;
corex_off = corex - originx; 
corey_off = corey - originy; 

%% Apply a rotation to the offset experimental data
% This rotation alligns the experimental X and Y axis with the X and Y
% axes of the plots... This transformation is useful for handling the
% experimental and theory grids. In this section we also calculate
% the rotation needed to line up the data such that the cores would appear 
% as in theory figures.. i.e. with core 1 at (1,0), core 2 at
% (-0.5,sqrt(3)/2), and core 3 at (-0.5,-sqrt(3)/2)). This rotation is not
% applied here, but just before plotting. 

% find the angle (in radians) for rotation: 
% angle for y-tilt data using mean gradients of the linear fitting above
angle_ysweep = -atan(1/fym); 
% angle for x-tilt data using mean gradients of the linear fitting above
angle_xsweep = atan(fxm);  
% the average angle (assumes x and y sweep are ~ orthogonal) 
angle_rot = mean([angle_ysweep angle_xsweep]); 
% this angle is the rotation that would need to be applied to line up the
% cores as in theory figures.. i.e. with core one at (1,0) 
angle_rot_for_core_allignment = atan(corey_off(3)/corex_off(3));
% calculate the difference between these angles (for the rotation to be
% applied just before plotting). 
angle_difference =  abs(angle_rot - angle_rot_for_core_allignment);

% create a matrix of x and y grid coordinates for data, 
v_for_cores = [corex_off;corey_off];
v_for_sweep = [COM_sweepx_off;COM_sweepy_off];

% create the rotation matrix using the angle calculated above
R = [cos(-angle_rot) -sin(-angle_rot); sin(-angle_rot) cos(-angle_rot)];

% Apply the rotation...
vo_for_cores = R*v_for_cores;
vo_for_sweep = R*v_for_sweep;

% x and y coordinates for the rotated cores
x_rotated_cores = vo_for_cores(1,:);
y_rotated_cores = vo_for_cores(2,:);

% x and y coordinates for the experimental centroid positions
x_rotated_sweep = vo_for_sweep(1,:);
y_rotated_sweep = vo_for_sweep(2,:);

%% Offset positions of data such that the centre of the grid is (0,0)
x_rotated_sweep_centred = x_rotated_sweep;% - mean(x_rotated_sweep); % COMs
y_rotated_sweep_centred = y_rotated_sweep;% - mean(y_rotated_sweep); % COMs

x_rotated_cores_centred = x_rotated_cores;% - mean(x_rotated_sweep); % cores 
y_rotated_cores_centred = y_rotated_cores;% - mean(y_rotated_sweep); % cores 

%% Work out the average pitch of a experimental data (and plot). 

yd = zeros(size(7,6));
xd = zeros(size(7,6));

for kk = 1:6
    yd(1,kk) = y_rotated_sweep(kk) - y_rotated_sweep(kk+1);
    yd(2,kk) = y_rotated_sweep(kk+7) - y_rotated_sweep(kk+8);
    yd(3,kk) = y_rotated_sweep(kk+14) - y_rotated_sweep(kk+15);
    yd(4,kk) = y_rotated_sweep(kk+21) - y_rotated_sweep(kk+22);
    yd(5,kk) = y_rotated_sweep(kk+28) - y_rotated_sweep(kk+29);
    yd(6,kk) = y_rotated_sweep(kk+35) - y_rotated_sweep(kk+36);
    yd(7,kk) = y_rotated_sweep(kk+42) - y_rotated_sweep(kk+43);
    
    xd(1,kk) = x_rotated_sweep((kk-1)*7+1) - x_rotated_sweep((kk*7)+1);
    xd(2,kk) = x_rotated_sweep((kk-1)*7+2) - x_rotated_sweep((kk*7)+2);
    xd(3,kk) = x_rotated_sweep((kk-1)*7+3) - x_rotated_sweep((kk*7)+3);
    xd(4,kk) = x_rotated_sweep((kk-1)*7+4) - x_rotated_sweep((kk*7)+4);
    xd(5,kk) = x_rotated_sweep((kk-1)*7+5) - x_rotated_sweep((kk*7)+5);
    xd(6,kk) = x_rotated_sweep((kk-1)*7+6) - x_rotated_sweep((kk*7)+5);
    xd(7,kk) = x_rotated_sweep((kk-1)*7+7) - x_rotated_sweep((kk*7)+5);    
end

% the average pitch of a experimental data grid
rd_pitch = (mean(mean(xd)) + mean(mean(yd)))/2;

% create grid coordinates to plot with.
x_grid = (-3:3).*rd_pitch + mean(x_rotated_sweep);
y_grid = (-3:3).*rd_pitch + mean(y_rotated_sweep);

% loop to plot the fitted square grid centred about the mean COM data point

x_rotated_sweep_fit = zeros(size(x_rotated_sweep));
y_rotated_sweep_fit = zeros(size(y_rotated_sweep));

n=1; % counter
for k = 7:-1:1
    for kk = 7:-1:1
        x_rotated_sweep_fit(n) = x_grid(k);
        y_rotated_sweep_fit(n) = y_grid(kk);
        n=n+1;
    end
end

%% Convert experimental centroid positions to polar coordinates 
% evaluate experimental values of phi [Rads] and rho [Pxls]. 
sweep_phis = atan2(y_rotated_sweep_centred,x_rotated_sweep_centred);
sweep_phis(sweep_phis<0) = sweep_phis(sweep_phis<0) + 2*pi;
sweep_rhos_pxl = (x_rotated_sweep_centred.^2 + y_rotated_sweep_centred.^2).^(0.5);

%% Normalise polar coordinates of experimental centroid positions
% normalise values of rho using the mean displacemnt of the cores from 
% the centre of the fibre. 
sweep_rhos = sweep_rhos_pxl ./ rad_mean;
% normalise the pitch value of the fitted grid
rd_pitch_norm = rd_pitch / rad_mean;


%% populate values of alpha and delta for theory grid
% The loop below is alsp piggy backed to remap the expermental fit grid to 
% polar coordinates. 

n=0;
micrometer_pos_theory=zeros([1 49]);
alpha_theory=zeros([1 49]);
theta_theory=zeros([1 49]);
delta_theory=zeros([1 49]);
for k = 3:-1:-3
    for kk = 3:-1:-3
        n = n+1;
        rx = k;
        ry = kk;
        micrometer_pos_theory(n) = sqrt(rx^2 + ry^2); %normalised
        micrometer_pos_theory(n) = micrometer_pos_theory(n)*xystepsize;% [m]
        % also use this loop to remap expermental fit grid to polar coords
        rho_data_fit(n) = sqrt((rx*rd_pitch_norm)^2 + (ry*rd_pitch_norm)^2);
        
        theta_theory(n) = atan(micrometer_pos_theory(n)./source_offset); % [Rad]
        
        % calculate value for delta = 2pi/wavelegth * theta * pitch / sqrt(3);
        delta_theory(n) = (2*pi/(wavelength)).*theta_theory(n).*pitch_tip./sqrt(3); 
        
        % Calculate alpha in Radians (modifiying to go between 0 and 2pi)
        alpha_theory(n) = atan2(ry,rx);
        alpha_theory(alpha_theory<0) = alpha_theory(alpha_theory<0) + 2*pi;
    end
end

pitch_of_square_theory_grid =  delta_theory(midp-1);

%% Calculate Theory rho / phi

% we start with alpha_theory and delta_theory. These are


% angle_rot is the angle in radians for which experimental data has been 
% rotated about (0,0), the mean core position. . 

% So take theory grid and see where this sits relative to the cores in the
% rotated experimental data. 
% -----------------------------------------------------------------------
% 1. Take rotated exeperimental data for cores and recentre about average 
%    core position. 
% 2. Apply the same transformation to the theory grid. 
% 3. Find angle between theory grid and core 1 axis
% 4. Apply rotation to transformed theory grid.
% 5. Calculate Theoretical values of P1 P2 and P3 from the rotated
%    and transformed theory grid. 
% 6. Calculate COM positions on fit-centred grid using P1, P2, P3 and the
%    core positions in these coordinates. 
% -----------------------------------------------------------------------
% 1. Take rotated exeperimental data for cores and recentre about average 
%    core position. These exist for the fitted data and are 
%    x_rotated_cores, y_rotated_cores. 

% 2. Apply a transformation to the theory grid to bring this into the 
%    fibre-centred coordinates. These exist and are x_rotated_sweep_fit and
%    y_rotated_sweep_fit. Recalculate essentially the same
%    thing but scale by the difference between the normalised pitchs. 

% create grid coordinates to plot with.
x_grid_theory = ((-3:3).*rd_pitch + mean(x_rotated_sweep))*(pitch_of_square_theory_grid/rd_pitch_norm);
y_grid_theory = ((-3:3).*rd_pitch + mean(y_rotated_sweep))*(pitch_of_square_theory_grid/rd_pitch_norm);

% loop to plot the fitted square grid centred about the mean COM data point

x_rotated_sweep_theory = zeros(size(x_rotated_sweep));
y_rotated_sweep_theory = zeros(size(y_rotated_sweep));
n=1; % counter
for k = 7:-1:1
    for kk = 7:-1:1
        x_rotated_sweep_theory(n) = x_grid_theory(k);
        y_rotated_sweep_theory(n) = y_grid_theory(kk);
        n=n+1;
    end
end

% 3. Find angle between theory grid and core 1 axis - to do this work out
%    angle of phi/alpha for each (not actual but relative to the +ve x-axis
%    of this coordinate space) and define core 1 as the core with the 
%    smallest angle. 

core_angles = atan2(y_rotated_cores,x_rotated_cores);
core_angles(core_angles<0) = core_angles(core_angles<0) + 2*pi;
[core_angle, core_index_a] = min(core_angles);

core_index_c = find(core_angles == max(core_angles));
core_index_b = find(core_angles == median(core_angles));


% 4. Apply rotation to transformed theory grid.

% create a matrix of x and y grid coordinates, same for cores.
v_for_theory = [x_rotated_sweep_theory;y_rotated_sweep_theory];
v_for_theory_cores = [x_rotated_cores;y_rotated_cores];

% create the rotation matrix using the angle calculated above
R = [cos(-core_angle) -sin(-core_angle); sin(-core_angle) cos(-core_angle)];

% Apply the rotation...
vo_for_theory = R*v_for_theory;
vo_for_theory_cores = R*v_for_theory_cores;

% x and y coordinates for the theory position grid in core-alligned frame. 
x_rotated2core_sweep_theory = vo_for_theory(1,:) ./ rad_mean;
y_rotated2core_sweep_theory = vo_for_theory(2,:) ./ rad_mean;

x_rotated2core_cores = vo_for_theory_cores(1,:) ./ rad_mean;
y_rotated2core_cores = vo_for_theory_cores(2,:) ./ rad_mean;

% 5. Calculate Theoretical values of P_a P_b and P_c from the rotated
%    and transformed theory grid. 

theory_input_angle = atan2(y_rotated2core_sweep_theory,x_rotated2core_sweep_theory);
%theory_angle(theory_angle<0) = sweep_phis(theory_angle<0) + 2*pi;
theory_input_displacement = (x_rotated2core_sweep_theory.^2 + y_rotated2core_sweep_theory.^2).^(0.5);

gamma=pi/2;
dp0=theory_input_displacement.*(cos(theory_input_angle+2*pi/3)-cos(theory_input_angle));
d0m=theory_input_displacement.*(cos(theory_input_angle)-cos(theory_input_angle-2*pi/3));
dmp=theory_input_displacement.*(cos(theory_input_angle-2*pi/3)-cos(theory_input_angle+2*pi/3)); 
pa=1+(2/9)*(1-cos(gamma)).*(2*cos(dmp)-cos(d0m)-cos(dp0))+(2/3).*sin(gamma).*(sin(d0m)-sin(dp0)); 
pb=1+(2/9)*(1-cos(gamma)).*(2*cos(dp0)-cos(dmp)-cos(d0m))+(2/3).*sin(gamma).*(sin(dmp)-sin(d0m)); 
pc=1+(2/9)*(1-cos(gamma)).*(2*cos(d0m)-cos(dp0)-cos(dmp))+(2/3).*sin(gamma).*(sin(dp0)-sin(dmp)); 

% 6. Calculate COM positions on fit-centred grid using P_a, P_b, P_c and the
%    core positions in coordinates of the grid-centred data. 
%
theory_com_grid_centred_x = (x_rotated_cores_centred(core_index_a) .* pa...
    + x_rotated_cores_centred(core_index_b) .* pb...
    + x_rotated_cores_centred(core_index_c) .* pc)./(pa + pb + pc);

theory_com_grid_centred_y = (y_rotated_cores_centred(core_index_a) .* pa...
    + y_rotated_cores_centred(core_index_b) .* pb...
    + y_rotated_cores_centred(core_index_c) .* pc)./(pa + pb + pc);

% Now map these into polar coordinates to allow for plotting... 

theory_evaluated_phis = atan2(theory_com_grid_centred_y,theory_com_grid_centred_x);
theory_evaluated_phis(theory_evaluated_phis<0) = theory_evaluated_phis(theory_evaluated_phis<0) + 2*pi;
theory_evaluated_rhos_pxl = (theory_com_grid_centred_x.^2 + theory_com_grid_centred_y.^2).^(0.5);
theory_evaluated_rhos = theory_evaluated_rhos_pxl ./ rad_mean;

%% Plotting (using plot/scatter commands)
% Here we construct a polar plot using the regular cartesian plot commands. 
% This includes having to draw circles/ radial lines to create axes.
% The values of phi, rho, alpha and delta are converted back to cartesian
% coordinates here which may mean some variables are duplicated in the
% workspace, but it should be clearer to see that the same results are
% being plotted as in fig2,ax2. 

% convert data in polar coordinates to a cartesian basis. Note that the
% theory input is already aligned correctly so does not need the correction
% to alpha. 
x_experimental = sweep_rhos.*cos(sweep_phis-angle_difference);
y_experimental = sweep_rhos.*sin(sweep_phis-angle_difference);

x_theory_in = theory_input_displacement.*cos(theory_input_angle); 
y_theory_in = theory_input_displacement.*sin(theory_input_angle);

x_theory_out = theory_evaluated_rhos.*cos(theory_evaluated_phis-angle_difference);
y_theory_out = theory_evaluated_rhos.*sin(theory_evaluated_phis-angle_difference);

% Create figure and axes for plotting. Setup plotting environment. 
fig1 = figure('Color','w','units','centimeters','position',[5,5,4.3,3.8]);
ax3 = axes('Position',[0.05 0.05 0.9 0.9]); hold on
axis equal
ax3.LineWidth = 1;
ax3.FontName = 'Times';
ax3.FontSize = 10; 
ax3.XColor = 'none';
ax3.YColor = 'none';

% data set 1 has smaller values of rho, so requires a higher density of
% grid lines
if datasetnum == 1
    r_ticks = 0.05:0.05:max_rho(1);
elseif datasetnum == 2
    r_ticks = 0.1:0.1:max_rho(2);
elseif datasetnum == 3
    r_ticks = 0.1:0.1:max_rho(3);
end

% draw circles with radii increasing in steps of  0.1 up to the max radius
viscircles(zeros(length(r_ticks),2),r_ticks,'LineWidth',1,'Color',[0.85 0.85 0.85]);
% draw radial lines every 30 degrees. 
for plot_theta = 0:30:360
    plot([0 max_rho(datasetnum)*cosd(plot_theta)],[0 max_rho(datasetnum)*sind(plot_theta)],'LineWidth',1,'Color',[0.85 0.85 0.85]);
end
% draw a slightly darker circle at the maximum radius. 
viscircles([0 0],max_rho(datasetnum),'LineWidth',1,'Color',[0.75 0.75 0.75]);
% plot error bars for the experimental results 
errorbar(x_experimental,y_experimental,0.011*ones(size(x_experimental)),'both','LineStyle','none','Color',[0.25 0.25 0.25],'CapSize',4);
% plot experimental results
scplot1 = scatter(x_experimental,y_experimental,14,'MarkerEdgeColor',[0.25 0.25 0.25],'MarkerFaceColor',[1 1 1]); % scatter(x,y,marker_area,...)
% % plot input in blue 
% scplot2 = scatter(x_theory_in,y_theory_in,'b+');
% plot analytical output in red
scplot3 = scatter(x_theory_out,y_theory_out,'r+');


% annotate plot based on which data set is selected. 
if datasetnum == 1
    text(-0.02,0.2,'0.20','FontName','Times','FontSize',8)
    text(-0.02,0.15,'0.15','FontName','Times','FontSize',8)
    text(max_rho(datasetnum)+0.005, 0,'0','FontName','Times');
    text(-max_rho(datasetnum)-0.03, 0.005,'\pi','FontName','Times');
elseif datasetnum == 2
    text(-0.02,0.2,'0.20','FontName','Times','FontSize',8);
    text(-0.02,0.25,'0.25','FontName','Times','FontSize',8);
    text(-0.02,0.15,'0.15','FontName','Times','FontSize',8);
    text(max_rho(datasetnum)+0.01*1.1, 0,'0','FontName','Times');
    text(-max_rho(datasetnum)-0.03*1.25, 0.005,'\pi','FontName','Times');
elseif datasetnum == 3
    text(-0.04,0.4,'0.4','FontName','Times','FontSize',8)
    text(-0.04,0.5,'0.5','FontName','Times','FontSize',8)
    text(-0.04,0.3,'0.3','FontName','Times','FontSize',8)
    text(max_rho(datasetnum)+0.005*2.5, 0,'0','FontName','Times');
    text(-max_rho(datasetnum)-0.03*2.25, 0.005,'\pi','FontName','Times');
else 
end
