function [midLatsInDegrees, midLongsInDegrees] = ...
    GeodesicMidpoints(latOnesInDegrees, longOnesInDegrees, latTwosInDegrees, ...
    longTwosInDegrees)
% Find the midpoint along the geodesic between the given pairs of points.
% In the case in which a given pair of points are antipodes, there are an
% infinite number of geodesics between them, and thus an infinite number of
% geodesic midpoints. In this case, the midpoint on the equator with longitude
% closest to 0 is arbitrarily selected. All values are in degrees.
% midLatsInDegrees lie in [-90, 90], and midLongsInDegrees lie in (-180, 180].
%
%    usage: [midLatsInDegrees, midLongsInDegrees] = ...
%    GeodesicMidpoints(latOnesInDegrees, longOnesInDegrees, latTwosInDegrees, ...
%    longTwosInDegrees)
%
% Kurt von Laven (2021). Geodesic Midpoints 
% https://www.mathworks.com/matlabcentral/fileexchange/50000-geodesic-midpoints), 
% MATLAB Central File Exchange. Retrieved July 27, 2021.

    [oneXs, oneYs, oneZs] = Spherical2Cartesian(latOnesInDegrees, ...
                                                            longOnesInDegrees);
    [twoXs, twoYs, twoZs] = Spherical2Cartesian(latTwosInDegrees, ...
                                                            longTwosInDegrees);
    % Convert to Cartesian coordinates. This user-defined function runs faster
    % than the equivalent library function, because no error checking is
    % performed, and the radius is assumed to be 1.

    midXs  = ElementWiseMean(oneXs, twoXs);
    midYs  = ElementWiseMean(oneYs, twoYs);
    midZs  = ElementWiseMean(oneZs, twoZs);
    % Average the Cartesian cooordinates to find the midpoints.

    [midLatsInDegrees, midLongsInDegrees] = Cartesian2Spherical(midXs, ...
                                                                midYs, midZs);
    % Return to spherical coordinates. This user-defined function is more
    % appropriate to the task at hand than the equivalent library function,
    % because it marks points at the center (i.e., origin) of the sphere as NaN
    % (not a number).

    antipodes = AreBadValues(midLatsInDegrees);
    % Create a mask for the midpoints at the center of the sphere, for which
    % there is no corresponding latitude and longitude.

    midLatsInDegrees(antipodes) = 0;
    midLongsInDegrees(antipodes) = ...
                                    0.5 * (longOnesInDegrees(antipodes) + ...
                                    longTwosInDegrees(antipodes));
    % Handle the points corresponding to antipodes of the center point. There
    % is an entire circle of possible midpoints to choose from, so the one on
    % the equator that is closest to 0 degrees longitude is picked arbitrarily.
    % Note that this is not necessarily consistent with the convention in the
    % function Spherical2AzimuthalEquidistant for plotting the antipode of a
    % point on an azimuthal equidistant projection.

end
