
%------------------------------------------------------------------------%
%  Program: PLOT COMBINED SEQUENCE OF SPOT MAPS ON W. EUROPE MAP         %
%------------------------------------------------------------------------%
%  Purpose: Plot reported contacts on European map, plus coverage data   %
%           for all possible paths between active stations, plus         %
%           ionosonde ftEs data. Comment/uncomment sections as required. %
%  Uses:                                                                 %
%------------------------------------------------------------------------%
%  MatLab:  R2021b                                                       %
%------------------------------------------------------------------------%
%  Creation / modification history:                                      %
%  2021-11-27 - v1.00 - C J Deacon - Creation from original spot plot    %
%  2021-12-06 - v1.10 - C J Deacon - Reformat plots for readability      %
%  2022-02-28 - v1.20 - C J Deacon - This is now the master mapping      %
%                                    script: comment/uncomment plot      %
%                                    elements as required                %
%  2022-04-05 - v1.30 - C J Deacon - Add spot and ionosonde plot         % 
%                                    persistence                         %
%------------------------------------------------------------------------%

% - Clear memory -
clear
close all hidden

% - Define ionosonde locations, lat/long
Chilton = [51.57,-1.32];
Roquetes = [40.8,0.5];
Dourbes = [50.1,4.6];
Rome = [41.9,12.5];
Juliusruh = [54.6,13.4];
Pruhonice = [49.98,14.55];
Sopron = [47.63,16.72];

% - Combine the ionosonde locations into a single array of lat/long
Ionosondes = [Chilton ; Roquetes ; Dourbes ; Rome ; Juliusruh ; ...
    Pruhonice ; Sopron];

% - Load ftEs data
load('ftEsData.mat');
numiono = length(ftEsTime);     % number of ionsonde time samples

% - Set start and end times for plot sequence
% % Full set of figure plots for paper 1
% startdat = datenum('18-Aug-2018 13:07:30','dd-mmm-yyyy HH:MM:SS');
% enddat = datenum('18-Aug-2018 17:07:30','dd-mmm-yyyy HH:MM:SS');
% % Example time period only for paper 1
startdat = datenum('18-Aug-2018 14:37:30','dd-mmm-yyyy HH:MM:SS');
enddat = datenum('18-Aug-2018 14:52:30','dd-mmm-yyyy HH:MM:SS');
% % HG/S5 time period only for paper 2
% startdat = datenum('18-Aug-2018 15:37:30','dd-mmm-yyyy HH:MM:SS');
% enddat = datenum('18-Aug-2018 15:52:30','dd-mmm-yyyy HH:MM:SS');
% % All time periods for video
% startdat = datenum('18-Aug-2018 13:00:00','dd-mmm-yyyy HH:MM:SS');
% enddat = datenum('18-Aug-2018 17:00:00','dd-mmm-yyyy HH:MM:SS');

% - Set time window for plot sequence and spot sample
% plottime = 1;      % [minutes] spot time window per plot
plottime = 15;     % [minutes] spot time window per plot
persistm = 0;      % [minutes] no spot persistence
% persistm = 5;      % [minutes] spot persistence time

% - Define 1 hour in MATLAB time
hour = datenum('01:00','HH MM') - datenum('00:00','HH MM');

% - Define directory to save images
% normal
OutputDir  = './PLOTS';
% video with very large number of plots
% OutputDir  = 'C:\Temp video';

% - Set map geographical boundaries
% Landscape format
% latlim = [40 55];
% lonlim = [-7.5 22.5];
% Square format
latlim = [35 62.5];
lonlim = [-10 30];


% - Set up sheet name and plot colour arrays
SheetName = ["28 MHZ" "50 MHZ" "70 MHZ"];
SpotColor = ["#4d851c" "b" "r"];
PathColor = ["#4d851c" "b" "r"];
% PathColor = ["#a3fc95" "#8affff" "#fa7a7a"];

% - Calculate number of plots to create
plotstart = startdat;
plotdat = hour * plottime / 60;     % convert minutes to MATLAB time
persistd = hour * persistm / 60;     % convert minutes to MATLAB time
plotnum = (enddat - startdat)/plotdat;


% LOOP THROUGH PLOT TIME PERIODS

% All time periods
% for plot = 1 : plotnum
% Example time period only (for paper)
for plot = 7 : 7
% HG/S5 time period only
% for plot = 11 : 11

    % Initialise plot data
    plotend = plotstart + plotdat;
    samplstart = plotstart - persistd/2;      % persistence of earlier spots
    samplend = plotend + persistd/2;      % persistence of earlier spots
    % midno = 0;
    
    % PREPARE GRAPH SPACE
    % Full screen format
    % scrsz = get(0,'ScreenSize');
    % Landscape format
    % scrsz = [1,1,1450,1080];
    % Square format
    scrsz = [1,1,1080,1080];    
    figure('NumberTitle','off','Name','Spot Map','OuterPosition',scrsz);
    
    % Create geographic axes using default basemap
    geoaxes
    hold on
    geolimits(latlim,lonlim)
    geolimits('manual')
  
    % Add title
    title([datestr(plotstart) ' - ' datestr(plotend, 'HH:MM:SS')...
        ' [UTC]'],'Fontsize',18);
    
    % Maximise plot size within figure and set axes font
    ax = gca;
    outerpos = ax.OuterPosition;
    ti = ax.TightInset;
    left = outerpos(1) + ti(1) + 0.025;
    bottom = outerpos(2) + ti(2) + 0.030;
    ax_width = outerpos(3) - ti(1) - ti(3) - 0.05;
    ax_height = outerpos(4) - ti(2) - ti(4) - 0.05;
    ax.Position = [left bottom ax_width ax_height];
%     left = outerpos(1) + ti(1) + 0.015;
%     bottom = outerpos(2) + ti(2)+ 0.025;
%     ax_width = outerpos(3) - ti(1) - ti(3) - 0.03;
%     ax_height = outerpos(4) - ti(2) - ti(4) - 0.03;
%     ax.Position = [left bottom ax_width ax_height];
    ax.FontSize = 16;
    
    % PLOT COVERAGE DATA
    % Per time period, cycle through the 28, 50, 70 MHz data
    for SheetNo = 1 : 3

        % Clear working arrays from previous sheet
        clear SpotNos StnLoc StnLat StnLon

        % Load spot data
        load (['./Spotlist ' char(SheetName(SheetNo)) '.mat'])

        numspot = length(SpotTime);

        % TIME WINDOW
        % Select spots in desired time window
        indx = 0;
        for spot = 1:numspot
            if (SpotTime(spot) >= samplstart) && (SpotTime(spot) < samplend)
                indx = indx +1;
                SpotNos(indx) = spot;
            end
        end

        % Capture number of spots in the desired time window
        numspot_w = indx;
        
        % UNIQUE STATION LOCATIONS
        % Build consolidated lists of unique station location data for the
        % selected time window, if there are spots in the time window
        
        if numspot_w > 0
            
            % Set the first two locations from the first spot
            spotw = SpotNos(1);
            StnLoc(1) = SpotLoc1(spotw);
            StnLat(1) = SpotLat1(spotw);
            StnLon(1) = SpotLon1(spotw);
            StnLoc(2) = SpotLoc2(spotw);
            StnLat(2) = SpotLat2(spotw);
            StnLon(2) = SpotLon2(spotw);
            
            indx = 2;
            
            % Loop through the rest of the spot list for the time window
            % and build the list of unique locations
            for spot = 2:numspot_w
                spotw = SpotNos(spot);
                if ~ismember(SpotLoc1(spotw),StnLoc)
                    indx = indx + 1;
                    StnLoc(indx) = SpotLoc1(spotw);
                    StnLat(indx) = SpotLat1(spotw);
                    StnLon(indx) = SpotLon1(spotw);
                end
                if ~ismember(SpotLoc2(spotw),StnLoc)
                    indx = indx + 1;
                    StnLoc(indx) = SpotLoc2(spotw);
                    StnLat(indx) = SpotLat2(spotw);
                    StnLon(indx) = SpotLon2(spotw);
                end
            end
            
            % Create vector of aggregated location index numbers
            numlocs = indx;
            LocNos = 1:numlocs;
            
            % FIND ALL POSSIBLE PATHS
            % Identify all possible combinations of location numbers
            Paths = nchoosek(LocNos,2);
            numpaths = size(Paths,1);
            
            % CALCULATE MIDPOINTS
            % Find path midpoints for all possible paths > 750 km
            for path = 1:numpaths
                
                lat1 = StnLat(Paths(path,1));
                lon1 = StnLon(Paths(path,1));
                lat2 = StnLat(Paths(path,2));
                lon2 = StnLon(Paths(path,2));
                
                % Calculate path length
                tracklen = haversine (lat1,lon1,lat2,lon2);
                
                % Remove possible tropo, backscatter and multi-hop Es
                if tracklen >= 750 && tracklen <=2600
                    
                    % Find path midpoint
                    [latmid,longmid] = GeodesicMidpoints(lat1,lon1,lat2,lon2);
                    
                    % Plot midpoint
                    geoplot(latmid,longmid,'Marker','o','Color',PathColor(SheetNo),...
                        'MarkerSize',1)
                    
                    % Add to consolidated array of path midpoints
                    % midno = midno + 1;
                    % Midpoints(midno,:) = [latmid,longmid,SheetNo];
                    
                end
                
            end
        end
    end
    
    % PLOT SPOT DATA
    % Per time period, cycle through the 28, 50, 70 MHz data
    for SheetNo = 1 : 3
        
        % Load spot data
        load (['./Spotlist ' char(SheetName(SheetNo)) '.mat'])
        numspot = length(SpotTime);
        
        % Plot spots
        for spot = 1:numspot
            
            % Select spots in desired time window
            if SpotTime(spot) >= samplstart && SpotTime(spot) < samplend
                
                % Calculate path length
                tracklen = haversine (SpotLat1(spot), SpotLon1(spot), ...
                    SpotLat2(spot),SpotLon2(spot));
                
                % Remove possible tropo, backscatter and multi-hop Es
                if tracklen >= 750 && tracklen <= 2600 
                    
                    % Calculate the great circle track.
                    [lattrkgc,lontrkgc] = track2(SpotLat1(spot), SpotLon1(spot), ...
                        SpotLat2(spot),SpotLon2(spot));
                    
                    % Plot the great circle track
                    % Thin lines
                    geoplot (lattrkgc,lontrkgc,'color',PathColor(SheetNo),'linewidth',0.3)
%                     % Thicker lines
%                     geoplot (lattrkgc,lontrkgc,'color',PathColor(SheetNo),'linewidth',2)
                    
                    % Find midpoint
                    [latmid,longmid] = GeodesicMidpoints(SpotLat1(spot), ...
                        SpotLon1(spot), SpotLat2(spot),SpotLon2(spot));
                    
                    % Plot midpoint
% %                   Larger midpoint circles (for landscape format)
%                     geoplot(latmid,longmid,'Marker','o','Color',SpotColor(SheetNo),...
%                         'MarkerSize',10,'MarkerFaceColor',SpotColor(SheetNo),...
%                         'MarkerEdgeColor', 'black')
%                   Smaller midpoint circles (for square format)
                    geoplot(latmid,longmid,'Marker','o','Color',SpotColor(SheetNo),...
                        'MarkerSize',8,'MarkerFaceColor',SpotColor(SheetNo),...
                        'MarkerEdgeColor', 'black')
                    
                end
                
            end
            
        end

    end
        
    % PLOT IONOSONDE DATA
    colormap parula
    caxis([0 8.3]);

    % Select and plot ionosonde data
    for idat = 1:numiono

        % Select ionosonde data in desired time window
        if ftEsTime(idat) >= samplstart && ftEsTime(idat) < samplend

            % Plot ftEs as large coloured round marker per ionosonde
            geoscatter(Ionosondes(:,1), Ionosondes(:,2), 1200, ftEsDat(idat,:), ...
                'filled', 'o', 'MarkerEdgeColor', 'black')

            % Add ftEs value as text to each ionosonde marker
            for i = 1:7

                ftEs = ftEsDat(idat,i);

                if ftEs < 3
                    ftEsTcol = 'white';
                else
                    ftEsTcol = 'black';
                end

                ftEsText = num2str(ftEs,'%.1f');

                if ftEs == 0.0
                    ftEsText = '--';
                end

                text(Ionosondes(i,1), Ionosondes(i,2), ftEsText, ...
                    'HorizontalAlignment', 'center', 'FontSize', 16, ...
                    'FontWeight', 'bold', 'color', ftEsTcol );
            end

        end

    end

   
    % SAVE FIGURE
%     % Save as meaningfully-named figure
%     Filename = [OutputDir '/' 'Spots ' ...
%         datestr(plotstart, 'yyyymmddTHHMMSS') '.fig'];
%     saveas(gcf,Filename,'fig');

    % Save as a meaningfully-named image
    Filename = [OutputDir '/' 'Spots ' ...
        datestr(plotstart, 'yyyymmddTHHMMSS') '.tif'];
%    saveas(gcf,Filename,'tiffn');
    saveas(gcf,Filename,'tiff');

%     % Save as sequentially-named image
%     Filename = [OutputDir '/images/' 'image' num2str(plot) '.png'];
%     saveas(gcf,Filename,'png');
    
close
    
    plotstart = plotend;
    
end

