################################################################################
#
#  Analysis script for the manuscript "Understanding Freehand Cursorless 
#  Pointing Variability and Its Impact on Selection Performance"
#
#  James Whiffing, Tobias Langlotz, Christof Lutteroth, Adwait Sharma,
#  Christopher Clarke
#
################################################################################

library(lme4)
library(performance)
library(effects)
library(emmeans)
library(dplyr)
library(ARTool)
library(tidyverse)
library(ggpubr)
library(rstatix)
library(MBESS)
library(patchwork)
# library(extrafont)
# font_import(pattern="Linux Biolinum")


#######################################
## Stats tests functions
#######################################

get_grouped_data <- function(d, conditionOne, conditionTwo){
  cols2group <- c("PARTICIPANT", conditionOne, conditionTwo)
  
  d_grouped <- d %>%
    group_by_at(cols2group) %>%
    summarise(across(where(is.numeric), mean), .groups = "drop") %>%
    ungroup()
  
  return(d_grouped)
}

get_model_summary <- function(model){
  print(summary(model))
  cat(sprintf("AIC: %f\n", AIC(model)))
  cat(sprintf("BIC: %f\n", BIC(model)))
}

get_summary_stats_mean <- function(data, column_name, group){
  result <- data %>%
    group_by(across(all_of(group))) %>%
    summarise(
      across(all_of(column_name), list(mean = mean, sd = sd)),
      .groups = "drop"
    )
  print(result)
}

get_summary_stats_mean_text <- function(data, column_name, group){
  result <- data %>%
    group_by(across(all_of(group))) %>%
    summarise(
      across(all_of(column_name), list(mean = mean, sd = sd)),
      .groups = "drop"
    )
  
  for(i in 1:count(result)$n){
    mean_label <- sprintf("%s_mean",column_name)
    sd_label <- sprintf("%s_sd",column_name)
    print(sprintf("%s (M = %.2f, SD= %.2f)",result[[group]][i], result[[mean_label]][i], result[[sd_label]][i]))
  }
}

get_summary_stats_median <- function(data, column_name, group){
  result <- data %>%
    group_by(across(all_of(group))) %>%
    summarise(
      across(all_of(column_name), list(median = median, iqr = IQR)),
      .groups = "drop"
    )
  
  print(result)
}

get_summary_stats_median_text <- function(data, column_name, group=FALSE){
  if(group == FALSE){
    result <- data %>%
      summarise(
        across(all_of(column_name), list(median = median, iqr = IQR)),
        .groups = "drop")
    
    median_label <- sprintf("%s_median",column_name)
    iqr_label <- sprintf("%s_iqr",column_name)
    print(sprintf("Total (Mdn = %.2f, IQR= %.2f)", result[[median_label]][1], result[[iqr_label]][1]))
  }
  else{
    result <- data %>%
      group_by(across(all_of(group))) %>%
      summarise(
        across(all_of(column_name), list(median = median, iqr = IQR)),
        .groups = "drop")
    
    for(i in 1:count(result)$n){
      median_label <- sprintf("%s_median",column_name)
      iqr_label <- sprintf("%s_iqr",column_name)
      print(sprintf("%s (Mdn = %.2f, IQR= %.2f)",result[[group]][i], result[[median_label]][i], result[[iqr_label]][i]))
    }
  }
}

numformat <- function(val) { sub("^(-?)0.", "\\1.", sprintf("%.3f", val)) }

run_manipulation_check <- function(d, measure) {
  
  d <- d %>%
    mutate(CONDITION = paste(ACC_REQ, FOCUS_REQ, sep = "_"))
  
  # Create two ggboxplots
  plot1 <- ggboxplot(d, x = "ACC_REQ", y = measure, add = "jitter") + 
    labs(title = "Effect of ACC_REQ")
  
  plot2 <- ggboxplot(d, x = "FOCUS_REQ", y = measure, add = "jitter") + 
    labs(title = "Effect of FOCUS_REQ")
  
  # Combine them using patchwork
  plot(plot1 + plot2)
  
  # Shapiro-Wilk test for normality on residuals by factor
  shapiro_results <- d %>%
    group_by(ACC_REQ, FOCUS_REQ) %>%
    summarise(shapiro_p_value = shapiro.test(get(measure))$p.value,
              shapiro_statistic = shapiro.test(get(measure))$statistic) %>%
    ungroup()
  
  # Print the results of the Shapiro-Wilk test for each factor group
  print("Shapiro-Wilk Test for Normality:")
  print(shapiro_results)
  
  extreme_outliers <- d %>%
    group_by(CONDITION) %>%
    identify_outliers(measure)
  
  any_extreme_outliers <- any(extreme_outliers$is.extreme)
  print(sprintf("Extreme outliers: %s", any_extreme_outliers))
  print(sprintf("All normal cells: %s", all(shapiro_results$shapiro_p_value > 0.05)))
  
  run_art_check <- any_extreme_outliers | !all(shapiro_results$shapiro_p_value > 0.05)
  
  # Check if all Shapiro p-values are > 0.05
  if (!run_art_check) {
    print("Running 2-way RM ANOVA: all groups are normally distributed and no extreme outliers")
    
    get_summary_stats_mean(d, measure, "CONDITION")
    
    anova_results <- d %>%
      anova_test(
        dv = measure,             # Dependent variable
        wid = PARTICIPANT,        # Repeated measures (subject ID)
        within = c(ACC_REQ, FOCUS_REQ),  # Two within-subject factors
        effect.size = "pes" # use partial eta squared because we have two predictors
      )
    
    print(anova_results)
    
    for(i in 1:count(anova_results)$n){
      if(anova_results$p[i] < 0.05){
        
        # Extract relevant values
        F_value <- anova_results$F[i]  # First row F-statistic
        df1 <- anova_results$DFn[i]    # Numerator degrees of freedom
        df2 <- anova_results$DFd[i]    # Denominator degrees of freedom
        pes <- anova_results$pes[i]    # Partial eta squared
        
        # http://daniellakens.blogspot.com/2014/06/calculating-confidence-intervals-for.html
        Lims <- conf.limits.ncf(F.value = F_value, conf.level = 0.95, df.1 <- df1, df.2 <- df2)
        Lower.lim <- Lims$Lower.Limit/(Lims$Lower.Limit + df.1 + df.2 + 1)
        Upper.lim <- Lims$Upper.Limit/(Lims$Upper.Limit + df.1 + df.2 + 1)
        
        # Print results
        print(cat(anova_results$Effect[i]," is significant. Partial Eta Squared (η²p):", pes, "95% CI for η²p:", Lower.lim, "-", Upper.lim, "."))
        
        get_summary_stats_mean_text(d, measure, anova_results$Effect[i])
      }
      else if(!grepl(":", anova_results$Effect[i])){
        print(cat(anova_results$Effect[i], " is not significant:"))
        get_summary_stats_mean_text(d, measure, anova_results$Effect[i])
      }
    }
  } else{
    print("Using ART transformation: not all groups are normally distributed or there are extreme outliers")
    
    # https://cran.r-project.org/web/packages/ARTool/readme/README.html
    art_model <- art(as.formula(paste(measure, " ~ ACC_REQ * FOCUS_REQ + Error(PARTICIPANT)")), data = qd)
    art_summary <- summary(art_model) # Check that ART works (all values should be 0)
    print(art_summary)
    art_anova <- anova(art_model) # ANOVA ART
    print(art_anova)
    
    # Calculate effect sizes from the ANOVA results of ART model
    effect_sizes <- effectsize::eta_squared(art_model)
    print(effect_sizes)
    
    print("Significant results: ")
    
    for(i in 1:3){
      p_value <- sprintf("p = %s", numformat(art_anova$`Pr(>F)`[i]))
      if(art_anova$`Pr(>F)`[i] < 0.001){
        
        p_value = "p < .001"
      }
      if(art_anova$`Pr(>F)`[i] < 0.05){
        print(sprintf("The effect of %s on %s was significant, F(%d, %d) = %.2f, %s, $\\eta^2_p$ = %.3f [%.3f, %.3f]", 
                      art_anova$Term[i], measure, art_anova$Df[i], art_anova$Df.res[i], art_anova$`F value`[i], p_value, effect_sizes$Eta2_partial[i], effect_sizes$CI_low[i], effect_sizes$CI_high[i]))
        get_summary_stats_median_text(d, measure, art_anova$Term[i])
      }
      else{
        print(sprintf("The effect of %s on %s was NOT significant, F(%d, %d) = %.2f, %s, $\\eta^2_p$ = %.3f [%.3f, %.3f]", 
                      art_anova$Term[i], measure, art_anova$Df[i], art_anova$Df.res[i], art_anova$`F value`[i], p_value, effect_sizes$Eta2_partial[i], effect_sizes$CI_low[i], effect_sizes$CI_high[i]))
        if(!grepl(":", art_anova$Term[i])){
          get_summary_stats_median_text(d, measure, art_anova$Term[i])
        }
      }
    }
  }
}

run_art_two_way <- function(d, measure, conditionOne, conditionTwo){
  
  cols2group <- c(conditionOne, conditionTwo)
  
  shapiro_d <- d %>%
    group_by_at(cols2group) %>%
    summarise(shapiro_p_value = tryCatch({shapiro.test(get(measure))$p.value}, error = function(e) {print(sprintf("Unable to calculate Shapiro test values: %s", e$message)); 1}), 
              shapiro_statistic = tryCatch({shapiro.test(get(measure))$statistic}, error = function(e) {print(sprintf("Unable to calculate Shapiro test values: %s", e$message)); 1})) %>% 
    ungroup()
  
  print(shapiro_d)
  
  cols2group <- c("PARTICIPANT", conditionOne, conditionTwo)
  
  d_grouped <- d %>%
    group_by_at(cols2group) %>%
    summarise(across(where(is.numeric), mean), .groups = "drop") %>%
    ungroup()
  
  plot(ggline(d_grouped, x = conditionOne, y = measure,
              add = "mean_ci", color = conditionTwo,
              xlab = conditionOne, ylab = measure))
  
  
  # Runs a RM-ANOVA using ART
  art_model <- art(as.formula(paste(measure, " ~ ", conditionOne," * ", conditionTwo," + Error(PARTICIPANT)")), data = d_grouped)
  art_summary <- summary(art_model) # Check that ART works (all values should be 0)
  print(art_summary) # This appears 
  art_anova <- anova(art_model) 
  print(art_anova)
  
  # Calculate effect sizes from the ANOVA results of ART model
  effect_sizes <- effectsize::eta_squared(art_model)
  print(effect_sizes)
  
  for(i in 1:count(art_anova)$n){
    print("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~")
    p_value <- sprintf("p = %s", numformat(art_anova$`Pr(>F)`[i]))
    if(art_anova$`Pr(>F)`[i] < 0.001){
      
      p_value = "p < .001"
    }
    if(art_anova$`Pr(>F)`[i] < 0.05){
      print(sprintf("The effect of %s on %s was significant, F(%d, %d) = %.2f, %s, $\\eta^2_p$ = %.3f [%.3f, %.3f]", 
                    art_anova$Term[i], measure, art_anova$Df[i], art_anova$Df.res[i], art_anova$`F value`[i], p_value, effect_sizes$Eta2_partial[i], effect_sizes$CI_low[i], effect_sizes$CI_high[i]))
      
      if(!grepl(":", art_anova$Term[i])){
        get_summary_stats_median_text(d_grouped, measure, art_anova$Term[i])
        print(art.con(art_model, art_anova$Term[i], method = "pairwise", adjust = "holm"))
      }else{
        print("Significant interaction")
        if(conditionOne == "ACC_REQ" | conditionTwo == "ACC_REQ"){
          contrasts <- art.con(art_model, art_anova$Term[i], method = "pairwise", adjust = "holm")
          print(contrasts)
        }
        run_art_one_way_interaction(d, measure, conditionTwo, conditionOne)
      }
    }
    else{
      print(sprintf("The effect of %s on %s was NOT significant, F(%d, %d) = %.2f, %s, $\\eta^2_p$ = %.3f [%.3f, %.3f]", 
                    art_anova$Term[i], measure, art_anova$Df[i], art_anova$Df.res[i], art_anova$`F value`[i], p_value, effect_sizes$Eta2_partial[i], effect_sizes$CI_low[i], effect_sizes$CI_high[i]))
      if(!grepl(":", art_anova$Term[i])){
        get_summary_stats_median_text(d_grouped, measure, art_anova$Term[i])
      }
    }
  }
}

run_art_one_way_interaction <- function(d, measure, conditionOne, conditionTwo){
  
  conditionTwoLevels <- unique(d[[conditionTwo]])
  
  for(level in conditionTwoLevels){
    print("------------------------------------------------------")
    print("------------------------------------------------------")
    print("------------------------------------------------------")
    print(cat("Running at level: ", level))
    d_simple <- d[d[[conditionTwo]] == level, ]
    
    cols2group <- c(conditionOne)
    
    shapiro_d_simple <- d_simple %>%
      group_by_at(cols2group) %>%
      summarise(shapiro_p_value = shapiro.test(get(measure))$p.value, 
                shapiro_statistic = shapiro.test(get(measure))$statistic) %>% 
      ungroup()
    
    print(shapiro_d_simple)
    
    cols2group <- c("PARTICIPANT", conditionOne)
    
    d_simple_grouped <- d_simple %>%
      group_by_at(cols2group) %>%
      summarise(across(where(is.numeric), mean), .groups = "drop") %>%
      ungroup()
    
    # Runs a RM-ANOVA using ART
    art_model <- art(as.formula(paste(measure, " ~ ", conditionOne," + Error(PARTICIPANT)")), data = d_simple_grouped)
    art_summary <- summary(art_model) # Check that ART works (all values should be 0)
    print(art_summary) # This appears 
    art_anova <- anova(art_model) 
    print(art_anova)
    
    # Calculate effect sizes from the ANOVA results of ART model
    effect_sizes <- effectsize::eta_squared(art_model)
    print(effect_sizes)
    
    for(i in 1:count(art_anova)$n){
      if(art_anova$`Pr(>F)`[i] < 0.05){
        
        p_value <- sprintf("p = %s", numformat(art_anova$`Pr(>F)`[i]))
        if(art_anova$`Pr(>F)`[i] < 0.001){
          
          p_value = "p < .001"
        }
        
        print(sprintf("For %s, the effect of %s on %s was significant, F(%d, %d) = %.2f, %s, $\\eta^2_p$ = %.3f [%.3f, %.3f]", level, art_anova$Term[i], measure, art_anova$Df[i], art_anova$Df.res[i], art_anova$`F value`[i], p_value, effect_sizes$Eta2_partial[i], effect_sizes$CI_low[i], effect_sizes$CI_high[i]))
        print(art.con(art_model, art_anova$Term[i], method = "pairwise", adjust = "holm"))
        get_summary_stats_median_text(d_simple_grouped, measure, art_anova$Term[i])
      }
    }
  }
}


# This calculates median and IQR over the whole dataset
get_behaviour_table <- function(dataF, conditionOne, measures){
  basic <- c(conditionOne)
  all_measures <- append(basic, measures)
  
  long_data <- dataF %>%
    select(all_of(all_measures)) %>%
    pivot_longer(cols = c(measures), 
                 names_to = "Measure", values_to = "Value")
  
  summarized_data <- long_data %>%
    group_by(.data[[conditionOne]], Measure) %>%
    summarize(
      Median_Value = round(median(Value, na.rm = TRUE),2),
      IQR_Value = round(IQR(Value, na.rm = TRUE),2),
      .groups = "drop"
    )
  
  summarized_data <- summarized_data %>%
    mutate(Median_IQR = paste(Median_Value, " (", IQR_Value, ")", sep = ""))
  
  summarized_data <- summarized_data %>%
    mutate(Measure = factor(Measure, levels = measures)) %>%
    arrange(Measure)
  
  reshaped_data <- summarized_data %>%
    select(-Median_Value, -IQR_Value) %>%
    pivot_wider(names_from = c(conditionOne), 
                values_from = Median_IQR)
  
  new_hand_row <- tibble(Measure = "Right handed")
  new_all_row <- tibble(Measure = "Number of ALL trials")
  new_par_row <- tibble(Measure = "Number of participants")
  new_cond_rows <- tibble(Measure = character())
  
  for(pose in unique(ts[[conditionOne]])){
    
    num_trials <- count(ts)$n
    c_all <- count(ts[ts[[conditionOne]] == pose, ])$n
    
    right_hand <- count(ts[ts[[conditionOne]] == pose & ts$HAND == "RIGHT", ])$n
    new_hand_row <- new_hand_row %>% mutate(!!pose := sprintf("%d (%.2f%%)", right_hand, (right_hand / c_all * 100)))
    
    new_all_row <- new_all_row %>% mutate(!!pose := sprintf("%d (%.2f%%)", c_all, (c_all / num_trials * 100)))
    
    for(condition in unique(ts$CONDITION)){
      c <- count(ts[ts[[conditionOne]] == pose & ts$CONDITION == condition, ])$n
      
      if (condition %in% new_cond_rows$Measure) {
        new_cond_rows[new_cond_rows$Measure == condition, pose] <- sprintf("%d (%.2f%%)", c, (c / c_all * 100))
      } else {
        # Add a new row if it doesn’t exist yet
        new_cond_rows <- bind_rows(
          new_cond_rows,
          tibble(Measure = condition, !!pose := sprintf("%d (%.2f%%)", c, (c / c_all * 100)))
        )
      }
    }
    
    all_p <- length(unique(ts$PARTICIPANT))
    num_p <- length(unique(ts[ts[[conditionOne]] == pose, ]$PARTICIPANT))
    new_par_row <- new_par_row %>% mutate(!!pose := sprintf("%d (%.2f%%)", num_p, (num_p / all_p * 100)))
  }
  reshaped_data <- bind_rows(reshaped_data, new_hand_row)
  reshaped_data <- bind_rows(reshaped_data, new_par_row)
  reshaped_data <- bind_rows(reshaped_data, new_all_row)
  reshaped_data <- bind_rows(reshaped_data, new_cond_rows)
  
  return(reshaped_data)
}


#######################################
## Ray Calculations functions
#######################################
calculate_uncorrected_errors <- function(d, ray){
  original_yaw_label <- sprintf("%s_original_yaw_errors", tolower(ray))
  original_yaw_values <- sprintf("PINPOINTING_%s_ERROR_LED_YAW_MEDIAN", toupper(ray))
  d[[original_yaw_label]] <- d[[original_yaw_values]]
  
  original_pitch_label <- sprintf("%s_original_pitch_errors", tolower(ray))
  original_pitch_values <- sprintf("PINPOINTING_%s_ERROR_LED_PITCH_MEDIAN", toupper(ray))
  d[[original_pitch_label]] <- d[[original_pitch_values]]
  
  uncorrected_label <- sprintf("uncorrected_%s_errors", tolower(ray))
  d[[uncorrected_label]] = sqrt(d[[original_yaw_label]]^2 + d[[original_pitch_label]]^2) 
  
  return(d)
}

calculate_corrected_errors <- function(d, ray){
  ray <- toupper(ray)
  sprintf("PINPOINTING_%s_ERROR_LED_YAW_MEDIAN", toupper(ray))
  
  error_yaw_ray <- sprintf("PINPOINTING_%s_ERROR_LED_YAW_MEDIAN", ray)
  error_pitch_ray <- sprintf("PINPOINTING_%s_ERROR_LED_PITCH_MEDIAN", ray)
  
  # Replicated the algorithm from https://dl.acm.org/doi/pdf/10.1145/2702123.2702332
  yaw_ray <- sprintf("PINPOINTING_%s_YAW_MEDIAN", ray)
  pitch_ray <- sprintf("PINPOINTING_%s_PITCH_MEDIAN", ray)
  
  d[[sprintf("%s_yaw_cubed_pitch_linear", ray)]] = d[[yaw_ray]]^3 * d[[pitch_ray]]  # coef c
  d[[sprintf("%s_yaw_linear_pitch_cubed", ray)]] = d[[yaw_ray]] * d[[pitch_ray]]^3 # coef d
  d[[sprintf("%s_yaw_squared_pitch_squared", ray)]] = d[[yaw_ray]]^2 * d[[pitch_ray]]^2 # coef g
  d[[sprintf("%s_yaw_squared_pitch_linear", ray)]] = d[[yaw_ray]]^2 * d[[pitch_ray]]  # coef h
  d[[sprintf("%s_yaw_linear_pitch_squared", ray)]] = d[[yaw_ray]] * d[[pitch_ray]]^2 # coef i
  d[[sprintf("%s_yaw_linear_pitch_linear", ray)]] = d[[yaw_ray]] * d[[pitch_ray]] # coef l
  
  ols_yaw_formula <- as.formula(paste(error_yaw_ray, "~ I(", yaw_ray, "^4) +
    I(",pitch_ray,"^4) +
    ",sprintf("%s_yaw_cubed_pitch_linear", ray)," +
    ",sprintf("%s_yaw_linear_pitch_cubed", ray)," +
    I(",yaw_ray,"^3) +
    I(",pitch_ray,"^3) +
    ",sprintf("%s_yaw_squared_pitch_squared", ray)," +
    ",sprintf("%s_yaw_squared_pitch_linear", ray)," +
    ",sprintf("%s_yaw_linear_pitch_squared", ray)," +
    I(",yaw_ray,"^2) +
    I(",pitch_ray,"^2) +
    ",sprintf("%s_yaw_linear_pitch_linear", ray)," +
    ",yaw_ray," +
    ",pitch_ray))
  
  print(ols_yaw_formula)
  
  ols_yaw_model <- lm(ols_yaw_formula, data = d)
  get_model_summary(ols_yaw_model)
  
  ols_pitch_formula <- as.formula(paste(error_pitch_ray, "~ I(", yaw_ray, "^4) +
    I(",pitch_ray,"^4) +
    ",sprintf("%s_yaw_cubed_pitch_linear", ray)," +
    ",sprintf("%s_yaw_linear_pitch_cubed", ray)," +
    I(",yaw_ray,"^3) +
    I(",pitch_ray,"^3) +
    ",sprintf("%s_yaw_squared_pitch_squared", ray)," +
    ",sprintf("%s_yaw_squared_pitch_linear", ray)," +
    ",sprintf("%s_yaw_linear_pitch_squared", ray)," +
    I(",yaw_ray,"^2) +
    I(",pitch_ray,"^2) +
    ",sprintf("%s_yaw_linear_pitch_linear", ray)," +
    ",yaw_ray," +
    ",pitch_ray))
  
  ols_pitch_model <- lm(ols_pitch_formula, data = d)
  get_model_summary(ols_pitch_model)
  
  # Get predicted yaw values from the OLS model
  original_yaw_label <- sprintf("%s_original_yaw_errors", tolower(ray))
  ols_yaw_label <- sprintf("%s_ols_replicate_yaw_errors", tolower(ray))
  predicted_yaw_difference_label <- sprintf("%s_difference_ols_replicate_yaw_errors", tolower(ray))
  d[[ols_yaw_label]] <- predict(ols_yaw_model)
  d[[predicted_yaw_difference_label]] <- d[[original_yaw_label]] - d[[ols_yaw_label]]
  
  # Get predicted pitch values from the OLS models
  original_pitch_label <- sprintf("%s_original_pitch_errors", tolower(ray))
  ols_pitch_label <- sprintf("%s_ols_replicate_pitch_errors", tolower(ray))
  predicted_pitch_difference_label <- sprintf("%s_difference_ols_replicate_pitch_errors", tolower(ray))
  d[[ols_pitch_label]] <- predict(ols_pitch_model)
  d[[predicted_pitch_difference_label]] <- d[[original_pitch_label]] - d[[ols_pitch_label]]
  
  # Get overall errors
  ols_overall_error_label <- sprintf("%s_corrected_ols_replicate_errors",tolower(ray))
  d[[ols_overall_error_label]] = sqrt(d[[predicted_yaw_difference_label]]^2 + d[[predicted_pitch_difference_label]]^2)
  
  return(d)
}

check_for_and_remove_nans <- function(d, ray){
  original_yaw_label <- sprintf("%s_original_yaw_errors", tolower(ray))
  original_pitch_label <- sprintf("%s_original_pitch_errors", tolower(ray))
  
  na_points_yaw = d[!complete.cases(d[[original_yaw_label]]), ]
  
  if(count(na_points_yaw) > 0){
    print(sprintf("%s yaw: %s - %s - %s- %s - %s\n", ray, na_points_yaw$PARTICIPANT, na_points_yaw$CONDITION, na_points_yaw$TRIAL_IDX, na_points_yaw$ARM_POSE_CLASS, na_points_yaw$FINGER_INCORPORATION_CLASS))
  } else{
    print(sprintf("No yaw outliers for %s", ray))
  }
  
  na_points_pitch = d[!complete.cases(d[[original_pitch_label]]), ]
  if(count(na_points_pitch) > 0){
    cat(sprintf("%s pitch: %s - %s - %s- %s - %s\n", ray, na_points_pitch$PARTICIPANT, na_points_pitch$CONDITION, na_points_pitch$TRIAL_IDX, na_points_pitch$ARM_POSE_CLASS, na_points_pitch$FINGER_INCORPORATION_CLASS))
  } else{
    print(sprintf("No pitch outliers for %s", ray))
  }
  
  original_num_trials <- count(d)
  
  # Remove NaNs
  d <- d[complete.cases(d[[original_yaw_label]]), ]
  d <- d[complete.cases(d[[original_pitch_label]]), ]
  
  new_num_trials <- count(d)
  
  print(sprintf("Original num trials: %d - New num trials: %d", original_num_trials$n, new_num_trials$n))
  
  return(d)
}

#######################################
## Data setup
#######################################

ts <- read.csv("EncodedGestures_FINAL.csv")

# Convert Data into correct types
ts$ACC_REQ <- factor(ts$ACC_REQ)
ts$FOCUS_REQ <- factor(ts$FOCUS_REQ)
ts$PARTICIPANT <- factor(ts$PARTICIPANT)
ts$IS_DOMINANT_HAND <- factor(ts$IS_DOMINANT_HAND)
ts$TARGET_NAME <- factor(ts$TARGET_NAME)
ts$CONDITION <- factor(ts$CONDITION)
ts$HAND <- factor(ts$HAND, levels = c("LEFT", "RIGHT"))
ts$GESTURE_ORIENTATION_CLASS <- factor(ts$GESTURE_ORIENTATION_CLASS, levels = c("MEDIAL", "ALIGNED", "LATERAL"))
ts$ARM_POSE_CLASS <- factor(ts$ARM_POSE_CLASS, levels = c("OUTSTRETCHED", "PROXIMAL", "HIP-FIRE"))
ts$FINGER_INCORPORATION_CLASS <- factor(ts$FINGER_INCORPORATION_CLASS, levels = c("ADS", "OCCLUDED", "UNOCCLUDED"))

ts$TARGET_COL_NAME <- ordered(ts$TARGET_COL_NAME, levels = c("LEFTMOST", "LEFT", "CENTRE", "RIGHT", "RIGHTMOST"))
ts$TARGET_ROW_NAME <- ordered(ts$TARGET_ROW_NAME, levels = c("BOTTOM", "MIDDLE", "TOP"))
ts$HAND_DOMINANT <- factor(ts$IS_DOMINANT_HAND, ordered=TRUE, levels = c("True", "False"), labels=c("DOMINANT", "NON-DOMINANT"))
ts$HAND_USED_PERCENTAGE <- (as.numeric(ts$HAND) - 1) * 100

ts$ACTIVE_MS = ts$INITIAL_BALLISTIC_MS + ts$PINPOINTING_MS
ts$TrainedOn = "All"

ts$HAND_USED <- (as.numeric(ts$HAND) - 1) * 100

rays <- c("efrc_cyclops", "hfrc", "ifrc", "frc")

for(ray in rays){
  ts <- calculate_uncorrected_errors(ts, ray)
  ts <- check_for_and_remove_nans(ts, ray)
  ts <- calculate_corrected_errors(ts, ray)
}


################################################################################
# Systematic Compensation
################################################################################

ts_cols <- c("uncorrected_ifrc_errors", "ifrc_corrected_ols_replicate_errors",
             "uncorrected_frc_errors", "frc_corrected_ols_replicate_errors",
             "uncorrected_efrc_cyclops_errors", "efrc_cyclops_corrected_ols_replicate_errors",
             "uncorrected_hfrc_errors", "hfrc_corrected_ols_replicate_errors")

data_long_ts <- ts %>%
  pivot_longer(
    cols = ts_cols,
    names_to = "Model",          # New column for names
    values_to = "Error"         # New column for values
  )

data_long_ts %>%
  group_by(Model) %>%
  get_summary_stats(Error, show = c("median", "iqr", "min", "max"))

long_ts <- ts %>%
  pivot_longer(cols = ts_cols, 
               names_to = "Type", 
               values_to = "Error") %>%
  mutate(
    # Extract 'Ray' component using regex
    Ray = case_when(
      grepl("efrc_cyclops", Type) ~ "EFRC (Cyclops)",
      grepl("hfrc", Type) ~ "HFRC",
      grepl("ifrc", Type) ~ "IFRC",
      grepl("frc", Type) ~ "FRC"
    ),
    # Determine 'Model' type
    Model = case_when(
      grepl("uncorrected", Type) ~ "Uncorrected",
      grepl("ols_replicate", Type) ~ "Corrected"
    )
  ) %>%
  select(PARTICIPANT, Ray, Model, Error)  # Keep relevant columns

long_ts %>%
  group_by(PARTICIPANT, Ray, Model) %>%
  get_summary_stats(Error, show = c("median", "iqr", "min", "max"))

conditionOne <- "Ray"
conditionTwo <- "Model"
cols2group <- c("PARTICIPANT", "Ray", "Model")
measure <- "Error"

long_ts$Ray <- factor(long_ts$Ray)
long_ts$Model <- factor(long_ts$Model)

ts_grouped <- long_ts %>%
  group_by_at(cols2group) %>%
  summarise(across(where(is.numeric), mean), .groups = "drop") %>%
  ungroup()

plot(ggline(ts_grouped, x = conditionOne, y = measure,
            add = "mean_ci", color = conditionTwo,
            xlab = conditionOne, ylab = measure))


# Runs a RM-ANOVA using ART
art_model <- art(as.formula(paste(measure, " ~ ", conditionOne," * ", 
                                  conditionTwo," + Error(PARTICIPANT)")), 
                 data = ts_grouped)

# Check that ART works (all values should be 0)
art_summary <- summary(art_model) 

print(art_summary)
art_anova <- anova(art_model) 
print(art_anova)

# Calculate effect sizes from the ANOVA results of ART model
effect_sizes <- effectsize::eta_squared(art_model)
print(effect_sizes)

# At this point, it demonstrates there are significant differences between the
# rays, the models, and the interaction. We first look at the interaction.
for(i in 1:count(art_anova)$n){
  if(art_anova$`Pr(>F)`[i] < 0.05){
    
    p_value <- sprintf("p = %s", numformat(art_anova$`Pr(>F)`[i]))
    if(art_anova$`Pr(>F)`[i] < 0.001){
      
      p_value = "p < .001"
    }
    
    print(sprintf("The effect of %s on %s was significant, F(%d, %d) = %.2f, %s, $\\eta^2_p$ = %.3f [%.3f, %.3f]", 
                  art_anova$Term[i], measure, art_anova$Df[i], art_anova$Df.res[i], art_anova$`F value`[i], p_value, effect_sizes$Eta2_partial[i], effect_sizes$CI_low[i], effect_sizes$CI_high[i]))
    
    if(!grepl(":", art_anova$Term[i])){
      print(art.con(art_model, art_anova$Term[i], method = "pairwise", adjust = "holm"))
      get_summary_stats_median_text(ts_grouped, measure, art_anova$Term[i])
    }
  }
}
# This shows us that all the rays are significantly different to each other, 
# except EFRC and HFRC when including both corrected and uncorrected.


################################################################################
# To look at the interaction, we first look at each ray, and see whether or not 
# the corrected model is a significant improvement over the uncorrected.
# Because we are looking at 4 comparisons (one for each ray), we posthoc correct
# the p values

corrected_p <- c()
ray_comparisons <- tibble()
ray_effectsizes <- tibble()

name_rays <- c("EFRC (Cyclops)", "HFRC", "IFRC", "FRC")

for(ray in name_rays){
  ts_grouped_ray <- ts_grouped[ts_grouped$Ray == ray, ]
  
  wilcoxon_result <- ts_grouped_ray %>% wilcox_test(Error ~ Model, paired = TRUE, detailed = TRUE)
  corrected_p <- append(corrected_p, wilcoxon_result$p)
  wilcoxon_result <- mutate(wilcoxon_result, ray = ray)
  wilcoxon_result$ray = ray
  wilcoxon_effect_size <- ts_grouped_ray %>% wilcox_effsize(Error ~ Model, paired = TRUE)
  
  if(length(ray_comparisons) == 0){
    ray_comparisons <- wilcoxon_result
    ray_effectsizes <- wilcoxon_effect_size
  }else{
    ray_comparisons <- add_row(ray_comparisons, wilcoxon_result)
    ray_effectsizes <- add_row(ray_effectsizes, wilcoxon_effect_size)
  }
}

ray_comparisons <- mutate(ray_comparisons, p_adjust = p.adjust(corrected_p, "holm"))

# Summary stats
stats <- ts_grouped %>%
  group_by(Ray, Model) %>%
  get_summary_stats(Error, show = c("median", "iqr"))

# These values are reported in the paper in Table 2
print(stats)
ray_comparisons
ray_effectsizes

################################################################################
# Continuing with the interaction, we can look at each model, and see whether 
# the ray casting approaches are significantly different from each other.
# We only allude to this in the paper because this is a general overview of
# which models are better, while in the paper we break this down into different
# conditions and behaviours.

models <- c("Uncorrected", "Corrected")

for(model in models){
  ts_grouped_model <- ts_grouped[ts_grouped$Model == model, ]
  
  print(model)
  # Summary stats
  summary_stats <- ts_grouped_model %>%
    group_by(Ray) %>%
    get_summary_stats(Error, show = c("median", "iqr"))
  
  print(summary_stats)
  
  plot(ggboxplot(ts_grouped_model, x = "Ray", y = "Error", add = "point"))
  
  
  friedman_result = friedman.test(Error ~ Ray | PARTICIPANT, data = ts_grouped_model)
  print(friedman_result)
  if(friedman_result$p.value < 0.05){
    wilcoxon_results <- ts_grouped_model %>% wilcox_test(Error ~ Ray, paired = TRUE, detailed = TRUE, p.adjust.method = "holm")
    print(wilcoxon_results)
    wilcoxon_effect_sizes <- ts_grouped_model %>% wilcox_effsize(Error ~ Ray, paired = TRUE)
    print(wilcoxon_effect_sizes)
  } else{
    print("Results not significant according to Friedman test")
  }
  print("#####################################################")
}
# This shows us that for the uncorrected models the rays are all significantly different
# with the exception of EFRC and HFRC. However, for corrected models all rays are 
# significantly different to each other (including EFRC and HFRC).

################################################################################
# Plotting
################################################################################

iqr_bars <- function(y, na.rm = TRUE) {
  if (na.rm) y <- na.omit(y)
  q1 <- quantile(y, 0.25)
  q3 <- quantile(y, 0.75)
  data.frame(ymin = q1, ymax = q3)
}

plot_two_conditions_box <- function(d, measure, measure_label, x_vals, x_group, x_label, legend_label, color_palette, dodge_width = 0.2){
  plot <- ggplot(d, aes(x = .data[[x_vals]], y = .data[[measure]], 
                        fill = .data[[x_group]], color = .data[[x_group]])) +
    geom_boxplot(width = 0.5, position = position_dodge(width=0.8), alpha = 0.3, linewidth = 0.8) + # Boxplot inside violin
    stat_summary(fun = median, geom = "line", aes(group = .data[[x_group]]), position = position_dodge(width=0.8), size = 1) + # Connecting lines between medians
    stat_summary(fun = median, geom = "point", position = position_dodge(width=0.8), size = 3) + # Median points
    scale_fill_manual(name = legend_label, values = color_palette) + # Custom fill colors
    scale_color_manual(name = legend_label, values = color_palette) + # Custom border colors
    labs(x = x_label, y = measure_label) +
    theme_minimal()
  
  return(plot)
}

plot_two_conditions <- function(d, measure, measure_label, x_vals, x_group, x_label, legend_label, color_palette, dodge_width = 0.2){
  dodge_width <- 0.4
  plot <- ggplot(d, aes(x = .data[[x_vals]], y = .data[[measure]], 
                        color = .data[[x_group]], group = .data[[x_group]])) +
    stat_summary(fun = median, geom = "line", position = position_dodge(width = dodge_width), size = 1) + # Offset lines
    stat_summary(fun = median, geom = "point", position = position_dodge(width = dodge_width), size = 3) + # Offset points
    stat_summary(fun.data = iqr_bars, geom = "errorbar", position = position_dodge(width = dodge_width), size = 1) + # Offset error bars
    scale_color_manual(name = legend_label, values = color_palette) + # Custom colors and legend title
    labs(x = x_label, y = measure_label) +
    theme_minimal()
  
  return(plot)
}

plot_stacked_columns <- function(df, measure, fill_label, label_one, colour_one, label_two, colour_two) {
  cols2group <- c("TARGET_COL_NAME", measure)
  
  # Summarize data
  df_summary <- df %>%
    group_by_at(cols2group) %>%
    summarise(count = n(), .groups = "drop") %>%  # Added .groups = "drop" to avoid warning
    group_by(TARGET_COL_NAME) %>%
    mutate(percentage = count / sum(count))
  
  # Plot the data
  plot <- ggplot(df_summary, aes(x = TARGET_COL_NAME, y = percentage, fill = as.factor(!!sym(measure)))) +
    geom_bar(stat = "identity") +
    scale_y_continuous(labels = scales::percent) +
    scale_fill_manual(
      values = c("0" = colour_one, "1" = colour_two),   # Optional: customize colors here
      labels = c("0" = label_one, "1" = label_two)  # Custom legend labels
    ) +
    labs(
      x = "Column\n(a)",
      y = "Percentage of Trials",
      fill = fill_label
    ) +
    theme_minimal()
  
  return (plot)
}

# group data by conditions
plot_condition_data <- get_grouped_data(ts, "ACC_REQ", "FOCUS_REQ")
plot_condition_data$ACC_REQ <- factor(plot_condition_data$ACC_REQ, levels = c("ACCURATE", "CASUAL"))
plot_condition_data$FOCUS_REQ <- factor(plot_condition_data$FOCUS_REQ, levels = c("FOCUSED", "DISTRACTED"))

# setup the levels and labels for plots
plot_font <- "Arial"
plot_focus_label <- "Focus"
plot_condition_labels <- c("ACCURATE" = "Accurate", "CASUAL" = "Casual")
plot_condition_colours <- c("FOCUSED" = "black", "DISTRACTED" = "blue")
plot_legend_labels_colour <- scale_color_manual(
  name = plot_focus_label,  # Legend title
  values = plot_condition_colours,  # Custom colors
  labels = c("DISTRACTED" = "Distracted", "FOCUSED" = "Focused")  # Custom labels
) 
plot_legend_labels_line <- scale_fill_manual (
  name = plot_focus_label,  # Legend title
  values = plot_condition_colours,  # Custom colors
  labels = c("DISTRACTED" = "Distracted", "FOCUSED" = "Focused")  # Custom labels
)

plot_finger_colours <- c("ADS" = "red", "UNOCCLUDED" = "black", "OCCLUDED" = "blue")
plot_arm_labels <- c("OUTSTRETCHED" = "Outstretched", 
                     "PROXIMAL" = "Proximal", 
                     "HIP-FIRE" = "Hip Fire")

plot_finger_colour_labels <- scale_color_manual(
  name = "Finger incorporation",  # Legend title
  values = plot_finger_colours,  # Custom colors
  labels = c("ADS" = "ADS",
             "OCCLUDED" = "Occluded", 
             "UNOCCLUDED" = "Unoccluded")
)

plot_arm_colours <- c("OUTSTRETCHED" = "red", "HIP-FIRE" = "black", "PROXIMAL" = "blue")
plot_orientation_labels <- c("MEDIAL" = "Medial",
                        "ALIGNED" = "Aligned", 
                        "LATERAL" = "Lateral")

plot_arm_colour_labels <- scale_color_manual(
  name = "Arm Pose",  # Legend title
  values = plot_arm_colours,  # Custom colors
  labels = c("OUTSTRETCHED" = "Outstretched", 
             "PROXIMAL" = "Proximal", 
             "HIP-FIRE" = "Hip Fire")
)


plot_column_label <- "Target Column"
plot_target_data <- get_grouped_data(ts, "TARGET_COL_NAME", "TARGET_ROW_NAME")
plot_target_colors <- c("TOP" = "red", "MIDDLE" = "black", "BOTTOM" = "blue")

plot_row_labels <- c("BOTTOM" = "Bottom",
                     "MIDDLE" = "Middle", 
                     "TOP" = "Top")

plot_column_labels <- c("LEFTMOST" = "Far left",
                        "LEFT" = "Left", 
                        "CENTRE" = "Middle",
                        "RIGHT" = "Right",
                        "RIGHTMOST" = "Far right")

plot_hand_colors <- c("LEFT" = "coral", "RIGHT" = "cyan3")
plot_hand_colour_labels_scale <- scale_color_manual(
  name = "Hand",  # Legend title
  values = plot_hand_colors,  # Custom colors
  labels = c("LEFT" = "Left", 
             "RIGHT" = "Right"
  )
)
plot_hand_colour_labels_fill <- scale_fill_manual(
  name = "Hand",  # Legend title
  values = plot_hand_colors,  # Custom colors
  labels = c("LEFT" = "Left", 
             "RIGHT" = "Right"
  )
)

################################################################################
# Manipulation Check
################################################################################

qd_wide <- read.csv("SubjectiveQuestionnaires.csv")

# Add participant labels and reshape to long format
qd <- qd_wide %>%
  mutate(PARTICIPANT = row_number()) %>%  # Add participant ID
  pivot_longer(cols = -PARTICIPANT,  
               names_to = c("Measure", "ACC_REQ", "FOCUS_REQ"),  
               names_pattern = "^(.*)_(ACCURATE|CASUAL)_(FOCUSED|DISTRACTED)$",  
               values_to = "Value") %>%
  pivot_wider(names_from = "Measure", values_from = "Value")  # Make measures separate columns

qd$ACC_REQ <- factor(qd$ACC_REQ)
qd$FOCUS_REQ <- factor(qd$FOCUS_REQ)
qd$PARTICIPANT <- factor(qd$PARTICIPANT)

qd <- qd %>%
  mutate(CONDITION = paste(ACC_REQ, FOCUS_REQ, sep = "-"))

qd <- qd %>%
  mutate(CONDITION = factor(CONDITION, levels = c("ACCURATE-FOCUSED",
                                                  "ACCURATE-DISTRACTED",
                                                  "CASUAL-FOCUSED",
                                                  "CASUAL-DISTRACTED"))) %>%
  arrange(CONDITION)

# View result
print(qd)

# The below are the results for Table 3 and Appendix D
run_manipulation_check(qd, "BRPE_GENERAL")

run_manipulation_check(qd, "NASATLX_EFFORT")
run_manipulation_check(qd, "NASATLX_FRUSTRATION")
run_manipulation_check(qd, "NASATLX_MENTAL_DEMAND")
run_manipulation_check(qd, "NASATLX_PHYSICAL_DEMAND")
run_manipulation_check(qd, "NASATLX_TEMPORAL_DEMAND")
run_manipulation_check(qd, "NASATLX_PERFORMANCE")

run_manipulation_check(qd, "IMI.EFFORT.IMPORTANCE_mean")
run_manipulation_check(qd, "IMI.INTEREST.ENJOYMENT_mean")
run_manipulation_check(qd, "IMI.PERCEIVED.COMPETENCE_mean")
run_manipulation_check(qd, "IMI.PRESSURE.TENSION_mean")


################################################################################
# OUTCOME MEASURES
################################################################################

# The below are results for Table 4 and Appendix E

################################################################################
# Temporal analysis
################################################################################

run_art_two_way(ts, "GESTURE_MS", "ACC_REQ", "FOCUS_REQ")
run_art_two_way(ts, "INITIAL_BALLISTIC_MS", "ACC_REQ", "FOCUS_REQ")
run_art_two_way(ts, "PINPOINTING_MS", "ACC_REQ", "FOCUS_REQ")
run_art_two_way(ts, "ACTIVE_MS", "ACC_REQ", "FOCUS_REQ")

################################################################################
# Biomechanical analysis
################################################################################

run_art_two_way(ts, "SHOULDER_TORQUE", "ACC_REQ", "FOCUS_REQ")
run_art_two_way(ts, "NICER_TOTAL_GESTURE", "ACC_REQ", "FOCUS_REQ")
run_art_two_way(ts, "DISTANCE_TRAVELLED_NORMALISED", "ACC_REQ", "FOCUS_REQ")
run_art_two_way(ts, "HAND_USED_PERCENTAGE", "ACC_REQ", "FOCUS_REQ")

################################################################################
# Accuracy analysis
################################################################################

run_art_two_way(ts, "efrc_cyclops_corrected_ols_replicate_errors", "ACC_REQ", "FOCUS_REQ")
run_art_two_way(ts, "hfrc_corrected_ols_replicate_errors", "ACC_REQ", "FOCUS_REQ")

run_art_two_way(ts, "ifrc_corrected_ols_replicate_errors", "ACC_REQ", "FOCUS_REQ")
run_art_two_way(ts, "frc_corrected_ols_replicate_errors", "ACC_REQ", "FOCUS_REQ")


################################################################################
# BEHAVIOUR TABLES
################################################################################

measures <- c("GESTURE_MS", 
              "INITIAL_BALLISTIC_MS", 
              "PINPOINTING_MS", 
              "ACTIVE_MS", 
              "SHOULDER_TORQUE",
              "NICER_TOTAL_GESTURE", 
              "DISTANCE_TRAVELLED_NORMALISED",
              "ifrc_corrected_ols_replicate_errors", 
              "frc_corrected_ols_replicate_errors",
              "efrc_cyclops_corrected_ols_replicate_errors",
              "hfrc_corrected_ols_replicate_errors")

# The below are the results for tables 6, 7, and 8
get_behaviour_table(ts,"ARM_POSE_CLASS", measures)
get_behaviour_table(ts,"FINGER_INCORPORATION_CLASS", measures)
get_behaviour_table(ts,"GESTURE_ORIENTATION_CLASS", measures)


# The below is for the trial counts for behaviour combinations seen in Tables 5 and 9
# Table 5
for (arm_pose in unique(ts$ARM_POSE_CLASS)) {
  print(arm_pose)
  arm_pose_group <- ts[ts$ARM_POSE_CLASS == arm_pose, ]
  trial_count <- arm_pose_group %>% 
    count(FINGER_INCORPORATION_CLASS, name = "count") %>% 
    mutate(count =  paste0(count, " (", round(100 *count/sum(count), 1), "%)")) %>%
    ungroup
  names(trial_count)[names(trial_count) == 'count'] <- 'Number of trials'
  names(trial_count)[names(trial_count) == "FINGER_INCORPORATION_CLASS"] <- 'Finger Incorporation'
  print(trial_count)
}

# Table 9
for (arm_pose in unique(ts$ARM_POSE_CLASS)) {
  print(arm_pose)
  arm_pose_group <- ts[ts$ARM_POSE_CLASS == arm_pose, ]
  trial_count <- arm_pose_group %>% 
    count(GESTURE_ORIENTATION_CLASS, name = "count") %>% 
    mutate(count =  paste0(count, " (", round(100 *count/sum(count), 1), "%)")) %>%
    ungroup
  names(trial_count)[names(trial_count) == 'count'] <- 'Number of trials'
  names(trial_count)[names(trial_count) == "GESTURE_ORIENTATION_CLASS"] <- 'Alignment'
  print(trial_count)
}
  
  
################################################################################
# Figures
################################################################################
# The below generates the graphs within Sections 4 and 5.

#############################################
## Systematic Offset Compensation - Figure 6
#############################################

plot_compensation_colours <- c("Corrected" = "brown3", "Uncorrected" = "darkcyan")

ts_grouped$Model <- factor(ts_grouped$Model, levels = c("Uncorrected", "Corrected"))

ylim_min <- 0
ylim_max <- 18
ts_grouped_eye <- ts_grouped[ts_grouped$Ray == "EFRC (Cyclops)" | ts_grouped$Ray == "HFRC", ]
eye_rays <- plot_two_conditions_box(ts_grouped_eye, "Error", "Error (degrees)", "Ray", "Model", "Ray", "Correction", plot_compensation_colours) + coord_cartesian(ylim = c(ylim_min, ylim_max))
eye_rays$labels$x <- "Head-Rooted Rays\n(a)"

ylim_min <- 0
ylim_max <- 45
ts_grouped_hand <- ts_grouped[ts_grouped$Ray == "IFRC" | ts_grouped$Ray == "FRC", ]
hand_rays <- plot_two_conditions_box(ts_grouped_hand, "Error", "Error (degrees)", "Ray", "Model", "Ray", "Correction", plot_compensation_colours) + coord_cartesian(ylim = c(ylim_min, ylim_max))
hand_rays$labels$x <- "Hand-Rooted Rays\n(b)"

compensation_plot <- (eye_rays + hand_rays) +
  plot_layout(guides = "collect") & 
  theme(legend.position = "top", legend.location = "plot", legend.text = element_text(family=plot_font, size=32), text = element_text(family=plot_font, size=34), axis.text.x = element_text(family=plot_font, size=30, angle = 45, vjust=1, hjust=1), axis.text.y = element_text(family=plot_font, size=28))

print(compensation_plot)

#############################################
## Temporal - Figure 7
#############################################

## Used to generate Figure ...
ges_plot <- plot_two_conditions_box(plot_condition_data, "GESTURE_MS", "Overall Gesture Duration (ms)", "ACC_REQ", "FOCUS_REQ", "Pointing Style\n(a)", plot_focus_label, plot_condition_colours)
init_plot <- plot_two_conditions_box(plot_condition_data, "INITIAL_BALLISTIC_MS", "Initial Ballistic Duration (ms)", "ACC_REQ", "FOCUS_REQ", "Pointing Style\n(b)", plot_focus_label, plot_condition_colours)
hold_plot <- plot_two_conditions_box(plot_condition_data, "PINPOINTING_MS", "Hold Duration (ms)", "ACC_REQ", "FOCUS_REQ", "Pointing Style\n(c)", plot_focus_label, plot_condition_colours)
act_plot <- plot_two_conditions_box(plot_condition_data, "ACTIVE_MS", "Selection Time (ms)", "ACC_REQ", "FOCUS_REQ", "Pointing Style\n(d)", plot_focus_label, plot_condition_colours)

ges_plot <- ges_plot + scale_x_discrete(labels = plot_condition_labels) + plot_legend_labels_colour + plot_legend_labels_line
init_plot <- init_plot + scale_x_discrete(labels = plot_condition_labels) + plot_legend_labels_colour + plot_legend_labels_line
hold_plot <- hold_plot + scale_x_discrete(labels = plot_condition_labels) + plot_legend_labels_colour + plot_legend_labels_line
act_plot <- act_plot + scale_x_discrete(labels = plot_condition_labels) + plot_legend_labels_colour + plot_legend_labels_line

temporal_condition_plot <- (ges_plot + init_plot + hold_plot + act_plot) +
  plot_layout(guides = "collect", ncol = 4) & 
  theme(legend.position = "top", legend.location = "plot", legend.text = element_text(family=plot_font, size=32), text = element_text(family=plot_font, size=34), axis.text.x = element_text(family=plot_font, size=30, angle = 45, vjust=1, hjust=1), axis.text.y = element_text(family=plot_font, size=28))

print(temporal_condition_plot)

#############################################
## Biomechancial - Figure 8
#############################################

st_plot <- plot_two_conditions_box(plot_condition_data, "SHOULDER_TORQUE", "Shoulder Torque (Nm)", "ACC_REQ", "FOCUS_REQ", "Pointing Style\n(a)", plot_focus_label, plot_condition_colours)
nicer_plot <- plot_two_conditions_box(plot_condition_data, "NICER", "NICER (%)", "ACC_REQ", "FOCUS_REQ", "Pointing Style\n(b)", plot_focus_label, plot_condition_colours)
dtn_plot <- plot_two_conditions_box(plot_condition_data, "DISTANCE_TRAVELLED_NORMALISED", "Normalised Hand Movement", "ACC_REQ", "FOCUS_REQ", "Pointing Style\n(c)", plot_focus_label, plot_condition_colours)
hand_plot <- plot_two_conditions_box(plot_condition_data, "HAND_USED", "Right hand %", "ACC_REQ", "FOCUS_REQ", "Pointing Style\n(d)", plot_focus_label, plot_condition_colours)

st_plot <- st_plot + scale_x_discrete(labels = plot_condition_labels) + plot_legend_labels_colour + plot_legend_labels_line
nicer_plot <- nicer_plot + scale_x_discrete(labels = plot_condition_labels) + plot_legend_labels_colour + plot_legend_labels_line
dtn_plot <- dtn_plot + scale_x_discrete(labels = plot_condition_labels) + plot_legend_labels_colour + plot_legend_labels_line
hand_plot <- hand_plot + scale_x_discrete(labels = plot_condition_labels) + plot_legend_labels_colour + plot_legend_labels_line

biomechanical_plot <- (st_plot + nicer_plot + dtn_plot + hand_plot) + 
  plot_layout(guides = "collect", ncol = 4) & 
  theme(legend.position = "top", legend.location = "plot", legend.text = element_text(family=plot_font, size=32), text = element_text(family=plot_font, size=34), axis.text.x = element_text(family=plot_font, size=30, angle = 45, vjust=1, hjust=1), axis.text.y = element_text(family=plot_font, size=28))

print(biomechanical_plot)

#############################################
## Accuracy - Figure 9
#############################################

ylim_min <- 0
ylim_max <- 12.5
efrc_plot <- plot_two_conditions_box(plot_condition_data, "efrc_cyclops_corrected_ols_replicate_errors", "EFRC Cyclops Error (degrees)", "ACC_REQ", "FOCUS_REQ", "Pointing Style\n(a)", plot_focus_label, plot_condition_colours)
hfrc_plot <- plot_two_conditions_box(plot_condition_data, "hfrc_corrected_ols_replicate_errors", "HFRC Error (degrees)", "ACC_REQ", "FOCUS_REQ", "Pointing Style\n(b)", plot_focus_label, plot_condition_colours)
efrc_plot <- efrc_plot + coord_cartesian(ylim = c(ylim_min, ylim_max))
hfrc_plot <- hfrc_plot + coord_cartesian(ylim = c(ylim_min, ylim_max))


ylim_min <- 0
ylim_max <- 25
ifrc_plot <- plot_two_conditions_box(plot_condition_data, "ifrc_corrected_ols_replicate_errors", "IFRC Error (degrees)", "ACC_REQ", "FOCUS_REQ", "Pointing Style\n(c)", plot_focus_label, plot_condition_colours)
frc_plot <- plot_two_conditions_box(plot_condition_data, "frc_corrected_ols_replicate_errors", "FRC Error (degrees)", "ACC_REQ", "FOCUS_REQ", "Pointing Style\n(d)", plot_focus_label, plot_condition_colours)
ifrc_plot <- ifrc_plot + coord_cartesian(ylim = c(ylim_min, ylim_max))
frc_plot <- frc_plot + coord_cartesian(ylim = c(ylim_min, ylim_max))

efrc_plot <- efrc_plot + scale_x_discrete(labels = plot_condition_labels) + plot_legend_labels_colour + plot_legend_labels_line
hfrc_plot <- hfrc_plot + scale_x_discrete(labels = plot_condition_labels) + plot_legend_labels_colour + plot_legend_labels_line
ifrc_plot <- ifrc_plot + scale_x_discrete(labels = plot_condition_labels) + plot_legend_labels_colour + plot_legend_labels_line
frc_plot <- frc_plot + scale_x_discrete(labels = plot_condition_labels) + plot_legend_labels_colour + plot_legend_labels_line

accuracy_plot <- efrc_plot + hfrc_plot + ifrc_plot + frc_plot +
  plot_layout(ncol = 4, guides = "collect") & 
  theme(legend.position = "top", legend.location = "plot", legend.text = element_text(family=plot_font, size=32), text = element_text(family=plot_font, size=34), axis.text.x = element_text(family=plot_font, size=30, angle = 45, vjust=1, hjust=1), axis.text.y = element_text(family=plot_font, size=28))

plot(accuracy_plot)


#############################################
## Behaviour Accuracy - Figure 12
#############################################
pose_data <- get_grouped_data(ts, "FINGER_INCORPORATION_CLASS", "ARM_POSE_CLASS")

ifrc_plot <- plot_two_conditions(ts, "ifrc_corrected_ols_replicate_errors", "IFRC Error (º)", "ARM_POSE_CLASS", "FINGER_INCORPORATION_CLASS", "Arm pose\n(a)", "Finger incorporation", plot_finger_colours)
frc_plot <- plot_two_conditions(ts, "frc_corrected_ols_replicate_errors", "FRC Error (º)", "ARM_POSE_CLASS", "FINGER_INCORPORATION_CLASS", "Arm pose\n(b)", "Finger incorporation", plot_finger_colours)
hfrc_plot <- plot_two_conditions(ts, "hfrc_corrected_ols_replicate_errors", "HFRC Error (º)", "ARM_POSE_CLASS", "FINGER_INCORPORATION_CLASS", "Arm pose\n(c)", "Finger incorporation", plot_finger_colours)
efrc_plot <- plot_two_conditions(ts, "efrc_cyclops_corrected_ols_replicate_errors", "EFRC (Cyclops) Error (º)", "ARM_POSE_CLASS", "FINGER_INCORPORATION_CLASS", "Arm pose\n(d)", "Finger incorporation", plot_finger_colours)

min_lim <- 1
max_lim <- 28
ifrc_plot <- ifrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))
frc_plot <- frc_plot + coord_cartesian(ylim = c(min_lim, max_lim))

max_lim <- 26
efrc_plot <- efrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))
hfrc_plot <- hfrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))

ifrc_plot <- ifrc_plot + scale_x_discrete(labels = plot_arm_labels) + plot_finger_colour_labels
frc_plot <- frc_plot + scale_x_discrete(labels = plot_arm_labels) + plot_finger_colour_labels
hfrc_plot <- hfrc_plot + scale_x_discrete(labels = plot_arm_labels) + plot_finger_colour_labels
efrc_plot <- efrc_plot + scale_x_discrete(labels = plot_arm_labels) + plot_finger_colour_labels

arm_finger_accuracy_plot <- (ifrc_plot + frc_plot + hfrc_plot + efrc_plot) +
  plot_layout(guides = "collect", ncol = 4) & 
  theme(legend.position = "top", legend.location = "plot", legend.text = element_text(family=plot_font, size=32), text = element_text(family=plot_font, size=34), axis.text.x = element_text(family=plot_font, size=30, angle = 45, vjust=1, hjust=1), axis.text.y = element_text(family=plot_font, size=28))

print(arm_finger_accuracy_plot)


#############################################
## Behaviour Accuracy - Figure 14
#############################################
pose_data <- get_grouped_data(ts, "GESTURE_ORIENTATION_CLASS", "ARM_POSE_CLASS")

ifrc_plot <- plot_two_conditions(ts, "ifrc_corrected_ols_replicate_errors", "IFRC Error (º)", "GESTURE_ORIENTATION_CLASS", "ARM_POSE_CLASS", "Arm pose\n(a)", "Arm pose\n(a)", plot_arm_colours)
frc_plot <- plot_two_conditions(ts, "frc_corrected_ols_replicate_errors", "FRC Error (º)", "GESTURE_ORIENTATION_CLASS", "ARM_POSE_CLASS", "Arm pose\n(b)", "Hand Torso Alignment", "Arm pose\n(b)", plot_arm_colours)
hfrc_plot <- plot_two_conditions(ts, "hfrc_corrected_ols_replicate_errors", "HFRC Error (º)", "GESTURE_ORIENTATION_CLASS", "ARM_POSE_CLASS", "Arm pose\n(c)", "Hand Torso Alignment", "Arm pose\n(c)", plot_arm_colours)
efrc_plot <- plot_two_conditions(ts, "efrc_cyclops_corrected_ols_replicate_errors", "EFRC (Cyclops) Error (º)", "GESTURE_ORIENTATION_CLASS", "ARM_POSE_CLASS", "Arm pose\n(d)", "Hand Torso Alignment", "Arm pose\n(d)", plot_arm_colours)

min_lim <- 1
max_lim <- 30
ifrc_plot <- ifrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))
frc_plot <- frc_plot + coord_cartesian(ylim = c(min_lim, max_lim))

max_lim <- 30
efrc_plot <- efrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))
hfrc_plot <- hfrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))

ifrc_plot <- ifrc_plot + scale_x_discrete(labels = plot_orientation_labels) + plot_arm_colour_labels
frc_plot <- frc_plot + scale_x_discrete(labels = plot_orientation_labels) + plot_arm_colour_labels
hfrc_plot <- hfrc_plot + scale_x_discrete(labels = plot_orientation_labels) + plot_arm_colour_labels
efrc_plot <- efrc_plot + scale_x_discrete(labels = plot_orientation_labels) + plot_arm_colour_labels

alignment_arm_accuracy_plot <- (ifrc_plot + frc_plot + hfrc_plot + efrc_plot) +
  plot_layout(guides = "collect", ncol = 4) & 
  theme(legend.position = "top", legend.location = "plot", legend.text = element_text(family=plot_font, size=32), text = element_text(family=plot_font, size=34), axis.text.x = element_text(family=plot_font, size=30, angle = 45, vjust=1, hjust=1), axis.text.y = element_text(family=plot_font, size=28))

print(alignment_arm_accuracy_plot)


#############################################
## Behaviour Accuracy - Figure 15
#############################################
pose_data <- get_grouped_data(ts, "TARGET_COL_NAME", "ARM_POSE_CLASS")

ifrc_plot <- plot_two_conditions(ts, "ifrc_corrected_ols_replicate_errors", "IFRC Error (º)", "TARGET_COL_NAME", "ARM_POSE_CLASS", "Arm pose\n(a)", "Arm pose\n(a)", plot_arm_colours)
frc_plot <- plot_two_conditions(ts, "frc_corrected_ols_replicate_errors", "FRC Error (º)", "TARGET_COL_NAME", "ARM_POSE_CLASS", "Arm pose\n(b)", "Row", "Arm pose\n(b)", plot_arm_colours)
hfrc_plot <- plot_two_conditions(ts, "hfrc_corrected_ols_replicate_errors", "HFRC Error (º)", "TARGET_COL_NAME", "ARM_POSE_CLASS", "Arm pose\n(c)", "Row", "Arm pose\n(c)", plot_arm_colours)
efrc_plot <- plot_two_conditions(ts, "efrc_cyclops_corrected_ols_replicate_errors", "EFRC (Cyclops) Error (º)", "TARGET_COL_NAME", "ARM_POSE_CLASS", "Arm pose\n(d)", "Row", "Arm pose", plot_arm_colours)

min_lim <- 1
max_lim <- 32
ifrc_plot <- ifrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))
frc_plot <- frc_plot + coord_cartesian(ylim = c(min_lim, max_lim))
efrc_plot <- efrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))
hfrc_plot <- hfrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))

ifrc_plot <- ifrc_plot + scale_x_discrete(labels = plot_column_labels) + plot_arm_colour_labels
frc_plot <- frc_plot + scale_x_discrete(labels = plot_column_labels) + plot_arm_colour_labels
hfrc_plot <- hfrc_plot + scale_x_discrete(labels = plot_column_labels) + plot_arm_colour_labels
efrc_plot <- efrc_plot + scale_x_discrete(labels = plot_column_labels) + plot_arm_colour_labels

arm_column_accuracy_plot <- (ifrc_plot + frc_plot + hfrc_plot + efrc_plot) +
  plot_layout(guides = "collect", ncol = 4) & 
  theme(legend.position = "top", legend.location = "plot", legend.text = element_text(family=plot_font, size=32), text = element_text(family=plot_font, size=34), axis.text.x = element_text(family=plot_font, size=30, angle = 45, vjust=1, hjust=1), axis.text.y = element_text(family=plot_font, size=28))

print(arm_column_accuracy_plot)


#############################################
## Behaviour Accuracy - Figure 16(a)
#############################################

ts$hand_var <- as.numeric(ts$HAND) - 1
hand_used_plot <- plot_stacked_columns(ts, "hand_var", "Hand Used", "Left", "coral", "Right", "cyan3")
hand_used_plot <- hand_used_plot + scale_x_discrete(labels = plot_column_labels)

hand_used_plot <- (hand_used_plot) +
  plot_layout(guides = "collect", ncol = 1) & 
  theme(legend.position = "top", legend.location = "plot", legend.text = element_text(family=plot_font, size=32), text = element_text(family=plot_font, size=34), axis.text.x = element_text(family=plot_font, size=30, angle = 45, vjust=1, hjust=1), axis.text.y = element_text(family=plot_font, size=28))

print(hand_used_plot)

#############################################
## Behaviour Accuracy - Figure 16(b)
#############################################

plot_data <- ts %>%
  group_by_at(c("PARTICIPANT", "HAND", "TARGET_COL_NAME")) %>%
  summarise(across(where(is.numeric), mean), .groups = "drop") %>%
  ungroup()

hand_hfrc_plot <- plot_two_conditions_box(plot_data, "hfrc_corrected_ols_replicate_errors", "HFRC Error (º)", "TARGET_COL_NAME", "HAND", "Column\n(b)", "Hand", colors)

min_lim <- 1
max_lim <- 10.5
hand_hfrc_plot <- hand_hfrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))
hand_hfrc_plot <- hand_hfrc_plot + scale_x_discrete(labels = plot_column_labels) + plot_hand_colour_labels_scale + plot_hand_colour_labels_fill


hand_column_accuracy_plot <- (hand_hfrc_plot) +
  plot_layout(guides = "collect", ncol = 1) & 
  theme(legend.position = "top", legend.location = "plot", legend.text = element_text(family=plot_font, size=32), text = element_text(family=plot_font, size=34), axis.text.x = element_text(family=plot_font, size=30, angle = 45, vjust=1, hjust=1), axis.text.y = element_text(family=plot_font, size=28))

print(hand_column_accuracy_plot)

#############################################
## Behaviour Accuracy - Figure 17
#############################################
pose_data <- get_grouped_data(ts, "TARGET_ROW_NAME", "ARM_POSE_CLASS")

ifrc_plot <- plot_two_conditions(ts, "ifrc_corrected_ols_replicate_errors", "IFRC Error (º)", "TARGET_ROW_NAME", "ARM_POSE_CLASS", "Arm pose\n(a)", "Arm pose\n(a)", plot_arm_colours)
frc_plot <- plot_two_conditions(ts, "frc_corrected_ols_replicate_errors", "FRC Error (º)", "TARGET_ROW_NAME", "ARM_POSE_CLASS", "Arm pose\n(b)", "Row", "Arm pose\n(b)", plot_arm_colours)
hfrc_plot <- plot_two_conditions(ts, "hfrc_corrected_ols_replicate_errors", "HFRC Error (º)", "TARGET_ROW_NAME", "ARM_POSE_CLASS", "Arm pose\n(c)", "Row", "Arm pose\n(c)", plot_arm_colours)
efrc_plot <- plot_two_conditions(ts, "efrc_cyclops_corrected_ols_replicate_errors", "EFRC (Cyclops) Error (º)", "TARGET_ROW_NAME", "ARM_POSE_CLASS", "Arm pose\n(d)", "Row", "Arm pose", plot_arm_colours)

min_lim <- 1
max_lim <- 46
ifrc_plot <- ifrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))
frc_plot <- frc_plot + coord_cartesian(ylim = c(min_lim, max_lim))
efrc_plot <- efrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))
hfrc_plot <- hfrc_plot + coord_cartesian(ylim = c(min_lim, max_lim))

ifrc_plot <- ifrc_plot + scale_x_discrete(labels = plot_row_labels) + plot_arm_colour_labels
frc_plot <- frc_plot + scale_x_discrete(labels = plot_row_labels) + plot_arm_colour_labels
hfrc_plot <- hfrc_plot + scale_x_discrete(labels = plot_row_labels) + plot_arm_colour_labels
efrc_plot <- efrc_plot + scale_x_discrete(labels = plot_row_labels) + plot_arm_colour_labels

arm_row_accuracy_plot <- (ifrc_plot + frc_plot + hfrc_plot + efrc_plot) +
  plot_layout(guides = "collect", ncol = 4) & 
  theme(legend.position = "top", legend.location = "plot", legend.text = element_text(family=plot_font, size=32), text = element_text(family=plot_font, size=34), axis.text.x = element_text(family=plot_font, size=30, angle = 45, vjust=1, hjust=1), axis.text.y = element_text(family=plot_font, size=28))

print(arm_row_accuracy_plot)
