# imports
import sys
from os.path import isfile, isdir, join, realpath, dirname
import warnings
import traceback

script_root = dirname(realpath(__file__))
sys.path.insert(1, script_root)

## Functional
from argparse import ArgumentParser
from pathlib import Path
from os import listdir
import os
import re
import copy
import numpy as np
import pandas as pd
import json
from itertools import combinations
import scipy.ndimage as sn

## data
from dependencies.Targets import Target

## Usability
from rich.table import Table
from rich.console import Console
from rich.progress import Progress

# Parallel Processing
from concurrent.futures import ThreadPoolExecutor, as_completed
from queue import Queue

# Static
## Regex
participant_regex = re.compile(r'^.*[A-Za-z0-9]+[-][A-Za-z0-9]+[-][A-Za-z0-9]+[-][A-Za-z0-9]+[-][A-Za-z0-9]+.*')
int_regex = re.compile(r"^[0-9]+$")
range_regex = re.compile(r"^[0-9]+[-][0-9]+$")
not_regex = re.compile(r"^[!].+$")
all_regex = re.compile(r"^[aA]([lL][lL])?$")
confirm_regex = re.compile(r"^[yY]([eE][sS])?$")
reject_regex = re.compile(r"^[nN][oO]?$")

console = Console()

HVP_KEY = "Hand Velocity Peaks"
R_KEY = "RH_R"
L_KEY = "LH_L"
R_VEL_KEY = "Right Hand"
L_VEL_KEY = "Left Hand"
IGNORE_HAND_KEY = "Ignore"
START_KEY = "Peak Starts"
END_KEY = "Peak Ends"
MANUAL_ANNOTATIONS_KEY = "Manual Annotations"
INITIAL_BALLISTIC_KEY = "Start Peak"
TERMINAL_BALLISTIC_KEY = "End Peak"
TARGET_KEY = "TARGETS"
    
conditions = [
    ("ACCURATE", "FOCUSED"),
    ("ACCURATE", "DISTRACTED"),
    ("CASUAL", "FOCUSED"),
    ("CASUAL", "DISTRACTED")
]

## Helpers
def generate_participant_table(participants):
    participants_table = Table()
    participants_table.add_column("#")
    participants_table.add_column("Selected")
    participants_table.add_column("Participant ID")

    for idx, row in enumerate(participants):
        if type(row) is tuple:
            participant = row[0]
            selected = row[1]
        else:
            participant = row
            selected = False
        selected_colour = "green" if selected else "red"
        participants_table.add_row(
            str(idx),
            f"[{selected_colour}]{selected}[/{selected_colour}]",
            participant
        )
    return participants_table

def update_missing_trials(missing_trials, condition, key, idx):
    condition_label = condition
    if condition_label not in missing_trials:
        missing_trials[condition_label] = {}
    if key not in missing_trials[condition_label]:
        missing_trials[condition_label][key] = {"COUNT": 0, "IDs": []}
    missing_trials[condition_label][key]["COUNT"] = missing_trials[condition_label][key]["COUNT"] + 1
    missing_trials[condition_label][key]["IDs"].append(idx)

# load trial data and participant metadata
def load_participant_data(participant, missing_trials, input_dir):
    metadata = None
    metadata_path = join(dirname(input_dir), "GestureAnnotations.json")
    if isfile(metadata_path):
        try:
            with open(metadata_path, 'r') as metadata_file:
                metadata = json.load(metadata_file)
                metadata = [md for md in metadata if md["ID"] == participant]
                if len(metadata) == 1:
                    metadata = metadata[0]
                else:
                    print(f"Unable to find participant ({participant}) in metadata {len(metadata)}")
                    metadata = None
        except Exception:
            print(f"Unable to load metadata file ({metadata_path})")
    else:
        print(f"Unable to find metadata file ({metadata_path}) for participant: {participant}")
    
    if metadata is None:
        return None, None
    
    participant_metadata = {k: v for k, v in metadata.items() if k != "Conditions"}
    participant_metadata["Conditions"] = {}
    participant_conditions = copy.deepcopy(metadata["Conditions"])

    pointing_trials = {}
    # read trials in conditions, load the dataframes
    for condition_data in participant_conditions:    
        condition = condition_data["Condition"]
        condition_metadata = {}
        print(f"Retrieving data for {participant} - {condition}")

        condition_label = condition.replace("-", "_")
        trials = condition_data["Trials"]
        loaded_condition_trials = set()
        
        for trial in trials:
            trial_path = join(input_dir, trial["Data"])
            trial_idx = trial["Trial Idx"]

            condition_metadata[trial_idx] = copy.deepcopy(trial)
                
            if isfile(trial_path):
                trial["DATA"] = pd.read_csv(trial_path)
                loaded_condition_trials.add(trial_idx)
            else:
                print(f"Unable to find file at: {trial_path}")
                if condition_label not in missing_trials:
                    missing_trials[condition_label] = {}
                update_missing_trials(missing_trials, condition, "TRIAL_RECORDING_MISSING", trial_idx)

        participant_metadata["Conditions"][condition] = condition_metadata

        missing_trial_list = list(set(range(0, 3*45)) - loaded_condition_trials)
        missing_trial_count = len(missing_trial_list)
        if missing_trial_count > 0:
            if condition_label not in missing_trials:
                missing_trials[condition_label] = {}
            missing_trials[condition_label]["TRIAL_RECORDING_MISSING"] = { "COUNT": missing_trial_count, "IDs": missing_trial_list }
        pointing_trials[condition] = trials.copy()
    
    return pointing_trials, participant_metadata

class json_serialize(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, np.integer):
            return int(obj)
        if isinstance(obj, np.floating):
            return float(obj)
        if isinstance(obj, np.ndarray):
            return obj.tolist()
        return json.JSONEncoder.default(self, obj)

# get the angular difference between 2 vectors
def get_vector_cosine_from_angles(a, b):
    def get_vector_cosine(x, y):
        d = np.dot(x, y)
        return np.rad2deg(np.arccos(d))
    a_yaw = np.deg2rad(a[1])
    a_pitch = np.deg2rad(a[0])
    b_yaw = np.deg2rad(b[1])
    b_pitch = np.deg2rad(b[0])
    x = np.array([np.cos(a_pitch) * np.cos(a_yaw), np.cos(a_pitch) * np.sin(a_yaw), np.sin(a_pitch)])
    y = np.array([np.cos(b_pitch) * np.cos(b_yaw), np.cos(b_pitch) * np.sin(b_yaw), np.sin(b_pitch)])

    return get_vector_cosine(x, y)

def discrete_feature_extraction(trial_data, participant, participant_metadata, condition, idx, trial_metadata, missing_trials):
    def get_duration(start, end):
        return np.array(range(start, end+1 if end==start else end, 1))
    
    # get the yaw and pitch from a 3d vector
    def get_vector_orientation(v0, inverse):
        yaw = np.rad2deg(np.arcsin(v0[1]))
        if inverse:
            yaw = 0 - yaw
        pitch = np.rad2deg(np.arcsin(v0[2]))
        return yaw, pitch

    # get the difference between the yaw and pitch of two vectors
    def get_angle_between_vectors(v0, v1, inverse):
        v0_yaw, v0_pitch = get_vector_orientation(v0, inverse)
        v1_yaw, v1_pitch = get_vector_orientation(v1, inverse)

        return v1_yaw - v0_yaw, v1_pitch - v0_pitch

    # get the difference between the yaw and pitch of 2 rays.
    def get_vector_alignment(prefix, features, vec_1_label, vec_2_label):
        if f"{prefix}{vec_1_label}_YAW_MEDIAN" not in features or f"{prefix}{vec_2_label}_YAW_MEDIAN" not in features:
            features[f"{prefix}{vec_1_label}_{vec_2_label}_ALIGNMENT_YAW"] = np.nan
            features[f"{prefix}{vec_1_label}_{vec_2_label}_ALIGNMENT_PITCH"] = np.nan
            return (np.nan, np.nan)
        features[f"{prefix}{vec_1_label}_{vec_2_label}_ALIGNMENT_YAW"] = features[f"{prefix}{vec_1_label}_YAW_MEDIAN"] - features[f"{prefix}{vec_2_label}_YAW_MEDIAN"]

        features[f"{prefix}{vec_1_label}_{vec_2_label}_ALIGNMENT_PITCH"] = features[f"{prefix}{vec_1_label}_PITCH_MEDIAN"] - features[f"{prefix}{vec_2_label}_PITCH_MEDIAN"]
        return (features[f"{prefix}{vec_1_label}_{vec_2_label}_ALIGNMENT_YAW"], features[f"{prefix}{vec_1_label}_{vec_2_label}_ALIGNMENT_PITCH"])

    # get the centre of mass and additional values for comptuation of shoulder torque
    def get_centre_of_mass(frames: pd.DataFrame, hand_key, body_key, gender):
        def get_vector_cosine(x, y):
            d = np.dot(x, y)
            return np.rad2deg(np.arccos(d))
        
        r = np.zeros((frames.shape[0], 3)) 
        r_mag = np.zeros(frames.shape[0])
        CoM = np.zeros((frames.shape[0], 3))
        shoulder_position = np.zeros(((frames.shape[0], 3)))
        shoulder_abduction = np.zeros(frames.shape[0])
        elbow_extension = np.zeros(frames.shape[0])
        for i in range(frames.shape[0]):
            frame = frames.iloc[i, :]
            shoulder_labels = [f"{body_key}_uarm {axis}" for axis in ["X", "Y", "Z"]]
            elbow_labels = [f"{body_key}_larm {axis}" for axis in ["X", "Y", "Z"]]
            wrist_labels = [f"{body_key}_hand {axis}" for axis in ["X", "Y", "Z"]]
            hand_labels = (
                [f"{hand_key}HandIn {axis}" for axis in ["X", "Y", "Z"]], 
                [f"{hand_key}HandOut {axis}" for axis in ["X", "Y", "Z"]]
            )
            
            # divide by 1000 to convert to meters
            shoulder_pos = frame[shoulder_labels].values / 1000
            elbow_pos = frame[elbow_labels].values / 1000
            wrist_pos = frame[wrist_labels].values / 1000
            hand_pos = ((frame[hand_labels[0]].values + frame[hand_labels[1]].values) / 2) / 1000

            hand_mass = 0.4
            if gender == "Male":
                ua_mass = 2.1
                fa_mass = 1.2
            else:
                ua_mass = 1.7
                fa_mass = 1
            la_mass = hand_mass + fa_mass
            arm_mass = hand_mass + fa_mass + ua_mass

            hand_com_ratio = 0.397
            fa_com_ratio = 0.424
            ua_com_ration = 0.452
            la_com_ratio = hand_mass / (la_mass)
            arm_com_ratio = la_mass / (arm_mass)

            # arm CoM components
            A = shoulder_pos + (ua_com_ration * (elbow_pos - shoulder_pos)) # Upper-arm CoM
            B = elbow_pos + (fa_com_ratio * (wrist_pos - elbow_pos)) # Forearm CoM
            C = wrist_pos + (hand_com_ratio * (hand_pos - wrist_pos)) # hand CoM

            D = B + (la_com_ratio * (C-B))
            CoM[i, :] = A + (arm_com_ratio * (D-A))
            
            r[i] = CoM[i] - shoulder_pos
            r_mag[i] = np.abs(np.linalg.norm(r))
            
            ua_vec = (elbow_pos) - (shoulder_pos)
            ua_vec = ua_vec / np.linalg.norm(ua_vec)
            fa_vec = (wrist_pos) - (elbow_pos)
            fa_vec = fa_vec / np.linalg.norm(fa_vec)

            # used to normalised angles relative to the sagittal axis
            normalised_to = ("pelvis", "l_uarm", "r_uarm")
            p0 = frame[[f"{normalised_to[0]} {axis}" for axis in ["X", "Y", "Z"]]].values
            p1 = frame[[f"{normalised_to[1]} {axis}" for axis in ["X", "Y", "Z"]]].values - p0
            p1 = p1 / np.linalg.norm(p1)
            p2 = frame[[f"{normalised_to[2]} {axis}" for axis in ["X", "Y", "Z"]]].values - p0
            p2 = p2 / np.linalg.norm(p2)
            normal = np.cross(p1, p2)
            
            shoulder_flexion, shoulder_abduction[i] = get_angle_between_vectors(normal, ua_vec, False)
            elbow_extension[i] = get_vector_cosine(ua_vec, fa_vec) # angle
            shoulder_position[i, :] = shoulder_pos

        return r, r_mag, CoM, shoulder_position, shoulder_abduction+90, elbow_extension
    
    # derive the torque at the shoulder in the current gesture
    def get_shoulder_torque(r, r_mag, coms_pos, shoulders, gender):
        mass = (3.7 if gender == "Male" else 3.1)
        g = np.array([0,0,9.81])
        com_vel = np.diff(coms_pos, axis=0) / 0.01
        com_vel = sn.median_filter(com_vel, size=3, mode="mirror")
        com_acc = np.diff(com_vel, axis=0) / 0.01
        com_acc = sn.median_filter(com_acc, size=3, mode="mirror")

        # https://dl.acm.org/doi/pdf/10.1145/2556288.2557130 - Eq. 14
        moment_of_inertia = 0.0201

        torque = np.zeros(com_acc.shape[0])
        for i in range(torque.shape[0]):

            ang_acc = com_acc[i] * r_mag[i+2]
            # https://dl.acm.org/doi/pdf/10.1145/2556288.2557130 - Eq. 13
            u = np.cross(coms_pos[i+1] - shoulders[i+2], coms_pos[i+2] - coms_pos[i+1])
            u = u / np.linalg.norm(u)

            # https://dl.acm.org/doi/pdf/10.1145/2556288.2557130 - Eq. 12
            inertia = u * moment_of_inertia
            # https://dl.acm.org/doi/pdf/10.1145/2556288.2557130 - Eq. 6
            torque[i] = np.linalg.norm(np.cross(r[i+2], (com_acc[i]-g)*mass) - np.cross(r[i+2], (mass*g) + (ang_acc*inertia)))

        return np.nanmean(torque)

    # derive the maximum torque at the shoulder in the current gesture
    def get_max_torque(average_shoulder_abduction, average_elbow_flexion, gender):
        G = 0.2845 if gender == "Male" else 0.1495
        # https://dl.acm.org/doi/pdf/10.1145/3658230 - Eq. 11
        mx_t = (227.338 + (0.525 * average_elbow_flexion) - (0.296 * average_shoulder_abduction)) * G
        return mx_t

    # compute the NICER metric for the current gesture
    def get_nicer(average_torque, average_shoulder_abduction, average_elbow_flexion, duration, gender):
        max_torque = get_max_torque(average_shoulder_abduction, average_elbow_flexion, gender)
        c_theta = 0
        if average_shoulder_abduction > 90:
            gendered_compensation_factor = (1230, 0.09) if gender == "Male" else (1005, 0.11)
            t_theta = np.sin(np.deg2rad(average_shoulder_abduction)) / gendered_compensation_factor[1]
            f_theta = gendered_compensation_factor[0] * (0.0095 / (1 + np.e**((66.4-average_shoulder_abduction) / 7.83)))
            # https://dl.acm.org/doi/pdf/10.1145/3658230 - Eq. 7-8
            c_theta = f_theta - t_theta
        
        adjusted_strength = ((average_torque + c_theta) / max_torque) * 100 

        # https://dl.acm.org/doi/pdf/10.1145/3658230 - Eq. 9
        nicer = (((duration-0.02) * (adjusted_strength**1.83) * 0.000218) / 14.86) * 100
        return nicer

    # compute the consumed endurance metric for the current gesture
    def get_consumed_endurance(average_torque, average_shoulder_abduction, average_elbow_flexion, gesture_duration, gender):
        max_torque = get_max_torque(average_shoulder_abduction, average_elbow_flexion, gender)
        strength = (average_torque / max_torque) * 100
        
        # Endurance curve is asymptotic, resulting in exertion below 15% one's max being treated as something that can be maintained indefinitely
        if strength < 15:
            # print("Skipping CE as below 15% exertion")
            return 0, 0
        else:
            # https://dl.acm.org/doi/pdf/10.1145/2556288.2557130 - Eq. 1
            endurance = (1236.5 / ((strength - 15)**0.618)) - 72.5
            
            # https://dl.acm.org/doi/pdf/10.1145/2556288.2557130 - Eq. 7
            # subtracting 20ms as we shorten the gesture duration by as much to calculate the acceleration
            ce = ((gesture_duration-0.02) / endurance) * 100
            return ce, endurance

    def get_ray(feature_array, joint_frame, r_idx, c_idx, ray_label, normalised_to, hand, compute_angular_error=False, direction=None, target_pos=None):
        side_key = "r" if hand == "RIGHT" else "l"
        hand_side_key = R_KEY if hand == "RIGHT" else L_KEY

        # Choose points from which to compute ray vector - this should instead have been passed in...
        ray_origin_labels = None
        if f"IFRC" in ray_label:
            ray_origin_labels = [f"{hand_side_key}Index2 {axis}" for axis in ["X", "Y", "Z"]]
            ray_direction_labels = [f"{hand_side_key}IndexTip {axis}" for axis in ["X", "Y", "Z"]]
        elif "HEAD" in ray_label:
            ray_origin_labels = [f"head {axis}" for axis in ["X", "Y", "Z"]]
            ray_direction_labels = [f"chin {axis}" for axis in ["X", "Y", "Z"]]
        elif "EFRC" in ray_label:
            ray_origin_labels = [f"Tobii3-Set-L2-R2 - {i} {j}" for i in [1, 4, 3, 6] for j in ["X", "Y", "Z"]]
            ray_direction_labels = [f"{hand_side_key}IndexTip {axis}" for axis in ["X", "Y", "Z"]]
        elif "UPPER_ARM" in ray_label:
            ray_origin_labels = [f"{ray_label[0].lower()}_uarm {axis}" for axis in ["X", "Y", "Z"]]
            ray_direction_labels = [f"{ray_label[0].lower()}_larm {axis}" for axis in ["X", "Y", "Z"]]
        elif "HAND_NEAR_SHOULDER" in ray_label:
            ray_origin_labels = [f"{ray_label[0].lower()}_uarm {axis}" for axis in ["X", "Y", "Z"]]
            ray_direction_labels = [f"{ray_label[0].lower()}_hand {axis}" for axis in ["X", "Y", "Z"]]
        elif "HFRC" in ray_label:
            ray_origin_labels = [f"head {axis}" for axis in ["X", "Y", "Z"]]
            ray_direction_labels = [f"{hand_side_key}IndexTip {axis}" for axis in ["X", "Y", "Z"]]
        elif f"{hand}_FRC" == ray_label:
            ray_origin_labels = [f"{side_key}_larm {axis}" for axis in ["X", "Y", "Z"]]
            ray_direction_labels = [f"{side_key}_hand {axis}" for axis in ["X", "Y", "Z"]]
        if ray_origin_labels is None:
            print(f"ray origin not found for {ray_label}")

        for joint_label in set([f"{point} X" for point in normalised_to]) | (set(ray_origin_labels) | set(ray_direction_labels)):
            if joint_label not in joint_frame:
                for i in c_idx:
                    feature_array[r_idx, i] = np.nan
                return None, None
            
        if "EFRC" in ray_label:
            tobii_markers = {
                "L1_R1": (joint_frame[[f"Tobii3-Set-L2-R2 - 1 {j}" for j in ["X", "Y", "Z"]]].values + joint_frame[[f"Tobii3-Set-L2-R2 - 4 {j}" for j in ["X", "Y", "Z"]]].values) / 2,
                "L3_R3": (joint_frame[[f"Tobii3-Set-L2-R2 - 3 {j}" for j in ["X", "Y", "Z"]]].values + joint_frame[[f"Tobii3-Set-L2-R2 - 6 {j}" for j in ["X", "Y", "Z"]]].values) / 2,
            }
            line_to_eye = tobii_markers["L3_R3"] - tobii_markers["L1_R1"]
            line_to_eye = line_to_eye / np.linalg.norm(line_to_eye)
            line_to_eye = line_to_eye*20
            ray_origin = tobii_markers["L3_R3"] + line_to_eye
        else:
            ray_origin = joint_frame[ray_origin_labels].values

        ray_direction = joint_frame[ray_direction_labels].values

        # compute ray
        ray = ray_direction - ray_origin
        ray_mag = np.linalg.norm(ray)
        ray = ray / ray_mag

        # used to normalised angles relative to the sagittal axis
        p0 = joint_frame[[f"{normalised_to[0]} {axis}" for axis in ["X", "Y", "Z"]]].values
        p1 = joint_frame[[f"{normalised_to[1]} {axis}" for axis in ["X", "Y", "Z"]]].values - p0
        p1 = p1 / np.linalg.norm(p1)
        p2 = joint_frame[[f"{normalised_to[2]} {axis}" for axis in ["X", "Y", "Z"]]].values - p0
        p2 = p2 / np.linalg.norm(p2)

        invert = "RIGHT" == hand
        normal = np.cross(p1, p2)
        
        # compute ray relative to sagittal axis
        yaw, pitch = get_angle_between_vectors(normal, ray, invert)

        # Compute the error to the target
        if compute_angular_error:
            led_ray = (np.nan, np.nan)
            led_pos = target_pos
            if led_pos is not None:
                led_ray = led_pos - ray_origin
                led_ray = (led_ray / np.linalg.norm(led_ray))
                led_ray = get_angle_between_vectors(normal, led_ray, invert)
                feature_array[r_idx, c_idx[-3]] = yaw - led_ray[0]
                feature_array[r_idx, c_idx[-2]] = pitch - led_ray[1]
                feature_array[r_idx, c_idx[-1]] = get_vector_cosine_from_angles((feature_array[r_idx, c_idx[-2]], feature_array[r_idx, c_idx[-3]]), (0,0))
            else:
                feature_array[r_idx, c_idx[-3]] = np.nan
                feature_array[r_idx, c_idx[-2]] = np.nan
                feature_array[r_idx, c_idx[-1]] = np.nan
            
        feature_array[r_idx, c_idx[0]] = yaw
        feature_array[r_idx, c_idx[1]] = pitch

    # compute the normal for a plane derived from 3 points
    def get_plane_normal(feature_array, joint_frame, r_idx, c_idx, points, hand, normalised_to=None):
        for joint_label in set(points) | set(normalised_to) if normalised_to is not None else set():
            if f"{joint_label} X" not in joint_frame:
                for i in range(c_idx):
                    feature_array[r_idx, i] = np.nan
                return None
        yaw = np.nan
        pitch = np.nan
        p0 = joint_frame[[f"{points[0]} {axis}" for axis in ["X", "Y", "Z"]]].values
        p1 = joint_frame[[f"{points[1]} {axis}" for axis in ["X", "Y", "Z"]]].values - p0
        p1 = p1 / np.linalg.norm(p1)
        p2 = joint_frame[[f"{points[2]} {axis}" for axis in ["X", "Y", "Z"]]].values - p0
        p2 = p2 / np.linalg.norm(p2)
        normal = np.cross(p1, p2)

        # used to normalised angles relative to the sagittal axis
        if normalised_to is not None:
            n0 = joint_frame[[f"{normalised_to[0]} {axis}" for axis in ["X", "Y", "Z"]]].values
            n1 = joint_frame[[f"{normalised_to[1]} {axis}" for axis in ["X", "Y", "Z"]]].values - n0
            n1 = n1 / np.linalg.norm(n1)
            n2 = joint_frame[[f"{normalised_to[2]} {axis}" for axis in ["X", "Y", "Z"]]].values - n0
            n2 = n2 / np.linalg.norm(n2)           

            normal = normal - np.cross(n1, n2)

        normal = normal / np.linalg.norm(normal)

        yaw, pitch = get_vector_orientation(normal, "RIGHT" == hand)
        feature_array[r_idx, c_idx[0]] = yaw
        feature_array[r_idx, c_idx[1]] = pitch
        return (yaw, pitch)
    
    # compute angle between 2 joints about a third
    def get_joint_angle(feature_array, joint_frame, r_idx, c_idx, plane_labels, v0_labels, v1_labels, orthogonal_plane: bool = False, perpendicular_plane: bool = False, shift: str | None = None, inverse: bool = False):
        for joint_label in set(plane_labels) | set(v0_labels) | set(v1_labels):
            if f"{joint_label} X" not in joint_frame:
                for i in c_idx:
                    feature_array[r_idx, i] = np.nan
                return None
            
        p0 = joint_frame[[f"{plane_labels[0]} {axis}" for axis in ["X", "Y", "Z"]]].values
        p1 = joint_frame[[f"{plane_labels[1]} {axis}" for axis in ["X", "Y", "Z"]]].values - p0
        p1 = p1 / np.linalg.norm(p1)
        if shift is not None:
            p2 = joint_frame[[f"{shift} {axis}" for axis in ["X", "Y", "Z"]]].values - p0
            p2 = joint_frame[[f"{plane_labels[2]} {axis}" for axis in ["X", "Y", "Z"]]].values - p0 - p2
        else:
            p2 = joint_frame[[f"{plane_labels[2]} {axis}" for axis in ["X", "Y", "Z"]]].values - p0
        p2 = p2 / np.linalg.norm(p2)           

        plane_normal = np.cross(p1, p2)
        if plane_labels[3] == plane_labels[1]:
            y_axis = p1
        elif plane_labels[3] == plane_labels[2]:
            y_axis = p2
        else: # only calculate if not a point already provided.
            y_axis = joint_frame[[f"{plane_labels[3]} {axis}" for axis in ["X", "Y", "Z"]]].values
            y_axis = y_axis - (np.dot(y_axis-p0, plane_normal) * plane_normal)
            y_axis = y_axis / np.linalg.norm(y_axis)
        
        if orthogonal_plane:
            x_axis = np.copy(plane_normal)
            plane_normal = np.cross(x_axis, y_axis)
        else:
            x_axis = np.cross(plane_normal, y_axis)

        if perpendicular_plane:
            old_y = np.copy(y_axis)
            y_axis = np.copy(plane_normal)
            plane_normal = old_y

        # shift points to lie on the plane provided
        v0_origin = joint_frame[[f"{v0_labels[0]} {axis}" for axis in ["X", "Y", "Z"]]].values
        v0_origin = v0_origin - (np.dot(v0_origin-p0, plane_normal) * plane_normal)
        v0 = joint_frame[[f"{v0_labels[1]} {axis}" for axis in ["X", "Y", "Z"]]].values
        v0 = (v0 - (np.dot(v0-p0, plane_normal) * plane_normal)) - v0_origin
        v0_mag = np.linalg.norm(v0)

        v1_origin = joint_frame[[f"{v1_labels[0]} {axis}" for axis in ["X", "Y", "Z"]]].values
        v1_origin = v1_origin - (np.dot(v1_origin-p0, plane_normal) * plane_normal)
        v1 = joint_frame[[f"{v1_labels[1]} {axis}" for axis in ["X", "Y", "Z"]]].values
        v1 = (v1 - (np.dot(v1-p0, plane_normal) * plane_normal)) - v1_origin
        v1_mag = np.linalg.norm(v1)

        angle = (np.rad2deg(np.arccos(np.dot(v0, v1) / (v0_mag * v1_mag))))
        if inverse:
            angle = 0 - angle

        feature_array[r_idx, c_idx[0]] = angle
        return angle
    
    def get_features(hand, data, peaks, metadata, trial_metadata):
        side_key = R_KEY if hand == "RIGHT" else L_KEY
        markerless_side_key = "r" if hand == "RIGHT" else "l"

        def get_inter_frame_distance(row):
            start = row[[f"{markerless_side_key}_hand {axis}" for axis in ["X", "Y", "Z"]]].values
            end = row[[f"{markerless_side_key}_hand {axis} SHIFTED" for axis in ["X", "Y", "Z"]]].values
            dist = np.sqrt(np.sum(np.square(end[0] - start[0]) + np.square(end[1] - start[1]) + np.square(end[2] - start[2])))
            return dist
        
        target_box = trial_metadata["TargetBox"]
        sub_target = trial_metadata["TargetLED"]
        target_row, target_col, target_label = Target.get_target_label(str(target_box["Row"]), str(target_box["Column"]))
        
        # intialise features from trial metadata
        features = {
            "PARTICIPANT": participant_metadata["ID"],
            "CONDITION": condition,
            "TRIAL_IDX": idx,
            "HAND": hand,
            "TARGET_ROW_NAME": target_row,
            "TARGET_COL_NAME": target_col,
            "TARGET_NAME": target_label,
            "LED_ROW": sub_target["Row"],
            "LED_COL": sub_target["Column"],
            "IS_DOMINANT_HAND": metadata["Dominant Hand"] == "BOTH" or metadata["Dominant Hand"] == hand,
            "ACC_REQ": condition.split('-')[0],
            "FOCUS_REQ": condition.split('-')[1],
        }
        hold_duration = get_duration(peaks[INITIAL_BALLISTIC_KEY][1], peaks[TERMINAL_BALLISTIC_KEY][0])
        gesture_duration = get_duration(peaks[INITIAL_BALLISTIC_KEY][0], peaks[TERMINAL_BALLISTIC_KEY][1])
        start_motion = get_duration(peaks[INITIAL_BALLISTIC_KEY][0], peaks[INITIAL_BALLISTIC_KEY][1])
        end_motion = get_duration(peaks[TERMINAL_BALLISTIC_KEY][0], peaks[TERMINAL_BALLISTIC_KEY][1])
        pre_gesture_motion = get_duration(0, peaks[INITIAL_BALLISTIC_KEY][0])

        # Get gesture temporal features
        features["GESTURE_START_DELAY_MS"] = peaks[INITIAL_BALLISTIC_KEY][0] * 10
        features["GESTURE_MS"] = gesture_duration.shape[0] * 10
        features["HOLD_PHASE_MS"] = hold_duration.shape[0] * 10
        features["START_PHASE_MS"] = start_motion.shape[0] * 10
        features["END_PHASE_MS"] = end_motion.shape[0] * 10
        features["ACTIVE_MS"] = features["START_PHASE_MS"] + features["HOLD_PHASE_MS"]
        
        # normalise duration lengths to ratio of gesture length
        features["HOLD_PHASE_RATIO"] = features["HOLD_PHASE_MS"] / features["ACTIVE_MS"]
        features["START_PHASE_RATIO"] = features["START_PHASE_MS"] / features["ACTIVE_MS"]
        features["END_PHASE_DURATION"] = features["END_PHASE_MS"] / features["ACTIVE_MS"]

        # extract target from metadata
        target = Target(
            trial_metadata["TargetBox"]["Row"], trial_metadata["TargetBox"]["Column"], trial_metadata["TargetLED"]["ID"], 
            np.array([trial_metadata["TargetBox"]["X"], trial_metadata["TargetBox"]["Y"], trial_metadata["TargetBox"]["Z"]]),
            np.array([trial_metadata["TargetBox"]["YAW"], trial_metadata["TargetBox"]["PITCH"], trial_metadata["TargetBox"]["ROLL"]]),
            np.array([trial_metadata["TargetLED"]["X"], trial_metadata["TargetLED"]["Y"], trial_metadata["TargetLED"]["Z"]])
        )
        target_pos = target.subtarget.to_array()

        # check that markerless body tracking data is present
        if "torso X" in data:
            distance_travelled = data.loc[gesture_duration, [f"{markerless_side_key}_hand {axis}" for axis in ["X", "Y", "Z"]]]
            distance_travelled = pd.concat([distance_travelled, pd.DataFrame({col: np.nan for col in distance_travelled.columns.values}, index=[distance_travelled.index.values[-1]])])
            distance_travelled[[f"{markerless_side_key}_hand {axis} SHIFTED" for axis in ["X", "Y", "Z"]]] = distance_travelled.shift(1)
            distance_travelled = distance_travelled.apply(get_inter_frame_distance, axis=1).sum()
            e = data.loc[gesture_duration, [f"{markerless_side_key}_larm {axis}" for axis in ["X", "Y", "Z"]]].values
            h = data.loc[gesture_duration, [f"{markerless_side_key}_hand {axis}" for axis in ["X", "Y", "Z"]]].values
            forearm_length = np.nanmean(np.sqrt(np.square(e[:, 0] - h[:, 0]) + np.square(e[:, 1] - h[:, 1]) + np.square(e[:, 2] - h[:, 2])))
            gender = metadata["Gender"]

            features["DISTANCE_TRAVELLED_NORMALISED"] = distance_travelled / forearm_length
            
            for prefix, frame_indexes, in [("", hold_duration), ("TOTAL_GESTURE_", gesture_duration[:-1])]:
                frames = data.iloc[frame_indexes]
                duration = (frames.shape[0])/100
                processing_steps = {}
                
                # Fatigue Measures
                r = np.zeros((frames.shape[0], 3))
                r_mag = np.zeros(frames.shape[0])
                com = np.zeros((frames.shape[0], 3))
                shoulder = np.zeros((frames.shape[0], 3))
                abduction = np.zeros(frames.shape[0])
                extension = np.zeros(frames.shape[0])
                r, r_mag, com, shoulder, abduction, extension = get_centre_of_mass(frames, side_key, markerless_side_key, gender)

                average_torque = get_shoulder_torque(r, r_mag, com, shoulder, gender)
                average_abduction = np.nanmean(abduction)
                average_extension = np.nanmean(extension)
                ce, endurance = get_consumed_endurance(average_torque, average_abduction, average_extension, duration, gender)
                nicer = get_nicer(average_torque, average_abduction, average_extension, duration, gender)
                
                features[f"{prefix}SHOULDER_TORQUE"] = average_torque
                features[f"{prefix}CONSUMED_ENDURANCE"] = ce
                features[f"{prefix}NICER"] = nicer

                # Compute Rays 
                if prefix != "TOTAL_GESTURE_":
                    processing_steps[f"{prefix}TORSO_ROTATION"] = { 
                        "LOGIC": lambda frame, f_idx, c_idx, feature_array: 
                            get_plane_normal(feature_array, frame, f_idx, c_idx, ("pelvis", "l_uarm", "r_uarm", "torso"), hand), 
                        "RETURNS": ["YAW", "PITCH"] 
                    }
                
                    ray = "HAND_NEAR_SHOULDER_RAY"
                    processing_steps[f"{prefix}{ray}"] = { 
                        "LOGIC": lambda frame, f_idx, c_idx, feature_array, ray=f"{hand}_{ray}", points=("pelvis", "l_uarm", "r_uarm", "torso"), angular=False, h=hand: 
                            get_ray(feature_array, frame, f_idx, c_idx, ray, points, h, compute_angular_error=angular), 
                        "RETURNS": ["YAW", "PITCH"] 
                    }

                    for ray in ["FRC", "EFRC_CYCLOPS", "IFRC", "HEAD", "UPPER_ARM", "HFRC"]:
                        if ray != "HEAD":
                            side = f"{hand}_"
                        else:
                            side = ""
                        compute_ray_calcs = True 
                        compute_angular_error = ray != "UPPER_ARM" and compute_ray_calcs
                        return_vals = ["YAW", "PITCH"]
                        if compute_angular_error:
                            return_vals.extend(["ERROR_LED_YAW", "ERROR_LED_PITCH", "ERROR_LED_ANGULAR"])
                        processing_steps[f"{prefix}{ray}"] = { 
                            "LOGIC": lambda frame, f_idx, c_idx, feature_array, ray=f"{side}{ray}", points=("pelvis", "l_uarm", "r_uarm", "torso"), angular=compute_angular_error, h=hand, tp=target_pos: 
                                get_ray(feature_array, frame, f_idx, c_idx, ray, points, h, compute_angular_error=angular, target_pos=tp), 
                            "RETURNS": return_vals
                        }

                    processing_steps[f"{prefix}ELBOW_EXTENSION"] = { "RETURNS": [""], "LOGIC": lambda frame, f_idx, c_idx, feature_array: get_joint_angle(feature_array, frame, f_idx, c_idx, (f"{markerless_side_key}_larm", f"{markerless_side_key}_uarm", f"{markerless_side_key}_hand", f"{markerless_side_key}_uarm"), (f"{markerless_side_key}_larm", f"{markerless_side_key}_hand"), (f"{markerless_side_key}_larm", f"{markerless_side_key}_uarm")) }
                
                
                # Determine the expected feature columns from the registered features
                feature_array_cols = 0
                for key, processing_logic in processing_steps.items():
                    returns = len(processing_logic["RETURNS"])
                    processing_logic["IDX"] = [i for i in range(feature_array_cols, feature_array_cols+returns)]
                    feature_array_cols += returns
                
                # Process the registered features for the current trial
                duration = frames.shape[0]
                if feature_array_cols > 0 and duration > 0:
                    feature_array = np.zeros((duration, feature_array_cols))
                    for frame in range(duration):
                        joint_frame = frames.iloc[frame]
                        for key, processing_logic in processing_steps.items():
                            processing_logic["LOGIC"](joint_frame, frame, processing_logic["IDX"], feature_array)
                    
                    if duration == 1:
                        medians = feature_array[0, :]
                    elif len(processing_steps.keys()) > 0:
                        with warnings.catch_warnings():
                            warnings.filterwarnings('ignore', r'All-NaN (slice|axis) encountered')
                            warnings.filterwarnings('ignore', r'Mean of empty slice')
                            medians = np.nanmedian(feature_array, axis=0)
                    for key, processing_logic in processing_steps.items():
                        idx_map = processing_logic["IDX"]
                        for f_idx, feature in enumerate(processing_logic["RETURNS"]):
                            features[f"{key}{'_' if feature != '' else ''}{feature}_MEDIAN"] = medians[idx_map[f_idx]]
                
                # Get Alignment of rays
                if prefix != "TOTAL_GESTURE_":
                    for ray_1, ray_2 in combinations(["FRC", "EFRC_CYCLOPS", "IFRC", "HEAD", "HFRC"], 2):
                        get_vector_alignment(prefix, features, ray_1, ray_2)

            # Get Torso movement
            pre_gesture_frames = data.iloc[pre_gesture_motion]
            pre_gesture_vals = np.zeros((pre_gesture_frames.shape[0], 2))
            for row in range(pre_gesture_frames.shape[0]):
                get_plane_normal(pre_gesture_vals, pre_gesture_frames.iloc[row], row, [0,1], ("pelvis", "l_uarm", "r_uarm", "torso"), hand)
            if pre_gesture_frames.shape[0] == 1:
                medians = pre_gesture_vals[0, :]
            else:
                medians = np.nanmedian(pre_gesture_vals, axis=0)
            for f_idx, feature in enumerate(["YAW", "PITCH"]):
                features[f"PRE_GESTURE_TORSO_ROTATION_{feature}_MEDIAN"] = medians[f_idx]

            torso_median_yaw_delta = np.abs(features[f"PRE_GESTURE_TORSO_ROTATION_YAW_MEDIAN"] - features[f"TORSO_ROTATION_YAW_MEDIAN"])
            torso_median_pitch_delta = np.abs(features[f"PRE_GESTURE_TORSO_ROTATION_PITCH_MEDIAN"] - features[f"TORSO_ROTATION_PITCH_MEDIAN"])
            features["TORSO_YAW_MEDIAN_DELTA"] = torso_median_yaw_delta
            features["TORSO_PITCH_MEDIAN_DELTA"] = torso_median_pitch_delta

            del features[f"PRE_GESTURE_TORSO_ROTATION_YAW_MEDIAN"]
            del features[f"PRE_GESTURE_TORSO_ROTATION_PITCH_MEDIAN"]
        else:
            print(f"Unable to locate trial data: {metadata['ID']}-{trial_metadata['Trial Idx']}")
        return features

    pointing_phases = None
    features = None
    hand = None
    if trial_data is not None:
        flags = {}

        # Retrieve trial flags, if present
        if "Flags" in trial_metadata:
            flags = set([flag for flag, v in trial_metadata["Flags"].items() if v is None or v])
        if len(set(flags) - {"NO_GAZE", "B_HANDSHAPE", "TARGET_TRACKING"}) > 0:
            console.log(f"{participant} - {condition} - {idx} | Trial omitted due to present exclusion flags: {flags}")
            flag = list(set(flags) - {"NO_GAZE", "B_HANDSHAPE"})[-1]
            update_missing_trials(missing_trials, condition, flag, idx)
            return None, None       

        # determine when pointing occurred
        hand_velocity_peaks = trial_metadata[HVP_KEY]
        right = {START_KEY:[], END_KEY: []} if R_VEL_KEY not in hand_velocity_peaks else hand_velocity_peaks[R_VEL_KEY]
        left = {START_KEY:[], END_KEY: []} if L_VEL_KEY not in hand_velocity_peaks else hand_velocity_peaks[L_VEL_KEY]
        right_gesture = { }
        left_gesture = { }

        if (MANUAL_ANNOTATIONS_KEY in left and IGNORE_HAND_KEY not in left) and (MANUAL_ANNOTATIONS_KEY in right and IGNORE_HAND_KEY not in right):
            console.log(f"{participant} - {condition} - {idx} | Unable to process as manual overrides given to both hands")
            update_missing_trials(missing_trials, condition, "BOTH_HANDS_HAVE_OVERRIDES", idx)
        else:
            # Derive right hand pointing phases
            if MANUAL_ANNOTATIONS_KEY in right and IGNORE_HAND_KEY not in right: # Manual overrides given
                if (INITIAL_BALLISTIC_KEY not in right[MANUAL_ANNOTATIONS_KEY] and TERMINAL_BALLISTIC_KEY not in right[MANUAL_ANNOTATIONS_KEY]):
                    console.log(f"{participant} - {condition} - {idx} | Unable to process as no manual overrides given {right}")
                else: 
                    # Derive pointing start phase   
                    if INITIAL_BALLISTIC_KEY in right[MANUAL_ANNOTATIONS_KEY] and len(right[MANUAL_ANNOTATIONS_KEY][INITIAL_BALLISTIC_KEY]):
                        right_gesture[INITIAL_BALLISTIC_KEY] = right[MANUAL_ANNOTATIONS_KEY][INITIAL_BALLISTIC_KEY]
                    elif len(right[START_KEY]) > 0 and len(right[END_KEY]) > 0:
                        right_gesture[INITIAL_BALLISTIC_KEY] = [right[START_KEY][0], right[END_KEY][0]]

                    # Derive pointing end phase
                    if TERMINAL_BALLISTIC_KEY in right[MANUAL_ANNOTATIONS_KEY] and len(right[MANUAL_ANNOTATIONS_KEY][TERMINAL_BALLISTIC_KEY]):
                        right_gesture[TERMINAL_BALLISTIC_KEY] = right[MANUAL_ANNOTATIONS_KEY][TERMINAL_BALLISTIC_KEY]
                    elif len(right[START_KEY]) > 0 and len(right[END_KEY]) > 0:
                        right_gesture[TERMINAL_BALLISTIC_KEY] = [right[START_KEY][-1], right[END_KEY][-1]]

            # Use detected hand velocity peaks for right hand pointing phases
            if len(right_gesture) == 0 and ((len(right[START_KEY]) == 2 and len(right[END_KEY]) == 2) and MANUAL_ANNOTATIONS_KEY not in left and (IGNORE_HAND_KEY in left or (len(left[START_KEY]) != 2 and len(left[END_KEY]) != 2))):
                right_gesture[INITIAL_BALLISTIC_KEY] = [right[START_KEY][0], right[END_KEY][0]]
                right_gesture[TERMINAL_BALLISTIC_KEY] = [right[START_KEY][-1], right[END_KEY][-1]]

            # Derive left hand pointing phases
            if MANUAL_ANNOTATIONS_KEY in left and IGNORE_HAND_KEY not in left:
                if (INITIAL_BALLISTIC_KEY not in left[MANUAL_ANNOTATIONS_KEY] and TERMINAL_BALLISTIC_KEY not in left[MANUAL_ANNOTATIONS_KEY]):
                    console.log(f"{participant} - {condition} - {idx} | Unable to process as no manual overrides given...")
                else:
                    # Derive pointing start phase
                    if INITIAL_BALLISTIC_KEY in left[MANUAL_ANNOTATIONS_KEY] and len(left[MANUAL_ANNOTATIONS_KEY][INITIAL_BALLISTIC_KEY]):
                        left_gesture[INITIAL_BALLISTIC_KEY] = left[MANUAL_ANNOTATIONS_KEY][INITIAL_BALLISTIC_KEY]
                    elif len(left[START_KEY]) > 0 and len(left[END_KEY]) > 0:
                        left_gesture[INITIAL_BALLISTIC_KEY] = [left[START_KEY][0], left[END_KEY][0]]

                    # Derive pointing end phase
                    if TERMINAL_BALLISTIC_KEY in left[MANUAL_ANNOTATIONS_KEY] and len(left[MANUAL_ANNOTATIONS_KEY][TERMINAL_BALLISTIC_KEY]):
                        left_gesture[TERMINAL_BALLISTIC_KEY] = left[MANUAL_ANNOTATIONS_KEY][TERMINAL_BALLISTIC_KEY]
                    elif len(left[START_KEY]) > 0 and len(left[END_KEY]) > 0:
                        left_gesture[TERMINAL_BALLISTIC_KEY] = [left[START_KEY][-1], left[END_KEY][-1]]
            
            # Use detected hand velocity peaks for left hand pointing phases
            if len(left_gesture) == 0 and ((len(left[START_KEY]) == 2 and len(left[END_KEY]) == 2) and MANUAL_ANNOTATIONS_KEY not in right and  (IGNORE_HAND_KEY in right or (len(right[START_KEY]) != 2 and len(right[END_KEY]) != 2))):
                left_gesture[INITIAL_BALLISTIC_KEY] = [left[START_KEY][0], left[END_KEY][0]]
                left_gesture[TERMINAL_BALLISTIC_KEY] = [left[START_KEY][-1], left[END_KEY][-1]]

        # determine which hand was used for pointing
        try:
            if INITIAL_BALLISTIC_KEY in right_gesture and TERMINAL_BALLISTIC_KEY in right_gesture and INITIAL_BALLISTIC_KEY in left_gesture and TERMINAL_BALLISTIC_KEY in left_gesture:
                print(f"{participant} - {condition} - {idx} | Unable to determine hand used for pointing gesture as motion has been detected for both hands.\nright: {right_gesture}, left: {left_gesture}")
                update_missing_trials(missing_trials, condition, "BOTH_HANDS_LABELLED", idx)
            elif INITIAL_BALLISTIC_KEY not in right_gesture and TERMINAL_BALLISTIC_KEY not in right_gesture and INITIAL_BALLISTIC_KEY not in left_gesture and TERMINAL_BALLISTIC_KEY not in left_gesture:
                print(f"{participant} - {condition} - {idx} | Unable to determine hand used for pointing gesture as no motion has been detected for either hand.\nright: {right_gesture}, left: {left_gesture}")
                update_missing_trials(missing_trials, condition, "NO_POINTING_LABELS", idx)
            elif INITIAL_BALLISTIC_KEY in right_gesture and TERMINAL_BALLISTIC_KEY in right_gesture:
                hand = "RIGHT"
                pointing_phases = right_gesture
            elif INITIAL_BALLISTIC_KEY in left_gesture and TERMINAL_BALLISTIC_KEY in left_gesture:
                hand = "LEFT"
                pointing_phases = left_gesture
            else:
                print(f"{participant} - {condition} - {idx} | Unable to determine pointing gesture as missing gesture components.\nright: {right_gesture}, left: {left_gesture}")
                update_missing_trials(missing_trials, condition, "MISSING_GESTURE_COMPONENTS", idx)
        except Exception as e:
            console.log(f"{participant} - {condition} - {idx} | Unable to process gesture components", e)
            console.log(f"Left peaks: {left}")
            console.log(f"Right peaks: {right}")
            console.log(f"Left motion: {left_gesture}")
            console.log(f"Right motion: {right_gesture}")

        if hand is not None:
            # validate pointing gesture phases are valid
            if pointing_phases[TERMINAL_BALLISTIC_KEY][0] < pointing_phases[INITIAL_BALLISTIC_KEY][1] or pointing_phases[INITIAL_BALLISTIC_KEY][1] < pointing_phases[INITIAL_BALLISTIC_KEY][0] or pointing_phases[TERMINAL_BALLISTIC_KEY][1] < pointing_phases[TERMINAL_BALLISTIC_KEY][0]:
                features = None
                console.log(f"Unable to derive gesture features as at least one identified gesture segment has an invalid range:{f' HOLD: {pointing_phases[INITIAL_BALLISTIC_KEY][1]}-{pointing_phases[TERMINAL_BALLISTIC_KEY][0]}' if pointing_phases[INITIAL_BALLISTIC_KEY][1] > pointing_phases[TERMINAL_BALLISTIC_KEY][0] else ''}{f' BALLISTIC_1: {pointing_phases[INITIAL_BALLISTIC_KEY][0]}-{pointing_phases[INITIAL_BALLISTIC_KEY][1]}' if pointing_phases[INITIAL_BALLISTIC_KEY][1] > pointing_phases[INITIAL_BALLISTIC_KEY][0] else ''}{f' BALLISTIC_2: {pointing_phases[TERMINAL_BALLISTIC_KEY][0]}-{pointing_phases[TERMINAL_BALLISTIC_KEY][1]}' if pointing_phases[TERMINAL_BALLISTIC_KEY][1] > pointing_phases[TERMINAL_BALLISTIC_KEY][0] else ''}")
            else:
                # Derive pointing gesture features from derived pointing gesture frames
                features = get_features(hand, trial_data, pointing_phases, participant_metadata, trial_metadata)

    return hand, features

# Program Start
def process_participant(participant, participant_path, input_dir):
    missing_trials = {}
    participant_data, participant_metadata = load_participant_data(participant, missing_trials, input_dir)

    pointing_gesture_stats = []
    print(f"Processing trial data for participant: {participant}, {participant_path}")
    for condition, trials in participant_data.items():
        print(f"Processing trial data for participant: {participant} - {condition}")
        for trial in trials:
            trial_idx = trial["Trial Idx"]
            trial_df = trial["DATA"]

            if "TargetLED" in trial:
                pointing_hand, pointing_gesture = discrete_feature_extraction(trial_df, participant, participant_metadata, condition, trial_idx, trial, missing_trials)
                if pointing_gesture is not None:
                    
                    pointing_gesture_stats.append(pointing_gesture)
                else:
                    console.log(f"Unable to proceed as unable to find pointing gesture: {condition} {trial_idx}")
            else:
                console.log(f"Unable to proceed as target array not available: {condition} {trial_idx}")
    
    return pointing_gesture_stats, missing_trials
                

def process_participants(participant_id, participant_path, input_dir, queue):
    try:
        encoded_gestures, missing_trials = process_participant(participant_id, participant_path, input_dir)
        queue.put((participant_id, encoded_gestures, missing_trials, f"Completed trials for participant {participant_id}"))
    except Exception as e:
        tb = traceback.format_exception(e)
        queue.put((participant_id, None, None, "Error processing trial for participant {}\n{}".format(participant_id, '\n'.join(tb))))


if __name__ == '__main__':
    parser = ArgumentParser(description="Aggregate Data From QTM, Theia, and Target Controller")
    parser.add_argument("-d", "--dir", type=Path, nargs="?", default=None)
    parser.add_argument("-o", "--output", type=Path, nargs="?", default=None)

    args = parser.parse_args()

    if args.dir is not None:
        participant_dir = args.dir
    else:
        parser.print_usage()
        parser.exit(1, "No input directory was provided")
    
    input_dir = dirname(participant_dir)
    if args.output is not None:
        output_dir = args.output
    else:
        output_dir = input_dir
    
    participants = [(d, join(participant_dir, d))  for d in listdir(participant_dir) if isdir(join(participant_dir, d)) and participant_regex.search(d)]

    if len(participants) < 1:
        parser.exit(1, f"Unable to locate participants in the provided directory: {participant_dir}")

    participant_table = generate_participant_table([(p[0], True) for p in participants])
    console.print(participant_table)
    
    # Process the DataFrame in parallel using 4 threads
    completion_queue = Queue()
    with ThreadPoolExecutor(max_workers=4) as executor:
        futures = [
            executor.submit(process_participants, participant, participant_path, participant_dir, completion_queue)
            for participant, participant_path in participants
        ]

        if output_dir.endswith(".csv"):
            encoded_gestures_path = output_dir
            invalid_trials_path = join(dirname(output_dir), "InvalidTrials.json")
        else:
            encoded_gestures_path = join(output_dir, "EncodedGesturesOutput.csv")
            invalid_trials_path = join(output_dir, "InvalidTrials.json")
        encoded_gestures = None
        if isfile(encoded_gestures_path):
            with open(encoded_gestures_path, 'r') as encoded_gestures_file:
                encoded_gestures = pd.read_csv(encoded_gestures_file)
                encoded_gestures.set_index('index', inplace=True)

        if isfile(invalid_trials_path): # make a backup
            os.replace(invalid_trials_path, f"{invalid_trials_path}.bk")
        invalid_trials = {}
        total_invalid = 0

        # Monitor completion
        with Progress() as p_bar:
            c_task = p_bar.add_task(f"Performing Gesture Encoding on dataset: ", total=len(participants))
            for future in as_completed(futures):
                while not completion_queue.empty():
                    participant_id, participant_encoded_gestures, participant_missing_trials, msg = completion_queue.get()
                    print(f"{participant_id} Processed")
                    
                    if participant_missing_trials is not None and len(participant_missing_trials) > 0:
                        invalid_trials[participant_id] = participant_missing_trials.copy()
                        total_invalid += len(participant_missing_trials)
                        
                        with open(invalid_trials_path, 'w') as invalid_trials_file:
                            json.dump(invalid_trials, invalid_trials_file)

                    if participant_encoded_gestures is None:
                        console.print(f"[red]Unable to encode gestures for participant: {participant_id} - {msg}[/red]")
                        p_bar.update(c_task, advance=1)
                        continue
                    
                    pointing_gestures_df = None
                    if len(participant_encoded_gestures) > 0:
                        pointing_gestures_df = pd.DataFrame(participant_encoded_gestures)
                        pointing_gestures_df.reset_index(drop=True, inplace=True)
                    
                    #  Write a file covering the stats for excluded trials, coverage, etc...
                    if encoded_gestures is not None:
                        old_data = encoded_gestures[encoded_gestures["PARTICIPANT"]==participant_id]
                        encoded_gestures = encoded_gestures.drop(old_data.index)
                        encoded_gestures = encoded_gestures.reset_index(drop=True)
                        
                        os.replace(encoded_gestures_path, f"{encoded_gestures_path}.bk") # make a backup

                        if pointing_gestures_df is not None:
                            encoded_gestures = pd.concat([encoded_gestures, pointing_gestures_df], ignore_index=True)
                            encoded_gestures = encoded_gestures.reset_index(drop=True)
                    else:
                        if pointing_gestures_df is not None:
                            encoded_gestures = pointing_gestures_df

                    if encoded_gestures is not None:
                        with open(encoded_gestures_path, "w") as gesture_encodings_file:
                            encoded_gestures.to_csv(gesture_encodings_file, index_label="index", lineterminator='\n')
                
                    p_bar.update(c_task, advance=1)
