#!/bin/bash
# lines starting with #SBATCH are options for the sbatch command
# version 03-09-2016

# Name of job and destinations for outputs

#SBATCH --job-name='res2_bands'
#SBATCH --output=StdOut.%j
#SBATCH --error=StdErr.%j

# Number of nodes (here 4 nodes with 16 CPUs each)
# The total number of nodes passed to mpirun will be nodes*ppn

#SBATCH --nodes=4
#SBATCH --ntasks-per-node=16

# Specify the account type and usage limits

#SBATCH --account=free
#SBATCH --mem=60000
#SBATCH --time=6:00:00

#SBATCH --mail-user=pysdw@bath.ac.uk
#SBATCH --mail-type=ALL

module purge
module load untested
module avail
module load slurm
module load espresso/5.2.0 

# check whether echo has the -e option
if test "`echo -e`" = "-e" ; then ECHO=echo ; else ECHO="echo -e" ; fi

# read in parameters from one file - restarts easier as restart scripts will pick up the same changes. 
. ./rel-pbesol_3D_res2/parametersG.txt
$ECHO "  Prefix is $prefix"

TMP_DIR="/home/e/pysdw/scratch/$TMP"
PSEUDO_DIR="/home/e/pysdw/pseudo"

# run from directory where this script is
TOPLEVEL_DIR=`pwd`
RESULTS_DIR=$TOPLEVEL_DIR/$RESULTS

PW_COMMAND='mpirun -np 64 pw.x'
PH_COMMAND='mpirun -np 64 ph.x'
DYNMAT_COMMAND='mpirun -np 1 dynmat.x'
MATDYN_COMMAND='mpirun -np 1 matdyn.x'
Q2R_COMMAND='mpirun -np 1 q2r.x'
PLOTBAND_COMMAND='mpirun -np 1 plotband.x'
PLOTRHO_COMMAND='mpirun -np 1 plotrho.x'
BANDS_COMMAND='mpirun -np 64 bands.x'
PROJWFC_COMMAND='mpirun -np 64 projwfc.x'
DOS_COMMAND='mpirun -np 64 dos.x'

if [ $computer != 'balena' ] 
   then
	# set the needed environment variables (for Feynman at pReSnt)
	. ../../environment_variables 

	# required executables and pseudopotentials (to be completed)
	BIN_LIST="pw.x ph.x q2r.x matdyn.x"
	PSEUDO_LIST=" "

	$ECHO "  checking that needed directories and files exist...\c"

	# check for directories
	for DIR in "$BIN_DIR" "$PSEUDO_DIR" ; do
	if test ! -d $DIR ; then
		$ECHO
		$ECHO "ERROR: $DIR not existent or not a directory"
		$ECHO "Aborting"
		exit 1
	fi
	done

	# check for executables
	for FILE in $BIN_LIST ; do
	if test ! -x $BIN_DIR/$FILE ; then
		$ECHO
		$ECHO "ERROR: $BIN_DIR/$FILE not existent or not executable"
		$ECHO "Aborting"
		exit 1
	fi
	done

	# check for pseudopotentials
	for FILE in $PSEUDO_LIST ; do
	if test ! -r $PSEUDO_DIR/$FILE ; then
		$ECHO
		$ECHO "ERROR: $PSEUDO_DIR/$FILE not existent or not readable"
		$ECHO "Aborting"
		exit 1
	fi
	done
	$ECHO " done"

        # how to run executables
	PW_COMMAND="$PARA_PREFIX $BIN_DIR/pw.x $PARA_POSTFIX"
	PH_COMMAND="$PARA_PREFIX $BIN_DIR/ph.x $PARA_POSTFIX"
	Q2R_COMMAND="$PARA_PREFIX $BIN_DIR/q2r.x $PARA_POSTFIX"
	MATDYN_COMMAND="$PARA_PREFIX $BIN_DIR/matdyn.x $PARA_POSTFIX"
	DYNMAT_COMMAND="$PARA_PREFIX $BIN_DIR/dynmat.x $PARA_POSTFIX"
	PLOTRHO_COMMAND="$BIN_DIR/plotrho.x"
	BANDS_COMMAND="$PARA_PREFIX $BIN_DIR/bands.x $PARA_POSTFIX"
	PLOTBAND_COMMAND="$BIN_DIR/plotband.x"
        PROJWFC_COMMAND="$PARA_PREFIX $BIN_DIR/projwfc.x $PARA_POSTFIX"
        DOS_COMMAND="$PARA_PREFIX $BIN_DIR/dos.x $PARA_POSTFIX"
   fi

$ECHO
$ECHO "  executables directory: $BIN_DIR"
$ECHO "  pseudo directory:      $PSEUDO_DIR"
$ECHO "  temporary directory:   $TMP_DIR"
$ECHO "  results directory:     $RESULTS_DIR"

$ECHO
$ECHO "  running pw.x as:     $PW_COMMAND"
$ECHO "  running ph.x as:     $PH_COMMAND"
$ECHO "  running q2r.x as:    $Q2R_COMMAND"
$ECHO "  running matdyn.x as: $MATDYN_COMMAND"
$ECHO "  running dynmat.x as: $DYNMAT_COMMAND"
$ECHO "  running bands.x as:  $BANDS_COMMAND"
$ECHO "  running plotband.x as: $PLOTBAND_COMMAND"
$ECHO "  running q2r.x as:    $Q2R_COMMAND"

run_host=hostname
run_date=date
run_dir=pwd

export OMP_NUM_THREADS=1

# not a restart; saving the script   

for DIR in "$TMP_DIR" "$RESULTS_DIR" ; do
if test ! -d $DIR ; then
	mkdir $DIR
fi
done

cp $0 $RESULTS_DIR
cp ./res2_postproc.c $RESULTS_DIR
cp ./rex2_ARPES_04.c  $RESULTS_DIR
cp ./organise_bands.c $RESULTS_DIR
cp ./projwfc_reader.c $RESULTS_DIR
cp ./rel-pbesol_3D_res2/* $RESULTS_DIR
cd $RESULTS_DIR

cat > run_details.txt << EOF
`$run_host`
`$run_date`
`$run_dir`
EOF

#========================================================================

# Extract relevant data from these lines and store in the correct format for use as an input

current_cell_parameters=$(tail -3 res2_3D_raw_cell_parameters)
current_atomic_positions=$(tail -$nats res2_3D_raw_cartesian_atomic_positions)

#
# calculation using relaxed parameters
#

cat > res2.scf.final.in << EOF
 &control
    calculation='scf'
    verbosity='high',
    restart_mode='from_scratch',
    prefix='$prefix',
    pseudo_dir = '$PSEUDO_DIR',
    outdir='$TMP_DIR',
    tprnfor=.true.,
    tstress=.true.,
 /
 &system    
    ibrav= 0, 
    A=   $celldmA,
    nat=  $nats, ntyp= $ntyp,
    lspinorb=$lspinorb, 
    noncolin=$noncolin,
    ecutwfc =$ecut,
    ecutrho = $rcut
 /
 &electrons
    diagonalization='cg',
    conv_thr =  $elthresh,
    mixing_mode='plain',
    mixing_beta = 0.3,
 /
CELL_PARAMETERS (alat)
$current_cell_parameters
ATOMIC_SPECIES
$atom1
$atom2
ATOMIC_POSITIONS (alat)
$current_atomic_positions
K_POINTS {automatic}
 $nkx $nky $nkz 0 0 0
EOF

#$ECHO " running the final scf calculation ; \c"
#$PW_COMMAND -input res2.scf.final.in &> res2.scf.final.out
#$ECHO " done"

grep 'highest' res2.scf.final.out > homo.txt
homo=$(awk '{print $NF}' homo.txt)
cc res2_postproc.c -lm -o res2_postproc
./res2_postproc $homo 2.000
emin=$(awk 'NR==1' energy_range.txt)
emax=$(awk 'NR==2' energy_range.txt)  

grep 'crystal' res2.scf.final.out -A4 > real_space.txt
grep 'reciprocal' res2.scf.final.out -A4 > recip.txt
alat=$(awk '/lattice parameter/ { print $5 }' res2.scf.final.out)

cc rex2_ARPES_04.c -lm -o rex2_ARPES_04
cc organise_bands.c -lm -o organise_bands
cc projwfc_reader.c -lm -o projwfc_reader

# to enter by hand comment out this:  
./rex2_ARPES_04 $alat 40 1 $curved $MM $NN 100 19.1

# to enter by hand, uncomment this:
# cp ../kvects_crystal.txt ./

#
# band structure calculation over grid of k points
#

cat > res2.band.head << EOF
 &control
    calculation='bands'
    restart_mode='from_scratch',
    prefix='$prefix',
    pseudo_dir = '$PSEUDO_DIR',
    outdir='$TMP_DIR',
    tprnfor=.true.,
    tstress=.true.,
 /
 &system    
    ibrav= 0, 
    A=   $celldmA,
    nat=  $nats, ntyp= $ntyp, nbnd=$nband,
    lspinorb=$lspinorb, 
    noncolin=$noncolin,
    ecutwfc =$ecut,
    ecutrho = $rcut
 /
 &electrons
    diagonalization='cg',
    conv_thr =  $elthresh,
    mixing_mode='plain',
    mixing_beta = 0.3,
 /
CELL_PARAMETERS (alat)
$current_cell_parameters
ATOMIC_SPECIES
$atom1
$atom2
ATOMIC_POSITIONS (alat)
$current_atomic_positions
EOF

# need a loop that goes over K1-K3 and M1-M3 - all way down to bottom, need to run organise on each so it needs output filename argument

for KDIR in "M1" "M2" "M3" "K1" "K2" ; do

cat res2.band.head kvects_tpiba_$KDIR.txt > res2.band.$KDIR.in

$ECHO "  running the band-structure calculation for res2...\c"
$PW_COMMAND < res2.band.$KDIR.in > res2.band.$KDIR.out
$ECHO " done"

# post-processing for band structure
cat > res2.bands.$KDIR.in << EOF
 &bands
    prefix='$prefix',
    outdir = '$TMP_DIR/'
    filband = 'res2bands.$KDIR.dat'
    lsym=.true.,
 /
EOF
$ECHO "  running the post-processing for band structure...\c"
$BANDS_COMMAND < res2.bands.$KDIR.in > res2.bands.$KDIR.out
$ECHO " done"

#
# rearrange the output for plotting and scale the k axes into Angstrom^{-1}
#
./organise_bands res2bands.$KDIR.dat $alat

#
# Keep the naming of the final result simple for Origin
#

mv ./bands.dat $KDIR.dat

#
# NEW 29-07-2016: run projwfc to analyse the wavefunctions
#

cat > pdos.$KDIR.in <<EOF
&projwfc
   outdir='$TMP_DIR/',
   prefix='$prefix',
   ngauss=0,
   degauss=0.005,
   Emin=$emin,
   Emax=$emax,
   DeltaE=0.02,
   kresolveddos=.true.,
/
EOF

$ECHO " running the PDOS calculation "
$PROJWFC_COMMAND < pdos.$KDIR.in > pdos.$KDIR.out
$ECHO " done "

#
# move the projections to a subdirectory since names not under my control
#

mkdir pdos_$KDIR
mv $prefix.pdos* ./pdos_$KDIR
cp $KDIR.dat ./pdos_$KDIR

#
# process those pdos files required (selected in the C code)
#

cp projwfc_reader ./pdos_$KDIR
cd ./pdos_$KDIR

FILES=`ls $prefix*`
#echo $FILES

for f in $FILES ; do
   #echo $f
   ./projwfc_reader ../kvects_inv_ang_M.txt $f $KDIR
done
cd ..

done

$ECHO
$ECHO "$EXAMPLE_DIR: done"






