#include <stdio.h>
#include <stdlib.h>
#include <math.h>

// version 5-12-2016 (nothing actually changed)

// convert Bohr to Angstrom
#define B2A 0.529177

#define PI  3.14159265359

// sqrt(2*m_e)/hbar in units A^-1 eV^-0.5
#define r2moh 0.51232

// excitation energy in eV, analyser work function, inner potential
// in this version, hv and Vinner are arguments 

//#define hv 97
#define wf 4.2
//#define Vinner 19.1

// ad, bd, cd: real space lattice vectors read in from QE
// as, bs, cs: reciprocal space lattice vectors read in from QE
// asc, bsc, csc: reciprocal space lattice vectors calculated here from real space ones.
// G, Z, K, M: special points in k space
// as2d, bs2D: reciprocal space lattice vectors projected onto the plane normal to cs, cs2D.

double Kx, Ky, Kz, Vx, Vy, Vz, hv, Vinner;
int steps, MM, NN, curved;
double alat, alatA, BZmult;
double AA, BB, CC, Alpha, Beta, Gamma;
double kk, ad[4], bd[4], cd[4], dd[4], ee[4], as[4], bs[4], cs[4], as2D[4], bs2D[4], cs2D[4];
double G[4], Z[4], M1[4], K1[4], M2[4], K2[4], M3[4], K3[4];
double asc[4], bsc[4], csc[4];

//=====================================================================
//
// 16-06-16. Implement curvature in k space to simulate ARPES: setup()
//
//=====================================================================

int scale(double a[4], double s)
{
  int i;

  for (i=1; i<4; i++) {
    a[i] *= s;
  }

  return 0;
}

double dotprod(double a[4], double b[4])
{
  return (a[1]*b[1] + a[2]*b[2] + a[3]*b[3]);
}

int crossprod(double a[4], double b[4], double c[4])
{
  c[1] =  a[2]*b[3]-a[3]*b[2];
  c[2] = -a[1]*b[3]+a[3]*b[1];
  c[3] =  a[1]*b[2]-a[2]*b[1];
  return 0;
}

double to_degrees(double theta)
{
  return 180.0*theta/PI;
}

double to_radians(double theta)
{
  return PI*theta/180.0;
}

// Given two (reciprocal) lattice vectors a and b assumed in the real space layer plane that define two M points
// (i.e, these could be projections of the true a*, b* onto the plane normal to c*)
// then find the coordinates of the K point between them (not trivial)
// This is not halfway between them, as appears to be assumed in Xcrysden

int find_k(double a[4], double b[4])
{
  Kx = a[2]/a[1] * (a[1]+a[2]*b[2]/b[1] - b[1] - b[2] * b[2] / b[1]) / (-a[2] / a[1] + b[2] / b[1]) / 0.2e1 + a[1] / 0.2e1;
  Ky = -(a[1]+a[2]*b[2] / b[1] - b[1] - b[2] * b[2] / b[1]) / (-a[2] / a[1] + b[2] / b[1]) / 0.2e1 + a[2] / 0.2e1;
  Kz = 0.0;

  return 0;
}

// find the magnitude of a vector

double mag(double a[4])
{
  return sqrt(dotprod(a,a));
}

// project a vector v onto a basis a,b,c. Returns result in a global variable V[].

int project(double v[4], double a[4], double b[4], double c[4])
{
  Vx = (b[1] * c[2] * v[3] - b[1] * c[3] * v[2] - c[1] * b[2] * v[3] + v[1] * b[2] * c[3] + c[1] * b[3] * v[2] - v[1] * b[3] * c[2]) / (a[1] * b[2] * c[3] - a[1] * b[3] * c[2] - a[2] * b[1] * c[3] + a[2] * b[3] * c[1] + a[3] * b[1] * c[2] - a[3] * b[2] * c[1]);

  Vy = -(a[1] * c[2] * v[3] - a[1] * c[3] * v[2] - a[2] * c[1] * v[3] + a[2] * c[3] * v[1] + a[3] * c[1] * v[2] - a[3] * c[2] * v[1]) / (a[1] * b[2] * c[3] - a[1] * b[3] * c[2] - a[2] * b[1] * c[3] + a[2] * b[3] * c[1] + a[3] * b[1] * c[2] - a[3] * b[2] * c[1]);

  Vz = (a[1] * b[2] * v[3] - a[1] * b[3] * v[2] - a[2] * b[1] * v[3] + a[2] * b[3] * v[1] + a[3] * b[1] * v[2] - a[3] * b[2] * v[1]) / (a[1] * b[2] * c[3] - a[1] * b[3] * c[2] - a[2] * b[1] * c[3] + a[2] * b[3] * c[1] + a[3] * b[1] * c[2] - a[3] * b[2] * c[1]);

  return 0;
}

// use the QE real space lattice to calculate my own recip lattice as a check
//
int calc_recip_vectors(double a[4], double b[4], double c[4], double ar[4], double br[4], double cr[4])
{
  double vol, s, t[4];

  crossprod(b,c,t);
  vol = dotprod(a,t);
  s = 1.0/vol;

  crossprod(b,c,ar);
  crossprod(c,a,br);
  crossprod(a,b,cr);

  scale(ar,s);
  scale(br,s);
  scale(cr,s);

  return 0;
}


// replace the QE reciprocal space lattice with my own calculated recip lattice for testing with dummy values
// take real space vectors in units of alat: resulting units will be (2*Pi/alat).
int use_my_recip_lattice(double a[4], double b[4], double c[4], double ar[4], double br[4], double cr[4])
{
  double vol, s, t[4];

  crossprod(b,c,t);
  vol = dotprod(a,t);
  s = 1.0/vol;

  crossprod(b,c,ar);
  crossprod(c,a,br);
  crossprod(a,b,cr);

  scale(ar,s);
  scale(br,s);
  scale(cr,s);

  printf("QE recip lattice replaced by my calculated ones!\n\n");

  return 0;
}


// read in the reciprocal and real space lattice vectors from grep output applied to an ...scf.out file.
// these wll be in units of (alat) and (2*Pi/alat) respectively.
int get_recip_vects()
{
  int i;
  char dummy[30],v1[30],v2[30],v3[30];
  FILE *fq;

  fq = fopen("recip.txt","r");
  for (i=1; i<12; i++) {
     fscanf(fq,"%s",dummy);
  }

  fscanf(fq,"%s %s %s %s",v1,v2,v3,dummy);
  as[1]=atof(v1);
  as[2]=atof(v2);
  as[3]=atof(v3);

  for (i=1; i<4; i++) {
     fscanf(fq,"%s",dummy);
  }

  fscanf(fq,"%s %s %s %s",v1,v2,v3,dummy);
  bs[1]=atof(v1);
  bs[2]=atof(v2);
  bs[3]=atof(v3);

  for (i=1; i<4; i++) {
     fscanf(fq,"%s",dummy);
  }

  fscanf(fq,"%s %s %s %s",v1,v2,v3,dummy);
  cs[1]=atof(v1);
  cs[2]=atof(v2);
  cs[3]=atof(v3);

  fclose(fq);

  return 0;
}

int get_real_vects()
{
  int i;
  char dummy[30],v1[30],v2[30],v3[30];
  FILE *fq;

  fq = fopen("real_space.txt","r");
  for (i=1; i<12; i++) {
     fscanf(fq,"%s",dummy);
  }

  fscanf(fq,"%s %s %s %s",v1,v2,v3,dummy);
  ad[1]=atof(v1);
  ad[2]=atof(v2);
  ad[3]=atof(v3);

  for (i=1; i<4; i++) {
     fscanf(fq,"%s",dummy);
  }

  fscanf(fq,"%s %s %s %s",v1,v2,v3,dummy);
  bd[1]=atof(v1);
  bd[2]=atof(v2);
  bd[3]=atof(v3);

  for (i=1; i<4; i++) {
     fscanf(fq,"%s",dummy);
  }

  fscanf(fq,"%s %s %s %s",v1,v2,v3,dummy);
  cd[1]=atof(v1);
  cd[2]=atof(v2);
  cd[3]=atof(v3);

  fclose(fq);

  return 0;
}


int calc_ABC(double a[4], double b[4], double c[4])
{
  AA = mag(a);
  BB = mag(b);
  CC = mag(c);

  Alpha = acos(dotprod(b,c)/BB/CC);
   Beta = acos(dotprod(a,c)/AA/CC);
  Gamma = acos(dotprod(a,b)/AA/BB);

  AA *= B2A * alat;
  BB *= B2A * alat;
  CC *= B2A * alat;

  return 0;
}

int setup_special_points()
{
  // unlike some earlier versions, use tpiba units
  // but need to use A-1 for calculation of hemisphere, then convert
  G[1]=0.0;
  G[2]=0.0;
  G[3]=0.0;

  Z[1]=0.0;
  Z[2]=0.0;
  Z[3]=0.5;

  M1[1]=as[1]/2.0;
  M1[2]=as[2]/2.0;
  M1[3]=0.0;

  //find_k(as,bs);
  //K1[1]=Kx;
  //K1[2]=Ky;
  //K1[3]=0.0;

  // see earlier versions for calculation of the rest of the "K" points

  M2[1]=bs[1]/2.0;
  M2[2]=bs[2]/2.0;
  M2[3]=0.0;

  M3[1]=(bs[1]-as[1])/2.0;
  M3[2]=(bs[2]-as[2])/2.0;
  M3[3]=0.0;

  return 0;
}

// following routine is in the end not used - it rotates the basis slightly but there's no
// way for the atomic positions to be updated by the same rotation, so useful only as a check

int make_2D_basis_from_ABC()
{
  double vol, alphat, betat, gammat, T1[4], T2[4], T3[4], temp[4];

  alphat = to_radians(90.0);
  betat  = alphat;
  gammat = Gamma;

  // First create the 2D real-space lattice vectors T1..T3
  T1[1] = AA;
  T1[2] = 0.0;
  T1[3] = 0.0;

  T2[1] = BB*cos(gammat);
  T2[2] = BB*sin(gammat);
  T2[3] = 0.0;

  T3[1] = 0.0;
  T3[2] = 0.0;
  T3[3] = CC;

  crossprod(T1,T2,temp);
  vol = dotprod(T3,temp);
  //printf("volume %f\n\n",vol);

  // make the 2D reciprocal lattice vectors as2D..cs2D
  crossprod(T2,T3,as2D);
  crossprod(T3,T1,bs2D);
  crossprod(T1,T2,cs2D);

  scale(as2D,2*PI/vol);
  scale(bs2D,2*PI/vol);
  scale(cs2D,2*PI/vol);

  return 0;
}

// simple approach: project recip lattice vectors a*, b* onto plane normal to c*
// tpiba units
int make_2D_basis()
{
  as2D[1] = as[1];
  as2D[2] = as[2];
  as2D[3] = 0.0;

  bs2D[1] = bs[1];
  bs2D[2] = bs[2];
  bs2D[3] = 0.0;

  cs2D[1] = 0.0;
  cs2D[2] = 0.0;
  cs2D[3] = cs[3];

  return 0;
}


// make sure file headers exist, and start from scratch, then append k points from now on
int init_output_files(int steps)
{
  FILE *fq;

  remove("kvects_tpiba.txt");
  remove("kvects_inv_ang.txt");

  fq = fopen("kvects_tpiba.txt","w");
  fprintf(fq,"K_POINTS { tpiba }\n");
  fprintf(fq,"%d\n",(2*steps+1)*(2*steps+1));
  fclose(fq);

  fq = fopen("kvects_inv_ang.txt","w");
  fprintf(fq,"K_POINTS { inverse angstrom }\n");
  fprintf(fq,"%d\n",(2*steps+1)*(2*steps+1));
  fclose(fq);

  return 0;
}


int write_one_point(double K[4])
{
  FILE *fq;

  fq = fopen("kvects_tpiba.txt","a");
  fprintf(fq,"%lf %lf %lf %lf\n",K[1],K[2],K[3], 1.0);
  fclose(fq);

  fq = fopen("kvects_inv_ang.txt","a");
  //project(K, as, bs, cs); this would be needed for crystal units
  fprintf(fq,"%lf %lf %lf %lf\n", 2*PI*K[1]/alatA, 2*PI*K[2]/alatA, 2*PI*K[3]/alatA, 1.0);
  fclose(fq);

  return 0;
}

// for plotting in Origin on top of contour plots; these are the lattice vectors
int write_basis(double K[4], double L[4], double M[4], double N[4])
{
  FILE *fq;

  fq = fopen("ad_basis_tpiba.txt","w");
  fprintf(fq,"%lf %lf %lf \n",0.0,0.0,0.0);
  fprintf(fq,"%lf %lf %lf \n",K[1],K[2],K[3]);
  fclose(fq);

  fq = fopen("bd_basis_tpiba.txt","w");
  fprintf(fq,"%lf %lf %lf \n",0.0,0.0,0.0);
  fprintf(fq,"%lf %lf %lf \n",L[1],L[2],L[3]);
  fclose(fq);

  fq = fopen("as_basis_tpiba.txt","w");
  fprintf(fq,"%lf %lf %lf \n",0.0,0.0,0.0);
  fprintf(fq,"%lf %lf %lf \n",M[1],M[2],M[3]);
  fclose(fq);

  fq = fopen("bs_basis_tpiba.txt","w");
  fprintf(fq,"%lf %lf %lf \n",0.0,0.0,0.0);
  fprintf(fq,"%lf %lf %lf \n",N[1],N[2],N[3]);
  fclose(fq);

  return 0;
}


double get_k_perp(double K[4])
{
  double Ekin, csth, kparasq, kperp, convert, kint;

  // Units of kperp, kpara need to be A^-1 to use formula but K[] is in tpiba units
  // so need to multiply by tpiba, with a in any required units - here, Angstrom
  convert = (2*PI)/(alatA);

  Ekin = hv - wf; // ignoring binding energy for now

  // calc kperp, which has to agree with experiment that we are at the Z point at 98 eV
  // this only follows if we use a fictitious value of Vinner which is larger than the real
  // value because (for example) LDA has underestimated the lattice parameters (or I'm using ReS2 data!!!).
  kparasq = convert*convert*dotprod(K,K);
  csth = 1.0 - kparasq/Ekin/(r2moh*r2moh);

  if (curved) {
    kperp = r2moh * sqrt(Ekin*csth + Vinner);
  }
  else {
    kperp = 0.0; // at present, flat means passing through Gamma.
  }

  // kperp in A-1 units, so divide by |cs| in A-1 for K[] as a multiple of the c* reciprocal lattice vector
  // to check we get a half-integer value at normal incidence (experimental fact)
  // and need to divide by the conversion factor only to get kperp in units of (2 Pi / alat).
  kint = kperp / (convert*cs[3]);
  K[3] = (kperp / convert); // older versions use this. Next line is equivalent if kint is in brackets.
  K[3] = (kint) * cs[3];

  return kint;
}


//
// calculate the Cartesian components of each point in reciprocal space, including k_perp curvature.
// this is only going to work if c* is exactly along (0 0 1), so need to prevent rx runs from changing that.
// am trying cell_dofree='xy' to forbid varying the c axis: this works.
//
// If we are scanning through a BZ point not at normal incidence, then it is not at the top of the hemisphere.
// Then, we have to choose a kpara

int setup(int ns, int m, int n, double *K)
{
  double sa,sb,va[4],vb[4],convert,kint;
  FILE *fp;

  convert = (2*PI)/(alatA);
  sa = 1.0/(double)ns;
  sb = 1.0/(double)ns;

  va[1]=(MM + BZmult*m*sa)*as2D[1];
  va[2]=(MM + BZmult*m*sa)*as2D[2];
  va[3]=0.0;

  vb[1]=(NN + BZmult*n*sb)*bs2D[1];
  vb[2]=(NN + BZmult*n*sb)*bs2D[2];
  vb[3]=0.0;

  K[1] = va[1] + vb[1];
  K[2] = va[2] + vb[2];
  K[3] = 0.0;

  kint = get_k_perp(K);

  if ((n==0) && (m==0)) {
     fp = fopen("centre.txt","w");
     fprintf(fp,"%f %f %f\n",convert*K[1],convert*K[2],convert*K[3]);
     printf("=======================================================\n");
     printf("cs[3] = %f tpiba, %f A-1\n",cs[3],cs[3]*convert);
     printf("kint=%f K[3] = %f tpiba \n", kint, K[3]);
     printf("=======================================================\n");
     fclose(fp);
  }

  return 0;
}


// sort out the steps in reciprocal space for the grid
int generate_grid(int steps)
{
  double KK[4];
  int i, j, count;

  // prepare the outputfiles
  init_output_files(steps);

  count=0;

  printf("\n");
  printf("GENERATE_GRID Generating grid of %d points in tpiba units.\n",(2*steps+1)*(2*steps+1));

  for (j=-steps; j<steps+1; j++) {
    for (i=-steps; i<steps+1; i++) {
      count++;
      setup(steps,i,j,KK);
      if (steps < 3) printf("GENERATE_GRID %5d %5d %12f %12f %12f\n",i,j,KK[1],KK[2],KK[3]);
      write_one_point(KK);
    }
  }
  printf("\n");

  write_basis(ad,bd,as,bs);

  return 0;
}


// make a 3D grid of k points for a constant energy surface plot.
// make this routine self-contained as it does not need k_perp considerations.
// divide the vector by 2 at teh last stage to get one primitive BZ centred on Gamma
int generate_volume(int steps)
{
  double KK[4], sa;
  int i, j, k, count;
  FILE *fp;

  fp = fopen("kvects_vol_tpiba.txt","w");
  fprintf(fp,"K_POINTS { tpiba }\n");
  fprintf(fp,"%d\n",(2*steps)*(2*steps)*(2*steps));

  count=0;
  sa = 1.0/(double)steps;

  printf("\n");
  printf("GENERATE_VOLUME Making volume of %d points in tpiba units.\n\n",(2*steps)*(2*steps)*(2*steps));

  for (k=-steps; k<steps; k++) {
    for (j=-steps; j<steps; j++) {
      for (i=-steps; i<steps; i++) {
          count++;
          KK[1]=(i*sa*as[1]) + (j*sa*bs[1]) + (k*sa*cs[1]);
          KK[2]=(i*sa*as[2]) + (j*sa*bs[2]) + (k*sa*cs[2]);
          KK[3]=(i*sa*as[3]) + (j*sa*bs[3]) + (k*sa*cs[3]);
          fprintf(fp,"%lf %lf %lf %lf\n",KK[1]/2.0,KK[2]/2.0,KK[3]/2.0, 1.0);
      }
    }
  }

  fclose(fp);
  return 0;
}


// Arguments are number of steps, offset integers of centre point
// Last two integers are same as those for setup()
// Here, points have to be in tpiba format (not tpiba_b) because K_perp can't be left to QE generation
// Adding one K point, orthogonal to M2

int generate_path(int ns)
{
  FILE *fp, *fq, *fr, *fs, *ft, *fu;
  double GZ[4], sa, kint, kpara1, kpara2, kpara3, kpara4, kpara5, convert, ascale;
  int i;

  convert = (2*PI)/(alatA);
  ascale = AA / 6.623; // computational over experimental lattice parameter, to scale k vectors below. 

  // GZ[] contains the in-plane coordinates of the centre point of the required cuts
  GZ[1] = MM*as2D[1] + NN*bs2D[1];
  GZ[2] = MM*as2D[2] + NN*bs2D[2];
  GZ[3] = 0.0;

  remove("kvects_tpiba_M1.txt");
  remove("kvects_tpiba_M2.txt");
  remove("kvects_tpiba_M3.txt");
  remove("kvects_tpiba_K1.txt");
  remove("kvects_tpiba_K2.txt");
  remove("kvects_tpiba_K3.txt");

  // going a little bit further than the next "tube" along - original factor was 1.0 here
  sa = 1.5/(double)ns;

  fp = fopen("kvects_tpiba_M1.txt","w");
  fprintf(fp,"K_POINTS { tpiba }\n");
  fprintf(fp,"%d\n",2*ns+1);

  fq = fopen("kvects_tpiba_M2.txt","w");
  fprintf(fq,"K_POINTS { tpiba }\n");
  fprintf(fq,"%d\n",2*ns+1);

  fr = fopen("kvects_tpiba_M3.txt","w");
  fprintf(fr,"K_POINTS { tpiba }\n");
  fprintf(fr,"%d\n",2*ns+1);

  fu = fopen("kvects_tpiba_K1.txt","w");
  fprintf(fu,"K_POINTS { tpiba }\n");
  fprintf(fu,"%d\n",2*ns+1);

  ft = fopen("kvects_tpiba_K2.txt","w");
  fprintf(ft,"K_POINTS { tpiba }\n");
  fprintf(ft,"%d\n",2*ns+1);

  fs = fopen("kvects_inv_ang_M.txt","w");
  fprintf(fs,"M1   M2   M3   K1  K2\n");

  for (i=-ns; i<ns+1; i++) {

     M1[1]=i*sa*as2D[1] + GZ[1];
     M1[2]=i*sa*as2D[2] + GZ[2];
     M1[3]=0.0;
     kint = get_k_perp(M1); // modifies M1[3] to the correct kz value; kint discarded here. 
     kpara1 = convert*copysign(sqrt(pow(i*sa*as2D[1],2) + pow(i*sa*as2D[2],2)),i);

     M2[1]=i*sa*bs2D[1] + GZ[1];
     M2[2]=i*sa*bs2D[2] + GZ[2];
     M2[3]=0.0;
     kint = get_k_perp(M2);
     kpara2 = convert*copysign(sqrt(pow(i*sa*bs2D[1],2) + pow(i*sa*bs2D[2],2)),i);

     M3[1]=i*sa*(bs2D[1]-as2D[1]);
     M3[2]=i*sa*(bs2D[2]-as2D[2]);
     kpara3 = convert*copysign(sqrt(pow(M3[1],2) + pow(M3[2],2)),i);
     M3[1] += GZ[1];
     M3[2] += GZ[2];
     M3[3]=0.0;
     kint = get_k_perp(M3);

     // construct K2 as if orthogonal to M2: is what was done in expt (is kx or ky dep. on definitions)
     // exchange indices of bs2D and change one sign == rotation by Pi/2 of the 2D vector.
     // rotation vector of path only, not of centre point offset.
     K2[1]=-i*sa*bs2D[2] + GZ[1];
     K2[2]=+i*sa*bs2D[1] + GZ[2];
     K2[3]=0.0;
     kint = get_k_perp(K2);
     kpara4 = convert*copysign(sqrt(pow(i*sa*bs2D[2],2) + pow(i*sa*bs2D[1],2)),i);

     // construct K1 as 2bs-as; this is only approximate, as the point halfway between M2 and M3
     K1[1]=i*sa*bs2D[1] + i*sa*(bs2D[1]-as2D[1]) + GZ[1];
     K1[2]=i*sa*bs2D[2] + i*sa*(bs2D[2]-as2D[2]) + GZ[2];
     K1[3]=0.0;
     kint = get_k_perp(K1);
     kpara5 = convert*copysign(sqrt(pow(i*sa*bs2D[1]+i*sa*(bs2D[1]-as2D[1]),2) + pow(i*sa*bs2D[2]+i*sa*(bs2D[2]-as2D[2]),2)),i);

     //
     // NEW 03-08: scale the computational vectors to the experimental lattice parameter: only for kvects_inv_ang_M.txt
     //

     fprintf(fp,"%lf %lf %lf 1.0\n",M1[1],M1[2],M1[3]);
     fprintf(fq,"%lf %lf %lf 1.0\n",M2[1],M2[2],M2[3]);
     fprintf(fr,"%lf %lf %lf 1.0\n",M3[1],M3[2],M3[3]);
     fprintf(fu,"%lf %lf %lf 1.0\n",K1[1],K1[2],K1[3]);
     fprintf(ft,"%lf %lf %lf 1.0\n",K2[1],K2[2],K2[3]);
     fprintf(fs,"%lf %lf %lf %lf %lf\n",ascale*kpara1,ascale*kpara2,ascale*kpara3,ascale*kpara5,ascale*kpara4);
  }

  fclose(fp);
  fclose(fq);
  fclose(fr);
  fclose(fu);
  fclose(ft);
  fclose(fs);

  return 0;
}

int generate_kz_path(int ns) {
    FILE *fp, *fq;
    int i;
    double km=6.0, kr=2.0, dk, kzc, kza, convert;
    double Ekin, Eexc;

    dk = kr/(2*ns);
    convert = (2*PI)/(alatA)*cs[3]; // check this!!! 

    fp = fopen("kvects_tpiba_KZ.txt","w");
    fq = fopen("kz_path_energy.txt","w");

    fprintf(fp,"K_POINTS { crystal }\n");
    fprintf(fp,"%d\n",2*ns+1);

    for (i=-ns; i<ns+1; i++) {
        kzc = km + i*dk;
        fprintf(fp,"0.00 0.00 %lf 1.00\n",kzc);

        kza = kzc * convert;
        Ekin = (kza*kza)/(r2moh*r2moh) - Vinner;
        Eexc = Ekin + wf;
        fprintf(fq,"%lf %lf %lf\n",Eexc,kzc,kza);
    }

    fclose(fp);
    fclose(fq);
}





//=====================================================================

int main( int argc, char *argv[] )
{
  // check command line inputs are there
  if (argc!=9) {
    printf("Requires eight arguments: alat (Bohr), no. of k steps, multiplier of BZ, (0=flat,1=curved), int offsets of BZ, hv, Vinner \n");
    exit(1);
  }

  // convert command line inputs. MM, NN are integer offsets of central point in calculated BZ section
  steps = atoi(argv[2]);
  alat  = atof(argv[1]);
  BZmult = atof(argv[3]);
  curved = atoi(argv[4]);
  MM = atoi(argv[5]);
  NN = atoi(argv[6]);
  alatA = B2A * alat;

  // new in version 04
  hv = atof(argv[7]);
  Vinner = atof(argv[8]);

  // Read in file recip.txt generated by grep from an ...scf.out file.
  // Extract the real space and reciprocal space lattice vectors.
  // Calculate the recip lattice vectors as a check
  // Then calculate A..C, Alpha..Gamma
  get_recip_vects();
  get_real_vects();
  calc_recip_vectors(ad,bd,cd,asc,bsc,csc);

  printf("\n");
  printf("MAIN Read in real space axes ad-cd (cart. coord. in units of alat)\n");
  printf("a  = %lf %lf %lf\n",ad[1],ad[2],ad[3]);
  printf("b  = %lf %lf %lf\n",bd[1],bd[2],bd[3]);
  printf("c  = %lf %lf %lf\n",cd[1],cd[2],cd[3]);
  printf("\n");
  printf("MAIN Read in reciprocal axes as-cs (cart. coord. in units of 2*pi/alat)\n");
  printf("a* = %lf %lf %lf\n",as[1],as[2],as[3]);
  printf("b* = %lf %lf %lf\n",bs[1],bs[2],bs[3]);
  printf("c* = %lf %lf %lf\n",cs[1],cs[2],cs[3]);
  printf("\n");
  printf("MAIN Calculated recip axes asc-csc (cart. coord. in units of 2*Pi/alat)\n");
  printf("a* = %lf %lf %lf\n",asc[1],asc[2],asc[3]);
  printf("b* = %lf %lf %lf\n",bsc[1],bsc[2],bsc[3]);
  printf("c* = %lf %lf %lf\n",csc[1],csc[2],csc[3]);
  printf("\n");

  // if uncommented, this puts my calculated recip lattice vectors in (2*Pi/alat) units in place of the ones read in.
  use_my_recip_lattice(ad,bd,cd,as,bs,cs);

  printf("MAIN real space axes alatA*ad-cd (cart. coord. in units of Angstrom)\n");
  printf("a  = %lf %lf %lf\n",alatA*ad[1],alatA*ad[2],alatA*ad[3]);
  printf("b  = %lf %lf %lf\n",alatA*bd[1],alatA*bd[2],alatA*bd[3]);
  printf("c  = %lf %lf %lf\n",alatA*cd[1],alatA*cd[2],alatA*cd[3]);
  printf("\n");
  printf("MAIN reciprocal axes (2*Pi/alatA)*as-cs (cart. coord. in units of inverse Angstrom)\n");
  printf("a* = %lf %lf %lf\n",(2*PI/alatA)*as[1],(2*PI/alatA)*as[2],(2*PI/alatA)*as[3]);
  printf("b* = %lf %lf %lf\n",(2*PI/alatA)*bs[1],(2*PI/alatA)*bs[2],(2*PI/alatA)*bs[3]);
  printf("c* = %lf %lf %lf\n",(2*PI/alatA)*cs[1],(2*PI/alatA)*cs[2],(2*PI/alatA)*cs[3]);
  printf("\n");
  printf("MAIN alat  = %lf Bohr\n", alat);
  printf("MAIN alatA = %lf Angstrom\n", alatA);
  printf("\n");
  printf("MAIN Check: a.c* = %lf\n",ad[1]*cs[1]+ad[2]*cs[2]+ad[3]*cs[3]);
  printf("MAIN Check: b.c* = %lf\n",bd[1]*cs[1]+bd[2]*cs[2]+bd[3]*cs[3]);

  // find the lattice parameters in ABC format for checking. NEW: AA used for scaling computational to experimental k vectors.
  calc_ABC(ad,bd,cd);
  printf("\n");
  printf("MAIN Lattice parameters  (A)     A: %8lf       B: %8lf         C: %8lf\n", AA, BB, CC);
  printf("MAIN Angles            (deg) Alpha: %8lf   Beta: %8lf   Gamma: %8lf\n",to_degrees(Alpha), to_degrees(Beta), to_degrees(Gamma));
  printf("\n");

  // Generate the basis of the 2D quasi-hexagonal grid of projected G..Z axes.
  make_2D_basis();
  printf("MAIN 2D projected recip latt. as_2D: %8lf %8lf %8lf  tpiba\n", as2D[1], as2D[2], as2D[3]);
  printf("MAIN 2D projected recip latt. bs_2D: %8lf %8lf %8lf  tpiba\n", bs2D[1], bs2D[2], bs2D[3]);
  printf("MAIN 2D projected recip latt. cs_2D: %8lf %8lf %8lf  tpiba\n", cs2D[1], cs2D[2], cs2D[3]);

  // make an input file for a "Fermi surface" script to plot energy band(s) over a section of the BZ
  generate_grid(steps);

  // make an input file for a dispersion script to plot bands as a function of a k point path
  // centred about any point on the (domed) surface. Re-use argument 'steps': this may mean this code
  // should be called immediately before the output is used and only one type of calculation
  // at a time will make sense - don't run this once to set up all BS calcs as they may need different
  // values of 'steps'.
  generate_path(steps);
  generate_kz_path(steps);

  // make a 3D grid of points, without k_perp curvature effects, for plotting constant energy surfaces.
  // this definitely needs different number of steps.
  generate_volume(8);

  return 0;
}
