#!/bin/bash
# lines starting with #SBATCH are options for the sbatch command
# 26-09-17

# Name of job and destinations for outputs

#SBATCH --job-name=resse48A
#SBATCH --output=StdOut.o.%j
#SBATCH --error=StdErr.e.%j

# Number of nodes (here 4 nodes with 16 CPUs each)
# The total number of nodes passed to mpirun will be nodes*ppn

#SBATCH --nodes=4
#SBATCH --ntasks-per-node=16

# Specify the account type and usage limits

#SBATCH --account=rc-ph1117
#SBATCH --time=24:00:00

#SBATCH --mail-user=pysdw@bath.ac.uk
#SBATCH --mail-type=ALL

# change to the scratch directory and make subdirectory for the results
# back up this input file in the results directory
 
echo Running on host `hostname` 
echo Start time is `date`
echo Directory is `pwd`

module purge
module load untested
module avail
module load slurm
module load espresso/5.2.0

export OMP_NUM_THREADS=1

TMP_DIR='/home/e/pysdw/scratch/tmp_ph_48A'
PSEUDO_DIR='/home/e/pysdw/pseudo'
RESULTS_DIR='/home/e/pysdw/resse/results_ph_48A'

# not a restart

mkdir $TMP_DIR
mkdir $RESULTS_DIR
cp $0 $RESULTS_DIR
cd $RESULTS_DIR

# How to run QE codes:

PW_COMMAND="mpirun -np 64 pw.x "
PH_COMMAND="mpirun -np 64 ph.x "
DYNMAT_COMMAND="mpirun -np 1 dynmat.x "

# computational parameters (default values commented out for reference)

ecut=60.00       #60
rcut=240.00      #300
elthresh=1.0d-10 #1E-12
fthresh=1.0d-4   #1E-4
ethresh=1.0d-5   #1E-5
nks=4            # 4
phthresh=1.0d-11 #1.0d-12
pthresh=0.5      #0.5

mass1=186.21     # Re
mass2=78.96      # Se
mass3=32.065     # S

atom1='Re  '$mass1'  Re.pz-n-nc.UPF'
atom2='Se  '$mass2'  Se.pz-n-nc.UPF'
atom3='S   '$mass3'  S.pz-n-nc.UPF'

#
# Initial vc-relax calculation with cell parameters and atomic positions expressed in crystal coordinates
#

cat > ReSSe.scf.in << EOF
 &control
    calculation='scf',
    max_seconds=86000,
    restart_mode='from_scratch',
    prefix='ReSSe',
    pseudo_dir = '$PSEUDO_DIR',
    outdir='$TMP_DIR',
    forc_conv_thr=$fthresh,
    etot_conv_thr=$ethresh,
    tprnfor=.true.,
    tstress=.true.,
 /
 &system    
    ibrav=  0, 
    A=13.194,
    nat=  48, ntyp= 3, 
    ecutwfc =$ecut,
    ecutrho = $rcut,
    nbnd= 156,
 /
 &electrons
    conv_thr =  $elthresh,
    mixing_mode='plain',
    mixing_beta = 0.7,
 /
 &ions
    ion_dynamics='bfgs',
 /
&cell
    cell_dynamics='bfgs',
 /
ATOMIC_SPECIES
$atom1
$atom2
$atom3
CELL_PARAMETERS (alat= 24.93304660)
   0.999776838   0.001139289  -0.000172164
  -0.490598021   0.892275713  -0.000049615
  -0.131137118  -0.091492203   0.477684889
ATOMIC_POSITIONS (crystal)
Re       0.033207143   0.158827711   0.492533302
Re       0.242776685   0.145121130   0.498583364
S        0.129108959   0.194405101   0.227613863
Se       0.356971775   0.167494100   0.233748665
Se       0.112277276   0.053474919   0.728120075
Se       0.377974048   0.068219918   0.697032193
Re       0.531251775   0.158093327   0.489340513
Re       0.742841186   0.144444808   0.494686585
Se       0.623261411   0.191330173   0.198493536
Se       0.859728856   0.167755941   0.233525907
Se       0.612124900   0.053444192   0.723708262
Se       0.877821236   0.068469861   0.697294705
Re       0.031709173   0.658167594   0.490366839
Re       0.242605496   0.644395690   0.495078526
Se       0.122968247   0.691172278   0.198822719
Se       0.357945160   0.667036325   0.232496327
Se       0.112399486   0.553525049   0.725083605
Se       0.377986157   0.568851425   0.697309300
Re       0.531480811   0.658002954   0.489837386
Re       0.742692217   0.644617848   0.495632028
Se       0.623270512   0.691211849   0.198846928
Se       0.858271895   0.667444002   0.233452701
Se       0.612253769   0.553569525   0.725125098
Se       0.877787385   0.568400588   0.697054906
Re      -0.031885702  -0.158175505  -0.491375130
Re      -0.243006168  -0.144570476  -0.497901097
Se      -0.123842197  -0.191395008  -0.200775320
Se      -0.358595311  -0.167438725  -0.235830110
Se      -0.111444564  -0.052859299  -0.726074305
Se      -0.378171632  -0.068672982  -0.699811841
Re       0.467851218  -0.158376896  -0.493046015
Re       0.256820858  -0.144752091  -0.497794126
Se       0.376380781  -0.191505204  -0.201716093
Se       0.141659945  -0.167435231  -0.234786599
Se       0.387145158  -0.053629695  -0.727516431
Se       0.121785484  -0.068029913  -0.698239276
Re      -0.031674740   0.341675193  -0.491145679
Re      -0.243105215   0.355142081  -0.497630945
Se      -0.123983222   0.308198740  -0.200998899
Se      -0.358418473   0.332336134  -0.235099888
Se      -0.112218781   0.446366477  -0.726342524
Se      -0.378380873   0.431319158  -0.698477934
Re       0.467938136   0.341683423  -0.490964270
Re       0.257029492   0.355052254  -0.494174794
Se       0.377477516   0.308972074  -0.198000291
Se       0.141460167   0.332401118  -0.231603240
Se       0.386545363   0.445602544  -0.725972461
Se       0.121917204   0.430615522  -0.698510064
K_POINTS {automatic}
 $nks $nks $nks 0 0 0
EOF

#$ECHO " running the scf calculation ; \c"
#$PW_COMMAND -input ReSSe.scf.in &> ReSSe.scf.out
#$ECHO " done"

cat > ReSSe.ph.in <<EOF
phonons of ReSSe
&inputph
   recover=.true.,
   alpha_mix(1)=0.3,
   tr2_ph=$phthresh,
   max_seconds=86000,
   prefix='ReSSe',
   ldisp=.false.,
   trans=.true.,
   epsil=.false.,
   lraman=.false.,
   amass(1)=186.21,
   amass(2)=78.96,
   amass(3)=32.065,
   outdir='$TMP_DIR/',
   fildyn='ReSSe.dyn',
/
0.0   0.0   0.0
EOF

$ECHO " running the phonon calculation"
$PH_COMMAND -input ReSSe.ph.in &> ReSSe.ph.out


cat > dynmat.in <<EOF
&input
   fildyn='ReSSe.dyn',
   q(1)=0.0,
   q(2)=0.0,
   q(3)=0.0,
   asr='simple'
/
EOF

$ECHO " running dynmat"
$DYNMAT_COMMAND -input dynmat.in &> dynmat.out



