#!/bin/bash
# lines starting with #SBATCH are options for the sbatch command
# 14-09-17

# Name of job and destinations for outputs

#SBATCH --job-name=test
#SBATCH --output=StdOut.o.%j
#SBATCH --error=StdErr.e.%j

# Number of nodes (here 4 nodes with 16 CPUs each)
# The total number of nodes passed to mpirun will be nodes*ppn

#SBATCH --nodes=4
#SBATCH --ntasks-per-node=16

# Specify the account type and usage limits

#SBATCH --account=rc-ph1117
#SBATCH --mem=60000
#SBATCH --time=24:00:00

#SBATCH --mail-user=pysdw@bath.ac.uk
#SBATCH --mail-type=ALL

# change to the scratch directory and make subdirectory for the results
# back up this input file in the results directory
 
echo Running on host `hostname` 
echo Start time is `date`
echo Directory is `pwd`

module purge
module load untested
module avail
module load slurm
module load espresso/5.2.0

export OMP_NUM_THREADS=1

TMP_DIR='/home/e/pysdw/scratch/tmp_j'
PSEUDO_DIR='/home/e/pysdw/pseudo'
RESULTS_DIR='/home/e/pysdw/resse/results_j'

# not a restart

mkdir $TMP_DIR
mkdir $RESULTS_DIR
cp $0 $RESULTS_DIR
cd $RESULTS_DIR

# How to run QE codes:

PW_COMMAND='mpirun -np 64 pw.x -nk 16'
PH_COMMAND='mpirun -np 64 ph.x -nk 16'
DYNMAT_COMMAND='mpirun -np 1 dynmat.x'

# computational parameters (default values commented out for reference)

ecut=60.00       #60
rcut=240.00      #300
elthresh=1.0d-12 #1E-12
fthresh=1.0d-5   # 1E-4
ethresh=1.0d-6   #1E-5
nks=8            # 4
phthresh=1.0d-12 #1.0d-12
pthresh=0.5      #0.5

mass1=186.21     # Re
mass2=78.96      # Se
mass3=32.065     # S

atom1='Re  '$mass1'  Re.pz-n-nc.UPF'
atom2='Se  '$mass2'  Se.pz-n-nc.UPF'
atom3='S   '$mass3'   S.pz-n-nc.UPF'

#
# Initial vc-relax calculation with cell parameters and atomic positions expressed in crystal coordinates
#

cat > ReSSe.scf.vcrelax0.in << EOF
 &control
    calculation='vc-relax',
    restart_mode='from_scratch',
    prefix='ReSSe',
    pseudo_dir = '$PSEUDO_DIR',
    outdir='$TMP_DIR',
    forc_conv_thr=$fthresh,
    etot_conv_thr=$ethresh,
    tprnfor=.true.,
    tstress=.true.,
 /
 &system    
    ibrav=  14, 
    A=6.597,
    B=6.710,  
    C=6.721,
    cosAB=-0.48344,
    cosAC=-0.25713,
    cosBC=-0.03211,
    nat=  12, ntyp= 2, 
    ecutwfc =$ecut,
    ecutrho = $rcut
 /
 &electrons
    conv_thr =  $elthresh,
    mixing_mode='plain',
    mixing_beta = 0.3,
 /
 &ions
    ion_dynamics='bfgs',
 /
&cell
    cell_dynamics='bfgs',
 /
ATOMIC_SPECIES
$atom1
$atom2
ATOMIC_POSITIONS {crystal}
Re       0.06315   0.31477   0.49138
Re       0.48723   0.29042   0.49681
Re       0.93685   0.68523   0.50862
Re       0.51277   0.70958   0.50319
Se       0.24816   0.38192   0.20614
Se       0.72049   0.33902   0.23868
Se       0.22610   0.10862   0.72366
Se       0.75496   0.13646   0.69664
Se       0.75184   0.61808   0.79386
Se       0.27951   0.66098   0.76132
Se       0.77390   0.89138   0.27634
Se        0.24504   0.86354   0.30336
K_POINTS {automatic}
 $nks $nks $nks 0 0 0
EOF

$ECHO " running the initial vc-relax calculation ; \c"
$PW_COMMAND -input ReSSe.scf.vcrelax0.in &> ReSSe.scf.vcrelax0.out
$ECHO " done"

# Get relevant unprocessed lines from output file that contain relevant data

raw_cart_at_pos=$(grep  'tau(' ReSSe.scf.vcrelax0.out)
raw_cell_params=$(grep -A 3 'CELL' ReSSe.scf.vcrelax0.out)
raw_f=$(grep 'Total force' ReSSe.scf.vcrelax0.out | tail -1)
raw_p=$(grep 'bar' ReSSe.scf.vcrelax0.out | tail -1)
current_time=$(grep 'time' ReSSe.scf.vcrelax0.out | tail -1)

cat > ReSSe_3D_raw_cartesian_atomic_positions << EOF
$raw_cart_at_pos
EOF

cat > ReSSe_3D_raw_cell_parameters << EOF
$raw_cell_params
EOF

cat > ReSSe_3D_raw_force << EOF
$raw_f
EOF

cat > ReSSe_3D_raw_pressure << EOF
$raw_p
EOF

# Extract relevant data from these lines and store in the correct format for use as an input

cart_at_pos=$(awk '{print $2,$7,$8,$9}' ReSSe_3D_raw_cartesian_atomic_positions)
current_cell_parameters=$(tail -3 ReSSe_3D_raw_cell_parameters)
current_force=$(awk '{print $1,$2,$3,$4 }' ReSSe_3D_raw_force)
current_pressure=$(awk '{print $5,$6 }' ReSSe_3D_raw_pressure)

cat > ReSSe_3D_cartesian_atomic_positions << EOF
$cart_at_pos
EOF

cat > tempfile_1 << EOF
$current_cell_parameters
EOF

# Split atomic positions file in two in order to seperate initial and final sets

head -12  ReSSe_3D_cartesian_atomic_positions > ReSSe_3D_init_cartesian_atomic_positions && sed -i '1,+11d' ReSSe_3D_cartesian_atomic_positions

# Assign data to variables for future use

input_atomic_positions=$(cat ReSSe_3D_init_cartesian_atomic_positions)
current_atomic_positions=$(cat ReSSe_3D_cartesian_atomic_positions)

# Write key data to seperate output file

initial_header="Initial positions:"
echo $initial_header >> ReSSe_3D_atomic_positions.txt
cat ReSSe_3D_init_cartesian_atomic_positions >> ReSSe_3D_atomic_positions.txt

vcr1_cp_header="Cell parameters after vc-relax 1:"
echo $vcr1_cp_header >> ReSSe_3D_atomic_positions.txt
cat tempfile_1 >> ReSSe_3D_atomic_positions.txt

vcr1_ap_header="Atomic positions after vc-relax 1:"
echo $vcr1_ap_header >> ReSSe_3D_atomic_positions.txt
cat ReSSe_3D_cartesian_atomic_positions >> ReSSe_3D_atomic_positions.txt

echo $current_force >> ReSSe_3D_atomic_positions.txt
echo $current_pressure >> ReSSe_3D_atomic_positions.txt
echo $current_time >> ReSSe_3D_atomic_positions.txt

#
# Initial relax calculation
#

cat > ReSSe.scf.relax0.in << EOF
 &control
    calculation='relax',
    restart_mode='from_scratch',
    prefix='ReSSe',
    pseudo_dir = '$PSEUDO_DIR',
    outdir='$TMP_DIR',
    forc_conv_thr=$fthresh,
    etot_conv_thr=$ethresh,
    tprnfor=.true.,
    tstress=.true.,
 /
 &system    
    ibrav=  0, 
    A=6.597,
    nat=  12, ntyp= 3, 
    ecutwfc =$ecut,
    ecutrho =$rcut
 /
 &electrons
    conv_thr =  $elthresh,
    mixing_mode='plain',
    mixing_beta = 0.3,
 /
 &ions
    ion_dynamics='bfgs',
 /
CELL_PARAMETERS { alat }
$current_cell_parameters
ATOMIC_SPECIES
$atom1
$atom2
$atom3
ATOMIC_POSITIONS {alat}
$current_atomic_positions
K_POINTS {automatic}
 $nks $nks $nks 0 0 0
EOF

$ECHO " running the initial relax calculation ; \c"
$PW_COMMAND -input ReSSe.scf.relax0.in &> ReSSe.scf.relax0.out
$ECHO " done"

# Get relevant unprocessed lines from output file that contain relevant data

raw_cart_at_pos=$(grep -A 12 'ATOMIC' ReSSe.scf.relax0.out)
raw_f=$(grep 'Total force' ReSSe.scf.relax0.out | tail -1)
raw_p=$(grep 'bar' ReSSe.scf.relax0.out | tail -1)
current_time=$(grep 'time' ReSSe.scf.relax0.out | tail -1)

cat > ReSSe_3D_raw_cartesian_atomic_positions << EOF
$raw_cart_at_pos
EOF

cat > ReSSe_3D_raw_force << EOF
$raw_f
EOF

cat > ReSSe_3D_raw_pressure << EOF
$raw_p
EOF

# Extract relevant data from these lines and store in the correct format for use as an input

current_atomic_positions=$(tail -12 ReSSe_3D_raw_cartesian_atomic_positions)
current_force=$(awk '{print $1,$2,$3,$4 }' ReSSe_3D_raw_force)
current_pressure=$(awk '{print $5,$6 }' ReSSe_3D_raw_pressure)

cat > tempfile_1 << EOF
$current_atomic_positions
EOF

# Write key data to seperate output file

r1_ap_header="Atomic positions after relax 1:"
echo $r1_ap_header >> ReSSe_3D_atomic_positions.txt
cat tempfile_1 >> ReSSe_3D_atomic_positions.txt

echo $current_force >> ReSSe_3D_atomic_positions.txt
echo $current_pressure >> ReSSe_3D_atomic_positions.txt
echo $current_time >> ReSSe_3D_atomic_positions.txt

loop_counter=0

# Find whether relax thresholds have been reached 

f=$(awk '{print $4 }' ReSSe_3D_raw_force)
fdelta=`echo $f - $fthresh | bc -l`
bigfdelta=`echo $fdelta \* 1000000 | bc -l`
bigfdelta=${bigfdelta%.*}

p=$(awk '{print $6 }' res2_3D_raw_pressure)
bp=`echo $p \* 10000 | bc -l`
pp=${bp%.*}
if [ $pp -lt 0 ] 
then p=`echo 0 - $p | bc -l` 
fi
pdelta=`echo $p - $pthresh | bc -l`
bigpdelta=`echo $pdelta \* 1000000 | bc -l`
bigpdelta=${bigpdelta%.*}

# If they have not, loop through vc-relax/relax cycles until they have been

while [ $bigfdelta -gt 0 ]||[ $bigpdelta -gt 0 ]  

do

loop_counter=$(($loop_counter+1))

#
# vc-relax calculation
#

cat > ReSSe.scf.vcrelax$loop_counter.in << EOF
 &control
    calculation='vc-relax',
    restart_mode='from_scratch',
    prefix='ReSSe',
    pseudo_dir = '$PSEUDO_DIR',
    outdir='$TMP_DIR',
    forc_conv_thr=$fthresh,
    etot_conv_thr=$ethresh,
    tprnfor=.true.,
    tstress=.true.,
 /
 &system    
    ibrav=  0, 
    A=6.597,
    nat=  12, ntyp= 3, 
    ecutwfc =$ecut,
    ecutrho = $rcut
 /
 &electrons
    conv_thr =  $elthresh,
    mixing_mode='plain',
    mixing_beta = 0.3,
 /
 &ions
    ion_dynamics='bfgs',
 /
&cell
    cell_dynamics='bfgs',
 /
CELL_PARAMETERS { alat }
$current_cell_parameters
ATOMIC_SPECIES
$atom1
$atom2
$atom3
ATOMIC_POSITIONS {alat}
$current_atomic_positions
K_POINTS {automatic}
 $nks $nks $nks 0 0 0
EOF

$ECHO " running the vc-relax calculation ; \c"
$PW_COMMAND -input ReSSe.scf.vcrelax$loop_counter.in &> ReSSe.scf.vcrelax$loop_counter.out
$ECHO " done"

# Get relevant unprocessed lines from output file that contain relevant data

raw_cart_at_pos=$(grep -A 12 'ATOMIC' ReSSe.scf.vcrelax$loop_counter.out)
raw_cell_params=$(grep -A 3 'CELL' ReSSe.scf.vcrelax$loop_counter.out)
raw_f=$(grep 'Total force' ReSSe.scf.vcrelax$loop_counter.out | tail -1)
raw_p=$(grep 'bar' ReSSe.scf.vcrelax$loop_counter.out | tail -1)
current_time=$(grep 'time' ReSSe.scf.vcrelax$loop_counter.out | tail -1)

cat > ReSSe_3D_raw_cartesian_atomic_positions << EOF
$raw_cart_at_pos
EOF

cat > ReSSe_3D_raw_cell_parameters << EOF
$raw_cell_params
EOF

cat > ReSSe_3D_raw_force << EOF
$raw_f
EOF

cat > ReSSe_3D_raw_pressure << EOF
$raw_p
EOF

# Extract relevant data from these lines and store in the correct format for use as an input

current_atomic_positions=$(tail -12 ReSSe_3D_raw_cartesian_atomic_positions)
current_cell_parameters=$(tail -3 ReSSe_3D_raw_cell_parameters)
current_force=$(awk '{print $1,$2,$3,$4 }' ReSSe_3D_raw_force)
current_pressure=$(awk '{print $5,$6 }' ReSSe_3D_raw_pressure)

cat > tempfile_1 << EOF
$current_atomic_positions
EOF

cat > tempfile_2 << EOF
$current_cell_parameters
EOF

# Write key data to seperate output file

vcr2_cp_header="Cell parameters after vc-relax $loop_counter:"
echo $vcr2_cp_header >> ReSSe_3D_atomic_positions.txt
cat tempfile_2 >> ReSSe_3D_atomic_positions.txt

vcr2_ap_header="Atomic positions after vc-relax $loop_counter:"
echo $vcr2_ap_header >> ReSSe_3D_atomic_positions.txt
cat tempfile_1 >> ReSSe_3D_atomic_positions.txt

echo $current_force >> ReSSe_3D_atomic_positions.txt
echo $current_pressure >> ReSSe_3D_atomic_positions.txt
echo $current_time >> ReSSe_3D_atomic_positions.txt

#
# relax calculation
#

cat > ReSSe.scf.relax$loop_counter.in << EOF
 &control
    calculation='relax',
    restart_mode='from_scratch',
    prefix='ReSSe',
    pseudo_dir = '$PSEUDO_DIR',
    outdir='$TMP_DIR',
    forc_conv_thr=$fthresh,
    etot_conv_thr=$ethresh,
    tprnfor=.true.,
    tstress=.true.,
 /
 &system    
    ibrav=  0, 
    nat=  12, ntyp= 3, 
    ecutwfc =$ecut,
    ecutrho = $rcut,
    A=6.597
 /
 &electrons
    conv_thr =  $elthresh,
    mixing_mode='plain',
    mixing_beta = 0.3,
 /
 &ions
    ion_dynamics='bfgs',
 /
CELL_PARAMETERS { alat }
$current_cell_parameters
ATOMIC_SPECIES
$atom1
$atom2
$atom3
ATOMIC_POSITIONS {alat}
$current_atomic_positions
K_POINTS {automatic}
 $nks $nks $nks 0 0 0
EOF

$ECHO " running the relax calculation ; \c"
$PW_COMMAND -input ReSSe.scf.relax$loop_counter.in &> ReSSe.scf.relax$loop_counter.out
$ECHO " done"

# Get relevant unprocessed lines from output file that contain relevant data

raw_cart_at_pos=$(grep -A 12 'ATOMIC' ReSSe.scf.relax$loop_counter.out)
raw_f=$(grep 'Total force' ReSSe.scf.relax$loop_counter.out | tail -1)
raw_p=$(grep 'bar' ReSSe.scf.relax$loop_counter.out | tail -1)
current_time=$(grep 'time' ReSSe.scf.relax$loop_counter.out | tail -1)

cat > ReSSe_3D_raw_cartesian_atomic_positions << EOF
$raw_cart_at_pos
EOF

cat > ReSSe_3D_raw_force << EOF
$raw_f
EOF

cat > ReSSe_3D_raw_pressure << EOF
$raw_p
EOF

# Extract relevant data from these lines and store in the correct format for use as an input

current_atomic_positions=$(tail -12 ReSSe_3D_raw_cartesian_atomic_positions)
current_force=$(awk '{print $1,$2,$3,$4 }' ReSSe_3D_raw_force)
current_pressure=$(awk '{print $5,$6 }' ReSSe_3D_raw_pressure)

cat > tempfile_1 << EOF
$current_atomic_positions
EOF

# Write key data to seperate output file

r2_ap_header="Atomic positions after relax $loop_counter:"
echo $r2_ap_header >> ReSSe_3D_atomic_positions.txt
cat tempfile_1 >> ReSSe_3D_atomic_positions.txt

echo $current_force >> ReSSe_3D_atomic_positions.txt
echo $current_pressure >> ReSSe_3D_atomic_positions.txt
echo $current_time >> ReSSe_3D_atomic_positions.txt

# Determine if relax thresholds have now been met

f=$(awk '{print $4 }' ReSSe_3D_raw_force)
fdelta=`echo $f - $fthresh | bc -l`
bigfdelta=`echo $fdelta \* 1000000 | bc -l`
bigfdelta=${bigfdelta%.*}

p=$(awk '{print $6 }' res2_3D_raw_pressure)
bp=`echo $p \* 10000 | bc -l`
pp=${bp%.*}
if [ $pp -lt 0 ] 
then p=`echo 0 - $p | bc -l` 
fi
pdelta=`echo $p - $pthresh | bc -l`
bigpdelta=`echo $pdelta \* 1000000 | bc -l`
bigpdelta=${bigpdelta%.*}

done

#
# phonon calculation using relaxed parameters
#

cat > ReSSe.scf.phon.in << EOF
 &control
    calculation='scf'
    restart_mode='from_scratch',
    prefix='ReSSe',
    pseudo_dir = '$PSEUDO_DIR',
    outdir='$TMP_DIR',
    tprnfor=.true.,
    tstress=.true.,
 /
 &system    
    ibrav= 0, 
    A=6.597,
    nat=  12, ntyp= 3, 
    ecutwfc =$ecut,
    ecutrho = $rcut
 /
 &electrons
    conv_thr =  $elthresh,
    mixing_mode='plain',
    mixing_beta = 0.3,
 /
CELL_PARAMETERS (alat)
$current_cell_parameters
ATOMIC_SPECIES
$atom1
$atom2
$atom3
ATOMIC_POSITIONS (alat)
$current_atomic_positions
K_POINTS {automatic}
 $nks $nks $nks 0 0 0
EOF

$ECHO " running the scf calculation ; \c"
$PW_COMMAND -input ReSSe.scf.phon.in &> ReSSe.scf.phon.out
$ECHO " done"

cat > ReSSe.ph.in <<EOF
phonons of ReSSe
&inputph
   tr2_ph=$phthresh,
   prefix='ReSSe',
   ldisp=.false.,
   trans=.true.,
   epsil=.true.,
   lraman=.true.,
   amass(1)=186.21,
   amass(2)=78.96,
   amass(3)=32.065,
   outdir='$TMP_DIR/',
   fildyn='ReSSe.dyn',
/
0.0   0.0   0.0
EOF

$ECHO " running the phonon calculation"
$PH_COMMAND -input ReSSe.ph.in &> ReSSe.ph.out
$ECHO " done"

cat > dynmat.in <<EOF
&input
   fildyn='ReSSe.dyn',
   q(1)=0.0,
   q(2)=0.0,
   q(3)=0.0,
   asr='simple'
/
EOF

$ECHO " running dynmat"
$DYNMAT_COMMAND -input dynmat.in &> dynmat.out
$ECHO " done"

# Get relevant unprocessed lines from output file that contain relevant data

raw_f=$(grep 'Total force' ReSSe.scf.phon.out)
raw_p=$(grep 'bar' ReSSe.scf.phon.out)
current_time=$(grep 'time' ReSSe.ph.out | tail -1)

cat > ReSSe_3D_raw_force << EOF
$raw_f
EOF

cat > ReSSe_3D_raw_pressure << EOF
$raw_p
EOF

# Extract relevant data from these lines and store in the correct format for use as an input

current_force=$(awk '{print $1,$2,$3,$4 }' ReSSe_3D_raw_force)
current_pressure=$(awk '{print $5,$6 }' ReSSe_3D_raw_pressure)

# Write key data to separate output file

final_header="After phonon calculation:"
echo $final_header >> ReSSe_3D_atomic_positions.txt
echo $current_force >> ReSSe_3D_atomic_positions.txt
echo $current_pressure >> ReSSe_3D_atomic_positions.txt
echo $current_time >> ReSSe_3D_atomic_positions.txt

# Delete redundant files - omit, on assumption that there will be a phonon restart

#rm ReSSe_3D_raw_*
#rm tempfile_*
#rm ReSSe_3D_init_cartesian_atomic_positions
#rm ReSSe_3D_cartesian_atomic_positions
